/******************************************************************************\
* Copyright (C) 2006 by RTD Embedded Technologies, Inc.   All rights reserved.
* Confidential and Proprietary, Not for Public Release
*------------------------------------------------------------------------------
* PROJECT.......... Board Support Library for SPM176431
* VERSION.......... (Defined in README.TXT)
*------------------------------------------------------------------------------
* CONTENT.......... C source file for API of peripheral Dual UART
* FILENAME......... bsl_duart.c
*------------------------------------------------------------------------------
* PERIPHERAL NAME.. Dual UART
* PERIPHERAL TYPE.. Single-device peripheral
* ACCESS MODE...... Direct (no handle)
* REGISTER ACCESS.. 8-bit wide
\******************************************************************************/
#define _BSL_DUART_MOD_

#include <bsl_duart.h>
#include <csl_irq.h>
#include <bsl_uint.h>

#if (BSL_DUART_SUPPORT)
/******************************************************************************\
*                         L O C A L   S E C T I O N
\******************************************************************************/


/******************************************************************************\
* static macro declarations
\******************************************************************************/

/* duart write macro using flash relative offset */
#define DUART_WRITE16(flashByteOffset,val16) \
    *(volatile Uint16 *)(((Uint32)(DUART_STARTING_ADDRESS)+(Uint32)(flashByteOffset))) \
    = ((Uint16)(val16))

/* duart read macro using flash relative offset */
#define DUART_READ16(flashByteOffset) \
    (*(volatile Uint16 *)((Uint32)(DUART_STARTING_ADDRESS)+(Uint32)(flashByteOffset)))


/******************************************************************************\
* static typedef declarations
\******************************************************************************/

/******************************************************************************\
* static function declarations
\******************************************************************************/

void DUART_init(void);
void DUART_ISR_CB(Uint16 Uart);
void DUART_operation(Uint16 Uart);

/******************************************************************************\
* static variable definitions
\******************************************************************************/
/* the buffers defined below are mapped into specific memory locations
   specified in the linker file */
#ifdef __cplusplus

extern "C" {
#endif

    #pragma DATA_SECTION(Uart1RxBuff,".Uart1RecieveCircBuff");
    volatile char Uart1RxBuff[SIZE_OF_UART_BUFF];

    #pragma DATA_SECTION(Uart1TxBuff,".Uart1TransmitCircBuff");
    volatile char Uart1TxBuff[SIZE_OF_UART_BUFF];

    #pragma DATA_SECTION(Uart2RxBuff,".Uart2RecieveCircBuff");
    volatile char Uart2RxBuff[SIZE_OF_UART_BUFF];

    #pragma DATA_SECTION(Uart2TxBuff,".Uart2TransmitCircBuff");
    volatile char Uart2TxBuff[SIZE_OF_UART_BUFF];

#ifdef TESTING
    #pragma DATA_SECTION(Uart1TestBuff,".Uart1TestingOnly");
    volatile char Uart1TestBuff[SIZE_OF_UART_TEST_BUFF];

    #pragma DATA_SECTION(Uart2TestBuff,".Uart2TestingOnly");
    volatile char Uart2TestBuff[SIZE_OF_UART_TEST_BUFF];
#endif

/*************************************************************/

#ifdef __cplusplus
}
#endif  // extern "C" {

/* these pointer mark the begining and end of data
   in the Transmit/Recieve circular ques */
volatile Uint32 Uart1TransmitOut = 0;
volatile Uint32 Uart1RecieveIn = 0;
volatile Uint32 Uart2TransmitOut = 0;
volatile Uint32 Uart2RecieveIn = 0;

volatile Uint32 Uart1TransmitIn = 0;
volatile Uint32 Uart1RecieveOut = 0;
volatile Uint32 Uart2TransmitIn = 0;
volatile Uint32 Uart2RecieveOut = 0;

#ifdef TESTING

volatile Uint32 Uart1TestBuffIn = 0;
volatile Uint32 Uart1TestBuffOut = 0;
volatile Uint32 Uart2TestBuffIn = 0;
volatile Uint32 Uart2TestBuffOut = 0;

#endif

/********************************************/
/* these flags are set in the ISR to give the main program
   information about the state of the UART if necessary */
/* the input buffer is currently full and if the space is not cleared
   shortly the Rcv FIFO will overflow */
volatile BOOL UART1_FAILED_RECIEVE_WRITE_TO_INPUT_BUFFER = FALSE;
volatile BOOL UART2_FAILED_RECIEVE_WRITE_TO_INPUT_BUFFER = FALSE;
/* this is set if there was a line status interrupt in the UART
   which could have been caused by ...
   Rx overrun error, parity error, framing error, or rx-break condition */
volatile BOOL UART1_LINE_STATUS_INTERRUPT_ERROR = FALSE;
volatile BOOL UART2_LINE_STATUS_INTERRUPT_ERROR = FALSE;
/* this is set if there was a modem status interrupt in the UART
   which was caused by change in modem status lines */
volatile BOOL UART1_MODEM_STATUS_INTERRUPT_ERROR = FALSE;
volatile BOOL UART2_MODEM_STATUS_INTERRUPT_ERROR = FALSE;
/* signals that the transmitter was turned off since there was nothing
   to transmit in the output buffer */
volatile BOOL UART1_TRANSMIT_IT_WAS_TURNED_OFF = FALSE;
volatile BOOL UART2_TRANSMIT_IT_WAS_TURNED_OFF = FALSE;
/* signals that a transmit break has been recieved */
volatile BOOL UART1_RECIEVED_TX_BREAK = FALSE;
volatile BOOL UART2_RECIEVED_TX_BREAK = FALSE;


#ifdef TESTING

volatile BOOL UART1_TESTING_BUFFER_OVERFLOW = FALSE;
volatile BOOL UART2_TESTING_BUFFER_OVERFLOW = FALSE;

volatile BOOL UART1_TESTING_BUFFER_UNDERFLOW = FALSE;
volatile BOOL UART2_TESTING_BUFFER_UNDERFLOW = FALSE;

volatile BOOL UART1_TESTING_BUFFER_ERROR = FALSE;
volatile BOOL UART2_TESTING_BUFFER_ERROR = FALSE;

char UART1_RX_DATA = 0xFF;
char UART1_TX_DATA = 0xFF;

char UART2_RX_DATA = 0xFF;
char UART2_TX_DATA = 0xFF;

#endif

/*************************************************/

/******************************************************************************\
* static function definitions
\******************************************************************************/
//==============================================================================
// Initializes the dual uart.
//
// Called from the BLS_init().
//
// The state of dual UART interrupt pins can be accessed through an
// EPLD register, and an interrupt can be generated for the internal interrupts
// in the UART chips on the ExtInt4 to ExtInt7.
//
// The interrupt pins are redirected to the ExtInt7. The ExtInt7 provides an
// interrupt controlled duart access method.
// In the vector.asm, the ExtInt7 interrupt vector must be unmodified!
//==============================================================================
void DUART_init(void)
{
    /* clear out uart buffers*/
    memset(Uart1RxBuff, 0x0, SIZE_OF_UART_BUFF);
    memset(Uart1TxBuff, 0x0, SIZE_OF_UART_BUFF);
    memset(Uart2RxBuff, 0x0, SIZE_OF_UART_BUFF);
    memset(Uart2TxBuff, 0x0, SIZE_OF_UART_BUFF);

#ifdef TESTING

    memset(Uart1TestBuff, 0x0, SIZE_OF_UART_TEST_BUFF);
    memset(Uart2TestBuff, 0x0, SIZE_OF_UART_TEST_BUFF);

#endif
    /*********************************************/

    /* set the line to 9600 baud, with a word length of 8 bits,
       1 stop bit, parity disabled, odd parity, not forced parity */
    DUART_Set_Line_Control(0, 9600, 3, 0, 0, 0, 0);
    DUART_Set_Line_Control(1, 9600, 3, 0, 0, 0, 0);
    /* RS 232 mode for both uarts*/
    DUART_Mode(2, 2);
    /* enable fifos, reset rcv fifo, fifo trigger level = 14, reset tx fifo */
    DUART_Set_Fifo_Control(0,1, 1, 3, 1);
    DUART_Set_Fifo_Control(1,1, 1, 3, 1);
    /* clear the sticky bit for the external interrupt in FPGA */
    UINT_CLEAR_STATUS_S( PBINT6IA );
    UINT_CLEAR_STATUS_S( PBINT7IA );
    /* turn the interrupt onto external interrupt source 7 */
    UINT_FSETS( EIT7SRC, PBINT6IT, ENABLE );
    UINT_FSETS( EIT7SRC, PBINT7IT, ENABLE );
    /* enable rcv, tx interrupts only */
    DUART_Set_Interrupt_Enables(0, 1, 1, 1, 1);
    DUART_Set_Interrupt_Enables(1, 1, 1, 1, 1);

}

//==============================================================================
// FLASH module's Callback Interrupt Service Routine
//
// This function is called, when there is a rising edge on the duart device's
// interrupt pins.
//==============================================================================
void DUART_ISR_CB(Uint16 Uart)
{


    /* clearing the sticky bit of the DUART Interrupt from the
       Interrupt Status register to enable further interrupts */
    switch(Uart)
    {
        /* uart1 */
        case 0:
        UINT_CLEAR_STATUS_S( PBINT6IA );
        break;

        /* uart2 */
        case 1:
        UINT_CLEAR_STATUS_S( PBINT7IA );
        break;

        /* uart1 */
        default:
        UINT_CLEAR_STATUS_S( PBINT6IA );
        break;

    }/* switch */

    /* doing the current duart operation */
    DUART_operation(Uart);
}

//==============================================================================
// The is the main loop of the duart controller
//==============================================================================
void DUART_operation(Uint16 Uart)
{
    Uint32 ISR, RoomInBuffer, DataInBuffer;
    int TxCount;
    volatile Uint32 * UartRecieveIn;
    volatile Uint32 * UartTransmitOut;
    volatile char * UartRxBuff;
    volatile char * UartTxBuff;
    Uint32 DUART_RHR_OFFSET, DUART_THR_OFFSET;
    //volatile BOOL * UART_RECIEVED_TX_BREAK;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            UartRecieveIn = &Uart1RecieveIn;
            UartTransmitOut = &Uart1TransmitOut;
            UartRxBuff = Uart1RxBuff;
            UartTxBuff = Uart1TxBuff;
            DUART_RHR_OFFSET = DUART1_RHR_OFFSET;
            DUART_THR_OFFSET = DUART1_THR_OFFSET;
            break;

        /* uart2 */
        case 1:
            UartRecieveIn = &Uart2RecieveIn;
            UartTransmitOut = &Uart2TransmitOut;
            UartRxBuff = Uart2RxBuff;
            UartTxBuff = Uart2TxBuff;
            DUART_RHR_OFFSET = DUART2_RHR_OFFSET;
            DUART_THR_OFFSET = DUART2_THR_OFFSET;
            break;

        /* uart1 */
        default:
            UartRecieveIn = &Uart1RecieveIn;
            UartTransmitOut = &Uart1TransmitOut;
            UartRxBuff = Uart1RxBuff;
            UartTxBuff = Uart1TxBuff;
            DUART_RHR_OFFSET = DUART1_RHR_OFFSET;
            DUART_THR_OFFSET = DUART1_THR_OFFSET;
            break;
    }/* switch */

    ISR = DUART_Get_5_ISR_bits(Uart);
while(ISR != NO_IT_PENDING)
{
    switch (ISR)
    {
        /* Line Status Register */
        case LINE_STATUS_REGISTER:
            /* set flag for main program */
            Set_DUART_Line_Status_IT(Uart);
            DUART_Clear_Line_Stat_IT(Uart);
            /* as long as a LSR interrupt bit will be set ... */
            while(DUART_Is_Rx_Overrun_Error(Uart) || DUART_Is_Rx_Parity_Error(Uart) ||
                    DUART_Is_Rx_Framing_Error(Uart) || DUART_Is_Rx_Break_Error(Uart) )
            {

                /* read from the buffer until it won't be set */
                DUART_READ16(DUART_RHR_OFFSET);
            }
            /* clear out interrupt */
            DUART_Clear_Line_Stat_IT(Uart);
            break;

        /* Rxed Data Time out */
        case RXED_DATA_TIME_OUT:
        /* Rxed Data Ready */
        case RXED_DATA_READY:
            /* determine amount of room in buffer */
            RoomInBuffer = DUART_Room_In_Rcv(Uart);

            if(RoomInBuffer)
            {
                /* this flag is unset because there is now room in the buffer */
                Reset_DUART_Failed_Recieve_Write(Uart);

                /* while room in buffer and data to place in the buffer */
                while(RoomInBuffer && DUART_Is_Rx_Data_Ready(Uart))
                {
                    /* write the byte */
                    *(UartRxBuff + (*UartRecieveIn) ) = DUART_READ16(DUART_RHR_OFFSET) & 0x00FF;

                    /* increment pointers */
                    (*UartRecieveIn)++;

                    /* check for wrap around */
                    if((*UartRecieveIn) == SIZE_OF_UART_BUFF)
                        (*UartRecieveIn) = 0;

                    /* one less space available in recieve buffer */
                    RoomInBuffer--;
                }
            }
            else
            {/* if no room in buffer */
                /* set flag saying there is no room in buffer */
                Set_DUART_Failed_Recieve_Write(Uart);

                /* turn off Recieve interrupts so it doesn't lock up */
                DUART_Set_Interrupt_Enables(Uart, 0, DUART_DO_NOT_CHANGE_VALUE,
                         DUART_DO_NOT_CHANGE_VALUE, DUART_DO_NOT_CHANGE_VALUE);

            }/* end if */

            break;

        /* Transmit Ready */
        case TRANSMIT_READY:
            /* if we are here this must not be turned off */
            Reset_DUART_Tx_IT_turned_off_by_ISR(Uart);
            /* determine amount of room in buffer */
            DataInBuffer = DUART_Data_In_Tx(Uart);
            if(DataInBuffer)
            {
                for(TxCount = 0;( (TxCount < 16) && (DataInBuffer > 0) ); TxCount++)
                {
                    /* write the byte */
                    DUART_WRITE16(DUART_THR_OFFSET,   *(UartTxBuff + (*UartTransmitOut) ) );

                    /* increment pointers */
                    (*UartTransmitOut)++;

                    /* check for wrap around */
                    if((*UartTransmitOut) == SIZE_OF_UART_BUFF)
                        (*UartTransmitOut) = 0;

                    /* one less item in buffer now */
                    DataInBuffer--;

                }/* wrote up to 16 characters to Fifo */
            }
            else
            {/* transmit buffer is empty so... */
                /* turn off Transmit interrupts so it doesn't lock up */
                Set_DUART_Tx_IT_turned_off_by_ISR(Uart);
                DUART_Set_Interrupt_Enables(Uart, DUART_DO_NOT_CHANGE_VALUE, 0,
                            DUART_DO_NOT_CHANGE_VALUE, DUART_DO_NOT_CHANGE_VALUE);
            }/* end if */

            break;

        /* Modem Status Register */
        case MODEM_STATUS_REGISTER:
            /* set flag for main program */
            Set_DUART_Modem_Status_IT(Uart);
            /* clear interrupt */
            DUART_Clear_Modem_Stat_IT(Uart);
            break;

        /* Rxed Xoff/ Special character (this interrupt is never enabled)*/
        case RXED_XOFF_SPECIAL_CHARACTER:
            break;

        /* CTS/RTS change of state (this interrupt is never enabled)*/
        case CTS_RTS_CHANGE_OF_STATE:
            break;

        /* None */
        case NO_IT_PENDING:
            break;

        /* error if it reaches here */
        default:
            break;

    }/* switch */
    ISR = DUART_Get_5_ISR_bits(Uart);
}/* while */

}/* DUART_operation */

//=============================================================================
// DUART_Get_LCR
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//=============================================================================
Uint16 DUART_Get_LCR(Uint16 Uart)
{
    Uint16 LCR_OFFSET;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            LCR_OFFSET = DUART1_LCR_OFFSET;
            break;

        /* uart2 */
        case 1:
            LCR_OFFSET = DUART2_LCR_OFFSET;
            break;

        /* uart1 by default */
        default:
            LCR_OFFSET = DUART1_LCR_OFFSET;
            break;
    }/* switch */

    /* read current LCR */
    return DUART_READ16(LCR_OFFSET);

}/* DUART_Get_LCR */
//=============================================================================
// DUART_Write_LCR
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//      CurrentLCR : the unsigned 8 bit LCR value to write to register
//
//=============================================================================
void DUART_Write_LCR(Uint16 Uart, Uint16 CurrentLCR)
{
    Uint16 LCR_OFFSET;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            LCR_OFFSET = DUART1_LCR_OFFSET;
            break;

        /* uart2 */
        case 1:
            LCR_OFFSET = DUART2_LCR_OFFSET;
            break;

        /* uart1 by default */
        default:
            LCR_OFFSET = DUART1_LCR_OFFSET;
            break;
    }/* switch */

    /* write current LCR */
    DUART_WRITE16(LCR_OFFSET, CurrentLCR);

}/* DUART_Write_LCR */
//=============================================================================
// DUART_Write_MCR
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//=============================================================================
void DUART_Write_MCR(Uint16 Uart, Uint16 CurrentMCR)
{
    Uint16 CurrentLCR, LCR_OFFSET, MCR_OFFSET;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            LCR_OFFSET = DUART1_LCR_OFFSET;
            MCR_OFFSET = DUART1_MCR_OFFSET;
            break;

        /* uart2 */
        case 1:
            LCR_OFFSET = DUART2_LCR_OFFSET;
            MCR_OFFSET = DUART2_MCR_OFFSET;
            break;

        /* uart1 by default */
        default:
            LCR_OFFSET = DUART1_LCR_OFFSET;
            MCR_OFFSET = DUART1_MCR_OFFSET;
            break;
    }/* switch */

    /* save current setting of LCR */
    CurrentLCR = DUART_READ16(LCR_OFFSET);

    /* turn off bit if necessary */
    if(CurrentLCR == 0x00BF)
        DUART_WRITE16(LCR_OFFSET, ( CurrentLCR & 0x007F) );

     DUART_WRITE16(MCR_OFFSET, CurrentMCR);

    /* turn bit back on if necessary */
    if(CurrentLCR & 0x00BF)
        DUART_WRITE16(LCR_OFFSET, CurrentLCR );

}/* DUART_Write_MCR */
//=============================================================================
// DUART_Get_MCR
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//=============================================================================
Uint16 DUART_Get_MCR(Uint16 Uart)
{
    Uint16 CurrentLCR, CurrentMCR, LCR_OFFSET, MCR_OFFSET;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            LCR_OFFSET = DUART1_LCR_OFFSET;
            MCR_OFFSET = DUART1_MCR_OFFSET;
            break;

        /* uart2 */
        case 1:
            LCR_OFFSET = DUART2_LCR_OFFSET;
            MCR_OFFSET = DUART2_MCR_OFFSET;
            break;

        /* uart1 by default */
        default:
            LCR_OFFSET = DUART1_LCR_OFFSET;
            MCR_OFFSET = DUART1_MCR_OFFSET;
            break;
    }/* switch */

    /* save current setting of LCR */
    CurrentLCR = DUART_READ16(LCR_OFFSET);

    /* turn off bit if necessary */
    if(CurrentLCR == 0x00BF)
        DUART_WRITE16(LCR_OFFSET, ( CurrentLCR & 0x007F) );

    CurrentMCR = DUART_READ16(MCR_OFFSET);

    /* turn bit back on if necessary */
    if(CurrentLCR & 0x00BF)
        DUART_WRITE16(LCR_OFFSET, CurrentLCR );

    return CurrentMCR;

}/* DUART_Get_MCR */
//=============================================================================
// DUART_Get_MSR
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//=============================================================================
Uint16 DUART_Get_MSR(Uint16 Uart)
{
    Uint16 CurrentLCR, CurrentMSR, LCR_OFFSET, MSR_OFFSET;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            LCR_OFFSET = DUART1_LCR_OFFSET;
            MSR_OFFSET = DUART1_MSR_OFFSET;
            break;

        /* uart2 */
        case 1:
            LCR_OFFSET = DUART2_LCR_OFFSET;
            MSR_OFFSET = DUART2_MSR_OFFSET;
            break;

        /* uart1 by default */
        default:
            LCR_OFFSET = DUART1_LCR_OFFSET;
            MSR_OFFSET = DUART1_MSR_OFFSET;
            break;
    }/* switch */

    /* save current setting of LCR */
    CurrentLCR = DUART_READ16(LCR_OFFSET);

    /* turn off bit if necessary */
    if(CurrentLCR == 0x00BF)
        DUART_WRITE16(LCR_OFFSET, ( CurrentLCR & 0x007F) );

    CurrentMSR = DUART_READ16(MSR_OFFSET);

    /* turn bit back on if necessary */
    if(CurrentLCR & 0x00BF)
        DUART_WRITE16(LCR_OFFSET, CurrentLCR );

    return CurrentMSR;

}/* DUART_Get_MSR */
//=============================================================================
// DUART_Get_LSR
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//=============================================================================
Uint16 DUART_Get_LSR(Uint16 Uart)
{
    Uint16 CurrentLCR, CurrentLSR, LCR_OFFSET, LSR_OFFSET;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            LCR_OFFSET = DUART1_LCR_OFFSET;
            LSR_OFFSET = DUART1_LSR_OFFSET;
            break;

        /* uart2 */
        case 1:
            LCR_OFFSET = DUART2_LCR_OFFSET;
            LSR_OFFSET = DUART2_LSR_OFFSET;
            break;

        /* uart1 by default */
        default:
            LCR_OFFSET = DUART1_LCR_OFFSET;
            LSR_OFFSET = DUART1_LSR_OFFSET;
            break;
    }/* switch */

    /* save current setting of LCR */
    CurrentLCR = DUART_READ16(LCR_OFFSET);

    /* turn off bit if necessary */
    if(CurrentLCR == 0x00BF)
        DUART_WRITE16(LCR_OFFSET, ( CurrentLCR & 0x007F) );

    CurrentLSR = DUART_READ16(LSR_OFFSET);

    /* always check here for break condition */
    if((CurrentLSR & 0x0018) == 0x0018)
        Set_DUART_RECIEVED_TX_BREAK(Uart);

    /* turn bit back on if necessary */
    if(CurrentLCR & 0x00BF)
        DUART_WRITE16(LCR_OFFSET, CurrentLCR );

    return CurrentLSR;

}/* DUART_Get_LSR */
//=============================================================================
// DUART_Get_ISR
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//=============================================================================
Uint16 DUART_Get_ISR(Uint16 Uart)
{
    Uint16 CurrentLCR, CurrentISR, LCR_OFFSET, ISR_OFFSET;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            LCR_OFFSET = DUART1_LCR_OFFSET;
            ISR_OFFSET = DUART1_ISR_OFFSET;
            break;

        /* uart2 */
        case 1:
            LCR_OFFSET = DUART2_LCR_OFFSET;
            ISR_OFFSET = DUART2_ISR_OFFSET;
            break;

        /* uart1 by default */
        default:
            LCR_OFFSET = DUART1_LCR_OFFSET;
            ISR_OFFSET = DUART1_ISR_OFFSET;
            break;
    }/* switch */

    /* save current setting of LCR */
    CurrentLCR = DUART_READ16(LCR_OFFSET);

    if(CurrentLCR & 0x0080)
        DUART_WRITE16(LCR_OFFSET, ( CurrentLCR & 0x007F) );

    /* read current value of ISR */
    CurrentISR = DUART_READ16(ISR_OFFSET);

    /* turn bit back on if necessary */
    if(CurrentLCR & 0x0080)
        DUART_WRITE16(LCR_OFFSET, CurrentLCR );

    return CurrentISR;

}/* DUART_Get_ISR */
//=============================================================================
// DUART_Get_IER
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//=============================================================================
Uint16 DUART_Get_IER(Uint16 Uart)
{
    Uint16 CurrentLCR, CurrentIER, LCR_OFFSET, IER_OFFSET;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            LCR_OFFSET = DUART1_LCR_OFFSET;
            IER_OFFSET = DUART1_IER_OFFSET;
            break;

        /* uart2 */
        case 1:
            LCR_OFFSET = DUART2_LCR_OFFSET;
            IER_OFFSET = DUART2_IER_OFFSET;
            break;

        /* uart1 by default */
        default:
            LCR_OFFSET = DUART1_LCR_OFFSET;
            IER_OFFSET = DUART1_IER_OFFSET;
            break;
    }/* switch */

    /* save current setting of LCR */
    CurrentLCR = DUART_READ16(LCR_OFFSET);

    if(CurrentLCR & 0x0080)
        DUART_WRITE16(LCR_OFFSET, ( CurrentLCR & 0x007F) );

    /* read current value of IER */
    CurrentIER = DUART_READ16(IER_OFFSET);

    /* turn bit back on if necessary */
    if(CurrentLCR & 0x0080)
        DUART_WRITE16(LCR_OFFSET, CurrentLCR );

    return CurrentIER;

}/* DUART_Get_IER */
//=============================================================================
// DUART_Get_IER
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//=============================================================================
void DUART_Write_IER(Uint16 Uart, Uint16 CurrentIER)
{
    Uint16 CurrentLCR, LCR_OFFSET, IER_OFFSET;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            LCR_OFFSET = DUART1_LCR_OFFSET;
            IER_OFFSET = DUART1_IER_OFFSET;
            break;

        /* uart2 */
        case 1:
            LCR_OFFSET = DUART2_LCR_OFFSET;
            IER_OFFSET = DUART2_IER_OFFSET;
            break;

        /* uart1 by default */
        default:
            LCR_OFFSET = DUART1_LCR_OFFSET;
            IER_OFFSET = DUART1_IER_OFFSET;
            break;
    }/* switch */

    /* save current setting of LCR */
    CurrentLCR = DUART_READ16(LCR_OFFSET);

    if(CurrentLCR & 0x0080)
        DUART_WRITE16(LCR_OFFSET, ( CurrentLCR & 0x007F) );

    /* write current value of IER */
    DUART_WRITE16(IER_OFFSET, CurrentIER);

    /* turn bit back on if necessary */
    if(CurrentLCR & 0x0080)
        DUART_WRITE16(LCR_OFFSET, CurrentLCR );

}/* DUART_Write_IER */
/******************************************************************************\
*                        G L O B A L   S E C T I O N
\******************************************************************************/


/******************************************************************************\
* global variable definitions
\******************************************************************************/


/******************************************************************************\
* global function definitions
\******************************************************************************/
#ifdef TESTING

/* four functions below were added for testing only */
void Get_Last_Inconsistent_Data(Uint16 Uart, char *uart_data, char *actual_data)
{
    switch(Uart)
    {
        /* uart1 */
        case 0:
        (*uart_data) = UART1_RX_DATA;
        (*actual_data) = UART1_TX_DATA;
            break;

        /* uart2 */
        case 1:
        (*uart_data) = UART2_RX_DATA;
        (*actual_data) = UART2_TX_DATA;
            break;

        /* uart1 */
        default:
        (*uart_data) = UART1_RX_DATA;
        (*actual_data) = UART1_TX_DATA;
            break;
    }/* switch */
}

Uint32 Data_In_Uart_Test(Uint16 Uart)
{
    Uint32 DataInBuffer;
    Uint32 UartRecieveOut, UartRecieveIn;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            UartRecieveOut = Uart1TestBuffOut;
            UartRecieveIn = Uart1TestBuffIn;
            break;

        /* uart2 */
        case 1:
            UartRecieveOut = Uart2TestBuffOut;
            UartRecieveIn = Uart2TestBuffIn;
            break;

        /* uart1 */
        default:
            UartRecieveOut = Uart1TestBuffOut;
            UartRecieveIn = Uart1TestBuffIn;
            break;
    }/* switch */

    /* calculate how much room is in the buffer */
    if( UartRecieveOut == UartRecieveIn)
        DataInBuffer = 0;
    if( UartRecieveOut < UartRecieveIn)
        DataInBuffer = (UartRecieveIn - UartRecieveOut);
    if( UartRecieveIn < UartRecieveOut)
        DataInBuffer = SIZE_OF_UART_TEST_BUFF - (UartRecieveOut - UartRecieveIn);

    return DataInBuffer;

}/* Data_In_Uart_Test */
Uint32 Room_In_Uart_Test(Uint16 Uart)
{
    Uint32 RoomInBuffer;
    Uint32 UartRecieveOut, UartRecieveIn;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            UartRecieveOut = Uart1TestBuffOut;
            UartRecieveIn = Uart1TestBuffIn;
            break;

        /* uart2 */
        case 1:
            UartRecieveOut = Uart2TestBuffOut;
            UartRecieveIn = Uart2TestBuffIn;
            break;

        /* uart1 */
        default:
            UartRecieveOut = Uart1TestBuffOut;
            UartRecieveIn = Uart1TestBuffIn;
            break;
    }/* switch */

    /* calculate how much room is in the buffer */
    if( UartRecieveOut == UartRecieveIn)
        RoomInBuffer = SIZE_OF_UART_TEST_BUFF - 1;
    if( UartRecieveOut < UartRecieveIn)
        RoomInBuffer = SIZE_OF_UART_TEST_BUFF - (UartRecieveIn - UartRecieveOut) - 1;
    if( UartRecieveIn < UartRecieveOut)
        RoomInBuffer = (UartRecieveOut - UartRecieveIn) - 1;

    return RoomInBuffer;

}/* Room_In_Uart_Test */
void DUART_Flush_Test(Uint16 Uart)
{
    volatile Uint32 *UartRecieveOut;
    volatile Uint32 *UartRecieveIn;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            UartRecieveOut = &Uart1TestBuffOut;
            UartRecieveIn = &Uart1TestBuffIn;
            break;

        /* uart2 */
        case 1:
            UartRecieveOut = &Uart2TestBuffOut;
            UartRecieveIn = &Uart2TestBuffIn;
            break;

        /* uart1 */
        default:
            UartRecieveOut = &Uart1TestBuffOut;
            UartRecieveIn = &Uart1TestBuffIn;
            break;
    }/* switch */

    /* flush buffer by setting in/out pointers to 0 */
    (*UartRecieveOut) = 0;
    (*UartRecieveIn) = 0;

}/*  DUART_Flush_Test */


#endif /* TESTING */
/********************************************************************************/

Uint32 DUART_Data_In_Rcv(Uint16 Uart)
{
    Uint32 DataInBuffer;
    Uint32 UartRecieveOut, UartRecieveIn;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            UartRecieveOut = Uart1RecieveOut;
            UartRecieveIn = Uart1RecieveIn;
            break;

        /* uart2 */
        case 1:
            UartRecieveOut = Uart2RecieveOut;
            UartRecieveIn = Uart2RecieveIn;
            break;

        /* uart1 */
        default:
            UartRecieveOut = Uart1RecieveOut;
            UartRecieveIn = Uart1RecieveIn;
            break;
    }/* switch */

    /* calculate how much room is in the buffer */
    if( UartRecieveOut == UartRecieveIn)
        DataInBuffer = 0;
    if( UartRecieveOut < UartRecieveIn)
        DataInBuffer = (UartRecieveIn - UartRecieveOut);
    if( UartRecieveIn < UartRecieveOut)
        DataInBuffer = SIZE_OF_UART_BUFF - (UartRecieveOut - UartRecieveIn);

    return DataInBuffer;

}/* DUART_Data_In_Rcv */

Uint32 DUART_Data_In_Tx(Uint16 Uart)
{
    Uint32 DataInBuffer;
    Uint32 UartTransmitOut, UartTransmitIn;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            UartTransmitOut = Uart1TransmitOut;
            UartTransmitIn = Uart1TransmitIn;
            break;

        /* uart2 */
        case 1:
            UartTransmitOut = Uart2TransmitOut;
            UartTransmitIn = Uart2TransmitIn;
            break;

        /* uart1 */
        default:
            UartTransmitOut = Uart1TransmitOut;
            UartTransmitIn = Uart1TransmitIn;
            break;
    }/* switch */

    /* calculate how much room is in the buffer */
    if( UartTransmitOut == UartTransmitIn)
        DataInBuffer = 0;
    if( UartTransmitOut < UartTransmitIn)
        DataInBuffer = (UartTransmitIn - UartTransmitOut);
    if( UartTransmitIn < UartTransmitOut)
        DataInBuffer = SIZE_OF_UART_BUFF - (UartTransmitOut - UartTransmitIn);

    return DataInBuffer;


}/* DUART_Data_In_Tx */

Uint32 DUART_Room_In_Rcv(Uint16 Uart)
{
    Uint32 RoomInBuffer;
    Uint32 UartRecieveOut, UartRecieveIn;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            UartRecieveOut = Uart1RecieveOut;
            UartRecieveIn = Uart1RecieveIn;
            break;

        /* uart2 */
        case 1:
            UartRecieveOut = Uart2RecieveOut;
            UartRecieveIn = Uart2RecieveIn;
            break;

        /* uart1 */
        default:
            UartRecieveOut = Uart1RecieveOut;
            UartRecieveIn = Uart1RecieveIn;
            break;
    }/* switch */

    /* calculate how much room is in the buffer */
    if( UartRecieveOut == UartRecieveIn)
        RoomInBuffer = SIZE_OF_UART_BUFF - 1;
    if( UartRecieveOut < UartRecieveIn)
        RoomInBuffer = SIZE_OF_UART_BUFF - (UartRecieveIn - UartRecieveOut) - 1;
    if( UartRecieveIn < UartRecieveOut)
        RoomInBuffer = (UartRecieveOut - UartRecieveIn) - 1;

    return RoomInBuffer;

}/* DUART_Room_In_Rcv */

Uint32 DUART_Room_In_Tx(Uint16 Uart)
{
   Uint32 RoomInBuffer;
    Uint32 UartTransmitOut, UartTransmitIn;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            UartTransmitOut = Uart1TransmitOut;
            UartTransmitIn = Uart1TransmitIn;
            break;

        /* uart2 */
        case 1:
            UartTransmitOut = Uart2TransmitOut;
            UartTransmitIn = Uart2TransmitIn;
            break;

        /* uart1 */
        default:
            UartTransmitOut = Uart1TransmitOut;
            UartTransmitIn = Uart1TransmitIn;
            break;
    }/* switch */

    /* calculate how much room is in the buffer */
    if( UartTransmitOut == UartTransmitIn)
        RoomInBuffer = SIZE_OF_UART_BUFF - 1;
    if( UartTransmitOut < UartTransmitIn)
        RoomInBuffer = SIZE_OF_UART_BUFF - (UartTransmitIn - UartTransmitOut) - 1;
    if( UartTransmitIn < UartTransmitOut)
        RoomInBuffer = (UartTransmitOut - UartTransmitIn) - 1;

    return RoomInBuffer;

}/* DUART_Room_In_Tx */

void DUART_Flush_Rcv(Uint16 Uart)
{
    volatile Uint32 *UartRecieveOut;
    volatile Uint32 *UartRecieveIn;


    switch(Uart)
    {
        /* uart1 */
        case 0:
            UartRecieveOut = &Uart1RecieveOut;
            UartRecieveIn = &Uart1RecieveIn;
            break;

        /* uart2 */
        case 1:
            UartRecieveOut = &Uart2RecieveOut;
            UartRecieveIn = &Uart2RecieveIn;
            break;

        /* uart1 */
        default:
            UartRecieveOut = &Uart1RecieveOut;
            UartRecieveIn = &Uart1RecieveIn;
            break;
    }/* switch */

    if(DUART_Is_Rx_IT_Enabled(Uart))
    {

        /* turn off interrupt */
        DUART_Set_Interrupt_Enables(Uart, 0, DUART_DO_NOT_CHANGE_VALUE,
                         DUART_DO_NOT_CHANGE_VALUE, DUART_DO_NOT_CHANGE_VALUE);

        /* flush buffer by setting in/out pointers to 0 */
        (*UartRecieveOut) = 0;
        (*UartRecieveIn) = 0;

        /* turn on interrupt */
        DUART_Set_Interrupt_Enables(Uart, 1, DUART_DO_NOT_CHANGE_VALUE,
                         DUART_DO_NOT_CHANGE_VALUE, DUART_DO_NOT_CHANGE_VALUE);

    }else{

        /* flush buffer by setting in/out pointers to 0 */
        (*UartRecieveOut) = 0;
        (*UartRecieveIn) = 0;

    }/* if Rx interrupt on */


}/* DUART_Flush_Rcv */

void DUART_Flush_Tx(Uint16 Uart)
{
    volatile Uint32 *UartRecieveOut;
    volatile Uint32 *UartRecieveIn;


    switch(Uart)
    {
        /* uart1 */
        case 0:
            UartRecieveOut = &Uart1TransmitOut;
            UartRecieveIn = &Uart1TransmitIn;
            break;

        /* uart2 */
        case 1:
            UartRecieveOut = &Uart2TransmitOut;
            UartRecieveIn = &Uart2TransmitIn;
            break;

        /* uart1 */
        default:
            UartRecieveOut = &Uart1TransmitOut;
            UartRecieveIn = &Uart1TransmitIn;
            break;
    }/* switch */

    if(DUART_Is_Tx_IT_Enabled(Uart))
    {

        /* turn off interrupt */
        DUART_Set_Interrupt_Enables(Uart, DUART_DO_NOT_CHANGE_VALUE, 0,
                         DUART_DO_NOT_CHANGE_VALUE, DUART_DO_NOT_CHANGE_VALUE);

        /* flush buffer by setting in/out pointers to 0 */
        (*UartRecieveOut) = 0;
        (*UartRecieveIn) = 0;

        /* turn on interrupt */
        DUART_Set_Interrupt_Enables(Uart, DUART_DO_NOT_CHANGE_VALUE, 1,
                         DUART_DO_NOT_CHANGE_VALUE, DUART_DO_NOT_CHANGE_VALUE);

    }else{

        /* flush buffer by setting in/out pointers to 0 */
        (*UartRecieveOut) = 0;
        (*UartRecieveIn) = 0;

    }/* if Rx interrupt on */


}/* DUART_Flush_Rcv */

Uint32 DUART_Put_Data(Uint16 Uart, char *StartPtr, Uint32 NumOfBytes)
{
    int inner_loop_count, outter_loop_count;
    char UserData;
    Uint32 ChunkOfBytes, UartTransmitInCopy;
    volatile Uint32 * UartTransmitIn;
    volatile char * UartTxBuff;

#ifdef TESTING

    volatile BOOL * UART_TESTING_BUFFER_OVERFLOW;

#endif


    switch(Uart)
    {
        /* uart1 */
        case 0:
            /* point to the indexer to use */
            UartTransmitIn = &Uart1TransmitIn;
            /* point to the appropriate transmist buffer */
            UartTxBuff = Uart1TxBuff;
            break;

        /* uart2 */
        case 1:
            /* point to the indexer to use */
            UartTransmitIn = &Uart2TransmitIn;
            /* point to the appropriate transmist buffer */
            UartTxBuff = Uart2TxBuff;
            break;

        /* uart1 */
        default:
            /* point to the indexer to use */
            UartTransmitIn = &Uart1TransmitIn;
            /* point to the appropriate transmist buffer */
            UartTxBuff = Uart1TxBuff;
            break;
    }/* switch */

    /* check if there is enough room */
    if(DUART_Room_In_Tx(Uart) < NumOfBytes)
        return 0;

    for( outter_loop_count = 0; outter_loop_count <= NumOfBytes/MAX_NUM_OF_BYTES_TO_COPY_TO_UART_AT_ONCE; outter_loop_count++)
    {
        /* determines how many bytes to copy in for loop while transmit IT is off */
        if( outter_loop_count == NumOfBytes/MAX_NUM_OF_BYTES_TO_COPY_TO_UART_AT_ONCE)
            /* on the last iteration of this loop it will copy the remainder of bytes */
            ChunkOfBytes = NumOfBytes - outter_loop_count * MAX_NUM_OF_BYTES_TO_COPY_TO_UART_AT_ONCE;
        else
            /* all except the last copy will copy exactly the defined number of bytes */
            ChunkOfBytes = MAX_NUM_OF_BYTES_TO_COPY_TO_UART_AT_ONCE;

        /* get copy of current transmit indexer */
        UartTransmitInCopy = *UartTransmitIn;

        /* write bytes to buffer while updating buffer variables */
        for( inner_loop_count = 0; inner_loop_count < ChunkOfBytes; inner_loop_count++)
        {

            /* retrieve data from users buffer in store in variable */
            UserData = *StartPtr;

            /* take users stored data and write it into our buffer for transmitting */
            *(UartTxBuff + UartTransmitInCopy ) = UserData;

            /* increment pointers */
            UartTransmitInCopy++;
            StartPtr++;

            /* check for wrap around */
            if(UartTransmitInCopy == SIZE_OF_UART_BUFF)
            UartTransmitInCopy = 0;

            /* one less space available in transmit buffer */


        }/* end of for that copies data into buffer*/


        /* if it is on turn... */
        if(DUART_Is_Tx_IT_Enabled(Uart))
        {
            /* turn it off */
            DUART_Set_Interrupt_Enables(Uart, DUART_DO_NOT_CHANGE_VALUE, 0,
                         DUART_DO_NOT_CHANGE_VALUE, DUART_DO_NOT_CHANGE_VALUE);

            /* copy new value into buffer pointer */
            *UartTransmitIn = UartTransmitInCopy;

            /* turn it back on */
            DUART_Set_Interrupt_Enables(Uart, DUART_DO_NOT_CHANGE_VALUE, 1,
                         DUART_DO_NOT_CHANGE_VALUE, DUART_DO_NOT_CHANGE_VALUE);

        }
        else
        /* if it was not turned on */
        {
            /* copy new value into buffer pointer */
            *UartTransmitIn = UartTransmitInCopy;

            /* if it is was previously turned off turn it back on */
            if( Is_DUART_Tx_IT_turned_off_by_ISR(Uart))
                DUART_Set_Interrupt_Enables(Uart, DUART_DO_NOT_CHANGE_VALUE, 1,
                         DUART_DO_NOT_CHANGE_VALUE, DUART_DO_NOT_CHANGE_VALUE);
        }/* endif */


    }/* end of outter for loop */

#ifdef TESTING

    switch(Uart)
    {
        /* uart1 */
        case 0:
            /* point to the indexer to use */
            UartTransmitIn = &Uart1TestBuffIn;
            /* point to the appropriate transmist buffer */
            UartTxBuff = Uart1TestBuff;
            UART_TESTING_BUFFER_OVERFLOW = &UART1_TESTING_BUFFER_OVERFLOW;
            break;

        /* uart2 */
        case 1:
            /* point to the indexer to use */
            UartTransmitIn = &Uart2TestBuffIn;
            /* point to the appropriate transmist buffer */
            UartTxBuff = Uart2TestBuff;
            UART_TESTING_BUFFER_OVERFLOW = &UART2_TESTING_BUFFER_OVERFLOW;
            break;

        /* uart1 */
        default:
            /* point to the indexer to use */
            UartTransmitIn = &Uart1TestBuffIn;
            /* point to the appropriate transmist buffer */
            UartTxBuff = Uart1TestBuff;
            UART_TESTING_BUFFER_OVERFLOW = &UART1_TESTING_BUFFER_OVERFLOW;
            break;
    }/* switch */


    StartPtr = StartPtr - NumOfBytes;

    /* check if there is enough room */
    if(Room_In_Uart_Test(Uart) >= NumOfBytes)
    {

        for( inner_loop_count = 0; inner_loop_count < NumOfBytes; inner_loop_count++)
        {

                /* retrieve data from users buffer in store in variable */
                UserData = *StartPtr;

                /* take users stored data and write it into our buffer for transmitting */
                *(UartTxBuff + (*UartTransmitIn) ) = UserData;

                /* increment pointers */
                (*UartTransmitIn)++;
                StartPtr++;

                /* check for wrap around */
                if((*UartTransmitIn) == SIZE_OF_UART_TEST_BUFF)
                (*UartTransmitIn) = 0;

                /* one less space available in transmit buffer */


        }/* end of for that copies data into buffer*/
    }else{

        /* JUST SET A FLAG HERE */
        (*UART_TESTING_BUFFER_OVERFLOW) = TRUE;
    }

    /*****************************************************************************************/
#endif /* TESTING */

    return NumOfBytes;

}/* DUART_Put_Data */

#ifdef TESTING

Uint32 DUART_Get_Test_Data_WO_Remove(Uint16 Uart, char* *StartPtr, Uint32 NumOfBytes)
{
    Uint32 DataInBuffer;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            *StartPtr =(void *) &Uart1TestBuff[Uart1TestBuffOut];
            break;

        /* uart2 */
        case 1:
            *StartPtr =(void *) &Uart2TestBuff[Uart2TestBuffOut];
            break;

        /* uart1 */
        default:
            *StartPtr =(void *) &Uart1TestBuff[Uart1TestBuffOut];
            break;
    }/* switch */


    /* how much is data is available */
    DataInBuffer = Data_In_Uart_Test(Uart);

    /* if there is enough data */
    if(NumOfBytes <= DataInBuffer)
    {   return NumOfBytes;}
    else
    {   /* else return how much data is in bufer */
        return DataInBuffer;
    }
}/* DUART_Get_Test_Data_WO_Remove */
/******************************************************************************/

#endif /* TESTING */

Uint32 DUART_Get_Data_WO_Remove(Uint16 Uart, char* *StartPtr, Uint32 NumOfBytes)
{
    Uint32 DataInBuffer;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            *StartPtr =(void *) &Uart1RxBuff[Uart1RecieveOut];
            break;

        /* uart2 */
        case 1:
            *StartPtr =(void *) &Uart2RxBuff[Uart2RecieveOut];
            break;

        /* uart1 */
        default:
            *StartPtr =(void *) &Uart1RxBuff[Uart1RecieveOut];
            break;
    }/* switch */


    /* how much is data is available */
    DataInBuffer = DUART_Data_In_Rcv(Uart);

    /* if there is enough data */
    if(NumOfBytes <= DataInBuffer)
    {   return NumOfBytes;}
    else
    {   /* else return how much data is in bufer */
        return DataInBuffer;
    }
}/* DUART_Get_Data_WO_Remove */

Uint32 DUART_Remove_Data(Uint16 Uart, Uint32 Bytes)
{
    Uint32 DataInBuffer, ActualBytesToRemove;
    BOOL Rx_IT_Enabled;
    volatile Uint32 * UartRecieveOut;

#ifdef TESTING

    volatile Uint32 * UartTestBuffOut;
    volatile BOOL *UART_TESTING_BUFFER_ERROR;
    volatile BOOL *UART_TESTING_BUFFER_UNDERFLOW;
    char *UART_RX_DATA;
    char *UART_TX_DATA;
    char *CompareRemove, *CompareTest;
    volatile char *RemoveBuffEnd, *TestBuffEnd;
    int CompareCount;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            UartTestBuffOut = &Uart1TestBuffOut;
            UART_TESTING_BUFFER_ERROR = &UART1_TESTING_BUFFER_ERROR;
            UART_TESTING_BUFFER_UNDERFLOW = &UART1_TESTING_BUFFER_UNDERFLOW;
            RemoveBuffEnd = &Uart1RxBuff[SIZE_OF_UART_BUFF-1];
            TestBuffEnd = &Uart1TestBuff[SIZE_OF_UART_TEST_BUFF-1];
            UART_RX_DATA = &UART1_RX_DATA;
            UART_TX_DATA = &UART1_TX_DATA;
            break;

        /* uart2 */
        case 1:
            UartTestBuffOut = &Uart2TestBuffOut;
            UART_TESTING_BUFFER_ERROR = &UART2_TESTING_BUFFER_ERROR;
            UART_TESTING_BUFFER_UNDERFLOW = &UART2_TESTING_BUFFER_UNDERFLOW;
            RemoveBuffEnd = &Uart2RxBuff[SIZE_OF_UART_BUFF-1];
            TestBuffEnd = &Uart2TestBuff[SIZE_OF_UART_TEST_BUFF-1];
            UART_RX_DATA = &UART2_RX_DATA;
            UART_TX_DATA = &UART2_TX_DATA;
            break;

        /* uart1 */
        default:
            UartTestBuffOut = &Uart1TestBuffOut;
            UART_TESTING_BUFFER_ERROR = &UART1_TESTING_BUFFER_ERROR;
            UART_TESTING_BUFFER_UNDERFLOW = &UART1_TESTING_BUFFER_UNDERFLOW;
            RemoveBuffEnd = &Uart1RxBuff[SIZE_OF_UART_BUFF-1];
            TestBuffEnd = &Uart1TestBuff[SIZE_OF_UART_TEST_BUFF-1];
            UART_RX_DATA = &UART1_RX_DATA;
            UART_TX_DATA = &UART1_TX_DATA;
            break;

    }/* switch */

    /*************************************************/

#endif /* TESTING */

    switch(Uart)
    {
        /* uart1 */
        case 0:
            UartRecieveOut = &Uart1RecieveOut;
            break;

        /* uart2 */
        case 1:
            UartRecieveOut = &Uart2RecieveOut;
            break;

        /* uart1 */
        default:
            UartRecieveOut = &Uart1RecieveOut;
            break;

    }/* switch */

    /* see how much data is available in rcv buffer */
    DataInBuffer = DUART_Data_In_Rcv(Uart);

    /* make sure you are not removing anymore bytes than what is available */
    if(Bytes <= DataInBuffer)
        ActualBytesToRemove = Bytes;
    else
        ActualBytesToRemove = DataInBuffer;

#ifdef TESTING

    if(DUART_Get_Test_Data_WO_Remove(Uart, &CompareTest, ActualBytesToRemove) >= DUART_Get_Data_WO_Remove(Uart, &CompareRemove, ActualBytesToRemove) )
    /* test_buffer shold have at least as many characters as recieve buffer */
    {
        for(CompareCount = 0; CompareCount < ActualBytesToRemove; CompareCount++)
        {

                /* they don't match and it is the first characters that do not match */
                if((*CompareTest) != (*CompareRemove)  && (*UART_TESTING_BUFFER_ERROR) != TRUE)
                {
                    /* set flags */
                    (*UART_TESTING_BUFFER_ERROR) = TRUE;

                    /* make copy of incorrectly compared data */
                    (*UART_RX_DATA) = (*CompareRemove);
                    (*UART_TX_DATA) = (*CompareTest);

                }
                /* increment ptrs */
                CompareTest++;
                CompareRemove++;

                /* check for wrap around */
                if(CompareRemove > RemoveBuffEnd )
                        CompareRemove -= (SIZE_OF_UART_BUFF);

                if(CompareTest > TestBuffEnd )
                        CompareTest -= (SIZE_OF_UART_TEST_BUFF);

        }/* for */

                /* actually remove the data  from our test buffer */
                (*UartTestBuffOut) += ActualBytesToRemove;
                if((*UartTestBuffOut) >= SIZE_OF_UART_TEST_BUFF)
                    (*UartTestBuffOut) -= SIZE_OF_UART_TEST_BUFF;
    }else{

        /* JUST SET FLAG */
        (*UART_TESTING_BUFFER_UNDERFLOW) = TRUE;

    }/* if */

    /*************************************************/

#endif /* TESTING */

    /* if it is on turn... */
    if(DUART_Is_Rx_IT_Enabled(Uart))
    {
        /* turn it off and save its previous state */
        DUART_Set_Interrupt_Enables(Uart, 0, DUART_DO_NOT_CHANGE_VALUE,
                     DUART_DO_NOT_CHANGE_VALUE, DUART_DO_NOT_CHANGE_VALUE);
        Rx_IT_Enabled = TRUE;
    }
    else
    {   Rx_IT_Enabled = FALSE; }

    /* remove bytes from buffer */
    *UartRecieveOut += ActualBytesToRemove;

    /* check for wrap around */
    if(*UartRecieveOut >= SIZE_OF_UART_BUFF)
        *UartRecieveOut -= (SIZE_OF_UART_BUFF);

    /* if it is was turned on or it was turned off because the buffer was full */
    if(Rx_IT_Enabled || Is_DUART_Failed_Recieve_Write(Uart))
        DUART_Set_Interrupt_Enables(Uart, 1, DUART_DO_NOT_CHANGE_VALUE,
                     DUART_DO_NOT_CHANGE_VALUE, DUART_DO_NOT_CHANGE_VALUE);


    /* return actual bytes removed */
    return ActualBytesToRemove;

}/* DUART_Remove_Data */

Uint32 DUART_Set_Line_Control(Uint16 Uart, Uint32 Baud, Uint32 WordLength, Uint32 StopBits,
            Uint32 ParityEnable, Uint32 ParityEven, Uint32 ForceParity)
{
    Uint16 CurrentLCR, DLL_OFFSET, DLM_OFFSET;
    Uint32 Divisor;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            DLL_OFFSET = DUART1_DLL_OFFSET;
            DLM_OFFSET = DUART1_DLM_OFFSET;
            break;

        /* uart2 */
        case 1:
            DLL_OFFSET = DUART2_DLL_OFFSET;
            DLM_OFFSET = DUART2_DLM_OFFSET;
            break;

        /* uart1 */
        default:
            DLL_OFFSET = DUART1_DLL_OFFSET;
            DLM_OFFSET = DUART1_DLM_OFFSET;
            break;
    }

    /* validate parameters */
    if((Baud > 921600) && Baud != DUART_DO_NOT_CHANGE_VALUE)
        return DUART_ERROR_INVALID_PARAMETER;

    if(WordLength & 0xFFFFFFFC  && WordLength != DUART_DO_NOT_CHANGE_VALUE)
        return DUART_ERROR_INVALID_PARAMETER;

    if( StopBits & 0xFFFFFFFE  &&  StopBits != DUART_DO_NOT_CHANGE_VALUE)
        return DUART_ERROR_INVALID_PARAMETER;

    if( ParityEnable & 0xFFFFFFFE  &&  ParityEnable != DUART_DO_NOT_CHANGE_VALUE)
        return DUART_ERROR_INVALID_PARAMETER;

    if( ParityEven & 0xFFFFFFFE  &&  ParityEven != DUART_DO_NOT_CHANGE_VALUE)
        return DUART_ERROR_INVALID_PARAMETER;

    if( ForceParity & 0xFFFFFFFE  &&  ForceParity != DUART_DO_NOT_CHANGE_VALUE)
        return DUART_ERROR_INVALID_PARAMETER;


    /* save current setting of LCR */
    CurrentLCR = DUART_Get_LCR(Uart);

    Divisor = 921600 / Baud;

    /* make sure writing to DLL & DLM */
    DUART_Write_LCR(Uart, 0x80);

    /* split up Divisor and write to DLL & DLM */
    DUART_WRITE16(DLL_OFFSET, ( Divisor & 0x00FF) );
    DUART_WRITE16(DLM_OFFSET, ( ( Divisor & 0xFF00 ) >> 8 ) );

    /* modify the current LCR with new settings */
    if( WordLength != DUART_DO_NOT_CHANGE_VALUE )
    {
        CurrentLCR &= 0x00FC;
        CurrentLCR |= WordLength;
    }

    if( StopBits != DUART_DO_NOT_CHANGE_VALUE )
    {
        CurrentLCR &= 0x00FB;
        CurrentLCR |= ( StopBits << 2);
    }

    if( ParityEnable != DUART_DO_NOT_CHANGE_VALUE )
    {
        CurrentLCR &= 0x00F7;
        CurrentLCR |= ( ParityEnable << 3);
    }

    if( ParityEven != DUART_DO_NOT_CHANGE_VALUE )
    {
        CurrentLCR &= 0x00EF;
        CurrentLCR |= ( ParityEven << 4);
    }

    if( ForceParity != DUART_DO_NOT_CHANGE_VALUE )
    {
        CurrentLCR &= 0x00DB;
        CurrentLCR |= ( ForceParity << 5);
    }

    /* make sure "divisor enable" is off and tx break is off */
    CurrentLCR &= 0x003F;

    /* rewrite the modified LCR back to the UART */
    DUART_Write_LCR(Uart, CurrentLCR);


    /* return the actual baud rate that was set */
    if( Divisor * Baud != 921600)
        return DUART_BAUD_RATE_ROUNDED_DOWN;
    else
        return  DUART_NO_ERROR;

}/* DUART_Set_Line_Control */

Uint32 DUART_Mode(Uint32 ModeUart1, Uint32 ModeUart2)
{
    Uint16 CurrentMode;

    /* check parameter */
    if(ModeUart1 & 0xFFFFFFFC  && ModeUart1 != DUART_DO_NOT_CHANGE_VALUE)
        return DUART_ERROR_INVALID_PARAMETER;

    if(ModeUart2 & 0xFFFFFFFC  && ModeUart2 != DUART_DO_NOT_CHANGE_VALUE)
        return DUART_ERROR_INVALID_PARAMETER;

    /* read current value of mode*/
    CurrentMode = DUART_READ16(DUART_MODE_OFFSET);

    /* change value if necessary */
    if(ModeUart1 != DUART_DO_NOT_CHANGE_VALUE)
    {
        CurrentMode &= 0xFFFC;
        CurrentMode |= ModeUart1;
    }

    if(ModeUart2 != DUART_DO_NOT_CHANGE_VALUE)
    {
        CurrentMode &= 0xFFCF;
        CurrentMode |= (ModeUart2 << 4);
    }

    /* write new value back */
    DUART_WRITE16(DUART_MODE_OFFSET, CurrentMode);

    return DUART_NO_ERROR;

}/* DUART_Mode */

Uint32 DUART_Set_Fifo_Control(Uint16 Uart, Uint32 Enable, Uint32 RcvFifoReset,
                                Uint32 RcvFifoTrigger, Uint32 TxFifoReset)
{
    Uint16 CurrentLCR, CurrentFCR, FCR_OFFSET;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            FCR_OFFSET = DUART1_FCR_OFFSET;
            break;

        /* uart2 */
        case 1:
            FCR_OFFSET = DUART2_FCR_OFFSET;
            break;

        /* uart1 */
        default:
            FCR_OFFSET = DUART1_FCR_OFFSET;
            break;
    }

    /* validate parameters */
    if( Enable & 0xFFFFFFFE  &&  Enable != DUART_DO_NOT_CHANGE_VALUE)
        return DUART_ERROR_INVALID_PARAMETER;

    if( RcvFifoReset & 0xFFFFFFFE  &&  RcvFifoReset != DUART_DO_NOT_CHANGE_VALUE)
        return DUART_ERROR_INVALID_PARAMETER;

    if( TxFifoReset & 0xFFFFFFFE  &&  TxFifoReset != DUART_DO_NOT_CHANGE_VALUE)
        return DUART_ERROR_INVALID_PARAMETER;

    if( RcvFifoTrigger & 0xFFFFFFFC  &&  RcvFifoTrigger != DUART_DO_NOT_CHANGE_VALUE)
        return DUART_ERROR_INVALID_PARAMETER;

    /* save current setting of LCR */
    CurrentLCR = DUART_Get_LCR(Uart);

    /* turn off bit if necessary */
    if(CurrentLCR & 0x0080)
        DUART_Write_LCR(Uart, ( CurrentLCR & 0x007F) );

    /* can not read current settings of FCR */
    CurrentFCR = 0;

    /* modify the current FCR with new settings */
    if( Enable != DUART_DO_NOT_CHANGE_VALUE )
    {
        CurrentFCR |= (Enable << 0);
    }

    if( RcvFifoReset != DUART_DO_NOT_CHANGE_VALUE )
    {
        CurrentFCR |= (RcvFifoReset << 1);
    }

    if( TxFifoReset != DUART_DO_NOT_CHANGE_VALUE )
    {
        CurrentFCR |= (TxFifoReset << 2);
    }

    if( RcvFifoTrigger != DUART_DO_NOT_CHANGE_VALUE )
    {
        CurrentFCR |= (RcvFifoTrigger << 6);
    }


    /* write current setting of FCR */
    DUART_WRITE16(FCR_OFFSET, CurrentFCR);

    /* turn bit back on if necessary */
    if(CurrentLCR & 0x0080)
        DUART_Write_LCR(Uart, CurrentLCR );


    return DUART_NO_ERROR;

}/* DUART_Set_Fifo_Control */

Uint32 DUART_Set_Interrupt_Enables(Uint16 Uart, Uint32 RcvIT, Uint32 TxIT,
                        Uint32 RcvLineStatIT, Uint32 ModemStatIT)
{
    Uint16 CurrentIER;

    /* validate parameters */
    if( RcvIT & 0xFFFFFFFE  &&  RcvIT != DUART_DO_NOT_CHANGE_VALUE)
        return DUART_ERROR_INVALID_PARAMETER;

    if( TxIT & 0xFFFFFFFE  &&  TxIT != DUART_DO_NOT_CHANGE_VALUE)
        return DUART_ERROR_INVALID_PARAMETER;

    if( RcvLineStatIT & 0xFFFFFFFE  &&  RcvLineStatIT != DUART_DO_NOT_CHANGE_VALUE)
        return DUART_ERROR_INVALID_PARAMETER;

    if( ModemStatIT & 0xFFFFFFFE  &&  ModemStatIT != DUART_DO_NOT_CHANGE_VALUE)
        return DUART_ERROR_INVALID_PARAMETER;

    /* read current setting of FCR */
    CurrentIER = DUART_Get_IER(Uart);

    /* modify the current LCR with new settings */
    if( RcvIT != DUART_DO_NOT_CHANGE_VALUE )
    {
        CurrentIER &= 0xFFFE;
        CurrentIER |= RcvIT;
    }

    if( TxIT != DUART_DO_NOT_CHANGE_VALUE )
    {
        CurrentIER &= 0xFFFD;
        CurrentIER |= (TxIT << 1);
    }

    if( RcvLineStatIT != DUART_DO_NOT_CHANGE_VALUE )
    {
        CurrentIER &= 0xFFFB;
        CurrentIER |= (RcvLineStatIT << 2);
    }

    if( ModemStatIT != DUART_DO_NOT_CHANGE_VALUE )
    {
        CurrentIER &= 0xFFF7;
        CurrentIER |= (ModemStatIT << 3);
    }


    /* write current setting of FCR */
    DUART_Write_IER(Uart, CurrentIER);

    return DUART_NO_ERROR;

}/* DUART_Set_Interrupt_Enables */

BOOL DUART_Is_Rx_IT_Enabled(Uint16 Uart)
{

    return ( (DUART_Get_IER(Uart) >> 0 ) & 1 );

}/* DUART_Is_Rx_IT_Enabled */

BOOL DUART_Is_Tx_IT_Enabled(Uint16 Uart)
{

    return ( (DUART_Get_IER(Uart) >> 1 ) & 1 );

}/* DUART_Is_Rx_IT_Enabled */

BOOL DUART_Is_Rx_Line_Status_IT_Enabled(Uint16 Uart)
{

    return ( (DUART_Get_IER(Uart) >> 2 ) & 1 );

}/* DUART_Is_Rx_Line_Status_IT_Enabled */

BOOL DUART_Is_Modem_Status_IT_Enabled(Uint16 Uart)
{

    return ( (DUART_Get_IER(Uart) >> 3 ) & 1 );

}/* DUART_Is_Modem_Status_IT_Enabled */

Uint32 DUART_Get_5_ISR_bits(Uint16 Uart)
{

    return (DUART_Get_ISR(Uart) & 0x001F);

}/* DUART_Get_5_ISR_bits */

void DUART_Clear_Line_Stat_IT(Uint16 Uart)
{
    DUART_Get_LSR(Uart);

}/* DUART_Clear_Line_Stat_IT */

void DUART_Clear_Modem_Stat_IT(Uint16 Uart)
{
    DUART_Get_MSR(Uart);

}/* DUART_Clear_Modem_Stat_IT */

BOOL DUART_Is_Rx_Data_Ready(Uint16 Uart)
{

    return ( (DUART_Get_LSR(Uart) >> 0 ) & 1 );

}/* DUART_Is_Rx_Data_Ready */

BOOL DUART_Is_Rx_Overrun_Error(Uint16 Uart)
{

    return ( (DUART_Get_LSR(Uart) >> 1 ) & 1 );

}/* DUART_Is_Rx_Overrun_Error */

BOOL DUART_Is_Rx_Parity_Error(Uint16 Uart)
{

    return ( (DUART_Get_LSR(Uart) >> 2 ) & 1 );

}/* DUART_Is_Rx_Parity_Error */

BOOL DUART_Is_Rx_Framing_Error(Uint16 Uart)
{

    return ( (DUART_Get_LSR(Uart) >> 3 ) & 1 );

}/* DUART_Is_Rx_Framing_Error */

BOOL DUART_Is_Rx_Break_Error(Uint16 Uart)
{

    return ( (DUART_Get_LSR(Uart) >> 4 ) & 1 );

}/* DUART_Is_Rx_Break_Error */

BOOL DUART_Is_THR_Empty(Uint16 Uart)
{

    return ( (DUART_Get_LSR(Uart) >> 5 ) & 1 );

}/* DUART_Is_THR_Empty */

BOOL DUART_Is_THR_TSR_Both_Empty(Uint16 Uart)
{

    return ( (DUART_Get_LSR(Uart) >> 6 ) & 1 );

}/* DUART_Is_THR_TSR_Both_Empty */

BOOL DUART_Is_Rx_FIFO_Global_Error(Uint16 Uart)
{
    return ( (DUART_Get_LSR(Uart) >> 7 ) & 1 );

}/* DUART_Is_Rx_FIFO_Global_Error */

void DUART_Enable_Tx_Break(Uint16 Uart)
{
    Uint16 CurrentLCR;

    /* read current LCR */
    CurrentLCR = DUART_Get_LCR(Uart);

    /* turn on break bit */
    CurrentLCR |= 0x0040;

    /* write back updated value */
    DUART_Write_LCR(Uart, CurrentLCR);

}/* DUART_Enable_Tx_Break */

void DUART_Disable_Tx_Break(Uint16 Uart)
{
    Uint16 CurrentLCR;

    /* read current LCR */
    CurrentLCR = DUART_Get_LCR(Uart);

    /* turn off break bit */
    CurrentLCR &= 0xFFBF;

    /* write back updated value */
    DUART_Write_LCR(Uart, CurrentLCR);

}/* DUART_Disable_Tx_Break */

void DUART_Enable_Transmitter(Uint16 Uart)
{
	DUART_Set_RTS(Uart, FALSE);

}/* DUART_Enable_Transmitter */

void DUART_Disable_Transmitter(Uint16 Uart)
{
	DUART_Set_RTS(Uart, TRUE);

}/* DUART_Disable_Transmitter */
#endif /* BSL_DUART_SUPPORT */
/******************************************************************************\
* End of bsl_duart.c
\******************************************************************************/
