/******************************************************************************\
* Copyright (C) 2004 by RTD Embedded Technologies, Inc.   All rights reserved.
* Confidential and Proprietary, Not for Public Release
*------------------------------------------------------------------------------
* PROJECT.......... Board Support Library for SPM6420 and SPM6430
* VERSION.......... (Defined in README.TXT)
*------------------------------------------------------------------------------
* CONTENT.......... Utility functions
* FILENAME......... bsl_util.c
\******************************************************************************/
#define _BSL_UINT_MOD_

#include <bsl_util.h>
#include <math.h>


/******************************************************************************\
*                         L O C A L   S E C T I O N
\******************************************************************************/


/******************************************************************************\
* static macro declarations
\******************************************************************************/


/******************************************************************************\
* static typedef declarations
\******************************************************************************/


/******************************************************************************\
* static function declarations
\******************************************************************************/


/******************************************************************************\
* static variable definitions
\******************************************************************************/


/******************************************************************************\
* static function definitions
\******************************************************************************/


/******************************************************************************\
*                        G L O B A L   S E C T I O N
\******************************************************************************/


/******************************************************************************\
* global variable definitions
\******************************************************************************/


/******************************************************************************\
* global function definitions
\******************************************************************************/

//==============================================================================
// Returns the value for the timer's period register according to the desired
// frequency and the timer and the round modes.
//
// parameters:
//	desiredFreqInHz		Desired frequency in Hz.
//	timerMode			Selects the operating mode of the timer. Can be pulse
//						or clock.
//						UTIL_GTP_PULSE_MODE for pulse mode timer setting
//						UTIL_GTP_CLOCK_MODE for clock mode timer setting
//	roundMode			Defines how to calculate the exact value of the period
//						register, if the value is not an integer number
//						according to the desired frequency.
//						UTIL_GTP_LOWER_FREQ or
//						UTIL_GTP_LONGER_PERIOD for getting a lower frequency
//						UTIL_GTP_HIGHER_FREQ or
//						UTIL_GTP_SHORTER_PERIOD for getting a higher frequency
//	gotFreqInHz			Pointer to a double-type variable to return the got
//						frequency after rounding. The pointer can be NULL.
//
// returns:
//	Timer's period register value.
//==============================================================================
Uint32 UTIL_getTimerPeriod(
	double desiredFreqInHz, Uint32 timerMode, Uint32 roundMode,
	double * gotFreqInHz)
{
	double	dbPeriod;	// floating-point period value
	Uint32	nPeriod;	// fixed-point period value

	// in 'C6416, the Timer works with CPU rate / 8

	// at pulse mode
	dbPeriod = (((double)BSL_boardDescriptor.dspNomSpeed) * (1.0e6 / 8.0)) / desiredFreqInHz;

	// at clock mode
	if( timerMode != UTIL_GTP_PULSE_MODE ) dbPeriod /= 2.0;

	if( roundMode == UTIL_GTP_LONGER_PERIOD ) dbPeriod = ceil(dbPeriod);
	else dbPeriod = floor(dbPeriod);

	// overflow protection (when the desired frequency is too low)
	if( dbPeriod > ((double)0xFFFFFFFFu) ) nPeriod = 0xFFFFFFFFu;
	else nPeriod = (Uint32)(dbPeriod);

	if( gotFreqInHz != 0 )
	{
		// at pulse mode
		*gotFreqInHz = ((double)BSL_boardDescriptor.dspNomSpeed) * (1.0e6 / 8.0) / (double)(nPeriod);

		// at clock mode
		if( timerMode != UTIL_GTP_PULSE_MODE ) *gotFreqInHz /= 2.0;
	}

	return nPeriod;
}


/******************************************************************************\
* End of bsl_util.c
\******************************************************************************/
