/******************************************************************************\
* File Name        : bsl_aim6020.h
* Target           : AIM6020 Analog Front-end (with SPM6020 dspModule) Board
* Compiler         : Code Composer Studio 2.0, 2.2
* Version          : 1.0
*
* This file provides functions, macros, constants, and structures to handle the
* resources of the AIM6020 Analog Interface Module board.
*
* Copyright (c) 2003, RTD Embedded Technologies, Inc.
\******************************************************************************/
/*
 *  Copyright 2002 by Texas Instruments Incorporated.
 *  All rights reserved. Property of Texas Instruments Incorporated.
 *  Restricted rights to use, duplicate or disclose this code are
 *  granted through contract.
 *  
 */
#ifndef _BSL_AIM6020_H_
#define _BSL_AIM6020_H_

#include <bsl_aim6020hal.h>
#include <csl_irq.h>


#if (BSL_AIM6020_SUPPORT)
/******************************************************************************\
* scope and inline control macros
\******************************************************************************/
#ifdef	__cplusplus
	#define	BSLAPI extern "C" far 
#else
	#define	BSLAPI extern far
#endif

#undef	USEDEFS
#undef	IDECL
#undef	IDEF

#ifdef	_BSL_AIM6020_MOD_
	#define IDECL BSLAPI
	#define USEDEFS
	#define IDEF
#else
	#ifdef	_INLINE
		#define	IDECL static inline
		#define	USEDEFS
		#define	IDEF  static inline
	#else
		#define	IDECL BSLAPI
	#endif
#endif


/******************************************************************************\
* global macro declarations
\******************************************************************************/
/* AIM6020_open() flags */
#define AIM6020_OPEN_RESET							0x00000001u

/* AIM6020_close() flags */
#define AIM6020_CLOSE_RESET							0x00000001u

/* AIM6020 Board Identifiers for AIM6020_open() */
#define AIM6020_BOARD0								0
#define AIM6020_BOARD1								1
#define AIM6020_BOARD2								2
#define AIM6020_BOARD3								3
#define AIM6020_BOARD_FIRST_AVAILABLE				(-1)
#define AIM6020_BOARD_FIRST_FREE					(-1)

/* for channel (converter) selection */
#define AIM6020_CHANNEL1							1
#define AIM6020_CHANNEL2							2
#define AIM6020_CHANNEL12							3
#define AIM6020_BOTH_CHANNELS						3

/* Start Source Signals for the CONFIG Register */
#define AIM6020_START_SIGNAL_SOFTWARE				0x00000000u
#define AIM6020_START_SIGNAL_SYNCBUS0				0x00000001u
#define AIM6020_START_SIGNAL_SYNCBUS1				0x00000002u
#define AIM6020_START_SIGNAL_SYNCBUS2				0x00000003u

/* Update Source Signals for the CONFIG Register */
#define AIM6020_UPDATE_SIGNAL_SOFTWARE				0x00000000u
#define AIM6020_UPDATE_SIGNAL_SYNCBUS0				0x00000001u
#define AIM6020_UPDATE_SIGNAL_SYNCBUS1				0x00000002u
#define AIM6020_UPDATE_SIGNAL_SYNCBUS2				0x00000003u

/* Filter Clock Selection for the FLTCFG Register */
#define AIM6020_FILTER_CLOCK1						0x00000000u
#define AIM6020_FILTER_CLOCK2						0x00000001u

/* Filter Switch for the FLTCFG Register */
#define AIM6020_FILTER_SWITCH_OFF					0x00000000u
#define AIM6020_FILTER_SWITCH_ON					0x00000001u
#define AIM6020_FILTER_SWITCH_DISABLED				0x00000000u
#define AIM6020_FILTER_SWITCH_ENABLED				0x00000001u
#define AIM6020_FILTER_OFF							0x00000000u
#define AIM6020_FILTER_ON							0x00000001u
#define AIM6020_FILTER_DISABLED						0x00000000u
#define AIM6020_FILTER_ENABLED						0x00000001u

/* Interrupt Source Signals for the INTCFG Register */
#define AIM6020_IT_SOURCE_DISABLED					0x00000000u
#define AIM6020_IT_SOURCE_AD1EOC					0x00000001u
#define AIM6020_IT_SOURCE_AD2EOC					0x00000002u
#define AIM6020_IT_SOURCE_PARLOAD					0x00000003u

/* for select the memory of the digital potentiometer */
/* After the power-on-reset, the AIM6020_DIGPOT_MEM_FILTER_OFF memory content
 * is loaded automatically.
 */
#define AIM6020_DIGPOT_MEM_FILTER_OFF				0
#define AIM6020_DIGPOT_MEM_FILTER_ON				1
#define AIM6020_DIGPOT_MEM_USER1					2
#define AIM6020_DIGPOT_MEM_USER2					3

/* Device Identifiers for both Analog-to-Digital converters. */
#define AIM6020_DIGPOT_DEVID_ADC					0x50
/* Device Identifiers for both Digital-to-Analog converters. */
#define AIM6020_DIGPOT_DEVID_DAC					0x51

/* Potentiometer Identifiers for both Analog-to-Digital converters. */
/* Offset for Analog-to-Digital converter channel 1 */
#define AIM6020_DIGPOT_POTID_AI1O					0x00
/* Offset for Analog-to-Digital converter channel 2 */
#define AIM6020_DIGPOT_POTID_AI2O					0x01
/* Gain for Analog-to-Digital converter channel 1 */
#define AIM6020_DIGPOT_POTID_AI1G					0x02
/* Gain for Analog-to-Digital converter channel 2 */
#define AIM6020_DIGPOT_POTID_AI2G					0x03

/* Offset for Digital-to-Analog converter channel 1 */
#define AIM6020_DIGPOT_POTID_AO1O					0x10
/* Offset for Digital-to-Analog converter channel 2 */
#define AIM6020_DIGPOT_POTID_AO2O					0x11
/* Gain for Digital-to-Analog converter channel 1 */
#define AIM6020_DIGPOT_POTID_AO1G					0x12
/* Gain for Digital-to-Analog converter channel 2 */
#define AIM6020_DIGPOT_POTID_AO2G					0x13


/******************************************************************************\
* global typedef declarations
\******************************************************************************/

/* channel configuration structure */
typedef struct {
	Uint32 regCONFIG;
	Uint32 regFLTCLK1;
	Uint32 regFLTCLK2;
	Uint32 regFLTCFG;
	Uint32 regINTCFG;
	Uint32 regDISPLAY;
} AIM6020_Config;

/* record structure for registers that contain write-only fields
 *
 * If you want to read these registers, you get values from this RecordedRegs
 * array. This array is updated with writing of the "write-only" registers
 * together.
 */
typedef struct {
	Uint32 regCONFIG;
	Uint32 regFLTCLK1;
	Uint32 regFLTCLK2;
	Uint32 regFLTCFG;
	Uint32 regINTCFG;
	Uint32 regDISPLAY;
} AIM6020_RecordedRegs;

/* object structure */
/* This structure can describe an AIM6020 (or AIM6021) board. */
typedef struct {
	Uint32	allocated;
	Uint32 baseAddr;
	AIM6020_RecordedRegs recordedRegs;
	Uint32	typeOfBoard;		/* 0x6020 or 0x6021 (hex number) */
	Uint8 *	pDisplayTable;		/* pointer to the display table */
} AIM6020_Obj, *AIM6020_Handle;


/******************************************************************************\
* global variable declarations
\******************************************************************************/
BSLAPI AIM6020_Obj _AIM6020_obj[4];


/******************************************************************************\
* global function declarations
\******************************************************************************/

/*
The table below shows which gain and offset values in the digital potmeters
mean what.

for ADC1 and ADC2					Filter ON				Filter OFF
-------------------------------------------------------------------------------
offset goes from - to +:		value 63 ---> 0			value 0 ---> 63
gain goes from min to max:		value 63 ---> 0			value 0 ---> 63

for DAC1 and DAC2					Filter ON
----------------------------------------------------
offset goes from - to +:		value 63 ---> 0
gain goes from min to max:		value  0 ---> 63
*/

//==============================================================================
//	Sets the new position of the selected digital potentiometer.
//
//	After the call of this function, the physical state of the potentiometer is
//	modified in about 1 ms. You can call the AIM6020_waitForDigPotPos() function
//	to wait proper long time for physical setting of the new position.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		potID			potentiometer identifier
//						constants from AIM6020_DIGPOT_POTID_x
//		position		the new position (valid: 0 to 63)
//
//	returns:
//		0	= failed
//		1	= successful
//==============================================================================
BSLAPI Uint32 AIM6020_setDigPotPosition(AIM6020_Handle hAIMboard, Uint32 potID,
	Uint32 position);

//==============================================================================
//	Reads and returns the current position of the selected digital
//	potentiometer.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		potID			potentiometer identifier
//						constants from AIM6020_DIGPOT_POTID_x
//
//	returns:
//		The position of the selected potentiometer, or error.
//		0 to 63		= valid position, all right
//		0xFFFFFFFF	= failed, dig. pot. position unknown
//==============================================================================
BSLAPI Uint32 AIM6020_getDigPotPosition(AIM6020_Handle hAIMboard, Uint32 potID);

//==============================================================================
//	Saves the current position of the selected digital potentiometer to the
//	given non-volatile memory cell.
//
//	The rutine is waiting for the end of internal save method, then it returns.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		potID			potentiometer identifier
//						constants from AIM6020_DIGPOT_POTID_x
//		memory			non-voltaile memory cell identifier
//						constants from AIM6020_DIGPOT_MEM_x
//
//	returns:
//		0	= failed
//		1	= successful
//==============================================================================
BSLAPI Uint32 AIM6020_saveDigPotPosition(AIM6020_Handle hAIMboard, Uint32 potID,
	Uint32 memory);

//==============================================================================
//	Loads the stored position of the selected digital potentiometer from the
//	given non-volatile memory cell.
//
//	After the call of this function, the physical state of the potentiometer is
//	modified in about 2 ms. You can call the AIM6020_waitForDigPotLoad()
//	function to wait proper long time for physical setting of the new position.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		potID			potentiometer identifier
//						constants from AIM6020_DIGPOT_POTID_x
//		memory			non-voltaile memory cell identifier
//						constants from AIM6020_DIGPOT_MEM_x
//
//	returns:
//		0	= failed
//		1	= successful
//==============================================================================
BSLAPI Uint32 AIM6020_loadDigPotPosition(AIM6020_Handle hAIMboard, Uint32 potID,
	Uint32 memory);

//==============================================================================
//	Saves the current positions of all the four digital potentiometers of the
//	selected digital potentiometer device to the given non-volatile memory
//	cells.
//
//	The rutine is waiting for the end of internal save method, then it returns.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		devID			potentiometer device identifier
//						constants from AIM6020_DIGPOT_DEVID_x
//		memory			non-voltaile memory cell identifier
//						constants from AIM6020_DIGPOT_MEM_x
//
//	returns:
//		0	= failed
//		1	= successful
//==============================================================================
BSLAPI Uint32 AIM6020_saveAllDigPotPositions(AIM6020_Handle hAIMboard,
	Uint32 devID, Uint32 memory);

//==============================================================================
//	Loads the stored positions of all the four digital potentiometers of the
//	selected digital potentiometer device from the given non-volatile memory
//	cells.
//
//	After the call of this function, the physical state of the potentiometer is
//	modified in about 2 ms. You can call the AIM6020_waitForDigPotLoad()
//	function to wait proper long time for physical setting of the new position.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		devID			potentiometer device identifier
//						constants from AIM6020_DIGPOT_DEVID_x
//		memory			non-voltaile memory cell identifier
//						constants from AIM6020_DIGPOT_MEM_x
//
//	returns:
//		0	= failed
//		1	= successful
//==============================================================================
BSLAPI Uint32 AIM6020_loadAllDigPotPositions(AIM6020_Handle hAIMboard,
	Uint32 devID, Uint32 memory);

//==============================================================================
//	Resets the selected AIM board.
//
//	Config reg.:	each ADC start and DAC update signal sources are set to
//					software sources
//	FltClk1 reg.:	maximum cut freq. for filter clock 1
//	FltClk2 reg.:	maximum cut freq. for filter clock 2
//	FltCfg reg.:	filters of both analog input channels and both analog output
//					channels get filter clock 1, and at both analog input
//					channels, the filters are off
//	IntrCfg reg.:	both PINT6 and PINT7 sources are disabled
//
//	On the display, there is a symbol 0.
//	Both DAC are written by 0 digital values, then they are updated.
//
//	THE DIGITAL POTENTIOMETER POSITIONS ARE UNMODIFIED!
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//==============================================================================
BSLAPI void AIM6020_reset(AIM6020_Handle hAIMboard);

//==============================================================================
//	Resets all the AIM boards what have been installed in the system currently.
//
//	See the description of the AIM6020_reset() to get information about the
//	default settings.
//==============================================================================
BSLAPI void AIM6020_resetAll();

//==============================================================================
//	Attempts to open the selected AIM board.
//
//	parameters:
//		boardID			board ID of the AIM board to be opened
//						valid: 0 to 3 for direct seelction
//						valid: -1 for find the first available (installed, but
//							not opened yed) board
//						constants from AIM6020_BOARDx
//		flags			0 = do nothing after a successful open
//						AIM6020_OPEN_RESET = reset the opened board
//
//	returns:
//		handle for the opened AIM board, or error
//		If the return value is INV, the board either does not exist in the
//		system or it has already been opened.
//		If the return value is not INV, it is a valid handle.
//==============================================================================
BSLAPI AIM6020_Handle AIM6020_open(int boardID, Uint32 flags);

//==============================================================================
//	Closes the given opened AIM board.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		flags			0 = do nothing before close
//						AIM6020_CLOSE_RESET = reset the board before close
//==============================================================================
BSLAPI void AIM6020_close(AIM6020_Handle hAIMboard, Uint32 flags);


/******************************************************************************\
* inline function declarations
\******************************************************************************/

//==============================================================================
//	Writes the Configuration register to set the start and update signal sources
//	of both ADCs and DACs.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		startSignalAD1	start signal source for analog-to-digital converter 1
//						constants from AIM6020_START_SIGNAL_x
//		startSignalAD2	start signal source for analog-to-digital converter 2
//						constants from AIM6020_START_SIGNAL_x
//		updateSignalDA1	update signal source for digital-to-analog converter 1
//						constants from AIM6020_UPDATE_SIGNAL_x
//		updateSignalDA2	update signal source for digital-to-analog converter 2
//						constants from AIM6020_UPDATE_SIGNAL_x
//==============================================================================
IDECL void AIM6020_setConfiguration(AIM6020_Handle hAIMboard,
	Uint32 startSignalAD1, Uint32 startSignalAD2,
	Uint32 updateSignalDA1, Uint32 updateSignalDA2);

//==============================================================================
//	Writes the Configuration register to set the start signal source of the
//	analog-to-digital converter 1.
//
//	The other settings of the configuration register are unmodified.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		startSignal		start signal source for analog-to-digital converter 1
//						constants from AIM6020_START_SIGNAL_x
//==============================================================================
IDECL void AIM6020_setAD1StartSignal(AIM6020_Handle hAIMboard, Uint32 startSignal);

//==============================================================================
//	Writes the Configuration register to set the start signal source of the
//	analog-to-digital converter 2.
//
//	The other settings of the configuration register are unmodified.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		startSignal		start signal source for analog-to-digital converter 2
//						constants from AIM6020_START_SIGNAL_x
//==============================================================================
IDECL void AIM6020_setAD2StartSignal(AIM6020_Handle hAIMboard, Uint32 startSignal);

//==============================================================================
//	Writes the Configuration register to set the update signal source of the
//	digital-to-analog converter 1.
//
//	The other settings of the configuration register are unmodified.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		updateSignal	update signal source for digital-to-analog converter 1
//						constants from AIM6020_UPDATE_SIGNAL_x
//==============================================================================
IDECL void AIM6020_setDA1UpdateSignal(AIM6020_Handle hAIMboard, Uint32 updateSignal);

//==============================================================================
//	Writes the Configuration register to set the update signal source of the
//	digital-to-analog converter 2.
//
//	The other settings of the configuration register are unmodified.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		updateSignal	update signal source for digital-to-analog converter 2
//						constants from AIM6020_UPDATE_SIGNAL_x
//==============================================================================
IDECL void AIM6020_setDA2UpdateSignal(AIM6020_Handle hAIMboard, Uint32 updateSignal);

//==============================================================================
//	Returns the previously set source of the start signal of the
//	analog-to-digital converter 1.
//
//	IT DOES NOT READ PHYSICALLY THE CONFIGURATION REGISTER OF THE SELECTED AIM
//	BOARD!
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		source of the start signal of the analog-to-digital converter 1
//		compare it to the constants of AIM6020_START_SIGNAL_x
//==============================================================================
IDECL Uint32 AIM6020_getAD1StartSignal(AIM6020_Handle hAIMboard);

//==============================================================================
//	Returns the previously set source of the start signal of the
//	analog-to-digital converter 2.
//
//	IT DOES NOT READ PHYSICALLY THE CONFIGURATION REGISTER OF THE SELECTED AIM
//	BOARD!
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		source of the start signal of the analog-to-digital converter 2
//		compare it to the constants of AIM6020_START_SIGNAL_x
//==============================================================================
IDECL Uint32 AIM6020_getAD2StartSignal(AIM6020_Handle hAIMboard);

//==============================================================================
//	Returns the previously set source of the update signal of the
//	digital-to-analog converter 1.
//
//	IT DOES NOT READ PHYSICALLY THE CONFIGURATION REGISTER OF THE SELECTED AIM
//	BOARD!
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		source of the update signal of the digital-to-analog converter 1
//		compare it to the constants of AIM6020_UPDATE_SIGNAL_x
//==============================================================================
IDECL Uint32 AIM6020_getDA1UpdateSignal(AIM6020_Handle hAIMboard);

//==============================================================================
//	Returns the previously set source of the update signal of the
//	digital-to-analog converter 2.
//
//	IT DOES NOT READ PHYSICALLY THE CONFIGURATION REGISTER OF THE SELECTED AIM
//	BOARD!
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		source of the update signal of the digital-to-analog converter 2
//		compare it to the constants of AIM6020_UPDATE_SIGNAL_x
//==============================================================================
IDECL Uint32 AIM6020_getDA2UpdateSignal(AIM6020_Handle hAIMboard);

//==============================================================================
//	Returns the result of the last conversation of the selected
//	analog-to-digital converter.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		chNum			select the analog-to-digital converter to be queried
//						constants from AIM6020_CHANNEL1 and AIM6020_CHANNEL2
//
//	returns:
//		The result of the last conversation of the selected analog-to-digital
//		converter in signed 16-bit integer format. The 12 bits of the result of
//		the conversation are shifted up by 4 bits and the lower 4 bits are
//		filled up by zeros.
//==============================================================================
IDECL Int16 AIM6020_readAD(AIM6020_Handle hAIMboard, Uint32 chNum);

//==============================================================================
//	Returns the result of the last conversation of the analog-to-digital
//	converter 1.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		The result of the last conversation of the analog-to-digital converter 1
//		in signed 16-bit integer format. The 12 bits of the result of the
//		conversation are shifted up by 4 bits and the lower 4 bits are filled up
//		by zeros.
//==============================================================================
IDECL Int16 AIM6020_readAD1(AIM6020_Handle hAIMboard);

//==============================================================================
//	Returns the result of the last conversation of the analog-to-digital
//	converter 2.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		The result of the last conversation of the analog-to-digital converter 2
//		in signed 16-bit integer format. The 12 bits of the result of the
//		conversation are shifted up by 4 bits and the lower 4 bits are filled up
//		by zeros.
//==============================================================================
IDECL Int16 AIM6020_readAD2(AIM6020_Handle hAIMboard);

//==============================================================================
//	Writes the selected digital-to-analog converter with the given value for the
//	next update.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		chNum			select the digital-to-analog converter to be accessed
//						constants from AIM6020_CHANNEL1 and AIM6020_CHANNEL2
//		data			the new digital value in signed 16-bit integer format
//						The upper 12 bits are valid only. The lower 4 bits lose.
//==============================================================================
IDECL void AIM6020_writeDA(AIM6020_Handle hAIMboard, Uint32 chNum, Int16 data);

//==============================================================================
//	Writes the digital-to-analog converter 1 with the given value for the next
//	update.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		data			the new digital value in signed 16-bit integer format
//						The upper 12 bits are valid only. The lower 4 bits lose.
//==============================================================================
IDECL void AIM6020_writeDA1(AIM6020_Handle hAIMboard, Int16 data);

//==============================================================================
//	Writes the digital-to-analog converter 2 with the given value for the next
//	update.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		data			the new digital value in signed 16-bit integer format
//						The upper 12 bits are valid only. The lower 4 bits lose.
//==============================================================================
IDECL void AIM6020_writeDA2(AIM6020_Handle hAIMboard, Int16 data);

//==============================================================================
//	Sends a start conversion signal to the selected analog-to-digital converter,
//	if the source of the start signal of the selected analog-to-digital
//	converter has been set to software source (AIM6020_START_SIGNAL_SOFTWARE).
//
//	If the source of the start signal of the selected analog-to-digital
//	converter has been set to not software source, the call of the function has
//	no effect.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		chNum			select the analog-to-digital converter to be accessed
//						constants from AIM6020_CHANNEL1 and AIM6020_CHANNEL2
//==============================================================================
IDECL void AIM6020_startAD(AIM6020_Handle hAIMboard, Uint32 chNum);

//==============================================================================
//	Sends a start conversion signal to the analog-to-digital converter 1, if the
//	source of the start signal of the analog-to-digital converter 1 has been set
//	to software source (AIM6020_START_SIGNAL_SOFTWARE).
//
//	If the source of the start signal of the analog-to-digital converter 1 has
//	been set to not software source, the call of the function has no effect.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//==============================================================================
IDECL void AIM6020_startAD1(AIM6020_Handle hAIMboard);

//==============================================================================
//	Sends a start conversion signal to the analog-to-digital converter 2, if the
//	source of the start signal of the analog-to-digital converter 2 has been set
//	to software source (AIM6020_START_SIGNAL_SOFTWARE).
//
//	If the source of the start signal of the analog-to-digital converter 2 has
//	been set to not software source, the call of the function has no effect.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//==============================================================================
IDECL void AIM6020_startAD2(AIM6020_Handle hAIMboard);

//==============================================================================
//	Sends start conversion signals to both analog-to-digital converters at a
//	time. This operation takes the time of a single register access only. This
//	function call is twice faster than AIM6020_startAD1() and AIM6020_startAD2()
//	together.
//
//	The call of the function has no effect at the analog-to-digital converter
//	whose start signal source has been set to not software source
//	(AIM6020_START_SIGNAL_SOFTWARE).
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//==============================================================================
IDECL void AIM6020_startAD12(AIM6020_Handle hAIMboard);

//==============================================================================
//	Sends an update signal to the selected digital-to-analog converter, if the
//	source of the update signal of the selected digital-to-analog converter has
//	been set to software source (AIM6020_UPDATE_SIGNAL_SOFTWARE).
//
//	If the source of the update signal of the selected digital-to-analog
//	converter has been set to not software source, the call of the function has
//	no effect.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		chNum			select the digital-to-analog converter to be accessed
//						constants from AIM6020_CHANNEL1 and AIM6020_CHANNEL2
//==============================================================================
IDECL void AIM6020_updateDA(AIM6020_Handle hAIMboard, Uint32 chNum);

//==============================================================================
//	Sends an update signal to the digital-to-analog converter 1, if the source
//	of the update signal of the digital-to-analog converter 1 has been set to
//	software source (AIM6020_UPDATE_SIGNAL_SOFTWARE).
//
//	If the source of the update signal of the digital-to-analog converter 1 has
//	been set to not software source, the call of the function has no effect.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//==============================================================================
IDECL void AIM6020_updateDA1(AIM6020_Handle hAIMboard);

//==============================================================================
//	Sends an update signal to the digital-to-analog converter 2, if the source
//	of the update signal of the digital-to-analog converter 2 has been set to
//	software source (AIM6020_UPDATE_SIGNAL_SOFTWARE).
//
//	If the source of the update signal of the digital-to-analog converter 2 has
//	been set to not software source, the call of the function has no effect.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//==============================================================================
IDECL void AIM6020_updateDA2(AIM6020_Handle hAIMboard);

//==============================================================================
//	Sends update signals to both digital-to-analog converters at a time. This
//	operation takes the time of a single register access only. This function
//	call is twice faster than AIM6020_updateDA1() and AIM6020_updateDA2()
//	together.
//
//	The call of the function has no effect at the digital-to-analog converter
//	whose update signal source has been set to not software source
//	(AIM6020_UPDATE_SIGNAL_SOFTWARE).
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//==============================================================================
IDECL void AIM6020_updateDA12(AIM6020_Handle hAIMboard);

//==============================================================================
//	Sets the clock division of the selected filter clock.
//
//	The base frequency of each filter clock (oscillator) is equal to the speed
//	of the SPM6020 board's DSP processor in MHz divided by 12800. The base
//	frequency means the maximum cutoff frequency what can be set.
//		at 200 MHz DSP processor, the base frequency is 15625 Hz
//		at 233 MHz DSP processor, the base frequency is 18203 Hz
//		at 250 MHz DSP processor, the base frequency is 19531 Hz
//
//	Cutoff frequency = DSP processor speed in MHz / 12800 / (Clock division + 1)
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		fltClkNum		select the filter clock (oscillator) to be set
//						constants from AIM6020_FILTER_CLOCKx
//		division		the new clock division number
//						valid: 0 to 255
//==============================================================================
IDECL void AIM6020_setFilterClock(AIM6020_Handle hAIMboard, Uint32 fltClkNum,
	Uint32 division);

//==============================================================================
//	Sets the clock division of the filter clock 1.
//
//	See details, at the description of the AIM6020_setFilterClock().
//	Cutoff frequency = DSP processor speed in MHz / 12800 / (Clock division + 1)
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		division		the new clock division number
//						valid: 0 to 255
//==============================================================================
IDECL void AIM6020_setFilterClock1(AIM6020_Handle hAIMboard, Uint32 division);

//==============================================================================
//	Sets the clock division of the filter clock 2
//
//	See details, at the description of the AIM6020_setFilterClock().
//	Cutoff frequency = DSP processor speed in MHz / 12800 / (Clock division + 1)
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		division		the new clock division number
//						valid: 0 to 255
//==============================================================================
IDECL void AIM6020_setFilterClock2(AIM6020_Handle hAIMboard, Uint32 division);

//==============================================================================
//	Returns the previously set clock division of the selected filter clock.
//
//	Cutoff frequency = DSP processor speed in MHz / 12800 / (Clock division + 1)
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		fltClkNum		select the filter clock (oscillator) to be set
//						constants from AIM6020_FILTER_CLOCKx
//
//	returns:
//		The division of the selected filter clock.
//		The returned value is between 0 to 255. The call of this function is
//		always successful.
//==============================================================================
IDECL Uint32 AIM6020_getFilterClock(AIM6020_Handle hAIMboard, Uint32 fltClkNum);

//==============================================================================
//	Returns the previously set clock division of the filter clock 1.
//
//	Cutoff frequency = DSP processor speed in MHz / 12800 / (Clock division + 1)
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		The division of the filter clock 1.
//		The returned value is between 0 to 255. The call of this function is
//		always successful.
//==============================================================================
IDECL Uint32 AIM6020_getFilterClock1(AIM6020_Handle hAIMboard);

//==============================================================================
//	Returns the previously set clock division of the filter clock 2.
//
//	Cutoff frequency = DSP processor speed in MHz / 12800 / (Clock division + 1)
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		The division of the filter clock 2.
//		The returned value is between 0 to 255. The call of this function is
//		always successful.
//==============================================================================
IDECL Uint32 AIM6020_getFilterClock2(AIM6020_Handle hAIMboard);

//==============================================================================
//	Writes the Filter Configuration register to set filter clock assignments for
//	both analog-to-digital and both digital-to-analog converters, and to enable
//	or disable the filter on the analog input channels.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		filtClkSelAD1	select the used filter clock for the analog-to-digital
//						converter 1
//						constants from AIM6020_FILTER_CLOCKx
//		filtClkSelAD2	select the used filter clock for the analog-to-digital
//						converter 2
//						constants from AIM6020_FILTER_CLOCKx
//		filtClkSelDA1	select the used filter clock for the digital-to-analog
//						converter 1
//						constants from AIM6020_FILTER_CLOCKx
//		filtClkSelDA2	select the used filter clock for the digital-to-analog
//						converter 2
//						constants from AIM6020_FILTER_CLOCKx
//		filtEnAD1		enable or disable the usage of the filter on the analog
//						input channel 1
//						constants from AIM6020_FILTER_OFF, AIM6020_FILTER_ON,
//						AIM6020_FILTER_DISABLED, AIM6020_FILTER_ENABLED
//		filtEnAD2		enable or disable the usage of the filter on the analog
//						input channel 2
//						constants from AIM6020_FILTER_OFF, AIM6020_FILTER_ON,
//						AIM6020_FILTER_DISABLED, AIM6020_FILTER_ENABLED
//==============================================================================
IDECL void AIM6020_setFilterConfiguration(AIM6020_Handle hAIMboard,
	Uint32 filtClkSelAD1, Uint32 filtClkSelAD2,
	Uint32 filtClkSelDA1, Uint32 filtClkSelDA2,
	Uint32 filtEnAD1, Uint32 filtEnAD2);

//==============================================================================
//	Writes the Filter Configuration register to set filter clock assignment for
//	the analog-to-digital converters 1.
//
//	The other settings of the Filter Configuration register are unmodified.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		filtClkSel		select the used filter clock for the analog-to-digital
//						converter 1
//						constants from AIM6020_FILTER_CLOCKx
//==============================================================================
IDECL void AIM6020_setAD1FilterClock(AIM6020_Handle hAIMboard, Uint32 filtClkSel);

//==============================================================================
//	Writes the Filter Configuration register to set filter clock assignment for
//	the analog-to-digital converters 2.
//
//	The other settings of the Filter Configuration register are unmodified.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		filtClkSel		select the used filter clock for the analog-to-digital
//						converter 2
//						constants from AIM6020_FILTER_CLOCKx
//==============================================================================
IDECL void AIM6020_setAD2FilterClock(AIM6020_Handle hAIMboard, Uint32 filtClkSel);

//==============================================================================
//	Writes the Filter Configuration register to set filter clock assignment for
//	the digital-to-analog converters 1.
//
//	The other settings of the Filter Configuration register are unmodified.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		filtClkSel		select the used filter clock for the digital-to-analog
//						converter 1
//						constants from AIM6020_FILTER_CLOCKx
//==============================================================================
IDECL void AIM6020_setDA1FilterClock(AIM6020_Handle hAIMboard, Uint32 filtClkSel);

//==============================================================================
//	Writes the Filter Configuration register to set filter clock assignment for
//	the digital-to-analog converters 2.
//
//	The other settings of the Filter Configuration register are unmodified.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		filtClkSel		select the used filter clock for the digital-to-analog
//						converter 2
//						constants from AIM6020_FILTER_CLOCKx
//==============================================================================
IDECL void AIM6020_setDA2FilterClock(AIM6020_Handle hAIMboard, Uint32 filtClkSel);

//==============================================================================
//	Writes the Filter Configuration register to enable or disable the filter
//	on the analog input channel 1.
//
//	The other settings of the Filter Configuration register are unmodified.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		bEnable			enable or disable the usage of the filter on the analog
//						input channel 1
//						constants from AIM6020_FILTER_OFF, AIM6020_FILTER_ON,
//						AIM6020_FILTER_DISABLED, AIM6020_FILTER_ENABLED
//==============================================================================
IDECL void AIM6020_enableAD1Filter(AIM6020_Handle hAIMboard, Uint32 bEnable);

//==============================================================================
//	Writes the Filter Configuration register to enable or disable the filter
//	on the analog input channel 2.
//
//	The other settings of the Filter Configuration register are unmodified.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		bEnable			enable or disable the usage of the filter on the analog
//						input channel 2
//						constants from AIM6020_FILTER_OFF, AIM6020_FILTER_ON,
//						AIM6020_FILTER_DISABLED, AIM6020_FILTER_ENABLED
//==============================================================================
IDECL void AIM6020_enableAD2Filter(AIM6020_Handle hAIMboard, Uint32 bEnable);

//==============================================================================
//	Returns the previously set filter clock assignment of the analog-to-digital
//	converters 1.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		The selected filter clock of the analog-to-digital converter 1.
//		constants from AIM6020_FILTER_CLOCKx
//==============================================================================
IDECL Uint32 AIM6020_getAD1FilterClock(AIM6020_Handle hAIMboard);

//==============================================================================
//	Returns the previously set filter clock assignment of the analog-to-digital
//	converters 2.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		The selected filter clock of the analog-to-digital converter 2.
//		constants from AIM6020_FILTER_CLOCKx
//==============================================================================
IDECL Uint32 AIM6020_getAD2FilterClock(AIM6020_Handle hAIMboard);

//==============================================================================
//	Returns the previously set filter clock assignment of the digital-to-analog
//	converters 1.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		The selected filter clock of the digital-to-analog converter 1.
//		constants from AIM6020_FILTER_CLOCKx
//==============================================================================
IDECL Uint32 AIM6020_getDA1FilterClock(AIM6020_Handle hAIMboard);

//==============================================================================
//	Returns the previously set filter clock assignment of the digital-to-analog
//	converters 1.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		The selected filter clock of the digital-to-analog converter 1.
//		constants from AIM6020_FILTER_CLOCKx
//==============================================================================
IDECL Uint32 AIM6020_getDA2FilterClock(AIM6020_Handle hAIMboard);

//==============================================================================
//	Returns the previously set filter enable of the analog-to-digital
//	converters 1.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		Returns whether or not the filter in enable for the analog input
//		channel 1.
//		constants from AIM6020_FILTER_DISABLED, AIM6020_FILTER_ENABLED,
//		AIM6020_FILTER_OFF, AIM6020_FILTER_ON
//==============================================================================
IDECL Uint32 AIM6020_getAD1FilterEnable(AIM6020_Handle hAIMboard);

//==============================================================================
//	Returns the previously set filter enable of the analog-to-digital
//	converters 2.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		Returns whether or not the filter in enable for the analog input
//		channel 2.
//		constants from AIM6020_FILTER_DISABLED, AIM6020_FILTER_ENABLED,
//		AIM6020_FILTER_OFF, AIM6020_FILTER_ON
//==============================================================================
IDECL Uint32 AIM6020_getAD2FilterEnable(AIM6020_Handle hAIMboard);

//==============================================================================
//	Writes the Interrupt Configuration register to set sources of the PINT6 and
//	PINT7 signals.
//
//	parameters:
//		hAIMboard			handle from the AIM6020_open()
//		sourceSelectPINT6	select the source of the PINT6 signal
//							This signal is connected to the ExtInt6 External
//							Interrupt 6 input of the DSP processor.
//							constants from AIM6020_IT_SOURCE_x
//		sourceSelectPINT7	select the source of the PINT7 signal
//							This signal can be connected to the ExtInt7 External
//							Interrupt 7 input of the DSP processor.
//							constants from AIM6020_IT_SOURCE_x
//==============================================================================
IDECL void AIM6020_setInterruptConfiguration(AIM6020_Handle hAIMboard,
	Uint32 sourceSelectPINT6, Uint32 sourceSelectPINT7);

//==============================================================================
//	Writes the Interrupt Configuration register to set source of the PINT6
//	signal.
//
//	The other settings of the Interrupt Configuration register are unmodified.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		sourceSelect	select the source of the PINT6 signal
//						This signal is connected to the ExtInt6 External
//						Interrupt 6 input of the DSP processor.
//						constants from AIM6020_IT_SOURCE_x
//==============================================================================
IDECL void AIM6020_setPINT6Source(AIM6020_Handle hAIMboard, Uint32 sourceSelect);

//==============================================================================
//	Writes the Interrupt Configuration register to set source of the PINT7
//	signal.
//
//	The other settings of the Interrupt Configuration register are unmodified.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		sourceSelect	select the source of the PINT7 signal
//						This signal can be connected to the ExtInt7 External
//						Interrupt 7 input of the DSP processor.
//						constants from AIM6020_IT_SOURCE_x
//==============================================================================
IDECL void AIM6020_setPINT7Source(AIM6020_Handle hAIMboard, Uint32 sourceSelect);

//==============================================================================
//	Returns the previously set source of the PINT6 signal.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		The previously set source of the PINT6 signal.
//		constants from AIM6020_IT_SOURCE_x
//==============================================================================
IDECL Uint32 AIM6020_getPINT6Source(AIM6020_Handle hAIMboard);

//==============================================================================
//	Returns the previously set source of the PINT7 signal.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		The previously set source of the PINT7 signal.
//		constants from AIM6020_IT_SOURCE_x
//==============================================================================
IDECL Uint32 AIM6020_getPINT7Source(AIM6020_Handle hAIMboard);

//==============================================================================
//	Writes the Serial Data register.
//
//	After the call of this function, if there is serial transmit clock and
//	transmit frame sync signal on the AIM6020 board's serial connector, the
//	written data will be sent through the serial port.
//
//	Look out! There is only a single 16-bit register for both write/read.
//	You can overwrite the last received word by a Write command.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		data			16-bit value to be written in the Output Serial Data
//						register
//						The lower 16 bits are valid.
//==============================================================================
IDECL void AIM6020_writeSerialData(AIM6020_Handle hAIMboard, Uint32 data);

//==============================================================================
//	Reads the Serial Data register.
//
//	If there has been serial receive clock and receive frame sync signal on the
//	AIM6020 board's serial connector, the AIM6020 has received a serial data and
//	has stored it in the Serial Data register. After this, you can read the
//	received serial data by a call of this function.
//
//	Look out! There is only a single 16-bit register for both write/read.
//	You can overwrite the last received word by a Write command.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		Content of the Input Serial Data register. The lower 16 bits are valid
//		only.
//==============================================================================
IDECL Uint32 AIM6020_readSerialData(AIM6020_Handle hAIMboard);

//==============================================================================
//	Writes the Display register of the selected AIM board to modify the digit
//	symbol and the state of the dot on the display.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		digitAndDot		digit symbol and dot enable
//						The lower 4 bits determine the hexa digit to be shown on
//						the display. If the 5th bit is set, the dot is light on
//						the display. Otherwise, the dot is dark.
//==============================================================================
IDECL void AIM6020_setDisplay(AIM6020_Handle hAIMboard, Uint32 digitAndDot);

//==============================================================================
//	Writes the Display register of the selected AIM board to modify only the
//	digit symbol on the display.
//
//	The state of the dot is unmodified on the display.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		digit			the new digit symbol
//						The lower 4 bits determine the hexa digit to be shown on
//						the display.
//==============================================================================
IDECL void AIM6020_setDisplayDigit(AIM6020_Handle hAIMboard, Uint32 digit);

//==============================================================================
//	Writes the Display register of the selected AIM board to enable the dot to
//	turn on on the display.
//
//	The hexa digit is unmodified on the display.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		bDotEnable		enable that the dot be light on the display
//						0	= disable (the dot will be dark)
//						1	= enable (the dot will be light)
//==============================================================================
IDECL void AIM6020_setDisplayDot(AIM6020_Handle hAIMboard, Uint32 bDotEnable);

//==============================================================================
//	Returns the previously written value of the Display register of the selected
//	AIM board.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		the current digit symbol and the current state of the dot
//		The lower 4 bits determine the hexa digit that is shown on the display.
//		If the 5th bit is set, the dot is light on the display. Otherwise, the
//		dot is dark.
//==============================================================================
IDECL Uint32 AIM6020_getDisplay(AIM6020_Handle hAIMboard);

//==============================================================================
//	Returns the previously set hexa digit symbol for the display.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		the current digit symbol
//		The lower 4 bits determine the hexa digit that is shown on the display.
//==============================================================================
IDECL Uint32 AIM6020_getDisplayDigit(AIM6020_Handle hAIMboard);

//==============================================================================
//	Returns the previously set state of the dot for the display.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		the current state of the dot
//		Look out! Not the 5th bit determines the state of the dot.
//		0	= dot is disabled (dark)
//		1	= dot is enabled (light)
//==============================================================================
IDECL Uint32 AIM6020_getDisplayDot(AIM6020_Handle hAIMboard);

//==============================================================================
//	Writes the I2C Data register of the selected AIM board to modify the voltage
//	level on the data wire of the I2C bus of the selected AIM board.
//
//	There is an I2C communication bus for digital potentiometers of both
//	analog-to-digital and both digital-to-analog converters.
//	The data wire of the I2C bus is driven by an open collector output with a
//	pull-up resistor.
//	The call of this function has no effect, if the I2C data output is disabled.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		bit				bit value to be set on the data wire of the I2C bus
//						0	= low voltage level (0 V)
//						1	= high voltage level (3.3 V)
//==============================================================================
IDECL void AIM6020_writeI2CData(AIM6020_Handle hAIMboard, Uint32 bit);

//==============================================================================
//	Writes the I2C Data register of the selected AIM board to set the voltage
//	level to high on the data wire of the I2C bus of the selected AIM board.
//
//	See details at the description of the AIM6020_writeI2CData().
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//==============================================================================
IDECL void AIM6020_setI2CDataHigh(AIM6020_Handle hAIMboard);

//==============================================================================
//	Writes the I2C Data register of the selected AIM board to set the voltage
//	level to low on the data wire of the I2C bus of the selected AIM board.
//
//	See details at the description of the AIM6020_writeI2CData().
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//==============================================================================
IDECL void AIM6020_setI2CDataLow(AIM6020_Handle hAIMboard);

//==============================================================================
//	Reads and returns the current logical level of the data wire of the I2C bus
//	of the selected AIM board.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		the current logical level of the data wire of the I2C bus
//		0	= there is a low voltage level on the data wire
//		1	= there is a high voltage level on the data wire
//==============================================================================
IDECL Uint32 AIM6020_readI2CData(AIM6020_Handle hAIMboard);

//==============================================================================
//	Writes the I2C Clock register of the selected AIM board to modify the
//	voltage level on the clock wire of the I2C bus of the selected AIM board.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		bit				bit value to be set on the clock wire of the I2C bus
//						0	= low voltage level (0 V)
//						1	= high voltage level (3.3 V)
//==============================================================================
IDECL void AIM6020_writeI2CClock(AIM6020_Handle hAIMboard, Uint32 bit);

//==============================================================================
//	Writes the I2C Clock register of the selected AIM board to set the voltage
//	level to high on the clock wire of the I2C bus of the selected AIM board.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//==============================================================================
IDECL void AIM6020_setI2CClockHigh(AIM6020_Handle hAIMboard);

//==============================================================================
//	Writes the I2C Clock register of the selected AIM board to set the voltage
//	level to low on the clock wire of the I2C bus of the selected AIM board.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//==============================================================================
IDECL void AIM6020_setI2CClockLow(AIM6020_Handle hAIMboard);

//==============================================================================
//	Writes the I2C Data Output Enable register of the selected AIM board to 
//	enable the output drive on the data wire of the I2C bus of the selected AIM
//	board.
//
//	See details at the description of the AIM6020_writeI2CData().
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//==============================================================================
IDECL void AIM6020_enableI2CDataOut(AIM6020_Handle hAIMboard);

//==============================================================================
//	Writes the I2C Data Output Enable register of the selected AIM board to 
//	disable the output drive on the data wire of the I2C bus of the selected AIM
//	board.
//
//	See details at the description of the AIM6020_writeI2CData().
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//==============================================================================
IDECL void AIM6020_disableI2CDataOut(AIM6020_Handle hAIMboard);

//==============================================================================
//	Provides a more or less accurate waiting loop in microsec steps.
//
//	There is a 'for' cycle. At the calculation of the length of the 'for' cycle,
//	the speed of the DSP processor is considered. The loop can be interrupted.
//	The interrupt time does not decrease the loop counter of the 'for' cycle.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//==============================================================================
IDECL void AIM6020_delay_us();

//==============================================================================
//	Provides a waiting loop to wait proper long time after the position setting
//	of a digital potentiometer by a call of the AIM6020_setDigPotPosition().
//
//	The waiting loop is based on the AIM6020_delay_us(). See details at the
//	description of the AIM6020_delay_us.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//==============================================================================
IDECL void AIM6020_waitForDigPotPos();

//==============================================================================
//	Provides a waiting loop to wait proper long time after the load of a digital
//	potentiometer setting by a call of the AIM6020_loadDigPotPosition() or the
//	AIM6020_loadAllDigPotPositions().
//
//	The waiting loop is based on the AIM6020_delay_us(). See details at the
//	description of the AIM6020_delay_us.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//==============================================================================
IDECL void AIM6020_waitForDigPotLoad();

//==============================================================================
//	Calls the AIM6020_setDigPotPosition() function to set a new offset value for
//	the analog-to-digital converter 1 (analog input channel 1).
//
//	See details at the description of the AIM6020_setDigPotPosition().
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		position		the new offset value (valid: 0 to 63)
//
//	returns:
//		The return value of the AIM6020_setDigPotPosition().
//		0	= failed
//		1	= successful
//==============================================================================
IDECL Uint32 AIM6020_setAD1offset(AIM6020_Handle hAIMboard, Uint32 position);

//==============================================================================
//	Calls the AIM6020_setDigPotPosition() function to set a new gain value for
//	the analog-to-digital converter 1 (analog input channel 1).
//
//	See details at the description of the AIM6020_setDigPotPosition().
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		position		the new gain value (valid: 0 to 63)
//
//	returns:
//		The return value of the AIM6020_setDigPotPosition().
//		0	= failed
//		1	= successful
//==============================================================================
IDECL Uint32 AIM6020_setAD1gain(AIM6020_Handle hAIMboard, Uint32 position);

//==============================================================================
//	Calls the AIM6020_setDigPotPosition() function to set a new offset value for
//	the analog-to-digital converter 2 (analog input channel 2).
//
//	See details at the description of the AIM6020_setDigPotPosition().
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		position		the new offset value (valid: 0 to 63)
//
//	returns:
//		The return value of the AIM6020_setDigPotPosition().
//		0	= failed
//		1	= successful
//==============================================================================
IDECL Uint32 AIM6020_setAD2offset(AIM6020_Handle hAIMboard, Uint32 position);

//==============================================================================
//	Calls the AIM6020_setDigPotPosition() function to set a new gain value for
//	the analog-to-digital converter 2 (analog input channel 2).
//
//	See details at the description of the AIM6020_setDigPotPosition().
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		position		the new gain value (valid: 0 to 63)
//
//	returns:
//		The return value of the AIM6020_setDigPotPosition().
//		0	= failed
//		1	= successful
//==============================================================================
IDECL Uint32 AIM6020_setAD2gain(AIM6020_Handle hAIMboard, Uint32 position);

//==============================================================================
//	Calls the AIM6020_setDigPotPosition() function to set a new offset value for
//	the digital-to-analog converter 1 (analog output channel 1).
//
//	See details at the description of the AIM6020_setDigPotPosition().
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		position		the new offset value (valid: 0 to 63)
//
//	returns:
//		The return value of the AIM6020_setDigPotPosition().
//		0	= failed
//		1	= successful
//==============================================================================
IDECL Uint32 AIM6020_setDA1offset(AIM6020_Handle hAIMboard, Uint32 position);

//==============================================================================
//	Calls the AIM6020_setDigPotPosition() function to set a new gain value for
//	the digital-to-analog converter 1 (analog output channel 1).
//
//	See details at the description of the AIM6020_setDigPotPosition().
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		position		the new gain value (valid: 0 to 63)
//
//	returns:
//		The return value of the AIM6020_setDigPotPosition().
//		0	= failed
//		1	= successful
//==============================================================================
IDECL Uint32 AIM6020_setDA1gain(AIM6020_Handle hAIMboard, Uint32 position);

//==============================================================================
//	Calls the AIM6020_setDigPotPosition() function to set a new offset value for
//	the digital-to-analog converter 2 (analog output channel 2).
//
//	See details at the description of the AIM6020_setDigPotPosition().
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		position		the new offset value (valid: 0 to 63)
//
//	returns:
//		The return value of the AIM6020_setDigPotPosition().
//		0	= failed
//		1	= successful
//==============================================================================
IDECL Uint32 AIM6020_setDA2offset(AIM6020_Handle hAIMboard, Uint32 position);

//==============================================================================
//	Calls the AIM6020_setDigPotPosition() function to set a new gain value for
//	the digital-to-analog converter 2 (analog output channel 2).
//
//	See details at the description of the AIM6020_setDigPotPosition().
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		position		the new gain value (valid: 0 to 63)
//
//	returns:
//		The return value of the AIM6020_setDigPotPosition().
//		0	= failed
//		1	= successful
//==============================================================================
IDECL Uint32 AIM6020_setDA2gain(AIM6020_Handle hAIMboard, Uint32 position);

//==============================================================================
//	Calls the AIM6020_getDigPotPosition() function to get a current offset value
//	of the analog-to-digital converter 1 (analog input channel 1).
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		The current offset value, or error.
//		0 to 63		= valid position, all right
//		0xFFFFFFFF	= failed, dig. pot. position unknown
//==============================================================================
IDECL Uint32 AIM6020_getAD1offset(AIM6020_Handle hAIMboard);

//==============================================================================
//	Calls the AIM6020_getDigPotPosition() function to get a current gain value
//	of the analog-to-digital converter 1 (analog input channel 1).
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		The current gain value, or error.
//		0 to 63		= valid position, all right
//		0xFFFFFFFF	= failed, dig. pot. position unknown
//==============================================================================
IDECL Uint32 AIM6020_getAD1gain(AIM6020_Handle hAIMboard);

//==============================================================================
//	Calls the AIM6020_getDigPotPosition() function to get a current offset value
//	of the analog-to-digital converter 2 (analog input channel 2).
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		The current offset value, or error.
//		0 to 63		= valid position, all right
//		0xFFFFFFFF	= failed, dig. pot. position unknown
//==============================================================================
IDECL Uint32 AIM6020_getAD2offset(AIM6020_Handle hAIMboard);

//==============================================================================
//	Calls the AIM6020_getDigPotPosition() function to get a current gain value
//	of the analog-to-digital converter 2 (analog input channel 2).
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		The current gain value, or error.
//		0 to 63		= valid position, all right
//		0xFFFFFFFF	= failed, dig. pot. position unknown
//==============================================================================
IDECL Uint32 AIM6020_getAD2gain(AIM6020_Handle hAIMboard);

//==============================================================================
//	Calls the AIM6020_getDigPotPosition() function to get a current offset value
//	of the digital-to-analog converter 1 (analog output channel 1).
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		The current offset value, or error.
//		0 to 63		= valid position, all right
//		0xFFFFFFFF	= failed, dig. pot. position unknown
//==============================================================================
IDECL Uint32 AIM6020_getDA1offset(AIM6020_Handle hAIMboard);

//==============================================================================
//	Calls the AIM6020_getDigPotPosition() function to get a current gain value
//	of the digital-to-analog converter 1 (analog output channel 1).
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		The current gain value, or error.
//		0 to 63		= valid position, all right
//		0xFFFFFFFF	= failed, dig. pot. position unknown
//==============================================================================
IDECL Uint32 AIM6020_getDA1gain(AIM6020_Handle hAIMboard);

//==============================================================================
//	Calls the AIM6020_getDigPotPosition() function to get a current offset value
//	of the digital-to-analog converter 2 (analog output channel 2).
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		The current offset value, or error.
//		0 to 63		= valid position, all right
//		0xFFFFFFFF	= failed, dig. pot. position unknown
//==============================================================================
IDECL Uint32 AIM6020_getDA2offset(AIM6020_Handle hAIMboard);

//==============================================================================
//	Calls the AIM6020_getDigPotPosition() function to get a current gain value
//	of the digital-to-analog converter 2 (analog output channel 2).
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//
//	returns:
//		The current gain value, or error.
//		0 to 63		= valid position, all right
//		0xFFFFFFFF	= failed, dig. pot. position unknown
//==============================================================================
IDECL Uint32 AIM6020_getDA2gain(AIM6020_Handle hAIMboard);

//==============================================================================
//	Writes all the configuration registers of the selected AIM board by a
//	AIM6020_Config type structure.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		config			AIM6020_Config type structure that contains the new
//						values for the registers
//==============================================================================
IDECL void AIM6020_config(AIM6020_Handle hAIMboard, AIM6020_Config *config);

//==============================================================================
//	Writes all the configuration registers of the selected AIM board. The new
//	register values are given in the parameters.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		config			new value for the Configuration register to set the
//						start and the update signal sources of both ADCs and
//						both DACs
//		fltclk1			new value for the Filter Clock 1 register to set the
//						division for the filter clock 1 (oscillator 1)
//		fltclk2			new value for the Filter Clock 2 register to set the
//						division for the filter clock 2 (oscillator 2)
//		fltcfg			new value for the Filter Configuration register to set
//						the filter clock assignments for both ADCs and both DACs
//						and to enable the analog input channel filters
//		intcfg			new value for the Interrupt Configuration register to set
//						the source of both PINT6 and PINT7 signals
//		display			direct value for the Display register to set the hexa
//						symbol and to enable dot
//						Look out! You can reach the display segments directly by
//						a call of this function.
//==============================================================================
IDECL void AIM6020_configArgs(AIM6020_Handle hAIMboard,
	Uint32 config, Uint32 fltclk1, Uint32 fltclk2, Uint32 fltcfg,
	Uint32 intcfg, Uint32 display);

//==============================================================================
//	Returns the previously set values of the Configuration registers of the
//	selected AIM board in a AIM6020_Config type structure.
//
//	parameters:
//		hAIMboard		handle from the AIM6020_open()
//		config			AIM6020_Config type structure to be filled up by the
//						current values of the registers
//==============================================================================
IDECL void AIM6020_getConfig(AIM6020_Handle hAIMboard, AIM6020_Config *config);


/******************************************************************************\
* inline function definitions
\******************************************************************************/
#ifdef USEDEFS

/*----------------------------------------------------------------------------*/
IDEF void AIM6020_setConfiguration(AIM6020_Handle hAIMboard,
	Uint32 startSignalAD1, Uint32 startSignalAD2,
	Uint32 updateSignalDA1, Uint32 updateSignalDA2)
{
	Uint32 gie;
	register Uint32 r0;
	
	r0 = (startSignalAD1 & 0x03) | ((startSignalAD2 & 0x03) << 2) |
		((updateSignalDA1 & 0x03) << 4) | ((updateSignalDA2 & 0x03) << 6);

	gie = IRQ_globalDisable();
	AIM6020_RSETH( hAIMboard, CONFIG, r0 );
	hAIMboard->recordedRegs.regCONFIG = r0;
	IRQ_globalRestore(gie);
}

IDEF void AIM6020_setAD1StartSignal(AIM6020_Handle hAIMboard, Uint32 startSignal)
{
	Uint32 gie;
	register Uint32 r0;
	
	r0 = (hAIMboard->recordedRegs.regCONFIG & 0xFCu) | (startSignal & 0x03);

	gie = IRQ_globalDisable();
	AIM6020_RSETH( hAIMboard, CONFIG, r0 );
	hAIMboard->recordedRegs.regCONFIG = r0;
	IRQ_globalRestore(gie);
}

IDEF void AIM6020_setAD2StartSignal(AIM6020_Handle hAIMboard, Uint32 startSignal)
{
	Uint32 gie;
	register Uint32 r0;
	
	r0 = (hAIMboard->recordedRegs.regCONFIG & 0xF3u) | ((startSignal & 0x03) << 2);

	gie = IRQ_globalDisable();
	AIM6020_RSETH( hAIMboard, CONFIG, r0 );
	hAIMboard->recordedRegs.regCONFIG = r0;
	IRQ_globalRestore(gie);
}

IDEF void AIM6020_setDA1UpdateSignal(AIM6020_Handle hAIMboard, Uint32 updateSignal)
{
	Uint32 gie;
	register Uint32 r0;
	
	r0 = (hAIMboard->recordedRegs.regCONFIG & 0xCFu) | ((updateSignal & 0x03) << 4);

	gie = IRQ_globalDisable();
	AIM6020_RSETH( hAIMboard, CONFIG, r0 );
	hAIMboard->recordedRegs.regCONFIG = r0;
	IRQ_globalRestore(gie);
}

IDEF void AIM6020_setDA2UpdateSignal(AIM6020_Handle hAIMboard, Uint32 updateSignal)
{
	Uint32 gie;
	register Uint32 r0;
	
	r0 = (hAIMboard->recordedRegs.regCONFIG & 0x3Fu) | ((updateSignal & 0x03) << 6);

	gie = IRQ_globalDisable();
	AIM6020_RSETH( hAIMboard, CONFIG, r0 );
	hAIMboard->recordedRegs.regCONFIG = r0;
	IRQ_globalRestore(gie);
}

IDEF Uint32 AIM6020_getAD1StartSignal(AIM6020_Handle hAIMboard)
{
	return (hAIMboard->recordedRegs.regCONFIG & 0x03);
}

IDEF Uint32 AIM6020_getAD2StartSignal(AIM6020_Handle hAIMboard)
{
	return ((hAIMboard->recordedRegs.regCONFIG >> 2) & 0x03);
}

IDEF Uint32 AIM6020_getDA1UpdateSignal(AIM6020_Handle hAIMboard)
{
	return ((hAIMboard->recordedRegs.regCONFIG >> 4) & 0x03);
}

IDEF Uint32 AIM6020_getDA2UpdateSignal(AIM6020_Handle hAIMboard)
{
	return ((hAIMboard->recordedRegs.regCONFIG >> 6) & 0x03);
}

/*----------------------------------------------------------------------------*/
IDEF Int16 AIM6020_readAD(AIM6020_Handle hAIMboard, Uint32 chNum)
{
	if( chNum == AIM6020_CHANNEL1 )
		return AIM6020_readAD1(hAIMboard);
	else /* if( chNum == AIM6020_CHANNEL2 ) */
		return AIM6020_readAD2(hAIMboard);
}

IDEF Int16 AIM6020_readAD1(AIM6020_Handle hAIMboard)
{
	return (AIM6020_RGETH(hAIMboard, AD1RD) & 0xFFF0u);
}

IDEF Int16 AIM6020_readAD2(AIM6020_Handle hAIMboard)
{
	return (AIM6020_RGETH(hAIMboard, AD2RD) & 0xFFF0u);
}

/*----------------------------------------------------------------------------*/
IDEF void AIM6020_writeDA(AIM6020_Handle hAIMboard, Uint32 chNum, Int16 data)
{
	if( chNum == AIM6020_CHANNEL1 )
		AIM6020_writeDA1(hAIMboard,data);
	else /* if( chNum == AIM6020_CHANNEL2 ) */
		AIM6020_writeDA2(hAIMboard,data);
}

IDEF void AIM6020_writeDA1(AIM6020_Handle hAIMboard, Int16 data)
{
	AIM6020_RSETH( hAIMboard, DA1WR, (data & 0xFFF0u) );
}

IDEF void AIM6020_writeDA2(AIM6020_Handle hAIMboard, Int16 data)
{
	AIM6020_RSETH( hAIMboard, DA2WR, (data & 0xFFF0u) );
}

/*----------------------------------------------------------------------------*/
IDEF void AIM6020_startAD(AIM6020_Handle hAIMboard, Uint32 chNum)
{
	if( chNum == AIM6020_CHANNEL1 )
		AIM6020_startAD1(hAIMboard);
	else if( chNum == AIM6020_CHANNEL2 )
		AIM6020_startAD2(hAIMboard);
	else /* if( chNum == AIM6020_BOTH_CHANNELS ) */
		AIM6020_startAD12(hAIMboard);
}

IDEF void AIM6020_startAD1(AIM6020_Handle hAIMboard)
{
	AIM6020_RGETH( hAIMboard, AD1STRT);
}

IDEF void AIM6020_startAD2(AIM6020_Handle hAIMboard)
{
	AIM6020_RGETH( hAIMboard, AD2STRT);
}

IDEF void AIM6020_startAD12(AIM6020_Handle hAIMboard)
{
	AIM6020_RGETH( hAIMboard, AD12STRT);
}

/*----------------------------------------------------------------------------*/
IDEF void AIM6020_updateDA(AIM6020_Handle hAIMboard, Uint32 chNum)
{
	if( chNum == AIM6020_CHANNEL1 )
		AIM6020_updateDA1(hAIMboard);
	else if( chNum == AIM6020_CHANNEL2 )
		AIM6020_updateDA2(hAIMboard);
	else /* if( chNum == AIM6020_BOTH_CHANNELS ) */
		AIM6020_updateDA12(hAIMboard);
}

IDEF void AIM6020_updateDA1(AIM6020_Handle hAIMboard)
{
	AIM6020_RSETH( hAIMboard, DA1UPDT, 0 );
}

IDEF void AIM6020_updateDA2(AIM6020_Handle hAIMboard)
{
	AIM6020_RSETH( hAIMboard, DA2UPDT, 0 );
}

IDEF void AIM6020_updateDA12(AIM6020_Handle hAIMboard)
{
	AIM6020_RSETH( hAIMboard, DA12UPDT, 0 );
}

/*----------------------------------------------------------------------------*/
IDEF void AIM6020_setFilterClock(AIM6020_Handle hAIMboard, Uint32 fltClkNum,
	Uint32 division)
{
	if( fltClkNum == AIM6020_FILTER_CLOCK1 )
		AIM6020_setFilterClock1(hAIMboard,division);
	else /* if( fltClkNum == AIM6020_FILTER_CLOCK2 ) */
		AIM6020_setFilterClock2(hAIMboard,division);
}

IDEF void AIM6020_setFilterClock1(AIM6020_Handle hAIMboard, Uint32 division)
{
	Uint32 gie;
	register Uint32 r0;

	r0 = division & 0xFFu;
	gie = IRQ_globalDisable();

	AIM6020_RSETH( hAIMboard, FLTCLK1, r0 );
	hAIMboard->recordedRegs.regFLTCLK1 = r0;

	IRQ_globalRestore(gie);
}

IDEF void AIM6020_setFilterClock2(AIM6020_Handle hAIMboard, Uint32 division)
{
	Uint32 gie;
	register Uint32 r0;

	r0 = division & 0xFFu;
	gie = IRQ_globalDisable();

	AIM6020_RSETH( hAIMboard, FLTCLK2, r0 );
	hAIMboard->recordedRegs.regFLTCLK2 = r0;

	IRQ_globalRestore(gie);
}

IDEF Uint32 AIM6020_getFilterClock(AIM6020_Handle hAIMboard, Uint32 fltClkNum)
{
	if( fltClkNum == AIM6020_FILTER_CLOCK1 )
		return AIM6020_getFilterClock1(hAIMboard);
	else /* if( fltClkNum == AIM6020_FILTER_CLOCK2 ) */
		return AIM6020_getFilterClock2(hAIMboard);
}

IDEF Uint32 AIM6020_getFilterClock1(AIM6020_Handle hAIMboard)
{
	return hAIMboard->recordedRegs.regFLTCLK1 & 0xFFu;
}

IDEF Uint32 AIM6020_getFilterClock2(AIM6020_Handle hAIMboard)
{
	return hAIMboard->recordedRegs.regFLTCLK2 & 0xFFu;
}

/*----------------------------------------------------------------------------*/
IDEF void AIM6020_setFilterConfiguration(AIM6020_Handle hAIMboard,
	Uint32 filtClkSelAD1, Uint32 filtClkSelAD2,
	Uint32 filtClkSelDA1, Uint32 filtClkSelDA2,
	Uint32 filtEnAD1, Uint32 filtEnAD2)
{
	Uint32 gie;
	register Uint32 r0;
	
	r0 = (filtClkSelAD1 & 1) | ((filtClkSelAD2 & 1) << 1) |
		((filtClkSelDA1 & 1) << 2) | ((filtClkSelDA2 & 1) << 3) |
		((filtEnAD1 & 1) << 4) | ((filtEnAD2 & 1) << 5);

	gie = IRQ_globalDisable();
	AIM6020_RSETH( hAIMboard, FLTCFG, r0 );
	hAIMboard->recordedRegs.regFLTCFG = r0;
	IRQ_globalRestore(gie);
}

IDEF void AIM6020_setAD1FilterClock(AIM6020_Handle hAIMboard, Uint32 filtClkSel)
{
	Uint32 gie;
	register Uint32 r0;
	
	r0 = (hAIMboard->recordedRegs.regFLTCFG & 0x3Eu) | (filtClkSel & 1);

	gie = IRQ_globalDisable();
	AIM6020_RSETH( hAIMboard, FLTCFG, r0 );
	hAIMboard->recordedRegs.regFLTCFG = r0;
	IRQ_globalRestore(gie);
}

IDEF void AIM6020_setAD2FilterClock(AIM6020_Handle hAIMboard, Uint32 filtClkSel)
{
	Uint32 gie;
	register Uint32 r0;
	
	r0 = (hAIMboard->recordedRegs.regFLTCFG & 0x3Du) | ((filtClkSel & 1) << 1);

	gie = IRQ_globalDisable();
	AIM6020_RSETH( hAIMboard, FLTCFG, r0 );
	hAIMboard->recordedRegs.regFLTCFG = r0;
	IRQ_globalRestore(gie);
}

IDEF void AIM6020_setDA1FilterClock(AIM6020_Handle hAIMboard, Uint32 filtClkSel)
{
	Uint32 gie;
	register Uint32 r0;
	
	r0 = (hAIMboard->recordedRegs.regFLTCFG & 0x3Bu) | ((filtClkSel & 1) << 2);

	gie = IRQ_globalDisable();
	AIM6020_RSETH( hAIMboard, FLTCFG, r0 );
	hAIMboard->recordedRegs.regFLTCFG = r0;
	IRQ_globalRestore(gie);
}

IDEF void AIM6020_setDA2FilterClock(AIM6020_Handle hAIMboard, Uint32 filtClkSel)
{
	Uint32 gie;
	register Uint32 r0;
	
	r0 = (hAIMboard->recordedRegs.regFLTCFG & 0x37u) | ((filtClkSel & 1) << 3);

	gie = IRQ_globalDisable();
	AIM6020_RSETH( hAIMboard, FLTCFG, r0 );
	hAIMboard->recordedRegs.regFLTCFG = r0;
	IRQ_globalRestore(gie);
}

IDEF void AIM6020_enableAD1Filter(AIM6020_Handle hAIMboard, Uint32 bEnable)
{
	Uint32 gie;
	register Uint32 r0;
	
	r0 = (hAIMboard->recordedRegs.regFLTCFG & 0x2Fu);
	if( bEnable ) r0 |= 0x10u;

	gie = IRQ_globalDisable();
	AIM6020_RSETH( hAIMboard, FLTCFG, r0 );
	hAIMboard->recordedRegs.regFLTCFG = r0;
	IRQ_globalRestore(gie);
}

IDEF void AIM6020_enableAD2Filter(AIM6020_Handle hAIMboard, Uint32 bEnable)
{
	Uint32 gie;
	register Uint32 r0;
	
	r0 = (hAIMboard->recordedRegs.regFLTCFG & 0x1Fu);
	if( bEnable ) r0 |= 0x20u;

	gie = IRQ_globalDisable();
	AIM6020_RSETH( hAIMboard, FLTCFG, r0 );
	hAIMboard->recordedRegs.regFLTCFG = r0;
	IRQ_globalRestore(gie);
}

IDEF Uint32 AIM6020_getAD1FilterClock(AIM6020_Handle hAIMboard)
{
	return (hAIMboard->recordedRegs.regFLTCFG & 1);
}

IDEF Uint32 AIM6020_getAD2FilterClock(AIM6020_Handle hAIMboard)
{
	return ((hAIMboard->recordedRegs.regFLTCFG >> 1) & 1);
}

IDEF Uint32 AIM6020_getDA1FilterClock(AIM6020_Handle hAIMboard)
{
	return ((hAIMboard->recordedRegs.regFLTCFG >> 2) & 1);
}

IDEF Uint32 AIM6020_getDA2FilterClock(AIM6020_Handle hAIMboard)
{
	return ((hAIMboard->recordedRegs.regFLTCFG >> 3) & 1);
}

IDEF Uint32 AIM6020_getAD1FilterEnable(AIM6020_Handle hAIMboard)
{
	return ((hAIMboard->recordedRegs.regFLTCFG >> 4) & 1);
}

IDEF Uint32 AIM6020_getAD2FilterEnable(AIM6020_Handle hAIMboard)
{
	return ((hAIMboard->recordedRegs.regFLTCFG >> 5) & 1);
}

/*----------------------------------------------------------------------------*/
IDEF void AIM6020_setInterruptConfiguration(AIM6020_Handle hAIMboard,
	Uint32 sourceSelectPINT6, Uint32 sourceSelectPINT7)
{
	Uint32 gie;
	register Uint32 r0;
	
	r0 = (sourceSelectPINT6 & 0x03) | ((sourceSelectPINT7 & 0x03) << 2);

	gie = IRQ_globalDisable();
	AIM6020_RSETH( hAIMboard, INTCFG, r0 );
	hAIMboard->recordedRegs.regINTCFG = r0;
	IRQ_globalRestore(gie);
}

IDEF void AIM6020_setPINT6Source(AIM6020_Handle hAIMboard, Uint32 sourceSelect)
{
	Uint32 gie;
	register Uint32 r0;
	
	r0 = (hAIMboard->recordedRegs.regINTCFG & 0xC) | (sourceSelect & 0x03);

	gie = IRQ_globalDisable();
	AIM6020_RSETH( hAIMboard, INTCFG, r0 );
	hAIMboard->recordedRegs.regINTCFG = r0;
	IRQ_globalRestore(gie);
}

IDEF void AIM6020_setPINT7Source(AIM6020_Handle hAIMboard, Uint32 sourceSelect)
{
	Uint32 gie;
	register Uint32 r0;
	
	r0 = (hAIMboard->recordedRegs.regINTCFG & 0x3) | ((sourceSelect & 0x03) << 2);

	gie = IRQ_globalDisable();
	AIM6020_RSETH( hAIMboard, INTCFG, r0 );
	hAIMboard->recordedRegs.regINTCFG = r0;
	IRQ_globalRestore(gie);
}

IDEF Uint32 AIM6020_getPINT6Source(AIM6020_Handle hAIMboard)
{
	return (hAIMboard->recordedRegs.regINTCFG & 0x03);
}

IDEF Uint32 AIM6020_getPINT7Source(AIM6020_Handle hAIMboard)
{
	return ((hAIMboard->recordedRegs.regINTCFG >> 2) & 0x03);
}

/*----------------------------------------------------------------------------*/
IDEF void AIM6020_writeSerialData(AIM6020_Handle hAIMboard, Uint32 data)
{
	AIM6020_RSETH( hAIMboard, SERIAL, (data & 0xFFFFu) );
}

IDEF Uint32 AIM6020_readSerialData(AIM6020_Handle hAIMboard)
{
	return (AIM6020_RGETH(hAIMboard, SERIAL) & 0xFFFFu);
}

/*----------------------------------------------------------------------------*/
/* It can appear a hexadecimal number with or without a dot on the display.
 * lower 4 bits of digitAndDot = hex number
 * 5th bit of digitAndDot = dot (0 = disabled, 1 = enabled)
 */
IDEF void AIM6020_setDisplay(AIM6020_Handle hAIMboard, Uint32 digitAndDot)
{
	Uint32 gie;
	register Uint32 r0;
	
	if( digitAndDot & 0x10u )
		r0 = hAIMboard->pDisplayTable[digitAndDot & 0x0Fu] | hAIMboard->pDisplayTable[16];
	else r0 = hAIMboard->pDisplayTable[digitAndDot & 0x0Fu];

	gie = IRQ_globalDisable();
	AIM6020_RSETH( hAIMboard, DISPLAY, r0 );
	hAIMboard->recordedRegs.regDISPLAY = r0;
	IRQ_globalRestore(gie);
}

IDEF void AIM6020_setDisplayDigit(AIM6020_Handle hAIMboard, Uint32 digit)
{
	Uint32 gie;
	register Uint32 r0;
	
	r0 = (hAIMboard->recordedRegs.regDISPLAY & hAIMboard->pDisplayTable[16]) |
		hAIMboard->pDisplayTable[digit & 0x0Fu];

	gie = IRQ_globalDisable();
	AIM6020_RSETH( hAIMboard, DISPLAY, r0 );
	hAIMboard->recordedRegs.regDISPLAY = r0;
	IRQ_globalRestore(gie);
}

IDEF void AIM6020_setDisplayDot(AIM6020_Handle hAIMboard, Uint32 bDotEnable)
{
	Uint32 gie;
	register Uint32 r0;
	
	r0 = (hAIMboard->recordedRegs.regDISPLAY & ~(hAIMboard->pDisplayTable[16])) |
		( bDotEnable ? hAIMboard->pDisplayTable[16] : 0x00 );

	gie = IRQ_globalDisable();
	AIM6020_RSETH( hAIMboard, DISPLAY, r0 );
	hAIMboard->recordedRegs.regDISPLAY = r0;
	IRQ_globalRestore(gie);
}

IDEF Uint32 AIM6020_getDisplay(AIM6020_Handle hAIMboard)
{
	return hAIMboard->recordedRegs.regDISPLAY;
}

IDEF Uint32 AIM6020_getDisplayDigit(AIM6020_Handle hAIMboard)
{
	return (hAIMboard->recordedRegs.regDISPLAY & ~(hAIMboard->pDisplayTable[16]));
}

/* return 0, if dot is disabled
 * return not 0, if dot is enabled
 */
IDEF Uint32 AIM6020_getDisplayDot(AIM6020_Handle hAIMboard)
{
	Uint32 dot_bit;
	
	dot_bit = hAIMboard->pDisplayTable[16];
	return ((hAIMboard->recordedRegs.regDISPLAY & dot_bit) == dot_bit);
}

/*----------------------------------------------------------------------------*/
IDEF void AIM6020_writeI2CData(AIM6020_Handle hAIMboard, Uint32 bit)
{
	AIM6020_RSETH( hAIMboard, I2CDAT, bit );
}

IDEF void AIM6020_setI2CDataHigh(AIM6020_Handle hAIMboard)
{
	AIM6020_RSETH( hAIMboard, I2CDAT, 1 );
}

IDEF void AIM6020_setI2CDataLow(AIM6020_Handle hAIMboard)
{
	AIM6020_RSETH( hAIMboard, I2CDAT, 0 );
}

IDEF Uint32 AIM6020_readI2CData(AIM6020_Handle hAIMboard)
{
	return (AIM6020_RGETH( hAIMboard, I2CDAT ) & 1);
}

/*----------------------------------------------------------------------------*/
IDEF void AIM6020_writeI2CClock(AIM6020_Handle hAIMboard, Uint32 bit)
{
	AIM6020_RSETH( hAIMboard, I2CCLK, bit );
}

IDEF void AIM6020_setI2CClockHigh(AIM6020_Handle hAIMboard)
{
	AIM6020_RSETH( hAIMboard, I2CCLK, 1 );
}

IDEF void AIM6020_setI2CClockLow(AIM6020_Handle hAIMboard)
{
	AIM6020_RSETH( hAIMboard, I2CCLK, 0 );
}

/*----------------------------------------------------------------------------*/
IDEF void AIM6020_enableI2CDataOut(AIM6020_Handle hAIMboard)
{
	AIM6020_RSETH( hAIMboard, I2COE, 1 );
}

IDEF void AIM6020_disableI2CDataOut(AIM6020_Handle hAIMboard)
{
	AIM6020_RSETH( hAIMboard, I2COE, 0 );
}

/*----------------------------------------------------------------------------*/
/* waits 'microsec' us */
IDEF void AIM6020_delay_us( Uint32 microsec )
{
	Uint32	i;
	
	// divide 8 <--- nop + sub + b + nop5 according to ASM code
	i = ( (microsec + 1) * BSL_boardDescriptor.dspNomSpeed) >> 3;
	for( ; i != 0; i-- )
	{
		asm("   nop   ");
	}
}

IDEF void AIM6020_waitForDigPotPos()
{
	AIM6020_delay_us(1000);		// 1 ms
}

IDEF void AIM6020_waitForDigPotLoad()
{
	AIM6020_delay_us(2000);		// 2 ms
}

/*----------------------------------------------------------------------------*/
IDEF Uint32 AIM6020_setAD1offset(AIM6020_Handle hAIMboard, Uint32 position)
{
	return AIM6020_setDigPotPosition(
		hAIMboard, AIM6020_DIGPOT_POTID_AI1O, position);
}

IDEF Uint32 AIM6020_setAD1gain(AIM6020_Handle hAIMboard, Uint32 position)
{
	return AIM6020_setDigPotPosition(
		hAIMboard, AIM6020_DIGPOT_POTID_AI1G, position);
}

IDEF Uint32 AIM6020_setAD2offset(AIM6020_Handle hAIMboard, Uint32 position)
{
	return AIM6020_setDigPotPosition(
		hAIMboard, AIM6020_DIGPOT_POTID_AI2O, position);
}

IDEF Uint32 AIM6020_setAD2gain(AIM6020_Handle hAIMboard, Uint32 position)
{
	return AIM6020_setDigPotPosition(
		hAIMboard, AIM6020_DIGPOT_POTID_AI2G, position);
}

IDEF Uint32 AIM6020_setDA1offset(AIM6020_Handle hAIMboard, Uint32 position)
{
	return AIM6020_setDigPotPosition(
		hAIMboard, AIM6020_DIGPOT_POTID_AO1O, position);
}

IDEF Uint32 AIM6020_setDA1gain(AIM6020_Handle hAIMboard, Uint32 position)
{
	return AIM6020_setDigPotPosition(
		hAIMboard, AIM6020_DIGPOT_POTID_AO1G, position);
}

IDEF Uint32 AIM6020_setDA2offset(AIM6020_Handle hAIMboard, Uint32 position)
{
	return AIM6020_setDigPotPosition(
		hAIMboard, AIM6020_DIGPOT_POTID_AO2O, position);
}

IDEF Uint32 AIM6020_setDA2gain(AIM6020_Handle hAIMboard, Uint32 position)
{
	return AIM6020_setDigPotPosition(
		hAIMboard, AIM6020_DIGPOT_POTID_AO2G, position);
}

/*----------------------------------------------------------------------------*/
IDEF Uint32 AIM6020_getAD1offset(AIM6020_Handle hAIMboard)
{
	return AIM6020_getDigPotPosition(hAIMboard, AIM6020_DIGPOT_POTID_AI1O);
}

IDEF Uint32 AIM6020_getAD1gain(AIM6020_Handle hAIMboard)
{
	return AIM6020_getDigPotPosition(hAIMboard, AIM6020_DIGPOT_POTID_AI1G);
}

IDEF Uint32 AIM6020_getAD2offset(AIM6020_Handle hAIMboard)
{
	return AIM6020_getDigPotPosition(hAIMboard, AIM6020_DIGPOT_POTID_AI2O);
}

IDEF Uint32 AIM6020_getAD2gain(AIM6020_Handle hAIMboard)
{
	return AIM6020_getDigPotPosition(hAIMboard, AIM6020_DIGPOT_POTID_AI2G);
}

IDEF Uint32 AIM6020_getDA1offset(AIM6020_Handle hAIMboard)
{
	return AIM6020_getDigPotPosition(hAIMboard, AIM6020_DIGPOT_POTID_AO1O);
}

IDEF Uint32 AIM6020_getDA1gain(AIM6020_Handle hAIMboard)
{
	return AIM6020_getDigPotPosition(hAIMboard, AIM6020_DIGPOT_POTID_AO1G);
}

IDEF Uint32 AIM6020_getDA2offset(AIM6020_Handle hAIMboard)
{
	return AIM6020_getDigPotPosition(hAIMboard, AIM6020_DIGPOT_POTID_AO2O);
}

IDEF Uint32 AIM6020_getDA2gain(AIM6020_Handle hAIMboard)
{
	return AIM6020_getDigPotPosition(hAIMboard, AIM6020_DIGPOT_POTID_AO2G);
}

/*----------------------------------------------------------------------------*/
IDEF void AIM6020_config(AIM6020_Handle hAim6020, AIM6020_Config *config)
{
	Uint32 gie;
	volatile Uint16 *base;
	volatile AIM6020_RecordedRegs *recregs;
	register Uint32 r0, r1, r2, r3, r4;

	base = (volatile Uint16 *)(hAim6020->baseAddr);
	recregs = (volatile AIM6020_RecordedRegs *) &(hAim6020->recordedRegs);

	gie = IRQ_globalDisable();

	r0 = config->regCONFIG;
	r1 = config->regFLTCLK1;
	r2 = config->regFLTCLK2;
	r3 = config->regFLTCFG;
	r4 = config->regINTCFG;
	base[_AIM6020_CONFIG_REGOFFS16] = recregs->regCONFIG = (Uint16)(r0);
	r0 = config->regDISPLAY;
	base[_AIM6020_FLTCLK1_REGOFFS16] = recregs->regFLTCLK1 = (Uint16)(r1);
	base[_AIM6020_FLTCLK2_REGOFFS16] = recregs->regFLTCLK2 = (Uint16)(r2);
	base[_AIM6020_FLTCFG_REGOFFS16] = recregs->regFLTCFG = (Uint16)(r3);
	base[_AIM6020_INTCFG_REGOFFS16] = recregs->regINTCFG = (Uint16)(r4);
	base[_AIM6020_DISPLAY_REGOFFS16] = recregs->regDISPLAY = (Uint16)(r0);

	IRQ_globalRestore(gie);
	
}

/*----------------------------------------------------------------------------*/
IDECL void AIM6020_configArgs(AIM6020_Handle hAim6020,
	Uint32 config, Uint32 fltclk1, Uint32 fltclk2, Uint32 fltcfg,
	Uint32 intcfg, Uint32 display)
{
	Uint32 gie;
	volatile Uint16 *base;
	volatile AIM6020_RecordedRegs *recregs;
	register Uint32 r0, r1, r2, r3, r4;

	base = (volatile Uint16 *)(hAim6020->baseAddr);
	recregs = (volatile AIM6020_RecordedRegs *) &(hAim6020->recordedRegs);

	gie = IRQ_globalDisable();

	r0 = config;
	r1 = fltclk1;
	r2 = fltclk2;
	r3 = fltcfg;
	r4 = intcfg;
	base[_AIM6020_CONFIG_REGOFFS16] = recregs->regCONFIG = (Uint16)(r0);
	r0 = display;
	base[_AIM6020_FLTCLK1_REGOFFS16] = recregs->regFLTCLK1 = (Uint16)(r1);
	base[_AIM6020_FLTCLK2_REGOFFS16] = recregs->regFLTCLK2 = (Uint16)(r2);
	base[_AIM6020_FLTCFG_REGOFFS16] = recregs->regFLTCFG = (Uint16)(r3);
	base[_AIM6020_INTCFG_REGOFFS16] = recregs->regINTCFG = (Uint16)(r4);
	base[_AIM6020_DISPLAY_REGOFFS16] = recregs->regDISPLAY = (Uint16)(r0);

	IRQ_globalRestore(gie);

}

/*----------------------------------------------------------------------------*/
IDECL void AIM6020_getConfig(AIM6020_Handle hAim6020, AIM6020_Config *config)
{
	Uint32 gie;
	volatile AIM6020_Config *cfg;
	volatile AIM6020_RecordedRegs *recregs;

	cfg = (volatile AIM6020_Config *) config;
	recregs = (volatile AIM6020_RecordedRegs *) &(hAim6020->recordedRegs);
	gie = IRQ_globalDisable();

	cfg->regCONFIG = recregs->regCONFIG;
	cfg->regFLTCLK1 = recregs->regFLTCLK1;
	cfg->regFLTCLK2 = recregs->regFLTCLK2;
	cfg->regFLTCFG = recregs->regFLTCFG;
	cfg->regINTCFG = recregs->regINTCFG;
	cfg->regDISPLAY = recregs->regDISPLAY;

	IRQ_globalRestore(gie);
}


/*----------------------------------------------------------------------------*/
#endif /* USEDEFS */


#endif /* BSL_AIM6020_SUPPORT */
#endif /* _BSL_AIM6020_H_ */
/******************************************************************************\
* End of bsl_aim6020.h
\******************************************************************************/

