/******************************************************************************\
* Copyright (C) 2006 by RTD Embedded Technologies, Inc.   All rights reserved.
* Confidential and Proprietary, Not for Public Release
*------------------------------------------------------------------------------
* PROJECT.......... Board Support Library for SPM176431
* VERSION.......... (Defined in README.TXT)
*------------------------------------------------------------------------------
* CONTENT.......... Header file for API of peripheral Dual UART
* FILENAME......... bsl_duart.h
*------------------------------------------------------------------------------
* PERIPHERAL NAME.. Dual UART
* PERIPHERAL TYPE.. Single-device peripheral
* ACCESS MODE...... Direct (no handle)
* REGISTER ACCESS.. 8-bit wide
*
*   NOTE:   All functions and variables in this file are only intended to be
*           utilized by a single thread in any .out file or project.  These
*           functions are not "thread safe."  To explain, in a multithreaded
*           application you might fork off a single thread with the sole purpose
*           of pulling serial data out of the input buffers and placing data
*           into the output buffer.
*
\******************************************************************************/
#ifndef _BSL_DUART_H_
#define _BSL_DUART_H_

#include <bsl.h>


#if (BSL_DUART_SUPPORT)
/******************************************************************************\
* scope and inline control macros
\******************************************************************************/

#ifdef  __cplusplus
    #define BSLAPI extern "C" far
#else
    #define BSLAPI extern far
#endif

#undef  USEDEFS
#undef  IDECL
#undef  IDEF

#ifdef _BSL_DUART_MOD_
    #define IDECL BSLAPI
    #define USEDEFS
    #define IDEF
#else
    #ifdef  _INLINE
        #define IDECL static inline
        #define USEDEFS
        #define IDEF  static inline
    #else
        #define IDECL BSLAPI
    #endif
#endif


/******************************************************************************\
* global macro declarations
\******************************************************************************/

#define DUART_DO_NOT_CHANGE_VALUE                     0xFFFFFFFF

#ifndef FALSE
#define FALSE                                       0
#endif

#ifndef FALSE
#define TRUE                                        1
#endif

// global dual uart operation functions' return values
#define DUART_NO_ERROR                              0
#define DUART_ERROR_INVALID_PARAMETER               1
#define DUART_BAUD_RATE_ROUNDED_DOWN                2

// if this is changed be sure to change the linker file also
// below 512 words -- 512 words * 2 bytes per word means it
// actually holds 1024 8-bit-characters
#define SIZE_OF_UART_BUFF                           1024

//for TESTING ONLY
#define SIZE_OF_UART_TEST_BUFF                      2064

// this determines the max number of characters that will be
// entered into the output buffer w/o toggling the tranmist
// interrupt in between
// by lowering this number it can prevent starvation of the
// transmit FIFO during the time period of large copies of
// data into the output buffer
// NOTE:  it does take time to toggle the transmit IT enable
// bit in the UART and make this number too low will cause
// severe ineffeciencies in your put_data functions
#define MAX_NUM_OF_BYTES_TO_COPY_TO_UART_AT_ONCE    32

// starting byte address of the flash device in the DSP's memory address space
#define DUART_STARTING_ADDRESS              0x60000000

// offsets for each register in the Dual UART's
#define DUART_MODE_OFFSET                   0x0C0800A0

    /* to access reg below LCR[7] = 0 */
#define DUART1_RHR_OFFSET                       0
#define DUART1_THR_OFFSET                       0
#define DUART1_IER_OFFSET                       2
#define DUART1_ISR_OFFSET                       4
#define DUART1_FCR_OFFSET                       4
#define DUART2_RHR_OFFSET                       16
#define DUART2_THR_OFFSET                       16
#define DUART2_IER_OFFSET                       18
#define DUART2_ISR_OFFSET                       20
#define DUART2_FCR_OFFSET                       20

#define DUART1_LCR_OFFSET                       6
#define DUART2_LCR_OFFSET                       22

    /* to access reg below LCR != 0xBF */
#define DUART1_MCR_OFFSET                       8
#define DUART1_LSR_OFFSET                       10
#define DUART1_MSR_OFFSET                       12
#define DUART1_SPR_OFFSET                       14
#define DUART2_MCR_OFFSET                       24
#define DUART2_LSR_OFFSET                       26
#define DUART2_MSR_OFFSET                       28
#define DUART2_SPR_OFFSET                       30

    /* to access reg below LCR[7] = 1 & LCR != 0xBF */
#define DUART1_DLL_OFFSET                       0
#define DUART1_DLM_OFFSET                       2
#define DUART1_AFR_OFFSET                       4
#define DUART2_DLL_OFFSET                       16
#define DUART2_DLM_OFFSET                       18
#define DUART2_AFR_OFFSET                       20

    /* to access reg below LCR = 0xBF */
#define DUART1_EFR_OFFSET                       4
#define DUART1_XON1_OFFSET                      8
#define DUART1_XON2_OFFSET                      10
#define DUART1_XOFF1_OFFSET                     12
#define DUART1_XOFF2_OFFSET                     14
#define DUART2_EFR_OFFSET                       20
#define DUART2_XON1_OFFSET                      24
#define DUART2_XON2_OFFSET                      26
#define DUART2_XOFF1_OFFSET                     28
#define DUART2_XOFF2_OFFSET                     30

/* these are the case of the ISR switch statement */
#define LINE_STATUS_REGISTER                    6
#define RXED_DATA_TIME_OUT                      12
#define RXED_DATA_READY                         4
#define TRANSMIT_READY                          2
#define MODEM_STATUS_REGISTER                   0
#define RXED_XOFF_SPECIAL_CHARACTER             16
#define CTS_RTS_CHANGE_OF_STATE                 32
#define NO_IT_PENDING                           1

/* 
   Bit Position for the Delta bits in the MSR register.
   Use these to test for delta changes to he various 
   bit fields. All the bits will be cleared after read,
   so it is imperative to read and store the MSR
   register before testing any of these bits.
 */
/**
    Bit is set when CTS has changed state.
 */
#define UART_MSR_DELTA_CTS      0x0001

/**
    Bit is set when DSR has changed state.
 */
#define UART_MSR_DELTA_DSR      0x0002

/**
    Bit is set when the RI line changes from 0 to 1
 */
#define UART_MSR_DELTA_RI       0x0004

/**
    Bit is set when CD has changed state.
 */
#define UART_MSR_DELTA_CD       0x0008


/******************************************************************************\
* global typedef declarations
\******************************************************************************/

// The user has to definiate a variable as DUART_Return-type variable to
// be able to receive the return value from the flash controller for the flash
// operation that was started by the user.

/******************************************************************************\
* global variable declarations
\******************************************************************************/


/******************************************************************************\
* global function declarations
\******************************************************************************/

//=============================================================================
// DUART_Data_In_Rcv
//  This function returns the amount of data in the UART 1 recieve buffer in
//  in terms of 8 bit characters.
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//  Returns
//      the amount characters in the buffer
//
//=============================================================================
Uint32 DUART_Data_In_Rcv(Uint16 Uart);
//=============================================================================
// DUART_Data_In_Tx
//  This function returns the amount of data in the UART 1 transmit buffer in
//  in terms of 8 bit characters.
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//  Returns
//      the amount characters in the buffer
//
//=============================================================================
Uint32 DUART_Data_In_Tx(Uint16 Uart);
//=============================================================================
// DUART_Room_In_Rcv
//  This function returns the amount room that is left for data in terms of
//  8 bit characters in the UART 1 reciever buffer.
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//  Returns
//      the amount characters that can still fit in the buffer
//
//=============================================================================
Uint32 DUART_Room_In_Rcv(Uint16 Uart);
//=============================================================================
// DUART_Room_In_Tx
//  This function returns the amount room that is left for data in terms of
//  8 bit characters in the UART 1 reciever buffer.
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//  Returns
//      the amount characters that can still fit in the buffer
//
//=============================================================================
Uint32 DUART_Room_In_Tx(Uint16 Uart);
//=============================================================================
// DUART_Flush_Rcv
//  This function flushes out the receive uart buffer
//
//=============================================================================
void DUART_Flush_Rcv(Uint16 Uart);
//=============================================================================
// DUART_Flush_Tx
//  This function flushes out the transmit uart buffer
//
//=============================================================================
void DUART_Flush_Tx(Uint16 Uart);
//=============================================================================
// DUART_Put_Data
//  This function places data into the output buffer for the user.  It accepts
//  pointer to the begining of the data and the number of characters to be
//  transfered to the buffer.  The first thing the function does is check if
//  there is enough room for all the character in the output buffer.  If there
//  is not enough room the function immediately returns the value 0(indicating
//  0 characters have been copied.  If there is enough room the function then
//  breaks down the data into chunks <= the size of
//  MAX_NUM_OF_BYTES_TO_COPY_TO_UART_AT_ONCE.  In the outter loop the transmit
//  interrupt is turned off during the copy of each chunk into the uart buffer.
//  After each a chunk has copied into the buffer, the transmit interrupt is
//  turned back on.  This allows any pending transmit interrupts to be serviced
//  with the new data that is now in the output buffer.  The inner loop is
//  where the data is actually copied.
//
//  Input:
//      Uart            0) uart1                1) uart2
//      Uint16 *StartPtr:  points to the address of data to enter
//                      this data is expected to have the first character in
//                      this buffer to be in the lower 8 bits at StartPtr.
//                      The second character is expected to be in the upper 8
//                      8 bits at StartPtr.  Third character is lower 8 bits
//                      of StartPtr++, etc...
//      NumOfBytes:     number of characters to copy into the transmit buffer
//                      in the format as specified above
//
//  Return:
//      it returns the number of character copied into the buffer.
//      this will be the number requested or 0
//
//=============================================================================
Uint32 DUART_Put_Data(Uint16 Uart, char *StartPtr, Uint32 NumOfBytes);
//=============================================================================
// DUART_Get_Data_WO_Remove
//  This function is passed a pointer to a pointer.  The pointer that this
//  pointer is pointed at will be redirected at the address of the first
//  character in the recieve buffer.  The function is also passed a boolean by
//  reference.  This boolean value will be true only if the first character of
//  the buffer is located in the upper 8 bit at the adress of StartPtr.  In
//  which case the second character is located in the lower 8 bits of
//  StartPtr++.  Third is located upper 8 bits of StartPtr++ and so on & etc.
//  Otherwise if this boolean(StartsInUpperHalf) is false, the first character
//  is located in the lower 8 bits of StartPtr.  The second character is
//  located in the upper 8 bits of StartPtr.  The third character is located
//  in the lower 8 bits of StartPtr++ and so on & etc.  The last parameter
//  passed to this function is the number of bytes requested.  This function
//  will either return this number if there is this many characters in the
//  buffer or return the total amount of characters currently in the buffer
//
//  NOTE:   This is a circular que.  The returned chunk of memory may not be
//      contigious.  For example, you may request 20 characters;  there may
//      be more than 20 characters in the buffer, but 10 might be in the last
//      10 locations of the buffer and 10 might be in the first ten locations
//      of the buffer.  UartxRxBuff[SIZE_OF_UART_BUFF -1] holds the last two
//      8 bit words in the buffer.  The first two characters of the buffer is
//      at Uart1RxBuff[0].
//
//  Input:
//      Uart        0) uart1                1) uart2
//      StartPtr:  ptr to a pointer that is passed by reference.  When the
//                 function returns it will change the pointer that it points
//                 to to the begining of the buffer
//      StartsInUpperHalf:
//                  boolean passed by reference.  When the function returns
//                  0) first character is in lower 8 bits in begining of the
//                  buffer
//                  1) first character is in upper 8 bits in begining of the
//                  buffer
//      NumOfBytes: how many characters you want
//
//  Return:
//      The number of characters you requested in the buffer if they are
//      available or the number of characters available
//
//=============================================================================
Uint32 DUART_Get_Data_WO_Remove(Uint16 Uart, char* *StartPtr, Uint32 NumOfBytes);
//=============================================================================
// DUART_Remove_Data
//  This function removes the specified(Bytes) number of characters from the
//  begining of the recieve buffer.  If there are not enough(Bytes) characters
//  in the recieve buffer it will remove all characters in the buffer
//
//  Inputs:
//      Uart:  0) selects uart1     1) selects uart2
//      Bytes:  how many bytes to remove
//
//  Returns:
//      the number of characters removed from the recieve buffer
//
//=============================================================================
Uint32 DUART_Remove_Data(Uint16 Uart, Uint32 Bytes);
//=============================================================================
// DUART_Set_Line_Control
//  This function initializes the UART setting that determine exactly how the
//  data will be sent/recieved over the send/recieve lines.
//
//  Inputs:
//      Uart:  0) selects uart1     1) selects uart2
//
//      Baud:   Enter the desired baud rate.  The uart has an extenal crystal
//              of 14.7456 MHz.  Typical baud rates range from 400 to 921.6K.
//              The baud rate must be an even divisor of 921600 for 0% error.
//
//      Wordlength: 0) 5 bits(default)
//                  1) 6 bits
//                  2) 7 bits
//                  3) 8 bits
//
//      StopBits:   0) 1 stop bit(default)
//                  1) if wordlength = 5 bits then 1.5 stop bits
//                     if wordlength = 6,7,8 bits then 2 stop bits
//
//      ParityEnable    0) No parity
//                      1) transmitter and reciever utilize parity bits
//
//      ParityEven      0) Odd number of 1's is transmitted and recieved
//                      1) Even number of 1's is transmitted and recieved
//
//      ForceParity     0) Parity is not forced
//                      1) the parity is forced to the above value in transmit
//                      and recieve by the uart
//
//              See EXAR XR16L2552 Rev 1.1.0 for more hardware info.
//
//  Returns:
//      The function returns one of three error codes:
//  DUART_BAUD_RATE_ROUNDED_DOWN:  You did not ask for a 0% error baud rate
//                                  and the function chose the closest
//                                  available baud by rounding down the divisor
//                                  to the nearest whole number
//
//  DUART_NO_ERRORS:                no errors in setting baud rate
//  DUART_ERROR_INVALID_PARAMETER:  invalid parameters entered
//
//=============================================================================
Uint32 DUART_Set_Line_Control(Uint16 Uart, Uint32 Baud, Uint32 WordLength, Uint32 StopBits,
            Uint32 ParityEnable, Uint32 ParityEven, Uint32 ForceParity);
//=============================================================================
// DUART_Mode
//  This function sets the protocol of the serial port.
//
//  Input:
//      ModeUart1:
//      ModeUart2:  0)Disabled
//                  1)Reserved
//                  2)RS-232 Mode
//                  3)RS-422/485 Mode
//
//  Returns:
//      The function returns one of two error codes:
//  DUART_NO_ERRORS:                no errors in setting baud rate
//  DUART_ERROR_INVALID_PARAMETER:  invalid parameters entered
//
//=============================================================================
Uint32 DUART_Mode(Uint32 ModeUart1, Uint32 ModeUart2);
//=============================================================================
// DUART_Set_Fifo_Control
//  This function can enable/disable fifos, reset recieve/transmit fifos, and
//  set the recieve fifo trigger value.
//
//  Input:
//      Uart:           0) uart1        1) uart2
//      Enable:         0)disable fifos 1)enable fifos
//      RcvFifoReset:   0) no reset     1) do a reset on recieve fifos
//      RcvFifoTrigger:(0) 1(default)  (1) 4   (2) 8   (3) 14
//          This is the number of characters the Fifo will need to contain
//          before a recieve fifo interrupt is genererated.
//      TxFifoReset:    0) no reset     1) do a reset on transmit fifos
//
//      See EXAR XR16L2552 Rev 1.1.0 for more hardware info.
//
//  Returns:
//      The function returns one of two error codes:
//  DUART_NO_ERRORS:                no errors in setting baud rate
//  DUART_ERROR_INVALID_PARAMETER:  invalid parameters entered
//
//=============================================================================
Uint32 DUART_Set_Fifo_Control(Uint16 Uart, Uint32 Enable, Uint32 RcvFifoReset,
                                Uint32 RcvFifoTrigger, Uint32 TxFifoReset);
//=============================================================================
// DUART_Set_Interrupt_Enables
//  This function enables the actual interrupts on the UART chip
//
//  Input:
//      Uart:           0) uart1                1) uart2
//      RcvIT:          0) interrupt disabled   1) interrupt enabled
//      TxIT:           0) interrupt disabled   1) interrupt enabled
//      RcvLineStatIT:  0) interrupt disabled   1) interrupt enabled
//      ModemStatIT:    0) interrupt disabled   1) interrupt enabled
//
//      See EXAR XR16L2552 Rev 1.1.0 for more hardware info.
//
//  Returns:
//      The function returns one of two error codes:
//  DUART_NO_ERRORS:                no errors in setting baud rate
//  DUART_ERROR_INVALID_PARAMETER:  invalid parameters entered
//
//=============================================================================
Uint32 DUART_Set_Interrupt_Enables(Uint16 Uart, Uint32 RcvIT, Uint32 TxIT,
                        Uint32 RcvLineStatIT, Uint32 ModemStatIT);
//=============================================================================
// DUART_Is_Rx_IT_Enabled
//  This function test if recieve interrupt is currently enabled on the chip
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//  Returns
//      Boolean value of 0) means disabled, 1) means enabled
//
//=============================================================================
BOOL DUART_Is_Rx_IT_Enabled(Uint16 Uart);
//=============================================================================
// DUART_Is_Tx_IT_Enabled
//  This function test if transmit interrupt is currently enabled on the chip
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//  Returns
//      Boolean value of 0) means disabled, 1) means enabled
//
//=============================================================================
BOOL DUART_Is_Tx_IT_Enabled(Uint16 Uart);
//=============================================================================
// DUART_Is_Rx_Line_Status_IT_Enabled
//  This function test if recieve line status interrupt is currently enabled
//  on the chip
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//  Returns
//      Boolean value of 0) means disabled, 1) means enabled
//
//=============================================================================
BOOL DUART_Is_Rx_Line_Status_IT_Enabled(Uint16 Uart);
//=============================================================================
// DUART_Is_Modem_Status_IT_Enabled
//  This function test if modem line status interrupt is currently enabled
//  on the chip
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//  Returns
//      Boolean value of 0) means disabled, 1) means enabled
//
//=============================================================================
BOOL DUART_Is_Modem_Status_IT_Enabled(Uint16 Uart);
//=============================================================================
// DUART_Get_5_ISR_bits
//  This function returns the value of the lowest 5 bits of the ISR register
//  int the UART(they are listed below from highest to lowest priority).
//
//      See EXAR XR16L2552 Rev 1.1.0 for more hardware info.
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//  Returns:
//      6)  Line Status---------------------( 0 0 0 1 1 0 )
//      12) Recieve Ready( Time-out)--------( 0 0 1 1 0 0 )
//      4)  Recieve Ready( Recieved data)---( 0 0 0 1 0 0 )
//      2)  Transmit Ready------------------( 0 0 0 0 1 0 )
//      0)  Modem Status--------------------( 0 0 0 0 0 0 )
//      16) Recieved Xoff/specieal char-----( 0 1 0 0 0 0 )
//      32) CTS/RTS-------------------------( 1 0 0 0 0 0 )
//      1)  None(default)-------------------( 0 0 0 0 0 1 )
//
//=============================================================================
Uint32 DUART_Get_5_ISR_bits(Uint16 Uart);
//=============================================================================
// DUART_Clear_Line_Stat_IT
//  This function clears a line status interrupt on a chip by reading the
//  line status register.  This is the only thing that is necessary to clear
//  this ineterrupt on the chip
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//      See EXAR XR16L2552 Rev 1.1.0 for more hardware info.
//
//=============================================================================
void DUART_Clear_Line_Stat_IT(Uint16 Uart);
//=============================================================================
// DUART_Clear_Modem_Stat_IT
//  This function clears a modem status interrupt on a chip by reading the
//  modem control register.  This is the only thing that is necessary to clear
//  this ineterrupt on the chip
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//      See EXAR XR16L2552 Rev 1.1.0 for more hardware info.
//
//=============================================================================
void DUART_Clear_Modem_Stat_IT(Uint16 Uart);
//=============================================================================
// DUART_Is_Rx_Data_Ready
//  This function checks the status of the recieve data ready bit in the LSR
//  register on the chip.
//
//      See EXAR XR16L2552 Rev 1.1.0 for more hardware info.
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//  Returns:
//      returns the value of the bit, 0/1
//
//=============================================================================
BOOL DUART_Is_Rx_Data_Ready(Uint16 Uart);
//=============================================================================
// DUART_Is_Rx_Overrun_Error
//  This function checks the status of the recieve overrun error bit in the LSR
//  register on the chip.
//
//      See EXAR XR16L2552 Rev 1.1.0 for more hardware info.
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//  Returns:
//      returns the value of the bit, 0/1
//
//=============================================================================
BOOL DUART_Is_Rx_Overrun_Error(Uint16 Uart);
//=============================================================================
// DUART_Is_Rx_Parity_Error
//  This function checks the status of the recieve parity error bit in the LSR
//  register on the chip.
//
//      See EXAR XR16L2552 Rev 1.1.0 for more hardware info.
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//  Returns:
//      returns the value of the bit, 0/1
//
//=============================================================================
BOOL DUART_Is_Rx_Parity_Error(Uint16 Uart);
//=============================================================================
// DUART_Is_Rx_Framing_Error
//  This function checks the status of the recieve framing error bit in the LSR
//  register on the chip.
//
//      See EXAR XR16L2552 Rev 1.1.0 for more hardware info.
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//  Returns:
//      returns the value of the bit, 0/1
//
//=============================================================================
BOOL DUART_Is_Rx_Framing_Error(Uint16 Uart);
//=============================================================================
// DUART_Is_Rx_Break_Error
//  This function checks the status of the recieve break error bit in the LSR
//  register on the chip.
//
//      See EXAR XR16L2552 Rev 1.1.0 for more hardware info.
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//  Returns:
//      returns the value of the bit, 0/1
//
//=============================================================================
BOOL DUART_Is_Rx_Break_Error(Uint16 Uart);
//=============================================================================
// DUART_Is_THR_Empty
//  This function checks the status of the transmit holding register empty bit
//  in the LSR register on the chip.
//
//      See EXAR XR16L2552 Rev 1.1.0 for more hardware info.
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//  Returns:
//      returns the value of the bit, 0/1
//
//=============================================================================
BOOL DUART_Is_THR_Empty(Uint16 Uart);
//=============================================================================
// DUART_Is_THR_TSR_Both_Empty
//  This function checks the status of the transmit holding register empty and
//  the transmit shift register empty bit in the LSR register on the chip.
//
//      See EXAR XR16L2552 Rev 1.1.0 for more hardware info.
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//  Returns:
//      returns the value of the bit, 0/1
//
//=============================================================================
BOOL DUART_Is_THR_TSR_Both_Empty(Uint16 Uart);
//=============================================================================
// DUART_Is_Rx_FIFO_Global_Error
//  This function checks the status of the recieve global error bit in the LSR
//  register on the chip.
//
//      See EXAR XR16L2552 Rev 1.1.0 for more hardware info.
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//  Returns:
//      returns the value of the bit, 0/1
//
//=============================================================================
BOOL DUART_Is_Rx_FIFO_Global_Error(Uint16 Uart);
//=============================================================================
// DUART_Enable_Tx_Break
//  This function enables the Transmission break bit int the UART's LCR.  This
//  will force the UART's transmitter to transmit a break signal.
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//=============================================================================
void DUART_Enable_Tx_Break(Uint16 Uart);
//=============================================================================
// DUART_Disable_Tx_Break
//  This function disables the Transmission break bit int the UART's LCR.  This
//  will force the UART's transmitter to stop transmitting a break signal.
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//=============================================================================
void DUART_Disable_Tx_Break(Uint16 Uart);
//=============================================================================
// DUART_Disable_Transmitter
//  This function controls the modem control line (RTS) which we uniquely use
//  to feed into the FPGA to indicate that the transmitters should be blocked
//  in the FPGA hardware.  This means that anything that is 'transmitted' in
//  software during this time will transmit off the UART chip but will never
//  make it off the DSP board itself.  It will be esentially thrown away by
//  FPGA's hardware.  It is important to realize though as far as software is
//  concerned it will continue to transmit.
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//=============================================================================
void DUART_Enable_Transmitter(Uint16 Uart);
//=============================================================================
// DUART_Disable_Transmitter
//  This function controls the modem control line (RTS) which we uniquely use
//  to feed into the FPGA to indicate that the transmitters should be blocked
//  in the FPGA hardware.  This means that anything that is 'transmitted' in
//  software during this time will transmit off the UART chip but will never
//  make it off the DSP board itself.  It will be esentially thrown away by
//  FPGA's hardware.  It is important to realize though as far as software is
//  concerned it will continue to transmit.
//
//  Input:
//      Uart:  0) selects uart1     1) selects uart2
//
//=============================================================================
void DUART_Disable_Transmitter(Uint16 Uart);
/******************************************************************************\
* inline function declarations
\******************************************************************************/

IDECL BOOL Is_DUART_Failed_Recieve_Write(Uint16 Uart);
IDECL BOOL Is_DUART_Line_Status_IT(Uint16 Uart);
IDECL BOOL Is_DUART_Modem_Status_IT(Uint16 Uart);
IDECL BOOL Is_DUART_Tx_IT_turned_off_by_ISR(Uint16 Uart);
IDECL BOOL Is_DUART_RECIEVED_TX_BREAK(Uint16 Uart);
IDECL void Set_DUART_Failed_Recieve_Write(Uint16 Uart);
IDECL void Set_DUART_Line_Status_IT(Uint16 Uart);
IDECL void Set_DUART_Modem_Status_IT(Uint16 Uart);
IDECL void Set_DUART_Tx_IT_turned_off_by_ISR(Uint16 Uart);
IDECL void Set_DUART_RECIEVED_TX_BREAK(Uint16 Uart);
IDECL void Reset_DUART_Failed_Recieve_Write(Uint16 Uart);
IDECL void Reset_DUART_Line_Status_IT(Uint16 Uart);
IDECL void Reset_DUART_Modem_Status_IT(Uint16 Uart);
IDECL void Reset_DUART_Tx_IT_turned_off_by_ISR(Uint16 Uart);
IDECL void Reset_DUART_RECIEVED_TX_BREAK(Uint16 Uart);
/* Modem Control Status control */
IDECL BOOL Is_DUART_CTS_Active(Uint16 Uart);
IDECL BOOL Is_DUART_DSR_Active(Uint16 Uart);
IDECL BOOL Is_DUART_DCD_Active(Uint16 Uart);
IDECL BOOL Is_DUART_RI_Active(Uint16 Uart);

IDECL void DUART_Set_DTR(Uint16 Uart, BOOL State);
IDECL void DUART_Set_RTS(Uint16 Uart, BOOL State);



#ifdef TESTING
IDECL BOOL Is_DUART_TESTING_BUFFER_OVERFLOW(Uint16 Uart);
IDECL BOOL Is_DUART_TESTING_BUFFER_UNDERFLOW(Uint16 Uart);
IDECL BOOL Is_DUART_TESTING_BUFFER_ERROR(Uint16 Uart);
#endif

/******************************************************************************\
* inline function definitions
\******************************************************************************/
#ifdef USEDEFS
void DUART_Write_MCR(Uint16 Uart, Uint16 CurrentMCR);
Uint16 DUART_Get_MCR(Uint16 Uart);
Uint16 DUART_Get_MSR(Uint16 Uart);

/*----------------------------------------------------------------------------*/
IDEF BOOL Is_DUART_Failed_Recieve_Write(Uint16 Uart)
{
    extern far volatile BOOL UART1_FAILED_RECIEVE_WRITE_TO_INPUT_BUFFER;
    extern far volatile BOOL UART2_FAILED_RECIEVE_WRITE_TO_INPUT_BUFFER;
    BOOL FLAG;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            FLAG = UART1_FAILED_RECIEVE_WRITE_TO_INPUT_BUFFER;
            break;

        /* uart2 */
        case 1:
            FLAG = UART2_FAILED_RECIEVE_WRITE_TO_INPUT_BUFFER;
            break;

        /* uart1 */
        default:
            FLAG = UART1_FAILED_RECIEVE_WRITE_TO_INPUT_BUFFER;
            break;

    }/* switch */

    return FLAG;

}
/*----------------------------------------------------------------------------*/
IDEF BOOL Is_DUART_Line_Status_IT(Uint16 Uart)
{
    extern far volatile BOOL UART1_LINE_STATUS_INTERRUPT_ERROR;
    extern far volatile BOOL UART2_LINE_STATUS_INTERRUPT_ERROR;
    BOOL FLAG;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            FLAG = UART1_LINE_STATUS_INTERRUPT_ERROR;
            break;

        /* uart2 */
        case 1:
            FLAG = UART2_LINE_STATUS_INTERRUPT_ERROR;
            break;

        /* uart1 */
        default:
            FLAG = UART1_LINE_STATUS_INTERRUPT_ERROR;
            break;

    }/* switch */

    return FLAG;
}
/*----------------------------------------------------------------------------*/
IDEF BOOL Is_DUART_Modem_Status_IT(Uint16 Uart)
{
    extern far volatile BOOL UART1_MODEM_STATUS_INTERRUPT_ERROR;
    extern far volatile BOOL UART2_MODEM_STATUS_INTERRUPT_ERROR;
    BOOL FLAG;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            FLAG = UART1_MODEM_STATUS_INTERRUPT_ERROR;
            break;

        /* uart2 */
        case 1:
            FLAG = UART2_MODEM_STATUS_INTERRUPT_ERROR;
            break;

        /* uart1 */
        default:
            FLAG = UART1_MODEM_STATUS_INTERRUPT_ERROR;
            break;

    }/* switch */

    return FLAG;
}
/*----------------------------------------------------------------------------*/
IDEF BOOL Is_DUART_Tx_IT_turned_off_by_ISR(Uint16 Uart)
{
    extern far volatile BOOL UART1_TRANSMIT_IT_WAS_TURNED_OFF;
    extern far volatile BOOL UART2_TRANSMIT_IT_WAS_TURNED_OFF;
    BOOL FLAG;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            FLAG = UART1_TRANSMIT_IT_WAS_TURNED_OFF;
            break;

        /* uart2 */
        case 1:
            FLAG = UART2_TRANSMIT_IT_WAS_TURNED_OFF;
            break;

        /* uart1 */
        default:
            FLAG = UART1_TRANSMIT_IT_WAS_TURNED_OFF;
            break;

    }/* switch */

    return FLAG;
}
/*----------------------------------------------------------------------------*/
IDEF BOOL Is_DUART_RECIEVED_TX_BREAK(Uint16 Uart)
{
    extern far volatile BOOL UART1_RECIEVED_TX_BREAK;
    extern far volatile BOOL UART2_RECIEVED_TX_BREAK;
    BOOL FLAG;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            FLAG = UART1_RECIEVED_TX_BREAK;
            break;

        /* uart2 */
        case 1:
            FLAG = UART2_RECIEVED_TX_BREAK;
            break;

        /* uart1 */
        default:
            FLAG = UART1_RECIEVED_TX_BREAK;
            break;

    }/* switch */

    return FLAG;
}
/*----------------------------------------------------------------------------*/
IDEF void Set_DUART_Failed_Recieve_Write(Uint16 Uart)
{
    extern far volatile BOOL UART1_FAILED_RECIEVE_WRITE_TO_INPUT_BUFFER;
    extern far volatile BOOL UART2_FAILED_RECIEVE_WRITE_TO_INPUT_BUFFER;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            UART1_FAILED_RECIEVE_WRITE_TO_INPUT_BUFFER = TRUE;
            break;

        /* uart2 */
        case 1:
            UART2_FAILED_RECIEVE_WRITE_TO_INPUT_BUFFER = TRUE;
            break;

        /* uart1 */
        default:
            UART1_FAILED_RECIEVE_WRITE_TO_INPUT_BUFFER = TRUE;
            break;

    }/* switch */



}
/*----------------------------------------------------------------------------*/
IDEF void Set_DUART_Line_Status_IT(Uint16 Uart)
{
    extern far volatile BOOL UART1_LINE_STATUS_INTERRUPT_ERROR;
    extern far volatile BOOL UART2_LINE_STATUS_INTERRUPT_ERROR;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            UART1_LINE_STATUS_INTERRUPT_ERROR = TRUE;
            break;

        /* uart2 */
        case 1:
            UART2_LINE_STATUS_INTERRUPT_ERROR = TRUE;
            break;

        /* uart1 */
        default:
            UART1_LINE_STATUS_INTERRUPT_ERROR = TRUE;
            break;

    }/* switch */


}
/*----------------------------------------------------------------------------*/
IDEF void Set_DUART_Modem_Status_IT(Uint16 Uart)
{
    extern far volatile BOOL UART1_MODEM_STATUS_INTERRUPT_ERROR;
    extern far volatile BOOL UART2_MODEM_STATUS_INTERRUPT_ERROR;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            UART1_MODEM_STATUS_INTERRUPT_ERROR = TRUE;
            break;

        /* uart2 */
        case 1:
            UART2_MODEM_STATUS_INTERRUPT_ERROR = TRUE;
            break;

        /* uart1 */
        default:
            UART1_MODEM_STATUS_INTERRUPT_ERROR = TRUE;
            break;

    }/* switch */

}
/*----------------------------------------------------------------------------*/
IDEF void Set_DUART_Tx_IT_turned_off_by_ISR(Uint16 Uart)
{
    extern far volatile BOOL UART1_TRANSMIT_IT_WAS_TURNED_OFF;
    extern far volatile BOOL UART2_TRANSMIT_IT_WAS_TURNED_OFF;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            UART1_TRANSMIT_IT_WAS_TURNED_OFF = TRUE;
            break;

        /* uart2 */
        case 1:
            UART2_TRANSMIT_IT_WAS_TURNED_OFF = TRUE;
            break;

        /* uart1 */
        default:
            UART1_TRANSMIT_IT_WAS_TURNED_OFF = TRUE;
            break;

    }/* switch */


}
/*----------------------------------------------------------------------------*/
IDEF void Set_DUART_RECIEVED_TX_BREAK(Uint16 Uart)
{
    extern far volatile BOOL UART1_RECIEVED_TX_BREAK;
    extern far volatile BOOL UART2_RECIEVED_TX_BREAK;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            UART1_RECIEVED_TX_BREAK = TRUE;
            break;

        /* uart2 */
        case 1:
            UART2_RECIEVED_TX_BREAK = TRUE;
            break;

        /* uart1 */
        default:
            UART1_RECIEVED_TX_BREAK = TRUE;
            break;

    }/* switch */


}
/*----------------------------------------------------------------------------*/
IDEF void Reset_DUART_Failed_Recieve_Write(Uint16 Uart)
{
    extern far volatile BOOL UART1_FAILED_RECIEVE_WRITE_TO_INPUT_BUFFER;
    extern far volatile BOOL UART2_FAILED_RECIEVE_WRITE_TO_INPUT_BUFFER;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            UART1_FAILED_RECIEVE_WRITE_TO_INPUT_BUFFER = FALSE;
            break;

        /* uart2 */
        case 1:
            UART2_FAILED_RECIEVE_WRITE_TO_INPUT_BUFFER = FALSE;
            break;

        /* uart1 */
        default:
            UART1_FAILED_RECIEVE_WRITE_TO_INPUT_BUFFER = FALSE;
            break;

    }/* switch */



}
/*----------------------------------------------------------------------------*/
IDEF void Reset_DUART_Line_Status_IT(Uint16 Uart)
{
    extern far volatile BOOL UART1_LINE_STATUS_INTERRUPT_ERROR;
    extern far volatile BOOL UART2_LINE_STATUS_INTERRUPT_ERROR;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            UART1_LINE_STATUS_INTERRUPT_ERROR = FALSE;
            break;

        /* uart2 */
        case 1:
            UART2_LINE_STATUS_INTERRUPT_ERROR = FALSE;
            break;

        /* uart1 */
        default:
            UART1_LINE_STATUS_INTERRUPT_ERROR = FALSE;
            break;

    }/* switch */


}
/*----------------------------------------------------------------------------*/
IDEF void Reset_DUART_Modem_Status_IT(Uint16 Uart)
{
    extern far volatile BOOL UART1_MODEM_STATUS_INTERRUPT_ERROR;
    extern far volatile BOOL UART2_MODEM_STATUS_INTERRUPT_ERROR;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            UART1_MODEM_STATUS_INTERRUPT_ERROR = FALSE;
            break;

        /* uart2 */
        case 1:
            UART2_MODEM_STATUS_INTERRUPT_ERROR = FALSE;
            break;

        /* uart1 */
        default:
            UART1_MODEM_STATUS_INTERRUPT_ERROR = FALSE;
            break;

    }/* switch */

}
/*----------------------------------------------------------------------------*/
IDEF void Reset_DUART_Tx_IT_turned_off_by_ISR(Uint16 Uart)
{
    extern far volatile BOOL UART1_TRANSMIT_IT_WAS_TURNED_OFF;
    extern far volatile BOOL UART2_TRANSMIT_IT_WAS_TURNED_OFF;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            UART1_TRANSMIT_IT_WAS_TURNED_OFF = FALSE;
            break;

        /* uart2 */
        case 1:
            UART2_TRANSMIT_IT_WAS_TURNED_OFF = FALSE;
            break;

        /* uart1 */
        default:
            UART1_TRANSMIT_IT_WAS_TURNED_OFF = FALSE;
            break;

    }/* switch */


}
/*----------------------------------------------------------------------------*/
IDEF void Reset_DUART_RECIEVED_TX_BREAK(Uint16 Uart)
{
    extern far volatile BOOL UART1_RECIEVED_TX_BREAK;
    extern far volatile BOOL UART2_RECIEVED_TX_BREAK;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            UART1_RECIEVED_TX_BREAK = FALSE;
            break;

        /* uart2 */
        case 1:
            UART2_RECIEVED_TX_BREAK = FALSE;
            break;

        /* uart1 */
        default:
            UART1_RECIEVED_TX_BREAK = FALSE;
            break;

    }/* switch */


}
/*----------------------------------------------------------------------------*/
IDECL BOOL Is_DUART_CTS_Active(Uint16 Uart)
{
	// Return Current DSR Status
	return ( (DUART_Get_MSR(Uart) >> 4 ) & 1 );
}
/*----------------------------------------------------------------------------*/
IDECL BOOL Is_DUART_DSR_Active(Uint16 Uart)
{
	// Return Current DSR Status
	return ( (DUART_Get_MSR(Uart) >> 5 ) & 1 );
}

/*----------------------------------------------------------------------------*/
IDECL BOOL Is_DUART_DCD_Active(Uint16 Uart)
{
	// Return Current DCD Status
	return ( (DUART_Get_MSR(Uart) >> 7 ) & 1 );
}

/*----------------------------------------------------------------------------*/
IDECL BOOL Is_DUART_RI_Active(Uint16 Uart)
{
	// Return Current RI Status
	return ( (DUART_Get_MSR(Uart) >> 6 ) & 1 );
}

/*----------------------------------------------------------------------------*/
IDECL void DUART_Set_DTR(Uint16 Uart, BOOL State)
{
    Uint16 CurrentMCR;

    /* read current MCR */
    CurrentMCR = DUART_Get_MCR(Uart);

	/**
	    Bit to set to activate the DTR line.
	*/
	if (State == TRUE)
	{
	    CurrentMCR |= 0x0001;
	}
	else
	{
    	CurrentMCR &= 0xFFFE;
	}

    /* write back updated value */
    DUART_Write_MCR(Uart, CurrentMCR);
}

/*----------------------------------------------------------------------------*/
IDECL void DUART_Set_RTS(Uint16 Uart, BOOL State)
{
    Uint16 CurrentMCR;

    /* read current MCR */
    CurrentMCR = DUART_Get_MCR(Uart);

	/**
	    Bit to set to activate the RTS line.
	*/
	if (State == TRUE)
	{
	    CurrentMCR |= 0x0002;
	}
	else
	{
    	CurrentMCR &= 0xFFFD;
	}

    /* write back updated value */
    DUART_Write_MCR(Uart, CurrentMCR);

}

/*----------------------------------------------------------------------------*/

#ifdef TESTING

IDEF BOOL Is_DUART_TESTING_BUFFER_OVERFLOW(Uint16 Uart)
{
    extern far volatile BOOL UART1_TESTING_BUFFER_OVERFLOW;
    extern far volatile BOOL UART2_TESTING_BUFFER_OVERFLOW;
    BOOL FLAG;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            FLAG = UART1_TESTING_BUFFER_OVERFLOW;
            break;

        /* uart2 */
        case 1:
            FLAG = UART2_TESTING_BUFFER_OVERFLOW;
            break;

        /* uart1 */
        default:
            FLAG = UART1_TESTING_BUFFER_OVERFLOW;
            break;

    }/* switch */

    return FLAG;

}
IDEF BOOL Is_DUART_TESTING_BUFFER_UNDERFLOW(Uint16 Uart)
{
    extern far volatile BOOL UART1_TESTING_BUFFER_UNDERFLOW;
    extern far volatile BOOL UART2_TESTING_BUFFER_UNDERFLOW;
    BOOL FLAG;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            FLAG = UART1_TESTING_BUFFER_UNDERFLOW;
            break;

        /* uart2 */
        case 1:
            FLAG = UART2_TESTING_BUFFER_UNDERFLOW;
            break;

        /* uart1 */
        default:
            FLAG = UART1_TESTING_BUFFER_UNDERFLOW;
            break;

    }/* switch */

    return FLAG;

}
IDEF BOOL Is_DUART_TESTING_BUFFER_ERROR(Uint16 Uart)
{
    extern far volatile BOOL UART1_TESTING_BUFFER_ERROR;
    extern far volatile BOOL UART2_TESTING_BUFFER_ERROR;
    BOOL FLAG;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            FLAG = UART1_TESTING_BUFFER_ERROR;
            break;

        /* uart2 */
        case 1:
            FLAG = UART2_TESTING_BUFFER_ERROR;
            break;

        /* uart1 */
        default:
            FLAG = UART1_TESTING_BUFFER_ERROR;
            break;

    }/* switch */

    return FLAG;

}
IDEF void Reset_DUART_TESTING_BUFFER_OVERFLOW(Uint16 Uart)
{
    extern far volatile BOOL UART1_TESTING_BUFFER_OVERFLOW;
    extern far volatile BOOL UART2_TESTING_BUFFER_OVERFLOW;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            UART1_TESTING_BUFFER_OVERFLOW = FALSE;
            break;

        /* uart2 */
        case 1:
            UART2_TESTING_BUFFER_OVERFLOW = FALSE;
            break;

        /* uart1 */
        default:
            UART1_TESTING_BUFFER_OVERFLOW = FALSE;
            break;

    }/* switch */

}
IDEF void Reset_DUART_TESTING_BUFFER_UNDERFLOW(Uint16 Uart)
{
    extern far volatile BOOL UART1_TESTING_BUFFER_UNDERFLOW;
    extern far volatile BOOL UART2_TESTING_BUFFER_UNDERFLOW;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            UART1_TESTING_BUFFER_UNDERFLOW = FALSE;
            break;

        /* uart2 */
        case 1:
            UART2_TESTING_BUFFER_UNDERFLOW = FALSE;
            break;

        /* uart1 */
        default:
            UART1_TESTING_BUFFER_UNDERFLOW = FALSE;
            break;

    }/* switch */

}
IDEF void Reset_DUART_TESTING_BUFFER_ERROR(Uint16 Uart)
{
    extern far volatile BOOL UART1_TESTING_BUFFER_ERROR;
    extern far volatile BOOL UART2_TESTING_BUFFER_ERROR;

    switch(Uart)
    {
        /* uart1 */
        case 0:
            UART1_TESTING_BUFFER_ERROR = FALSE;
            break;

        /* uart2 */
        case 1:
            UART2_TESTING_BUFFER_ERROR = FALSE;
            break;

        /* uart1 */
        default:
            UART1_TESTING_BUFFER_ERROR = FALSE;
            break;

    }/* switch */


}
#endif /* TESTING */
/*************************************************************/
#endif /* USEDEFS */


#endif /* BSL_DUART_SUPPORT */
#endif /* _BSL_DUART_H_ */
/******************************************************************************\
* End of bsl_duart.h
\******************************************************************************/
