/******************************************************************************\
* Copyright (C) 2008 by RTD Embedded Technologies, Inc.   All rights reserved.
* Confidential and Proprietary, Not for Public Release
*------------------------------------------------------------------------------
* PROJECT.......... Board Support Library for SPM64xx (176430, 173431, 186420)
* CONTENT.......... Main source file of Multi-Threaded BSL
\******************************************************************************/

/******************************************************************************\
*  NOTE:  This file should not be optimized.  Doing such can cause various
*         algorithms that do multiple read and writes to different memory
*         locations to be optimized to the point of breaking the code.
*         ie:  GetSDRAMSize()
\******************************************************************************/


#define _BSL_BSL_MOD_

#include <std.h>
#include <bsl.h>

#include <csl.h>
#include <csl_emifa.h>
#include <csl_emifb.h>
#include <csl_pci.h>
#include <csl_irq.h>
#include <c6x.h>        // for SAVE_AMR, RESTORE_AMR, SAVE_SAT, RESTORE_SAT

#include <bsl_umisc.h>
#include <bsl_uint.h>


/******************************************************************************\
*                         L O C A L   S E C T I O N
\******************************************************************************/


/******************************************************************************\
* static macro declarations
\******************************************************************************/

// offsets of HW info variables in EEPROM
#define EEVAR_DSPNOMSP_OFFS         14  // DSP processor's nominal speed in MHz
#define EEVAR_FLASHSZ_OFFS          15  // size of the installed Flash memory in MB
#define EEVAR_SDRAMSZ_OFFS          16  // size of the installed SDRAM memory in MB

// EMIF Configuration for SPM6420/6430 board with 600 MHz DSP processor
// These are default configuration values, and are overridden in SPM6400_EMIF_A_INIT()
//  and SPM6400_EMIF_B_INIT().
#define CFG_EMIFB_GBLCTL_600            0x00092024
#define CFG_EMIFB_CECTL1_600            0x33514D10
#define CFG_EMIFB_CECTL2_600            0x33514D10
#define CFG_EMIFB_CECTL3_600            0x11814610

#if (BOARD_SPM6420_SUPPORT)
#define CFG_EMIFB_CECTL0_600            0x13410D10
#endif

#if (BOARD_SPM6430_SUPPORT)
#define CFG_EMIFB_CECTL0_600            0x13410D10
#endif

#if (BOARD_SPM6431_SUPPORT)
#define CFG_EMIFB_CECTL0_600            0x83F8CF1F
#endif


#define MEM_RANGE_START_FLASH           0x67E00000
#define MEM_RANGE_START_SDRAM           0x80000000
#define MEM_FPGA_SWITCH_READ            0x6C080016
#define MEM_FPGA_LED                    0x6C08003A
#define EMIF_SDRAM_CTL_MASK             0x7FFFF003
#define EMIF_SDRAM_CTL_DEFAULT          0x0248F000
//
// This function sets the size of SDRAM in MB (32,64,128,256,512)
// led will turn on solid if there is an error
//
void GetSDRAMSize ()
{
    Uint32 EMIFA_temp;
    Uint32 temp00000000, temp00001000, temp00000800;
    Uint32 temp08000000, temp04000000, temp02000000;
    Uint32 temp10000000, temp10000008;
    int SDRAMSize = 0;

    // EMIF A SDRAM SIZES
    //-------------------
    //
    //  size    rows    banks   columns     cs  EMIFA_SDRAMCTL  EMIFA_CE1
    // -----    ----    -----   -------     --  --------------  ----------
    //  32MB    4k(12)    4      256(8)     1     0x57116000    0xFFFFFFDF
    // 128MB    8k(13)    4      512(9)     1     0x53116000    0xFFFFFFDF
    // 256MB    8k(13)    4      1k(10)     1     0x5B116000    0xFFFFFFDF
    // 512MB    8k(13)    4      1k(10)     2     0x5B116000    0xFFFFFFDF
    //
    // SDRAM Control
    //  b30         SDBSZ   1 = two bank select pins
    //  b[29:28]    SDRSZ   01 = 12 row address pins, 10 = 13 row address pins
    //  b[27:26]    SDCSZ   01 = 8 column address pins , 00 = 9 column address pins
    //  b25         RFEN    1 = refresh is enabled
    //  b24         INIT    1 = initialize SDRAM (automatically reset when finished)
    //  b[23:20]    TRCD    1 = ( (trcd = 20ns)/(10ns) ) -1
    //  b[19:16]    TRP     1 = ( (trp = 15ns)/(10ns) ) -1
    //  b[15:12]    TRC     6 = ( (trc = 66ns)/(10ns) ) - 1
    //  b0          SLFRFR  0 = self refresh is disabled

    //*(int *)EMIFA_SDRAMCTL = 0x57116000;  // For  32MB
    //*(int *)EMIFA_SDRAMCTL = 0x53116000;  // For 128MB
    //*(int *)EMIFA_SDRAMCTL = 0x5B116000;  // For 256MB/512MB


    // ADDRESS LINE ASSIGNMENTS
    //
    //  size    Internal Address
    // ------   ----------------
    //  32MB    EEEE...B Brrrrrrr rrrrrCCC CCCCC...
    // 128MB    EEEE.rBB rrrrrrrr rrrrCCCC CCCCC...
    // 256MB    EEEErBBr rrrrrrrr rrrCCCCC CCCCC...
    //
    //  E = Chip Enable Decode
    //  B = Bank Address Line
    //  r = Row Address Line
    //  C = Column Address Line
    //  . = Not Used or not connected to chip
    //
    // The internal address is sent to the SDRAM chip in two cycles.  First the ROW address is sent.  It is
    // the internal address shifted right by the number of column bit, i.e. the row and the bank address.
    // Then the COLUMN addres is sent, which is just the column address bits.  The BANK address bits are
    // physically always attached to the same address pins; therefore we always set the number of rows address
    // lines to 12.  The 13th address line then appears above the bank addresses.


    // First, set the SDRAM to the largest size, two banks of 512Mb devices = 512 MB. row=13, col=10, b=2
    EMIFA_temp = EMIFA_RGET(SDCTL);
    EMIFA_temp &= 0x80000FFE;
    EMIFA_temp |= 0x5B116000; // Change to initiate the SDRAM controller.
    EMIFA_RSET(SDCTL, EMIFA_temp);

    // FIND NUMBER OF COLUMN ADDRESS LINES.

    //back up all SDRAM locations that will be used
    temp00000000 = *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x00000000);
    temp00001000 = *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x00001000);
    temp00000800 = *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x00000800);

    // Write a value of 10 to SDRAM offset 0x00000000.  This will not be overwritten if all 10 column address
    // lines are used by the SDRAM chip.
    *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x00000000) = 10;

    // Write a value of 9 to SDRAM offset 0x00001000 (10th column bit set).  If there are less than 10 column
    // address lines used, this will be written to offset 0x00000000.
    *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x00001000) = 9;

    // Write a value of 8 to SDRAM offset 0x00000800 (9th column bit set).  If there are less than 9 column
    // address lines used, this will be written to offset 0x00000000.
    *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x00000800) = 8;


    // Read from SDRAM offset 0x00000000 to see which value was written there last.  Check to make sure it is
    // a valid number of columns, and set the SDRAMCTL register accordingly.

    EMIFA_temp = EMIFA_RGET(SDCTL);
    EMIFA_temp &= 0xF3FFFFFF;

    switch (*(Uint32 *)(MEM_RANGE_START_SDRAM))
    {
        case 8: // 8 column address lines
            EMIFA_temp |= 0x04000000;
            break;

        case 9: // 9 column address lines
            EMIFA_temp |= 0x00000000;
            break;

        case 10: // 10 column address lines
            EMIFA_temp |= 0x08000000;
            break;

        default: // Something went horribly wrong
            *(Uint16 *)(MEM_FPGA_LED) = 0x0001;//cause led to stay on solid
    }

    //restore all SDRAM locations that were used
    *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x00000000) = temp00000000;
    *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x00001000) = temp00001000;
    *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x00000800) = temp00000800;

    EMIFA_RSET(SDCTL, EMIFA_temp);

    // FIND MEMORY SIZE IN FIRST BANK

    //back up all SDRAM locations that will be used
    temp00000000 = *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x00000000);
    temp08000000 = *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x08000000);
    temp04000000 = *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x04000000);
    temp02000000 = *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x02000000);
    temp10000000 = *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x10000000);
    temp10000008 = *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x10000008);

    // Write a value of 256 to SDRAM offset 0x00000000.  This will not be overwritten if there is 256MB in first bank
    *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x00000000) = 256;

    // Write a value of 128 to SDRAM offset 0x08000000 (128MB boundary).  If there is less than 256MB in the first
    // bank, this will be written to offset 0x00000000.
    *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x08000000) = 128;

    // Write a value of 64 to SDRAM offset 0x04000000 (64MB boundary).  If there is less than 64MB in the first
    // bank, this will be written to offset 0x00000000.
    *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x04000000) = 64;

    // Write a value of 32 to SDRAM offset 0x02000000 (32MB boundary).  If there is less than 32MB in the first
    // bank, this will be written to offset 0x00000000.
    *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x02000000) = 32;

    // Read from SDRAM offset 0x00000000 to see which value was written there last.  Check to make sure it is
    // a valid memory size.
    switch (SDRAMSize = *(Uint32 *)(MEM_RANGE_START_SDRAM))
    {
        case 256:
            // Now we also check to see if the second bank is installed.  If it is not, we will read 0xFFFFFFFF.
            *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x10000000) = 0xA5A5A5A5;
            *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x10000008) = 0x5A5A5A5A;
            if ( (*(Uint32 *)(MEM_RANGE_START_SDRAM + 0x10000000) == 0xA5A5A5A5) &&
                (*(Uint32 *)(MEM_RANGE_START_SDRAM + 0x10000008) == 0x5A5A5A5A) )
            {
                // Memory exists in the second bank.  Assume it is the same size as the first bank.
                SDRAMSize *= 2;
            }

        case 128:
        case 64:
        case 32:
            // Memory size is detected and setup.
            BSL_boardDescriptor.sdramSize = SDRAMSize;
            break;

        default: // Something went horribly wrong
            BSL_boardDescriptor.sdramSize = 32;
            *(Uint16 *)(MEM_FPGA_LED) = 0x0001;//cause led to stay on solid
    }

    //restore all SDRAM locations that will be used
    *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x00000000) = temp00000000;
    *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x08000000) = temp08000000;
    *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x04000000) = temp04000000;
    *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x02000000) = temp02000000;
    *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x10000000) = temp10000000;
    *(Uint32 *)(MEM_RANGE_START_SDRAM + 0x10000008) = temp10000008;

}

//
// Sets the size of the flash chip in MB (2 or 4)
// led will turn on solid if there is an error
//
void GetFlashSize ()
{
    Uint16 FlashManuf;
    Uint16 FlashDevice;

    // Check to see if flash write is enabled
    if ( *(Uint16 *)(MEM_FPGA_SWITCH_READ) & 0x04 )
    {
        //Flash write is disabled
        //Place in the 2MB default value
        BSL_boardDescriptor.flashSize = 2;
        BSL_boardDescriptor.flashAutoDetected = 0;
        return;
    }

    // Reset Flash Chip
    *(Uint16 *)(MEM_RANGE_START_FLASH + 0x0000) = 0x00F0;

    // Enter Autoselect
    *(Uint16 *)(MEM_RANGE_START_FLASH + 0x0AAA) = 0x00AA;
    *(Uint16 *)(MEM_RANGE_START_FLASH + 0x0554) = 0x0055;
    *(Uint16 *)(MEM_RANGE_START_FLASH + 0x0AAA) = 0x0090;

    // Get Flash Info
    FlashManuf = *(Uint16 *)(MEM_RANGE_START_FLASH + 0x0000);
    FlashDevice = *(Uint16 *)(MEM_RANGE_START_FLASH + 0x0002);

    // Reset Flash Chip
    *(Uint16 *)(MEM_RANGE_START_FLASH + 0x0000) = 0x00F0;

    switch (FlashManuf)
    {
        case 0x0001: // AMD/Spansion
            switch (FlashDevice)
            {
                case 0x2249: // 2MB AM29LV160DB
                    BSL_boardDescriptor.flashSize = 2;
                    BSL_boardDescriptor.flashAutoDetected = 1;
                    return;

                case 0x22F9: // 4MB S29AL032DB
                    BSL_boardDescriptor.flashSize = 4;
                    BSL_boardDescriptor.flashAutoDetected = 1;
                    return;

            }
            break;
    }

    // We are still here, so something must have gone wrong

    BSL_boardDescriptor.flashSize = 2;
    BSL_boardDescriptor.flashAutoDetected = 0;
    *(Uint16 *)(MEM_FPGA_LED) = 0x0001;//cause led to stay on solid
    return;

}

//
// Get the frequency of the DSP processor
// led will turn on solid if there is an error
//
void GetBoardSpeed ()
{
    int speed;
    int BoardSpeed;
    int DSPSpeeds[] = {500,600,720,850,1000};

   //Get the board frequency from the fpga
    BoardSpeed = UMISC_RGET(DSPSPEED);

    //Catorgize it into one of the speeds listed int DSPSpeeds
    //if it is with 50Hz of any of them
    BSL_boardDescriptor.dspNomSpeed = 0;
    for (speed = 0;speed < 5;speed++)
    {
        if ( ((BoardSpeed-50) < DSPSpeeds[speed]) && ((BoardSpeed+50) > DSPSpeeds[speed]))
        {
            BSL_boardDescriptor.dspNomSpeed = DSPSpeeds[speed];
            break;
        }
    }

    //if it isn't within 50Hz of the frequencies, give it the lowest frequency
    if (BSL_boardDescriptor.dspNomSpeed == 0)
    {
        //if there is an error just select the lowest speed
        BSL_boardDescriptor.dspNomSpeed = DSPSpeeds[0];
        *(Uint16 *)(MEM_FPGA_LED) = 0x0001;//cause led to stay on solid

    }


}

// function to configure the EMIF_A
SPM6400_EMIF_A_INIT()
{
    Uint32 EMIFA_temp;

    EMIFA_temp = EMIFA_RGET(GBLCTL);
    EMIFA_temp &= 0xFFF0D007;
    EMIFA_temp |= 0x000820A0;
    EMIFA_RSET(GBLCTL, EMIFA_temp);

    // CECTL0/1
    EMIFA_RSET(CECTL0, 0xFFFFFFDF); // No reserved bits, just write a value
    EMIFA_RSET(CECTL1, 0xFFFFFFDF);

    // SDTIM
    EMIFA_temp = EMIFA_RGET(SDTIM);
    EMIFA_temp &= 0xFC000000;
    EMIFA_temp |= 0x00000190;
    EMIFA_RSET(SDTIM, EMIFA_temp);

    // SDEXT
    EMIFA_temp = EMIFA_RGET(SDEXT);
    EMIFA_temp &= 0xFFE00000;
    EMIFA_temp |= 0x00055CA8;
    EMIFA_RSET(SDEXT, EMIFA_temp);


}

// function to configure the EMIF_B
SPM6400_EMIF_B_INIT()
{
    Uint32 EMIFB_temp;

#if (BOARD_SPM6420_SUPPORT)
    Uint32 PBus_temp;
    Uint32 Timer_Tick_Count;
    Uint16 pbctl_temp;
    TIMER_Handle hTimer;
#endif //BOARD_SPM6420_SUPPORT

    /*
        Do some basic EMIFB setup so the platformBus can be configured
    */

    // GBLCTL
    EMIFB_temp = EMIFB_RGET(GBLCTL);
    EMIFB_temp &= 0xFFF0D007;
    EMIFB_temp |= 0x000920A0;
    EMIFB_RSET(GBLCTL, EMIFB_temp);

    // CECTL0-3
    EMIFB_RSET( CECTL0, CFG_EMIFB_CECTL0_600);  // No reserved bits, just write a value
    EMIFB_RSET( CECTL1, CFG_EMIFB_CECTL1_600);
    EMIFB_RSET( CECTL2, CFG_EMIFB_CECTL2_600);
    EMIFB_RSET( CECTL3, CFG_EMIFB_CECTL3_600);

#if (BOARD_SPM6420_SUPPORT)

    // Setup platformBus and put it into Reset
    // PlatformBus must be in Reset for the bus speed to be changed.
    pbctl_temp = UMISC_RGET(PBCTL0);
    UMISC_RSET(PBCTL1, 0x0100);

    UMISC_RSET(PBCTL0, pbctl_temp | 0x8000);

    /*
        Change platformBus speed to 50 MHz

        We must do this without touching any of the reserved bits in the
        GBLCTL register.  Modifiying the reserved bits can crash the 1GHz
        version of the DSP.
    */

    // Read in the current EMIFB GBLCTL register
    PBus_temp = EMIFB_RGET(GBLCTL);

    // Wipe out all the bits we want to change, without affecting any reserved bits.
    PBus_temp &= 0xFFF0D007;

    // Set the bits
    PBus_temp |= 0x000520A0;

    // Write the modified value
    EMIFB_RSET(GBLCTL, PBus_temp);



    /*
        The platformBus must be held in reset for 50us after a speed
        change.  Since there is no udelay() function in the DSP's C runtime
        library, we have to use one of the onboard timers for the delay.

        The timers count based on the DSP clock speed.  Therefore, we
        can't delay for a fixed number of timer ticks.  We already know
        the DSP frequency in MHz, so we can use it to calculate the
        number of timer ticks.

        This value has to be divided by 8, because the timers use 1/8 the
        DSP clock.
    */
    Timer_Tick_Count = ((BSL_boardDescriptor.dspNomSpeed * 50) / 8);

    hTimer = TIMER_open(TIMER_DEVANY, TIMER_OPEN_RESET);

    TIMER_reset(hTimer);

    // This will configure and start the timer in one operation.
    TIMER_configArgs(hTimer, 0x000002C0, 0xFFFFFFFF, 0x00000000);

    while( TIMER_getCount(hTimer) < Timer_Tick_Count);


    // Bring platformBus out of reset
    UMISC_RSET(PBCTL0, (pbctl_temp & 0x40FC)|(0x0220 & ~0x40FC));

    // Delay for another 50us to make sure the platformBus made it all the
    // way out of reset before returning.
    TIMER_reset(hTimer);
    TIMER_configArgs(hTimer, 0x000002C0, 0xFFFFFFFF, 0x00000000);
    while( TIMER_getCount(hTimer) < Timer_Tick_Count);

    // Reset and close the timer when we're done, so it is restored to a
    // known state.
    TIMER_reset(hTimer);
    TIMER_close(hTimer);

#endif //BOARD_SPM6420_SUPPORT

}


/******************************************************************************\
* static typedef declarations
\******************************************************************************/


/******************************************************************************\
* static function declarations
\******************************************************************************/

#ifdef __cplusplus
extern "C" {
#endif

// Initialization functions of COMM and FLASH modules
extern far void COMM_init();    // defined in bsl_comm.c
extern far void FLASH_init();   // defined in bsl_flash.c

#if (BSL_DUART_SUPPORT)
extern far void DUART_init();   // defined in bsl_duart.c
#endif /* BSL_DUART_SUPPORT */

// Callback Interrupt Service Routine of COMM and FLASH modules
extern far void COMM_ISR_CB();  // defined in bsl_comm.c
extern far void FLASH_ISR_CB(); // defined in bsl_flash.c

#if (BSL_DUART_SUPPORT)
extern far void DUART_ISR_CB(Uint16 Uart);  // defined in bsl_duart.c
#endif /* BSL_DUART_SUPPORT */

#ifdef __cplusplus
}
#endif  // extern "C" {


/******************************************************************************\
* static variable definitions
\******************************************************************************/


/******************************************************************************\
* static function definitions
\******************************************************************************/

void BSL_ISR_DISPATCHER_CB()    // DO NOT USE interrupt keyword!!
{
    // In this ISR, there are assembly instructions whose results depend on
    // AMR and SAT. That is, the AMR and SAT have to be saved and restored.
    Uint32  temp_AMR, temp_SAT; // to save the AMR and SAT

    // save AMR and SAT
    SAVE_AMR(temp_AMR);
    SAVE_SAT(temp_SAT);

    do
    {
        if( UINT_FGETSQ( STATUS, P2LDBIA, ACTIVE ) ) COMM_ISR_CB();
        if( UINT_FGETSQ( STATUS, FRYBYIA, ACTIVE ) ) FLASH_ISR_CB();

#if (BSL_DUART_SUPPORT)
        if( UINT_FGETSQ( STATUS, PBINT6IA, ACTIVE ) ) DUART_ISR_CB(0);
        if( UINT_FGETSQ( STATUS, PBINT7IA, ACTIVE ) ) DUART_ISR_CB(1);
#endif /* BSL_DUART_SUPPORT */

    }
#if (BSL_DUART_SUPPORT)
    while( UINT_RGET(STATUS) & (_UINT_STATUS_P2LDBIA_ACTIVE_SV | _UINT_STATUS_FRYBYIA_ACTIVE_SV
             | _UINT_STATUS_PBINT6IA_ACTIVE_SV | _UINT_STATUS_PBINT7IA_ACTIVE_SV) );
#else
    while( UINT_RGET(STATUS) & (_UINT_STATUS_P2LDBIA_ACTIVE_SV | _UINT_STATUS_FRYBYIA_ACTIVE_SV) );
#endif /* BSL_DUART_SUPPORT */

    // restore AMR and SAT
    RESTORE_AMR(temp_AMR);
    RESTORE_SAT(temp_SAT);
}


/******************************************************************************\
*                        G L O B A L   S E C T I O N
\******************************************************************************/


/******************************************************************************\
* global variable definitions
\******************************************************************************/

BSL_BoardDescriptor BSL_boardDescriptor;


/******************************************************************************\
* global function definitions
\******************************************************************************/

/*----------------------------------------------------------------------------*/
void BSL_init()
{


    Uint16 CopyOfEIT7SRC;
    Uint32 EMIFA_temp;
    volatile Uint32 trctl;

    //------------------------------------------------------
    // initializing of the DSP interrupt conditions
    //------------------------------------------------------
    IER = 0x0003;   // enable Reset and NMI only
    ICR = 0xFFFF;   // clear all pending ITs
    CSR |= 1;       // enable Global Interrupt Enable (GIE)

    //------------------------------------------------------
    // initializing of the Chip Support Library
    //------------------------------------------------------

    CSL_init();

    //------------------------------------------------------
    // Set priority for HPI (PCI) Accesses into DSP Memory
    //------------------------------------------------------

    /*
        Increase HPI to Urgent priority.  This is intended to prevent a
        deadlock condition where the DSP and the CPU keep attempting to
        poll the same register in memory and collide.

        By putting the HPI to the higest priority possible, we allow the
        CPU to bully the DSP.  This makes sense in a CoProcessor type system,
        where the CPU "owns" the bus.

        HPI priority is controlled via the TRCTL register.  The procedure for
        changing HPI priority is documented in SPRU578C.  Note that we deviate
        from the procedure a bit.  We do not wait for EDMAs to finish, and we
        do not flush the L2 cache.  Since we are very early in board init, we
        do not have to worry about these things so much.
    */

    trctl = *(Uint32 *) _PCI_TRCTL_ADDR;
    trctl |= 0x00000100; // Set bit 8 (TRSTALL) to stall the host
    trctl &= 0xFFFFFFCF; // Set the PRI bits to Urgent (00)
    *(Uint32 *)(_PCI_TRCTL_ADDR) = trctl;

    trctl = *(Uint32 *) _PCI_TRCTL_ADDR;
    trctl &= 0xFFFFFEFF; // Clear TRSTALL to let transactions continue
    *(Uint32 *)(_PCI_TRCTL_ADDR) = trctl;


    //------------------------------------------------------
    // reading HW information
    //------------------------------------------------------

    // pre-initializing EMIFB for the fastest board to reach the FPGA registers
    EMIFB_RSET( GBLCTL, CFG_EMIFB_GBLCTL_600 );
    EMIFB_RSET( CECTL3, CFG_EMIFB_CECTL3_600 );

    // set "DSP is in use" flag
    UMISC_RFSETS( DHCOMM, DSPINUSE, INUSE );

    //Determine the Nominal DSP speed
    //This must be done before the call to SPM6400_EMIF_B_INIT
    //so that we can correctly set are delays in the function.
    //We will only detect the board speed if it has not previously
    //been detected.
    GetBoardSpeed();

    /*************************************************************************
    Below there are two methods for setting the BSL_boardDescriptor
    variables.  Are default library does auto detection.  The alternative
    is dead setting all of these values.  To deadset all values...
        -Comment out the above call to GetBoardSpeed()
        -Comment out the code below which auto detects SDRAM
        -Uncomment the code that hard codes these data and set the correct
        values for your board
        -Comment ou the call below to GetFlashSize()
    *************************************************************************/

    //Determine the SDRAM size
    //We will only detect the SDRAM size if the SDRAM Control Register
    //still has it's default value from reset, because otherwise we
    //have already set up the SDRAM
    EMIFA_temp = EMIFA_RGET(SDCTL);
    if((EMIFA_temp & EMIF_SDRAM_CTL_MASK) == EMIF_SDRAM_CTL_DEFAULT)
    {

        //initialize the EMIF registers
        SPM6400_EMIF_A_INIT();

        //Determine the SDRAM size
        GetSDRAMSize();

    }

    //Always initialize the EMIFB registers everytime we are called
    SPM6400_EMIF_B_INIT();


    // Alternative Method (commented out, but left in for reference)

    //BSL_boardDescriptor.dspNomSpeed   = 600;
    //BSL_boardDescriptor.flashSize = 4;
    //BSL_boardDescriptor.flashAutoDetected = 0; //Not automatically detected
    //BSL_boardDescriptor.sdramSize = 128;
    //SPM6400_EMIF_A_INIT();
    // set the SDRAM control register
    //EMIFA_temp = EMIFA_RGET(SDCTL);
    //EMIFA_temp &= 0x80000FFE;
    //EMIFA_temp |= 0x5A116000;
    //              8 column         9 column      10 column
    //EMIFA_temp |= 0x04000000; //or 0x00000000 or 0x08000000
    //EMIFA_RSET(SDCTL, EMIFA_temp);
    //SPM6400_EMIF_B_INIT();


    //Determine the flash size
    //This must be done after the call to SPM6400_EMIF_B_INIT
    //so that we can use EMIFB to talk to the flash chip
    //We will only detect the flash size if it has not previously
    //been detected.
    GetFlashSize();

    //------------------------------------------------------
    // initializing Communication and Flash Controller
    // after the initialization of EMIF
    //------------------------------------------------------

    IRQ_disable(IRQ_EVT_EXTINT7);

    // ExtInt7 must be a rising edge sensitive pin (EXTPOL.XIP7 = 0)
    IRQ_RSET( EXTPOL, IRQ_RGET(EXTPOL) & (~0x8) );

    // Disable every sources for ExtInt7 pin
    // The proper sources will be set by the COMM_init() and by the FLASH_init()
    UINT_RSET( EIT7SRC, 0 );

    COMM_init();
    FLASH_init();
#if (BSL_DUART_SUPPORT)
    DUART_init();
#endif /* BSL_DUART_SUPPORT */

    IRQ_clear(IRQ_EVT_EXTINT7);
    IRQ_enable(IRQ_EVT_EXTINT7);

    //------------------------------------------------------
    // Incase an interrupt has already occured before the
    // Source 7 IRQ was enable, we will force an edge
    //------------------------------------------------------
    // copy current value
    CopyOfEIT7SRC = UINT_RGET( EIT7SRC);
    // turn off all sources( forces interupt line low if it
    // wasn't already low
    UINT_RSET( EIT7SRC, 0 );
    // put interrupt line back to its previous state
    UINT_RSET( EIT7SRC, CopyOfEIT7SRC );

    //------------------------------------------------------
    // initializing PCI Bus peripheral Controller
    //------------------------------------------------------

    IRQ_disable(IRQ_EVT_DSPINT);

    PCIBUS_init();

    // enables DSPINT's interrupt on Int13 by default
    // for the PCI peripheral interrupts
    IRQ_clear(IRQ_EVT_DSPINT);
    IRQ_enable(IRQ_EVT_DSPINT);
}


/*----------------------------------------------------------------------------*/

/******************************************************************************\
* End of bsl_bsl.c
\******************************************************************************/

