/******************************************************************************\
* Copyright (C) 2004 by RTD Embedded Technologies, Inc.   All rights reserved.
* Confidential and Proprietary, Not for Public Release
*------------------------------------------------------------------------------
* PROJECT.......... Board Support Library for SPM6420
* VERSION.......... (Defined in README.TXT)
*------------------------------------------------------------------------------
* CONTENT.......... PCI Device Module of BSL
* FILENAME......... bsl_pdev.c
\******************************************************************************/
#define _BSL_PDEV_MOD_

#include <bsl_pdev.h>
#include <bsl_pcibus.h>


#if (BSL_PDEV_SUPPORT)
/******************************************************************************\
*                         L O C A L   S E C T I O N
\******************************************************************************/


/******************************************************************************\
* static macro declarations
\******************************************************************************/

// this many PCI devices can be opened at a time
#define PDEV_MAX_OPENED_PCI_DEVICES			4

// PCI config register number in a PCI device's header
#define PDEV_HDR_PCIIDR_REGOFFS32			0
#define PDEV_HDR_PCICRSR_REGOFFS32			1
#define PDEV_HDR_PCIBAR0_REGOFFS32			4
#define PDEV_HDR_PCIBAR1_REGOFFS32			5
#define PDEV_HDR_PCIBAR2_REGOFFS32			6
#define PDEV_HDR_PCIBAR3_REGOFFS32			7


/******************************************************************************\
* static typedef declarations
\******************************************************************************/

// structure to maintain the opened PCI devices
typedef struct {
	Uint32 vendorAndDeviceID;
	Uint32 ordinal;
} PDEV_OpenedDevices;


/******************************************************************************\
* static function declarations
\******************************************************************************/

Bool	PDEV_isOpened(Uint32 vendorAndDeviceID, Uint32 ordinal);
Uint32	PDEV_canBeOpened();
Uint32	PDEV_findDevice(Uint32 vendorAndDeviceID, Uint32 ordinal,
			PDEV_DeviceObj *deviceObj);
Uint32	PDEV_readPCIcfg(Uint32 reg, Uint32 dev, Uint32 *val);
Uint32	PDEV_writePCIcfg(Uint32 reg, Uint32 dev, Uint32 val);


/******************************************************************************\
* static variable definitions
\******************************************************************************/

// stores the opened PCI devices ({0,0} means, the 'slot' is empty)
PDEV_OpenedDevices _PDEV_openedDevices[PDEV_MAX_OPENED_PCI_DEVICES] = {
	{0, 0}, {0, 0}, {0, 0}, {0, 0}
};


/******************************************************************************\
* static function definitions
\******************************************************************************/

//==============================================================================
// Returns TRUE, if the given PCI device is opened currently.
//
// parameters:
//		vendorAndDeviceID	Vendor and Device ID of the queried PCI device.
//		ordinal				Ordinal number of the queried PCI device within the
//							set of the same Vendor and Device ID devices.
//
// returns:
//		TRUE, if the queried PCI device is opened currently.
//		Otherwise, FALSE.
//==============================================================================
Bool PDEV_isOpened(Uint32 vendorAndDeviceID, Uint32 ordinal)
{
	Uint32 dev;

	for( dev = 0; dev < PDEV_MAX_OPENED_PCI_DEVICES; dev++)
	{
		if( (_PDEV_openedDevices[dev].vendorAndDeviceID == vendorAndDeviceID)
			&& (_PDEV_openedDevices[dev].ordinal == ordinal) ) return TRUE;
	}
	return FALSE;
}

//==============================================================================
// Returns the index of the slot that can be used to store a new opened PCI
// device for maintain.
// If there is no more room, the returned value is greater than the greatest
// index.
//==============================================================================
Uint32 PDEV_canBeOpened()
{
	Uint32 dev;

	for( dev = 0; dev < PDEV_MAX_OPENED_PCI_DEVICES; dev++)
	{
		if( (_PDEV_openedDevices[dev].vendorAndDeviceID == 0)
			&& (_PDEV_openedDevices[dev].ordinal == 0) ) return dev;
	}

	/* there is no more place */
	return (PDEV_MAX_OPENED_PCI_DEVICES + 1);
}

//==============================================================================
// Attempts to find a PCI device and fills up a PDEV_DeviceObj structure to
// handle the found device.
//
// This function can find PCI device with 'function number' 0 only.
//
// parameters:
//		vendorAndDeviceID	Vendor and Device ID of the queried PCI device.
//							The PDEV_PCI_BOARD_RTD_x macros should be used.
//		ordinal				Ordinal number of the queried PCI device within the
//							set of the same Vendor and Device ID devices.
//							Must be at least 1.
//		pDeviceObj			Pointer to the PCI device descriptor to be filled
//							up, if the requested device has been found.
//
// returns:
//		PDEV_DEVICE_FOUND		All right. Device has been found.
//		PDEV_DEVICE_NOT_FOUND	No error, but the device has not been found.
//		PDEV_PCI_CONFIG_ERROR	PCI configuration cycle error has occured.
//==============================================================================
Uint32 PDEV_findDevice(Uint32 vendorAndDeviceID, Uint32 ordinal,
	PDEV_DeviceObj * pDeviceObj)
{
	Uint32	dev, ven_dev_id, ord;
	Uint32	pcicrsr;
	Uint32	bar;

	ord = 1;

	for( dev = 0; dev <= 15; dev++ )
	{
		// read PCI Configuration and check error
		if( PDEV_readPCIcfg(PDEV_HDR_PCIIDR_REGOFFS32,dev,&ven_dev_id)
			== PCIBUS_TOE_NO_ERROR )
		{
			// a PCI device has been found
			if( ven_dev_id == vendorAndDeviceID )
			{
				// an appropriate PCI device has been found according to the
				// given Vendor and Device IDs
				if( ord >= ordinal )
					// the ">=" is a protection for parameter 'ordinal' = 0
					// thus, the ordinal = 0 and 1 mean 1
				{
					// the required PCI device has been found

					// stores the location of the device
					// If later, a PCI config cycle has to be applied for the
					// opened device, you can use this device number.
					pDeviceObj->location = dev;

					pDeviceObj->barTypes = 0;

					// read PCICRSR reg.
					if( PDEV_readPCIcfg(
							PDEV_HDR_PCICRSR_REGOFFS32,dev,&pcicrsr)
						!= PCIBUS_TOE_NO_ERROR )
					{
						return PDEV_PCI_CONFIG_ERROR;
					}

					// clear PCICRSR reg's error flags and set flags to
					// response Memory and I/O cycles.
					pcicrsr |= 0xF8000007;

					// write PCICRSR reg. back
					if( PDEV_writePCIcfg(
							PDEV_HDR_PCICRSR_REGOFFS32,dev,pcicrsr)
						!= PCIBUS_TOE_NO_ERROR )
					{
						return PDEV_PCI_CONFIG_ERROR;
					}

					// read Base Address registers
					for( bar = 0; bar < PDEV_NUMBER_OF_BARS; bar++ )
					{
						if( PDEV_readPCIcfg(
								PDEV_HDR_PCIBAR0_REGOFFS32 + bar,
								dev, &(pDeviceObj->baseAddress[bar]) )
							!= PCIBUS_TOE_NO_ERROR )
						{
							return PDEV_PCI_CONFIG_ERROR;
						}

						// store the Type Of the base address (Memory or I/O)
						pDeviceObj->barTypes |=
							(pDeviceObj->baseAddress[bar] & 0xF) << (4 * bar);

						// make clear Base Address without control bits in the
						// lower bits
						pDeviceObj->baseAddress[bar] &= ~0xF;
					}

					// The requested PCI device has been found, the major
					// information about the device has been stored in the
					// pDeviceObj.
					return PDEV_DEVICE_FOUND;
				}
				// search the next device with the same Vendor and Device IDs
				ord++;
			}
			else
			{
				// NOT this device what we want to find
				// do nothing, go on
			}
		}
	}

	// The requested PCI device has not been found.
	return PDEV_DEVICE_NOT_FOUND;
}

//==============================================================================
// Initiates a PCI Configuration Read Cycle to read a PCI Configuration
// register of a PCI device.
//
// parameters:
//		reg			ordinal number of the register to be read
//		dev			PCI device number of the PCI deivce to be reached
//		val			pointer to an Uint32-type variable to be filled up with the
//					value of the read register
//
// returns:
//		error codes according to the PCIBUS_getError()
//		PCIBUS_TOE_NO_ERROR = All right.
//==============================================================================
Uint32 PDEV_readPCIcfg(Uint32 reg, Uint32 dev, Uint32 *val)
{
	PCIBUS_AccessReqObj	reqCfgCycle;
	SEM_Obj	sem;

	// init semaphore
	SEM_new( &sem, 0 );

	// make a request for the PCI Configuration Cycle
	PCIBUS_makeReq_Cfg_SEM(
		&reqCfgCycle, reg, 0, dev, val, PCIBUS_TRANSFER_FROM_PCI,
		PCIBUS_TOS_SEM_FOR_TSK, &sem);

	// reading
	PCIBUS_accessReq( &reqCfgCycle );

	// waiting for data transfer to complete
	SEM_pend( &sem, SYS_FOREVER );

	// check error
	return PCIBUS_getError( &reqCfgCycle );
}

//==============================================================================
// Initiates a PCI Configuration Write Cycle to write a PCI Configuration
// register of a PCI device.
//
// parameters:
//		reg			ordinal number of the register to be read
//		dev			PCI device number of the PCI deivce to be reached
//		bus			PCI bus number of the PCI deivce to be reached
//		val			32-bit word to be written into the selected register of the
//					accessed PCI device
//
// returns:
//		error codes according to the PCIBUS_getError()
//		PCIBUS_TOE_NO_ERROR = All right.
//==============================================================================
Uint32 PDEV_writePCIcfg(Uint32 reg, Uint32 dev, Uint32 val)
{
	PCIBUS_AccessReqObj	reqCfgCycle;
	SEM_Obj	sem;
	Uint32	value = val;

	// init semaphore
	SEM_new( &sem, 0 );

	// make a request for the PCI Configuration Cycle
	PCIBUS_makeReq_Cfg(
		&reqCfgCycle, reg, 0, dev, &value, PCIBUS_TRANSFER_TO_PCI,
		PCIBUS_TOS_SEM_FOR_TSK, &sem);

	// reading
	PCIBUS_accessReq( &reqCfgCycle );

	// waiting for data transfer to complete
	SEM_pend( &sem, SYS_FOREVER );

	// check error
	return PCIBUS_getError( &reqCfgCycle );
}


/******************************************************************************\
*                        G L O B A L   S E C T I O N
\******************************************************************************/


/******************************************************************************\
* global variable definitions
\******************************************************************************/


/******************************************************************************\
* global function definitions
\******************************************************************************/

//==============================================================================
// Opens a PCI Board according to its Vendor and Device ID and its ordinal
// number.
//
// parameters:
//		vendorAndDeviceID	Vendor and Device ID of the PCI device to be opened.
//							The PDEV_PCI_BOARD_RTD_x macros should be used.
//							(The 0th register in the Configuration Space.)
//		ordinal				Ordinal number of the PCI device within the
//							set of the same Vendor and Device ID devices.
//							Must be at least 1.
//		pDeviceObj			Pointer to the PCI device descriptor to be filled
//							up.
//
// returns:
//		PDEV_DEVICE_HAS_ALREADY_OPENED	The requested PCI device has already
//										been opened. It is opened currently.
//		PDEV_TOO_MANY_OPENED_DEVICE		There is no more room to maintain a new
//										opened PCI device.
//		PDEV_DEVICE_OPEN_SUCCESSFUL		The requested PCI device has been found
//										and opened. All right.
//		PDEV_DEVICE_NOT_FOUND			No error, but the device has not been
//										found.
//		PDEV_PCI_CONFIG_ERROR			PCI configuration cycle error has
//										occured.
//==============================================================================
Uint32 PDEV_openPCIDevice(Uint32 vendorAndDeviceID, Uint32 ordinal,
	PDEV_DeviceObj * pDeviceObj)
{
	Uint32	newdev;
	Uint32	err;

	// check whether the PCI device has already opened or not
	if( PDEV_isOpened(vendorAndDeviceID, ordinal) )
		return PDEV_DEVICE_HAS_ALREADY_OPENED;

	// check whether the program can maintain a new PCI device or not
	if( (newdev = PDEV_canBeOpened()) > PDEV_MAX_OPENED_PCI_DEVICES )
		return PDEV_TOO_MANY_OPENED_DEVICE;

	// attempt to find the given device
	if( (err = PDEV_findDevice(vendorAndDeviceID, ordinal, pDeviceObj))
		 == PDEV_DEVICE_FOUND )
	{
		// PDEV_findDevice has already filled up the some members of the deviceObj
		// structure, but not all ones.
		pDeviceObj->vendorAndDeviceID = vendorAndDeviceID;
		pDeviceObj->ordinal = ordinal;

		// record the opened device's parameters
		_PDEV_openedDevices[newdev].vendorAndDeviceID = vendorAndDeviceID;
		_PDEV_openedDevices[newdev].ordinal = ordinal;

		return PDEV_DEVICE_OPEN_SUCCESSFUL;
	}

	return err;
}

//==============================================================================
// Closes a previously opened PCI device and remove it from the maintain.
//
// parameters:
//		pDeviceObj		Pointer to the pDeviceObj of the opened PCI device to be
//						closed.
//
// returns:
//		TRUE	All right.
//		FALSE	The specified PCI device is not opened currently.
//==============================================================================
BOOL PDEV_closePCIDevice(PDEV_DeviceObj * pDeviceObj)
{
	Uint32 dev;
	Uint32 vendorAndDeviceID;
	Uint32 ordinal;

	vendorAndDeviceID = pDeviceObj->vendorAndDeviceID;
	ordinal = pDeviceObj->ordinal;

	for( dev = 0; dev < PDEV_MAX_OPENED_PCI_DEVICES; dev++)
	{
		if( (_PDEV_openedDevices[dev].vendorAndDeviceID == vendorAndDeviceID)
			&& (_PDEV_openedDevices[dev].ordinal == ordinal) )
		{
			// clear maintaining data of the closed PCI device
			_PDEV_openedDevices[dev].vendorAndDeviceID = 0;
			_PDEV_openedDevices[dev].ordinal = 0;

			return TRUE;
		}
	}

	// device is not opened currently
	return FALSE;
}

//==============================================================================
// Reads a PCI peripheral register of a DSP chip with integrated PCI interface.
// This function can be used for C6416 and C6205.
//
// parameters:
//		pDeviceObj		Pointer to the PCI device descriptor of an opened PCI
//						device. (PDEV_openPCIdevice fills it up.)
//		pciRegOffs		Direct BAR1 offset to select the appropriate register.
//						Use PDEV_SPM6?XX_PCI_REG_? macros to reach HSR, HDCR,
//						or DSPP register.
//		pValue			pointer to Uint32-type variable that will contain the
//						read value of the selected register
//
// returns:
//		Returns after the necessary PCI data trasfers only.
//		Gives TRUE, if all right, and pValue is filled up. Otherwise, FALSE.
//==============================================================================
BOOL PDEV_read_PciReg(PDEV_DeviceObj *pDeviceObj,
	Uint32 pciRegOffs, Uint32 *pValue)
{
	Uint32	reg;

	if( PCIBUS_read( &reg,
		PDEV_PCI_ADDRESS(pDeviceObj, BAR1, pciRegOffs), 1)
		!= PCIBUS_TOE_NO_ERROR) return FALSE;
	*pValue = reg;

	return TRUE;
}

//==============================================================================
// Writes a PCI peripheral register of a DSP chip with integrated PCI interface.
// This function can be used for C6416 and C6205.
//
// parameters:
//		pDeviceObj		Pointer to the PCI device descriptor of an opened PCI
//						device. (PDEV_openPCIdevice fills it up.)
//		pciRegOffs		Direct BAR1 offset to select the appropriate register.
//						Use PDEV_SPM6?XX_PCI_REG_? macros to reach HSR, HDCR,
//						or DSPP register.
//		value			32-bit value to be written into the selected register
//
// returns:
//		Returns after the necessary PCI data trasfers only.
//		Gives TRUE, if all right. Otherwise, FALSE.
//==============================================================================
BOOL PDEV_write_PciReg(PDEV_DeviceObj *pDeviceObj,
	Uint32 pciRegOffs, Uint32 value)
{
	Uint32	reg;

	reg = value;
	if( PCIBUS_write( &reg,
		PDEV_PCI_ADDRESS(pDeviceObj, BAR1, pciRegOffs), 1)
		!= PCIBUS_TOE_NO_ERROR) return FALSE;

	return TRUE;
}

//==============================================================================
// Reads a memory block from an opened SPM64xx dspModule board.
//
// parameters:
//		pDeviceObj		Pointer to the PCI device descriptor of the opened
//						SPM64xx board. (PDEV_openPCIdevice fills it up.)
//		pBuf			Pointer to the buffer to be filled up with the read
//						32-bit words.
//		addrInDev		Starting byte address of the transfer within the opened
//						board's 4-GB address space.
//						Must be aligned to 32-bit word boundary.
//		numOf32bWs		Number of the 32-bit words to be read.
//						Must be up to 65535.
//
// returns:
//		Returns after the necessary PCI data trasfers only.
//		Gives TRUE, if all right. Otherwise, FALSE.
//==============================================================================
BOOL PDEV_SPM64xx_read(PDEV_DeviceObj *pDeviceObj,
	Uint32 *pBuf, Uint32 addrInDev, Uint16 numOf32bWs)
{
	// write DSPP reg. to set location of the 4-MB window for BAR0
	if( !PDEV_write_PciReg(pDeviceObj,
		PDEV_SPM64XX_PCI_REG_DSPP,
		PDEV_DSP_PCI_PAGE_NUMBER(addrInDev)) ) return FALSE;

	// read memory block from BAR0
	if( PCIBUS_read( pBuf,
		PDEV_PCI_ADDRESS(pDeviceObj, BAR0,
		PDEV_DSP_PCI_PAGE_OFFSET(addrInDev)), numOf32bWs)
		!= PCIBUS_TOE_NO_ERROR) return FALSE;

	return TRUE;
}

//==============================================================================
// Writes a memory block into the memory space of an opened SPM64xx dspModule
// board.
//
// parameters:
//		pDeviceObj		Pointer to the PCI device descriptor of the opened
//						SPM64xx board. (PDEV_openPCIdevice fills it up.)
//		pBuf			Pointer to the buffer to get 32-bit words to be written
//						into the memory space of the SPM64xx board.
//		addrInDev		Starting byte address of the transfer within the opened
//						board's 4-GB address space.
//						Must be aligned to 32-bit word boundary.
//		numOf32bWs		Number of the 32-bit words to be read.
//						Must be up to 65535.
//
// returns:
//		Returns after the necessary PCI data trasfers only.
//		Gives TRUE, if all right. Otherwise, FALSE.
//==============================================================================
BOOL PDEV_SPM64xx_write(PDEV_DeviceObj *pDeviceObj,
	Uint32 *pBuf, Uint32 addrInDev, Uint16 numOf32bWs)
{
	// write DSPP reg. to set location of the 4-MB window for BAR0
	if( !PDEV_write_PciReg(pDeviceObj,
		PDEV_SPM64XX_PCI_REG_DSPP,
		PDEV_DSP_PCI_PAGE_NUMBER(addrInDev)) ) return FALSE;

	// read memory block from BAR0
	if( PCIBUS_write( pBuf,
		PDEV_PCI_ADDRESS(pDeviceObj, BAR0,
		PDEV_DSP_PCI_PAGE_OFFSET(addrInDev)), numOf32bWs)
		!= PCIBUS_TOE_NO_ERROR) return FALSE;

	return TRUE;
}

//==============================================================================
// Reads a memory block from an opened SPM62xx dspModule board.
//
// parameters:
//		pDeviceObj		Pointer to the PCI device descriptor of the opened
//						SPM62xx board. (PDEV_openPCIdevice fills it up.)
//		pBuf			Pointer to the buffer to be filled up with the read
//						32-bit words.
//		addrInDev		Starting byte address of the transfer within the opened
//						board's 4-GB address space.
//						Must be aligned to 32-bit word boundary.
//		numOf32bWs		Number of the 32-bit words to be read.
//						Must be up to 65535.
//
// returns:
//		Returns after the necessary PCI data trasfers only.
//		Gives TRUE, if all right. Otherwise, FALSE.
//==============================================================================
BOOL PDEV_SPM62xx_read(PDEV_DeviceObj *pDeviceObj,
	Uint32 *pBuf, Uint32 addrInDev, Uint16 numOf32bWs)
{
	Uint32	regXBISA;

	// write XBISA register to set the starting address of the data transfer
	// XBISA is at offset 0x0C of LAS1
	regXBISA = addrInDev & 0xFFFFFFFC;	// address autoincrement enabled
	if( PCIBUS_write(&regXBISA, PDEV_PCI_ADDRESS(pDeviceObj, LAS1, 0x0C), 1)
		!= PCIBUS_TOE_NO_ERROR) return FALSE;

	// read memory block from LAS0
	if( PCIBUS_read( pBuf, PDEV_PCI_ADDRESS(pDeviceObj, LAS0, 0), numOf32bWs)
		!= PCIBUS_TOE_NO_ERROR) return FALSE;

	return TRUE;
}

//==============================================================================
// Writes a memory block into the memory space of an opened SPM62xx dspModule
// board.
//
// parameters:
//		pDeviceObj		Pointer to the PCI device descriptor of the opened
//						SPM62xx board. (PDEV_openPCIdevice fills it up.)
//		pBuf			Pointer to the buffer to get 32-bit words to be written
//						into the memory space of the SPM62xx board.
//		addrInDev		Starting byte address of the transfer within the opened
//						board's 4-GB address space.
//						Must be aligned to 32-bit word boundary.
//		numOf32bWs		Number of the 32-bit words to be read.
//						Must be up to 65535.
//
// returns:
//		Returns after the necessary PCI data trasfers only.
//		Gives TRUE, if all right. Otherwise, FALSE.
//==============================================================================
BOOL PDEV_SPM62xx_write(PDEV_DeviceObj *pDeviceObj,
	Uint32 *pBuf, Uint32 addrInDev, Uint16 numOf32bWs)
{
	Uint32	regXBISA;

	// write XBISA register to set the starting address of the data transfer
	// XBISA is at offset 0x0C of LAS1
	regXBISA = addrInDev & 0xFFFFFFFC;	// address autoincrement enabled
	if( PCIBUS_write(&regXBISA, PDEV_PCI_ADDRESS(pDeviceObj, LAS1, 0x0C), 1)
		!= PCIBUS_TOE_NO_ERROR) return FALSE;

	// read memory block from LAS0
	if( PCIBUS_write( pBuf, PDEV_PCI_ADDRESS(pDeviceObj, LAS0, 0), numOf32bWs)
		!= PCIBUS_TOE_NO_ERROR) return FALSE;

	return TRUE;
}


#endif /* BSL_PDEV_SUPPORT */
/******************************************************************************\
* End of bsl_pdev.c
\******************************************************************************/

