/******************************************************************************\
* Copyright (C) 2004 by RTD Embedded Technologies, Inc.   All rights reserved.
* Confidential and Proprietary, Not for Public Release
*------------------------------------------------------------------------------
* PROJECT.......... Board Support Library for SPM6420
* VERSION.......... (Defined in README.TXT)
*------------------------------------------------------------------------------
* CONTENT.......... PCI Device Module of BSL
* FILENAME......... bsl_pdev.h
\******************************************************************************/
#ifndef _BSL_PDEV_H_
#define _BSL_PDEV_H_

#include <bsl.h>


#if (BSL_PDEV_SUPPORT)
/******************************************************************************\
* scope and inline control macros
\******************************************************************************/
#ifdef	__cplusplus
	#define	BSLAPI extern "C" far
#else
	#define	BSLAPI extern far
#endif

#undef	USEDEFS
#undef	IDECL
#undef	IDEF

#ifdef	_BSL_PDEV_MOD_
	#define IDECL BSLAPI
	#define USEDEFS
	#define IDEF
#else
	#ifdef	_INLINE
		#define	IDECL static inline
		#define	USEDEFS
		#define	IDEF  static inline
	#else
		#define	IDECL BSLAPI
	#endif
#endif


/******************************************************************************\
* global macro declarations
\******************************************************************************/

// Supported PCI boards to open them by a call of PDEV_openDevice().
#define PDEV_PCI_BOARD_RTD_SPM6020			0x60201435u
#define PDEV_PCI_BOARD_RTD_SPM6030			0x60301435u
#define PDEV_PCI_BOARD_RTD_DM7520			0x75201435u
#define PDEV_PCI_BOARD_RTD_DM7521			0x75211435u
#define PDEV_PCI_BOARD_RTD_DM7530			0x75301435u
#define PDEV_PCI_BOARD_RTD_DM7531			0x75311435u
#define PDEV_PCI_BOARD_RTD_DM7540			0x75401435u
#define PDEV_PCI_BOARD_RTD_SPM6420			0x64201435u
#define PDEV_PCI_BOARD_RTD_SPM6430			0x64301435u

// return values for PDEV_openPCIDevice()
#define PDEV_DEVICE_OPEN_SUCCESSFUL			0
#define PDEV_DEVICE_FOUND					0
#define PDEV_DEVICE_HAS_ALREADY_OPENED		1
#define PDEV_DEVICE_NOT_FOUND				2
#define PDEV_TOO_MANY_OPENED_DEVICE			3
#define PDEV_PCI_CONFIG_ERROR				4

// base addresses in the array baseAddress of PDEV_DeviceObj
#define PDEV_BASE_ADDRESS_TO_LCFG			0
#define PDEV_BASE_ADDRESS_TO_LCFG_MEM		0
#define PDEV_BASE_ADDRESS_TO_LCFG_IO		1
#define PDEV_BASE_ADDRESS_TO_LAS0			2
#define PDEV_BASE_ADDRESS_TO_LAS1			3

#define PDEV_BASE_ADDRESS_TO_BAR0			0
#define PDEV_BASE_ADDRESS_TO_BAR1			1
#define PDEV_BASE_ADDRESS_TO_BAR2			2
#define PDEV_BASE_ADDRESS_TO_BAR3			3

#define PDEV_NUMBER_OF_BARS					4

// Direct BAR1 offsets for PCI configuration registers
// Use these macros for PDEV_read/write_PciReg
#define PDEV_SPM64XX_PCI_REG_HSR			(0x01C1FFF0u - 0x01800000u)
#define PDEV_SPM64XX_PCI_REG_HDCR			(0x01C1FFF4u - 0x01800000u)
#define PDEV_SPM64XX_PCI_REG_DSPP			(0x01C1FFF8u - 0x01800000u)

#define PDEV_SPM62XX_PCI_REG_HSR			(0x01A7FFF0u - 0x01800000u)
#define PDEV_SPM62XX_PCI_REG_HDCR			(0x01A7FFF4u - 0x01800000u)
#define PDEV_SPM62XX_PCI_REG_DSPP			(0x01A7FFF8u - 0x01800000u)

#define PDEV_SPM67XX_PCI_REG_HSR			(0x01A7FFF0u - 0x01800000u)
#define PDEV_SPM67XX_PCI_REG_HDCR			(0x01A7FFF4u - 0x01800000u)
#define PDEV_SPM67XX_PCI_REG_DSPP			(0x01A7FFF8u - 0x01800000u)

// used to calculate DSPP value and BAR0 offset
#define PDEV_DSP_PCI_PAGE_NUMBER(addr)		(((Uint32)(addr))>>22)
#define PDEV_DSP_PCI_PAGE_OFFSET(addr)		(((Uint32)(addr))&0x003FFFFFu)

//==============================================================================
// Provides a PCI address according to the opened PCI device, the selected
// region, and the offset
//
// parameters:
//		pDeviceObj		pointer to a PDEV_DeviceObj of an opened PCI device
//		REGION			select a region
//						use suffix of PDEV_BASE_ADDRESS_TO_<REGION> constants
//		offset			byte offset within the selected region
//
// returns:
//		an absolute (within the 4 GB system memory) PCI address according
//		to the given region and offset of the selected (opened) PCI device
//==============================================================================
#define PDEV_PCI_ADDRESS(pDeviceObj,REGION,offset) \
	((pDeviceObj)->baseAddress[PDEV_BASE_ADDRESS_TO_##REGION]+((Uint32)(offset)))


/******************************************************************************\
* global typedef declarations
\******************************************************************************/

// descriptor for a PCI device
// handles 4 BARs only (instead of 6)
typedef struct {
	Uint32	baseAddress[PDEV_NUMBER_OF_BARS];
	Uint32	vendorAndDeviceID;
	Uint32	ordinal;	// ordinal number of the PCI device within the same
						// Vendor and Device ID devices
	Uint16	location;	// func/dev of the openen PCI device
	Uint16	barTypes;	// bit mask to determine the BAR is memory or IO mapped
} PDEV_DeviceObj;


/******************************************************************************\
* global variable declarations
\******************************************************************************/


/******************************************************************************\
* global function declarations
\******************************************************************************/

//==============================================================================
// Opens a PCI Board according to its Vendor and Device ID and its ordinal
// number.
//
// parameters:
//		vendorAndDeviceID	Vendor and Device ID of the PCI device to be opened.
//							The PDEV_PCI_BOARD_RTD_x macros should be used.
//							(The 0th register in the Configuration Space.)
//		ordinal				Ordinal number of the PCI device within the
//							set of the same Vendor and Device ID devices.
//							Must be at least 1.
//		pDeviceObj			Pointer to the PCI device descriptor to be filled
//							up.
//
// returns:
//		PDEV_DEVICE_HAS_ALREADY_OPENED	The requested PCI device has already
//										been opened. It is opened currently.
//		PDEV_TOO_MANY_OPENED_DEVICE		There is no more room to maintain a new
//										opened PCI device.
//		PDEV_DEVICE_OPEN_SUCCESSFUL		The requested PCI device has been found
//										and opened. All right.
//		PDEV_DEVICE_NOT_FOUND			No error, but the device has not been
//										found.
//		PDEV_PCI_CONFIG_ERROR			PCI configuration cycle error has
//										occured.
//==============================================================================
BSLAPI Uint32 PDEV_openPCIDevice(Uint32 vendorAndDeviceID, Uint32 ordinal,
	PDEV_DeviceObj * pDeviceObj);

//==============================================================================
// Closes a previously opened PCI device and remove it from the maintain.
//
// parameters:
//		pDeviceObj		Pointer to the pDeviceObj of the opened PCI device to be
//						closed.
//
// returns:
//		TRUE	All right.
//		FALSE	The specified PCI device is not opened currently.
//==============================================================================
BSLAPI BOOL PDEV_closePCIDevice(PDEV_DeviceObj * pDeviceObj);

//==============================================================================
// Reads a PCI peripheral register of a DSP chip with integrated PCI interface.
// This function can be used for C6416 and C6205.
//
// parameters:
//		pDeviceObj		Pointer to the PCI device descriptor of an opened PCI
//						device. (PDEV_openPCIdevice fills it up.)
//		pciRegOffs		Direct BAR1 offset to select the appropriate register.
//						Use PDEV_SPM6?XX_PCI_REG_? macros to reach HSR, HDCR,
//						or DSPP register.
//		pValue			pointer to Uint32-type variable that will contain the
//						read value of the selected register
//
// returns:
//		Returns after the necessary PCI data trasfers only.
//		Gives TRUE, if all right, and pValue is filled up.
//		Otherwise, FALSE.
//==============================================================================
BSLAPI BOOL PDEV_read_PciReg(PDEV_DeviceObj *pDeviceObj,
	Uint32 pciRegOffs, Uint32 *pValue);

//==============================================================================
// Writes a PCI peripheral register of a DSP chip with integrated PCI interface.
// This function can be used for C6416 and C6205.
//
// parameters:
//		pDeviceObj		Pointer to the PCI device descriptor of an opened PCI
//						device. (PDEV_openPCIdevice fills it up.)
//		pciRegOffs		Direct BAR1 offset to select the appropriate register.
//						Use PDEV_SPM6?XX_PCI_REG_? macros to reach HSR, HDCR,
//						or DSPP register.
//		value			32-bit value to be written into the selected register
//
// returns:
//		Returns after the necessary PCI data trasfers only.
//		Gives TRUE, if all right. Otherwise, FALSE.
//==============================================================================
BSLAPI BOOL PDEV_write_PciReg(PDEV_DeviceObj *pDeviceObj,
	Uint32 pciRegOffs, Uint32 value);

//==============================================================================
// Reads a memory block from an opened SPM64xx dspModule board.
//
// parameters:
//		pDeviceObj		Pointer to the PCI device descriptor of the opened
//						SPM64xx board. (PDEV_openPCIdevice fills it up.)
//		pBuf			Pointer to the buffer to be filled up with the read
//						32-bit words.
//		addrInDev		Starting byte address of the transfer within the opened
//						board's 4-GB address space.
//						Must be aligned to 32-bit word boundary.
//		numOf32bWs		Number of the 32-bit words to be read.
//						Must be up to 16383.
//
// returns:
//		Returns after the necessary PCI data trasfers only.
//		Gives TRUE, if all right. Otherwise, FALSE.
//==============================================================================
BSLAPI BOOL PDEV_SPM64xx_read(PDEV_DeviceObj *pDeviceObj,
	Uint32 *pBuf, Uint32 addrInDev, Uint16 numOf32bWs);

//==============================================================================
// Writes a memory block into the memory space of an opened SPM64xx dspModule
// board.
//
// parameters:
//		pDeviceObj		Pointer to the PCI device descriptor of the opened
//						SPM64xx board. (PDEV_openPCIdevice fills it up.)
//		pBuf			Pointer to the buffer to get 32-bit words to be written
//						into the memory space of the SPM64xx board.
//		addrInDev		Starting byte address of the transfer within the opened
//						board's 4-GB address space.
//						Must be aligned to 32-bit word boundary.
//		numOf32bWs		Number of the 32-bit words to be read.
//						Must be up to 16383.
//
// returns:
//		Returns after the necessary PCI data trasfers only.
//		Gives TRUE, if all right. Otherwise, FALSE.
//==============================================================================
BSLAPI BOOL PDEV_SPM64xx_write(PDEV_DeviceObj *pDeviceObj,
	Uint32 *pBuf, Uint32 addrInDev, Uint16 numOf32bWs);

//==============================================================================
// Reads a memory block from an opened SPM62xx dspModule board.
//
// parameters:
//		pDeviceObj		Pointer to the PCI device descriptor of the opened
//						SPM62xx board. (PDEV_openPCIdevice fills it up.)
//		pBuf			Pointer to the buffer to be filled up with the read
//						32-bit words.
//		addrInDev		Starting byte address of the transfer within the opened
//						board's 4-GB address space.
//						Must be aligned to 32-bit word boundary.
//		numOf32bWs		Number of the 32-bit words to be read.
//						Must be up to 16383.
//
// returns:
//		Returns after the necessary PCI data trasfers only.
//		Gives TRUE, if all right. Otherwise, FALSE.
//==============================================================================
BSLAPI BOOL PDEV_SPM62xx_read(PDEV_DeviceObj *pDeviceObj,
	Uint32 *pBuf, Uint32 addrInDev, Uint16 numOf32bWs);

//==============================================================================
// Writes a memory block into the memory space of an opened SPM62xx dspModule
// board.
//
// parameters:
//		pDeviceObj		Pointer to the PCI device descriptor of the opened
//						SPM62xx board. (PDEV_openPCIdevice fills it up.)
//		pBuf			Pointer to the buffer to get 32-bit words to be written
//						into the memory space of the SPM62xx board.
//		addrInDev		Starting byte address of the transfer within the opened
//						board's 4-GB address space.
//						Must be aligned to 32-bit word boundary.
//		numOf32bWs		Number of the 32-bit words to be read.
//						Must be up to 16383.
//
// returns:
//		Returns after the necessary PCI data trasfers only.
//		Gives TRUE, if all right. Otherwise, FALSE.
//==============================================================================
BSLAPI BOOL PDEV_SPM62xx_write(PDEV_DeviceObj *pDeviceObj,
	Uint32 *pBuf, Uint32 addrInDev, Uint16 numOf32bWs);


/******************************************************************************\
* inline function declarations
\******************************************************************************/


/******************************************************************************\
* inline function definitions
\******************************************************************************/
#ifdef USEDEFS
/*----------------------------------------------------------------------------*/
#endif /* USEDEFS */


#endif /* BSL_PDEV_SUPPORT */
#endif /* _BSL_PDEV_H_ */
/******************************************************************************\
* End of bsl_pdev.h
\******************************************************************************/

