
/*****************************************************************************
*
*  spi.c - CC3000 Host Driver Implementation.
*  Copyright (C) 2011 Texas Instruments Incorporated - http://www.ti.com/
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*    Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the
*    documentation and/or other materials provided with the   
*    distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
*****************************************************************************/

//*****************************************************************************
//
///*******************************************************************************
// *                              INCLUDE FILES
// ******************************************************************************/
#include "hw_gpio.h"
#include "hw_qspi1.h"
#include "hci.h"
#include "spi.h"
#include "evnt_handler.h"
#include "MCF5253_QSPI.h"


#define READ                    3
#define WRITE                   1
#define HI(value)               (((value) & 0xFF00) >> 8)
#define LO(value)               ((value) & 0x00FF)
#define HEADERS_SIZE_EVNT       (SPI_HEADER_SIZE + 5)

#define SPI_HEADER_SIZE			(5)

#define 	eSPI_STATE_POWERUP 				 (0)
#define 	eSPI_STATE_INITIALIZED  		 (1)
#define 	eSPI_STATE_IDLE					 (2)
#define 	eSPI_STATE_WRITE_IRQ	   		 (3)
#define 	eSPI_STATE_WRITE_FIRST_PORTION   (4)
#define 	eSPI_STATE_WRITE_EOT			 (5)
#define 	eSPI_STATE_READ_IRQ				 (6)
#define 	eSPI_STATE_READ_FIRST_PORTION	 (7)
#define 	eSPI_STATE_READ_EOT				 (8)

#define CC3000_BUFFER_MAGIC_NUMBER (0xDE)
unsigned long TPoint[100] = {0};
unsigned long Ctr = 0;
unsigned long CtrRead = 0;
unsigned long CtrWrite =  0;
extern unsigned long TrackPoint[30] ;
unsigned long CounterWrite = 0 ;
 ///*******************************************************************************
// *                              LOCAL STRUCTURES / TYPEDEFS
// ******************************************************************************/
//int IRQControl = 1;
typedef struct
{
	gcSpiHandleRx  SPIRxHandler;
	unsigned short usTxPacketLength;
	unsigned short usRxPacketLength;
	unsigned long  ulSpiState;
	unsigned char *pTxPacket;
	unsigned char *pRxPacket;

}tSpiInformation;


tSpiInformation sSpiInformation;


// buffer for 5 bytes of SPI HEADER
unsigned char tSpiReadHeader[] = {READ, 0, 0, 0, 0};

char spi_buffer[CC3000_RX_BUFFER_SIZE ];

unsigned char wlan_tx_buffer[CC3000_TX_BUFFER_SIZE];

unsigned char Data_Tx_Done = 0;
///*******************************************************************************
// *                              VARIABLES
// ******************************************************************************/

///*******************************************************************************
// *                              LOCAL PROTOTYPES
// ******************************************************************************/
//
void SpiWriteDataSynchronous(unsigned char *data, unsigned short size);
void SpiWriteAsync(const unsigned char *data, unsigned short size);
void SpiPauseSpi(void);
void SpiResumeSpi(void);
void SSIContReadOperation(void);
#define QSPI_TRANSMIT_ADDRESS 0x00          /*QSPI RAM transmit address*/
#define QSPI_RECEIVE_ADDRESS  0x10          /*QSPI RAM receive address*/
// the allocated size) for the purpose of detection of the overrun. The location
// of the memory where the magic number resides shall never be written. In case 
// it is written - the overrun occurred and either receive function or send
// function will stuck forever.
#define QSPI_COMMAND_ADDRESS  0x20          /*QSPI RAM command address*/


#define ASSERT_CS()			(MCF_GPIO_GPIO_OUT &= ~(0x00000001 << 15))
#define DEASSERT_CS()		(MCF_GPIO_GPIO_OUT |= (0x00000001 << 15))	


//*****************************************************************************
//
//!  SpiClose
//!
//!  @param  none
//!
//!  @return none
//!
//!  @brief  Close Spi interface
//
//*****************************************************************************
void
SpiClose(void)
{
	if (sSpiInformation.pRxPacket)
	{
		sSpiInformation.pRxPacket = 0;
	}
	
	//	Disable Interrupt in GPIOA module...
	tSLInformation.WlanInterruptDisable();
}


//*****************************************************************************
//
//!  SpiOpen
//!
//!  @param  none
//!
//!  @return none
//!
//!  @brief  Open Spi interface 
//
//*****************************************************************************
void 
SpiOpen(gcSpiHandleRx pfRxHandler)
{
	sSpiInformation.ulSpiState = eSPI_STATE_POWERUP;
	sSpiInformation.SPIRxHandler = pfRxHandler;
	sSpiInformation.usTxPacketLength = 0;
	sSpiInformation.pTxPacket = NULL;
	sSpiInformation.pRxPacket = (unsigned char *)spi_buffer;
	sSpiInformation.usRxPacketLength = 0;
	spi_buffer[CC3000_RX_BUFFER_SIZE - 1] = CC3000_BUFFER_MAGIC_NUMBER;
	wlan_tx_buffer[CC3000_TX_BUFFER_SIZE - 1] = CC3000_BUFFER_MAGIC_NUMBER;
	
	// Enable interrupt on the GPIOA pin of WLAN IRQ
	tSLInformation.WlanInterruptEnable();
}

//*****************************************************************************
//
//!  init_spi
//!
//!  @param  none
//!
//!  @return none
//!
//!  @brief  initializes an SPI interface
//
//*****************************************************************************

int init_spi(void)
{
/*
	MCF_QSPI_QMR	= 0xA002;
	MCF_QSPI_QDLYR	= 0x0000;
	MCF_QSPI_QIR	= 0xD10D;
	*/
	/*
	MCF_QSPI_QMR	= 0xA110;
	MCF_QSPI_QDLYR	= 0x0808;
	MCF_QSPI_QIR	= 0xD10D;
	*/
	//MCF_QSPI_QDLYR	= 0x0002;
	//MCF_QSPI_QDLYR	= 0x7F6E;
	
	MCF_QSPI_QDLYR	= 0x0000;
	MCF_QSPI_QMR	= 0xA104;//Baud Changes as CC3000 onlyy support 0-16Mhz
	MCF_QSPI_QWR 	= 0x1000;//make CS active low(Inactive state is 1)
	MCF_QSPI_QIR	= 0x110D;
	DEASSERT_CS();
	return(0);
	//MCF_QSPI_QAR 	= QSPI_COMMAND_ADDRESS;
	//MCF_QSPI_QDR 	= 0x0E00;//CONT - 1, BITSE - 0, DT - 1, DSCK - 1
/*
	MCF_QSPI_QMR	= 0xA210;//MSTR - 1, DOHI - 0, BITS - 1000, CPOL - 1, CPHA - 0, BAUD - 2187500
	MCF_QSPI_QDLYR	= 0x0101;//SPE - Disable, QCD - 1, DTL - 1
	MCF_QSPI_QIR	= 0xD10D;//WCEFB - 1, ABRTB - 1, ABRTL - 1, WCEFE - 0, ABRTE - 0, SPIFE - 1 WCEF - 1, ABRT - 1, SPIF - 1
	MCF_QSPI_QWR	= 0x1000;//HALT -0, WREN - 0, WRTO - 0, QSPI_CS - 1, ENDQP - 0000, CPTQ - 0000, NEWQP - 0000
*/
}

long
SpiFirstWrite(unsigned char *ucBuf, unsigned short usLength)
{
    
	long i;
	ASSERT_CS();
	for(i = 0 ; i <= 3000;i++)
	{
		asm("nop");
		asm("nop");
		asm("nop");
		asm("nop");
		asm("nop");
	}
   
    SpiWriteDataSynchronous(ucBuf, 4);
    
    for(i = 0 ; i <= 3000;i++)
	{
		asm("nop");
		asm("nop");
		asm("nop");
		asm("nop");
		asm("nop");
	
	}
	
    SpiWriteDataSynchronous(ucBuf + 4, usLength - 4);

    // From this point on - operate in a regular way
    sSpiInformation.ulSpiState = eSPI_STATE_IDLE;

    DEASSERT_CS();
    
    return(0);
}


//*****************************************************************************
//
//!  SpiWrite
//!
//!  @param  pUserBuffer  buffer to write
//!  @param  usLength     buffer's length
//!
//!  @return none
//!
//!  @brief  Spi write operation
//
//*****************************************************************************
long
SpiWrite(unsigned char *pUserBuffer, unsigned short usLength)
{
	unsigned char ucPad = 0;
	
	// Figure out the total length of the packet in order to figure out if there 
	// is padding or not
	if(!(usLength & 0x0001))
	{
		ucPad++;
	}
	
	pUserBuffer[0] = WRITE;
	pUserBuffer[1] = HI(usLength + ucPad);
	pUserBuffer[2] = LO(usLength + ucPad);
	pUserBuffer[3] = 0;
	pUserBuffer[4] = 0;
	
	usLength += (SPI_HEADER_SIZE + ucPad);
	
	// The magic number that resides at the end of the TX/RX buffer (1 byte after 
	// the allocated size) for the purpose of detection of the overrun. If the 
	// magic number is overwritten - buffer overrun occurred - and we will stuck 
	// here forever!
	if (wlan_tx_buffer[CC3000_TX_BUFFER_SIZE - 1] != CC3000_BUFFER_MAGIC_NUMBER)
	{
		while (1)
			;
	}
	
	if (sSpiInformation.ulSpiState == eSPI_STATE_POWERUP)
	{
		while (sSpiInformation.ulSpiState != eSPI_STATE_INITIALIZED)
			;
	}
	
	if (sSpiInformation.ulSpiState == eSPI_STATE_INITIALIZED)
	{
		// This is time for first TX/RX transactions over SPI: the IRQ is down - 
		// so need to send read buffer size command
		SpiFirstWrite(pUserBuffer, usLength);
	}
	else 
	{
		// We need to prevent here race that can occur in case 2 back to back 
		// packets are sent to the  device, so the state will move to IDLE and once 
		//again to not IDLE due to IRQ
		asm("move.w  #0x2700,SR");
		
		
		while (sSpiInformation.ulSpiState != eSPI_STATE_IDLE)
		{
			;
		}
		tSLInformation.WlanInterruptDisable();
		sSpiInformation.ulSpiState = eSPI_STATE_WRITE_IRQ;
		sSpiInformation.pTxPacket = pUserBuffer;
		sSpiInformation.usTxPacketLength = usLength;
		
		
		
		// Re-enable IRQ - if it was not disabled - this is not a problem...
		tSLInformation.WlanInterruptEnable();
		asm("move.w  #0x2000,SR");
		// Assert the CS line and wait till SSI IRQ line is active and then
		// initialize write operation
		ASSERT_CS();
		
		
		// check for a missing interrupt between the CS assertion and enabling back the interrupts
		if (tSLInformation.ReadWlanInterruptPin() == 0)
		{
            SpiWriteDataSynchronous(sSpiInformation.pTxPacket, sSpiInformation.usTxPacketLength);

			sSpiInformation.ulSpiState = eSPI_STATE_IDLE;

			DEASSERT_CS();
		}
	}
	
	// Due to the fact that we are currently implementing a blocking situation
	// here we will wait till end of transaction
	while (eSPI_STATE_IDLE != sSpiInformation.ulSpiState);
	
	
	return(0);
}

 
//*****************************************************************************
//
//!  SpiWriteDataSynchronous
//!
//!  @param  data  buffer to write
//!  @param  size  buffer's size
//!
//!  @return none
//!
//!  @brief  Spi write operation
//
//*****************************************************************************
void
SpiWriteDataSynchronous(unsigned char *data, unsigned short size)
{
	
	unsigned char DataReceived = 0;
    
    unsigned char ucsize = size,i;
    
  
    
    while(size)
    {
    	MCF_QSPI_QAR = QSPI_COMMAND_ADDRESS;
    	
		MCF_QSPI_QDR = 0x0E00;//CONT - 0, BITSE - 0, DT - 1, DSCK - 1
		
		Data_Tx_Done = 0;
		
		MCF_QSPI_QAR = QSPI_TRANSMIT_ADDRESS;
		
		MCF_QSPI_QDR = *data;
		
		MCF_QSPI_QDLYR |= 0x8000;//SPE -Ebnabel
		
		while(!Data_Tx_Done);
		
		MCF_QSPI_QAR = QSPI_RECEIVE_ADDRESS;
		
		DataReceived = MCF_QSPI_QDR;
	
		data++;
		
		size--;
    }
    
    /*
   	MCF_QSPI_QAR = QSPI_COMMAND_ADDRESS;
	
	for(i = 0; i < size ; i++)
	{
		MCF_QSPI_QDR = 0x3F00;//CONT - 1, BITSE - 0, DT - 1, DSCK - 1
	}
	
	MCF_QSPI_QAR = QSPI_TRANSMIT_ADDRESS;
	
	for(i = 0; i < size ; i++)
	{
		MCF_QSPI_QDR = *data;
		
		data++;
	}
	
	Data_Tx_Done = 0;
	
	MCF_QSPI_QWR |= (MCF_QSPI_QWR_ENDQP(ucsize)
					|MCF_QSPI_QWR_NEWQP(0x00));
			
	MCF_QSPI_QDLYR  |= 0x8000 ;//SPE -Ebnabel
	
	while(!Data_Tx_Done);
	
	MCF_QSPI_QAR = QSPI_RECEIVE_ADDRESS;
	
	for(i = 0; i < size ; i ++)
	{
		DataReceived = MCF_QSPI_QDR;
	}
   */
	
}

//*****************************************************************************
//
//! SpiReadDataSynchronous
//!
//!  @param  data  buffer to read
//!  @param  size  buffer's size
//!
//!  @return none
//!
//!  @brief  Spi read operation
//
//*****************************************************************************
void
SpiReadDataSynchronous(unsigned char *data, unsigned short size)
{
	
	long i = 0,j= 0;
	
	unsigned char *data_to_send = tSpiReadHeader;
	
	for (i = 0; i < size; i ++)
	{
		MCF_QSPI_QAR = QSPI_COMMAND_ADDRESS;
	
		//if(i < (size -1))
		//	MCF_QSPI_QDR = 0xBE00;//CONT - 1, BITSE - 0, DT - 1, DSCK - 1
		//else
		MCF_QSPI_QDR = 0x0E00;
		
		Data_Tx_Done = 0;
		
		MCF_QSPI_QAR = QSPI_TRANSMIT_ADDRESS;
		
		MCF_QSPI_QDR = data_to_send[0];
		
		MCF_QSPI_QDLYR |= 0x8000;//SPE -Ebnabel
		
		while(!Data_Tx_Done);
		
		MCF_QSPI_QAR = QSPI_RECEIVE_ADDRESS;
		
		data[i] = MCF_QSPI_QDR;
		
	} 
	/*
	
	long i = 0;
    
	unsigned char *data_to_send = tSpiReadHeader;
	
	unsigned char ucsize = size;
	
	MCF_QSPI_QAR = QSPI_COMMAND_ADDRESS;
	
	for(i = 0; i < size ; i++)
	{
		MCF_QSPI_QDR = 0x3E00;//CONT - 1, BITSE - 0, DT - 1, DSCK - 1
	}
	
	MCF_QSPI_QAR = QSPI_TRANSMIT_ADDRESS;
	
	for(i = 0; i < size ; i++)
	{
		MCF_QSPI_QDR = data_to_send[i];
	}
	
	Data_Tx_Done = 0;
	
	MCF_QSPI_QWR |= (MCF_QSPI_QWR_ENDQP(ucsize)
					|MCF_QSPI_QWR_NEWQP(0x00) );
		
	MCF_QSPI_QDLYR |= 0x8000;//SPE -Ebnabel
		
	while(!Data_Tx_Done);
	
	MCF_QSPI_QAR = QSPI_RECEIVE_ADDRESS;
	
	for(i = 0; i < size ; i ++)
	{
		data[i] = MCF_QSPI_QDR;
	}
	
	*/
}


//*****************************************************************************
//
//!  SpiReadHeader
//!
//!  \param  buffer
//!
//!  \return none
//!
//!  \brief   This function enter point for read flow: first we read minimal 5 
//!	          SPI header bytes and 5 Event Data bytes
//
//*****************************************************************************
void
SpiReadHeader(void)
{
	SpiReadDataSynchronous(sSpiInformation.pRxPacket, 10);
}


//*****************************************************************************
//
//!  SpiReadDataCont
//!
//!  @param  None
//!
//!  @return None
//!
//!  @brief  This function processes received SPI Header and in accordance with 
//!	         it - continues reading the packet
//
//*****************************************************************************
long
SpiReadDataCont(void)
{
	long data_to_recv;
	unsigned char *evnt_buff, type;
	
	//determine what type of packet we have
	evnt_buff =  sSpiInformation.pRxPacket;
	data_to_recv = 0;
	STREAM_TO_UINT8((char *)(evnt_buff + SPI_HEADER_SIZE), HCI_PACKET_TYPE_OFFSET,
									type);
	
    switch(type)
    {
        case HCI_TYPE_DATA:
        {
			//
			// We need to read the rest of data..
			STREAM_TO_UINT16((char *)(evnt_buff + SPI_HEADER_SIZE), 
											 HCI_DATA_LENGTH_OFFSET, data_to_recv);
			if (!((HEADERS_SIZE_EVNT + data_to_recv) & 1))
			{	
				data_to_recv++;
			}
			
			if (data_to_recv)
			{
				SpiReadDataSynchronous(evnt_buff + 10, data_to_recv);
			}
			break;
		}
	case HCI_TYPE_EVNT:
		{
			// Calculate the rest length of the data
			STREAM_TO_UINT8((char *)(evnt_buff + SPI_HEADER_SIZE), 
											HCI_EVENT_LENGTH_OFFSET, data_to_recv);
			
			data_to_recv -= 1;
			
			// Add padding byte if needed
			if ((HEADERS_SIZE_EVNT + data_to_recv) & 1)
			{
				
				data_to_recv++;
			}
			
			if (data_to_recv)
			{
				SpiReadDataSynchronous(evnt_buff + 10, data_to_recv);
			}
			
			sSpiInformation.ulSpiState = eSPI_STATE_READ_EOT;
			break;
		}
	}
	
	return (0);
}

void 
SpiTriggerRxProcessing(void)
{
	
	SpiPauseSpi();
	//
	// Trigger Rx processing
	//
	DEASSERT_CS();
        
        // The magic number that resides at the end of the TX/RX buffer (1 byte after the allocated size)
        // for the purpose of detection of the overrun. If the magic number is overriten - buffer overrun 
        // occurred - and we will stuck here forever!
	if (sSpiInformation.pRxPacket[CC3000_RX_BUFFER_SIZE - 1] != CC3000_BUFFER_MAGIC_NUMBER)
	{
		while (1);
	}
	
	sSpiInformation.ulSpiState = eSPI_STATE_IDLE;
	sSpiInformation.SPIRxHandler(sSpiInformation.pRxPacket + SPI_HEADER_SIZE);
}


void
SSIContReadOperation(void)
{
	//
	// The header was read - continue with  the payload read
	//
	if (!SpiReadDataCont())
	{
		
		
		//
		// All the data was read - finalize handling by switching to teh task
		//	and calling from task Event Handler
		//
		SpiTriggerRxProcessing();
	}
}
/*****************************************************************************
 * QSPIIsr: QSPI Interrupt Service Routine                                   *                               
 *          Saves the received data into sram. Enables the sendStatus flag.	 *                                                             
 *                                                                           *
 * Parameters: none.                                                         *
 *                                                                           *
 *                                                                           *
 * Return : none.                                                            *
 *****************************************************************************/

void QSPI_ISR(void)
{
   MCF_QSPI_QIR |= MCF_QSPI_QIR_SPIF;
	
   Data_Tx_Done = 1;
   
  
}
void QSPI_IRQ_ISR(void)
{
	
	MCF_GPIO_GPIO_INT_CLEAR |= MCF_GPIO_GPIO_INT_CLEAR_GPIO_INT_CLEAR13;
	//if(!IRQControl)
	//	while(!IRQControl);
	//IRQControl = 0;
	if (sSpiInformation.ulSpiState == eSPI_STATE_POWERUP)
	{
		/* This means IRQ line was low call a callback of HCI Layer to inform on event */
 		sSpiInformation.ulSpiState = eSPI_STATE_INITIALIZED;
	}
	else if (sSpiInformation.ulSpiState == eSPI_STATE_IDLE)
	{
		
		sSpiInformation.ulSpiState = eSPI_STATE_READ_IRQ;
		ASSERT_CS();
		//
		// Wait for TX/RX Compete which will come as DMA interrupt
		// 
        SpiReadHeader();
			
		sSpiInformation.ulSpiState = eSPI_STATE_READ_EOT;
			
		//
		//
		//
		SSIContReadOperation();
	}
	else if (sSpiInformation.ulSpiState == eSPI_STATE_WRITE_IRQ)
	{
		SpiWriteDataSynchronous(sSpiInformation.pTxPacket, sSpiInformation.usTxPacketLength);
		sSpiInformation.ulSpiState = eSPI_STATE_IDLE;
		
		DEASSERT_CS();
		
	}

	//IRQControl = 1;

}

void SpiPauseSpi(void)
{
	WlanInterruptDisable();
	//WriteWlanPin(0);
}
void SpiResumeSpi(void)
{
	//WriteWlanPin(1);
	WlanInterruptEnable();
}
