///*******************************************************************************
// * Copyright (c) 2007, Cummins Engine Company, Inc.  All Rights Reserved.
// *
// *          FILENAME: hw_qspi.h
// *
// *      FILE PURPOSE: This file contains the constants, types and
// *                    function prototypes for the hw_qspi.c module.
// *
// * CONTROLLING DESIGN
// *       DOCUMENT(S):
// *******************************************************************************
// * IMPLEMENTATION CONCEPTS:
// *
// *******************************************************************************
// * FUNCTION INDEX:
// *
// ******************************************************************************/
//#ifndef HW_QSPI_H
//#define HW_QSPI_H
//
///*******************************************************************************
// *                              INCLUDE FILES
// ******************************************************************************/
//#include "hw_configuration.h"
//#include "hw_clock.h"
//
///*******************************************************************************
// *                              MACROS
// ******************************************************************************/
//#define QSPI_CMD_RAM_START_ADDRESS      (0x20)
//#define QSPI_RX_RAM_START_ADDRESS       (0x10)
//#define QSPI_TX_RAM_START_ADDRESS       (0x00)
//
//#define QSPI_RAM_BLOCK_SIZE             (0x10)
//
//#define AUX_1_CPU_CHIP_SELECT           (0xE)
//#define AUX_2_CPU_CHIP_SELECT           (0xB)
//#define TOUCH_SCREEN_CHIP_SELECT        (0xD)
//
//#define MCF_QSPI_INT_CNTRL_REG      MCF_INTC0_ICR31
//
///*******************************************************************************
//*   Macro Name:   SET_QSPI_BAUD_RATE
//*
//*   Description:  The baud rate is selected by writing a value in the
//*       range 2255. A value of zero disables the QSPI. A value of 1 is an
//*       invalid setting. The desired QSPI_CLK baud rate is related to the
//*       internal bus clock and QMR[BAUD] by the following expression:
//*
//*                                    fsys / 3
//*           QMR[BAUD] = ----------------------------------
//*                        2  [desired QSPI_CLK baud rate]
//*
//*   Inputs:  Processor Internal Bus Clock Frequency
//*            baud_rate
//*
//*   Outputs: None
//*
//*******************************************************************************/
//#define QSPI_BAUD_RATE_MULTIPLIER   (2)
//
//#define SET_QSPI_BAUD_RATE(baud_rate)                                          \
//            CPU_INT_BUS_FREQ / (QSPI_BAUD_RATE_MULTIPLIER * baud_rate)
//
//#define RR5_TOUCH_SPI_BAUD_RATE     (312500)
//
//#define RR5_SPI_BAUD_RATE           (1250000)   /* desired RR5 baud rate */
//
///*******************************************************************************
//*
//*   QSPI Transfer Delays
//*
//*           The QSPI supports programmable delays for the QSPI_CS signals
//*           before and after a transfer. The time between QSPI_CS
//*           assertion and the leading QSPI_CLK edge, and the time between
//*           the end of one transfer and the beginning of the next, are
//*           both independently programmable.
//*
//********************************************************************************
//*
//*   Macro Name:   RR5_SPI_QDLYR_QCD
//*
//*   Description:  The chip select to clock delay enable bit in command RAM,
//*                 QCR[DSCK], enables the programmable delay period from QSPI_CS
//*                 assertion until the leading edge of QSPI_CLK. QDLYR[QCD]
//*                 determines the period of delay before the leading edge of
//*                 QSPI_CLK. The following expression determines the actual
//*                 delay before the QSPI_CLK leading edge:
//*
//*                                              QDLYR[QCD]
//*                 QSPI_CS-to-QSPI_CLK delay = ------------
//*                                               fsys / 3
//*
//*                 QDLYR[QCD] has a range of 1127.
//*
//*                 When QDLYR[QCD] or QCR[DSCK] equals zero, the standard delay
//*                 of one-half the QSPI_CLK period is used.
//*                                                    (NOTE: RR5 uses this mode.)
//*
//********************************************************************************
//*
//*   Macro Name:   RR5_SPI_QDLYR_DTL
//*
//*   Description:  The command RAM delay after transmit enable bit, QCR[DT],
//*                 enables the programmable delay period from the negation of
//*                 the QSPI_CS signals until the start of the next transfer.
//*                 The delay after transfer can be used to provide a peripheral
//*                 deselect interval. A delay can also be inserted between
//*                 consecutive transfers to allow serial A/D converters to
//*                 complete conversion. There are two transfer delay options:
//*                  (1) the user can choose to delay a standard period after
//*                      serial transfer is complete
//*                  - or -
//*                  (2) the user can specify a delay period
//*
//*                 Writing a value to QDLYR[DTL] specifies a delay period.
//*                 QCR[DT] determines whether the standard delay period
//*                 (DT = 0) or the specified delay period (DT = 1) is used.
//*
//*                 The following expression is used to calculate the delay
//*                 when DT equals 1:                  (NOTE: RR5 uses this mode.)
//*
//*                                         32  QDLYR[DTL]
//*                 Delay after transfer = -----------------            (DT = 1)
//*                                              fsys/3
//*
//*                 ...where QDLYR[DTL] has a range of 1255.
//*
//*                 A zero value for DTL causes a delay-after-transfer value
//*                 of 8192/fsys/3. Standard delay period (DT = 0) is
//*                 calculated by the following:
//*
//*                                                    17
//*                 Standard delay after transfer = --------            (DT = 0)
//*                                                  fsys/3
//*
//*******************************************************************************/
//
//#ifdef CLOCK_180_MHZ
//    /* running with a 180MHz clock */
//    /*
//     * With BAUD = 0x60, QCD = 0x4 and DTL = 0x14...
//     * 1 SPI clock cycle = 3.2uS
//     * 1 SPI byte = 25.6uS (plus QCD = 6.7nS and DTL = 10.7uS)
//     * 1 14-byte block transfer requires roughly ~521uS including interrupt
//     *   software overhead
//     */
//  #define RR5_SPI_QDLYR_QCD       (0x04)
//  #define RR5_SPI_QDLYR_DTL       (0x14)
//#endif
//
//#ifdef CLOCK_210_MHZ  
//    /* running with a 210MHz clock */
//    /*
//     * If QSPI baud = 312.5KHz, QMR[BAUD] = 0x70
//     *              = 625.0KHz, QMR[BAUD] = 0x38
//     *              = 937.5KHz, QMR[BAUD] = 0x4A
//     *              =  1.25MHz, QMR[BAUD] = 0x1C
//     *
//     * If QSPI baud = 312.5KHz, QCD = 0x5 and DTL = 0x18...
//     * 1 SPI clock cycle = 3.2uS
//     * 1 SPI byte = 25.6uS (plus QCD = 8.3nS and DTL = 12.8uS)
//     * 1 14-byte block transfer requires roughly ~521uS including interrupt
//     *   software overhead
//     * If QSPI baud = 1.25MHz and DTL = 12.5uS...
//     * 1 14-byte block transfer requires roughly ~38.1uS including interrupt
//     *   software overhead
//     */
//  /*ok_8_3_06 #define RR5_SPI_QDLYR_QCD       (0x05) */
//  /*ok_8_3_06 #define RR5_SPI_QDLYR_DTL       (0x18) */
//  #define RR5_SPI_QDLYR_QCD       (0x03)
//  /*
//   * NOTE: The auxiliary processor needs a certain amount of time to process
//   *       each byte of an SPI transfer. The packets are presently 14 bytes in
//   *       size. The DTL delay is set to allow ample time for the Aux. CPU
//   *       to do it's 'housekeeping'. The below setting of 0x1A provides 
//   *       roughly 12.5uS of time between each byte of data transferred.
//   *       This is pretty close to the minimum required (as of 8.4.2006) with
//   *       the Aux. CPU's running at 8MHz and the SPI baud rate set to 1.25MHz.
//   *                                                       D. Staley (8.4.06)
//   */
//  /*ok@1MHz #define RR5_SPI_QDLYR_DTL       (0x16) */
//  #define RR5_SPI_QDLYR_DTL       (0x1A)
//#endif
//
//#ifdef CLOCK_240_MHZ
//  #define RR5_SPI_QDLYR_QCD       (0x04)
//  /*
//   * NOTE: The auxiliary processor needs a certain amount of time to process
//   *       each byte of an SPI transfer. The packets are presently 3 bytes in
//   *       size. The DTL delay is set to allow ample time for the Aux. CPU
//   *       to do it's 'housekeeping'. I tested a setting of 0x40, which provides
//   *       roughly 25.6uS of time between each byte of data transferred and it
//   *       worked fine. However, a setting of 0x30 (19.2us) did not work.
//   *                                                       D. Staley (1.15.2008)
//   *       The auxiliary processor software revision number was sometimes
//   *       getting out of sync. I increased the DTL to 38.4uS and the
//   *       loss-of-sync problem went away.                 D. Staley (2.15.2008)
//   *
//   *       0x40 =  64 = 25.6 uS
//   *       0x48 =  72 = 28.8 uS         D. Staley (1.15.2008)
//   *       0x60 =  96 = 38.4 uS         D. Staley (2.15.2008)
//   *       0x70 = 112 = 44.8 uS
//   */
//  #define RR5_SPI_QDLYR_DTL       (0x60)
//#endif
//
///*******************************************************************************
// *                              STRUCTURES / TYPEDEFS
// ******************************************************************************/
//
///*******************************************************************************
// * QSPI Mode Register (QMR)
// *
// * The QMR determines the basic operating modes of the QSPI module. Parameters
// *  such as QSPI_CLK polarity and phase, baud rate, master mode operation, and
// *  transfer size are determined by this register. The data output high
// *  impedance enable, DOHIE, controls the operation of QSPI_DOUT between data
// *  transfers. When DOHIE is cleared, QSPI_DOUT is actively driven between
// *  transfers. When DOHIE is set, QSPI_DOUT assumes a high impedance state.
// *
// *                                        NOTE
// * Because the QSPI does not operate in slave mode, the master mode enable
// * bit (QMR[MSTR]) must be set for the QSPI module to operate correctly.
// *
// * MSTR - Master mode enable.
// *        0 Reserved, do not use.
// *        1 The QSPI is in master mode. Must be set for the QSPI module to
// *          operate correctly.
// *
// * DOHIE - Data output high impedance enable. Selects QSPI_DOUT mode of
// *          operation.
// *         0 Default value after reset. QSPI_DOUT is actively driven between
// *           transfers.
// *         1 QSPI_DOUT is high impedance between transfers.
// *
// * BITS - Transfer size. Determines the number of bits to be transferred for
// *         each entry in the queue.
// *
// * CPOL - Clock polarity. Defines the clock polarity of QSPI_CLK.
// *        0 The inactive state value of QSPI_CLK is logic level 0.
// *        1 The inactive state value of QSPI_CLK is logic level 1.
// *
// * CPHA - Clock phase. Defines the QSPI_CLK clock-phase.
// *        0 Data captured on the leading edge of QSPI_CLK and changed on the
// *          following edge of QSPI_CLK.
// *        1 Data changed on the leading edge of QSPI_CLK and captured on the
// *          following edge of QSPI_CLK.
// *
// * BAUD - Baud rate divider. The baud rate is selected by writing a value in
// *        the range 2255. A value of zero disables the QSPI. A value of
// *        1 is an invalid setting. The desired QSPI_CLK baud rate is related
// *        to the internal bus clock and QMR[BAUD] by the following expression:
// *
// *                                  fsys / 3
// *        QMR[BAUD] = ------------------------------------
// *                     (2  [desired QSPI_CLK baud rate])
// *
// ******************************************************************************/
//
//#define BAUD_TO_QMR(x)     ((bitfield16_t)(CPU_INT_BUS_FREQ/(2*(x))))
//
//typedef union
//{
//   struct
//   {
//      bitfield16_t MSTR     : 1;
//      bitfield16_t DOHIE    : 1;
//      bitfield16_t BITS     : 4;
//      bitfield16_t CPOL     : 1;
//      bitfield16_t CPHA     : 1;
//      bitfield16_t BAUD     : 8;
//
//   } B;
//   uint16_t R;
//
//}  qspi_qmr_type;
//
//#define QSPI_QM_REG(MSTR,DOHIE,BITS,CPOL,CPHA,BAUD) \
//  (MSTR << 15 | DOHIE << 14 | BITS << 10 | CPOL << 9 | CPHA << 8 | BAUD)
//
///*******************************************************************************
// * QSPI Delay Register (QDLYR)
// *
// * SPE - QSPI enable. When set, the QSPI initiates transfers in master mode by
// *       executing commands in the command RAM. Automatically cleared by the
// *       QSPI when a transfer completes. The user can also clear this bit to
// *       abort transfer unless QIR[ABRTL] is set. The recommended method for
// *       aborting transfers is to set QWR[HALT].
// *
// * QCD - QSPI_CLK delay. When the DSCK bit in the command RAM is set this
// *        field determines the length of the delay from assertion of the
// *        chip selects to valid QSPI_CLK transition.
// *
// *       The following expression determines the actual delay before the
// *        QSPI_CLK leading edge:
// *
// *                                     QDLYR[QCD]
// *        QSPI_CS-to-QSPI_CLK delay = ------------
// *                                      fsys / 3
// *
// *        (QDLYR[QCD] has a range of 1127.)
// *
// * DTL - Delay after transfer. When the DT bit in the command RAM is set
// *        this field determines the length of delay after the serial transfer.
// *
// *       QCR[DT] determines whether the standard delay period (DT = 0) or the
// *       specified delay period (DT = 1) is used. The following expression is
// *       used to calculate the delay when DT = 1:
// *
// *                               32  QDLYR[DTL]
// *       Delay after transfer = -----------------
// *                                   fsys / 3
// *
// *       (QDLYR[DTL] has a range of 1255.) A zero value for DTL causes a
// *        delay-after-transfer value of 8192/Fsys/3.
// *
// ******************************************************************************/
//#define US_TO_QCD(x)        ((bitfield16_t)(bitfield16_t*(x)))
//#define US_TO_DTL(x)        ((bitfield16_t)(bitfield16_t*(x)/32))
//
//typedef union
//{
//   struct
//   {
//      bitfield16_t SPE      : 1;
//      bitfield16_t QCD      : 7;
//      bitfield16_t DTL      : 8;
//
//   } B;
//   uint16_t R;
//
//}  qspi_qdlyr_type;
//
//#define QSPI_DLY_REG(SPE,QCD,DTL) \
//  (SPE << 15 | QCD << 8 | DTL)
//
///*******************************************************************************
// * QSPI Wrap Register (QWR)
// *
// * HALT - Halt transfers. Assertion of this bit causes the QSPI to stop
// *        execution of commands once it has completed execution of the
// *        current command.
// *
// * WREN - Wraparound enable. Enables wraparound mode.
// *        0 Execution stops after executing the command pointed to by
// *          QWR[ENDQP].
// *        1 After executing command pointed to by QWR[ENDQP], wrap back to
// *          entry zero, or the entry pointed to by QWR[NEWQP] and continue
// *          execution.
// *
// * WRTO - Wraparound location. Determines where the QSPI wraps to in
// *         wraparound mode.
// *        0 Wrap to RAM entry zero.
// *        1 Wrap to RAM entry pointed to by QWR[NEWQP].
// *
// * CSIV - QSPI_CS inactive level.
// *        0 QSPI chip select outputs return to zero when not driven from the
// *          value in the current command RAM entry during a transfer (that is,
// *          inactive state is 0, chip selects are active high).
// *        1 QSPI chip select outputs return to one when not driven from the
// *          value in the current command RAM entry during a transfer (that is,
// *          inactive state is 1, chip selects are active low).
// *
// * ENDQP - End of queue pointer. Points to the RAM entry that contains the
// *          last transfer description in the queue.
// *
// * CPTQP - Completed queue entry pointer. Points to the RAM entry that contains
// *          the last command to have been completed. This field is read only.
// *
// * NEWQP - Start of queue pointer. This 4-bit field points to the first entry
// *          in the RAM to be executed on initiating a transfer.
// *
// ******************************************************************************/
//
//typedef union
//{
//   struct
//   {
//      bitfield16_t HALT     : 1;
//      bitfield16_t WREN     : 1;
//      bitfield16_t WRTO     : 1;
//      bitfield16_t CSIV     : 1;
//      bitfield16_t ENDQP    : 4;
//      bitfield16_t CPTQP    : 4;
//      bitfield16_t NEWQP    : 4;
//
//   } B;
//   uint16_t R;
//
//}  qspi_qwr_type;
//
//#define QSPI_QW_REG(HALT,WREN,WRTO,ENDQP,NEWQP)                                \
//    (SPIFIE<<15 | WREN<<14 | WRTO<<13 | CSIV<<12 | ENDQP<<8 | CPTQP<<4 | NEWQP)
//
///*******************************************************************************
// * QSPI Interrupt Register (QIR)
// *
// * WCEFB - Write collision access error enable. A write collision occurs
// *         during a data transfer when the RAM entry containing the current
// *         command is written to by the CPU with the QDR. When this bit is
// *         asserted, the write access to QDR results in an access error.
// *
// * ABRTB - Abort access error enable. An abort occurs when QDLYR[SPE] is
// *         cleared during a transfer. When set, an attempt to clear QDLYR[SPE]
// *         during a transfer results in an access error.
// *
// * ABRTL - Abort lock-out. When set, QDLYR[SPE] cannot be cleared by writing
// *         to the QDLYR. QDLYR[SPE] is only cleared by the QSPI when a transfer
// *         completes.
// *
// * WCEFE - Write collision (WCEF) interrupt enable.
// *         0 Write collision interrupt disabled
// *         1 Write collision interrupt enabled
// *
// * ABRTE - Abort (ABRT) interrupt enable.
// *         0 Abort interrupt disabled
// *         1 Abort interrupt enabled
// *
// * SPIFE - QSPI finished (SPIF) interrupt enable.
// *         0 SPIF interrupt disabled
// *         1 SPIF interrupt enabled
// *
// * WCEF - Write collision error flag. Indicates that an attempt has been made
// *        to write to the RAM entry that is currently being executed. Writing
// *        a 1 to this bit (w1c) clears it and writing 0 has no effect.
// *
// * ABRT - Abort flag. Indicates that QDLYR[SPE] has been cleared by the user
// *        writing to the QDLYR rather than by completion of the command queue
// *        by the QSPI. Writing a 1 to this bit (w1c) clears it and writing 0
// *        has no effect.
// *
// * SPIF - QSPI finished flag. Asserted when the QSPI has completed all the
// *        commands in the queue. Set on completion of the command pointed to
// *        by QWR[ENDQP], and on completion of the current command after
// *        assertion of QWR[HALT]. In wraparound mode, this bit is set every
// *        time the command pointed to by QWR[ENDQP] is completed. Writing a
// *        1 to this bit (w1c) clears it and writing 0 has no effect.
// *
// ******************************************************************************/
//
//typedef union
//{
//   struct
//   {
//      bitfield16_t WCEFB    : 1;
//      bitfield16_t ABRTB    : 1;
//      bitfield16_t          : 1;
//      bitfield16_t ABRTL    : 1;
//      bitfield16_t WCEFE    : 1;
//      bitfield16_t ABRTE    : 1;
//      bitfield16_t          : 1;
//      bitfield16_t SPIFE    : 1;
//      bitfield16_t          : 4;
//      bitfield16_t WCEF     : 1;
//      bitfield16_t ABRT     : 1;
//      bitfield16_t          : 1;
//      bitfield16_t SPIF     : 1;
//
//   } B;
//   uint16_t R;
//
//}  qspi_qir_type;
//
///*******************************************************************************
// * QSPI Address Register (QAR)
// *
// * The QAR is used to specify the location in the QSPI RAM that read and write
// *  operations affect. The transmit RAM is located at addresses 0x0 to 0xF, the
// *  receive RAM is located at 0x10 to 0x1F, and the command RAM is located at
// *  0x20 to 0x2F. These addresses refer to the QSPI RAM space, not the device
// *  memory map.
// *
// ******************************************************************************/
//
//typedef union
//{
//   struct
//   {
//      bitfield16_t          : 12;
//      bitfield16_t ADDR     : 4;
//
//   } B;
//   uint16_t R;
//
//}  qspi_qar_type;
//
///*******************************************************************************
// * QSPI Data Register (QDR)
// *
// * The QDR is used to access QSPI RAM indirectly. The CPU reads and writes all
// *  data from and to the QSPI RAM through this register.
// *
// ******************************************************************************/
//
//typedef uint16_t qspi_qdr_type;
//
///*******************************************************************************
// * Command RAM Registers (QCR0QCR15)
// *
// * The command RAM is accessed using the upper byte of the QDR; the QSPI
// *  cannot modify information in command RAM. There are 16 bytes in the
// *  command RAM. Each byte is divided into two fields. The chip select
// *  field enables external peripherals for transfer. The command field
// *  provides transfer operations.
// *
// *                                  NOTE
// *   The command RAM is accessed only using the most significant byte
// *   of QDR and indirect addressing based on QAR[ADDR].
// *
// * CONT - Continuous.
// *        0 Chip selects return to inactive level defined by QWR[CSIV] when
// *          a single word transfer is complete.
// *        1 Chip selects return to inactive level defined by QWR[CSIV] only
// *          after the transfer of the queue entries (max of 16 words).
// *
// *          Note: In order to keep the chip selects asserted for transfers
// *                beyond 16 words, the QWR[CSIV] bit must be set to control
// *                the level that the chip selects return to after the first
// *                transfer.
// *
// * BITSE - Bits per transfer enable.
// *         0 Eight bits
// *         1 Number of bits set in QMR[BITS]
// *
// * DT - Delay after transfer enable.
// *      0 Default reset value.
// *      1 The QSPI provides a variable delay at the end of serial transfer
// *        to facilitate interfacing with peripherals that have a latency
// *        requirement. The delay between transfers is determined by QDLYR[DTL].
// *
// * DSCK - Chip select to QSPI_CLK delay enable.
// *        0 Chip select valid to QSPI_CLK transition is one-half QSPI_CLK
// *          period.
// *        1 QDLYR[QCD] specifies the delay from QSPI_CS valid to QSPI_CLK.
// *
// * QSPI_CS - Peripheral chip selects. Used to select an external device for
// *           serial data transfer. More than one chip select may be active
// *           at once, and more than one device can be connected to each chip
// *           select. Bits 11-8 map directly to the corresponding QSPI_CSn
// *           pins. If more than four chip selects are needed, then an external
// *           demultiplexor can be used with the QSPI_CSn pins.
// *
// *           Note: Not all chip selects may be available on all device
// *                 packages.
// *
// ******************************************************************************/
//
//typedef union
//{
//   struct
//   {
//      bitfield16_t CONT     : 1;
//      bitfield16_t BITSE    : 1;
//      bitfield16_t DT       : 1;
//      bitfield16_t DSCK     : 1;
//      bitfield16_t QSPI_CS  : 4;
//      bitfield16_t          : 8;
//
//   } B;
//   uint16_t R;
//
//}  qspi_qcr_type;
//
//typedef struct
//{
//  uint16_t  Rx_Buffer[QSPI_RAM_BLOCK_SIZE];
//  uint16_t  Tx_Buffer[QSPI_RAM_BLOCK_SIZE];
//  uint8_t   Rx_Index;
//  uint8_t   Tx_Index;
//  uint16_t  Tx_Error_Count;
//  uint16_t  Sync_Check_Count;
//  uint8_t   Chip_Select;
//  uint8_t   Is_In_Sync;
//} qspi_aux_cpu_t;
//
///*******************************************************************************
// *                              VARIABLES
// ******************************************************************************/
//
//extern qspi_aux_cpu_t  sAux_1_Cpu;
//extern qspi_aux_cpu_t  sAux_2_Cpu;
//
///*******************************************************************************
// *                              PROTOTYPES
// ******************************************************************************/
//
//void vQSPI_Init( void );
//
//void vQSPI_Configure( void );
//
//#endif /* HW_QSPI_H */
