/*
 * main.c
 */

#include <stdbool.h>
#include <stdint.h>

#include <inc/tm4c123gh6pm.h>

#define CLOCK_FREQ_80MHz_RCC2_DIVISORS     (2 << SYSCTL_RCC2_SYSDIV2_S)
#define CLOCK_FREQ_66_67MHz_RCC2_DIVISORS ((2 << SYSCTL_RCC2_SYSDIV2_S) | SYSCTL_RCC2_SYSDIV2LSB)
#define CLOCK_FREQ_50MHz_RCC2_DIVISORS    ((3 << SYSCTL_RCC2_SYSDIV2_S) | SYSCTL_RCC2_SYSDIV2LSB)
#define CLOCK_FREQ_44_44MHz_RCC2_DIVISORS  (4 << SYSCTL_RCC2_SYSDIV2_S)
#define CLOCK_FREQ_40MHz_RCC2_DIVISORS    ((4 << SYSCTL_RCC2_SYSDIV2_S) | SYSCTL_RCC2_SYSDIV2LSB)

#define RCC2_DIVISORS CLOCK_FREQ_80MHz_RCC2_DIVISORS

void prvGetRegistersFromStack( uint32_t *pulFaultStackAddress )
{
/* These are volatile to try and prevent the compiler/linker optimising them
away as the variables never actually get used.  If the debugger won't show the
values of the variables, make them global my moving their declaration outside
of this function. */
volatile uint32_t r0;
volatile uint32_t r1;
volatile uint32_t r2;
volatile uint32_t r3;
volatile uint32_t r12;
volatile uint32_t lr; /* Link register. */
volatile uint32_t pc; /* Program counter. */
volatile uint32_t psr;/* Program status register. */

    r0 = pulFaultStackAddress[ 0 ];
    r1 = pulFaultStackAddress[ 1 ];
    r2 = pulFaultStackAddress[ 2 ];
    r3 = pulFaultStackAddress[ 3 ];

    r12 = pulFaultStackAddress[ 4 ];
    lr = pulFaultStackAddress[ 5 ];
    pc = pulFaultStackAddress[ 6 ];
    psr = pulFaultStackAddress[ 7 ];

    /* When the following line is hit, the variables contain the register values. */
    for( ;; );
}

void HardFault_Handler(void)
{
    __asm volatile
    (
        " tst lr, #4                                                \n"
        " ite EQ                                                    \n"
        " mrseq r0, msp                                             \n"
        " mrsne r0, psp                                             \n"
        " ldr r1, [r0, #24]                                         \n"
        " ldr r2, handler2_address_const                            \n"
        " bx r2                                                     \n"
        "handler2_address_const: .word prvGetRegistersFromStack    \n"
    );
}

void await_not_busy (void)
{
	while (FLASH_FMC_R & FLASH_FMC_ERASE)
	{

	}
}

void erase_page (uint32_t address)
{
	await_not_busy ();

	asm ("	mov r5, #5");
	asm ("	mov r6, #6");
	asm ("	mov r7, #7");
	asm ("	mov r8, #8");
	asm ("	mov r9, #9");
	asm ("	mov r10, sp");
	asm ("	mov r11, sp");
	asm ("	mov r12, sp");

    FLASH_FMA_R = address;
    FLASH_FMC_R = (FLASH_FMC_WRKEY | FLASH_FMC_ERASE);

    asm ("	sub r11, #4");
    asm ("	sub r11, #8");
    asm ("	sub r11, #16");
    asm ("	sub r11, #32");
    asm ("	sub r11, #64");
    asm ("	sub r11, #128");
	await_not_busy ();
    asm ("	sub r11, #256");
    asm ("	sub r11, #512");

    for (;;)
    {

    }
}

void SetupClock() {
  // Run using internal oscillator
  // Use RCC2 instead of RCC
  SYSCTL_RCC2_R |= SYSCTL_RCC2_USERCC2;
  // Disable PLL
  SYSCTL_RCC2_R |= SYSCTL_RCC2_BYPASS2;
  SYSCTL_RCC2_R |= SYSCTL_RCC2_OSCSRC2_IO;
  // Power on the 400 Mhz PLL
  SYSCTL_RCC2_R &= ~SYSCTL_RCC2_PWRDN2;
  SYSCTL_RCC2_R |= SYSCTL_RCC2_DIV400;

  // Set up system divider (2 * SYSDIV + 1 + LSB)
  SYSCTL_RCC2_R = (SYSCTL_RCC2_R &
                   ~(SYSCTL_RCC2_SYSDIV2_M | SYSCTL_RCC2_SYSDIV2LSB)) |
                		   RCC2_DIVISORS;

  // Wait for the PLL to lock by polling PLLLRIS
  while ((SYSCTL_RIS_R & SYSCTL_RIS_PLLLRIS) == 0) {}

  // Enable PLL
  SYSCTL_RCC2_R &= ~SYSCTL_RCC2_BYPASS2;
}

int main(void)
{
	SetupClock ();
	erase_page (73 * 1024);
	
	return 0;
}
