/*
 * main.c
 *
 * Program to test umktime/ulocaltime routines, and RTC calendar on TM4C123 Launchpad
 * This program enters hibernate, waits for 5 seconds, wakes up and in interrupt send
 * to UART0 the formatted date/time.
 *
 * There are two sets of variables, one for reading, one for writing. Usually you must
 * use only one, but here are two for debugging/comparations.
 *
 */

#include <stdint.h>
#include <stdbool.h>
#include <stdio.h>
#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
#include "inc/hw_hibernate.h"
#include "driverlib/debug.h"
#include "driverlib/fpu.h"
#include "driverlib/gpio.h"
#include "driverlib/hibernate.h"
#include "driverlib/interrupt.h"
#include "driverlib/pin_map.h"
#include "driverlib/rom.h"
#include "driverlib/rom_map.h"
#include "driverlib/sysctl.h"
#include "driverlib/uart.h"
#include "utils/ustdlib.h"

//*****************************************************************************
//
// Global data to be used by this example
//
//*****************************************************************************
time_t  calendar_write;
time_t  calendar_read;
struct  tm tm1 = {50,59,23,5,6,2014-1900,5,0,0}; // to compute seconds since epoch
struct  tm tm2 = {0};                            // to read back the calendar
uint8_t buf_er[] = "Wrong initialization structure!\n";
uint8_t buf_rz[48];

void UARTSend(uint8_t *pui8Buffer, uint32_t ui32Count);
//*****************************************************************************
//
// The error routine that is called if the driver library encounters an error.
//
//*****************************************************************************
#ifdef DEBUG
void
__error__(char *pcFilename, uint32_t ui32Line)
{
}
#endif
//*****************************************************************************
//
// Hibernate handler
//
//*****************************************************************************
void HibernateHandler(void)
{
  // reset interrupt flag
  uint32_t ui32Status = HibernateIntStatus(1);
  ROM_HibernateIntClear(ui32Status);
  //
  calendar_read = HibernateRTCGet();
  ulocaltime(calendar_read, &tm2);
  // format the output for uart
  strftime((char *)buf_rz, 32, "%c\r\n", &tm2);
  UARTSend((uint8_t *)buf_rz, 32);
  // Keep the interrupt hapening
  HibernateRTCMatchSet(0,HibernateRTCGet()+5);
}

//*****************************************************************************
//
// Send a string to the UART.
//
//*****************************************************************************
void
UARTSend(uint8_t *pui8Buffer, uint32_t ui32Count)
{
    //
    // Loop while there are more characters to send.
    //
    while(ui32Count--)
    {
        //
        // Write the next character to the UART.
        //
        ROM_UARTCharPut(UART0_BASE, *pui8Buffer++);
    }
}

int main(void) {
    //
    // Enable lazy stacking for interrupt handlers.  This allows floating-point
    // instructions to be used within interrupt handlers, but at the expense of
    // extra stack usage.
    //
    ROM_FPUEnable();
    ROM_FPULazyStackingEnable();

    //
    // Set the clocking to run directly from the crystal.
    //
    ROM_SysCtlClockSet(SYSCTL_SYSDIV_1 | SYSCTL_USE_OSC | SYSCTL_OSC_MAIN |
                       SYSCTL_XTAL_16MHZ);
    //
    // Enable the GPIO port that is used for the on-board LED.
    //
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOF);

    //
    // Enable the GPIO pins for the LED (PF2).
    //
    ROM_GPIOPinTypeGPIOOutput(GPIO_PORTF_BASE, GPIO_PIN_2);

    //
    // Enable the peripherals used by this example.
    //
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_UART0);
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);

    //
    // Enable processor interrupts.
    //
    ROM_IntMasterEnable();

    //
    // Set GPIO A0 and A1 as UART pins.
    //
    ROM_GPIOPinConfigure(GPIO_PA0_U0RX);
    ROM_GPIOPinConfigure(GPIO_PA1_U0TX);
    ROM_GPIOPinTypeUART(GPIO_PORTA_BASE, GPIO_PIN_0 | GPIO_PIN_1);

    //
    // Configure the UART for 115,200, 8-N-1 operation.
    //
    ROM_UARTConfigSetExpClk(UART0_BASE, ROM_SysCtlClockGet(), 115200,
                           (UART_CONFIG_WLEN_8 | UART_CONFIG_STOP_ONE |
                            UART_CONFIG_PAR_NONE));
    //
    // Configure the hibernation module
    //
    ROM_HibernateEnableExpClk(0);
    ROM_HibernateRTCEnable();
    HibernateClockConfig(HIBERNATE_OSC_HIGHDRIVE);
    //
    HibernateIntRegister(HibernateHandler);
    HibernateIntEnable(HIBERNATE_INT_RTC_MATCH_0);
    HibernateWakeSet(HIBERNATE_WAKE_RTC);

    //Set up counter mode for RTC
    HibernateCounterMode(HIBERNATE_COUNTER_RTC);
    //
    // Calculate the calendar data to be written
    //
    calendar_write = umktime(&tm1);
    if (calendar_write == (uint32_t)(-1)){
    	UARTSend((uint8_t *)buf_er, sizeof(buf_er));
    	while(1);
    }
    else {
    	HibernateRTCSet(calendar_write);
    	//Set an interrupt for the RTC after one second.
    	HibernateRTCMatchSet(0,HibernateRTCGet()+5);
    }

	while(1){
		//do nothing - safety
	}
	// return 0;
}
