//*****************************************************************************
//
// sw01267.c - Example use case of the EEPROM emulation drivers.
//
// Copyright (c) 2005-2009 Luminary Micro, Inc.  All rights reserved.
// 
// Software License Agreement
// 
// Luminary Micro, Inc. (LMI) is supplying this software for use solely and
// exclusively on LMI's microcontroller products.
// 
// The software is owned by LMI and/or its suppliers, and is protected under
// applicable copyright laws.  All rights are reserved.  You may not combine
// this software with "viral" open-source software in order to form a larger
// program.  Any use in violation of the foregoing restrictions may subject
// the user to criminal sanctions under applicable laws, as well as to civil
// liability for the breach of the terms and conditions of this license.
// 
// THIS SOFTWARE IS PROVIDED "AS IS".  NO WARRANTIES, WHETHER EXPRESS, IMPLIED
// OR STATUTORY, INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE.
// LMI SHALL NOT, IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR
// CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
// 
// This is part of revision 5132 of the EEPROM Emulation Example.
//
//*****************************************************************************

#include "inc/hw_types.h"
#include "inc/hw_memmap.h"
#include "driverlib/sysctl.h"
#include "driverlib/systick.h"
#include "driverlib/uart.h"
#include "driverlib/gpio.h"
#include "utils/ustdlib.h"
#include "utils/uartstdio.h"
#include "utils/cmdline.h"
#include "softeeprom/softeeprom.h"

//*****************************************************************************
//
//! \addtogroup example_list
//!
//! This software is intended to provide a simple example of using the EEPROM
//! emulation drivers.  UART0 is used as a user interface to allow the user to
//! read and write emulated EEPROM variables.  In addition, the user can dump
//! and clear the emulated EEPROM contents.
//!
//
//*****************************************************************************

//*****************************************************************************
//
//! \page softeeprom_example_intro Introduction
//!
//! This module provides a simple example using the EEPROM emulation
//! drivers. UART0 is used as a user interface to allow the user to read and
//! write up to 16 emulated EEPROM variables.  In addition, the user can dump 
//! and clear the emulated EEPROM contents.  The example is configured for a
//! board with an 8-MHz crystal.  Change the appropriate parameter in the call to
//! SysCtlClockSet() if a different crystal is used on the board.
//!
//! To run the example:
//! -# Start Hyperterminal.
//! -# Select the COM port associated with the board.
//! -# Select 115200 bits per second.
//! -# Select 8 data bits.
//! -# Select no parity.
//! -# Select 1 stop bit.
//! -# Select no flow control.
//!
//! Once the example is running, you should see a prompt in the Hyperterminal
//! window.  IDs 0 - 15 are supported.  The data size is 16 bits.
//!
//!
//! Available commands:
//!
//! help : Display list of commands
//!
//! - h    : alias for help
//!
//! - ?    : alias for help
//!
//! - w    : Write soft EEPROM - Usage: w <ID> <Value>
//!
//! - r    : Read soft EEPROM  - Usage: r <ID>
//!
//! - c    : Clear soft EEPROM - Usage: c
//!
//! - d    : Dump soft EEPROM  - Usage: d
//!
//!
//!
//! Example:
//!
//! > r 0
//!
//! ID 0: 0xffff (65535) - The variable with this ID has not been initialized.
//! 
//! > r 1
//!
//! ID 1: 0xffff (65535) - The variable with this ID has not been initialized.
//! 
//! > w 0 0xbeef
//! 
//! > w 1 0xdead
//! 
//! > r 0
//!
//! ID 0: 0xbeef (48879)
//! 
//! > r 1
//!
//! ID 1: 0xdead (57005)
//! 
//! > c
//! 
//! > r 0
//!
//! ID 0: 0xffff (65535) - The variable with this ID has not been initialized.
//! 
//! > r 1
//!
//! ID 1: 0xffff (65535) - The variable with this ID has not been initialized.
//!
//
//*****************************************************************************

//*****************************************************************************
//
//! \defgroup softeeprom_example Definitions
//! @{
//
//*****************************************************************************

//*****************************************************************************
//
//! The error code returned when an error occurs that does not need to be
//! handled other than notifying the user.  -1 and -2 are already taken by the
//! command line processor.
//
//*****************************************************************************
#define ERROR_UNHANDLED		    -3

//*****************************************************************************
//
//! The size of the buffer that holds the command line entry.
//
//*****************************************************************************
#define CMD_BUF_SIZE            64

//*****************************************************************************
//
//! The buffer that holds the command line entry.
//
//*****************************************************************************
static char g_cCmdBuf[CMD_BUF_SIZE];

//*****************************************************************************
//
// The error routine that is called if the driver library encounters an error.
//
//*****************************************************************************
#ifdef DEBUG
void
__error__(char *pcFilename, unsigned long ulLine)
{
}
#endif

//*****************************************************************************
//
//! Outputs the error and quits.
//!
//! This function will output an error message for the given error code and
//! enter an infinite loop.  The message is sent out via UART0.
//!
//! \param ulError is the error code returned by the soft EEPROM drivers.
//!
//! \return None.
//
//*****************************************************************************
void
OutputErrorAndQuit(unsigned long ulError)
{
    //
    // Switch and output based on the error code.
    //
    switch(ulError & 0x7FFF)
    {
        case ERR_NOT_INIT:
            UARTprintf("\r\nERROR: Soft EEPROM not initialized!");
            break;
        case ERR_ILLEGAL_ID:
            UARTprintf("\r\nERROR: Illegal ID used!");
            break;
        case ERR_PG_ERASE:
            UARTprintf("\r\nERROR: Soft EEPROM page erase error!");
            break;
        case ERR_PG_WRITE:
            UARTprintf("\r\nERROR: Soft EEPROM page write error!");
            break;
        case ERR_ACTIVE_PG_CNT:
            UARTprintf("\r\nERROR: Active soft EEPROM page count error!");
            break;
        case ERR_RANGE:
            UARTprintf("\r\nERROR: Soft EEPROM specified out of range!");
            break;
        case ERR_AVAIL_ENTRY:
            UARTprintf("\r\nERROR: Next available entry error!");
            break;
        case ERR_TWO_ACTIVE_NO_FULL:
            UARTprintf("\r\nERROR: Two active pages found but not full!");
            break;
        default:
            UARTprintf("\r\nERROR: Unidentified Error");
            break;
    }
                           
    //
    // Did the error occur during the swap operation?
    //
    if(ulError & ERR_SWAP)
    {
        //
        // Indicate that the error occurred during the swap operation.
        //
        UARTprintf("\r\nOccurred during the swap operation.");
    }
    
    //
    // Enter an infinite loop
    //
    while(1)
    {
        //
        // Do nothing.
        //
    }
}

//*****************************************************************************
//
//! Implements the help command.
//!
//! This function implements the "help" command.  It prints a simple list of
//! the available commands with a brief description. An argument count (argc)
//! of zero is expected.
//!
//! \param argc is the argument count from the command line.
//!
//! \param argv is a pointer to the argument buffer from the command line.
//!
//! \return This function always returns a value of 0.
//
//*****************************************************************************
int
Cmd_help(int argc, char *argv[])
{
    tCmdLineEntry *pEntry;

    //
    // Tell the user what this program allows you to do.
    //
    UARTprintf("\n------------------------------------------------------\n");
    UARTprintf("This application allows you to read, write, clear,\n");
    UARTprintf("and dump the soft EEPROM.  IDs 0-15 are supported.\n");

    //
    // Print some header text before listing the available commands.
    //
    UARTprintf("\nAvailable commands\n");
    UARTprintf("------------------\n");

    //
    // Point at the beginning of the command table.
    //
    pEntry = &g_sCmdTable[0];

    //
    // Enter a loop to read each entry from the command table.  The end of the
    // table has been reached when the command name is NULL.
    //
    while(pEntry->pcCmd)
    {
        //
        // Print the command name and the brief description.
        //
        UARTprintf("%s%s\n", pEntry->pcCmd, pEntry->pcHelp);

        //
        // Advance to the next entry in the table.
        //
        pEntry++;
    }

    //
    // Return success.
    //
    return(0);
}

//*****************************************************************************
//
//! Implements the read command.
//!
//! This function implements the "r" (read) command.  It reads the data for the
//! specified ID from the soft EEPROM and displays the value to the user. An
//! argument count (argc) of one is expected.
//!
//! \param argc is the argument count from the command line.
//!
//! \param argv is a pointer to the argument buffer from the command line.
//!
//! \return A value of 0 indicates that the command was successful.  A non-zero
//! value indicates a failure.
//
//*****************************************************************************
int
Cmd_read(int argc, char *argv[])
{
    long lEEPROMRetStatus;    
    tBoolean bFound;
    unsigned short usData;
    unsigned char ucID;

    //
    // Check that there is at least 1 argument provided with the command.
    // Any extra will simply be ignored.
    //
    if(argc < 2)
    {
        //
        // Print the error to the user.
        //
        UARTprintf("The read command requires an argument.\n");
        UARTprintf("Type help for usage details.\n");

        //
        // Return ERROR_UNHANDLED so that the caller knows there was an 
        // error, but that it does not need to handle the error.
        //
        return(ERROR_UNHANDLED);
    }

    //
    // Get the ID for the variable to read.  Use a radix of zero to handle
    // both decimal and hexidecimal.
    //
    ucID = ustrtoul(argv[1], '\0', 0);

    //
    // Check that the ID is in the range supported by this example.
    //
    if(ucID > 15)
    {
        //
        // Print the error to the user.
        //
        UARTprintf("Only IDs 0-15 are supported by this example.\n");

        //
        // Return ERROR_UNHANDLED so that the caller knows there was an 
        // error, but that it does not need to handle the error.
        //
        return(ERROR_UNHANDLED);
    }

    //
    // Read the value associated with the given ID from the EEPROM.
    //
    lEEPROMRetStatus = SoftEEPROMRead(ucID, &usData, &bFound);
    
    //
    // Check to see if an error occurred
    //
    if(lEEPROMRetStatus != 0)
    {
        //
        // Indicate that an error occurred during a read operation.
        //
        UARTprintf("\rAn error occurred during a soft EEPROM read operation");
                   
        //
        // Output the error to the UART and quit.
        //
        OutputErrorAndQuit(lEEPROMRetStatus);
    }

    //
    // Output the result to the user.
    //
    UARTprintf("ID %d: 0x%04X (%d) %s\n", ucID, usData, usData, bFound ? "" : 
               "- The variable with this ID has not been initialized.");

    //
    // Return success.
    //
    return(0);
}

//*****************************************************************************
//
//! Implements the write command.
//!
//! This function implements the "w" (write) command.  It writes the specified 
//! data for the specified ID to the soft EEPROM.  An argument count (argc) of
//! two is expected.
//!
//! \param argc is the argument count from the command line.
//!
//! \param argv is a pointer to the argument buffer from the command line.
//!
//! \return A value of 0 indicates that the command was successful.  A non-zero
//! value indicates a failure.
//
//*****************************************************************************
int
Cmd_write(int argc, char *argv[])
{
    long lEEPROMRetStatus;
    unsigned char ucID;
    unsigned short usData;

    //
    // Check that there is at least 2 arguments provided with the command.
    // Any extra will simply be ignored.
    //
    if(argc < 3)
    {
        //
        // Print the error to the user.
        //
        UARTprintf("The write command requires two arguments.\n");
        UARTprintf("Type help for usage details.\n");

        //
        // Return ERROR_UNHANDLED so that the caller knows there was an 
        // error, but that it does not need to handle the error.
        //
        return(ERROR_UNHANDLED);
    }
    
    //
    // Get the ID for the variable to write.  Use a radix of zero to handle
    // both decimal and hexidecimal.
    //
    ucID = ustrtoul(argv[1], '\0', 0);

    //
    // Check that the ID is in the range supported by this example.
    //
    if(ucID > 15)
    {
        //
        // Print the error to the user.
        //
        UARTprintf("Only IDs 0-15 are supported by this example.\n");

        //
        // Return ERROR_UNHANDLED so that the caller knows there was an 
        // error, but that it does not need to handle the error.
        //
        return(ERROR_UNHANDLED);
    }
    
    //
    // Get the value for the variable to write.  Use a radix of zero to handle
    // both decimal and hexidecimal.
    //
    usData = ustrtoul(argv[2], '\0', 0);

    //
    // Write the specified value for the specified ID to the EEPROM.
    //
    lEEPROMRetStatus = SoftEEPROMWrite(ucID, usData);
    
    //
    // Check to see if an error occurred
    //
    if(lEEPROMRetStatus != 0)
    {
        //
        // Indicate that an error occurred during a write operation.
        //
        UARTprintf("\rAn error occurred during a soft EEPROM write "          \
                   "operation");
                   
        //
        // Output the error to the UART and quit.
        //
        OutputErrorAndQuit(lEEPROMRetStatus);
    }

    //
    // Return success.
    //
    return(0);
}

//*****************************************************************************
//
//! Implements the clear command.
//!
//! This function implements the "c" (clear) command.  It clears the contents
//! of the soft EEPROM. An argument count (argc) of zero is expected.
//!
//! \param argc is the argument count from the command line.
//!
//! \param argv is a pointer to the argument buffer from the command line.
//!
//! \return A value of 0 indicates that the command was successful.  A non-zero
//! value indicates a failure.
//
//*****************************************************************************
int
Cmd_clear(int argc, char *argv[])
{
    long lEEPROMRetStatus;

    //
    // Clear the EEPROM.
    //
    lEEPROMRetStatus = SoftEEPROMClear();
    
    //
    // Check to see if an error occurred
    //
    if(lEEPROMRetStatus != 0)
    {
        //
        // Indicate that an error occurred during a clear operation.
        //
        UARTprintf("\rAn error occurred during a soft EEPROM clear "          \
                   "operation");
                   
        //
        // Output the error to the UART and quit.
        //
        OutputErrorAndQuit(lEEPROMRetStatus);
    }

    //
    // Return success.
    //
    return(0);
}

//*****************************************************************************
//
//! Implements the dump command.
//!
//! This function implements the "d" (dump) command.  It dumps the contents of
//! the soft EEPROM. An argument count (argc) of zero is expected.
//!
//! \param argc is the argument count from the command line.
//!
//! \param argv is a pointer to the argument buffer from the command line.
//!
//! \return A value of 0 indicates that the command was successful.  A non-zero
//! value indicates a failure.
//
//*****************************************************************************
int
Cmd_dump(int argc, char *argv[])
{
    tBoolean bFound;
    long lEEPROMRetStatus;
    unsigned short usData;
    unsigned char ucID;

    //
    // Loop through the IDs.
    //
    for(ucID = 0; ucID <= 15; ucID++)
    { 
        //
        // Read the value associated with the given ID from the EEPROM.
        //
        lEEPROMRetStatus = SoftEEPROMRead(ucID, &usData, &bFound);
        
        //
        // Check to see if an error occurred
        //
        if(lEEPROMRetStatus != 0)
        {
            //
            // Indicate that an error occurred during a read operation.
            //
            UARTprintf("\rAn error occurred during a soft EEPROM read"        \
                       "operation");
        
            //
            // Output the error to the UART and quit.
            //
            OutputErrorAndQuit(lEEPROMRetStatus);
        }

        //
        // Output the result to the user.
        //
        UARTprintf("ID %d: 0x%04X (%d) %s\n", ucID, usData, usData, bFound ? ""
                    : "- The variable with this ID has not been initialized.");
    }

    //
    // Return success.
    //
    return(0);
}

//*****************************************************************************
//
//! The table that holds the command names, implementing functions, and brief
//! description.
//
//*****************************************************************************
tCmdLineEntry g_sCmdTable[] =
{
    { "help",   Cmd_help,      " : Display list of commands" },
    { "h",      Cmd_help,   "    : alias for help" },
    { "?",      Cmd_help,   "    : alias for help" },
    { "w",      Cmd_write,  "    : Write soft EEPROM - Usage: w <ID> <Value>"},
    { "r",      Cmd_read,   "    : Read soft EEPROM  - Usage: r <ID>" },
    { "c",      Cmd_clear,  "    : Clear soft EEPROM - Usage: c" },
    { "d",      Cmd_dump,   "    : Dump soft EEPROM  - Usage: d" },
    { 0, 0, 0 }
};



//*****************************************************************************
//
// The main application entry point.  The emulated EEPROM is initialized and
// then an infinite loop is entered waiting for user input.
//
//*****************************************************************************
int
main(void)
{
    int iCmdRedStatus;
    long lEEPROMRetStatus;

    SysCtlClockSet(SYSCTL_SYSDIV_1 | SYSCTL_USE_OSC | SYSCTL_OSC_MAIN |
                   SYSCTL_XTAL_8MHZ);
    
    //
    // Enable the peripherals used by this example.
    //
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);
    
    //
    // Set GPIO A0 and A1 as UART pins.
    //
    GPIOPinTypeUART(GPIO_PORTA_BASE, GPIO_PIN_0 | GPIO_PIN_1);

    //
    // Initialize the UARTStdio module for use with UART0.
    //
    UARTStdioInit(0);

    //
    // Initialize the EEPROM emulation region.
    //
    lEEPROMRetStatus = SoftEEPROMInit(0x4000, 0x5000, 0x800);
    
    //
    // Check to see if an error occurred.
    //
    if(lEEPROMRetStatus != 0)
    {
        //
        // Indicate that an error occurred during initialization.
        //
        UARTprintf("\rAn error occurred during Soft EEPROM initialization!"); 
        
        //
        // Output the error to the UART and quit.
        //
        OutputErrorAndQuit(lEEPROMRetStatus);
    }

    //
    // Print hello message to user.
    //
    UARTprintf("\n\nSoft EEPROM Example Program\n");
    UARTprintf("Type \'help\' for help.\n");
  
    //
    // Enter infinite loop waiting for user input.
    //
    while(1)
    {
        //
        // Print a prompt to the console.
        //
        UARTprintf("\n> ");

        //
        // Get a line of text from the user.
        //
        UARTgets(g_cCmdBuf, sizeof(g_cCmdBuf));

        //
        // Pass the line from the user to the command processor.  It will be
        // parsed and valid commands executed.
        //
        iCmdRedStatus = CmdLineProcess(g_cCmdBuf);
        
        //
        // Handle the case of bad command.
        //
        if(iCmdRedStatus == CMDLINE_BAD_CMD)
        {
            UARTprintf("Bad command!\n");
        }
        
        //
        // Handle the case of too many arguments.
        //
        else if(iCmdRedStatus == CMDLINE_TOO_MANY_ARGS)
        {
            UARTprintf("Too many arguments for command processor!\n");
        }
    }
}

//*****************************************************************************
//
// Close the Doxygen group.
//! @}
//
//*****************************************************************************
