/** @file sys_pmm.h
*   @brief PMM Driver Header File
*   @date 5.Oct.2012
*   @version 03.03.00
*   
*   This file contains:
*   - Definitions
*   - Types
*   .
*   which are relevant for the System driver.
*/

/* (c) Texas Instruments 2009-2012, All rights reserved. */

#ifndef __SYS_PMM_H__
#define __SYS_PMM_H__

#include "sys_common.h"

//Bit Masks
#define PMM_LOGICPDPWRCTRL0_LOGICPDON0 (0xF << 24)		/*PD2*/
#define PMM_LOGICPDPWRCTRL0_LOGICPDON1 (0xF << 16)		/*PD3*/
#define PMM_LOGICPDPWRCTRL0_LOGICPDON2 (0xF << 8)		/*PD4*/
#define PMM_LOGICPDPWRCTRL0_LOGICPDON3 (0xF << 0)		/*PD5*/

#define PMM_MEMPDPWRCTRL0_MEMPDON0 (0xF << 24)			/*RAM_PD1*/
#define PMM_MEMPDPWRCTRL0_MEMPDON1 (0xF << 16)			/*RAM_PD2*/
#define PMM_MEMPDPWRCTRL0_MEMPDON2 (0xF << 8)			/*RAM_PD3*/

#define PMM_LOGICPDPWRSTAT_DOMAINON	(1 << 8)
#define PMM_LOGICPDPWRSTAT_LOGICPDPWRSTAT	(0x3 << 0)
#define PMM_MEMPDPWRSTAT_DOMAINON	(1 << 8)
#define PMM_MEMPDPWRSTAT_MEMPDPWRSTAT	(0x3 << 0)
#define PMM_GLOBALCTRL1_AUTOCLKWAKEENA	(1 << 0)

//Configuration Register Values
#define PMM_LOGICPDPWRCTRL0_C0NFIGVALUE   (0xA << 24) \
										| (0xA << 16) \
										| (0xA << 8)  \
										| (0xA << 0)

#define PMM_MEMPDPWRCTRL0_CONFIGVALUE 	  (0xA << 24) \
										| (0xA << 16) \
										| (0xA << 8)

#define PMM_PDCLKDISREG_CONFIGVALUE		  ((0xA >> 3 ) << 0)\
										| ((0xA >> 3) << 1)\
										| ((0xA >> 3) << 2)\
										| ((0xA >> 3) << 3)

#define PMM_GLOBALCTRL1_CONFIGVALUE		0 << 8 | 0 << 0

typedef enum pmmLogicPDTag
{
	PMM_LOGICPD1	= 4U,	/*-- NOT USED*/
	PMM_LOGICPD2	= 0U,
	PMM_LOGICPD3	= 1U,
	PMM_LOGICPD4	= 2U,
	PMM_LOGICPD5	= 3U
}pmm_LogicPD_t;

typedef enum pmmMemPDTag
{
	PMM_MEMPD1 = 0U,
	PMM_MEMPD2 = 1U,
	PMM_MEMPD3 = 2U
}pmm_MemPD_t;

typedef enum pmmModeTag
{
	LockStep				= 0x0,
	SelfTest				= 0x6,
	ErrorForcing			= 0x9,
	SelfTestErrorForcing	= 0xF
}pmm_Mode_t;

/*Pmm Configuration Registers*/
typedef struct pmm_config_reg
{
	uint32_t CONFIG_LOGICPDPWRCTRL0;
	uint32_t CONFIG_MEMPDPWRCTRL0;
	uint32_t CONFIG_PDCLKDISREG;
	uint32_t CONFIG_GLOBALCTRL1;
}pmm_config_reg_t;

/* Pmm Register Frame Definition */
/** @struct pmmBase
*   @brief Pmm Register Frame Definition
*
*   This type is used to access the Pmm Registers.
*/
/** @typedef pmmBase_t
*   @brief Pmm Register Frame Type Definition
*
*   This type is used to access the Pmm Registers.
*/
typedef volatile struct pmmBase
{
	uint32_t LOGICPDPWRCTRL0;		/* 0x0000: Logic Power Domain Control Register 0 */	
    uint32_t :32U;					/* 0x0004 */
    uint32_t :32U;					/* 0x0008 */
    uint32_t :32U;					/* 0x000C */
	uint32_t MEMPDPWRCTRL0;			/* 0x0010: Memory Power Domain Control Register 0 */
    uint32_t :32U;					/* 0x0014 */
    uint32_t :32U;					/* 0x0018 */
    uint32_t :32U;					/* 0x001C */
    uint32_t PDCLKDISREG;			/* 0x0020: Power Domain Clock Disable Register */
    uint32_t PDCLKDISSETREG;		/* 0x0024: Power Domain Clock Disable Set Register */
    uint32_t PDCLKDISCLRREG;		/* 0x0028: Power Domain Clock Disable Clear Register */
    uint32_t :32U;					/* 0x002C */
    uint32_t :32U;					/* 0x0030 */
    uint32_t :32U;					/* 0x0034 */
    uint32_t :32U;					/* 0x0038 */
    uint32_t :32U;					/* 0x003C */
    uint32_t LOGICPDPWRSTAT[4];		/* 0x0040, 0x0044, 0x0048, 0x004C: Logic Power Domain PD2 Power Status Register 
										- 0: PD2
										- 1: PD3
										- 2: PD4
										- 3: PD5 */
    uint32_t :32U;					/* 0x0050 */
    uint32_t :32U;					/* 0x0054 */
    uint32_t :32U;					/* 0x0058 */
    uint32_t :32U;					/* 0x005C */
    uint32_t :32U;					/* 0x0060 */
    uint32_t :32U;					/* 0x0064 */
    uint32_t :32U;					/* 0x0068 */
    uint32_t :32U;					/* 0x006C */
    uint32_t :32U;					/* 0x0070 */
    uint32_t :32U;					/* 0x0074 */
    uint32_t :32U;					/* 0x0078 */
    uint32_t :32U;					/* 0x007C */
    uint32_t MEMPDPWRSTAT[3];		/* 0x0080, 0x0084, 0x0088: Memory Power Domain Power Status Register 
										- 0: RAM_PD1
										- 1: RAM_PD2
										- 2: RAM_PD3 */
    uint32_t :32U;					/* 0x008C */
    uint32_t :32U;					/* 0x0090 */
    uint32_t :32U;					/* 0x0094 */
    uint32_t :32U;					/* 0x0098 */
    uint32_t :32U;					/* 0x009C */
    uint32_t GLOBALCTRL1;			/* 0x00A0: Global Control Register 1 */
    uint32_t :32U;					/* 0x00A4 */
    uint32_t GLOBALSTAT;			/* 0x00A8: Global Status Register */
    uint32_t PRCKEYREG;				/* 0x00AC: PSCON Diagnostic Compare Key Register */
    uint32_t LPDDCSTAT1;			/* 0x00B0: LogicPD PSCON Diagnostic Compare Status Register 1 */
    uint32_t LPDDCSTAT2;			/* 0x00B4: LogicPD PSCON Diagnostic Compare Status Register 2 */
    uint32_t MPDDCSTAT1;			/* 0x00B8: Memory PD PSCON Diagnostic Compare Status Register 1 */
    uint32_t MPDDCSTAT2;			/* 0x00BC: Memory PD PSCON Diagnostic Compare Status Register 2 */
    uint32_t ISODIAGSTAT;			/* 0x00C0: Isolation Diagnostic Status Register */
}pmmBase_t;


/** @def pmmREG
*   @brief Pmm Register Frame Pointer
*
*   This pointer is used by the Pmm driver to access the Pmm registers.
*/
#define pmmREG ((pmmBase_t *)0xFFFF0000)

/* Pmm Interface Functions */
void pmmInit(void);
void pmmTurnONLogicPowerDomain(pmm_LogicPD_t logicPD);
void pmmTurnONMemPowerDomain(pmm_MemPD_t memPD);
void pmmTurnOFFLogicPowerDomain(pmm_LogicPD_t logicPD);
void pmmTurnOFFMemPowerDomain(pmm_MemPD_t memPD);
boolean_t pmmIsLogicPowerDomainActive(pmm_LogicPD_t logicPD);
boolean_t pmmIsMemPowerDomainActive(pmm_MemPD_t memPD);
void pmmGetConfigValue(pmm_config_reg_t *config_reg, config_value_type_t type);
void pmmSetMode(pmm_Mode_t mode);
boolean_t pmmPerformSelfTest();
#endif

