/*
 * Copyright (c) 2013, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 *  ======== empty_min.c ========
 */
/* XDCtools Header files */
#include <xdc/std.h>
#include <xdc/cfg/global.h>
#include <xdc/runtime/Log.h>				//needed for any Log_info() call

/* BIOS Header files */
#include <ti/sysbios/BIOS.h>
#include <ti/sysbios/knl/Semaphore.h>		//when using Semaphores (dynamically)

/* TI-RTOS Header files */
#include <ti/drivers/GPIO.h>
// #include <ti/drivers/I2C.h>
// #include <ti/drivers/SPI.h>
// #include <ti/drivers/UART.h>
// #include <ti/drivers/Watchdog.h>
// #include <ti/drivers/WiFi.h>

/* Example/Board Header files */
#include "Board.h"

#include "driverlib/adc.h"
#include "driverlib/sysctl.h"
#include "driverlib/timer.h"
#include "inc/hw_ints.h"// Removed because it created a bunch of macro redefinitions
#include "inc/hw_memmap.h"
#include "driverlib/interrupt.h"


/* Constants */
#define TARGET_IS_BLIZZARD_RA1
#define ADCPP 					0xFC0
#define FULL_FIFOS_PER_SECOND 	125000 	//1 million samples total per second, spread out over 8 channels = (1M/8) = 125,000 samples per second for each channel
										//In addition, the global counter only increments when 8 channels have been sampled,
										//so when the global counter reaches 125,000 then really 1 million samples have been taken.
#define SECONDS_PER_MINUTE		60

/* Global Variables */
static volatile uint32_t g_ui32Counter = 0;
static volatile uint32_t g_ui32TCounter = 0;
static volatile uint32_t pui32ThresholdedData[128];

//**
//InitTimer
//**
void
InitTimer(void)
{
	SysCtlPeripheralEnable(SYSCTL_PERIPH_TIMER2);
	// Configure Timer0B as a 16-bit periodic timer.
	TimerConfigure(TIMER2_BASE, TIMER_CFG_SPLIT_PAIR | TIMER_CFG_B_PERIODIC);//TIMER_CFG_B_ONE_SHOT);
	// Set the Timer0B load value to 800: 1ms//.
	TimerLoadSet(TIMER2_BASE, TIMER_B, 1280);//640);//SysCtlClockGet() / (1000*1000/8));//80 million divided by 1 million means
											//an over flow every 80 ticks at 80MHz.  Divide by 8 because one trigger signals 8 ADC samples
	// Configure the Timer0B interrupt for timer timeout.
	TimerIntEnable(TIMER2_BASE, TIMER_TIMB_TIMEOUT);
	// Enable the ADC trigger output
	TimerControlTrigger(TIMER2_BASE, TIMER_B,true);
	// Enable the Timer0B interrupt on the processor (NVIC).
	IntEnable(INT_TIMER2B);
	// Initialize the interrupt counter.
	g_ui32Counter = 0;
	//Enable (start) in main program.
}

void CustomTimerIntHandler()
{
	TimerIntClear(TIMER2_BASE, TIMER_TIMB_TIMEOUT);
	g_ui32TCounter++;
//	Log_info1("LED TOGGLED [%u] TIMES",g_ui32TCounter);		// send toggle count to UIA
//
//	GPIO_toggle(Board_LED0);
};

/**********************************************
 * InitADC
 *
 */
void InitADC()
{
		uint32_t uiDummy;
		SysCtlADCSpeedSet(SYSCTL_ADCSPEED_1MSPS);// This instruction must be placed here before the ADC peripheral is enabled.  This is a quirk in the system: not datasheet operation. Found on forum.
		//Enable the ADC peripheral
	 	SysCtlPeripheralEnable(SYSCTL_PERIPH_ADC0);
	    // Select the ports to be used.
	    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOE);
	    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOD);
	    // Select the analog ADC function for these pins.
	    GPIOPinTypeADC(GPIO_PORTE_BASE, GPIO_PIN_1|GPIO_PIN_2|GPIO_PIN_3|GPIO_PIN_5);
	    GPIOPinTypeADC(GPIO_PORTD_BASE, GPIO_PIN_0|GPIO_PIN_1|GPIO_PIN_2|GPIO_PIN_3);

	    // Enable sample sequence 0 with a timer signal trigger.  Sequence 0
	    // will do eight samples when the timer sends a signal to start the
	    // conversion sequence.  Each ADC module has 4 programmable sequences,
	    // sequence 0 to sequence 3.
	    ADCSequenceConfigure(ADC0_BASE, 0, /*ADC_TRIGGER_ALWAYSADC_TRIGGER_PROCESSOR*/ADC_TRIGGER_TIMER, 0);
	    // Configure step 0 on sequence 3.  Sample channel 0 (ADC_CTL_CH0) in
	    // single-ended mode (default) and configure the interrupt flag
	    // (ADC_CTL_IE) to be set when the sample is done.  Tell the ADC logic
	    // that this is the last conversion on sequence 3 (ADC_CTL_END).  Sequence
	    // 3 has only one programmable step.  Sequence 1 and 2 have 4 steps, and
	    // sequence 0 has 8 programmable steps.  Since we are only doing a single
	    // conversion using sequence 3 we will only configure step 0.  For more
	    // information on the ADC sequences and steps, reference the datasheet.
	    ADCSequenceStepConfigure(ADC0_BASE, 0, 0, ADC_CTL_CH0);//PE3
	    ADCSequenceStepConfigure(ADC0_BASE, 0, 1, ADC_CTL_CH1);//PE2
	    ADCSequenceStepConfigure(ADC0_BASE, 0, 2, ADC_CTL_CH2);//PE1
	    ADCSequenceStepConfigure(ADC0_BASE, 0, 3, ADC_CTL_CH4);//PD3
	    ADCSequenceStepConfigure(ADC0_BASE, 0, 4, ADC_CTL_CH5);//PD2
	    ADCSequenceStepConfigure(ADC0_BASE, 0, 5, ADC_CTL_CH6);//PD1
	    ADCSequenceStepConfigure(ADC0_BASE, 0, 6, ADC_CTL_CH7);//PD0
	    ADCSequenceStepConfigure(ADC0_BASE, 0, 7, ADC_CTL_CH8 | ADC_CTL_IE | ADC_CTL_END);//PE5

	    // Since sample sequence 3 is now configured, it must be enabled.
	    ADCSequenceEnable(ADC0_BASE, 0);

	    // Clear the interrupt status flag.  This is done to make sure the
	    // interrupt flag is cleared before we sample.
	    ADCIntClear(ADC0_BASE, 0);

	    // Allow ADC completions to trigger the interrupt handler
	    ADCIntEnable(ADC0_BASE,0);
	    IntEnable(INT_ADC0SS0);//This instruction controls whether or not the ADC will run.

	    //Set ADC to 1MSPS speed
	    //SysCtlADCSpeedSet(SYSCTL_ADCSPEED_1MSPS);////CAUTION: Putting this instruction elsewhere may cause a FAULT_ISR
	    //to trigger at some random time.
	    //////CAUTION: This instruction also will screw up the program, so use it on one run, abort the run, then start
	    //a new debug run without this command.
	    //The prior value will be preserved.
	    /*#define SYSCTL_ADCSPEED_1MSPS   0x00000F00  // 1,000,000 samples per second
			#define SYSCTL_ADCSPEED_500KSPS 0x00000A00  // 500,000 samples per second
			#define SYSCTL_ADCSPEED_250KSPS 0x00000500  // 250,000 samples per second
			#define SYSCTL_ADCSPEED_125KSPS 0x00000000  // 125,000 samples per second*/

	    //Delay
	    uiDummy = 0x3840;
	   // SysCtlADCSpeedSet(SYSCTL_ADCSPEED_1MSPS);

}

/******************/
//ADC Interrupt Handler
/******************/
void ADCSeq0IntHandler()
{
	static uint32_t ui32TempVal;
	static uint32_t ui32Threshold=50;

	ADCIntClear(ADC0_BASE,0); //THIS IS THE FUNCTION THAT FAILS TO CLEAR THE INTERRUPT WHEN DEBUGGING
	g_ui32Counter++;
	ui32TempVal = HWREG(ADC0_BASE + ADCSSFIFO0);

	//Basic processing
	if(ui32TempVal>ui32Threshold)
	{
		pui32ThresholdedData[g_ui32Counter%128]=ui32TempVal;
	}
	else
	{
		pui32ThresholdedData[g_ui32Counter%128]=0;
	};
	if(g_ui32Counter==(FULL_FIFOS_PER_SECOND*SECONDS_PER_MINUTE))
    {
		Semaphore_post(ADC_Done_Sem);
    };
}

void ADC_Done_Task(void)
{
	Semaphore_pend(ADC_Done_Sem,BIOS_WAIT_FOREVER);
	GPIO_write(Board_LED0, Board_LED_OFF);
}

/*
 *  ======== main ========
 */
Int main(Void)
{

    /* Call board init functions */
    Board_initGeneral();
    Board_initGPIO();
    // Board_initDMA();
    // Board_initI2C();
    // Board_initSPI();
    // Board_initUART();
    // Board_initUSB(Board_USBDEVICE);
    // Board_initWatchdog();
    // Board_initWiFi();

    //////////////////////////////////////////
    InitTimer();
    InitADC();

    Log_info1("CLOCK SPEED [%u]",SysCtlClockGet());
    Log_info1("ADC SPEED [%u]",SysCtlADCSpeedGet());

	//////////////////////////////

	// Start Timer0B. and the ADC (Don't trust the debugger (ever))
	TimerEnable(TIMER2_BASE, TIMER_B);
	//////////////////////////////

    /* Turn on user LED  */
    GPIO_write(Board_LED0, Board_LED_ON);

    /* Start BIOS */
    BIOS_start();

    return (0);
}
