"""
RegistersGUI.py

Copyright (c) 2007 National Semiconductor Corporation.
All Rights Reserved.

The register panel GUI base class. Can be subclassed to provide
additional device specific register functionality.

ALPFGA boards are not supported in this module

Revision History
072214 a0224961
Created file, derived from RegistersGUI.py, modified to display and modify registers of remote device.
Emulates device registers if in demo mode or no device present
Note: Does not take into account blocked access to registers 0x80-0xff in serializer

073114 a0224961
Fixed issue where I2C on Forward channel was not active.  This was caused by Remote device register being different between SER and DES.  
Note that there can be an intermittent issue extending to other tabs when the link toggles on/off.  This occurred when plugged into a UH925, but could not be repeated.
Also hides registers that cannot be accessed if DES is truing to read SER registers

080814 a0224961
Fixed issue where registers would not load because there was no non-testdie profile for 94x devices
"""

from NS1FoldPanelBar import NSFoldPanelBar
from wx.lib.foldpanelbar import CaptionBarStyle
from Utils import *

import os
import sys
import time
import pickle

from ProfileClass import *

REMOTE_ID_REG = 0x07
#But is 0x06 in 925

class RegSettings:
    def __init__( s):
        VerboseRegisters = False


#***************************************************************************
class RegFoldPanelBar( NSFoldPanelBar):

    def __init__( s, *args, **argd):

        args = list(args)
        s.verboseFlag = args[4]
        args.remove( args[4])

        NSFoldPanelBar.__init__( s, *args, **argd)
        s.childToPanelDict = {}
        s.currFocusPanel = None
        s.normBackColour = s.GetParent().GetBackgroundColour()

        s.Bind( wx.EVT_LEFT_DOWN, s.FPSetFocus)
        s.childToPanelDict[s] = s

    def Expand( s, foldPanel):
        foldPanel.Freeze()

        if not foldPanel.regInitialized:
            foldPanel.regInitialized = True

            reg = foldPanel.register
            regSz = wx.BoxSizer( wx.VERTICAL)
            if s.verboseFlag:
                if len( reg.description.strip()):
                    regDesc = WriteText( foldPanel.newWin, reg.description, wrap=foldPanel.regParent.regWinWidth)
                    regSz.Add( regDesc, flag=wx.BOTTOM, border=10)

            bitFGSz = wx.GridBagSizer( 8, 0)
            bitFGSz.SetEmptyCellSize( (0,0))
            col = 0
            if s.verboseFlag:
                headers = ("Bit(s)","Type","Default","Name","Description")
            else:
                headers = ("Bit(s)","Name")

            for headTxt in headers:
                htxt = WriteText( foldPanel.newWin, headTxt, underline=True)
                if headTxt == "Bit(s)": bitFGSz.Add( htxt, (0,col), flag=wx.ALIGN_LEFT, border=7)
                else: bitFGSz.Add( htxt, (0,col), flag=wx.LEFT, border=7)
                col += 1

            row = 1
            hzLines = []
            bitBoxsAll = []
            for bits in reg.regBits:
                if s.readOnly:
                    if bits.startBitIdx == bits.endBitIdx:
                        ttxt = WriteText( foldPanel.newWin, str( bits.startBitIdx), wrap=20)
                    else:
                        ttxt = WriteText( foldPanel.newWin, "%d:%d" % (bits.startBitIdx, bits.endBitIdx), wrap=20)
                    bitFGSz.Add( ttxt, (row,0), flag=wx.ALIGN_LEFT|wx.LEFT, border=7)
                else:
                    bitBoxs = BuildCheckboxes( foldPanel.newWin, "", range( bits.startBitIdx, bits.endBitIdx-1, -1), maxSpan=4,
                                               spanCol=True, space=(3,1), noStaticBox=True, labelLeft=True)
                    for bit in bitBoxs[1:]:
                        wx.EVT_CHECKBOX( foldPanel.newWin, bit.GetId(), foldPanel.regParent.OnRegBitCheck)

                    bitBoxs[0].SetEmptyCellSize( (-2,0))
                    bitFGSz.Add( bitBoxs[0], (row,0), flag=wx.ALIGN_RIGHT|wx.RIGHT, border=5)
                    bitBoxsAll.extend( bitBoxs[1:])

                if s.verboseFlag:
                    ttxt = WriteText( foldPanel.newWin, bits.type, wrap=30)
                    bitFGSz.Add( ttxt, (row,1), flag=wx.ALIGN_LEFT|wx.LEFT, border=7)
                    dtxt = WriteText( foldPanel.newWin, bits.defaultValue, wrap=30)
                    bitFGSz.Add( dtxt, (row,2), flag=wx.ALIGN_LEFT|wx.LEFT, border=7)

                nameFlags = wx.ALIGN_LEFT|wx.LEFT
                nameTxt = bits.fieldName
                nameCol = 3
                wrap = 80
                if not s.verboseFlag:
                    nameFlags = wx.ALIGN_LEFT|wx.ALIGN_CENTER_VERTICAL|wx.LEFT
                    nameTxt = bits.fieldName.replace( '\n', ' ')
                    nameCol = 1
                    wrap += 200+30+30

                ntxt = WriteText( foldPanel.newWin, nameTxt, wrap=wrap)
                bitFGSz.Add( ntxt, (row,nameCol), flag=nameFlags, border=7)

                if s.verboseFlag:
                    dtxt = WriteText( foldPanel.newWin, bits.description, wrap=200)
                    bitFGSz.Add( dtxt, (row,4), flag=wx.ALIGN_LEFT|wx.LEFT, border=7)

                if s.verboseFlag:
                    if bits != reg.regBits[-1]:
                        row += 1
                        ln = wx.StaticLine( foldPanel.newWin, size=(-1,1))
                        hzLines.append( (bitFGSz, ln))
                        bitFGSz.Add( ln, (row,0), (1,5), flag=wx.ALIGN_LEFT|wx.ALIGN_CENTER_VERTICAL, border=0)
                        row += 1
                else:
                    row += 1

            maxX = 0
            for bitChk in bitBoxsAll:
                sz = bitChk.GetClientSize()
                if sz[0] > maxX: maxX = sz[0]

            for bitChk in bitBoxsAll:
                bitChk.SetMinSize( (maxX, bitChk.GetClientSize()[1]))

            regSz.Add( bitFGSz, flag=wx.RIGHT, border=7)

            foldPanel.newWin.SetSizer( regSz)
            foldPanel.newWin.Layout()
            foldPanel.Layout()

            if not s.readOnly:
                foldPanel.regParent.regListPanels[foldPanel] = [reg, bitBoxsAll]

            for l in hzLines:
                l[1].SetSize( (l[0].GetSize()[0]+5, 2))

            s.ResizeScrolledWindow()

            def BindToAll( win):
                ch = win.GetChildren()
                if len( ch):
                    for chnode in ch:
                        BindToAll( chnode)
                if win not in s.childToPanelDict:
                    win.Bind( wx.EVT_LEFT_DOWN, s.FPSetFocus)
                    s.childToPanelDict[win] = foldPanel

            if not s.readOnly:
                BindToAll( foldPanel)

        NSFoldPanelBar.Expand( s, foldPanel)

        if not s.readOnly:
            evt = wx.MouseEvent( foldPanel.GetId())
            evt.SetEventObject( foldPanel)
            s.FPSetFocus( evt, forceUpdate=True)

        yPos = foldPanel.GetRect()[1]
        s.parent.regListWin.Scroll( -1, yPos / 15)

        foldPanel.Thaw()

    def FPSetFocus( s, event, forceUpdate=False):
        objEvt = event.GetEventObject()
        obj = s.childToPanelDict[objEvt]

        if s.currFocusPanel and obj != s.currFocusPanel:
            s.currFocusPanel.Freeze()
            s.currFocusPanel.SetBackgroundColour( s.normBackColour)
            s.currFocusPanel.Refresh( False)

            style = s.currFocusPanel.GetCaptionStyle()
            style.SetFirstColour( s.parent.REG_UNSELECTED_COLOR_1)
            style.SetSecondColour( s.parent.REG_UNSELECTED_COLOR_2)
            s.currFocusPanel.ApplyCaptionStyle( style)
            s.currFocusPanel.Thaw()

        if forceUpdate or obj != s.currFocusPanel:
            obj.Freeze()
            obj.SetBackgroundColour( wx.TheColourDatabase.Find( 'LIGHT BLUE'))
            obj.Refresh( False)

            style = obj.GetCaptionStyle()
            style.SetFirstColour( s.parent.REG_SELECTED_COLOR_1)
            style.SetSecondColour( s.parent.REG_SELECTED_COLOR_2)
            obj.ApplyCaptionStyle( style)

            s.currFocusPanel = obj

            s.parent.SetRegFocus( s.currFocusPanel)
            obj.Thaw()

        event.Skip()

    def Collapse( s, foldPanel):
        foldPanel.Freeze()
        NSFoldPanelBar.Collapse( s, foldPanel)
        foldPanel.Thaw()


#***************************************************************************
class RemoteRegistersGUI:

    def __init__( s, parent, board, readOnly=False):
        s.REG_SELECTED_COLOR_1 = wx.TheColourDatabase.Find( 'LIGHT BLUE')
        s.REG_SELECTED_COLOR_2 = wx.TheColourDatabase.Find( 'SKY BLUE')
        s.REG_UNSELECTED_COLOR_1 = wx.Colour( 240, 240, 240)
        s.REG_UNSELECTED_COLOR_2 = wx.TheColourDatabase.Find( 'LIGHT GREY')

        s.parent = parent
        s.board = board
        s.readOnly = readOnly

        s.appCWD = os.path.dirname(__file__)
        s.currFocusPanel = None
        s.regListPanels = []
        
        #Activate Demo mode if I2C bus not active
        
        if(s.board.demoMode):#In demo mode, so activate demo mode
            s.demoMode = True
        elif(s.board.GetLinkStatus()):#Linked to partner, disable demo mode
            s.demoMode = False
        else:#No link, blank panel
            MyPopup( s.parent, "Error: No Partner Device Found: %s" %(s.board.partnerShortDescription), "Error")
            return
       
        #Get remote device address, register depends on device
        s.remIDReg = 0x07
        if('925' in s.board.shortDescription):
            s.remIDReg = 0x06
        if('926' in s.board.shortDescription):
            s.remIDReg = 0x07
        if('927' in s.board.shortDescription):
            s.remIDReg = 0x06
        if('928' in s.board.shortDescription):
            s.remIDReg = 0x07
        if('929' in s.board.shortDescription):
            s.remIDReg = 0x06
        if('940' in s.board.shortDescription):
            s.remIDReg = 0x07
        if('947' in s.board.shortDescription):
            s.remIDReg = 0x06
        if('948' in s.board.shortDescription):
            s.remIDReg = 0x07
        if('949' in s.board.shortDescription):
            s.remIDReg = 0x06
        if('913' in s.board.shortDescription):
            s.remIDReg = 0x08
        if('914' in s.board.shortDescription):
            s.remIDReg = 0x06
        s.partnerAddr = s.board.ReadReg(s.remIDReg)#TODO: change to variable on device
            
        
        # This is the horizontal width to hold the formatted text with a possible vert. scroll bar
        # See the BuildRegisterDisplay for the various pixel widths used for this calculation
        s.regWinWidth = 5+10+s.parent.GetTextExtent("Bit(s)")[0] + (7*4) + 30+30+130+258
        # Build the register info tab panel
        if s.readOnly:
            regListHdr = WriteText( s.parent, "Imported Register Descriptions:")
        else:
            regListHdr = WriteText( s.parent, "")
        s.regListWinSz = wx.BoxSizer( wx.HORIZONTAL)
        s.regListWin = wx.ScrolledWindow( s.parent, -1, style=wx.TAB_TRAVERSAL|wx.SUNKEN_BORDER|wx.CLIP_CHILDREN)
        s.regListWin.Bind( wx.EVT_MOUSEWHEEL, s.OnWheel)
        s.regListWinSz.Add( s.regListWin, flag=wx.EXPAND, proportion=1)

        s.regListWin.SetBackgroundColour( wx.SystemSettings.GetColour( wx.SYS_COLOUR_WINDOW))
        s.regListPanelBar = None

        s.regSizer = wx.BoxSizer( wx.VERTICAL)
        s.firstLineSizer = wx.BoxSizer( wx.HORIZONTAL)
        s.firstLineSizer.Add( regListHdr, flag=wx.LEFT|wx.ALIGN_CENTER_VERTICAL, border=15)
        s.firstLineSizer.AddSpacer( (0,3))

        if not s.readOnly:
            s.regSizer.Add( s.firstLineSizer, flag=wx.BOTTOM|wx.TOP, border=15)
        else:
            s.regSizer.Add( s.firstLineSizer, flag=wx.BOTTOM, border=3)
        
        #Get Remote Device ProfileClass, which contains register information
        profileName = s.board.partnerShortDescription
        if(profileName[-2:] == 'TD'):#Test Die Devices have 'TD' in shortname but '_ENG' in profile folder
            profileName = profileName[:-3] + '_ENG'
        s.remProfile = FindProfileFromProfileName(appCWD, profileName)

        if not s.readOnly:
            try:
                regList = s.remProfile.registers
            except:
                try:#If a non-testdie device is requested, but no non-testdie profile exists, then use the testdie profile
                    s.remProfile = FindProfileFromProfileName(appCWD, profileName+'_ENG')
                    regList = s.remProfile.registers
                except:
                    MyPopup( s.parent, "Error: No Partner Device Profile Found: %s Profile Name: %s" %(s.board.partnerShortDescription,profileName), "Error")
                    return
            
            #Higher registers cannot be read from Serializer, so remove those
            remName = s.board.partnerShortDescription
            hiddenList = []
            s.registers = []
            if('925' in remName):
                hiddenList = range(0x80,0xFF)
            if('927' in remName):
                hiddenList = range(0x80,0xFF)
            if('947' in remName):
                hiddenList = range(0x80,0xF9)
            if('949' in remName):
                hiddenList = range(0x80,0xF9)
            for i in range(0,len(regList)):
                if(regList[i].address not in hiddenList):
                    s.registers.append(regList[i])
            
            if len( s.registers):
                s.regSize = s.registers[0].regBits[0].startBitIdx
            else:
                s.regSize = 31

            numDigits = 2
            if s.regSize == 15: numDigits = 4
            elif s.regSize == 31: numDigits = 8
            
            #Emulate registers if in demo mode
            s.emuRegisters = [0] * (2**(s.regSize+1))
            for reg in s.registers:
                s.emuRegisters[reg.address] = s.getDefaultRegValue(reg)
                
            #Assemble GUI
            s.valueSizer = wx.BoxSizer( wx.HORIZONTAL)

            s.valEditBox = BuildHexEditBox( s.parent, "Value:", numDigits=numDigits, enable=True)
            s.valueSizer.Add( s.valEditBox[0], flag=wx.ALIGN_BOTTOM|wx.RIGHT, border=10)

            s.applyBtn = BuildButton( s.parent, "Apply", s.OnApplyClick, style=wx.BU_EXACTFIT, enable=False)
            s.valueSizer.Add( s.applyBtn, flag=wx.ALIGN_BOTTOM|wx.RIGHT, border=5)

            s.refreshBtn = BuildButton( s.parent, "Refresh", s.OnRefreshClick, style=wx.BU_EXACTFIT, enable=False)
            s.valueSizer.Add( s.refreshBtn, flag=wx.ALIGN_BOTTOM|wx.LEFT, border=20)

            s.regRefrBtn = BuildButton( s.parent, "Refresh All", s.RefreshAllRegsClick, style=wx.BU_EXACTFIT, enable=False)
            s.valueSizer.Add( s.regRefrBtn, flag=wx.LEFT, border=10)
            
            verbEn = True
            s.verboseChk = wx.CheckBox( s.parent, wx.NewId(), "Verbose Descriptions")
            s.verboseChk.SetValue( verbEn)
            wx.EVT_CHECKBOX( s.parent, s.verboseChk.GetId(), s.OnVerboseCheck)
            s.valueSizer.Add( s.verboseChk, flag=wx.LEFT|wx.RIGHT|wx.ALIGN_CENTER_VERTICAL, border=25)
            
            devName = s.board.partnerShortDescription
            if(s.demoMode):
                devName = "Emulation:" + devName
            else:
                devName = hex(s.partnerAddr) + ': ' + devName
            s.remDevNameL = wx.StaticText(s.parent, label=devName)
            s.valueSizer.Add(s.remDevNameL, flag=wx.LEFT|wx.RIGHT|wx.ALIGN_CENTER_VERTICAL, border=25)

            s.regDumpBtn = BuildButton( s.parent, "Display", s.RegOnDumpRegsClick)
            s.regLoadBtn = BuildButton( s.parent, "Load", s.RegOnLoadRegsClick)
            s.regSaveBtn = BuildButton( s.parent, "Save", s.RegOnSaveRegsClick)

            s.regBtnSizer = wx.BoxSizer( wx.VERTICAL)
            s.regBtnSizer.Add( s.regDumpBtn, flag=wx.BOTTOM, border=10)
            s.regBtnSizer.Add( s.regLoadBtn, flag=wx.BOTTOM, border=10)
            s.regBtnSizer.Add( s.regSaveBtn, flag=wx.BOTTOM, border=10)

            s.regListWinSz.Add( s.regBtnSizer, flag=wx.LEFT, border=15)
            s.firstLineSizer.Add( s.valueSizer)

        if not s.readOnly:
            s.BuildRegisterDisplay()

        s.regSizer.Add( s.regListWinSz, flag=wx.LEFT|wx.RIGHT|wx.BOTTOM|wx.EXPAND, proportion=1, border=15)

        s.regListWin.SetMinSize( (20, 20))
        s.regSizer.SetItemMinSize( s.regListWin, 20, 20)

        if not s.readOnly:
            s.parent.SetSizer( s.regSizer)
            s.parent.Layout()

        s.regListWin.SetScrollRate( 15, 15)
        return

    def OnVerboseCheck( s, event):
        s.BuildRegisterDisplay()

        try:
            setFile = open( os.path.join( s.appCWD, "reg.settings"))
            regSettings = pickle.load( setFile)
            setFile.close()
        except:
            regSettings = RegSettings()

        try:
            regSettings.VerboseRegisters = s.verboseChk.GetValue()
            setFile = open( os.path.join( s.appCWD, "reg.settings"), 'w')
            pickle.dump( regSettings, setFile)
            setFile.close()
        except:
            None

    wheelScroll = 0
    def OnWheel( s, event):
        # The wheel scrolling in wxWidgets is a bit busted. If we use the wheel to scroll,
        # we get a repaint event for all child windows in the view area. If we use the scroll
        # bar directly, we only get a single repaint for the affected subwindow. Plus if the
        # scroll position is at the top, then spinning the wheel causes repaint events
        # for our subwindows... so to reduce the flicker this causes, do the wheel handling
        # ourselves.
        delta = event.GetWheelDelta()
        rot = event.GetWheelRotation()
        linesPer = event.GetLinesPerAction()
        lines = -1 * (rot / delta) * linesPer
        s.regListWin.ScrollLines( lines)
        return

    def BuildRegisterDisplay( s):
        s.regListWin.Freeze()
        if s.regListPanelBar:
            s.regListPanelBar.Destroy()

        if not s.readOnly:
            #s.registers = s.remProfile.registers
            s.hasMultiplePages = s.remProfile.hasMultiplePages
        else:
            # Caller is our profile editor, "board" is a reference to the register list
            #s.registers = s.remProfile.registers
            s.hasMultiplePages = False
            for reg in s.registers:
                if reg.page > 0:
                    s.hasMultiplePages = True
                    break

        if len( s.registers) == 0:
            s.regListWin.Thaw()
            return

        if len( s.registers) > 100:
            s.regProg = wx.ProgressDialog( "Creating Register Display", "Please wait while the register list is created.",
                                           len( s.registers), s.regListWin)
        if s.readOnly:
            verFlag = True
        else:
            verFlag = s.verboseChk.GetValue()

        s.regListPanelBar = RegFoldPanelBar( s.regListWin, -1, wx.DefaultPosition, (-1,-1), verFlag)
        s.regListPanelBar.readOnly = s.readOnly
        s.regListPanelBar.parent = s
        s.regListPanels = {}

        if len( s.registers):
            s.regSize = s.registers[0].regBits[0].startBitIdx
        else:
            s.regSize = 31

        regCnt = 0

        for reg in s.registers:
            name = ""
            if s.hasMultiplePages:
                if reg.page == -1: name = "Base - "
                else: name = "Page %d - " % (reg.page)
            if reg.address > 0xFF:
                name += "0x%04X" % reg.address
            else:
                name += "0x%02X" % reg.address
            name += " - %s" % reg.regName
            
            #Get root directory for ALP, where RegisterGroup.png is located, originally s.appCWD
            ALProot = os.path.split(os.path.split(s.appCWD)[0])[0]
            
            item = s.regListPanelBar.AddFoldPanel( name, collapsed=True,icon=os.path.join( ALProot, 'RegisterGroup.png'))

            if not s.readOnly:
                item._captionBar.Bind( wx.EVT_LEFT_DOWN, s.regListPanelBar.FPSetFocus)
                s.regListPanelBar.childToPanelDict[item._captionBar] = item

            style = CaptionBarStyle()
            style.SetFirstColour( s.REG_UNSELECTED_COLOR_1)
            style.SetSecondColour( s.REG_UNSELECTED_COLOR_2)
            style.SetCaptionStyle( fpb.CAPTIONBAR_GRADIENT_V)
            item.ApplyCaptionStyle( style)

            item.register = reg
            item.regParent = s
            item.regInitialized = False
            newWin = wx.Panel( item, -1, size=(s.regWinWidth+15, -1), style=wx.TAB_TRAVERSAL|wx.CLIP_CHILDREN)
            item.newWin = newWin

            if not s.readOnly:
                s.regListPanels[item] = [reg, None]

            s.regListPanelBar.AddFoldPanelWindow( item, newWin, fpb.FPB_ALIGN_WIDTH, leftSpacing=10, rightSpacing=10)

            regCnt += 1
            if len( s.registers) > 100:
                s.regProg.Update( regCnt)

        if not s.readOnly:
            s.applyBtn.Enable( False)
            s.refreshBtn.Enable( False)
            s.regRefrBtn.Enable( False)

            fmt = "%02X"
            if s.regSize == 15: fmt = "%04X"
            elif s.regSize == 31: fmt = "%08X"
            s.valEditBox[1].SetLabel( fmt % 0)

        s.regListIntSz = wx.BoxSizer( wx.VERTICAL)
        s.regListIntSz.Add( s.regListPanelBar)
        s.regListWin.SetSizer( s.regListIntSz)
        s.regListPanelBar.ResizeScrolledWindow()
        s.regListWin.Thaw()

    def SetRegFocus( s, currFocusPanel):
        s.currFocusPanel = currFocusPanel
        if s.currFocusPanel:
            s.RefreshBitValues()

        en = (s.currFocusPanel != None)
        s.applyBtn.Enable( en)
        s.refreshBtn.Enable( en)
        s.regRefrBtn.Enable( en)

    def OnRefreshClick( s, event):
        s.RefreshBitValues()

    def RefreshBitValues( s, useThisPanel=None):
        if not s.currFocusPanel:
            return

        if useThisPanel:
            regData = s.regListPanels[ useThisPanel ]
        else:
            regData = s.regListPanels[ s.currFocusPanel ]

        addr = regData[0].address
        if regData[0].page != -1: addr |= regData[0].page << 5
        regVal = s.ReadRemReg( addr)

        if regData[1]:
            for bit in range( s.regSize, -1, -1):
                regData[1][s.regSize-bit].SetValue( regVal & (1 << bit))

        if not useThisPanel:
            fmt = "%02X"
            if s.regSize == 15: fmt = "%04X"
            elif s.regSize == 31: fmt = "%08X"
            s.valEditBox[1].SetLabel( fmt % regVal)

    def RefreshAllRegsClick( s, event):
        s.RefreshAllRegs()

    def RefreshAllRegs( s):
        s.RefreshBitValues()
        for regPan in s.regListPanels:
            if s.regListPanels[regPan][1]:
                s.RefreshBitValues( useThisPanel=regPan)

    def OnRegBitCheck( s, event):
        regData = s.regListPanels[ s.currFocusPanel ]
        regVal = 0x00
        for bit in range( s.regSize, -1, -1):
            if regData[1][s.regSize-bit].GetValue():
                regVal |= 1 << bit
        fmt = "%02X"
        if s.regSize == 15: fmt = "%04X"
        elif s.regSize == 31: fmt = "%08X"
        s.valEditBox[1].SetLabel( fmt % regVal)

    def OnApplyClick( s, event):
        regData = s.regListPanels[ s.currFocusPanel ]
        regVal = int( s.valEditBox[1].GetLabel(), 16)

        addr = regData[0].address
        if regData[0].page != -1: addr |= regData[0].page << 5

        s.WriteRemReg( addr, regVal)
        s.RefreshBitValues()

    def RegOnDumpRegsClick( s, event):
        info = s.board.GetBoardInfo()

        if not info['nanoBoard'] and (info['lptMdioBoard'] or not hasattr( s.board, 'GetBaseboardObject')):
            binfo = s.board.GetParentDevice().GetDeviceInfo()
            banner = "Operational Register Display - %s, %s, MDIO Address %d" % (s.board.partnerShortDescription,
                    FormatBusInfo( binfo), s.board.GetPortMDIOAddress())
        elif info['nanoBoard']:
            banner = "Register Display - ALP Nano - %s, I2C Address 0x%02X" % (s.board.partnerShortDescription, s.board.partnerAddr)
        else:
            baseBoard = s.board.GetBaseboardObject()
            bbinfo = baseBoard.GetBoardInfo()
            banner = "Register Display - %s %d - %s, Connector %d" % (bbinfo['shortName'], bbinfo['devIndex']+1,\
                     s.board.partnerShortDescription, info['connector']+1)

        if s.hasMultiplePages:
            msg = banner + "\n\nRegister\t\tData\tName\n"
        else:
            msg = banner + "\n\nRegister\tData\tName\n"

        for reg in s.registers:
            addr = reg.address
            if reg.page != -1: addr |= reg.page << 5
            regVal = s.ReadRemReg( addr)

            if s.hasMultiplePages:
                if reg.page == -1: msg += "Base - "
                else: msg += "Page %d - " % (reg.page)

            msg += "0x%04X\t" % (reg.address)

            if s.regSize == 7:
                fmt = "0x%02X"
            elif s.regSize == 15:
                fmt = "0x%04X"
            elif s.regSize == 31:
                fmt = "0x%08X"
            msg += fmt % regVal
            msg += "\t" + reg.regName + "\n"

        dlg = wx.lib.dialogs.ScrolledMessageDialog( s.parent, msg, banner, size=(600,400))
        dlg.Show( True)
        event.Skip()

    def RegOnLoadRegsClick( s, event):
        wildcard = "Device Register Data (*.nrd)|*.nrd|" \
                   "All files (*.*)|*.*"
        dlg = wx.FileDialog( s.parent, message="Load Register State",
                            defaultDir=os.getcwd(), defaultFile="",
                            wildcard=wildcard,
                            style=wx.OPEN|wx.CHANGE_DIR|wx.HIDE_READONLY)
        if dlg.ShowModal() == wx.ID_OK:
            try:
                vals = ParseDataFile( dlg.GetFilename(), "REGISTERS")
                devName = "N/A"
                for n in vals:
                    if n[0] == "Device":
                        devName = n[1]
                        break
                comments = "None"
                for n in vals:
                    if n[0] == "Comments":
                        comments = n[1]
                        break
                date = "N/A"
                for n in vals:
                    if n[0] == "Date":
                        date = n[1]
                        break
                time = "N/A"
                for n in vals:
                    if n[0] == "Time":
                        time = n[1]
                        break

                msg = "Apply the following register file to the selected device registers?\n\n%s\nDate/Time: %s, %s\nFile Comments: %s" % (devName, date, time, comments)
                answer = MyPopup( s.parent, msg, "Apply Register Values", style=wx.ICON_QUESTION|wx.YES_NO)
                if answer == wx.ID_YES and len( s.registers):
                    for n in vals:
                        if n[0] == "Reg":
                            page, off, val = n[1].split(',')

                            s.WriteRemReg( (int(page) << 5) | int( off, 16), int( val,16))
                    MyPopup( s.parent, "The register values were successfully written.", "Status")
            except:
                MyPopup( s.parent, "The specified file is not a valid register information file.", "Error")
        dlg.Destroy()
        s.RefreshAllRegs()
        event.Skip()
        return

    def RegOnSaveRegsClick( s, event):
        wildcard = "Device Register Data (*.nrd)|*.nrd|" \
                   "All files (*.*)|*.*"
        dlg = wx.FileDialog( s.parent, message="Save Register Data",
                            defaultDir=os.getcwd(), defaultFile="",
                            wildcard=wildcard,
                            style=wx.SAVE|wx.CHANGE_DIR|wx.HIDE_READONLY|wx.OVERWRITE_PROMPT)
        if dlg.ShowModal() == wx.ID_OK:
            multiDlg = wx.TextEntryDialog( s.parent, "Register Data Comments:", \
                                          "Enter Descriptive Comments", \
                                          style = wx.OK|wx.CANCEL|wx.CENTRE)
            ccode = multiDlg.ShowModal()
            multiDlg.Destroy()
            if ccode != wx.ID_OK:
                event.Skip()
                return

            comments = multiDlg.GetValue().strip()
            try:
                s.fh = open( dlg.GetPath(), 'w')
                s.fh.write( "[REGISTERS]\n")

                info = s.board.GetBoardInfo()
                if not info['nanoBoard'] and (info['lptMdioBoard'] or not hasattr( s.board, 'GetBaseboardObject')):
                    binfo = s.board.GetParentDevice().GetDeviceInfo()
                    banner = "Device = %s, %s, MDIO Address %d\n" % (s.board.partnerShortDescription,
                             FormatBusInfo( binfo), s.board.GetPortMDIOAddress())
                elif info['nanoBoard']:
                    banner = "Device = ALP Nano - %s, I2C Address 0x%02X\n" % (s.board.partnerShortDescription, info['i2cAddress'])
                else:
                    baseBoard = s.board.GetBaseboardObject()
                    bbinfo = baseBoard.GetBoardInfo()
                    banner = "Device = %s %d - %s, Connector %d\n" % (bbinfo['shortName'], bbinfo['devIndex']+1,\
                             s.board.partnerShortDescription, info['connector']+1)

                s.fh.write( banner)
                s.fh.write( "Comments = \"%s\"\n" % comments)
                t = time.localtime()
                s.fh.write( time.strftime( "Date = %m/%d/%Y", t) + "\n")
                s.fh.write( time.strftime( "Time = %H:%M:%S", t) + "\n")

                if s.regSize == 7:
                    fmt = "0x%02X"
                elif s.regSize == 15:
                    fmt = "0x%04X"
                elif s.regSize == 31:
                    fmt = "0x%08X"

                if len( s.registers):
                    for reg in s.registers:
                        addr = reg.address
                        if reg.page != -1: addr |= reg.page << 5
                        regVal = s.ReadRemReg( addr)
                        regValFmt = fmt % regVal
                        rgpg = reg.page
                        if reg.page == -1: rgpg = 0
                        s.fh.write( "Reg = %d,0x%04X,%s\n" % (rgpg, reg.address, regValFmt))

                s.fh.close()
                MyPopup( s.parent, "The device's register state has been successfully saved.", "Status")
            except:
                MyPopup( s.parent, "An error occurred saving the register data to the specified file.", "Error")
        dlg.Destroy()
        event.Skip()
        return
    #Read from target register
    def ReadReg( s, regAddr):
        regVal = s.board.ReadReg( regAddr)
        return regVal
    #Write to target register
    def WriteReg( s, regAddr, regVal):
        s.board.WriteReg( regAddr, regVal)
    #Read from remote register
    def ReadRemReg(s, regAddr):
        if(s.demoMode):
            regVal = s.emuRegisters[regAddr]
        else:
            regVal = s.board.ReadI2C(s.partnerAddr, regAddr)
        return regVal
    #Write to remote register
    def WriteRemReg(s, regAddr, regVal):
        if(s.demoMode):
            s.emuRegisters[regAddr] = regVal
        else:
            s.board.WriteI2C(s.partnerAddr, regAddr, regVal)
        return
    #Check to see if we are in deo mode by attempting to read.  Returns Null type if in demomode or no receiver present
    def checkRemReg(s, regAddr):
        #regVal = s.board.ReadI2C(s.partnerAddr, regAddr)
        regVal = s.ReadReg(regAddr)
    #Construct default value of register for emulation
    def getDefaultRegValue(s, reg):
        val = 0
        for bit in reg.regBits:
            try:
                bitVal = int(bit.defaultValue,16)
            except:
                bitVal = 0
            val = val | (bitVal << int(bit.endBitIdx))
        return val
