"""
DS90UB933_Lib.py

<Insert library module description>

Copyright (c) 2016 Texas Instruments Incorporated.
All Rights Reserved.
"""

import os
import sys
import NanoALP
from array import array
from ALPExtensionModule import ALPExtensionModule

# List of register lists that specify the default value and access type for the reg.
# When WriteReg is called below the access type is checked and applied to the operation.
READ_WRITE = 0x1
READ_ONLY  = 0x0

DS90UB933RegDefaults = {0x00: [0x00, READ_WRITE],  # /reg 00
                        0x01: [0x01, READ_WRITE],  # /reg 01
                        0x02: [0x02, READ_WRITE],  # /reg 02
                        0x03: [0x03, READ_WRITE],  # /reg 03
                        0x04: [0x04, READ_WRITE],  # /reg 04
                        0x05: [0x05, READ_WRITE],  # /reg 05
                        0x06: [0x06, READ_WRITE],  # /reg 06
                        0x07: [0x07, READ_WRITE],  # /reg 07
                        0x08: [0x08, READ_WRITE],  # /reg 08
                        0x09: [0x09, READ_WRITE],  # /reg 09
                        0x0A: [0x0A, READ_WRITE],  # /reg 0A
                        0x0B: [0x0B, READ_WRITE],  # /reg 0B
                        0x0C: [0x0C, READ_WRITE],  # /reg 0C
                        0x0D: [0x0D, READ_WRITE],  # /reg 0D
                        0x0E: [0x0E, READ_WRITE],  # /reg 0E
                        0x0F: [0x0F, READ_WRITE],  # /reg 0F
                        0x10: [0x10, READ_WRITE],  # /reg 10
                        0x11: [0x11, READ_WRITE],  # /reg 11
                        0x12: [0x12, READ_WRITE],  # /reg 12
                        0x13: [0x13, READ_WRITE],  # /reg 13
                        0x14: [0x14, READ_WRITE],  # /reg 14
                        0x16: [0x16, READ_WRITE],  # /reg 16
                        0x1E: [0x1E, READ_WRITE],  # /reg 1E
                        0x23: [0x23, READ_WRITE],  # /reg 23
                        0x29: [0x29, READ_WRITE],  # /reg 29
                        0x2A: [0x2A, READ_WRITE],  # /reg 2A
                        0x34: [0x34, READ_WRITE],  # /reg 34
                        0x35: [0x35, READ_WRITE] } # /reg 35


class DS90UB933( ALPExtensionModule): 	
    shortDescription = 'DS90UB933'
    partnerShortDescription = 'DS90UB914'
    
    def __init__( s, *args, **argd):
        ALPExtensionModule.__init__( s, *args, **argd)
        s.devAddr = 0xB0
        s.txAddr = 0xC0
        s.rxAddr = 0xB0
        s.emulRegs = DS90UB933RegDefaults

    def SetDeviceSlaveAddr(s, slaveAddr):
        s.devAddr = slaveAddr

    def ReadReg( s, regIndex):

        if s.demoMode:
            try:
                return s.emulRegs[regIndex][0]
            except IndexError:
                return 0

        if hasattr(s.alpBaseBoard,'shortName'):
            if s.alpBaseBoard.shortName == 'USB2ANY':
                readStatus = s.alpBaseBoard.USB2ANY.i2c_readByteMult(s.devAddr>>1,regIndex,1)
            elif s.alpBaseBoard.shortName == 'Aardvark':
                readValue = s.alpBaseBoard.Aardvark.readI2C(s.devAddr>>1,regIndex,1)
                readStatus = []
                if not readValue:
                    readStatus.append(0)
                else:
                    readStatus.insert(0,1)
                    readStatus.append(readValue.tolist())
            else:
                readStatus = NanoALP.ReadI2C( s.GetALPInterface(), s.devAddr, regIndex, 1)
        else:
            readStatus = NanoALP.ReadI2C( s.GetALPInterface(), s.devAddr, regIndex, 1)

        return int(array( 'B', readStatus[1]).tolist()[0])

    def WriteReg( s, regIndex, regVal):
        if s.demoMode:
            try:
                s.emulRegs[regIndex][0]= regVal
            except IndexError:
                pass
            return

        if hasattr(s.alpBaseBoard,'shortName'):
            if s.alpBaseBoard.shortName == 'USB2ANY':
                s.alpBaseBoard.USB2ANY.i2c_writeByteMult(s.devAddr>>1,regIndex,regVal)
            elif s.alpBaseBoard.shortName == 'Aardvark':
                s.alpBaseBoard.Aardvark.writeI2C(s.devAddr>>1,regIndex,regVal)
            else:
                NanoALP.WriteI2C( s.GetALPInterface(), s.devAddr, regIndex, array( 'B', regVal).tostring())
        else:
            NanoALP.WriteI2C( s.GetALPInterface(), s.devAddr, regIndex, array( 'B', [regVal,]).tostring())

    def ReadI2C( s, devAddr, regIndex, regLength=1):
        if s.demoMode:
            s.rc = 1
            if (regLength == 1):
                if (devAddr == s.devAddr):
                    return s.emulRegs[regIndex][0]
                elif (devAddr == s.rxAddr):
                    return s.emulRxRegs[regIndex][0]
                elif (devAddr == s.txAddr):  #s.txAddr[0]
                    return s.emulTxRegs[regIndex][0]
                else:   # emulate read from missing device
                    s.rc = 0
		    return regIndex
            else:
                readStatus = []
                for i in range(regLength):
                    if (devAddr == s.devAddr):
                        readStatus += [s.emulRegs[regIndex+i][0]]
                    elif (devAddr == s.rxAddr):
                        readStatus += [s.emulRxRegs[regIndex+i][0]]
                    elif (devAddr == s.txAddr):  #s.txAddr[0]
                        readStatus += [s.emulTxRegs[regIndex+i][0]]
                    else:   # emulate read from missing device
		        s.rc = 0
                        readStatus += [regIndex+i]
                return readStatus
         
        if hasattr(s.alpBaseBoard,'shortName'):
            if s.alpBaseBoard.shortName == 'USB2ANY':
                readStatus = s.alpBaseBoard.USB2ANY.i2c_readByteMult(devAddr>>1,regIndex,1)
            elif s.alpBaseBoard.shortName == 'Aardvark':
                readValue = s.alpBaseBoard.Aardvark.readI2C(devAddr>>1,regIndex,1)
                readStatus = []
                if not readValue:
                    readStatus.append(0)
                else:
                    readStatus.insert(0,1)
                    readStatus.append(readValue.tolist())
            else:
                readStatus = NanoALP.ReadI2C( s.GetALPInterface(), devAddr, regIndex, 1)
        else:
            readStatus = NanoALP.ReadI2C( s.GetALPInterface(), devAddr, regIndex, 1)
         
        #readStatus = NanoALP.ReadI2C( s.GetALPInterface(), devAddr, regIndex, regLength)
        s.rc = readStatus[0]
        if (regLength == 1):
            return int(array( 'B', readStatus[1]).tolist()[0])
        else:
            return array( 'B', readStatus[1]).tolist()
            
    # WriteI2C writes to the specified I2C address
    def WriteI2C( s, devAddr, regIndex, regVal):
        if s.demoMode:
            s.rc = 1
	    if (devAddr == s.devAddr):  # Only write if address matches
                try:    # Multi-byte write
                    for i in range(len( regVal)):
                        s.emulRegs[regIndex+i][0]= regVal[i]
                except:
                    s.emulRegs[regIndex][0]= regVal
            elif (devAddr == s.rxAddr):
                try:    # Multi-byte write
                    for i in range(len( regVal)):
                        s.emulRxRegs[regIndex+i][0]= regVal[i]
                except:
                    s.emulRxRegs[regIndex][0]= regVal
            elif (devAddr == s.txAddr[0]):
                try:    # Multi-byte write
                    for i in range(len( regVal)):
                        s.emulTxRegs[regIndex+i][0]= regVal[i]
                except:
                    s.emulTxRegs[regIndex][0]= regVal
	    else:
		s.rc = 0
            return
        try:    # Do multi-byte write
            len(regVal)
            reg_list = regVal
        except:
            reg_list = [regVal,]
        
        if hasattr(s.alpBaseBoard,'shortName'):
            if s.alpBaseBoard.shortName == 'USB2ANY':
                s.alpBaseBoard.USB2ANY.i2c_writeByteMult(devAddr>>1,regIndex,regVal)
            elif s.alpBaseBoard.shortName == 'Aardvark':
                s.alpBaseBoard.Aardvark.writeI2C(devAddr>>1,regIndex,regVal)
            else:
                NanoALP.WriteI2C( s.GetALPInterface(), devAddr, regIndex, array( 'B', regVal).tostring())
        else:
            NanoALP.WriteI2C( s.GetALPInterface(), devAddr, regIndex, array( 'B', [regVal,]).tostring())
        
        #s.rc = NanoALP.WriteI2C( s.GetALPInterface(), devAddr, regIndex, array( 'B', reg_list).tostring())

            
    def Deinit( s):
        """Called to cleanup state and resources when we're being unloaded."""
        pass
        
    def GetLinkStatus(s):
        # Cable link detect
        if s.demoMode:
            return 1
        # link = s.ReadRXReg(0x00) #Check if the slave's DES ID is the same as the local Device ID
         #Check bit 0 of register 0x1C
        elif ((s.ReadReg(0x0C) & 0x01) == 0x01):
            return s.ReadReg(0x06)
        else:
            return 0x00

     # ReadRXReg reads from the I2C address specified in register 0x07
    def ReadTXReg( s, regIndex, regLength=1):

        readVal = s.ReadI2C(s.txAddr, regIndex, regLength)
        return readVal          

