"""
DS90UB953_ENG_GUI.py


Copyright (c) 2016 Texas Instruments Incorporated.
All Rights Reserved.
"""

import os
import sys
import wx
from Utils import *
from ScriptingGUI import *
from RegistersGUI import *
from RemoteRegistersGUI import *
from InternalRegistersGUI import InternalRegistersGUI
from InternalRegistersGUI import *
from RemoteRegistersGUI import *
from time import sleep

import DS90UB953_ENG_Lib

# Definitions
DEF_SPACE           = 15
PANEL_ORIGIN_LOC    = (DEF_SPACE, DEF_SPACE)
STATUS_TIMER        = 1000  # milliseconds

ValidAddressList = ["0x30", \
                    "0x3A"]

class DS90UB953_ENG_GUI:

    def __init__( s, auiMgr, panel, alpBoards, boardObj):
        s.auiMgr = auiMgr
        s.mainPanel = panel
        s.alpBoards = alpBoards
        s.board = boardObj
        s.linkSts = 'No'
        
        # These are the tabbed panels we support for this device
        # s.deviceTabNames = ('Information', 'Slave Address', 'Registers','Scripting', 'CSI Registers', 'Remote Registers')
        s.deviceTabNames = ('Information', 'Registers','Scripting', 'CSI Registers', 'Remote Registers')

        # This is the tab dispatch table
        s.tabDispatch = { s.deviceTabNames[0]: s.UpdateInformationTab,
                          # s.deviceTabNames[1]: s.UpdateSlaveAddressTab,
                          s.deviceTabNames[1]: s.UpdateRegistersTab,
                          s.deviceTabNames[2]: s.UpdateScriptingTab,
                          s.deviceTabNames[3]: s.UpdateCSIRegistersTab,
                          s.deviceTabNames[4]: s.UpdateRemoteRegistersTab
                          }

        # Build our main notebook with the tabbed panels we want
        s.nb = wx.Notebook( s.mainPanel, -1)
        s.nb.Bind( wx.EVT_NOTEBOOK_PAGE_CHANGED, s.OnTabSelect)

        s.nbPanels = {}
        s.nbInitialized = {}
        for nbPage in s.deviceTabNames:
            nbPanel = wx.Panel( s.nb, -1, style=wx.TAB_TRAVERSAL)
            s.nbPanels[nbPage] = nbPanel
            s.nbInitialized[nbPage] = False
            s.nb.AddPage( nbPanel, nbPage)

        s.tabSizer = wx.BoxSizer( wx.VERTICAL)
        s.topSpacerWin = wx.Window( s.mainPanel, -1, size=(-1,5))
        s.topSpacerWin.SetBackgroundColour( s.nb.GetBackgroundColour())
        s.tabSizer.Add( s.topSpacerWin, 0.5, flag=wx.EXPAND)
        s.tabSizer.Add( s.nb, 1, flag=wx.EXPAND)
        s.mainPanel.SetSizer( s.tabSizer)
        s.mainPanel.Layout()
        s.nb.Show( True)

        if('953' in s.board.GetBoardInfo()['shortName']):
            modName = s.board.profile.profileName + "_CSI.txt"
            s.CSIRegFilePath = os.path.join(s.board.profileDir, modName)
            if (os.path.exists(s.CSIRegFilePath)):
                s.InsertTab(-1, 'CSI Registers', s.UpdateCSIRegistersTab)
            # modName = s.board.profile.profileName + "_FPD3.txt"
            # s.FPD3RegFilePath = os.path.join(s.board.profileDir, modName)
            # if (os.path.exists(s.FPD3RegFilePath)):
                # s.InsertTab(-1, 'FPD3 Registers', s.UpdateFPD3RegistersTab)
            

    def OnTabSelect( s, nbEvent):
        # This dispatches to the tabbed panel handler when that tab has been selected
        # by the user.
        
        s.currTabIdx = nbEvent.GetSelection()
        s.currTabTxt = s.nb.GetPageText( s.currTabIdx)
        s.panel = s.nbPanels[s.currTabTxt]
        
        if not s.nbInitialized[s.currTabTxt] and s.currTabTxt in s.tabDispatch:
            s.tabDispatch[s.currTabTxt]()
            s.nbInitialized[s.currTabTxt] = True
        
        if s.currTabTxt == 'CSI Registers':
            s.SetCSIRegistersTab()
        # if s.currTabTxt == 'FPD3 Registers':
            # s.UpdateFPD3RegistersTab()
            
        nbEvent.Skip()
        
    def UpdateRegistersTab( s):
        s.regGUI = RegistersGUI( s.panel, s.board)
    
    def UpdateScriptingTab( s):
        s.scriptingGUI = ScriptingGUI( s.panel, s.alpBoards, s.board)

    def UpdateInformationTab(s):
        # Panel heading
        #s.headingTxt = WriteText( s.panel, s.board.longDescription, position=None, bold=True)
        
        # Build the status boxes
        s.BuildDevStsBox()
        
        # Build the link status box
        s.BuildLinkStsBox()
        
        # Adjust the local/partner boxes to be the same horiz. width
        ResizeStaticGroupBoxesToMax([s.devStsBoxSz, s.linkStsBoxSz, s.partStsBoxSz])
 
 
        # Add the boxes to the panel sizer
        s.mainSizer = wx.GridBagSizer( DEF_SPACE, DEF_SPACE)
        #s.mainSizer.Add( s.headingTxt,   (0,0), border=DEF_SPACE, flag=wx.TOP|wx.LEFT|wx.RIGHT)
        s.mainSizer.Add( s.devStsPanel,  (0,0), border=DEF_SPACE, flag=wx.TOP|wx.LEFT|wx.RIGHT)
        s.mainSizer.Add( s.partStsPanel, (1,0), border=DEF_SPACE, flag=wx.LEFT|wx.RIGHT)
        s.mainSizer.Add( s.linkStsPanel, (2,0), border=DEF_SPACE, flag=wx.LEFT|wx.RIGHT)

        # Lay it all out
        s.panel.SetAutoLayout( True)
        s.panel.SetSizer( s.mainSizer)
        s.mainSizer.Layout()
        
        # Polling timer
        s.timer = wx.Timer( s.panel)
        wx.EVT_TIMER( s.panel, s.timer.GetId(), s.OnTimer)
        s.timer.Start(STATUS_TIMER)

    def UpdateInformationTab_old(s):
        tab = wx.BoxSizer(wx.VERTICAL)
        txt = WriteText(s.panel, "DS90UB953 FPD-Link III Serializer CSI-2 (TI Only)", bold=True)
        tx2 = WriteText(s.panel, " ", bold=True)
        s.deviceEntryBs = wx.BoxSizer(wx.HORIZONTAL)
        s.txtLbl = WriteText(s.panel, "Device Slave Address", bold=True)
        s.txtCtrl = wx.ListBox(s.panel, choices = ValidAddressList, style=wx.LB_SINGLE )
        s.txtCtrl.SetSelection(0)
        devaddr = int(s.txtCtrl.GetStringSelection(), 16)
        s.board.SetDeviceSlaveAddr(devaddr)
        s.deviceEntryBs.Add(s.txtLbl, flag= wx.ALL | wx.ALIGN_CENTER_HORIZONTAL | wx.ALIGN_CENTER_VERTICAL, border = 10)
        s.deviceEntryBs.Add(s.txtCtrl, flag= wx.ALL | wx.ALIGN_CENTER_HORIZONTAL | wx.ALIGN_CENTER_VERTICAL, border = 10)
        wx.EVT_LISTBOX(s.txtCtrl, s.txtCtrl.GetId(), s.txtEvt)
        
        tab.Add(txt, flag= wx.ALL | wx.ALIGN_CENTER_HORIZONTAL | wx.ALIGN_CENTER_VERTICAL, border = 15)
        tab.Add(tx2, flag= wx.ALL | wx.ALIGN_CENTER_HORIZONTAL | wx.ALIGN_CENTER_VERTICAL, border = 15)
        tab.Add(s.deviceEntryBs, flag= wx.ALL | wx.ALIGN_CENTER_HORIZONTAL | wx.ALIGN_CENTER_VERTICAL, border = 15)
        
        s.panel.SetSizer(tab)
        s.panel.Layout()
        s.nb.GetParent().FitInside()
        
    def OnTimer( s, event):
        if s.currTabTxt == 'Information':
            s.RefreshInformationTab()

    def RefreshInformationTab(s):
        s.UpdateLinkInfo()
        s.UpdateDevInfo()

    def UpdateSlaveAddressTab(s):
        tab = wx.BoxSizer(wx.VERTICAL)
        # txt = WriteText(s.panel, s.board.profile.shortName, bold=True)
        tx2 = WriteText(s.panel, " ", bold=True)
        s.deviceEntryBs = wx.BoxSizer(wx.HORIZONTAL)
        s.txtLbl = WriteText(s.panel, "Device Slave Address", bold=True)
        s.txtCtrl = wx.ListBox(s.panel, choices = ValidAddressList, style=wx.LB_SINGLE )
        s.txtCtrl.SetSelection(0)
        devaddr = int(s.txtCtrl.GetStringSelection(), 16)
        s.board.SetDeviceSlaveAddr(devaddr)
        s.deviceEntryBs.Add(s.txtLbl, flag= wx.ALL | wx.ALIGN_CENTER_HORIZONTAL | wx.ALIGN_CENTER_VERTICAL, border = 10)
        s.deviceEntryBs.Add(s.txtCtrl, flag= wx.ALL | wx.ALIGN_CENTER_HORIZONTAL | wx.ALIGN_CENTER_VERTICAL, border = 10)
        wx.EVT_LISTBOX(s.txtCtrl, s.txtCtrl.GetId(), s.txtEvt)
        
        # tab.Add(txt, flag= wx.ALL | wx.ALIGN_CENTER_HORIZONTAL | wx.ALIGN_CENTER_VERTICAL, border = 15)
        tab.Add(tx2, flag= wx.ALL | wx.ALIGN_CENTER_HORIZONTAL | wx.ALIGN_CENTER_VERTICAL, border = 15)
        tab.Add(s.deviceEntryBs, flag= wx.ALL | wx.ALIGN_CENTER_HORIZONTAL | wx.ALIGN_CENTER_VERTICAL, border = 15)
        
        s.panel.SetSizer(tab)
        s.panel.Layout()
        s.nb.GetParent().FitInside()

    def UpdateRemoteRegistersTab(s):
        # if s.board.ReadReg(0x07) == 0x00:    # If the slaveAlias has not been set, sets it to the SlaveID
            # s.board.WriteReg(0x07, s.board.ReadReg(0x06)) 
            # MyPopup(s.panel ,"SlaveAlias(0x07) was set to 0x00. It has been set to SlaveID(0x06) to connect to remote device", "Register Change")
        s.remregGUI = RemoteRegistersGUI(s.panel,s.board)


    def txtEvt(s, evt):
        devaddr = int(s.txtCtrl.GetStringSelection(), 16)
        s.board.SetDeviceSlaveAddr(devaddr)

    def BuildDevStsBox(s):
        # Build local device status box
        s.devStsPanel = wx.Panel( s.panel, style=wx.TAB_TRAVERSAL|wx.CLIP_CHILDREN)
        s.devStsLabel = "Device Information"
        s.devStsBox = wx.StaticBox( s.devStsPanel, -1, s.devStsLabel)
        s.devStsBoxSz = wx.StaticBoxSizer( s.devStsBox, wx.VERTICAL)
        s.devStsPanel.SetSizer( s.devStsBoxSz)
        
        s.devStsCol0Sz = wx.BoxSizer( wx.VERTICAL)
        s.devStsCol1Sz = wx.BoxSizer( wx.VERTICAL)
        s.devStsCol0TxtCtrls = []
        s.devStsCol1TxtCtrls = []
            
        minLabelField = s.panel.GetTextExtent( "1234567890123456")
        minStsField = s.panel.GetTextExtent( "1234567890123456789012345678") 

        for row in ('Device:','Revision:','I2C Address:','Clocking Mode:', 'Link Cabling:', 'BCC Parity Errs:', 'RX Lock Status:' ):
            txt = WriteText( s.devStsPanel, row, None, False)
            s.devStsCol0Sz.Add( txt)
            s.devStsCol0Sz.SetItemMinSize( txt, *minLabelField)
            s.devStsCol0TxtCtrls.append( txt)
            txt = WriteText( s.devStsPanel, "", None, False)
            s.devStsCol1Sz.Add( txt, flag=wx.LEFT|wx.RIGHT, border=10)
            s.devStsCol1Sz.SetItemMinSize( txt, *minStsField)
            s.devStsCol1TxtCtrls.append( txt)
            
        s.devStsColSz = wx.BoxSizer( wx.HORIZONTAL)
        s.devStsColSz.Add( s.devStsCol0Sz)
        s.devStsColSz.Add( s.devStsCol1Sz)       
        s.devStsBoxSz.Add( s.devStsColSz, flag=wx.ALL, border=5)
        
        #Build partner device status box
        s.partStsPanel = wx.Panel( s.panel, style=wx.TAB_TRAVERSAL|wx.CLIP_CHILDREN)
        s.partStsLabel = "Partner Information"
        s.partStsBox = wx.StaticBox( s.partStsPanel, -1, s.partStsLabel)
        s.partStsBoxSz = wx.StaticBoxSizer( s.partStsBox, wx.VERTICAL)
        s.partStsPanel.SetSizer( s.partStsBoxSz)
        
        s.partStsCol0Sz = wx.BoxSizer( wx.VERTICAL)
        s.partStsCol1Sz = wx.BoxSizer( wx.VERTICAL)
        s.partStsCol0TxtCtrls = []
        s.partStsCol1TxtCtrls = []
            
        minLabelField = s.panel.GetTextExtent( "1234567890123456")
        minStsField = s.panel.GetTextExtent( "1234567890123456789012345678")

        for row in ('Device:','Revision:','I2C Address:','Mode:'):
            txt = WriteText( s.partStsPanel, row, None, False)
            s.partStsCol0Sz.Add( txt)
            s.partStsCol0Sz.SetItemMinSize( txt, *minLabelField)
            s.partStsCol0TxtCtrls.append( txt)
            txt = WriteText( s.partStsPanel, "", None, False)
            s.partStsCol1Sz.Add( txt, flag=wx.LEFT|wx.RIGHT, border=10)
            s.partStsCol1Sz.SetItemMinSize( txt, *minStsField)
            s.partStsCol1TxtCtrls.append( txt)
            
        s.partStsColSz = wx.BoxSizer( wx.HORIZONTAL)
        s.partStsColSz.Add( s.partStsCol0Sz)
        s.partStsColSz.Add( s.partStsCol1Sz)       
        s.partStsBoxSz.Add( s.partStsColSz, flag=wx.ALL, border=5)
    
        # Update both boxes
        s.UpdateDevInfo()    
        
    def UpdateLinkInfo(s):
        s.linkSts = 'No'
        if s.board.GetLinkStatus():
            s.linkSts = 'Yes'
        s.linkStsCol1TxtCtrls[0].SetLabel(s.linkSts)
            
    def BuildLinkStsBox(s):
        s.linkStsPanel = wx.Panel( s.panel, style=wx.TAB_TRAVERSAL|wx.CLIP_CHILDREN)
        s.linkStsLabel = "Current Link Status"
        s.linkStsBox = wx.StaticBox( s.linkStsPanel, -1, s.linkStsLabel)
        s.linkStsBoxSz = wx.StaticBoxSizer( s.linkStsBox, wx.VERTICAL)
        s.linkStsPanel.SetSizer( s.linkStsBoxSz)
        
        s.linkStsCol0Sz = wx.BoxSizer( wx.VERTICAL)
        s.linkStsCol1Sz = wx.BoxSizer( wx.VERTICAL)
        s.linkStsCol0TxtCtrls = []
        s.linkStsCol1TxtCtrls = []

        
        minLabelField = s.panel.GetTextExtent( "Linked to Deserializer:")
        minStsField = s.panel.GetTextExtent( "000000")

        row = 'Linked to Deserializer:'
       # for row in rows:
        txt = WriteText( s.linkStsPanel, row, None, False)
        

        s.linkStsCol0Sz.Add( txt)
        s.linkStsCol0Sz.SetItemMinSize( txt, *minLabelField)
        s.linkStsCol0TxtCtrls.append( txt)

        s.linkStsCol0Sz.SetItemMinSize( txt, *minLabelField)
        s.linkStsCol0TxtCtrls.append( txt)
        txt = WriteText( s.linkStsPanel, "", None, False)
        s.linkStsCol1Sz.Add( txt, flag=wx.LEFT|wx.RIGHT, border=10)
        s.linkStsCol1Sz.SetItemMinSize( txt, *minStsField)
        s.linkStsCol1TxtCtrls.append( txt)
            
        s.linkStsColSz = wx.BoxSizer( wx.HORIZONTAL)
        s.linkStsColSz.Add( s.linkStsCol0Sz)
        s.linkStsColSz.Add( s.linkStsCol1Sz)       
        s.linkStsBoxSz.Add( s.linkStsColSz, flag=wx.ALL, border=5)
        
        s.UpdateLinkInfo()
        

    def SetDevInfo(s):
        s.board.GetDevInfo()
        BoardInfo = s.board.GetBoardInfo()
       
        s.board.shortDescription = s.board.modelName
        s.board.longDescription = s.board.modelName + ' FPD-Link III Serializer'
        
        s.board.partnerShortDescription = s.board.rxmodelName
        s.board.partnerLongDescription = s.board.rxmodelName + ' FPD-Link III Deserializer'
    
        return 1
        
        
    def UpdateDevInfo(s):
        s.SetDevInfo()
        
        if s.linkSts == 'No':
            s.board.partnerLongDescription = 'UNKNOWN DESERIALIZER'
            s.partStsPanel.Disable()
        else:
            s.partStsPanel.Enable()
            
        #Checking Revision Number
        txrev = s.board.ReadReg(s.board.tx.REVISION_ID )
        txrev &= s.board.tx.REV_MASK_ID
        txrev >>= 4
        
        #Checking I2C Address 
        txAddr = s.board.ReadReg(s.board.tx.I2C_DEVICE_ID)
        if ((txAddr & 0x1) == 0x1):  #checking the override bit
            txAddr = (txAddr & 0xFE)  

        #Checking PCLK lock
        locked = s.board.ReadReg(s.board.tx.GENERAL_STATUS)
        if ((locked & s.board.tx.RX_LOCK_DETECT) == 0x40): 
            locked = 'True'
        else:
            locked = 'False'

        #Checking Mode         
        txModeSts = s.board.GetTxMode()
            
        #Checking Coax Mode         
        txCablingModeSts = s.board.GetTxCableMode()

        par_err_cnt = s.board.GetParityErrors()
        #print "Accumulating parity errors
        par_err_txt = "%d" % par_err_cnt
            
        txInfo = (s.board.longDescription, txrev,  \
                    "0x%02X" % txAddr, txModeSts, \
                    txCablingModeSts, par_err_txt ,locked)
            
        # dev = txInfo
        i=0
        for row in txInfo:
            txt = "%s" % row
            s.devStsCol1TxtCtrls[i].SetLabel( txt)
            i+=1
            
        #Partner:   
        #Checking I2C Address 
        rxAddr = s.board.ReadTXReg(s.board.tx.DES_ID)
        if ((rxAddr & 0x01) == 0x01):  #checking the override bit
            rxAddr &= 0xFE

        #Checking Revision Number
        rxrev = s.board.ReadRXReg(0x3)
        rxrev &= 0xF0
        rxrev >>= 4
        
        #Checking Mode         
        rxModeSts = s.board.GetRxMode()
        
        #Checking Cabling Mode         
        rxCablingModeSts = s.board.GetRxCableMode()
        
        rxInfo = (s.board.partnerLongDescription, rxrev, \
                    "0x%02X" % rxAddr, rxModeSts)
                    
        # partInfo  = rxInfo
        s.linkSts = s.board.GetLinkStatus()
        
        i=0
        for row in rxInfo:
            txt = "%s" % row
            s.partStsCol1TxtCtrls[i].SetLabel( txt)
            i+=1
        
    def DestroyPane( s):
        # Called by the framework when we're exiting the app or reenumerating
        # devices. We need to clean up async processes, etc.
        pass

    def UpdateCSIRegistersTab(s):
        s.CSIRegGUI = InternalRegistersGUI(s.panel, s.board, s.CSIRegFilePath)

    def SetCSIRegistersTab(s):
        rdata = s.board.ReadReg(s.board.tx.ANA_IA_CTL)
        rdata &= 0xE3  
        s.board.WriteReg(s.board.tx.ANA_IA_CTL, (rdata))
        
    # def UpdateFPD3RegistersTab(s):
        # s.FPD3RegGUI = InternalRegistersGUI(s.panel, s.board, s.FPD3RegFilePath)
        # rdata = s.board.ReadReg(s.board.tx.ANA_IA_CTL)
        # s.board.WriteReg(s.board.tx.ANA_IA_CTL, (rdata))
    
    def InsertTab(s, pos=1, name=None, updateRoutine=None):
        try:
            if pos == -1:
                # append
                s.deviceTabNames.append(name)
            else:
                # insert
                s.deviceTabNames.insert(pos, name)
            s.tabDispatch[name] = updateRoutine
        except:
            pass