"""
DS90UB960_ENG_GUI.py

<Insert GUI module description>

Copyright (c) Texas Instruments
All Rights Reserved.
"""

import os
import sys

import wx
from Utils import *
from ScriptingGUI import *
from ChannelRegistersGUI import ChannelRegistersGUI as RegistersGUI 
from InternalRegistersGUI import InternalRegistersGUI
from InternalRegistersGUI import *
from RemoteRegistersGUI import RemoteRegistersGUI
from time import sleep

import DS90UB960_ENG_Lib

# Definitions
DEF_SPACE           = 15
PANEL_ORIGIN_LOC    = (DEF_SPACE, DEF_SPACE)
STATUS_TIMER        = 1000  # milliseconds


ValidAddressList = ["0x60", \
                    "0x64", \
                    "0x68", \
                    "0x6C", \
                    "0x70", \
                    "0x74", \
                    "0x78", \
                    "0x7A"]

class DS90UB960_ENG_GUI:

    def __init__( s, auiMgr, panel, alpBoards, boardObj):
        s.auiMgr = auiMgr
        s.mainPanel = panel
        s.alpBoards = alpBoards
        s.board = boardObj
        
        # These are the tabbed panels we support for this device
        s.deviceTabNames = ('Information', 'GPIO', 'Forwarding', 'Registers', 'Scripting', 'CSI Registers', 'CSI AN Registers', 'Remote Registers')

        # This is the tab dispatch table
        s.tabDispatch = { s.deviceTabNames[0]: s.UpdateInformationTab,
                          s.deviceTabNames[1]: s.CreateGPIOTab,
                          s.deviceTabNames[2]: s.CreateForwardingTab,
                          s.deviceTabNames[3]: s.UpdateRegistersTab,
                          s.deviceTabNames[4]: s.UpdateScriptingTab,
                          s.deviceTabNames[5]: s.UpdateCSIRegistersTab,
                          s.deviceTabNames[6]: s.UpdateCSIANRegistersTab,
                          s.deviceTabNames[7]: s.UpdateRemoteRegistersTab
                          }

        # Initial values for RX POrt selection
        s.saved_rxPortSel = 0
        s.saved_rxPortWrMask = 0
        s.saved_rxCSISel = 0
        s.saved_rxCSIWrMask = 0

        # Build our main notebook with the tabbed panels we want
        s.nb = wx.Notebook( s.mainPanel, -1)
        s.nb.Bind( wx.EVT_NOTEBOOK_PAGE_CHANGED, s.OnTabSelect)

        s.nbPanels = {}
        s.nbInitialized = {}
        for nbPage in s.deviceTabNames:
            nbPanel = wx.Panel( s.nb, -1, style=wx.TAB_TRAVERSAL)
            s.nbPanels[nbPage] = nbPanel
            s.nbInitialized[nbPage] = False
            s.nb.AddPage( nbPanel, nbPage)

        s.tabSizer = wx.BoxSizer( wx.VERTICAL)
        s.topSpacerWin = wx.Window( s.mainPanel, -1, size=(-1,5))
        s.topSpacerWin.SetBackgroundColour( s.nb.GetBackgroundColour())
        s.tabSizer.Add( s.topSpacerWin, 0.5, flag=wx.EXPAND)
        s.tabSizer.Add( s.nb, 1, flag=wx.EXPAND)
        s.mainPanel.SetSizer( s.tabSizer)
        s.mainPanel.Layout()
        s.nb.Show( True)
        
        if('960' in s.board.GetBoardInfo()['shortName']):
            modName = s.board.profile.profileName + "_CSI.txt"
            s.CSIRegFilePath = os.path.join(s.board.profileDir, modName)
            if (os.path.exists(s.CSIRegFilePath)):
                s.InsertTab(-1, 'CSI Registers', s.UpdateCSIRegistersTab)
            modName = s.board.profile.profileName + "_CSI_AN.txt"
            s.CSIANRegFilePath = os.path.join(s.board.profileDir, modName)
            if (os.path.exists(s.CSIANRegFilePath)):
                s.InsertTab(-1, 'CSI AN Registers', s.UpdateCSIANRegistersTab)
            # modName = s.board.profile.profileName + "_FPD3.txt"
            # s.FPD3RegFilePath = os.path.join(s.board.profileDir, modName)
            # if (os.path.exists(s.FPD3RegFilePath)):
                # s.InsertTab(-1, 'FPD3 Registers', s.UpdateFPD3RegistersTab)


    def BuildButton(s, buttonLabel, onClick, name1="", name2=""):
        ButtonId = wx.NewId()
        ButtonInst = wx.Button( s.panel, ButtonId, buttonLabel)
        ButtonInst.Id = ButtonId
        ButtonInst.name1 = name1
        if (name2 == ""):
            ButtonInst.name2 = name1
        else:
            ButtonInst.name2 = name2
        wx.EVT_BUTTON( s.panel, ButtonId, onClick)
        ButtonInst.Disable()
        return(ButtonId, ButtonInst)

    def OnTabSelect( s, nbEvent):
        # This dispatches to the tabbed panel handler when that tab has been selected
        # by the user.
        
        # save previous TabTxt value
        try:
            oldTabTxt = s.currTabTxt
        except:
            oldTabTxt = ""
        
        # Set new values
        s.currTabIdx = nbEvent.GetSelection()
        s.currTabTxt = s.nb.GetPageText( s.currTabIdx)
        s.panel = s.nbPanels[s.currTabTxt]

        if s.currTabTxt != 'Information':
            s.timer.Stop()
        
        # When exiting Register tab, Get port selections
        if (oldTabTxt == 'Registers'):
            s.board.Get_FPD3_PORT_SEL()
            s.board.Get_CSI_PORT_SEL()        

        # Restore RX Port selection on exit of certain tabs
        restore_list = ['Information', 'GPIO', 'Forwarding']
        #if (oldTabTxt != s.currTabTxt) and (oldTabTxt in restore_list):
        if (oldTabTxt != s.currTabTxt) and (oldTabTxt in restore_list):
            #sleep(0.5)  # not sure why, but this fixes read/write access
            s.board.rxPortSel = s.saved_rxPortSel
            s.board.rxPortWrMask = s.saved_rxPortWrMask
            s.board.Set_FPD3_PORT_SEL()
            s.board.rxCSISel = s.saved_rxCSISel
            s.board.rxCSIWrMask = s.saved_rxCSIWrMask
            s.board.Set_CSI_PORT_SEL()
            
        # Save RX Port selection on entry of certain tabs
        if (s.currTabTxt in restore_list):
            s.board.Get_FPD3_PORT_SEL()
            s.saved_rxPortSel = s.board.rxPortSel
            s.saved_rxPortWrMask = s.board.rxPortWrMask
            s.board.Get_CSI_PORT_SEL()
            s.saved_rxCSISel = s.board.rxCSISel
            s.saved_rxCSIWrMask = s.board.rxCSIWrMask
        
        if not s.nbInitialized[s.currTabTxt] and s.currTabTxt in s.tabDispatch:
            s.tabDispatch[s.currTabTxt]()
            s.nbInitialized[s.currTabTxt] = True
        elif s.currTabTxt == 'Information':
            # Refresh status
            s.RefreshInformationTab()
            s.timer.Start(STATUS_TIMER)
        elif s.currTabTxt == 'GPIO':
            s.UpdateGPIOTab()
            s.RefreshGPIOTab()
            s.timer.Start(STATUS_TIMER)
        elif s.currTabTxt == 'Forwarding':
            s.UpdateForwardingTab()
            s.RefreshForwardingTab()
            s.timer.Start(STATUS_TIMER)
        elif s.currTabTxt == 'CSI Registers':
            s.SetCSIRegistersTab()
        elif s.currTabTxt == 'CSI AN Registers':
            s.SetCSIANRegistersTab()
        # elif s.currTabTxt == 'Remote Registers':
            # s.UpdateRemoteRegistersTab()
            
            
        nbEvent.Skip()
        
    def UpdateRegistersTab( s):
        s.regGUI = RegistersGUI( s.panel, s.board)
    
    def UpdateScriptingTab( s):
        s.scriptingGUI = ScriptingGUI( s.panel, s.alpBoards, s.board)
        try:
            loadFile = open( os.path.join( os.path.dirname( __file__), 'UserScriptButtons.nsb'))
            s.scriptingGUI.userScriptButtons = pickle.load( loadFile)
            loadFile.close()
            s.scriptingGUI.RefreshUserScriptButtons()
        except:
            pass

    def UpdateDevInfo(s):
        s.board.GetDevInfo()
        
        if s.board.rxTestdie:
            rxrev = "%d (TDA1)" % s.board.rxRevId
            #s.board.ApplyTestdieWorkarounds()
        else:
            rxrev = "%d" % s.board.rxRevId

        refclkfreq = s.board.GetRefClkFreq()
            
        rxInfo = (s.board.longDescription, rxrev, \
                    "0x%02x" % s.board.rxAddr, refclkfreq)

        i=0
        for row in rxInfo:
            txt = "%s" % row
            s.devStsCol1TxtCtrls[i].SetLabel( txt)
            i+=1
            
    def UpdateRemoteRegistersTab(s):
        # if s.board.ReadReg(0x5C) == 0x00:    # If the slaveAlias has not been set, sets it to the SlaveID
            # s.board.WriteReg(0x5C, s.board.ReadReg(0x5B)) 
            # MyPopup(s.panel ,"SlaveAlias(0x5C) was set to 0x00. It has been set to SlaveID(0x5B) to connect to remote device", "Register Change")
        s.remregGUI = RemoteRegistersGUI(s.panel,s.board)

    def UpdateLinkInfo(s):
        if (not s.board.rxValid):
            return (0)
        else:
            # Check each port status
            for portIndex in range(4):
                s.UpdateLinkInfo_portX(portIndex)
        return
        
    def UpdateLinkInfo_portX(s, portIndex):
            if (portIndex == 0):
                column_num = s.linkStsCol1TxtCtrls
            elif (portIndex == 1):
                column_num = s.linkStsCol2TxtCtrls
            elif (portIndex == 2):
                column_num = s.linkStsCol3TxtCtrls
            else:
                column_num = s.linkStsCol4TxtCtrls

            freq_txt = ""
            pass_txt ="No"
            bc_freq_txt = ""
            horiz_txt = ""
            horiz_txt_en = True
            vert_txt = ""
            vert_txt_en = True
            eq_txt = ""
            sfilter_txt = ""
            if (s.board.GetRxPortStatus(portIndex)):
                    freq = s.board.GetRxPortFreq(portIndex)
                    freq_txt = "%4d MHz" % freq
                    horiz_len = s.board.GetRxLineLen(portIndex)
                    horiz_txt = "%4d bytes" % horiz_len
                    vert_len = s.board.GetRxLineCnt(portIndex)
                    vert_txt = "%4d lines" % vert_len
                    if (s.board.rx_port_sts2[portIndex] & 0xC0):
                        horiz_txt_en = False
                    if (s.board.rx_port_sts2[portIndex] & 0x1):
                        vert_txt_en = False
            else:
                    freq_txt = "No"
            if (s.board.rxPassSts[portIndex]):
                    pass_txt = "Pass"
            bc_freq = s.board.GetRxPortBCFreq(portIndex)
            bc_freq_txt = "%.2f MHz" % bc_freq
            eq_setting = s.board.ReadRXPortReg(portIndex, s.board.rx.AEQ_STS)
            eq_txt = "%1x / %1x" % ((eq_setting >> 3), (eq_setting & 0x7))
            sfilter_sts = s.board.ReadRXPortReg(portIndex, s.board.rx.SFILTER_STS_0, 2)
            sfil_cdly = sfilter_sts[0] & 0x3F
            sfil_ddly = sfilter_sts[1] & 0x3F
            if (sfil_cdly):
                sfilter_txt = "%6x cdly" % sfil_cdly
            else:
                sfilter_txt = "%6x ddly" % sfil_ddly
            # Accumulate parity errors
            if (s.board.rxParityFlag[portIndex]):
                s.board.par_err_cnt[portIndex] += s.board.GetParityErrors(portIndex)
                #print "Accumulating parity errors: %d" % s.board.par_err_cnt[portIndex]
            par_err_txt = "%d" % s.board.par_err_cnt[portIndex]
            # Accumulate encoder errors
            if ((s.board.rx_port_sts2[portIndex] & s.board.rx.FPD3_ENCODE_ERROR) == s.board.rx.FPD3_ENCODE_ERROR):
                s.board.encode_err_cnt[portIndex] += 1
            encode_err_txt = "%d" % s.board.encode_err_cnt[portIndex]
            
            # Link Status/Freq
            column_num[2].SetLabel( freq_txt)
            # Grey out if unstable or changing
            if (s.board.rxLockStsChg[portIndex]): 
                s.board.lock_chg_cnt[portIndex] += 1
                column_num[2].Disable( )
            else: column_num[2].Enable( )

            # Pass Status
            column_num[3].SetLabel( pass_txt)
                        
            # Horizontal
            column_num[4].SetLabel( horiz_txt)
            # Grey out if unstable or changing
            if (horiz_txt_en and (pass_txt == "Pass")): column_num[4].Enable( )
            else: column_num[4].Disable( )

            # Vertical
            column_num[5].SetLabel( vert_txt)
            if (pass_txt == "Pass"): column_num[5].Enable( )
            else: column_num[5].Disable( )

            # BC Freq
            column_num[6].SetLabel( bc_freq_txt)
            
            # EQ Setting
            column_num[7].SetLabel( eq_txt)

            # SFILTER Setting
            column_num[8].SetLabel( sfilter_txt)

            # Lock Lost count
            lock_chg_txt = "%d" % s.board.lock_chg_cnt[portIndex]
            column_num[9].SetLabel( lock_chg_txt)
            
            # Parity Error counter
            column_num[10].SetLabel( par_err_txt)

            # Encoder Error counter
            column_num[11].SetLabel( encode_err_txt)
            
            return

    # Update RX Port Config
    def UpdateRxPortConfig(s):
        if (not s.board.rxValid):
            return (0)
        else:
            s.board.rx_port_ctl_val = s.board.ReadReg(s.board.rx.RX_PORT_CTL)
            s.port0EnableChk.SetValue(s.board.rx_port_ctl_val & s.board.rx.PORT_EN0)
            s.port1EnableChk.SetValue(s.board.rx_port_ctl_val & s.board.rx.PORT_EN1)
            s.port2EnableChk.SetValue(s.board.rx_port_ctl_val & s.board.rx.PORT_EN2)
            s.port3EnableChk.SetValue(s.board.rx_port_ctl_val & s.board.rx.PORT_EN3)
            
            # RX_PORT_CTL per port
            s.board.port_cfg_val[0] = s.board.ReadRXPortReg(0, s.board.rx.PORT_CONFIG)
            s.board.port_cfg_val[1] = s.board.ReadRXPortReg(1, s.board.rx.PORT_CONFIG)
            s.board.port_cfg_val[2] = s.board.ReadRXPortReg(2, s.board.rx.PORT_CONFIG)
            s.board.port_cfg_val[3] = s.board.ReadRXPortReg(3, s.board.rx.PORT_CONFIG)

            s.port0InputMode.SetSelection((s.board.port_cfg_val[0] >> 0) & 0x3)
            s.port1InputMode.SetSelection((s.board.port_cfg_val[1] >> 0) & 0x3)
            s.port2InputMode.SetSelection((s.board.port_cfg_val[2] >> 0) & 0x3)
            s.port3InputMode.SetSelection((s.board.port_cfg_val[3] >> 0) & 0x3)

            s.port0Cabling.SetSelection((s.board.port_cfg_val[0] >> 2) & 0x1)
            s.port1Cabling.SetSelection((s.board.port_cfg_val[1] >> 2) & 0x1)
            s.port2Cabling.SetSelection((s.board.port_cfg_val[2] >> 2) & 0x1)
            s.port3Cabling.SetSelection((s.board.port_cfg_val[3] >> 2) & 0x1)

            # PORT_PASS_CTL per port
            s.board.port_pass_ctl_val[0] = s.board.ReadRXPortReg(0, s.board.rx.PORT_PASS_CTL)
            s.board.port_pass_ctl_val[1] = s.board.ReadRXPortReg(1, s.board.rx.PORT_PASS_CTL)
            s.board.port_pass_ctl_val[2] = s.board.ReadRXPortReg(2, s.board.rx.PORT_PASS_CTL)
            s.board.port_pass_ctl_val[3] = s.board.ReadRXPortReg(3, s.board.rx.PORT_PASS_CTL)

            s.port0PassThresh.SetSelection((s.board.port_pass_ctl_val[0] >> 0) & 0x3)
            s.port1PassThresh.SetSelection((s.board.port_pass_ctl_val[1] >> 0) & 0x3)
            s.port2PassThresh.SetSelection((s.board.port_pass_ctl_val[2] >> 0) & 0x3)
            s.port3PassThresh.SetSelection((s.board.port_pass_ctl_val[3] >> 0) & 0x3)
            
        return
                 
    # Update CSI Info
    def UpdateCSIInfo(s):
        if (not s.board.rxValid):
            return (0)
        else:
            # Check each port status
            pass_txt = ["No", "No"]
            sync_txt = ["No","No"]
            for portIndex in range(2):
                s.board.GetCSITXStatus(portIndex)
                if (s.board.csi_pass_sts[portIndex]):
                    pass_txt[portIndex] = "Pass"
                if (s.board.csi_sync_sts[portIndex]):
                    sync_txt[portIndex] = "Yes"
                    
            # Pass Status
            s.csiStsCol1TxtCtrls[2].SetLabel( pass_txt[0])
            s.csiStsCol2TxtCtrls[2].SetLabel( pass_txt[1])
            
            # Synchronized Status
            s.csiStsCol1TxtCtrls[3].SetLabel( sync_txt[0])
            s.csiStsCol2TxtCtrls[3].SetLabel( sync_txt[1])
            
        return
     
    def BuildDevStsBox(s):
        # Build local device status box
        s.devStsPanel = wx.Panel( s.panel, style=wx.TAB_TRAVERSAL|wx.CLIP_CHILDREN)
        s.devStsLabel = "Device Information"
        s.devStsBox = wx.StaticBox( s.devStsPanel, -1, s.devStsLabel)
        s.devStsBoxSz = wx.StaticBoxSizer( s.devStsBox, wx.VERTICAL)
        s.devStsPanel.SetSizer( s.devStsBoxSz)
        
        s.devStsCol0Sz = wx.BoxSizer( wx.VERTICAL)
        s.devStsCol1Sz = wx.BoxSizer( wx.VERTICAL)
        s.devStsCol0TxtCtrls = []
        s.devStsCol1TxtCtrls = []
            
        minLabelField = s.panel.GetTextExtent( "I2C Address:")
        #        minStsField = s.panel.GetTextExtent( s.board.modelName + " FPD-Link III Deserializer to CSI-2 ADAS HUB")
        minStsField = s.panel.GetTextExtent( "DS90UB96x FPD-Link III Deserializer to CSI-2 ADAS HUB")
        for row in ('Device:','Revision:','I2C Address:', 'Refclk Freq:'):
            txt = WriteText( s.devStsPanel, row, None, False)
            s.devStsCol0Sz.Add( txt)
            s.devStsCol0Sz.SetItemMinSize( txt, *minLabelField)
            s.devStsCol0TxtCtrls.append( txt)
            txt = WriteText( s.devStsPanel, "", None, False)
            s.devStsCol1Sz.Add( txt, flag=wx.LEFT|wx.RIGHT, border=10)
            s.devStsCol1Sz.SetItemMinSize( txt, *minStsField)
            s.devStsCol1TxtCtrls.append( txt)
            
        s.devStsColSz = wx.BoxSizer( wx.HORIZONTAL)
        s.devStsColSz.Add( s.devStsCol0Sz)
        s.devStsColSz.Add( s.devStsCol1Sz)       
        
        # Testdie workarounds
        # s.UntrimmedChk = wx.CheckBox( s.devStsPanel, wx.NewId(), "Untrimmed UB964 Workarounds")
        # wx.EVT_CHECKBOX( s.UntrimmedChk, s.UntrimmedChk.GetId(), s.UntrimmedClick)
        # s.UntrimmedChk.SetValue(False)
            
        # Build status box
        s.devStsBoxSz.Add( s.devStsColSz, flag=wx.ALL, border=5)
        # s.devStsBoxSz.Add( s.UntrimmedChk, flag=wx.ALL, border=5)
        
        # Build partner device status box
        s.partStsPanel = wx.Panel( s.panel, style=wx.TAB_TRAVERSAL|wx.CLIP_CHILDREN)
        s.partStsLabel = "Partner Information"
        s.partStsBox = wx.StaticBox( s.partStsPanel, -1, s.partStsLabel)
        s.partStsBoxSz = wx.StaticBoxSizer( s.partStsBox, wx.VERTICAL)
        s.partStsPanel.SetSizer( s.partStsBoxSz)
        
        s.partStsCol0Sz = wx.BoxSizer( wx.VERTICAL)
        s.partStsCol1Sz = wx.BoxSizer( wx.VERTICAL)
        s.partStsCol0TxtCtrls = []
        s.partStsCol1TxtCtrls = []
            
        minLabelField = s.panel.GetTextExtent( "I2C Address:")
        minStsField = s.panel.GetTextExtent( "DS90UB953 FPD-Link III Serializer")
        for row in ('Device:','Revision:','I2C Address:'):
            txt = WriteText( s.partStsPanel, row, None, False)
            s.partStsCol0Sz.Add( txt)
            s.partStsCol0Sz.SetItemMinSize( txt, *minLabelField)
            s.partStsCol0TxtCtrls.append( txt)
            txt = WriteText( s.partStsPanel, "", None, False)
            s.partStsCol1Sz.Add( txt, flag=wx.LEFT|wx.RIGHT, border=10)
            s.partStsCol1Sz.SetItemMinSize( txt, *minStsField)
            s.partStsCol1TxtCtrls.append( txt)
            
        s.partStsColSz = wx.BoxSizer( wx.HORIZONTAL)
        s.partStsColSz.Add( s.partStsCol0Sz)
        s.partStsColSz.Add( s.partStsCol1Sz)       
        s.partStsBoxSz.Add( s.partStsColSz, flag=wx.ALL, border=5)
    
        # Update both boxes
        s.UpdateDevInfo()    
                
    # Forwarding Enable check box
    def onPort0EnableClick(s, event):
        isChecked = s.port0EnableChk.IsChecked()
        if (isChecked):
            s.board.rx_port_ctl_val |= s.board.rx.PORT_EN0
        else:
            s.board.rx_port_ctl_val &= (0xFF & ~s.board.rx.PORT_EN0)
        s.board.WriteReg(s.board.rx.RX_PORT_CTL, s.board.rx_port_ctl_val)
        return
        
    # Forwarding Enable check box
    def onPort1EnableClick(s, event):
        isChecked = s.port1EnableChk.IsChecked()
        if (isChecked):
            s.board.rx_port_ctl_val |= s.board.rx.PORT_EN1
        else:
            s.board.rx_port_ctl_val &= (0xFF & ~s.board.rx.PORT_EN1)
        s.board.WriteReg(s.board.rx.RX_PORT_CTL, s.board.rx_port_ctl_val)
        return
        
    # Forwarding Enable check box
    def onPort2EnableClick(s, event):
        isChecked = s.port2EnableChk.IsChecked()
        if (isChecked):
            s.board.rx_port_ctl_val |= s.board.rx.PORT_EN2
        else:
            s.board.rx_port_ctl_val &= (0xFF & ~s.board.rx.PORT_EN2)
        s.board.WriteReg(s.board.rx.RX_PORT_CTL, s.board.rx_port_ctl_val)
        return
        
    # Forwarding Enable check box
    def onPort3EnableClick(s, event):
        isChecked = s.port3EnableChk.IsChecked()
        if (isChecked):
            s.board.rx_port_ctl_val |= s.board.rx.PORT_EN3
        else:
            s.board.rx_port_ctl_val &= (0xFF & ~s.board.rx.PORT_EN3)
        s.board.WriteReg(s.board.rx.RX_PORT_CTL, s.board.rx_port_ctl_val)
        return

    # Input Mode selection
    def port0InputModeClick(s, event):
        portInputMode = (s.port0InputMode.GetSelection())
        s.board.port_cfg_val[0] &= 0xFC
        s.board.port_cfg_val[0] |= (portInputMode << 0)
        s.board.WriteRXPortReg(0, s.board.rx.PORT_CONFIG, s.board.port_cfg_val[0])
        return

    # Input Mode selection
    def port1InputModeClick(s, event):
        portInputMode = (s.port1InputMode.GetSelection())
        s.board.port_cfg_val[1] &= 0xFC
        s.board.port_cfg_val[1] |= (portInputMode << 0)
        s.board.WriteRXPortReg(1, s.board.rx.PORT_CONFIG, s.board.port_cfg_val[1])
        return

    # Input Mode selection
    def port2InputModeClick(s, event):
        portInputMode = (s.port2InputMode.GetSelection())
        s.board.port_cfg_val[2] &= 0xFC
        s.board.port_cfg_val[2] |= (portInputMode << 0)
        s.board.WriteRXPortReg(2, s.board.rx.PORT_CONFIG, s.board.port_cfg_val[2])
        return

    # Input Mode selection
    def port3InputModeClick(s, event):
        portInputMode = (s.port3InputMode.GetSelection())
        s.board.port_cfg_val[3] &= 0xFC
        s.board.port_cfg_val[3] |= (portInputMode << 0)
        s.board.WriteRXPortReg(3, s.board.rx.PORT_CONFIG, s.board.port_cfg_val[3])
        return

    # Cabling selection
    def port0CablingClick(s, event):
        portCabling = (s.port0Cabling.GetSelection())
        s.board.port_cfg_val[0] &= (0xFF & ~s.board.rx.COAX_MODE)
        s.board.port_cfg_val[0] |= portCabling << 2
        s.board.WriteRXPortReg(0, s.board.rx.PORT_CONFIG, s.board.port_cfg_val[0])
        return

    # Cabling selection
    def port1CablingClick(s, event):
        portCabling = (s.port1Cabling.GetSelection())
        s.board.port_cfg_val[1] &= (0xFF & ~s.board.rx.COAX_MODE)
        s.board.port_cfg_val[1] |= portCabling << 2
        s.board.WriteRXPortReg(1, s.board.rx.PORT_CONFIG, s.board.port_cfg_val[1])
        return

    # Cabling selection
    def port2CablingClick(s, event):
        portCabling = (s.port2Cabling.GetSelection())
        s.board.port_cfg_val[2] &= (0xFF & ~s.board.rx.COAX_MODE)
        s.board.port_cfg_val[2] |= portCabling << 2
        s.board.WriteRXPortReg(2, s.board.rx.PORT_CONFIG, s.board.port_cfg_val[2])
        return

    # Cabling selection
    def port3CablingClick(s, event):
        portCabling = (s.port3Cabling.GetSelection())
        s.board.port_cfg_val[3] &= (0xFF & ~s.board.rx.COAX_MODE)
        s.board.port_cfg_val[3] |= portCabling << 2
        s.board.WriteRXPortReg(3, s.board.rx.PORT_CONFIG, s.board.port_cfg_val[3])
        return

    # PASS_THRESHOLD selection
    def port0PassThreshClick(s, event):
        PassThreshold = (s.port0PassThresh.GetSelection())
        s.board.port_pass_ctl_val[0] &= 0xFC
        s.board.port_pass_ctl_val[0] |= (PassThreshold << 0)
        s.board.WriteRXPortReg(0, s.board.rx.PORT_PASS_CTL, s.board.port_pass_ctl_val[0])
        return

    # PASS_THRESHOLD selection
    def port1PassThreshClick(s, event):
        PassThreshold = (s.port1PassThresh.GetSelection())
        s.board.port_pass_ctl_val[1] &= 0xFC
        s.board.port_pass_ctl_val[1] |= (PassThreshold << 0)
        s.board.WriteRXPortReg(1, s.board.rx.PORT_PASS_CTL, s.board.port_pass_ctl_val[1])
        return
        
    # PASS_THRESHOLD selection
    def port2PassThreshClick(s, event):
        PassThreshold = (s.port2PassThresh.GetSelection())
        s.board.port_pass_ctl_val[2] &= 0xFC
        s.board.port_pass_ctl_val[2] |= (PassThreshold << 0)
        s.board.WriteRXPortReg(2, s.board.rx.PORT_PASS_CTL, s.board.port_pass_ctl_val[2])
        return
        
    # PASS_THRESHOLD selection
    def port3PassThreshClick(s, event):
        PassThreshold = (s.port3PassThresh.GetSelection())
        s.board.port_pass_ctl_val[3] &= 0xFC
        s.board.port_pass_ctl_val[3] |= (PassThreshold << 0)
        s.board.WriteRXPortReg(3, s.board.rx.PORT_PASS_CTL, s.board.port_pass_ctl_val[3])
        return
        
    def BuildPortConfigBox(s):
        InputModeTxt = s.board.rxPortModeTxt
        CablingTxt = ["TP", "Coax"]
        PassThreshTxt = ["Disable", "1 Frame", "2 Frames", "3 Frames"]

        # RX Port 0
        # Port enable check box
        s.port0EnableChk = wx.CheckBox( s.panel, wx.NewId(), "RX port 0")
        wx.EVT_CHECKBOX( s.port0EnableChk, s.port0EnableChk.GetId(), s.onPort0EnableClick)

        # FPD3 Mode Drop-down
        (s.port0InputModeSz, s.port0InputMode, s.port0InputModeTxt) = \
            BuildComboBox( s.panel, "", choices=InputModeTxt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.port0InputMode.GetId(), s.port0InputModeClick)

        # Cabling Drop-down
        (s.port0CablingSz, s.port0Cabling, s.port0CablingTxt) = \
            BuildComboBox( s.panel, "", choices=CablingTxt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.port0Cabling.GetId(), s.port0CablingClick)

        # Pass Threshold
        (s.port0PassThreshSz, s.port0PassThresh, s.port0PassThreshTxt) = \
            BuildComboBox( s.panel, "", choices=PassThreshTxt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.port0PassThresh.GetId(), s.port0PassThreshClick)

        # RX Port 1
        # Port enable check box
        s.port1EnableChk = wx.CheckBox( s.panel, wx.NewId(), "RX port 1")
        wx.EVT_CHECKBOX( s.port1EnableChk, s.port1EnableChk.GetId(), s.onPort1EnableClick)

        # FPD3 Mode Drop-down
        (s.port1InputModeSz, s.port1InputMode, s.port1InputModeTxt) = \
            BuildComboBox( s.panel, "", choices=InputModeTxt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.port1InputMode.GetId(), s.port1InputModeClick)

        # Cabling Drop-down
        (s.port1CablingSz, s.port1Cabling, s.port1CablingTxt) = \
            BuildComboBox( s.panel, "", choices=CablingTxt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.port1Cabling.GetId(), s.port1CablingClick)

        # Pass Threshold
        (s.port1PassThreshSz, s.port1PassThresh, s.port1PassThreshTxt) = \
            BuildComboBox( s.panel, "", choices=PassThreshTxt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.port1PassThresh.GetId(), s.port1PassThreshClick)

        # RX Port 2
        # Port enable check box
        s.port2EnableChk = wx.CheckBox( s.panel, wx.NewId(), "RX port 2")
        wx.EVT_CHECKBOX( s.port2EnableChk, s.port2EnableChk.GetId(), s.onPort2EnableClick)

        # FPD3 Mode Drop-down
        (s.port2InputModeSz, s.port2InputMode, s.port2InputModeTxt) = \
            BuildComboBox( s.panel, "", choices=InputModeTxt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.port2InputMode.GetId(), s.port2InputModeClick)

        # Cabling Drop-down
        (s.port2CablingSz, s.port2Cabling, s.port2CablingTxt) = \
            BuildComboBox( s.panel, "", choices=CablingTxt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.port2Cabling.GetId(), s.port2CablingClick)

        # Pass Threshold
        (s.port2PassThreshSz, s.port2PassThresh, s.port2PassThreshTxt) = \
            BuildComboBox( s.panel, "", choices=PassThreshTxt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.port2PassThresh.GetId(), s.port2PassThreshClick)

        # RX Port 3
        # Port enable check box
        s.port3EnableChk = wx.CheckBox( s.panel, wx.NewId(), "RX port 3")
        wx.EVT_CHECKBOX( s.port3EnableChk, s.port3EnableChk.GetId(), s.onPort3EnableClick)

        # FPD3 Mode Drop-down
        (s.port3InputModeSz, s.port3InputMode, s.port3InputModeTxt) = \
            BuildComboBox( s.panel, "", choices=InputModeTxt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.port3InputMode.GetId(), s.port3InputModeClick)

        # Cabling Drop-down
        (s.port3CablingSz, s.port3Cabling, s.port3CablingTxt) = \
            BuildComboBox( s.panel, "", choices=CablingTxt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.port3Cabling.GetId(), s.port3CablingClick)

        # Pass Threshold
        (s.port3PassThreshSz, s.port3PassThresh, s.port3PassThreshTxt) = \
            BuildComboBox( s.panel, "", choices=PassThreshTxt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.port3PassThresh.GetId(), s.port3PassThreshClick)

        # Left side text
        txt0 = WriteText( s.panel, "Port Enable", None, False)
        txt1 = WriteText( s.panel, "Input Mode", None, False)
        txt2 = WriteText( s.panel, "Cabling", None, False)
        txt3 = WriteText( s.panel, "Pass Threshold", None, False)
            
        # Configurations
        s.RxPortCfg = wx.GridBagSizer( 0, 0)
        s.RxPortCfg.Add( txt0, (0,0), border=5, flag=wx.TOP|wx.BOTTOM|wx.RIGHT)
        s.RxPortCfg.Add( txt1, (1,0), border=5, flag=wx.TOP|wx.BOTTOM|wx.RIGHT)
        s.RxPortCfg.Add( txt2, (2,0), border=5, flag=wx.TOP|wx.BOTTOM|wx.RIGHT)
        s.RxPortCfg.Add( txt3, (3,0), border=5, flag=wx.TOP|wx.BOTTOM|wx.RIGHT)
        s.RxPortCfg.Add( s.port0EnableChk, (0,1), border=5, flag=wx.TOP|wx.BOTTOM)
        s.RxPortCfg.Add( s.port0InputModeSz, (1,1), border=5, flag=wx.TOP|wx.BOTTOM)
        s.RxPortCfg.Add( s.port0Cabling, (2,1), border=5, flag=wx.TOP|wx.BOTTOM)
        s.RxPortCfg.Add( s.port0PassThresh, (3,1), border=5, flag=wx.TOP|wx.BOTTOM)
        s.RxPortCfg.Add( s.port1EnableChk, (0,2), border=5, flag=wx.TOP|wx.BOTTOM)
        s.RxPortCfg.Add( s.port1InputModeSz, (1,2), border=5, flag=wx.TOP|wx.BOTTOM)
        s.RxPortCfg.Add( s.port1Cabling, (2,2), border=5, flag=wx.TOP|wx.BOTTOM)
        s.RxPortCfg.Add( s.port1PassThresh, (3,2), border=5, flag=wx.TOP|wx.BOTTOM)
        s.RxPortCfg.Add( s.port2EnableChk, (0,3), border=5, flag=wx.TOP|wx.BOTTOM)
        s.RxPortCfg.Add( s.port2InputModeSz, (1,3), border=5, flag=wx.TOP|wx.BOTTOM)
        s.RxPortCfg.Add( s.port2Cabling, (2,3), border=5, flag=wx.TOP|wx.BOTTOM)
        s.RxPortCfg.Add( s.port2PassThresh, (3,3), border=5, flag=wx.TOP|wx.BOTTOM)
        s.RxPortCfg.Add( s.port3EnableChk, (0,4), border=5, flag=wx.TOP|wx.BOTTOM)
        s.RxPortCfg.Add( s.port3InputModeSz, (1,4), border=5, flag=wx.TOP|wx.BOTTOM)        
        s.RxPortCfg.Add( s.port3Cabling, (2,4), border=5, flag=wx.TOP|wx.BOTTOM)
        s.RxPortCfg.Add( s.port3PassThresh, (3,4), border=5, flag=wx.TOP|wx.BOTTOM)
        
        # create static box
        s.RxPortCfgSB = wx.StaticBox( s.panel, -1, "RX Port Configuration")
        s.RxPortCfgSBSz = wx.StaticBoxSizer( s.RxPortCfgSB, wx.VERTICAL)
        
        s.RxPortCfgSBSz.Add( s.RxPortCfg, flag=wx.ALL, border=5)


    def UntrimmedClick( s, event):
        isChecked = s.UntrimmedChk.IsChecked()
        s.board.ApplyUntrimmedWorkarounds()
        # after applying workarounds, update device info
        s.UpdateDevInfo
            
    def BuildLinkStsBox(s):
        s.linkStsPanel = wx.Panel( s.panel, style=wx.TAB_TRAVERSAL|wx.CLIP_CHILDREN)
        s.linkStsLabel = "Current RX Port Status"
        s.linkStsBox = wx.StaticBox( s.linkStsPanel, -1, s.linkStsLabel)
        s.linkStsBoxSz = wx.StaticBoxSizer( s.linkStsBox, wx.VERTICAL)
        s.linkStsPanel.SetSizer( s.linkStsBoxSz)
        
        s.linkStsCol0Sz = wx.BoxSizer( wx.VERTICAL)
        s.linkStsCol1Sz = wx.BoxSizer( wx.VERTICAL)
        s.linkStsCol2Sz = wx.BoxSizer( wx.VERTICAL)
        s.linkStsCol3Sz = wx.BoxSizer( wx.VERTICAL)
        s.linkStsCol4Sz = wx.BoxSizer( wx.VERTICAL)
        s.linkStsCol0TxtCtrls = []
        s.linkStsCol1TxtCtrls = []
        s.linkStsCol2TxtCtrls = []
        s.linkStsCol3TxtCtrls = []
        s.linkStsCol4TxtCtrls = []
        
        minLabelField = s.panel.GetTextExtent( "123456789012345")
        minStsField = s.panel.GetTextExtent( "123456789012")
        minFreqField = s.panel.GetTextExtent( "255 MHz")
        rows = ('  Port #','-------------','Linked:','Pass Sts:','Horizontal:', 'Vertical:', 'BC Freq:', 'EQ Hi/Lo:', 'S-Filter', 'Lock Chg Cnt:', 'Parity Errs:', 'Encoder Errs:')

        for row in rows:
            txt = WriteText( s.linkStsPanel, row, None, False)
            s.linkStsCol0Sz.Add( txt)
            s.linkStsCol0Sz.SetItemMinSize( txt, *minLabelField)
            s.linkStsCol0TxtCtrls.append( txt)
            txt = WriteText( s.linkStsPanel, "", None, False)
            s.linkStsCol1Sz.Add( txt, flag=wx.LEFT|wx.RIGHT, border=10)
            s.linkStsCol1Sz.SetItemMinSize( txt, *minStsField)
            s.linkStsCol1TxtCtrls.append( txt)
            txt = WriteText( s.linkStsPanel, "", None, False)
            s.linkStsCol2Sz.Add( txt, flag=wx.LEFT|wx.RIGHT, border=10)
            s.linkStsCol2Sz.SetItemMinSize( txt, *minStsField)
            s.linkStsCol2TxtCtrls.append( txt)
            txt = WriteText( s.linkStsPanel, "", None, False)
            s.linkStsCol3Sz.Add( txt, flag=wx.LEFT|wx.RIGHT, border=10)
            s.linkStsCol3Sz.SetItemMinSize( txt, *minStsField)
            s.linkStsCol3TxtCtrls.append( txt)
            txt = WriteText( s.linkStsPanel, "", None, False)
            s.linkStsCol4Sz.Add( txt, flag=wx.LEFT|wx.RIGHT, border=10)
            s.linkStsCol4Sz.SetItemMinSize( txt, *minStsField)
            s.linkStsCol4TxtCtrls.append( txt)
        
        # port #
        s.linkStsCol1TxtCtrls[0].SetLabel( "  0")
        s.linkStsCol2TxtCtrls[0].SetLabel( "  1")
        s.linkStsCol3TxtCtrls[0].SetLabel( "  2")
        s.linkStsCol4TxtCtrls[0].SetLabel( "  3")
        s.linkStsCol1TxtCtrls[1].SetLabel( "---------")
        s.linkStsCol2TxtCtrls[1].SetLabel( "---------")
        s.linkStsCol3TxtCtrls[1].SetLabel( "---------")
        s.linkStsCol4TxtCtrls[1].SetLabel( "---------")
        
        s.linkStsColSz = wx.BoxSizer( wx.HORIZONTAL)
        s.linkStsColSz.Add( s.linkStsCol0Sz)
        s.linkStsColSz.Add( s.linkStsCol1Sz)       
        s.linkStsColSz.Add( s.linkStsCol2Sz)       
        s.linkStsColSz.Add( s.linkStsCol3Sz)       
        s.linkStsColSz.Add( s.linkStsCol4Sz)       
        s.linkStsBoxSz.Add( s.linkStsColSz, flag=wx.ALL, border=5)
        
        #s.UpdateLinkInfo()
    
    # CSI TX status box
    def BuildCSIStsBox(s):
        s.csiStsPanel = wx.Panel( s.panel, style=wx.TAB_TRAVERSAL|wx.CLIP_CHILDREN)
        s.csiStsLabel = "Current CSI TX Status"
        s.csiStsBox = wx.StaticBox( s.csiStsPanel, -1, s.csiStsLabel)
        s.csiStsBoxSz = wx.StaticBoxSizer( s.csiStsBox, wx.VERTICAL)
        s.csiStsPanel.SetSizer( s.csiStsBoxSz)
        
        s.csiStsCol0Sz = wx.BoxSizer( wx.VERTICAL)
        s.csiStsCol1Sz = wx.BoxSizer( wx.VERTICAL)
        s.csiStsCol2Sz = wx.BoxSizer( wx.VERTICAL)
        s.csiStsCol0TxtCtrls = []
        s.csiStsCol1TxtCtrls = []
        s.csiStsCol2TxtCtrls = []
        
        minLabelField = s.panel.GetTextExtent( "Input Mode:")
        minStsField = s.panel.GetTextExtent( "RAW10/100")
        minFreqField = s.panel.GetTextExtent( "255 MHz")
        rows = ('  Port #','-----------','Pass Sts:','Sync Sts:')

        for row in rows:
            txt = WriteText( s.csiStsPanel, row, None, False)
            s.csiStsCol0Sz.Add( txt)
            s.csiStsCol0Sz.SetItemMinSize( txt, *minLabelField)
            s.csiStsCol0TxtCtrls.append( txt)
            txt = WriteText( s.csiStsPanel, "", None, False)
            s.csiStsCol1Sz.Add( txt, flag=wx.LEFT|wx.RIGHT, border=10)
            s.csiStsCol1Sz.SetItemMinSize( txt, *minStsField)
            s.csiStsCol1TxtCtrls.append( txt)
            txt = WriteText( s.csiStsPanel, "", None, False)
            s.csiStsCol2Sz.Add( txt, flag=wx.LEFT|wx.RIGHT, border=10)
            s.csiStsCol2Sz.SetItemMinSize( txt, *minStsField)
            s.csiStsCol2TxtCtrls.append( txt)
        
        # port #
        s.csiStsCol1TxtCtrls[0].SetLabel( "  0")
        s.csiStsCol2TxtCtrls[0].SetLabel( "  1")
        s.csiStsCol1TxtCtrls[1].SetLabel( "---------")
        s.csiStsCol2TxtCtrls[1].SetLabel( "---------")
        
        s.csiStsColSz = wx.BoxSizer( wx.HORIZONTAL)
        s.csiStsColSz.Add( s.csiStsCol0Sz)
        s.csiStsColSz.Add( s.csiStsCol1Sz)       
        s.csiStsColSz.Add( s.csiStsCol2Sz)       
        s.csiStsBoxSz.Add( s.csiStsColSz, flag=wx.ALL, border=5)
        
    def ResetStatsClick(s, event):
        s.board.lock_chg_cnt = [0, 0, 0, 0]
        s.board.par_err_cnt = [0, 0, 0, 0]
        s.board.encode_err_cnt = [0, 0, 0, 0]
        s.UpdateLinkInfo()
        
    def RestartAEQClick(s, event):
        s.board.RestartAeq(0)
        s.board.RestartAeq(1)
        s.board.RestartAeq(2)
        s.board.RestartAeq(3)
    def RestartDigitalClick(s, event):
        s.board.DigitalReset()


    def BuildButtonBox(s):
        s.ResetStatsButtonID, s.ResetStatsButton = s.BuildButton("Reset Statistics", s.ResetStatsClick, "")
        s.ResetStatsButton.Enable()

        s.RestartAEQButtonID, s.RestartAEQButton = s.BuildButton("Restart AEQ", s.RestartAEQClick, "")
        s.RestartAEQButton.Enable()
        
        s.RestartDigitalButtonID, s.RestartDigitalButton = s.BuildButton("Digital Reset", s.RestartDigitalClick, "")
        s.RestartDigitalButton.Enable()

        # Configurations
        s.InfoButtons =  wx.BoxSizer( wx.VERTICAL)
        s.InfoButtons.Add( s.ResetStatsButton, border=DEF_SPACE, flag=wx.BOTTOM)
        s.InfoButtons.Add( s.RestartAEQButton, border=DEF_SPACE, flag=wx.BOTTOM)
        s.InfoButtons.Add( s.RestartDigitalButton, border=DEF_SPACE, flag=wx.BOTTOM)
        
        # create static box
        s.InfoButtonsSB = wx.StaticBox( s.panel, -1, "Diagnostic Controls")
        s.InfoButtonsSBSz = wx.StaticBoxSizer( s.InfoButtonsSB, wx.VERTICAL)
        
        s.InfoButtonsSBSz.Add( s.InfoButtons, flag=wx.ALL, border=5)


    def UpdateInformationTab(s):
        # Panel heading
        #s.headingTxt = WriteText( s.panel, s.board.longDescription, position=None, bold=True)
        
        # Build the status boxes
        s.BuildDevStsBox()
        
        # Build the link status box
        s.BuildLinkStsBox()
        
        # Build RX Port Configuration box
        s.BuildPortConfigBox()
        
        # Build the CSI TX status box
        s.BuildCSIStsBox()
        
        # Build buttons
        s.BuildButtonBox()
        
        # Adjust the local/partner boxes to be the same horiz. width
        ResizeStaticGroupBoxesToMax([s.devStsBoxSz, s.linkStsBoxSz, s.RxPortCfgSBSz])
 
        # Add the boxes to the panel sizer
        s.mainSizer = wx.GridBagSizer( DEF_SPACE, DEF_SPACE)
        #s.mainSizer.Add( s.headingTxt,   (0,0), border=DEF_SPACE, flag=wx.TOP|wx.LEFT|wx.RIGHT)
        s.mainSizer.Add( s.devStsPanel,  (0,0), border=DEF_SPACE, flag=wx.TOP|wx.LEFT)
        s.mainSizer.Add( s.partStsPanel, (0,1), border=DEF_SPACE, flag=wx.TOP|wx.LEFT)
        s.mainSizer.Add( s.RxPortCfgSBSz, (1,0), border=DEF_SPACE, flag=wx.LEFT)
        s.mainSizer.Add( s.linkStsPanel, (2,0), border=DEF_SPACE, flag=wx.LEFT)
        s.mainSizer.Add( s.InfoButtonsSBSz, (1,1), border=DEF_SPACE, flag=wx.LEFT)
        s.mainSizer.Add( s.csiStsPanel, (2,1), border=DEF_SPACE, flag=wx.LEFT)

        # Lay it all out
        s.panel.SetAutoLayout( True)
        s.panel.SetSizer( s.mainSizer)
        s.mainSizer.Layout()
        
        # Refresh tab
        s.RefreshInformationTab()
        
        # Polling timer
        s.timer = wx.Timer( s.panel)
        wx.EVT_TIMER( s.panel, s.timer.GetId(), s.OnTimer)
        s.timer.Start(STATUS_TIMER)

    def OnTimer( s, event):
        if s.currTabTxt == 'Information':
            s.UpdateLinkInfo()
            s.UpdateCSIInfo()
            s.UpdateDevInfo()
        elif s.currTabTxt == 'GPIO':
            s.RefreshGPIOTab()
        elif s.currTabTxt == 'Forwarding':
            s.RefreshForwardingTab()
        
    def RefreshInformationTab(s):
        #print "RefreshInformation"
        s.UpdateLinkInfo()
        s.UpdateRxPortConfig()
        s.UpdateCSIInfo()
        s.UpdateDevInfo()
        return
        
    def DestroyPane( s):
        # Called by the framework when we're exiting the app or reenumerating
        # devices. We need to clean up async processes, etc.
        #Check all devices for various issues
        s.timer.Stop()
        del s.timer
        pass
            
    def UpdateInformationTab_old(s):
        tab = wx.BoxSizer(wx.VERTICAL)
        txt = WriteText(s.panel, "DS90UB960 FPD-Link III HUB CSI2 Deserializer (TI Only)", bold=True)
        tx2 = WriteText(s.panel, " ", bold=True)
        s.deviceEntryBs = wx.BoxSizer(wx.HORIZONTAL)
        s.txtLbl = WriteText(s.panel, "Device Slave Address", bold=True)
        s.txtCtrl = wx.ListBox(s.panel, choices = ValidAddressList, style=wx.LB_SINGLE )
        s.txtCtrl.SetSelection(0)
        devaddr = int(s.txtCtrl.GetStringSelection(), 16)
        s.board.SetDeviceSlaveAddr(devaddr)
        s.deviceEntryBs.Add(s.txtLbl, flag= wx.ALL | wx.ALIGN_CENTER_HORIZONTAL | wx.ALIGN_CENTER_VERTICAL, border = 10)
        s.deviceEntryBs.Add(s.txtCtrl, flag= wx.ALL | wx.ALIGN_CENTER_HORIZONTAL | wx.ALIGN_CENTER_VERTICAL, border = 10)
        wx.EVT_LISTBOX(s.txtCtrl, s.txtCtrl.GetId(), s.txtEvt)
        
        tab.Add(txt, flag= wx.ALL | wx.ALIGN_CENTER_HORIZONTAL | wx.ALIGN_CENTER_VERTICAL, border = 15)
        tab.Add(tx2, flag= wx.ALL | wx.ALIGN_CENTER_HORIZONTAL | wx.ALIGN_CENTER_VERTICAL, border = 15)
        tab.Add(s.deviceEntryBs, flag= wx.ALL | wx.ALIGN_CENTER_HORIZONTAL | wx.ALIGN_CENTER_VERTICAL, border = 15)
        
        s.panel.SetSizer(tab)
        s.panel.Layout()
        s.nb.GetParent().FitInside()

    def BuildFrameSyncPanel(s):
        s.FPS_Selected = True
        s.FSduty_Selected = True
        s.fs_clk_sel = 0
        if (s.board.rxPortCnt > 2):
            fs_choices = ["RX0 BC", "RX1 BC", "RX2 BC", "RX3 BC", "25/12.5 MHz"]
        else:
            fs_choices = ["RX0 BC", "RX1 BC", "Reserved", "Reserved", "25/12.5 MHz"]
        
        # Static box for topology map
        s.FrameSyncBox = wx.StaticBox( s.panel, -1, "FrameSync Generator")
        s.FrameSyncSizer = wx.StaticBoxSizer( s.FrameSyncBox, wx.VERTICAL)
        s.FrameSyncGrid = wx.GridBagSizer( 0,0)
        s.FrameSyncSizer.Add(s.FrameSyncGrid)

        ## CREATING ENABLE BOX        
        # FSync clock source drop-down
        (s.FSclkSourceSz, s.FSclkSource, s.FSclkSourceTxt) = \
            BuildComboBox( s.panel, "FSync Ref:", choices=fs_choices, \
                labelOrientation=wx.LEFT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.FSclkSource.GetId(), s.FSclkSourceClick)
        #s.FSclkSource.SetSelection(s.fs_clk_sel)
            
        # FSync Reference period Status (changes with Ref setting)
        s.FSrefSz = wx.BoxSizer( wx.HORIZONTAL)
        s.FSrefTxtCtrls = []
        
        minStsField = s.panel.GetTextExtent( "not selected")

        txt = WriteText( s.panel, "FSync Ref Period:", None, False)
        s.FSrefSz.Add( txt)
        s.FSrefTxtCtrls.append( txt)
        txt = WriteText( s.panel, "not selected", None, False)
        s.FSrefSz.Add( txt, flag=wx.LEFT|wx.RIGHT, border=10)
        s.FSrefSz.SetItemMinSize( txt, *minStsField)
        s.FSrefTxtCtrls.append( txt)
        
        # Create the Frames/second control
        (s.FSfpsCtlSz, s.FSfpsCtl, s.FSfpsCtlTxt) = BuildEditBox( s.panel, "FSync rate (fps)", numChars=3, labelOrientation=wx.LEFT, enable=True)
        wx.EVT_TEXT_ENTER(s.panel, s.FSfpsCtl.GetId(), s.FSfpsCtlClick)
        s.FSfpsCtlVal = 60
        s.FSfpsCtl.SetValue("%3d" % (s.FSfpsCtlVal))
        
        # Create the FrameSync Period control
        (s.FSperiodCtlSz, s.FSperiodCtl, s.FSperiodCtlTxt) = BuildEditBox( s.panel, "-OR-   FSync period (us)", numChars=10, labelOrientation=wx.LEFT, enable=True)
        wx.EVT_TEXT_ENTER(s.panel, s.FSperiodCtl.GetId(), s.FSperiodCtlClick)
        s.FSperiodCtlVal = 1e6 / s.FSfpsCtlVal
        s.FSperiodCtl.SetValue("%.2f" % (s.FSperiodCtlVal))
        
        # Create the  Duty Cycle control
        (s.FSdutyCycleSz, s.FSdutyCycle, s.FSdutyCycleTxt) = BuildEditBox( s.panel, "Duty Cycle %", numChars=2, labelOrientation=wx.LEFT, enable=True)
        wx.EVT_TEXT_ENTER(s.panel, s.FSdutyCycle.GetId(), s.FSdutyCycleClick)
        s.FSdutyCycleVal = 50
        s.FSdutyCycle.SetValue("%2d" % (s.FSdutyCycleVal))
        
        # Create the FrameSync High Period control
        (s.FSperiodHighCtlSz, s.FSperiodHighCtl, s.FSperiodHighCtlTxt) = BuildEditBox( s.panel, "-OR-   FSync High (us)", numChars=10, labelOrientation=wx.LEFT, enable=True)
        wx.EVT_TEXT_ENTER(s.panel, s.FSperiodHighCtl.GetId(), s.FSperiodHighCtlClick)
        s.FSperiodHighCtlVal = s.FSperiodCtlVal * s.FSdutyCycleVal/100
        s.FSperiodHighCtl.SetValue("%.2f" % (s.FSperiodHighCtlVal))

        # Create the StartStop buttons
        s.FSyncStartButtonID, s.FSyncStartButton = s.BuildButton("Start", s.FSyncStartClick, "")
        s.FSyncStartButton.Enable()
        s.FSyncStopButtonID, s.FSyncStopButton = s.BuildButton("Stop", s.FSyncStopClick, "")
        s.FSyncStopButton.Disable()

        # Button Grid
        #s.topoButtonSizer = wx.GridBagSizer( 0, 0)
        #s.topoButtonSizer.Add( s.topoReadButton,  (0,0), border=0, flag=wx.ALL)
        #s.topoButtonSizer.Add( s.topoWriteButton,  (0,1), border=0, flag=wx.ALL)

        # Static box for topology map
        # s.FrameSyncBox = wx.StaticBox( s.panel, -1, "FrameSync Generator")
        s.FrameSyncSizer = wx.StaticBoxSizer( s.FrameSyncBox, wx.VERTICAL)
        #s.FrameSyncSizer.Add( s.FSfpsCtlSz, flag=wx.ALL, border=5)
        #s.FrameSyncSizer.Add( s.FSperiodCtlSz, flag=wx.ALL, border=5)
        #s.FrameSyncSizer.Add( s.FSdutyCycleSz, flag=wx.ALL, border=5)
        
        s.FrameSyncGridSizer = wx.GridBagSizer( 0, 0)
        s.FrameSyncGridSizer.Add( s.FSclkSourceSz, (0,0), border=5, flag=wx.ALL)
        s.FrameSyncGridSizer.Add( s.FSrefSz, (0,1), border=5, flag=wx.ALL)
        s.FrameSyncGridSizer.Add( s.FSfpsCtlSz, (1,0), border=5, flag=wx.ALL)
        s.FrameSyncGridSizer.Add( s.FSperiodCtlSz, (1,1), border=5, flag=wx.ALL)
        s.FrameSyncGridSizer.Add( s.FSdutyCycleSz, (2,0), border=5, flag=wx.ALL)
        s.FrameSyncGridSizer.Add( s.FSperiodHighCtlSz, (2,1), border=5, flag=wx.ALL)
        s.FrameSyncGridSizer.Add( s.FSyncStartButton, (3,0), border=5, flag=wx.ALL)
        s.FrameSyncGridSizer.Add( s.FSyncStopButton, (3,1), border=5, flag=wx.ALL)

        s.FrameSyncSizer.Add( s.FrameSyncGridSizer, flag=wx.ALL, border=5)

        #s.FrameSyncSizer.Add( s.topoButtonSizer, flag=wx.ALL, border=5)

    # Update FS clock reference period display
    def UpdateFSclkSourceTxt(s):
        if (s.fs_clk_sel == 4):
            ref_freq = s.board.GetRefFreq() / 8
            fs_clk_period = 1000/ref_freq
            fs_clk_txt = "%2d ns" % fs_clk_period
        else:
            portIndex = s.fs_clk_sel & 0x3
            bc_freq = s.board.GetRxPortBCFreq(portIndex)
            fs_clk_period = (30.0)/(bc_freq)
            fs_clk_txt = "%.2f us" % fs_clk_period
        s.FSrefTxtCtrls[1].SetLabel( fs_clk_txt)

    # Frame Sync Clock source
    def FSclkSourceClick(s, event):
        s.fs_clk_sel = (s.FSclkSource.GetSelection())
        s.UpdateFSclkSourceTxt()

    # Compute FrameSync values based on options selected
    def UpdateFrameSync(s):
        if (s.FPS_Selected):
            s.FSperiodCtlVal = 1e6 / s.FSfpsCtlVal
        else:
            s.FSfpsCtlVal = 1e6 / s.FSperiodCtlVal
        
        if (s.FSduty_Selected):
            s.FSperiodHighCtlVal = s.FSperiodCtlVal * s.FSdutyCycleVal/100
        else:
            s.FSdutyCycleVal = s.FSperiodHighCtlVal * 100 / s.FSperiodCtlVal

        # Update the screen in case the value was adjusted
        s.FSfpsCtl.SetValue("%3d" % (s.FSfpsCtlVal))
        s.FSperiodCtl.SetValue("%.2f" % (s.FSperiodCtlVal))
        s.FSdutyCycle.SetValue("%3d" % (s.FSdutyCycleVal))
        s.FSperiodHighCtl.SetValue("%.2f" % (s.FSperiodHighCtlVal))

    def FSfpsCtlClick(s, event):
        s.FPS_Selected = True

        try:
            fps_value = int(s.FSfpsCtl.GetValue(), 10)
        except:
            fps_value = s.FSfpsCtlVal
        #
        # Sanity-check
        if fps_value not in range(1, 1000):
            MyPopup(s.panel, "FPS must be between 1 and 999", "Warning")
        else:
            s.FSfpsCtlVal = fps_value
                
        s.UpdateFrameSync()

    def FSperiodCtlClick(s, event):
        s.FPS_Selected = False

        try:
            fs_period = long(s.FSperiodCtl.GetValue(), 10)
        except:
            fs_period = s.FSperiodCtlVal
        #
        # Sanity-check
        if (fs_period < 1):
            MyPopup(s.panel, "FSync period must be an integer greater than 0", "Warning")
        else:
            s.FSperiodCtlVal = fs_period
        
        s.UpdateFrameSync()

    def FSdutyCycleClick(s, event):
        s.FSduty_Selected = True

        try:
            duty_cycle = int(s.FSdutyCycle.GetValue(), 10)
        except:
            duty_cycle = s.FSdutyCycleVal
        #
        # Sanity-check
        if duty_cycle not in range(1, 100):
            MyPopup(s.panel, "FPS must be between 1 and 99", "Warning")
        else:
            s.FSdutyCycleVal = duty_cycle

        s.UpdateFrameSync()

    def FSperiodHighCtlClick(s, event):
        s.FSduty_Selected = False

        try:
            fs_high = long(s.FSperiodHighCtl.GetValue(), 10)
        except:
            fs_high = s.FSperiodHighCtlVal
        #
        # Sanity-check
        if (fs_high < 1):
            MyPopup(s.panel, "FSync High Time must be an integer greater than 0", "Warning")
        else:
            s.FSperiodHighCtlVal = fs_high
         
        s.UpdateFrameSync()
    
    def FSyncStartClick(s, event):
        # Determine FS_CTL value
        # Temporary, use default modes
        fs_ctl_val = (s.fs_clk_sel << 4) | 0x1
        
        # Get FSync clock period
        # temporary -- use RX0 BC frame period
        if (s.fs_clk_sel == 4):
            ref_freq = s.board.GetRefFreq() / 8
            fs_clk_period = 1.0/ref_freq
        else:
            portIndex = s.fs_clk_sel & 0x3
            bc_freq = s.board.GetRxPortBCFreq(portIndex)
            fs_clk_period = (30.0)/(bc_freq)
        
        # High/Low settings
        fs_period = long(s.FSperiodCtlVal / fs_clk_period)
        fs_high = long(s.FSperiodHighCtlVal / fs_clk_period)
        fs_low = fs_period - fs_high
        fs_high -= 1
        fs_low -= 1
        
        # Set timer controls
        s.board.WriteRXReg(s.board.rx.FS_HI_TIMER1, ((fs_high >> 8) & 0xFF)) 
        s.board.WriteRXReg(s.board.rx.FS_HI_TIMER0, (fs_high & 0xFF))
        s.board.WriteRXReg(s.board.rx.FS_LO_TIMER1, ((fs_low >> 8) & 0xFF)) 
        s.board.WriteRXReg(s.board.rx.FS_LO_TIMER0, (fs_low & 0xFF))
        
        # Start FrameSync
        s.board.WriteRXReg(s.board.rx.FS_CTL, fs_ctl_val)
        
        s.FSyncStartButton.SetLabel("Running")
        s.FSyncStartButton.Disable()
        s.FSyncStopButton.Enable()


        
    def FSyncStopClick(s, event):
        # Disable FrameSync
        fs_ctl_val = s.board.ReadRXReg(s.board.rx.FS_CTL) & 0xFE
        s.board.WriteRXReg(s.board.rx.FS_CTL, fs_ctl_val)
        
        s.FSyncStartButton.SetLabel("Start")
        s.FSyncStartButton.Enable()
        s.FSyncStopButton.Disable()
        
    def BuildBCGPIOBox(s):

        # Get Current BC GPIO control values
        for portIndex in range(4):
            s.board.bc_gpio_ctl0[portIndex] = s.board.ReadRXPortReg(portIndex, s.board.rx.BC_GPIO_CTL0)
            s.board.bc_gpio_ctl1[portIndex] = s.board.ReadRXPortReg(portIndex, s.board.rx.BC_GPIO_CTL1)

        # RX Port 0
        # BC GPIO 0 Source drop-down
        (s.RX0_BCgpio0SourceSz, s.RX0_BCgpio0Source, s.RX0_BCgpio0SourceTxt) = \
            BuildComboBox( s.panel, "BC GPIO0:", choices=["GPIO PIN 0", "GPIO PIN 1", "GPIO PIN 2", "GPIO PIN 3", "GPIO PIN 4", "GPIO PIN 5", "GPIO PIN 6", "GPIO PIN 7", "Const=0", "Const=1", "FrameSync"], \
                labelOrientation=wx.LEFT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.RX0_BCgpio0Source.GetId(), s.RX0_BCgpio0SourceClick)
        s.RX0_BCgpio0Source.SetSelection((s.board.bc_gpio_ctl0[0]) & 0xF)

        # BC GPIO 1 Source drop-down
        (s.RX0_BCgpio1SourceSz, s.RX0_BCgpio1Source, s.RX0_BCgpio1SourceTxt) = \
            BuildComboBox( s.panel, "BC GPIO1:", choices=["GPIO PIN 0", "GPIO PIN 1", "GPIO PIN 2", "GPIO PIN 3", "GPIO PIN 4", "GPIO PIN 5", "GPIO PIN 6", "GPIO PIN 7", "Const=0", "Const=1", "FrameSync"], \
                labelOrientation=wx.LEFT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.RX0_BCgpio1Source.GetId(), s.RX0_BCgpio1SourceClick)
        s.RX0_BCgpio1Source.SetSelection((s.board.bc_gpio_ctl0[0] >> 4) & 0xF)

        # BC GPIO 2 Source drop-down
        (s.RX0_BCgpio2SourceSz, s.RX0_BCgpio2Source, s.RX0_BCgpio2SourceTxt) = \
            BuildComboBox( s.panel, "BC GPIO2:", choices=["GPIO PIN 0", "GPIO PIN 1", "GPIO PIN 2", "GPIO PIN 3", "GPIO PIN 4", "GPIO PIN 5", "GPIO PIN 6", "GPIO PIN 7", "Const=0", "Const=1", "FrameSync"], \
                labelOrientation=wx.LEFT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.RX0_BCgpio2Source.GetId(), s.RX0_BCgpio2SourceClick)
        s.RX0_BCgpio2Source.SetSelection((s.board.bc_gpio_ctl1[0]) & 0xF)

        # BC GPIO 3 Source drop-down
        (s.RX0_BCgpio3SourceSz, s.RX0_BCgpio3Source, s.RX0_BCgpio3SourceTxt) = \
            BuildComboBox( s.panel, "BC GPIO3:", choices=["GPIO PIN 0", "GPIO PIN 1", "GPIO PIN 2", "GPIO PIN 3", "GPIO PIN 4", "GPIO PIN 5", "GPIO PIN 6", "GPIO PIN 7", "Const=0", "Const=1", "FrameSync"], \
                labelOrientation=wx.LEFT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.RX0_BCgpio3Source.GetId(), s.RX0_BCgpio3SourceClick)
        s.RX0_BCgpio3Source.SetSelection((s.board.bc_gpio_ctl1[0] >> 4) & 0xF)


        # Configurations
        s.RX0_BCGPIOstgs = wx.GridBagSizer( 0, 0)
        s.RX0_BCGPIOstgs.Add( s.RX0_BCgpio0SourceSz, (0,0), border=5, flag=wx.ALL)
        s.RX0_BCGPIOstgs.Add( s.RX0_BCgpio1SourceSz, (1,0), border=5, flag=wx.ALL)
        s.RX0_BCGPIOstgs.Add( s.RX0_BCgpio2SourceSz, (2,0), border=5, flag=wx.ALL)
        s.RX0_BCGPIOstgs.Add( s.RX0_BCgpio3SourceSz, (3,0), border=5, flag=wx.ALL)

        # create static box
        s.RX0_BCgpioStgsSB = wx.StaticBox( s.panel, -1, "RX0 BC GPIO")
        s.RX0_BCgpioStgsSBSz = wx.StaticBoxSizer( s.RX0_BCgpioStgsSB, wx.VERTICAL)
        
        s.RX0_BCgpioStgsSBSz.Add( s.RX0_BCGPIOstgs, flag=wx.ALL, border=5)

        # RX Port 1
        # BC GPIO 0 Source drop-down
        (s.RX1_BCgpio0SourceSz, s.RX1_BCgpio0Source, s.RX1_BCgpio0SourceTxt) = \
            BuildComboBox( s.panel, "BC GPIO0:", choices=["GPIO PIN 0", "GPIO PIN 1", "GPIO PIN 2", "GPIO PIN 3", "GPIO PIN 4", "GPIO PIN 5", "GPIO PIN 6", "GPIO PIN 7", "Const=0", "Const=1", "FrameSync"], \
                labelOrientation=wx.LEFT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.RX1_BCgpio0Source.GetId(), s.RX1_BCgpio0SourceClick)
        s.RX1_BCgpio0Source.SetSelection((s.board.bc_gpio_ctl0[1]) & 0xF)

        # BC GPIO 1 Source drop-down
        (s.RX1_BCgpio1SourceSz, s.RX1_BCgpio1Source, s.RX1_BCgpio1SourceTxt) = \
            BuildComboBox( s.panel, "BC GPIO1:", choices=["GPIO PIN 0", "GPIO PIN 1", "GPIO PIN 2", "GPIO PIN 3", "GPIO PIN 4", "GPIO PIN 5", "GPIO PIN 6", "GPIO PIN 7", "Const=0", "Const=1", "FrameSync"], \
                labelOrientation=wx.LEFT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.RX1_BCgpio1Source.GetId(), s.RX1_BCgpio1SourceClick)
        s.RX1_BCgpio1Source.SetSelection((s.board.bc_gpio_ctl0[1] >> 4) & 0xF)

        # BC GPIO 2 Source drop-down
        (s.RX1_BCgpio2SourceSz, s.RX1_BCgpio2Source, s.RX1_BCgpio2SourceTxt) = \
            BuildComboBox( s.panel, "BC GPIO2:", choices=["GPIO PIN 0", "GPIO PIN 1", "GPIO PIN 2", "GPIO PIN 3", "GPIO PIN 4", "GPIO PIN 5", "GPIO PIN 6", "GPIO PIN 7", "Const=0", "Const=1", "FrameSync"], \
                labelOrientation=wx.LEFT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.RX1_BCgpio2Source.GetId(), s.RX1_BCgpio2SourceClick)
        s.RX1_BCgpio2Source.SetSelection((s.board.bc_gpio_ctl1[1]) & 0xF)

        # BC GPIO 3 Source drop-down
        (s.RX1_BCgpio3SourceSz, s.RX1_BCgpio3Source, s.RX1_BCgpio3SourceTxt) = \
            BuildComboBox( s.panel, "BC GPIO3:", choices=["GPIO PIN 0", "GPIO PIN 1", "GPIO PIN 2", "GPIO PIN 3", "GPIO PIN 4", "GPIO PIN 5", "GPIO PIN 6", "GPIO PIN 7", "Const=0", "Const=1", "FrameSync"], \
                labelOrientation=wx.LEFT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.RX1_BCgpio3Source.GetId(), s.RX1_BCgpio3SourceClick)
        s.RX1_BCgpio3Source.SetSelection((s.board.bc_gpio_ctl1[1] >> 4) & 0xF)


        # Configurations
        s.RX1_BCGPIOstgs = wx.GridBagSizer( 0, 0)
        s.RX1_BCGPIOstgs.Add( s.RX1_BCgpio0SourceSz, (0,0), border=5, flag=wx.ALL)
        s.RX1_BCGPIOstgs.Add( s.RX1_BCgpio1SourceSz, (1,0), border=5, flag=wx.ALL)
        s.RX1_BCGPIOstgs.Add( s.RX1_BCgpio2SourceSz, (2,0), border=5, flag=wx.ALL)
        s.RX1_BCGPIOstgs.Add( s.RX1_BCgpio3SourceSz, (3,0), border=5, flag=wx.ALL)

        # create static box
        s.RX1_BCgpioStgsSB = wx.StaticBox( s.panel, -1, "RX1 BC GPIO")
        s.RX1_BCgpioStgsSBSz = wx.StaticBoxSizer( s.RX1_BCgpioStgsSB, wx.VERTICAL)
        
        s.RX1_BCgpioStgsSBSz.Add( s.RX1_BCGPIOstgs, flag=wx.ALL, border=5)

        # RX Port 2
        # BC GPIO 0 Source drop-down
        (s.RX2_BCgpio0SourceSz, s.RX2_BCgpio0Source, s.RX2_BCgpio0SourceTxt) = \
            BuildComboBox( s.panel, "BC GPIO0:", choices=["GPIO PIN 0", "GPIO PIN 1", "GPIO PIN 2", "GPIO PIN 3", "GPIO PIN 4", "GPIO PIN 5", "GPIO PIN 6", "GPIO PIN 7", "Const=0", "Const=1", "FrameSync"], \
                labelOrientation=wx.LEFT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.RX2_BCgpio0Source.GetId(), s.RX2_BCgpio0SourceClick)
        s.RX2_BCgpio0Source.SetSelection((s.board.bc_gpio_ctl0[2]) & 0xF)

        # BC GPIO 1 Source drop-down
        (s.RX2_BCgpio1SourceSz, s.RX2_BCgpio1Source, s.RX2_BCgpio1SourceTxt) = \
            BuildComboBox( s.panel, "BC GPIO1:", choices=["GPIO PIN 0", "GPIO PIN 1", "GPIO PIN 2", "GPIO PIN 3", "GPIO PIN 4", "GPIO PIN 5", "GPIO PIN 6", "GPIO PIN 7", "Const=0", "Const=1", "FrameSync"], \
                labelOrientation=wx.LEFT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.RX2_BCgpio1Source.GetId(), s.RX2_BCgpio1SourceClick)
        s.RX2_BCgpio1Source.SetSelection((s.board.bc_gpio_ctl0[2] >> 4) & 0xF)

        # BC GPIO 2 Source drop-down
        (s.RX2_BCgpio2SourceSz, s.RX2_BCgpio2Source, s.RX2_BCgpio2SourceTxt) = \
            BuildComboBox( s.panel, "BC GPIO2:", choices=["GPIO PIN 0", "GPIO PIN 1", "GPIO PIN 2", "GPIO PIN 3", "GPIO PIN 4", "GPIO PIN 5", "GPIO PIN 6", "GPIO PIN 7", "Const=0", "Const=1", "FrameSync"], \
                labelOrientation=wx.LEFT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.RX2_BCgpio2Source.GetId(), s.RX2_BCgpio2SourceClick)
        s.RX2_BCgpio2Source.SetSelection((s.board.bc_gpio_ctl1[2]) & 0xF)

        # BC GPIO 3 Source drop-down
        (s.RX2_BCgpio3SourceSz, s.RX2_BCgpio3Source, s.RX2_BCgpio3SourceTxt) = \
            BuildComboBox( s.panel, "BC GPIO3:", choices=["GPIO PIN 0", "GPIO PIN 1", "GPIO PIN 2", "GPIO PIN 3", "GPIO PIN 4", "GPIO PIN 5", "GPIO PIN 6", "GPIO PIN 7", "Const=0", "Const=1", "FrameSync"], \
                labelOrientation=wx.LEFT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.RX2_BCgpio3Source.GetId(), s.RX2_BCgpio3SourceClick)
        s.RX2_BCgpio3Source.SetSelection((s.board.bc_gpio_ctl1[2] >> 4) & 0xF)


        # Configurations
        s.RX2_BCGPIOstgs = wx.GridBagSizer( 0, 0)
        s.RX2_BCGPIOstgs.Add( s.RX2_BCgpio0SourceSz, (0,0), border=5, flag=wx.ALL)
        s.RX2_BCGPIOstgs.Add( s.RX2_BCgpio1SourceSz, (1,0), border=5, flag=wx.ALL)
        s.RX2_BCGPIOstgs.Add( s.RX2_BCgpio2SourceSz, (2,0), border=5, flag=wx.ALL)
        s.RX2_BCGPIOstgs.Add( s.RX2_BCgpio3SourceSz, (3,0), border=5, flag=wx.ALL)

        # create static box
        s.RX2_BCgpioStgsSB = wx.StaticBox( s.panel, -1, "RX2 BC GPIO")
        s.RX2_BCgpioStgsSBSz = wx.StaticBoxSizer( s.RX2_BCgpioStgsSB, wx.VERTICAL)
        
        s.RX2_BCgpioStgsSBSz.Add( s.RX2_BCGPIOstgs, flag=wx.ALL, border=5)

        # RX Port 3
        # BC GPIO 0 Source drop-down
        (s.RX3_BCgpio0SourceSz, s.RX3_BCgpio0Source, s.RX3_BCgpio0SourceTxt) = \
            BuildComboBox( s.panel, "BC GPIO0:", choices=["GPIO PIN 0", "GPIO PIN 1", "GPIO PIN 2", "GPIO PIN 3", "GPIO PIN 4", "GPIO PIN 5", "GPIO PIN 6", "GPIO PIN 7", "Const=0", "Const=1", "FrameSync"], \
                labelOrientation=wx.LEFT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.RX3_BCgpio0Source.GetId(), s.RX3_BCgpio0SourceClick)
        s.RX3_BCgpio0Source.SetSelection((s.board.bc_gpio_ctl0[3]) & 0xF)

        # BC GPIO 1 Source drop-down
        (s.RX3_BCgpio1SourceSz, s.RX3_BCgpio1Source, s.RX3_BCgpio1SourceTxt) = \
            BuildComboBox( s.panel, "BC GPIO1:", choices=["GPIO PIN 0", "GPIO PIN 1", "GPIO PIN 2", "GPIO PIN 3", "GPIO PIN 4", "GPIO PIN 5", "GPIO PIN 6", "GPIO PIN 7", "Const=0", "Const=1", "FrameSync"], \
                labelOrientation=wx.LEFT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.RX3_BCgpio1Source.GetId(), s.RX3_BCgpio1SourceClick)
        s.RX3_BCgpio1Source.SetSelection((s.board.bc_gpio_ctl0[3] >> 4) & 0xF)

        # BC GPIO 2 Source drop-down
        (s.RX3_BCgpio2SourceSz, s.RX3_BCgpio2Source, s.RX3_BCgpio2SourceTxt) = \
            BuildComboBox( s.panel, "BC GPIO2:", choices=["GPIO PIN 0", "GPIO PIN 1", "GPIO PIN 2", "GPIO PIN 3", "GPIO PIN 4", "GPIO PIN 5", "GPIO PIN 6", "GPIO PIN 7", "Const=0", "Const=1", "FrameSync"], \
                labelOrientation=wx.LEFT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.RX3_BCgpio2Source.GetId(), s.RX3_BCgpio2SourceClick)
        s.RX3_BCgpio2Source.SetSelection((s.board.bc_gpio_ctl1[3]) & 0xF)

        # BC GPIO 3 Source drop-down
        (s.RX3_BCgpio3SourceSz, s.RX3_BCgpio3Source, s.RX3_BCgpio3SourceTxt) = \
            BuildComboBox( s.panel, "BC GPIO3:", choices=["GPIO PIN 0", "GPIO PIN 1", "GPIO PIN 2", "GPIO PIN 3", "GPIO PIN 4", "GPIO PIN 5", "GPIO PIN 6", "GPIO PIN 7", "Const=0", "Const=1", "FrameSync"], \
                labelOrientation=wx.LEFT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.RX3_BCgpio3Source.GetId(), s.RX3_BCgpio3SourceClick)
        s.RX3_BCgpio3Source.SetSelection((s.board.bc_gpio_ctl1[3] >> 4) & 0xF)


        # Configurations
        s.RX3_BCGPIOstgs = wx.GridBagSizer( 0, 0)
        s.RX3_BCGPIOstgs.Add( s.RX3_BCgpio0SourceSz, (0,0), border=5, flag=wx.ALL)
        s.RX3_BCGPIOstgs.Add( s.RX3_BCgpio1SourceSz, (1,0), border=5, flag=wx.ALL)
        s.RX3_BCGPIOstgs.Add( s.RX3_BCgpio2SourceSz, (2,0), border=5, flag=wx.ALL)
        s.RX3_BCGPIOstgs.Add( s.RX3_BCgpio3SourceSz, (3,0), border=5, flag=wx.ALL)

        # create static box
        s.RX3_BCgpioStgsSB = wx.StaticBox( s.panel, -1, "RX3 BC GPIO")
        s.RX3_BCgpioStgsSBSz = wx.StaticBoxSizer( s.RX3_BCgpioStgsSB, wx.VERTICAL)
        
        s.RX3_BCgpioStgsSBSz.Add( s.RX3_BCGPIOstgs, flag=wx.ALL, border=5)

        return
        
    def RX0_BCgpio0SourceClick(s, event):
        bc_gpio_sel = (s.RX0_BCgpio0Source.GetSelection())
        s.board.bc_gpio_ctl0[0] &= 0xF0
        s.board.bc_gpio_ctl0[0] |= (bc_gpio_sel)
        s.board.WriteRXPortReg(0, s.board.rx.BC_GPIO_CTL0, s.board.bc_gpio_ctl0[0])
        return

    def RX0_BCgpio1SourceClick(s, event):
        bc_gpio_sel = (s.RX0_BCgpio1Source.GetSelection())
        s.board.bc_gpio_ctl0[0] &= 0x0F
        s.board.bc_gpio_ctl0[0] |= (bc_gpio_sel << 4)
        s.board.WriteRXPortReg(0, s.board.rx.BC_GPIO_CTL0, s.board.bc_gpio_ctl0[0])
        return
        
    def RX0_BCgpio2SourceClick(s, event):
        bc_gpio_sel = (s.RX0_BCgpio2Source.GetSelection())
        s.board.bc_gpio_ctl1[0] &= 0xF0
        s.board.bc_gpio_ctl1[0] |= (bc_gpio_sel)
        s.board.WriteRXPortReg(0, s.board.rx.BC_GPIO_CTL1, s.board.bc_gpio_ctl1[0])
        return
        
    def RX0_BCgpio3SourceClick(s, event):
        bc_gpio_sel = (s.RX0_BCgpio3Source.GetSelection())
        s.board.bc_gpio_ctl1[0] &= 0x0F
        s.board.bc_gpio_ctl1[0] |= (bc_gpio_sel << 4)
        s.board.WriteRXPortReg(0, s.board.rx.BC_GPIO_CTL1, s.board.bc_gpio_ctl1[0])
        return

    def RX1_BCgpio0SourceClick(s, event):
        bc_gpio_sel = (s.RX1_BCgpio0Source.GetSelection())
        s.board.bc_gpio_ctl0[1] &= 0xF0
        s.board.bc_gpio_ctl0[1] |= (bc_gpio_sel)
        s.board.WriteRXPortReg(1, s.board.rx.BC_GPIO_CTL0, s.board.bc_gpio_ctl0[1])
        return

    def RX1_BCgpio1SourceClick(s, event):
        bc_gpio_sel = (s.RX1_BCgpio1Source.GetSelection())
        s.board.bc_gpio_ctl0[1] &= 0x0F
        s.board.bc_gpio_ctl0[1] |= (bc_gpio_sel << 4)
        s.board.WriteRXPortReg(1, s.board.rx.BC_GPIO_CTL0, s.board.bc_gpio_ctl0[1])
        return
        
    def RX1_BCgpio2SourceClick(s, event):
        bc_gpio_sel = (s.RX1_BCgpio2Source.GetSelection())
        s.board.bc_gpio_ctl1[1] &= 0xF0
        s.board.bc_gpio_ctl1[1] |= (bc_gpio_sel)
        s.board.WriteRXPortReg(1, s.board.rx.BC_GPIO_CTL1, s.board.bc_gpio_ctl1[1])
        return
        
    def RX1_BCgpio3SourceClick(s, event):
        bc_gpio_sel = (s.RX1_BCgpio3Source.GetSelection())
        s.board.bc_gpio_ctl1[1] &= 0x0F
        s.board.bc_gpio_ctl1[1] |= (bc_gpio_sel << 4)
        s.board.WriteRXPortReg(1, s.board.rx.BC_GPIO_CTL1, s.board.bc_gpio_ctl1[1])
        return

    def RX2_BCgpio0SourceClick(s, event):
        bc_gpio_sel = (s.RX2_BCgpio0Source.GetSelection())
        s.board.bc_gpio_ctl0[2] &= 0xF0
        s.board.bc_gpio_ctl0[2] |= (bc_gpio_sel)
        s.board.WriteRXPortReg(2, s.board.rx.BC_GPIO_CTL0, s.board.bc_gpio_ctl0[2])
        return

    def RX2_BCgpio1SourceClick(s, event):
        bc_gpio_sel = (s.RX2_BCgpio1Source.GetSelection())
        s.board.bc_gpio_ctl0[2] &= 0x0F
        s.board.bc_gpio_ctl0[2] |= (bc_gpio_sel << 4)
        s.board.WriteRXPortReg(2, s.board.rx.BC_GPIO_CTL0, s.board.bc_gpio_ctl0[2])
        return
        
    def RX2_BCgpio2SourceClick(s, event):
        bc_gpio_sel = (s.RX2_BCgpio2Source.GetSelection())
        s.board.bc_gpio_ctl1[2] &= 0xF0
        s.board.bc_gpio_ctl1[2] |= (bc_gpio_sel)
        s.board.WriteRXPortReg(2, s.board.rx.BC_GPIO_CTL1, s.board.bc_gpio_ctl1[2])
        return
        
    def RX2_BCgpio3SourceClick(s, event):
        bc_gpio_sel = (s.RX2_BCgpio3Source.GetSelection())
        s.board.bc_gpio_ctl1[2] &= 0x0F
        s.board.bc_gpio_ctl1[2] |= (bc_gpio_sel << 4)
        s.board.WriteRXPortReg(2, s.board.rx.BC_GPIO_CTL1, s.board.bc_gpio_ctl1[2])
        return

    def RX3_BCgpio0SourceClick(s, event):
        bc_gpio_sel = (s.RX3_BCgpio0Source.GetSelection())
        s.board.bc_gpio_ctl0[3] &= 0xF0
        s.board.bc_gpio_ctl0[3] |= (bc_gpio_sel)
        s.board.WriteRXPortReg(3, s.board.rx.BC_GPIO_CTL0, s.board.bc_gpio_ctl0[3])
        return

    def RX3_BCgpio1SourceClick(s, event):
        bc_gpio_sel = (s.RX3_BCgpio1Source.GetSelection())
        s.board.bc_gpio_ctl0[3] &= 0x0F
        s.board.bc_gpio_ctl0[3] |= (bc_gpio_sel << 4)
        s.board.WriteRXPortReg(3, s.board.rx.BC_GPIO_CTL0, s.board.bc_gpio_ctl0[3])
        return
        
    def RX3_BCgpio2SourceClick(s, event):
        bc_gpio_sel = (s.RX3_BCgpio2Source.GetSelection())
        s.board.bc_gpio_ctl1[3] &= 0xF0
        s.board.bc_gpio_ctl1[3] |= (bc_gpio_sel)
        s.board.WriteRXPortReg(3, s.board.rx.BC_GPIO_CTL1, s.board.bc_gpio_ctl1[3])
        return
        
    def RX3_BCgpio3SourceClick(s, event):
        bc_gpio_sel = (s.RX3_BCgpio3Source.GetSelection())
        s.board.bc_gpio_ctl1[3] &= 0x0F
        s.board.bc_gpio_ctl1[3] |= (bc_gpio_sel << 4)
        s.board.WriteRXPortReg(3, s.board.rx.BC_GPIO_CTL1, s.board.bc_gpio_ctl1[3])
        return

    def BuildGPIOBox(s):
        # Get Current pin control values
        s.board.gpio0_pin_ctl = s.board.ReadRXReg(s.board.rx.GPIO0_PIN_CTL)
        s.board.gpio1_pin_ctl = s.board.ReadRXReg(s.board.rx.GPIO1_PIN_CTL)
        s.board.gpio2_pin_ctl = s.board.ReadRXReg(s.board.rx.GPIO2_PIN_CTL)
        s.board.gpio3_pin_ctl = s.board.ReadRXReg(s.board.rx.GPIO3_PIN_CTL)
        s.board.gpio4_pin_ctl = s.board.ReadRXReg(s.board.rx.GPIO4_PIN_CTL)
        s.board.gpio5_pin_ctl = s.board.ReadRXReg(s.board.rx.GPIO5_PIN_CTL)
        s.board.gpio6_pin_ctl = s.board.ReadRXReg(s.board.rx.GPIO6_PIN_CTL)
        s.board.gpio7_pin_ctl = s.board.ReadRXReg(s.board.rx.GPIO7_PIN_CTL)

        s.rxport_txt = ["RX GPIO 0", "RX GPIO 1", "RX GPIO 2", "RX GPIO 3", "RX Lock", "RX PASS", "FrameValid", "LineValid"]
        s.Devsts_txt = ["Output Val", "OR-Lock", "AND-Lock", "AND-Pass", "FrameSync", "", "", ""]
        s.txport_txt = ["Pass (AND)", "Pass (OR)", "FrameValid", "LineValid", "Sync-ed", "TX Interrupt", "", ""]
        
        # GPIO Pin 0
        # Output enable check box
        s.gpio0EnableChk = wx.CheckBox( s.panel, wx.NewId(), "GPIO 0")
        wx.EVT_CHECKBOX( s.gpio0EnableChk, s.gpio0EnableChk.GetId(), s.onGPIO0Click)
        if (s.board.gpio0_pin_ctl & s.board.rx.GPIO_OUT_EN):
            s.gpio0EnableChk.SetValue(True)
  
        # Output Source drop-down
        (s.gpio0SourceSz, s.gpio0Source, s.gpio0SourceTxt) = \
            BuildComboBox( s.panel, "", choices=["RX Port 0", "RX Port 1", "RX Port 2", "RX Port 3", "Device Sts", "Reserved", "CSI TX0", "CSI TX1"], \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio0Source.GetId(), s.gpio0SourceClick)
        s.gpio0Source.SetSelection((s.board.gpio0_pin_ctl >> 2) & 0x7)

        # Output RX Port Select drop-down
        (s.gpio0PortSz, s.gpio0Port, s.gpio0PortTxt) = \
            BuildComboBox( s.panel, "", choices=s.rxport_txt, \
                labelOrientation=wx.RIGHT, enable=False, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio0Port.GetId(), s.gpio0PortClick)
        s.gpio0Port.SetSelection((s.board.gpio0_pin_ctl >> 5) & 0x7)
        
        # Output Port Select drop-down
        (s.gpio0DevstsSz, s.gpio0Devsts, s.gpio0DevstsTxt) = \
            BuildComboBox( s.panel, "", choices=s.Devsts_txt, \
                labelOrientation=wx.RIGHT, enable=False, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio0Devsts.GetId(), s.gpio0DevstsClick)
        s.gpio0Devsts.SetSelection((s.board.gpio0_pin_ctl >> 5) & 0x7)

        # Output TX Port Select drop-down
        (s.gpio0TXPortSz, s.gpio0TXPort, s.gpio0TXPortTxt) = \
            BuildComboBox( s.panel, "", choices=s.txport_txt, \
                labelOrientation=wx.RIGHT, enable=False, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio0TXPort.GetId(), s.gpio0TXPortClick)
        s.gpio0TXPort.SetSelection((s.board.gpio0_pin_ctl >> 5) & 0x7)
            
        # Output Value drop-down
        (s.gpio0OutValSz, s.gpio0OutVal, s.gpio0OutValTxt) = \
            BuildComboBox( s.panel, "", choices=["0", "1"], \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio0OutVal.GetId(), s.gpio0OutValClick)
        s.gpio0OutVal.SetSelection((s.board.gpio0_pin_ctl >> 1) & 0x1)

        # Enable one of Port or DevSts option drop-downs
        if ((s.board.gpio0_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_DEVICE_STS):
            s.gpio0Devsts.Enable()
        elif (((s.board.gpio0_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0) \
                or ((s.board.gpio0_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0)):
            s.gpio0TXPort.Enable()
        else:
            s.gpio0Port.Enable()

        # GPIO Pin 1
        # Output enable check box
        s.gpio1EnableChk = wx.CheckBox( s.panel, wx.NewId(), "GPIO 1")
        wx.EVT_CHECKBOX( s.gpio1EnableChk, s.gpio1EnableChk.GetId(), s.onGPIO1Click)
        if (s.board.gpio1_pin_ctl & s.board.rx.GPIO_OUT_EN):
            s.gpio1EnableChk.SetValue(True)
  
        # Output Source drop-down
        (s.gpio1SourceSz, s.gpio1Source, s.gpio1SourceTxt) = \
            BuildComboBox( s.panel, "", choices=["RX Port 0", "RX Port 1", "RX Port 2", "RX Port 3", "Device Sts", "Reserved", "CSI TX0", "CSI TX1"], \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio1Source.GetId(), s.gpio1SourceClick)
        s.gpio1Source.SetSelection((s.board.gpio1_pin_ctl >> 2) & 0x7)

        # Output Port Select drop-down
        (s.gpio1PortSz, s.gpio1Port, s.gpio1PortTxt) = \
            BuildComboBox( s.panel, "", choices=s.rxport_txt, \
                labelOrientation=wx.RIGHT, enable=False, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio1Port.GetId(), s.gpio1PortClick)
        s.gpio1Port.SetSelection((s.board.gpio1_pin_ctl >> 5) & 0x7)
            
        # Output Port Select drop-down
        (s.gpio1DevstsSz, s.gpio1Devsts, s.gpio1DevstsTxt) = \
            BuildComboBox( s.panel, "", choices=s.Devsts_txt, \
                labelOrientation=wx.RIGHT, enable=False, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio1Devsts.GetId(), s.gpio1DevstsClick)
        s.gpio1Devsts.SetSelection((s.board.gpio1_pin_ctl >> 5) & 0x7)

        # Output TX Port Select drop-down
        (s.gpio1TXPortSz, s.gpio1TXPort, s.gpio1TXPortTxt) = \
            BuildComboBox( s.panel, "", choices=s.txport_txt, \
                labelOrientation=wx.RIGHT, enable=False, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio1TXPort.GetId(), s.gpio1TXPortClick)
        s.gpio1TXPort.SetSelection((s.board.gpio1_pin_ctl >> 5) & 0x7)
            
        # Output Value drop-down
        (s.gpio1OutValSz, s.gpio1OutVal, s.gpio1OutValTxt) = \
            BuildComboBox( s.panel, "", choices=["0", "1"], \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio1OutVal.GetId(), s.gpio1OutValClick)
        s.gpio1OutVal.SetSelection((s.board.gpio1_pin_ctl >> 1) & 0x1)

        # Enable one of Port or DevSts option drop-downs
        if ((s.board.gpio1_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_DEVICE_STS):
            s.gpio1Devsts.Enable()
        elif (((s.board.gpio1_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0) \
                or ((s.board.gpio1_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0)):
            s.gpio1TXPort.Enable()
        else:
            s.gpio1Port.Enable()

        # GPIO Pin 2
        # Output enable check box
        s.gpio2EnableChk = wx.CheckBox( s.panel, wx.NewId(), "GPIO 2")
        wx.EVT_CHECKBOX( s.gpio2EnableChk, s.gpio2EnableChk.GetId(), s.onGPIO2Click)
        if (s.board.gpio2_pin_ctl & s.board.rx.GPIO_OUT_EN):
            s.gpio2EnableChk.SetValue(True)
  
        # Output Source drop-down
        (s.gpio2SourceSz, s.gpio2Source, s.gpio2SourceTxt) = \
            BuildComboBox( s.panel, "", choices=["RX Port 0", "RX Port 1", "RX Port 2", "RX Port 3", "Device Sts", "Reserved", "CSI TX0", "CSI TX1"], \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio2Source.GetId(), s.gpio2SourceClick)
        s.gpio2Source.SetSelection((s.board.gpio2_pin_ctl >> 2) & 0x7)

        # Output Port Select drop-down
        (s.gpio2PortSz, s.gpio2Port, s.gpio2PortTxt) = \
            BuildComboBox( s.panel, "", choices=s.rxport_txt, \
                labelOrientation=wx.RIGHT, enable=False, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio2Port.GetId(), s.gpio2PortClick)
        s.gpio2Port.SetSelection((s.board.gpio2_pin_ctl >> 5) & 0x7)
            
        # Output Port Select drop-down
        (s.gpio2DevstsSz, s.gpio2Devsts, s.gpio2DevstsTxt) = \
            BuildComboBox( s.panel, "", choices=s.Devsts_txt, \
                labelOrientation=wx.RIGHT, enable=False, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio2Devsts.GetId(), s.gpio2DevstsClick)
        s.gpio2Devsts.SetSelection((s.board.gpio2_pin_ctl >> 5) & 0x7)

        # Output TX Port Select drop-down
        (s.gpio2TXPortSz, s.gpio2TXPort, s.gpio2TXPortTxt) = \
            BuildComboBox( s.panel, "", choices=s.txport_txt, \
                labelOrientation=wx.RIGHT, enable=False, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio2TXPort.GetId(), s.gpio2TXPortClick)
        s.gpio2TXPort.SetSelection((s.board.gpio2_pin_ctl >> 5) & 0x7)
            
        # Output Value drop-down
        (s.gpio2OutValSz, s.gpio2OutVal, s.gpio2OutValTxt) = \
            BuildComboBox( s.panel, "", choices=["0", "1"], \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio2OutVal.GetId(), s.gpio2OutValClick)
        s.gpio2OutVal.SetSelection((s.board.gpio2_pin_ctl >> 1) & 0x1)

        # Enable one of Port or DevSts option drop-downs
        if ((s.board.gpio2_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_DEVICE_STS):
            s.gpio2Devsts.Enable()
        elif (((s.board.gpio2_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0) \
                or ((s.board.gpio2_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0)):
            s.gpio2TXPort.Enable()
        else:
            s.gpio2Port.Enable()

        # GPIO Pin 3
        # Output enable check box
        s.gpio3EnableChk = wx.CheckBox( s.panel, wx.NewId(), "GPIO 3")
        wx.EVT_CHECKBOX( s.gpio3EnableChk, s.gpio3EnableChk.GetId(), s.onGPIO3Click)
        if (s.board.gpio3_pin_ctl & s.board.rx.GPIO_OUT_EN):
            s.gpio3EnableChk.SetValue(True)
  
        # Output Source drop-down
        (s.gpio3SourceSz, s.gpio3Source, s.gpio3SourceTxt) = \
            BuildComboBox( s.panel, "", choices=["RX Port 0", "RX Port 1", "RX Port 2", "RX Port 3", "Device Sts", "Reserved", "CSI TX0", "CSI TX1"], \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio3Source.GetId(), s.gpio3SourceClick)
        s.gpio3Source.SetSelection((s.board.gpio3_pin_ctl >> 2) & 0x7)

        # Output Port Select drop-down
        (s.gpio3PortSz, s.gpio3Port, s.gpio3PortTxt) = \
            BuildComboBox( s.panel, "", choices=s.rxport_txt, \
                labelOrientation=wx.RIGHT, enable=False, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio3Port.GetId(), s.gpio3PortClick)
        s.gpio3Port.SetSelection((s.board.gpio3_pin_ctl >> 5) & 0x7)
            
        # Output Port Select drop-down
        (s.gpio3DevstsSz, s.gpio3Devsts, s.gpio3DevstsTxt) = \
            BuildComboBox( s.panel, "", choices=s.Devsts_txt, \
                labelOrientation=wx.RIGHT, enable=False, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio3Devsts.GetId(), s.gpio3DevstsClick)
        s.gpio3Devsts.SetSelection((s.board.gpio3_pin_ctl >> 5) & 0x7)

        # Output TX Port Select drop-down
        (s.gpio3TXPortSz, s.gpio3TXPort, s.gpio3TXPortTxt) = \
            BuildComboBox( s.panel, "", choices=s.txport_txt, \
                labelOrientation=wx.RIGHT, enable=False, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio3TXPort.GetId(), s.gpio3TXPortClick)
        s.gpio3TXPort.SetSelection((s.board.gpio3_pin_ctl >> 5) & 0x7)
            
        # Output Value drop-down
        (s.gpio3OutValSz, s.gpio3OutVal, s.gpio3OutValTxt) = \
            BuildComboBox( s.panel, "", choices=["0", "1"], \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio3OutVal.GetId(), s.gpio3OutValClick)
        s.gpio3OutVal.SetSelection((s.board.gpio3_pin_ctl >> 1) & 0x1)

        # Enable one of Port or DevSts option drop-downs
        if ((s.board.gpio3_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_DEVICE_STS):
            s.gpio3Devsts.Enable()
        elif (((s.board.gpio3_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0) \
                or ((s.board.gpio3_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0)):
            s.gpio3TXPort.Enable()
        else:
            s.gpio3Port.Enable()

        # GPIO Pin 4
        # Output enable check box
        s.gpio4EnableChk = wx.CheckBox( s.panel, wx.NewId(), "GPIO 4")
        wx.EVT_CHECKBOX( s.gpio4EnableChk, s.gpio4EnableChk.GetId(), s.onGPIO4Click)
        if (s.board.gpio4_pin_ctl & s.board.rx.GPIO_OUT_EN):
            s.gpio4EnableChk.SetValue(True)
  
        # Output Source drop-down
        (s.gpio4SourceSz, s.gpio4Source, s.gpio4SourceTxt) = \
            BuildComboBox( s.panel, "", choices=["RX Port 0", "RX Port 1", "RX Port 2", "RX Port 3", "Device Sts", "Reserved", "CSI TX0", "CSI TX1"], \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio4Source.GetId(), s.gpio4SourceClick)
        s.gpio4Source.SetSelection((s.board.gpio4_pin_ctl >> 2) & 0x7)

        # Output Port Select drop-down
        (s.gpio4PortSz, s.gpio4Port, s.gpio4PortTxt) = \
            BuildComboBox( s.panel, "", choices=s.rxport_txt, \
                labelOrientation=wx.RIGHT, enable=False, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio4Port.GetId(), s.gpio4PortClick)
        s.gpio4Port.SetSelection((s.board.gpio4_pin_ctl >> 5) & 0x7)
            
        # Output Port Select drop-down
        (s.gpio4DevstsSz, s.gpio4Devsts, s.gpio4DevstsTxt) = \
            BuildComboBox( s.panel, "", choices=s.Devsts_txt, \
                labelOrientation=wx.RIGHT, enable=False, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio4Devsts.GetId(), s.gpio4DevstsClick)
        s.gpio4Devsts.SetSelection((s.board.gpio4_pin_ctl >> 5) & 0x7)

        # Output TX Port Select drop-down
        (s.gpio4TXPortSz, s.gpio4TXPort, s.gpio4TXPortTxt) = \
            BuildComboBox( s.panel, "", choices=s.txport_txt, \
                labelOrientation=wx.RIGHT, enable=False, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio4TXPort.GetId(), s.gpio4TXPortClick)
        s.gpio4TXPort.SetSelection((s.board.gpio4_pin_ctl >> 5) & 0x7)
            
        # Output Value drop-down
        (s.gpio4OutValSz, s.gpio4OutVal, s.gpio4OutValTxt) = \
            BuildComboBox( s.panel, "", choices=["0", "1"], \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio4OutVal.GetId(), s.gpio4OutValClick)
        s.gpio4OutVal.SetSelection((s.board.gpio4_pin_ctl >> 1) & 0x1)

        # Enable one of Port or DevSts option drop-downs
        if ((s.board.gpio4_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_DEVICE_STS):
            s.gpio4Devsts.Enable()
        elif (((s.board.gpio4_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0) \
                or ((s.board.gpio4_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0)):
            s.gpio4TXPort.Enable()
        else:
            s.gpio4Port.Enable()

        # GPIO Pin 5
        # Output enable check box
        s.gpio5EnableChk = wx.CheckBox( s.panel, wx.NewId(), "GPIO 5")
        wx.EVT_CHECKBOX( s.gpio5EnableChk, s.gpio5EnableChk.GetId(), s.onGPIO5Click)
        if (s.board.gpio5_pin_ctl & s.board.rx.GPIO_OUT_EN):
            s.gpio5EnableChk.SetValue(True)
  
        # Output Source drop-down
        (s.gpio5SourceSz, s.gpio5Source, s.gpio5SourceTxt) = \
            BuildComboBox( s.panel, "", choices=["RX Port 0", "RX Port 1", "RX Port 2", "RX Port 3", "Device Sts", "Reserved", "CSI TX0", "CSI TX1"], \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio5Source.GetId(), s.gpio5SourceClick)
        s.gpio5Source.SetSelection((s.board.gpio5_pin_ctl >> 2) & 0x7)

        # Output Port Select drop-down
        (s.gpio5PortSz, s.gpio5Port, s.gpio5PortTxt) = \
            BuildComboBox( s.panel, "", choices=s.rxport_txt, \
                labelOrientation=wx.RIGHT, enable=False, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio5Port.GetId(), s.gpio5PortClick)
        s.gpio5Port.SetSelection((s.board.gpio5_pin_ctl >> 5) & 0x7)
            
        # Output Port Select drop-down
        (s.gpio5DevstsSz, s.gpio5Devsts, s.gpio5DevstsTxt) = \
            BuildComboBox( s.panel, "", choices=s.Devsts_txt, \
                labelOrientation=wx.RIGHT, enable=False, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio5Devsts.GetId(), s.gpio5DevstsClick)
        s.gpio5Devsts.SetSelection((s.board.gpio5_pin_ctl >> 5) & 0x7)

        # Output TX Port Select drop-down
        (s.gpio5TXPortSz, s.gpio5TXPort, s.gpio5TXPortTxt) = \
            BuildComboBox( s.panel, "", choices=s.txport_txt, \
                labelOrientation=wx.RIGHT, enable=False, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio5TXPort.GetId(), s.gpio5TXPortClick)
        s.gpio5TXPort.SetSelection((s.board.gpio5_pin_ctl >> 5) & 0x7)
            
        # Output Value drop-down
        (s.gpio5OutValSz, s.gpio5OutVal, s.gpio5OutValTxt) = \
            BuildComboBox( s.panel, "", choices=["0", "1"], \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio5OutVal.GetId(), s.gpio5OutValClick)
        s.gpio5OutVal.SetSelection((s.board.gpio5_pin_ctl >> 1) & 0x1)

        # Enable one of Port or DevSts option drop-downs
        if ((s.board.gpio5_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_DEVICE_STS):
            s.gpio5Devsts.Enable()
        elif (((s.board.gpio5_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0) \
                or ((s.board.gpio5_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0)):
            s.gpio5TXPort.Enable()
        else:
            s.gpio5Port.Enable()

        # GPIO Pin 6
        # Output enable check box
        s.gpio6EnableChk = wx.CheckBox( s.panel, wx.NewId(), "GPIO 6")
        wx.EVT_CHECKBOX( s.gpio6EnableChk, s.gpio6EnableChk.GetId(), s.onGPIO6Click)
        if (s.board.gpio6_pin_ctl & s.board.rx.GPIO_OUT_EN):
            s.gpio6EnableChk.SetValue(True)
  
        # Output Source drop-down
        (s.gpio6SourceSz, s.gpio6Source, s.gpio6SourceTxt) = \
            BuildComboBox( s.panel, "", choices=["RX Port 0", "RX Port 1", "RX Port 2", "RX Port 3", "Device Sts", "Reserved", "CSI TX0", "CSI TX1"], \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio6Source.GetId(), s.gpio6SourceClick)
        s.gpio6Source.SetSelection((s.board.gpio6_pin_ctl >> 2) & 0x7)

        # Output Port Select drop-down
        (s.gpio6PortSz, s.gpio6Port, s.gpio6PortTxt) = \
            BuildComboBox( s.panel, "", choices=s.rxport_txt, \
                labelOrientation=wx.RIGHT, enable=False, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio6Port.GetId(), s.gpio6PortClick)
        s.gpio6Port.SetSelection((s.board.gpio6_pin_ctl >> 5) & 0x7)
            
        # Output Port Select drop-down
        (s.gpio6DevstsSz, s.gpio6Devsts, s.gpio6DevstsTxt) = \
            BuildComboBox( s.panel, "", choices=s.Devsts_txt, \
                labelOrientation=wx.RIGHT, enable=False, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio6Devsts.GetId(), s.gpio6DevstsClick)
        s.gpio6Devsts.SetSelection((s.board.gpio6_pin_ctl >> 5) & 0x7)

        # Output TX Port Select drop-down
        (s.gpio6TXPortSz, s.gpio6TXPort, s.gpio6TXPortTxt) = \
            BuildComboBox( s.panel, "", choices=s.txport_txt, \
                labelOrientation=wx.RIGHT, enable=False, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio6TXPort.GetId(), s.gpio6TXPortClick)
        s.gpio6TXPort.SetSelection((s.board.gpio6_pin_ctl >> 5) & 0x7)
            
        # Output Value drop-down
        (s.gpio6OutValSz, s.gpio6OutVal, s.gpio6OutValTxt) = \
            BuildComboBox( s.panel, "", choices=["0", "1"], \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio6OutVal.GetId(), s.gpio6OutValClick)
        s.gpio6OutVal.SetSelection((s.board.gpio6_pin_ctl >> 1) & 0x1)

        # Enable one of Port or DevSts option drop-downs
        if ((s.board.gpio6_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_DEVICE_STS):
            s.gpio6Devsts.Enable()
        elif (((s.board.gpio6_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0) \
                or ((s.board.gpio6_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0)):
            s.gpio6TXPort.Enable()
        else:
            s.gpio6Port.Enable()

        # GPIO Pin 7
        # Output enable check box
        s.gpio7EnableChk = wx.CheckBox( s.panel, wx.NewId(), "GPIO 7")
        wx.EVT_CHECKBOX( s.gpio7EnableChk, s.gpio7EnableChk.GetId(), s.onGPIO7Click)
        if (s.board.gpio7_pin_ctl & s.board.rx.GPIO_OUT_EN):
            s.gpio7EnableChk.SetValue(True)
  
        # Output Source drop-down
        (s.gpio7SourceSz, s.gpio7Source, s.gpio7SourceTxt) = \
            BuildComboBox( s.panel, "", choices=["RX Port 0", "RX Port 1", "RX Port 2", "RX Port 3", "Device Sts", "Reserved", "CSI TX0", "CSI TX1"], \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio7Source.GetId(), s.gpio7SourceClick)
        s.gpio7Source.SetSelection((s.board.gpio7_pin_ctl >> 2) & 0x7)

        # Output Port Select drop-down
        (s.gpio7PortSz, s.gpio7Port, s.gpio7PortTxt) = \
            BuildComboBox( s.panel, "", choices=s.rxport_txt, \
                labelOrientation=wx.RIGHT, enable=False, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio7Port.GetId(), s.gpio7PortClick)
        s.gpio7Port.SetSelection((s.board.gpio7_pin_ctl >> 5) & 0x7)
            
        # Output Port Select drop-down
        (s.gpio7DevstsSz, s.gpio7Devsts, s.gpio7DevstsTxt) = \
            BuildComboBox( s.panel, "", choices=s.Devsts_txt, \
                labelOrientation=wx.RIGHT, enable=False, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio7Devsts.GetId(), s.gpio7DevstsClick)
        s.gpio7Devsts.SetSelection((s.board.gpio7_pin_ctl >> 5) & 0x7)

        # Output TX Port Select drop-down
        (s.gpio7TXPortSz, s.gpio7TXPort, s.gpio7TXPortTxt) = \
            BuildComboBox( s.panel, "", choices=s.txport_txt, \
                labelOrientation=wx.RIGHT, enable=False, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio7TXPort.GetId(), s.gpio7TXPortClick)
        s.gpio7TXPort.SetSelection((s.board.gpio7_pin_ctl >> 5) & 0x7)
            
        # Output Value drop-down
        (s.gpio7OutValSz, s.gpio7OutVal, s.gpio7OutValTxt) = \
            BuildComboBox( s.panel, "", choices=["0", "1"], \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.gpio7OutVal.GetId(), s.gpio7OutValClick)
        s.gpio7OutVal.SetSelection((s.board.gpio7_pin_ctl >> 1) & 0x1)

        # Enable one of Port or DevSts option drop-downs
        if ((s.board.gpio7_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_DEVICE_STS):
            s.gpio7Devsts.Enable()
        elif (((s.board.gpio7_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0) \
                or ((s.board.gpio7_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0)):
            s.gpio7TXPort.Enable()
        else:
            s.gpio7Port.Enable()

        # Left side text
        txt0 = WriteText( s.panel, "Output Enable", None, False)
        txt1 = WriteText( s.panel, "Source Option", None, False)
        txt2 = WriteText( s.panel, "RX Port Option", None, False)
        txt3 = WriteText( s.panel, "Status Option", None, False)
        txt4 = WriteText( s.panel, "TX Port Option", None, False)
        txt5 = WriteText( s.panel, "Output Value", None, False)
            
        # Configurations
        s.GPIOstgs = wx.GridBagSizer( 0, 0)
        s.GPIOstgs.Add( txt0, (0,0), border=5, flag=wx.TOP|wx.BOTTOM|wx.RIGHT)
        s.GPIOstgs.Add( txt1, (1,0), border=5, flag=wx.TOP|wx.BOTTOM|wx.RIGHT)
        s.GPIOstgs.Add( txt2, (2,0), border=5, flag=wx.TOP|wx.BOTTOM|wx.RIGHT)
        s.GPIOstgs.Add( txt3, (3,0), border=5, flag=wx.TOP|wx.BOTTOM|wx.RIGHT)
        s.GPIOstgs.Add( txt4, (4,0), border=5, flag=wx.TOP|wx.BOTTOM|wx.RIGHT)
        s.GPIOstgs.Add( txt5, (5,0), border=5, flag=wx.TOP|wx.BOTTOM|wx.RIGHT)
        s.GPIOstgs.Add( s.gpio0EnableChk, (0,1), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio0SourceSz, (1,1), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio0PortSz, (2,1), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio0DevstsSz, (3,1), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio0TXPortSz, (4,1), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio0OutValSz, (5,1), border=5, flag=wx.TOP|wx.BOTTOM)
        
        s.GPIOstgs.Add( s.gpio1EnableChk, (0,2), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio1SourceSz, (1,2), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio1PortSz, (2,2), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio1DevstsSz, (3,2), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio1TXPortSz, (4,2), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio1OutValSz, (5,2), border=5, flag=wx.TOP|wx.BOTTOM)
        
        s.GPIOstgs.Add( s.gpio2EnableChk, (0,3), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio2SourceSz, (1,3), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio2PortSz, (2,3), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio2DevstsSz, (3,3), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio2TXPortSz, (4,3), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio2OutValSz, (5,3), border=5, flag=wx.TOP|wx.BOTTOM)
        
        s.GPIOstgs.Add( s.gpio3EnableChk, (0,4), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio3SourceSz, (1,4), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio3PortSz, (2,4), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio3DevstsSz, (3,4), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio3TXPortSz, (4,4), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio3OutValSz, (5,4), border=5, flag=wx.TOP|wx.BOTTOM)
        
        s.GPIOstgs.Add( s.gpio4EnableChk, (0,5), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio4SourceSz, (1,5), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio4PortSz, (2,5), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio4DevstsSz, (3,5), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio4TXPortSz, (4,5), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio4OutValSz, (5,5), border=5, flag=wx.TOP|wx.BOTTOM)
        
        s.GPIOstgs.Add( s.gpio5EnableChk, (0,6), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio5SourceSz, (1,6), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio5PortSz, (2,6), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio5DevstsSz, (3,6), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio5TXPortSz, (4,6), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio5OutValSz, (5,6), border=5, flag=wx.TOP|wx.BOTTOM)
        
        s.GPIOstgs.Add( s.gpio6EnableChk, (0,7), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio6SourceSz, (1,7), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio6PortSz, (2,7), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio6DevstsSz, (3,7), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio6TXPortSz, (4,7), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio6OutValSz, (5,7), border=5, flag=wx.TOP|wx.BOTTOM)
        
        s.GPIOstgs.Add( s.gpio7EnableChk, (0,8), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio7SourceSz, (1,8), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio7PortSz, (2,8), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio7DevstsSz, (3,8), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio7TXPortSz, (4,8), border=5, flag=wx.TOP|wx.BOTTOM)
        s.GPIOstgs.Add( s.gpio7OutValSz, (5,8), border=5, flag=wx.TOP|wx.BOTTOM)
        
        # create static box
        s.gpioStgsSB = wx.StaticBox( s.panel, -1, "GPIO Pin Control")
        s.gpioStgsSBSz = wx.StaticBoxSizer( s.gpioStgsSB, wx.VERTICAL)
        
        s.gpioStgsSBSz.Add( s.GPIOstgs, flag=wx.ALL, border=5)

        return
        
    def onGPIO0Click(s, event):
        isChecked = s.gpio0EnableChk.IsChecked()
        if (isChecked):
            s.board.gpio0_pin_ctl |= s.board.rx.GPIO_OUT_EN
        else:
            s.board.gpio0_pin_ctl &= 0xFE
        s.board.WriteRXReg(s.board.rx.GPIO0_PIN_CTL, s.board.gpio0_pin_ctl)
        return
            
    def gpio0SourceClick(s, event):
        gpio0_source = (s.gpio0Source.GetSelection())
        s.board.gpio0_pin_ctl &= 0xE3
        s.board.gpio0_pin_ctl |= (gpio0_source << 2)
        s.board.WriteRXReg(s.board.rx.GPIO0_PIN_CTL, s.board.gpio0_pin_ctl)
        if (gpio0_source == 5):
            # Disable all options
            s.gpio0Port.Disable()
            s.gpio0TXPort.Disable()
            s.gpio0Devsts.Disable()
        elif (gpio0_source == 4):
            # Disable other options
            s.gpio0Port.Disable()
            s.gpio0TXPort.Disable()
            # Enable Device Status options
            s.gpio0Devsts.Enable()
        elif ((gpio0_source == 6) or (gpio0_source == 7)):
            # Disable other options
            s.gpio0Port.Disable()
            s.gpio0Devsts.Disable()
            # Enable TX Port options
            s.gpio0TXPort.Enable()
        else:
            # Enable Port Source options
            s.gpio0Port.Enable()
            # Disable other options
            s.gpio0TXPort.Disable()
            s.gpio0Devsts.Disable()
        #s.UpdateGPIOTab()
        return
        
    def gpio0PortClick(s, event):
        gpio0_port_option = (s.gpio0Port.GetSelection())
        s.gpio0Devsts.SetSelection(gpio0_port_option)
        s.gpio0TXPort.SetSelection(gpio0_port_option)
        s.board.gpio0_pin_ctl &= 0x1F
        s.board.gpio0_pin_ctl |= (gpio0_port_option << 5)
        s.board.WriteRXReg(s.board.rx.GPIO0_PIN_CTL, s.board.gpio0_pin_ctl)
        return
        
        
    def gpio0DevstsClick(s, event):
        gpio0_devsts_option = (s.gpio0Devsts.GetSelection())
        s.gpio0Port.SetSelection(gpio0_devsts_option)
        s.gpio0TXPort.SetSelection(gpio0_devsts_option)
        s.board.gpio0_pin_ctl &= 0x1F
        s.board.gpio0_pin_ctl |= (gpio0_devsts_option << 5)
        s.board.WriteRXReg(s.board.rx.GPIO0_PIN_CTL, s.board.gpio0_pin_ctl)
        return
                
    def gpio0TXPortClick(s, event):
        gpio0_txport_option = (s.gpio0TXPort.GetSelection())
        s.gpio0Devsts.SetSelection(gpio0_txport_option)
        s.gpio0Port.SetSelection(gpio0_txport_option)
        s.board.gpio0_pin_ctl &= 0x1F
        s.board.gpio0_pin_ctl |= (gpio0_txport_option << 5)
        s.board.WriteRXReg(s.board.rx.GPIO0_PIN_CTL, s.board.gpio0_pin_ctl)
        return
        
    def gpio0OutValClick(s, event):
        gpio0_outval = (s.gpio0OutVal.GetSelection()) & 0x1
        s.board.gpio0_pin_ctl &= 0xFD
        s.board.gpio0_pin_ctl |= (gpio0_outval << 1)
        s.board.WriteRXReg(s.board.rx.GPIO0_PIN_CTL, s.board.gpio0_pin_ctl)
        return
        
    def onGPIO1Click(s, event):
        isChecked = s.gpio1EnableChk.IsChecked()
        if (isChecked):
            s.board.gpio1_pin_ctl |= s.board.rx.GPIO_OUT_EN
        else:
            s.board.gpio1_pin_ctl &= 0xFE
        s.board.WriteRXReg(s.board.rx.GPIO1_PIN_CTL, s.board.gpio1_pin_ctl)
        return
            
    def gpio1SourceClick(s, event):
        gpio1_source = (s.gpio1Source.GetSelection())
        s.board.gpio1_pin_ctl &= 0xE3
        s.board.gpio1_pin_ctl |= (gpio1_source << 2)
        s.board.WriteRXReg(s.board.rx.GPIO1_PIN_CTL, s.board.gpio1_pin_ctl)
        if (gpio1_source == 5):
            # Disable all options
            s.gpio1Port.Disable()
            s.gpio1TXPort.Disable()
            s.gpio1Devsts.Disable()
        elif (gpio1_source == 4):
            # Disable other options
            s.gpio1Port.Disable()
            s.gpio1TXPort.Disable()
            # Enable Device Status options
            s.gpio1Devsts.Enable()
        elif ((gpio1_source == 6) or (gpio1_source == 7)):
            # Disable other options
            s.gpio1Port.Disable()
            s.gpio1Devsts.Disable()
            # Enable TX Port options
            s.gpio1TXPort.Enable()
        else:
            # Enable Port Source options
            s.gpio1Port.Enable()
            # Disable other options
            s.gpio1TXPort.Disable()
            s.gpio1Devsts.Disable()
            
        return
        
    def gpio1PortClick(s, event):
        gpio1_port_option = (s.gpio1Port.GetSelection())
        s.gpio1Devsts.SetSelection(gpio1_port_option)
        s.gpio1TXPort.SetSelection(gpio1_port_option)
        s.board.gpio1_pin_ctl &= 0x1F
        s.board.gpio1_pin_ctl |= (gpio1_port_option << 5)
        s.board.WriteRXReg(s.board.rx.GPIO1_PIN_CTL, s.board.gpio1_pin_ctl)
        return
        
        
    def gpio1DevstsClick(s, event):
        gpio1_devsts_option = (s.gpio1Devsts.GetSelection())
        s.gpio1Port.SetSelection(gpio1_devsts_option)
        s.gpio1TXPort.SetSelection(gpio1_devsts_option)
        s.board.gpio1_pin_ctl &= 0x1F
        s.board.gpio1_pin_ctl |= (gpio1_devsts_option << 5)
        s.board.WriteRXReg(s.board.rx.GPIO1_PIN_CTL, s.board.gpio1_pin_ctl)
        return
                
    def gpio1TXPortClick(s, event):
        gpio1_txport_option = (s.gpio1TXPort.GetSelection())
        s.gpio1Devsts.SetSelection(gpio1_txport_option)
        s.gpio1Port.SetSelection(gpio1_txport_option)
        s.board.gpio1_pin_ctl &= 0x1F
        s.board.gpio1_pin_ctl |= (gpio1_txport_option << 5)
        s.board.WriteRXReg(s.board.rx.GPIO1_PIN_CTL, s.board.gpio1_pin_ctl)
        return
          
    def gpio1OutValClick(s, event):
        gpio1_outval = (s.gpio1OutVal.GetSelection()) & 0x1
        s.board.gpio1_pin_ctl &= 0xFD
        s.board.gpio1_pin_ctl |= (gpio1_outval << 1)
        s.board.WriteRXReg(s.board.rx.GPIO1_PIN_CTL, s.board.gpio1_pin_ctl)
        return
        
    def onGPIO2Click(s, event):
        isChecked = s.gpio2EnableChk.IsChecked()
        if (isChecked):
            s.board.gpio2_pin_ctl |= s.board.rx.GPIO_OUT_EN
        else:
            s.board.gpio2_pin_ctl &= 0xFE
        s.board.WriteRXReg(s.board.rx.GPIO2_PIN_CTL, s.board.gpio2_pin_ctl)
        return
            
    def gpio2SourceClick(s, event):
        gpio2_source = (s.gpio2Source.GetSelection())
        s.board.gpio2_pin_ctl &= 0xE3
        s.board.gpio2_pin_ctl |= (gpio2_source << 2)
        s.board.WriteRXReg(s.board.rx.GPIO2_PIN_CTL, s.board.gpio2_pin_ctl)
        if (gpio2_source == 5):
            # Disable all options
            s.gpio2Port.Disable()
            s.gpio2TXPort.Disable()
            s.gpio2Devsts.Disable()
        elif (gpio2_source == 4):
            # Disable other options
            s.gpio2Port.Disable()
            s.gpio2TXPort.Disable()
            # Enable Device Status options
            s.gpio2Devsts.Enable()
        elif ((gpio2_source == 6) or (gpio2_source == 7)):
            # Disable other options
            s.gpio2Port.Disable()
            s.gpio2Devsts.Disable()
            # Enable TX Port options
            s.gpio2TXPort.Enable()
        else:
            # Enable Port Source options
            s.gpio2Port.Enable()
            # Disable other options
            s.gpio2TXPort.Disable()
            s.gpio2Devsts.Disable()
            
        return
        
    def gpio2PortClick(s, event):
        gpio2_port_option = (s.gpio2Port.GetSelection())
        s.gpio2Devsts.SetSelection(gpio2_port_option)
        s.gpio2TXPort.SetSelection(gpio2_port_option)
        s.board.gpio2_pin_ctl &= 0x1F
        s.board.gpio2_pin_ctl |= (gpio2_port_option << 5)
        s.board.WriteRXReg(s.board.rx.GPIO2_PIN_CTL, s.board.gpio2_pin_ctl)
        return
        
        
    def gpio2DevstsClick(s, event):
        gpio2_devsts_option = (s.gpio2Devsts.GetSelection())
        s.gpio2Port.SetSelection(gpio2_devsts_option)
        s.gpio2TXPort.SetSelection(gpio2_devsts_option)
        s.board.gpio2_pin_ctl &= 0x1F
        s.board.gpio2_pin_ctl |= (gpio2_devsts_option << 5)
        s.board.WriteRXReg(s.board.rx.GPIO2_PIN_CTL, s.board.gpio2_pin_ctl)
        return
                
    def gpio2TXPortClick(s, event):
        gpio2_txport_option = (s.gpio2TXPort.GetSelection())
        s.gpio2Devsts.SetSelection(gpio2_txport_option)
        s.gpio2Port.SetSelection(gpio2_txport_option)
        s.board.gpio2_pin_ctl &= 0x1F
        s.board.gpio2_pin_ctl |= (gpio2_txport_option << 5)
        s.board.WriteRXReg(s.board.rx.GPIO2_PIN_CTL, s.board.gpio2_pin_ctl)
        return
        
    def gpio2OutValClick(s, event):
        gpio2_outval = (s.gpio2OutVal.GetSelection()) & 0x1
        s.board.gpio2_pin_ctl &= 0xFD
        s.board.gpio2_pin_ctl |= (gpio2_outval << 1)
        s.board.WriteRXReg(s.board.rx.GPIO2_PIN_CTL, s.board.gpio2_pin_ctl)
        return
        
    def onGPIO3Click(s, event):
        isChecked = s.gpio3EnableChk.IsChecked()
        if (isChecked):
            s.board.gpio3_pin_ctl |= s.board.rx.GPIO_OUT_EN
        else:
            s.board.gpio3_pin_ctl &= 0xFE
        s.board.WriteRXReg(s.board.rx.GPIO3_PIN_CTL, s.board.gpio3_pin_ctl)
        return
            
    def gpio3SourceClick(s, event):
        gpio3_source = (s.gpio3Source.GetSelection())
        s.board.gpio3_pin_ctl &= 0xE3
        s.board.gpio3_pin_ctl |= (gpio3_source << 2)
        s.board.WriteRXReg(s.board.rx.GPIO3_PIN_CTL, s.board.gpio3_pin_ctl)
        if (gpio3_source == 5):
            # Disable all options
            s.gpio3Port.Disable()
            s.gpio3TXPort.Disable()
            s.gpio3Devsts.Disable()
        elif (gpio3_source == 4):
            # Disable other options
            s.gpio3Port.Disable()
            s.gpio3TXPort.Disable()
            # Enable Device Status options
            s.gpio3Devsts.Enable()
        elif ((gpio3_source == 6) or (gpio3_source == 7)):
            # Disable other options
            s.gpio3Port.Disable()
            s.gpio3Devsts.Disable()
            # Enable TX Port options
            s.gpio3TXPort.Enable()
        else:
            # Enable Port Source options
            s.gpio3Port.Enable()
            # Disable other options
            s.gpio3TXPort.Disable()
            s.gpio3Devsts.Disable()
            
        return
        
    def gpio3PortClick(s, event):
        gpio3_port_option = (s.gpio3Port.GetSelection())
        s.gpio3Devsts.SetSelection(gpio3_port_option)
        s.gpio3TXPort.SetSelection(gpio3_port_option)
        s.board.gpio3_pin_ctl &= 0x1F
        s.board.gpio3_pin_ctl |= (gpio3_port_option << 5)
        s.board.WriteRXReg(s.board.rx.GPIO3_PIN_CTL, s.board.gpio3_pin_ctl)
        return
        
        
    def gpio3DevstsClick(s, event):
        gpio3_devsts_option = (s.gpio3Devsts.GetSelection())
        s.gpio3Port.SetSelection(gpio3_devsts_option)
        s.gpio3TXPort.SetSelection(gpio3_devsts_option)
        s.board.gpio3_pin_ctl &= 0x1F
        s.board.gpio3_pin_ctl |= (gpio3_devsts_option << 5)
        s.board.WriteRXReg(s.board.rx.GPIO3_PIN_CTL, s.board.gpio3_pin_ctl)
        return
                
    def gpio3TXPortClick(s, event):
        gpio3_txport_option = (s.gpio3TXPort.GetSelection())
        s.gpio3Devsts.SetSelection(gpio3_txport_option)
        s.gpio3Port.SetSelection(gpio3_txport_option)
        s.board.gpio3_pin_ctl &= 0x1F
        s.board.gpio3_pin_ctl |= (gpio3_txport_option << 5)
        s.board.WriteRXReg(s.board.rx.GPIO3_PIN_CTL, s.board.gpio3_pin_ctl)
        return
        
    def gpio3OutValClick(s, event):
        gpio3_outval = (s.gpio3OutVal.GetSelection()) & 0x1
        s.board.gpio3_pin_ctl &= 0xFD
        s.board.gpio3_pin_ctl |= (gpio3_outval << 1)
        s.board.WriteRXReg(s.board.rx.GPIO3_PIN_CTL, s.board.gpio3_pin_ctl)
        return
        
    def onGPIO4Click(s, event):
        isChecked = s.gpio4EnableChk.IsChecked()
        if (isChecked):
            s.board.gpio4_pin_ctl |= s.board.rx.GPIO_OUT_EN
        else:
            s.board.gpio4_pin_ctl &= 0xFE
        s.board.WriteRXReg(s.board.rx.GPIO4_PIN_CTL, s.board.gpio4_pin_ctl)
        return
            
    def gpio4SourceClick(s, event):
        gpio4_source = (s.gpio4Source.GetSelection())
        s.board.gpio4_pin_ctl &= 0xE3
        s.board.gpio4_pin_ctl |= (gpio4_source << 2)
        s.board.WriteRXReg(s.board.rx.GPIO4_PIN_CTL, s.board.gpio4_pin_ctl)
        if (gpio4_source == 5):
            # Disable all options
            s.gpio4Port.Disable()
            s.gpio4TXPort.Disable()
            s.gpio4Devsts.Disable()
        elif (gpio4_source == 4):
            # Disable other options
            s.gpio4Port.Disable()
            s.gpio4TXPort.Disable()
            # Enable Device Status options
            s.gpio4Devsts.Enable()
        elif ((gpio4_source == 6) or (gpio4_source == 7)):
            # Disable other options
            s.gpio4Port.Disable()
            s.gpio4Devsts.Disable()
            # Enable TX Port options
            s.gpio4TXPort.Enable()
        else:
            # Enable Port Source options
            s.gpio4Port.Enable()
            # Disable other options
            s.gpio4TXPort.Disable()
            s.gpio4Devsts.Disable()
            
        return
        
    def gpio4PortClick(s, event):
        gpio4_port_option = (s.gpio4Port.GetSelection())
        s.gpio4Devsts.SetSelection(gpio4_port_option)
        s.gpio4TXPort.SetSelection(gpio4_port_option)
        s.board.gpio4_pin_ctl &= 0x1F
        s.board.gpio4_pin_ctl |= (gpio4_port_option << 5)
        s.board.WriteRXReg(s.board.rx.GPIO4_PIN_CTL, s.board.gpio4_pin_ctl)
        return
        
        
    def gpio4DevstsClick(s, event):
        gpio4_devsts_option = (s.gpio4Devsts.GetSelection())
        s.gpio4Port.SetSelection(gpio4_devsts_option)
        s.gpio4TXPort.SetSelection(gpio4_devsts_option)
        s.board.gpio4_pin_ctl &= 0x1F
        s.board.gpio4_pin_ctl |= (gpio4_devsts_option << 5)
        s.board.WriteRXReg(s.board.rx.GPIO4_PIN_CTL, s.board.gpio4_pin_ctl)
        return
                
    def gpio4TXPortClick(s, event):
        gpio4_txport_option = (s.gpio4TXPort.GetSelection())
        s.gpio4Devsts.SetSelection(gpio4_txport_option)
        s.gpio4Port.SetSelection(gpio4_txport_option)
        s.board.gpio4_pin_ctl &= 0x1F
        s.board.gpio4_pin_ctl |= (gpio4_txport_option << 5)
        s.board.WriteRXReg(s.board.rx.GPIO4_PIN_CTL, s.board.gpio4_pin_ctl)
        return
        
    def gpio4OutValClick(s, event):
        gpio4_outval = (s.gpio4OutVal.GetSelection()) & 0x1
        s.board.gpio4_pin_ctl &= 0xFD
        s.board.gpio4_pin_ctl |= (gpio4_outval << 1)
        s.board.WriteRXReg(s.board.rx.GPIO4_PIN_CTL, s.board.gpio4_pin_ctl)
        return
        
    def onGPIO5Click(s, event):
        isChecked = s.gpio5EnableChk.IsChecked()
        if (isChecked):
            s.board.gpio5_pin_ctl |= s.board.rx.GPIO_OUT_EN
        else:
            s.board.gpio5_pin_ctl &= 0xFE
        s.board.WriteRXReg(s.board.rx.GPIO5_PIN_CTL, s.board.gpio5_pin_ctl)
        return
            
    def gpio5SourceClick(s, event):
        gpio5_source = (s.gpio5Source.GetSelection())
        s.board.gpio5_pin_ctl &= 0xE3
        s.board.gpio5_pin_ctl |= (gpio5_source << 2)
        s.board.WriteRXReg(s.board.rx.GPIO5_PIN_CTL, s.board.gpio5_pin_ctl)
        if (gpio5_source == 5):
            # Disable all options
            s.gpio5Port.Disable()
            s.gpio5TXPort.Disable()
            s.gpio5Devsts.Disable()
        elif (gpio5_source == 4):
            # Disable other options
            s.gpio5Port.Disable()
            s.gpio5TXPort.Disable()
            # Enable Device Status options
            s.gpio5Devsts.Enable()
        elif ((gpio5_source == 6) or (gpio5_source == 7)):
            # Disable other options
            s.gpio5Port.Disable()
            s.gpio5Devsts.Disable()
            # Enable TX Port options
            s.gpio5TXPort.Enable()
        else:
            # Enable Port Source options
            s.gpio5Port.Enable()
            # Disable other options
            s.gpio5TXPort.Disable()
            s.gpio5Devsts.Disable()
           
        return
        
    def gpio5PortClick(s, event):
        gpio5_port_option = (s.gpio5Port.GetSelection())
        s.gpio5Devsts.SetSelection(gpio5_port_option)
        s.gpio5TXPort.SetSelection(gpio5_port_option)
        s.board.gpio5_pin_ctl &= 0x1F
        s.board.gpio5_pin_ctl |= (gpio5_port_option << 5)
        s.board.WriteRXReg(s.board.rx.GPIO5_PIN_CTL, s.board.gpio5_pin_ctl)
        return
        
        
    def gpio5DevstsClick(s, event):
        gpio5_devsts_option = (s.gpio5Devsts.GetSelection())
        s.gpio5Port.SetSelection(gpio5_devsts_option)
        s.gpio5TXPort.SetSelection(gpio5_devsts_option)
        s.board.gpio5_pin_ctl &= 0x1F
        s.board.gpio5_pin_ctl |= (gpio5_devsts_option << 5)
        s.board.WriteRXReg(s.board.rx.GPIO5_PIN_CTL, s.board.gpio5_pin_ctl)
        return
                
    def gpio5TXPortClick(s, event):
        gpio5_txport_option = (s.gpio5TXPort.GetSelection())
        s.gpio5Devsts.SetSelection(gpio5_txport_option)
        s.gpio5Port.SetSelection(gpio5_txport_option)
        s.board.gpio5_pin_ctl &= 0x1F
        s.board.gpio5_pin_ctl |= (gpio5_txport_option << 5)
        s.board.WriteRXReg(s.board.rx.GPIO5_PIN_CTL, s.board.gpio5_pin_ctl)
        return
        
    def gpio5OutValClick(s, event):
        gpio5_outval = (s.gpio5OutVal.GetSelection()) & 0x1
        s.board.gpio5_pin_ctl &= 0xFD
        s.board.gpio5_pin_ctl |= (gpio5_outval << 1)
        s.board.WriteRXReg(s.board.rx.GPIO5_PIN_CTL, s.board.gpio5_pin_ctl)
        return
        
    def onGPIO6Click(s, event):
        isChecked = s.gpio6EnableChk.IsChecked()
        if (isChecked):
            s.board.gpio6_pin_ctl |= s.board.rx.GPIO_OUT_EN
        else:
            s.board.gpio6_pin_ctl &= 0xFE
        s.board.WriteRXReg(s.board.rx.GPIO6_PIN_CTL, s.board.gpio6_pin_ctl)
        return
            
    def gpio6SourceClick(s, event):
        gpio6_source = (s.gpio6Source.GetSelection())
        s.board.gpio6_pin_ctl &= 0xE3
        s.board.gpio6_pin_ctl |= (gpio6_source << 2)
        s.board.WriteRXReg(s.board.rx.GPIO6_PIN_CTL, s.board.gpio6_pin_ctl)
        if (gpio6_source == 5):
            # Disable all options
            s.gpio6Port.Disable()
            s.gpio6TXPort.Disable()
            s.gpio6Devsts.Disable()
        elif (gpio6_source == 4):
            # Disable other options
            s.gpio6Port.Disable()
            s.gpio6TXPort.Disable()
            # Enable Device Status options
            s.gpio6Devsts.Enable()
        elif ((gpio6_source == 6) or (gpio6_source == 7)):
            # Disable other options
            s.gpio6Port.Disable()
            s.gpio6Devsts.Disable()
            # Enable TX Port options
            s.gpio6TXPort.Enable()
        else:
            # Enable Port Source options
            s.gpio6Port.Enable()
            # Disable other options
            s.gpio6TXPort.Disable()
            s.gpio6Devsts.Disable()
            
        return
        
    def gpio6PortClick(s, event):
        gpio6_port_option = (s.gpio6Port.GetSelection())
        s.gpio6Devsts.SetSelection(gpio6_port_option)
        s.gpio6TXPort.SetSelection(gpio6_port_option)
        s.board.gpio6_pin_ctl &= 0x1F
        s.board.gpio6_pin_ctl |= (gpio6_port_option << 5)
        s.board.WriteRXReg(s.board.rx.GPIO6_PIN_CTL, s.board.gpio6_pin_ctl)
        return
        
        
    def gpio6DevstsClick(s, event):
        gpio6_devsts_option = (s.gpio6Devsts.GetSelection())
        s.gpio6Port.SetSelection(gpio6_devsts_option)
        s.gpio6TXPort.SetSelection(gpio6_devsts_option)
        s.board.gpio6_pin_ctl &= 0x1F
        s.board.gpio6_pin_ctl |= (gpio6_devsts_option << 5)
        s.board.WriteRXReg(s.board.rx.GPIO6_PIN_CTL, s.board.gpio6_pin_ctl)
        return
                
    def gpio6TXPortClick(s, event):
        gpio6_txport_option = (s.gpio6TXPort.GetSelection())
        s.gpio6Devsts.SetSelection(gpio6_txport_option)
        s.gpio6Port.SetSelection(gpio6_txport_option)
        s.board.gpio6_pin_ctl &= 0x1F
        s.board.gpio6_pin_ctl |= (gpio6_txport_option << 5)
        s.board.WriteRXReg(s.board.rx.GPIO6_PIN_CTL, s.board.gpio6_pin_ctl)
        return
        
    def gpio6OutValClick(s, event):
        gpio6_outval = (s.gpio6OutVal.GetSelection()) & 0x1
        s.board.gpio6_pin_ctl &= 0xFD
        s.board.gpio6_pin_ctl |= (gpio6_outval << 1)
        s.board.WriteRXReg(s.board.rx.GPIO6_PIN_CTL, s.board.gpio6_pin_ctl)
        return
        
    def onGPIO7Click(s, event):
        isChecked = s.gpio7EnableChk.IsChecked()
        if (isChecked):
            s.board.gpio7_pin_ctl |= s.board.rx.GPIO_OUT_EN
        else:
            s.board.gpio7_pin_ctl &= 0xFE
        s.board.WriteRXReg(s.board.rx.GPIO7_PIN_CTL, s.board.gpio7_pin_ctl)
        return
            
    def gpio7SourceClick(s, event):
        gpio7_source = (s.gpio7Source.GetSelection())
        s.board.gpio7_pin_ctl &= 0xE3
        s.board.gpio7_pin_ctl |= (gpio7_source << 2)
        s.board.WriteRXReg(s.board.rx.GPIO7_PIN_CTL, s.board.gpio7_pin_ctl)
        if (gpio7_source == 5):
            # Disable all options
            s.gpio7Port.Disable()
            s.gpio7TXPort.Disable()
            s.gpio7Devsts.Disable()
        elif (gpio7_source == 4):
            # Disable other options
            s.gpio7Port.Disable()
            s.gpio7TXPort.Disable()
            # Enable Device Status options
            s.gpio7Devsts.Enable()
        elif ((gpio7_source == 6) or (gpio7_source == 7)):
            # Disable other options
            s.gpio7Port.Disable()
            s.gpio7Devsts.Disable()
            # Enable TX Port options
            s.gpio7TXPort.Enable()
        else:
            # Enable Port Source options
            s.gpio7Port.Enable()
            # Disable other options
            s.gpio7TXPort.Disable()
            s.gpio7Devsts.Disable()
            
        return
        
    def gpio7PortClick(s, event):
        gpio7_port_option = (s.gpio7Port.GetSelection())
        s.gpio7Devsts.SetSelection(gpio7_port_option)
        s.gpio7TXPort.SetSelection(gpio7_port_option)
        s.board.gpio7_pin_ctl &= 0x1F
        s.board.gpio7_pin_ctl |= (gpio7_port_option << 5)
        s.board.WriteRXReg(s.board.rx.GPIO7_PIN_CTL, s.board.gpio7_pin_ctl)
        return
        
        
    def gpio7DevstsClick(s, event):
        gpio7_devsts_option = (s.gpio7Devsts.GetSelection())
        s.gpio7Port.SetSelection(gpio7_devsts_option)
        s.gpio7TXPort.SetSelection(gpio7_devsts_option)
        s.board.gpio7_pin_ctl &= 0x1F
        s.board.gpio7_pin_ctl |= (gpio7_devsts_option << 5)
        s.board.WriteRXReg(s.board.rx.GPIO7_PIN_CTL, s.board.gpio7_pin_ctl)
        return
                
    def gpio7TXPortClick(s, event):
        gpio7_txport_option = (s.gpio7TXPort.GetSelection())
        s.gpio7Devsts.SetSelection(gpio7_txport_option)
        s.gpio7Port.SetSelection(gpio7_txport_option)
        s.board.gpio7_pin_ctl &= 0x1F
        s.board.gpio7_pin_ctl |= (gpio7_txport_option << 5)
        s.board.WriteRXReg(s.board.rx.GPIO7_PIN_CTL, s.board.gpio7_pin_ctl)
        return
        
    def gpio7OutValClick(s, event):
        gpio7_outval = (s.gpio7OutVal.GetSelection()) & 0x1
        s.board.gpio7_pin_ctl &= 0xFD
        s.board.gpio7_pin_ctl |= (gpio7_outval << 1)
        s.board.WriteRXReg(s.board.rx.GPIO7_PIN_CTL, s.board.gpio7_pin_ctl)
        return
        
    def UpdateGPIOSts(s):
        if (not s.board.rxValid):
            return (0)
        else:
            # Read GPIO Pin Status
            pin_sts = s.board.GetGPIOPinStatus()
            
            # Set GPIO pin status for each port
            for portIndex in range(8):
                txt = "%d" % pin_sts[portIndex]
                s.gpioStsCol1TxtCtrls[portIndex].SetLabel(txt)
        return
     
    def BuildGPIOStsBox(s):
        s.gpioStsPanel = wx.Panel( s.panel, style=wx.TAB_TRAVERSAL|wx.CLIP_CHILDREN)
        s.gpioStsLabel = "GPIO Pin Status"
        s.gpioStsBox = wx.StaticBox( s.gpioStsPanel, -1, s.gpioStsLabel)
        s.gpioStsBoxSz = wx.StaticBoxSizer( s.gpioStsBox, wx.VERTICAL)
        s.gpioStsPanel.SetSizer( s.gpioStsBoxSz)
        
        s.gpioStsCol0Sz = wx.BoxSizer( wx.VERTICAL)
        s.gpioStsCol1Sz = wx.BoxSizer( wx.VERTICAL)
        s.gpioStsCol0TxtCtrls = []
        s.gpioStsCol1TxtCtrls = []

        minLabelField = s.panel.GetTextExtent( "GPIO 0:")
        minStsField = s.panel.GetTextExtent( " 0 ")
        rows = ('GPIO 0:','GPIO 1:','GPIO 2:','GPIO 3:','GPIO 4:','GPIO 5:','GPIO 6:','GPIO 7:')

        for row in rows:
            txt = WriteText( s.gpioStsPanel, row, None, False)
            s.gpioStsCol0Sz.Add( txt)
            s.gpioStsCol0Sz.SetItemMinSize( txt, *minLabelField)
            s.gpioStsCol0TxtCtrls.append( txt)
            txt = WriteText( s.gpioStsPanel, "", None, False)
            s.gpioStsCol1Sz.Add( txt, flag=wx.LEFT|wx.RIGHT, border=10)
            s.gpioStsCol1Sz.SetItemMinSize( txt, *minStsField)
            s.gpioStsCol1TxtCtrls.append( txt)
                
        s.gpioStsColSz = wx.BoxSizer( wx.HORIZONTAL)
        s.gpioStsColSz.Add( s.gpioStsCol0Sz)
        s.gpioStsColSz.Add( s.gpioStsCol1Sz)       
        s.gpioStsBoxSz.Add( s.gpioStsColSz, flag=wx.ALL, border=5)
        
        s.UpdateGPIOSts()

    def UpdateGPIOTab(s):
        # Get Current pin control values
        s.board.gpio0_pin_ctl = s.board.ReadRXReg(s.board.rx.GPIO0_PIN_CTL)
        s.board.gpio1_pin_ctl = s.board.ReadRXReg(s.board.rx.GPIO1_PIN_CTL)
        s.board.gpio2_pin_ctl = s.board.ReadRXReg(s.board.rx.GPIO2_PIN_CTL)
        s.board.gpio3_pin_ctl = s.board.ReadRXReg(s.board.rx.GPIO3_PIN_CTL)
        s.board.gpio4_pin_ctl = s.board.ReadRXReg(s.board.rx.GPIO4_PIN_CTL)
        s.board.gpio5_pin_ctl = s.board.ReadRXReg(s.board.rx.GPIO5_PIN_CTL)
        s.board.gpio6_pin_ctl = s.board.ReadRXReg(s.board.rx.GPIO6_PIN_CTL)
        s.board.gpio7_pin_ctl = s.board.ReadRXReg(s.board.rx.GPIO7_PIN_CTL)
        # GPIO Pin 0
        s.gpio0EnableChk.SetValue(s.board.gpio0_pin_ctl & s.board.rx.GPIO_OUT_EN)
        s.gpio0Source.SetSelection((s.board.gpio0_pin_ctl >> 2) & 0x7)
        s.gpio0Port.SetSelection((s.board.gpio0_pin_ctl >> 5) & 0x7)
        s.gpio0Devsts.SetSelection((s.board.gpio0_pin_ctl >> 5) & 0x7)
        s.gpio0OutVal.SetSelection((s.board.gpio0_pin_ctl >> 1) & 0x1)
        if ((s.board.gpio0_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_DEVICE_STS):
            s.gpio0Devsts.Enable()
        elif (((s.board.gpio0_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0) \
                or ((s.board.gpio0_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0)):
            s.gpio0TXPort.Enable()
        else:
            s.gpio0Port.Enable()
        # GPIO Pin 1
        s.gpio1EnableChk.SetValue(s.board.gpio1_pin_ctl & s.board.rx.GPIO_OUT_EN)
        s.gpio1Source.SetSelection((s.board.gpio1_pin_ctl >> 2) & 0x7)
        s.gpio1Port.SetSelection((s.board.gpio1_pin_ctl >> 5) & 0x7)
        s.gpio1Devsts.SetSelection((s.board.gpio1_pin_ctl >> 5) & 0x7)
        s.gpio1OutVal.SetSelection((s.board.gpio1_pin_ctl >> 1) & 0x1)
        if ((s.board.gpio1_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_DEVICE_STS):
            s.gpio1Devsts.Enable()
        elif (((s.board.gpio1_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0) \
                or ((s.board.gpio1_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0)):
            s.gpio1TXPort.Enable()
        else:
            s.gpio1Port.Enable()
       # GPIO Pin 2
        s.gpio2EnableChk.SetValue(s.board.gpio2_pin_ctl & s.board.rx.GPIO_OUT_EN)
        s.gpio2Source.SetSelection((s.board.gpio2_pin_ctl >> 2) & 0x7)
        s.gpio2Port.SetSelection((s.board.gpio2_pin_ctl >> 5) & 0x7)
        s.gpio2Devsts.SetSelection((s.board.gpio2_pin_ctl >> 5) & 0x7)
        s.gpio2OutVal.SetSelection((s.board.gpio2_pin_ctl >> 1) & 0x1)
        if ((s.board.gpio2_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_DEVICE_STS):
            s.gpio2Devsts.Enable()
        elif (((s.board.gpio2_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0) \
                or ((s.board.gpio2_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0)):
            s.gpio2TXPort.Enable()
        else:
            s.gpio2Port.Enable()
        # GPIO Pin 3
        s.gpio3EnableChk.SetValue(s.board.gpio3_pin_ctl & s.board.rx.GPIO_OUT_EN)
        s.gpio3Source.SetSelection((s.board.gpio3_pin_ctl >> 2) & 0x7)
        s.gpio3Port.SetSelection((s.board.gpio3_pin_ctl >> 5) & 0x7)
        s.gpio3Devsts.SetSelection((s.board.gpio3_pin_ctl >> 5) & 0x7)
        s.gpio3OutVal.SetSelection((s.board.gpio3_pin_ctl >> 1) & 0x1)
        if ((s.board.gpio3_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_DEVICE_STS):
            s.gpio3Devsts.Enable()
        elif (((s.board.gpio3_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0) \
                or ((s.board.gpio3_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0)):
            s.gpio3TXPort.Enable()
        else:
            s.gpio3Port.Enable()
        # GPIO Pin 4
        s.gpio4EnableChk.SetValue(s.board.gpio4_pin_ctl & s.board.rx.GPIO_OUT_EN)
        s.gpio4Source.SetSelection((s.board.gpio4_pin_ctl >> 2) & 0x7)
        s.gpio4Port.SetSelection((s.board.gpio4_pin_ctl >> 5) & 0x7)
        s.gpio4Devsts.SetSelection((s.board.gpio4_pin_ctl >> 5) & 0x7)
        s.gpio4OutVal.SetSelection((s.board.gpio4_pin_ctl >> 1) & 0x1)
        if ((s.board.gpio4_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_DEVICE_STS):
            s.gpio4Devsts.Enable()
        elif (((s.board.gpio4_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0) \
                or ((s.board.gpio4_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0)):
            s.gpio4TXPort.Enable()
        else:
            s.gpio4Port.Enable()
        # GPIO Pin 5
        s.gpio5EnableChk.SetValue(s.board.gpio5_pin_ctl & s.board.rx.GPIO_OUT_EN)
        s.gpio5Source.SetSelection((s.board.gpio5_pin_ctl >> 2) & 0x7)
        s.gpio5Port.SetSelection((s.board.gpio5_pin_ctl >> 5) & 0x7)
        s.gpio5Devsts.SetSelection((s.board.gpio5_pin_ctl >> 5) & 0x7)
        s.gpio5OutVal.SetSelection((s.board.gpio5_pin_ctl >> 1) & 0x1)
        if ((s.board.gpio5_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_DEVICE_STS):
            s.gpio5Devsts.Enable()
        elif (((s.board.gpio5_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0) \
                or ((s.board.gpio5_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0)):
            s.gpio5TXPort.Enable()
        else:
            s.gpio5Port.Enable()
        # GPIO Pin 6
        s.gpio6EnableChk.SetValue(s.board.gpio6_pin_ctl & s.board.rx.GPIO_OUT_EN)
        s.gpio6Source.SetSelection((s.board.gpio6_pin_ctl >> 2) & 0x7)
        s.gpio6Port.SetSelection((s.board.gpio6_pin_ctl >> 5) & 0x7)
        s.gpio6Devsts.SetSelection((s.board.gpio6_pin_ctl >> 5) & 0x7)
        s.gpio6OutVal.SetSelection((s.board.gpio6_pin_ctl >> 1) & 0x1)
        if ((s.board.gpio6_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_DEVICE_STS):
            s.gpio6Devsts.Enable()
        elif (((s.board.gpio6_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0) \
                or ((s.board.gpio6_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0)):
            s.gpio6TXPort.Enable()
        else:
            s.gpio6Port.Enable()
        # GPIO Pin 7
        s.gpio7EnableChk.SetValue(s.board.gpio7_pin_ctl & s.board.rx.GPIO_OUT_EN)
        s.gpio7Source.SetSelection((s.board.gpio7_pin_ctl >> 2) & 0x7)
        s.gpio7Port.SetSelection((s.board.gpio7_pin_ctl >> 5) & 0x7)
        s.gpio7Devsts.SetSelection((s.board.gpio7_pin_ctl >> 5) & 0x7)
        s.gpio7OutVal.SetSelection((s.board.gpio7_pin_ctl >> 1) & 0x1)        
        if ((s.board.gpio7_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_DEVICE_STS):
            s.gpio7Devsts.Enable()
        elif (((s.board.gpio7_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0) \
                or ((s.board.gpio7_pin_ctl & 0x1C) == s.board.rx.GPIO_OUT_SRC_TX_PORT0)):
            s.gpio7TXPort.Enable()
        else:
            s.gpio7Port.Enable()
        # Get Current BC GPIO control values
        for portIndex in range(4):
            s.board.bc_gpio_ctl0[portIndex] = s.board.ReadRXPortReg(portIndex, s.board.rx.BC_GPIO_CTL0)
            s.board.bc_gpio_ctl1[portIndex] = s.board.ReadRXPortReg(portIndex, s.board.rx.BC_GPIO_CTL1)
        # RX Port 0
        s.RX0_BCgpio0Source.SetSelection((s.board.bc_gpio_ctl0[0]) & 0xF)
        s.RX0_BCgpio1Source.SetSelection((s.board.bc_gpio_ctl0[0] >> 4) & 0xF)
        s.RX0_BCgpio2Source.SetSelection((s.board.bc_gpio_ctl1[0]) & 0xF)
        s.RX0_BCgpio3Source.SetSelection((s.board.bc_gpio_ctl1[0] >> 4) & 0xF)
        # RX Port 1
        s.RX1_BCgpio0Source.SetSelection((s.board.bc_gpio_ctl0[1]) & 0xF)
        s.RX1_BCgpio1Source.SetSelection((s.board.bc_gpio_ctl0[1] >> 4) & 0xF)
        s.RX1_BCgpio2Source.SetSelection((s.board.bc_gpio_ctl1[1]) & 0xF)
        s.RX1_BCgpio3Source.SetSelection((s.board.bc_gpio_ctl1[1] >> 4) & 0xF)
        # RX Port 2
        s.RX2_BCgpio0Source.SetSelection((s.board.bc_gpio_ctl0[2]) & 0xF)
        s.RX2_BCgpio1Source.SetSelection((s.board.bc_gpio_ctl0[2] >> 4) & 0xF)
        s.RX2_BCgpio2Source.SetSelection((s.board.bc_gpio_ctl1[2]) & 0xF)
        s.RX2_BCgpio3Source.SetSelection((s.board.bc_gpio_ctl1[2] >> 4) & 0xF)
        # RX Port 3
        s.RX3_BCgpio0Source.SetSelection((s.board.bc_gpio_ctl0[3]) & 0xF)
        s.RX3_BCgpio1Source.SetSelection((s.board.bc_gpio_ctl0[3] >> 4) & 0xF)
        s.RX3_BCgpio2Source.SetSelection((s.board.bc_gpio_ctl1[3]) & 0xF)
        s.RX3_BCgpio3Source.SetSelection((s.board.bc_gpio_ctl1[3] >> 4) & 0xF)
        
        # Get Current ref frequency settings
        s.UpdateFSclkSourceTxt()
        
    def CreateGPIOTab(s):
        # Panel heading
        #s.headingTxt = WriteText( s.panel, s.board.longDescription, position=None, bold=True)
        
        # Build the status boxes
        s.BuildGPIOBox()
        s.BuildGPIOStsBox()
        s.BuildBCGPIOBox()
        s.BuildFrameSyncPanel()
        # Build second Row
        s.gpioRow2Sizer = wx.BoxSizer( wx.HORIZONTAL)
        s.gpioRow2Sizer.Add( s.gpioStsPanel, flag=wx.ALL, border=5 )
        s.gpioRow2Sizer.Add( s.RX0_BCgpioStgsSBSz, flag=wx.ALL, border=5 )
        s.gpioRow2Sizer.Add( s.RX1_BCgpioStgsSBSz, flag=wx.ALL, border=5 )
        s.gpioRow2Sizer.Add( s.RX2_BCgpioStgsSBSz, flag=wx.ALL, border=5 )
        s.gpioRow2Sizer.Add( s.RX3_BCgpioStgsSBSz, flag=wx.ALL, border=5 )

        # Add the boxes to the panel sizer
        s.gpioSizer = wx.BoxSizer( wx.VERTICAL)
        
        s.gpioSizer.Add( s.gpioStgsSBSz, flag=wx.ALL, border=5 )
        s.gpioSizer.Add( s.gpioRow2Sizer, flag=wx.ALL, border=5 )
        s.gpioSizer.Add( s.FrameSyncSizer, flag=wx.ALL, border=5 )
        
        # Lay it all out
        s.panel.SetAutoLayout( True)
        s.panel.SetSizer( s.gpioSizer)
        s.gpioSizer.Layout()
        
        # Polling timer
        #s.timer = wx.Timer( s.panel)
        #wx.EVT_TIMER( s.panel, s.timer.GetId(), s.OnTimer)
        s.timer.Start(STATUS_TIMER)
        
    def RefreshGPIOTab(s):
        s.UpdateGPIOSts()
        return

    # Forwarding Enable check box
    def onPort0EnClick(s, event):
        isChecked = s.port0FwdEnableChk.IsChecked()
        if (isChecked):
            s.board.fwd_ctl1_val &= (0xFF & ~s.board.rx.FWD_PORT0_DIS)
        else:
            s.board.fwd_ctl1_val |= s.board.rx.FWD_PORT0_DIS
        return
        
    # Port Assignment drop down
    def port0AssignClick(s, event):
        portAssign = (s.port0Assign.GetSelection())
        s.board.fwd_ctl1_val &= (0xFF & ~s.board.rx.VC0_MAP)
        s.board.fwd_ctl1_val |= (portAssign << 0)
        return
        
    # Forwarding Enable check box
    def onPort1EnClick(s, event):
        isChecked = s.port1FwdEnableChk.IsChecked()
        if (isChecked):
            s.board.fwd_ctl1_val &= (0xFF & ~s.board.rx.FWD_PORT1_DIS)
        else:
            s.board.fwd_ctl1_val |= s.board.rx.FWD_PORT1_DIS
        return
        
    # Port Assignment drop down
    def port1AssignClick(s, event):
        portAssign = (s.port1Assign.GetSelection())
        s.board.fwd_ctl1_val &= (0xFF & ~s.board.rx.VC1_MAP)
        s.board.fwd_ctl1_val |= (portAssign << 1)
        return
        
    # Forwarding Enable check box
    def onPort2EnClick(s, event):
        isChecked = s.port2FwdEnableChk.IsChecked()
        if (isChecked):
            s.board.fwd_ctl1_val &= (0xFF & ~s.board.rx.FWD_PORT2_DIS)
        else:
            s.board.fwd_ctl1_val |= s.board.rx.FWD_PORT2_DIS
        return
        
    # Port Assignment drop down
    def port2AssignClick(s, event):
        portAssign = (s.port2Assign.GetSelection())
        s.board.fwd_ctl1_val &= (0xFF & ~s.board.rx.VC2_MAP)
        s.board.fwd_ctl1_val |= (portAssign << 2)
        return
        
    # Forwarding Enable check box
    def onPort3EnClick(s, event):
        isChecked = s.port3FwdEnableChk.IsChecked()
        if (isChecked):
            s.board.fwd_ctl1_val &= (0xFF & ~s.board.rx.FWD_PORT3_DIS)
        else:
            s.board.fwd_ctl1_val |= s.board.rx.FWD_PORT3_DIS
        return
        
    # Port Assignment drop down
    def port3AssignClick(s, event):
        portAssign = (s.port3Assign.GetSelection())
        s.board.fwd_ctl1_val &= (0xFF & ~s.board.rx.VC3_MAP)
        s.board.fwd_ctl1_val |= (portAssign << 3)
        return
        
    def ForwardApplyClick(s, event):
        s.board.WriteReg(s.board.rx.FWD_CTL1, s.board.fwd_ctl1_val)
        
    def BuildForwardingCtlBox(s):
        port_assign_txt = ["TX port 0", "TX port 1"]
        
        # RX Port 0
        # Forward enable check box
        s.port0FwdEnableChk = wx.CheckBox( s.panel, wx.NewId(), "RX port 0")
        wx.EVT_CHECKBOX( s.port0FwdEnableChk, s.port0FwdEnableChk.GetId(), s.onPort0EnClick)
  
        # Output Assignment drop-down
        (s.port0AssignSz, s.port0Assign, s.port0AssignTxt) = \
            BuildComboBox( s.panel, "", choices=port_assign_txt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.port0Assign.GetId(), s.port0AssignClick)

        # RX Port 1
        # Forward enable check box
        s.port1FwdEnableChk = wx.CheckBox( s.panel, wx.NewId(), "RX port 1")
        wx.EVT_CHECKBOX( s.port1FwdEnableChk, s.port1FwdEnableChk.GetId(), s.onPort1EnClick)
  
        # Output Assignment drop-down
        (s.port1AssignSz, s.port1Assign, s.port1AssignTxt) = \
            BuildComboBox( s.panel, "", choices=port_assign_txt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.port1Assign.GetId(), s.port1AssignClick)

        # RX Port 2
        # Forward enable check box
        s.port2FwdEnableChk = wx.CheckBox( s.panel, wx.NewId(), "RX port 2")
        wx.EVT_CHECKBOX( s.port2FwdEnableChk, s.port2FwdEnableChk.GetId(), s.onPort2EnClick)
  
        # Output Assignment drop-down
        (s.port2AssignSz, s.port2Assign, s.port2AssignTxt) = \
            BuildComboBox( s.panel, "", choices=port_assign_txt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.port2Assign.GetId(), s.port2AssignClick)

        # RX Port 3
        # Forward enable check box
        s.port3FwdEnableChk = wx.CheckBox( s.panel, wx.NewId(), "RX port 3")
        wx.EVT_CHECKBOX( s.port3FwdEnableChk, s.port3FwdEnableChk.GetId(), s.onPort3EnClick)
  
        # Output Assignment drop-down
        (s.port3AssignSz, s.port3Assign, s.port3AssignTxt) = \
            BuildComboBox( s.panel, "", choices=port_assign_txt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.port3Assign.GetId(), s.port3AssignClick)

        # Create the Apply buttons
        s.ForwardApplyButtonID, s.ForwardApplyButton = s.BuildButton("Apply", s.ForwardApplyClick, "")
        s.ForwardApplyButton.Enable()

        # Left side text
        txt0 = WriteText( s.panel, "Forward Enable", None, False)
        txt1 = WriteText( s.panel, "Port Assignment", None, False)
            
        # Configurations
        s.ForwardStgs = wx.GridBagSizer( 0, 0)
        s.ForwardStgs.Add( txt0, (0,0), border=5, flag=wx.TOP|wx.BOTTOM|wx.RIGHT)
        s.ForwardStgs.Add( txt1, (1,0), border=5, flag=wx.TOP|wx.BOTTOM|wx.RIGHT)
        s.ForwardStgs.Add( s.port0FwdEnableChk, (0,1), border=5, flag=wx.TOP|wx.BOTTOM)
        s.ForwardStgs.Add( s.port0AssignSz, (1,1), border=5, flag=wx.TOP|wx.BOTTOM)
        s.ForwardStgs.Add( s.port1FwdEnableChk, (0,2), border=5, flag=wx.TOP|wx.BOTTOM)
        s.ForwardStgs.Add( s.port1AssignSz, (1,2), border=5, flag=wx.TOP|wx.BOTTOM)
        s.ForwardStgs.Add( s.port2FwdEnableChk, (0,3), border=5, flag=wx.TOP|wx.BOTTOM)
        s.ForwardStgs.Add( s.port2AssignSz, (1,3), border=5, flag=wx.TOP|wx.BOTTOM)
        s.ForwardStgs.Add( s.port3FwdEnableChk, (0,4), border=5, flag=wx.TOP|wx.BOTTOM)
        s.ForwardStgs.Add( s.port3AssignSz, (1,4), border=5, flag=wx.TOP|wx.BOTTOM)
        s.ForwardStgs.Add( s.ForwardApplyButton, (2,2), border=5, flag=wx.TOP|wx.BOTTOM)
        
        
        # create static box
        s.ForwardStgsSB = wx.StaticBox( s.panel, -1, "RX Port Forwarding Control")
        s.ForwardStgsSBSz = wx.StaticBoxSizer( s.ForwardStgsSB, wx.VERTICAL)
        
        s.ForwardStgsSBSz.Add( s.ForwardStgs, flag=wx.ALL, border=5)

    # Replicate Mode check box
    def replicateClick(s, event):
        isChecked = s.replicateChk.IsChecked()
        if (isChecked):
            s.board.replicate_en = True
            s.board.fwd_ctl2_val |= s.board.rx.CSI_REPLICATE_MODE
            s.csi1EnableChk.Disable()
            s.csi1Lanes.Disable()
            s.csi1contClk.Disable()
            s.csi1Cal.Disable()
            s.csi1FwdMode.Disable()
        else:
            s.board.replicate_en = False
            s.board.fwd_ctl2_val &= 0x7F
            s.csi1EnableChk.Enable()
            s.csi1Lanes.Enable()
            s.csi1contClk.Enable()
            s.csi1Cal.Enable()
            s.csi1FwdMode.Enable()
        return
        
    # CSI TX  Enable check box
    def csi0EnClick(s, event):
        isChecked = s.csi0EnableChk.IsChecked()
        if (isChecked):
            s.board.csi_ctl1_val[0] |= s.board.rx.CSI_ENABLE
        else:
            s.board.csi_ctl1_val[0] &= (0xFF & ~s.board.rx.CSI_ENABLE)
        return
        
    # CSI datarate drop down
    def csi0RateClick(s, event):
        rateAssign = (s.csi0Rate.GetSelection())
        s.board.csi_pll_ctl_val[0] &= 0xFC
        s.board.csi_pll_ctl_val[0] |= (rateAssign << 0)

        if (rateAssign == 0):
            s.board.csi_pll_ctl_val[0] |= s.board.rx.CSITX_200M # 1.6Gbps
        if (rateAssign == 1):
            s.board.csi_pll_ctl_val[0] |= s.board.rx.CSITX_100M_TST # 1.2Gbps
        if (rateAssign == 2):
            s.board.csi_pll_ctl_val[0] |= s.board.rx.CSITX_100M # 800Mbps
        if (rateAssign == 3):
            s.board.csi_pll_ctl_val[0] |= s.board.rx.CSITX_50M # 400Mbps
        return
        
    # Number of lanes drop down
    def csi0LanesClick(s, event):
        laneAssign = (s.csi0Lanes.GetSelection())
        s.board.csi_ctl1_val[0] &= (0xCF)
        s.board.csi_ctl1_val[0] |= (laneAssign << 4)
        return
        
    # CSI Continuous Clock drop down
    def csi0contClkClick(s, event):
        enabled = (s.csi0contClk.GetSelection())
        if (enabled):
            s.board.csi_ctl1_val[0] |= s.board.rx.CSI_CONTS_CLOCK
        else:
            s.board.csi_ctl1_val[0] &= (0xFF & ~s.board.rx.CSI_CONTS_CLOCK)
        return

    # CSI Calibration drop down
    def csi0CalClick(s, event):
        cal_select = (s.csi0Cal.GetSelection())
        cal_init_en = cal_select & 0x1
        cal_per_en = cal_select >> 1
        
        if (cal_init_en):
            s.board.csi_ctl1_val[0] |= s.board.rx.CSI_CAL_EN
        else:
            s.board.csi_ctl1_val[0] &= (0xFF & ~s.board.rx.CSI_CAL_EN)
        
        if (cal_per_en):
            s.board.csi_ctl2_val[0] |= s.board.rx.CSI_CAL_PERIODIC
        else:
            s.board.csi_ctl2_val[0] &= (0xFF & ~s.board.rx.CSI_CAL_PERIODIC)
        return

    # Forwarding Mode drop down
    # Selections:  0: Disabled
    #                      1: Best Effort/RoundRobin
    #                      2: Synchronized (not merged)
    #                      3: Line Interleave
    #                      4: Line Concatenation
    def csi0FwdModeClick(s, event):
        fwd_select = (s.csi0FwdMode.GetSelection())
        s.board.fwd_ctl2_val &= 0xF2
        if (fwd_select == 1):
            s.board.fwd_ctl2_val |= s.board.rx.CSI0_RR_FWD
        elif (fwd_select == 2):
            s.board.fwd_ctl2_val |= s.board.rx.CSI0_SYNC_FWD_BASIC
        elif (fwd_select == 3):
            s.board.fwd_ctl2_val |= s.board.rx.CSI0_SYNC_LINE_INTERLEAVE
        elif (fwd_select == 4):
            s.board.fwd_ctl2_val |= s.board.rx.CSI0_SYNC_LINE_CONCAT
        return

    # CSI TX  Enable check box
    def csi1EnClick(s, event):
        isChecked = s.csi1EnableChk.IsChecked()
        if (isChecked):
            s.board.csi_ctl1_val[1] |= s.board.rx.CSI_ENABLE
        else:
            s.board.csi_ctl1_val[1] &= (0xFF & ~s.board.rx.CSI_ENABLE)
        return
        
    def csi1RateClick(s, event):
        rateAssign = (s.csi1Rate.GetSelection())
        s.board.csi_pll_ctl_val[1] &= 0xFC
        s.board.csi_pll_ctl_val[1] |= (rateAssign << 0)

        if (rateAssign == 0):
            s.board.csi_pll_ctl_val[1] |= s.board.rx.CSITX_200M # 1.6Gbps
        if (rateAssign == 1):
            s.board.csi_pll_ctl_val[1] |= s.board.rx.CSITX_100M_TST # 1.2Gbps
        if (rateAssign == 2):
            s.board.csi_pll_ctl_val[1] |= s.board.rx.CSITX_100M # 800Mbps
        if (rateAssign == 3):
            s.board.csi_pll_ctl_val[1] |= s.board.rx.CSITX_50M # 400Mbps
        return
        
    # Number of lanes drop down
    def csi1LanesClick(s, event):
        laneAssign = (s.csi1Lanes.GetSelection())
        s.board.csi_ctl1_val[1] &= (0xCF)
        s.board.csi_ctl1_val[1] |= (laneAssign << 4)
        return
        
    # Continuous drop down
    def csi1contClkClick(s, event):
        enabled = (s.csi1contClk.GetSelection())
        if (enabled):
            s.board.csi_ctl1_val[1] |= s.board.rx.CSI_CONTS_CLOCK
        else:
            s.board.csi_ctl1_val[1] &= (0xFF & ~s.board.rx.CSI_CONTS_CLOCK)
        return

    # CSI Calibration drop down
    def csi1CalClick(s, event):
        cal_select = (s.csi1Cal.GetSelection())
        cal_init_en = cal_select & 0x1
        cal_per_en = cal_select >> 1
        
        if (cal_init_en):
            s.board.csi_ctl1_val[1] |= s.board.rx.CSI_CAL_EN
        else:
            s.board.csi_ctl1_val[1] &= (0xFF & ~s.board.rx.CSI_CAL_EN)
        
        if (cal_per_en):
            s.board.csi_ctl2_val[1] |= s.board.rx.CSI_CAL_PERIODIC
        else:
            s.board.csi_ctl2_val[1] &= (0xFF & ~s.board.rx.CSI_CAL_PERIODIC)
        return

    # Forwarding Mode drop down
    # Selections:  0: Disabled
    #                      1: Best Effort/RoundRobin
    #                      2: Synchronized (not merged)
    #                      3: Line Interleave
    #                      4: Line Concatenation
    def csi1FwdModeClick(s, event):
        fwd_select = (s.csi1FwdMode.GetSelection())
        s.board.fwd_ctl2_val &= 0xCD
        if (fwd_select == 1):
            s.board.fwd_ctl2_val |= s.board.rx.CSI1_RR_FWD
        elif (fwd_select == 2):
            s.board.fwd_ctl2_val |= s.board.rx.CSI1_SYNC_FWD_BASIC
        elif (fwd_select == 3):
            s.board.fwd_ctl2_val |= s.board.rx.CSI1_SYNC_LINE_INTERLEAVE
        elif (fwd_select == 4):
            s.board.fwd_ctl2_val |= s.board.rx.CSI1_SYNC_LINE_CONCAT
        return

    # Apply button for CSI Tx Port configuration
    def csiTxApplyClick(s, event):
        for portIndex in range(2):
            s.board.WriteRXCSItxReg(portIndex, s.board.rx.CSI_CTL1, s.board.csi_ctl1_val[portIndex])
            s.board.WriteRXCSItxReg(portIndex, s.board.rx.CSI_CTL2, s.board.csi_ctl2_val[portIndex])
        s.board.WriteReg(s.board.rx.FWD_CTL2, s.board.fwd_ctl2_val)

    def BuildCSITxBox(s):

        # Replicate Mode check box
        s.replicateChk = wx.CheckBox( s.panel, wx.NewId(), "Replicate Mode")
        wx.EVT_CHECKBOX( s.replicateChk, s.replicateChk.GetId(), s.replicateClick)

        # TX Port 0
        # CSI TX  enable check box
        s.csi0EnableChk = wx.CheckBox( s.panel, wx.NewId(), "CSI TX 0")
        wx.EVT_CHECKBOX( s.csi0EnableChk, s.csi0EnableChk.GetId(), s.csi0EnClick)
  
        # CSI Datarate Assignment drop-down
        rate_assign_txt = ["1.6 Gbps", "1.2 Gbps", "800 Mbps", "400 Mbps"]
        (s.csi0RateSz, s.csi0Rate, s.csi0RateTxt) = \
            BuildComboBox( s.panel, "", choices=rate_assign_txt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.csi0Rate.GetId(), s.csi0RateClick)
  
        # Number of Lanes Assignment drop-down
        lane_assign_txt = ["4 Lanes", "3 Lanes", "2 Lanes", "1 Lane"]
        (s.csi0LanesSz, s.csi0Lanes, s.csi0LanesTxt) = \
            BuildComboBox( s.panel, "", choices=lane_assign_txt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.csi0Lanes.GetId(), s.csi0LanesClick)

        #Continuous clock enable/disable
        enable_txt = ["Disable", "Enable"]
        (s.csi0contClkSz, s.csi0contClk, s.csi0contClkTxt) = \
            BuildComboBox( s.panel, "", choices=enable_txt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.csi0contClk.GetId(), s.csi0contClkClick)

        # Calibration sequence enable/disable
        calibration_txt = ["Disable", "Initial Only", "Periodic Only", "Init + Per"]
        (s.csi0CalSz, s.csi0Cal, s.csi0CalTxt) = \
            BuildComboBox( s.panel, "", choices=calibration_txt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.csi0Cal.GetId(), s.csi0CalClick)

        # Forwarding Mode
        fwd_mode_txt = ["Disabled", "Best Effort", "Synchronized", "Line Interleave", "Line Concat", ""]
        (s.csi0FwdModeSz, s.csi0FwdMode, s.csi0FwdModeTxt) = \
            BuildComboBox( s.panel, "", choices=fwd_mode_txt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.csi0FwdMode.GetId(), s.csi0FwdModeClick)

        # TX Port 1
        # CSI TX  enable check box
        s.csi1EnableChk = wx.CheckBox( s.panel, wx.NewId(), "CSI TX 1")
        wx.EVT_CHECKBOX( s.csi1EnableChk, s.csi1EnableChk.GetId(), s.csi1EnClick)

        # CSI Datarate Assignment drop-down
        rate_assign_txt = ["1.6 Gbps", "1.2 Gbps", "800 Mbps", "400 Mbps"]
        (s.csi1RateSz, s.csi1Rate, s.csi1RateTxt) = \
            BuildComboBox( s.panel, "", choices=rate_assign_txt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.csi1Rate.GetId(), s.csi1RateClick)
        
        # Number of Lanes Assignment drop-down
        lane_assign_txt = ["4 Lanes", "3 Lanes", "2 Lanes", "1 Lane"]
        (s.csi1LanesSz, s.csi1Lanes, s.csi1LanesTxt) = \
            BuildComboBox( s.panel, "", choices=lane_assign_txt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.csi1Lanes.GetId(), s.csi1LanesClick)

        #Continuous clock enable/disable
        (s.csi1contClkSz, s.csi1contClk, s.csi1contClkTxt) = \
            BuildComboBox( s.panel, "", choices=enable_txt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.csi1contClk.GetId(), s.csi1contClkClick)

        # Calibration sequence enable/disable
        calibration_txt = ["Disable", "Initial Only", "Periodic Only", "Init + Per"]
        (s.csi1CalSz, s.csi1Cal, s.csi1CalTxt) = \
            BuildComboBox( s.panel, "", choices=calibration_txt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.csi1Cal.GetId(), s.csi1CalClick)

        # Forwarding Mode
        fwd_mode_txt = ["Disabled", "Best Effort", "Synchronized", "Line Interleave", "Line Concat", ""]
        (s.csi1FwdModeSz, s.csi1FwdMode, s.csi1FwdModeTxt) = \
            BuildComboBox( s.panel, "", choices=fwd_mode_txt, \
                labelOrientation=wx.RIGHT, enable=True, 
                style = wx.CB_DROPDOWN|wx.CB_READONLY)
        wx.EVT_COMBOBOX(s.panel, s.csi1FwdMode.GetId(), s.csi1FwdModeClick)

        # Create the Apply buttons
        s.csiTxApplyButtonID, s.csiTxApplyButton = s.BuildButton("Apply", s.csiTxApplyClick, "")
        s.csiTxApplyButton.Enable()

        # Update default values

        # Left side text
        txt_replicate = WriteText( s.panel, "Replicate Mode", None, False)
        txt0 = WriteText( s.panel, "CSI TX Enable", None, False)
        txt1 = WriteText( s.panel, "CSI Datarate", None, False)
        txt2 = WriteText( s.panel, "Number of Lanes", None, False)
        txt3 = WriteText( s.panel, "Continuous Clock", None, False)
        txt4 = WriteText( s.panel, "Calibration Sequence", None, False)
        txt5 = WriteText( s.panel, "Forwarding Mode", None, False)
        
        # Configurations
        s.csiTxStgs = wx.GridBagSizer( 0, 0)
        s.csiTxStgs.Add( txt_replicate, (0,0), border=5, flag=wx.TOP|wx.BOTTOM|wx.RIGHT)
        s.csiTxStgs.Add( txt0, (1,0), border=5, flag=wx.TOP|wx.BOTTOM|wx.RIGHT)
        s.csiTxStgs.Add( txt1, (2,0), border=5, flag=wx.TOP|wx.BOTTOM|wx.RIGHT)        
        s.csiTxStgs.Add( txt2, (3,0), border=5, flag=wx.TOP|wx.BOTTOM|wx.RIGHT)
        s.csiTxStgs.Add( txt3, (4,0), border=5, flag=wx.TOP|wx.BOTTOM|wx.RIGHT)
        s.csiTxStgs.Add( txt4, (5,0), border=5, flag=wx.TOP|wx.BOTTOM|wx.RIGHT)
        s.csiTxStgs.Add( txt5, (6,0), border=5, flag=wx.TOP|wx.BOTTOM|wx.RIGHT)
        s.csiTxStgs.Add( s.replicateChk, (0,1), border=5, flag=wx.TOP|wx.BOTTOM)
        s.csiTxStgs.Add( s.csi0EnableChk, (1,1), border=5, flag=wx.TOP|wx.BOTTOM)
        s.csiTxStgs.Add( s.csi0RateSz, (2,1), border=5, flag=wx.TOP|wx.BOTTOM)
        s.csiTxStgs.Add( s.csi0LanesSz, (3,1), border=5, flag=wx.TOP|wx.BOTTOM)
        s.csiTxStgs.Add( s.csi0contClkSz, (4,1), border=5, flag=wx.TOP|wx.BOTTOM)
        s.csiTxStgs.Add( s.csi0CalSz, (5,1), border=5, flag=wx.TOP|wx.BOTTOM)
        s.csiTxStgs.Add( s.csi0FwdModeSz, (6,1), border=5, flag=wx.TOP|wx.BOTTOM)
        s.csiTxStgs.Add( s.csi1EnableChk, (1,2), border=5, flag=wx.TOP|wx.BOTTOM)
        s.csiTxStgs.Add( s.csi1RateSz, (2,2), border=5, flag=wx.TOP|wx.BOTTOM)
        s.csiTxStgs.Add( s.csi1LanesSz, (3,2), border=5, flag=wx.TOP|wx.BOTTOM)
        s.csiTxStgs.Add( s.csi1contClkSz, (4,2), border=5, flag=wx.TOP|wx.BOTTOM)
        s.csiTxStgs.Add( s.csi1CalSz, (5,2), border=5, flag=wx.TOP|wx.BOTTOM)
        s.csiTxStgs.Add( s.csi1FwdModeSz, (6,2), border=5, flag=wx.TOP|wx.BOTTOM)
        s.csiTxStgs.Add( s.csiTxApplyButton, (7,1), border=5, flag=wx.TOP|wx.BOTTOM)
        
        
        # create static box
        s.csiTxStgsSB = wx.StaticBox( s.panel, -1, "CSI Transmitter Control")
        s.csiTxStgsSBSz = wx.StaticBoxSizer( s.csiTxStgsSB, wx.VERTICAL)
        
        s.csiTxStgsSBSz.Add( s.csiTxStgs, flag=wx.ALL, border=5)

    def CreateForwardingTab(s):
        # Panel heading
        #s.headingTxt = WriteText( s.panel, s.board.longDescription, position=None, bold=True)
        
        # Build the status boxes
        s.BuildForwardingCtlBox()
        s.BuildCSITxBox()
        
        # Update status based on current settings
        s.UpdateForwardingTab()

        # Build second Row
        #s.gpioRow2Sizer = wx.BoxSizer( wx.HORIZONTAL)
        #s.gpioRow2Sizer.Add( s.gpioStsPanel, flag=wx.ALL, border=5 )
        #s.gpioRow2Sizer.Add( s.RX0_BCgpioStgsSBSz, flag=wx.ALL, border=5 )
        #s.gpioRow2Sizer.Add( s.RX1_BCgpioStgsSBSz, flag=wx.ALL, border=5 )
        #s.gpioRow2Sizer.Add( s.RX2_BCgpioStgsSBSz, flag=wx.ALL, border=5 )
        #s.gpioRow2Sizer.Add( s.RX3_BCgpioStgsSBSz, flag=wx.ALL, border=5 )

        # Add the boxes to the panel sizer
        s.forwardingSizer = wx.BoxSizer( wx.VERTICAL)
        
        s.forwardingSizer.Add( s.ForwardStgsSBSz, flag=wx.ALL, border=5 )
        s.forwardingSizer.Add( s.csiTxStgsSBSz, flag=wx.ALL, border=5 )
        #s.forwardingSizer.Add( s.gpioRow2Sizer, flag=wx.ALL, border=5 )
        #s.forwardingSizer.Add( s.FrameSyncSizer, flag=wx.ALL, border=5 )
        
        # Lay it all out
        s.panel.SetAutoLayout( True)
        s.panel.SetSizer( s.forwardingSizer)
        s.forwardingSizer.Layout()
        
        # Polling timer
        #s.timer = wx.Timer( s.panel)
        #wx.EVT_TIMER( s.panel, s.timer.GetId(), s.OnTimer)
        s.timer.Start(STATUS_TIMER)
        
    def UpdateForwardingTab(s):
        # Get Current Forwarding control values
        [s.board.fwd_ctl1_val, s.board.fwd_ctl2_val] = s.board.ReadRXReg(s.board.rx.FWD_CTL1, 2)
        
        s.port0FwdEnableChk.SetValue(not (s.board.fwd_ctl1_val & s.board.rx.FWD_PORT0_DIS))
        s.port0Assign.SetSelection((s.board.fwd_ctl1_val >> 0) & 0x1)

        s.port1FwdEnableChk.SetValue(not (s.board.fwd_ctl1_val & s.board.rx.FWD_PORT1_DIS))
        s.port1Assign.SetSelection((s.board.fwd_ctl1_val >> 1) & 0x1)

        s.port2FwdEnableChk.SetValue(not (s.board.fwd_ctl1_val & s.board.rx.FWD_PORT2_DIS))
        s.port2Assign.SetSelection((s.board.fwd_ctl1_val >> 2) & 0x1)

        s.port3FwdEnableChk.SetValue(not (s.board.fwd_ctl1_val & s.board.rx.FWD_PORT3_DIS))
        s.port3Assign.SetSelection((s.board.fwd_ctl1_val >> 3) & 0x1)

        # Replicate
        s.replicateChk.SetValue(s.board.fwd_ctl2_val & s.board.rx.CSI_REPLICATE_MODE)

        # Current CSI TX controls
        for portIndex in range(2):
            [s.board.csi_ctl1_val[portIndex], s.board.csi_ctl2_val[portIndex]] = \
                s.board.ReadRXCSItxReg(portIndex, s.board.rx.CSI_CTL1,2)

        s.csi0EnableChk.SetValue(s.board.csi_ctl1_val[0] & s.board.rx.CSI_ENABLE)
        s.csi0Rate.SetSelection((s.board.csi_pll_ctl_val[0] >> 0) & 0x3)
        s.csi0Lanes.SetSelection((s.board.csi_ctl1_val[0] >> 4) & 0x3)
        s.csi0contClk.SetSelection((s.board.csi_ctl1_val[0] >> 1) & 0x1)
        csi0_cal_select = ((s.board.csi_ctl1_val[0] >> 6) & 0x1)
        csi0_cal_select |= ((s.board.csi_ctl2_val[0] << 1) & 0x2)
        s.csi0Cal.SetSelection(csi0_cal_select)
        csi0_fwd_mode = (s.board.fwd_ctl2_val & 0x1) | ((s.board.fwd_ctl2_val >> 1) & 0x6)
        if (csi0_fwd_mode > 2):
            if (csi0_fwd_mode == 4):
                csi0_fwd_mode = 3
            elif (csi0_fwd_mode == 6):
                csi0_fwd_mode = 4
            else:
                csi0_fwd_mode = 5
        s.csi0FwdMode.SetSelection(csi0_fwd_mode)
                
        s.csi1EnableChk.SetValue(s.board.csi_ctl1_val[1] & s.board.rx.CSI_ENABLE)
        s.csi1Rate.SetSelection((s.board.csi_pll_ctl_val[1] >> 0) & 0x3)
        s.csi1Lanes.SetSelection((s.board.csi_ctl1_val[1] >> 4) & 0x3)
        s.csi1contClk.SetSelection((s.board.csi_ctl1_val[1] >> 1) & 0x1)
        csi1_cal_select = ((s.board.csi_ctl1_val[1] >> 6) & 0x1)
        csi1_cal_select |= ((s.board.csi_ctl2_val[1] << 1) & 0x2)
        s.csi1Cal.SetSelection(csi1_cal_select)
        csi1_fwd_mode = ((s.board.fwd_ctl2_val >> 1) & 0x1) | ((s.board.fwd_ctl2_val >> 3) & 0x6)
        if (csi1_fwd_mode > 2):
            if (csi1_fwd_mode == 4):
                csi1_fwd_mode = 3
            elif (csi1_fwd_mode == 6):
                csi1_fwd_mode = 4
            else:
                csi1_fwd_mode = 5
        s.csi1FwdMode.SetSelection(csi1_fwd_mode)

        return
        
    def RefreshForwardingTab(s):
        #s.UpdateGPIOSts()
        return

    def txtEvt(s, evt):
        devaddr = int(s.txtCtrl.GetStringSelection(), 16)
        s.board.SetDeviceSlaveAddr(devaddr)
      
    def UpdateCSIRegistersTab(s):
        s.CSIRegGUI = InternalRegistersGUI(s.panel, s.board, s.CSIRegFilePath)
        s.SetCSIRegistersTab()

    def SetCSIRegistersTab(s):
        rdata = s.board.ReadReg(s.board.rx.ANA_IA_CTL)
        rdata &= 0xC3    
        s.board.WriteReg(s.board.rx.ANA_IA_CTL, rdata)
        
    def UpdateCSIANRegistersTab(s):
        s.CSIANRegGUI = InternalRegistersGUI(s.panel, s.board, s.CSIANRegFilePath)
        s.SetCSIANRegistersTab()
        
    def SetCSIANRegistersTab(s):
        rdata = s.board.ReadReg(s.board.rx.ANA_IA_CTL)
        rdata &= ~(0xF << 2) # clear bits 5..2
        rdata |= (0x7 << 2)
        s.board.WriteReg(s.board.rx.ANA_IA_CTL, rdata)

    # def UpdateFPD3RegistersTab(s):
        # s.FPD3RegGUI = InternalRegistersGUI(s.panel, s.board, s.FPD3RegFilePath)
        # rdata = s.board.ReadReg(s.board.tx.ANA_IA_CTL)
        # s.board.WriteReg(s.board.tx.ANA_IA_CTL, (rdata))
        
    def InsertTab(s, pos=1, name=None, updateRoutine=None):
        try:
            if pos == -1:
                # append
                s.deviceTabNames.append(name)
            else:
                # insert
                s.deviceTabNames.insert(pos, name)
            s.tabDispatch[name] = updateRoutine
        except:
            pass