/*
 * tusb926x-Erase-tool-0.2.0.0
 *
 * The software is a Linux-based tool allowing the TUSB926x in-circuit erasing of the SPI Flash via the 
 *   universal serial bus (USB).
 *
 * Copyright (C) 2015 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

#include <stdbool.h>
#include <stdio.h>
#include <string.h>
#include <hid.h>
#include <fcntl.h>
#include <linux/hiddev.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <string>
#include <stdlib.h>
#include <linux/hid.h>
#include <usb.h>
#include <time.h>
#include "SCSI_Erase.cpp"
/*HID structures*/

struct hiddev_devinfo dev_info;
struct hiddev_usage_ref uref;
struct hiddev_field_info finfo;
struct hiddev_report_info rep_info;

/*Global variables*/
bool EFlash, Sel, SelPath, Loop, Print, Help = false;
uint devVID;
;
unsigned int devPID;
uchar PIDTI[2];

int HIDDevicesCount = 0;

char* Device = NULL;
char devlist[10][100] =
{ '0', '0', '0', '0', '0', '0', '0', '0', '0', '0' };

void HIDGetFeatureReport(char* DevName)
{
	int FR, DevNameSize;
	char* FileName = NULL;
	int ret = -1;
	uint i, j;

	uref.value = 1;
	DevNameSize = strlen(DevName);
	FileName = new char[DevNameSize + 10];

	strcpy(FileName, "/dev/usb/");
	strncat(FileName, DevName, DevNameSize);

	FR = open(FileName, O_RDWR);
	delete[] FileName;

	if (ioctl(FR, HIDIOCINITREPORT, 0) >= 0)
	{
		rep_info.report_id = HID_REPORT_ID_FIRST;
		rep_info.report_type = HID_REPORT_TYPE_FEATURE;

		ret = ioctl(FR, HIDIOCGDEVINFO, &dev_info);

		if (ret == -1)
		{
			printf("ERROR:  HIDIOCGDEVINFO  failed\n");
		}
		else
		{
			devVID = dev_info.vendor;
			devPID = dev_info.product & 0x0000FFFF;
		}

		ret = ioctl(FR, HIDIOCGREPORTINFO, &rep_info);

		if (ret == -1)
		{
			printf("ERROR:  HIDIOCGREPORTINFO  failed\n");
		}

		while (ret >= 0)
		{
			for (i = 0; i < rep_info.num_fields; i++)
			{
				finfo.report_type = rep_info.report_type;
				finfo.report_id = rep_info.report_id;
				finfo.field_index = i;
				if (-1 == ioctl(FR, HIDIOCGFIELDINFO, &finfo))
				{
					printf("HIDIOCGFIELDINFO FAILED \n");
				}

				for (j = 0; j < finfo.maxusage; j++)
				{
					uref.report_type = finfo.report_type;
					uref.report_id = finfo.report_id;
					uref.field_index = i;
					uref.usage_index = j;
					if (-1 == ioctl(FR, HIDIOCGUCODE, &uref))
					{
						fprintf(stderr, "HIDIOCGUCODE (%s)\n", strerror(errno));
					}
					if (-1 == ioctl(FR, HIDIOCGUSAGE, &uref))
					{
						fprintf(stderr, "HIDIOCGUSAGE (%s)\n", strerror(errno));
					}
					if (uref.value == 0x60)
					{
						PIDTI[1] = 0x60;
					}
					if (uref.value == 0x92)
					{
						PIDTI[0] = 0x92;
					}

				}

			}
			rep_info.report_id |= HID_REPORT_ID_NEXT;
			ret = ioctl(FR, HIDIOCGREPORTINFO, &rep_info);
		}
	}
	close(FR);
}

void GetHIDDevices(const char *DevNamePrefix)
{
	DIR *dp = NULL;
	struct dirent *ep = NULL;
	char * DevName = NULL;
	char** HIDDevPathName = NULL;
	int i = 0;
	unsigned int DevNamePrefixL = 0;

	int nn = 0;

	HIDDevPathName = new char*[128];
	DevNamePrefixL = strlen(DevNamePrefix);
	HIDDevicesCount = 0;

	dp = opendir("/dev/usb");
	if (dp == NULL)
	{
		delete[] HIDDevPathName;
		return;
	}

	while ((ep = readdir(dp)))
	{
		if ((strlen(ep->d_name)) > DevNamePrefixL)
		{
			DevName = new char[DevNamePrefixL + 1];
			strncpy(DevName, ep->d_name, DevNamePrefixL);
			DevName[DevNamePrefixL] = '\0';

			i = strcmp(DevName, DevNamePrefix);
			delete[] DevName;

			if (i == 0)
			{
				if (DevNamePrefix == "hiddev")
				{
					DevName = new char[strlen(ep->d_name) + 1];
					strncpy(DevName, ep->d_name, strlen(ep->d_name));
					DevName[strlen(ep->d_name)] = '\0';

					HIDGetFeatureReport(DevName);

					if (PIDTI[0] == 0x92 && PIDTI[1] == 0x60)
					{
						if (dev_info.devnum < 10)
						{
							if (devPID != 0x926b)
							{
								nn =
										sprintf(devlist[HIDDevicesCount + 1],
												"VID = 0x0%x / PID = 0x%x %s Bus/Dev=00%x/00%d",
												devVID, devPID, ep->d_name,
												dev_info.busnum,
												dev_info.devnum);
							}
						}
						else
						{
							if (dev_info.devnum < 100)
							{
								if (devPID != 0x926b)
								{
									nn =
											sprintf(
													devlist[HIDDevicesCount + 1],
													"VID = 0x0%x / PID = 0x%x %s Bus/Dev=00%x/0%d",
													devVID, devPID, ep->d_name,
													dev_info.busnum,
													dev_info.devnum);
								}
							}
							else
							{
								if (devPID != 0x926b)
								{
									nn =
											sprintf(
													devlist[HIDDevicesCount + 1],
													"VID = 0x0%x / PID = 0x%x %s Bus/Dev=00%x/%d",
													devVID, devPID, ep->d_name,
													dev_info.busnum,
													dev_info.devnum);
								}
							}
						}
						if (devPID != 0x926b)
						{
							HIDDevPathName[HIDDevicesCount] = DevName;
							HIDDevicesCount++;

							if (Print)
							{
								if (HIDDevicesCount == 1)
									printf("\nTUSB926x Firmware Device Instances (HID Interface) :\n====================================================\n");

								printf("[%d] %s\n", HIDDevicesCount,
										devlist[HIDDevicesCount]);
							}
						}
						else
						{
							delete[] DevName;
						}
					}
				}
			}
		}
	}
	for (unsigned int i0 = 0; i0 < HIDDevicesCount; i0++)
	{
		delete[] HIDDevPathName[i0];
	}
	delete[] HIDDevPathName;
	DevNamePrefix = NULL;

	(void) closedir(dp);
}

char* GetDevName(int selection)  //obtain the bus path device
{
	int DataIndex;
	char* ListDevice = NULL;

	ListDevice = new char[strlen(devlist[selection]) - (strlen(devlist[selection]) - 7) + 1];

	for (DataIndex = 0; DataIndex < (strlen(devlist[selection]) - (strlen(devlist[selection]) - 7)); DataIndex++)
	{
		ListDevice[DataIndex] = devlist[selection][DataIndex + (strlen(devlist[selection]) - 7)];
	}
	ListDevice[7] = '\0';
	return ListDevice;
}

/***************************************/
/* Create a lis of all the hid device  */
/***************************************/

bool UpdateDeviceList()
{

	//find all the TUSB926x HID devices
	GetHIDDevices("hiddev");

	if ((HIDDevicesCount == 0) && (!Loop) && (!Loop) && (SCSIDevicesCount == 0))
	{
		printf("No TUSB9260 firmware devices have been detected.\n\n");
		return false;
	}

	return true;
}

/***************************************************
 * HID reports
 ***************************************************/

char *hid_id[32];

bool device_iterator(struct usb_dev_handle const* usbdev, void* custom,
		unsigned int len)
{
	bool ret = false;
	char current_dev_path[10];
	const struct usb_device *device = usb_device(
			(struct usb_dev_handle *) usbdev);

	//Obtain the device's full path
	sprintf(current_dev_path, "%s/%s", device->bus->dirname, device->filename);

	if (!strcmp(Device, current_dev_path))
	{
		ret = true;
	}

	return ret;
}

void WriteOutputReport(uchar OutReportBuffer[]) //send the repor package to the device.
{
	int i;
	hid_return ret;
	HIDInterface* hid;
	HIDInterfaceMatcher matcher;
	int const PATH_IN[1] =
	{ 0xffb00003 };

	for (i = 0; i < 32; i++)
		hid_id[i] = NULL;

	ret = hid_init();

	hid = hid_new_HIDInterface();
	matcher.vendor_id = 0x0451;
	matcher.product_id = 0x9260;
	matcher.matcher_fn = device_iterator;

	while ((ret = hid_force_open(hid, 0, &matcher, 2))
			!= HID_RET_DEVICE_NOT_FOUND)  //find and open the valid 926x device.
	{
		hid_write_identification(stdout, hid);

		//debug fprintf(stderr,"Sending Report......\n");
		ret = hid_set_output_report(hid, PATH_IN, 1, (char*) OutReportBuffer,
				9);  //send the report package.
		if (ret != HID_RET_SUCCESS)
		{
			fprintf(stderr,
					"hid_set_output_report failed with return code %d\n", ret);
		}

		hid_close(hid);
		break;
	}
	hid_delete_HIDInterface(&hid);
	ret = hid_cleanup();
	return;
}

void EraseFlash(int selection)
{
	int status;
	char CurrentProccess[1][1024];

	// if the device pat is specified, then verify if the path exist.
	if (!SelPath)
		Device = GetDevName(selection);

	printf("\nErasing flash of device %s\n\n", Device);
	// debug printf ("Device =%s\n", Device);
	status = sprintf(CurrentProccess[0],
			"issuing HID Enable Re-program command");
	uchar OutBuff[9] =
	{ 0x07, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
	WriteOutputReport(OutBuff);

	//Poison the Flash

	status = sprintf(CurrentProccess[0],
			"issuing HID Enable Re-program command");
	OutBuff[0] = 0x03;
	OutBuff[1] = 0x00;
	WriteOutputReport(OutBuff);
	sleep(4);

	//Erase Flash
	printf("Device =%s\n", Device);
	OutBuff[0] = 0x02;
	OutBuff[1] = 0x00;
	WriteOutputReport(OutBuff);

	sleep(4);
	printf("\nErase flash succeeded.\n");
}

void Print_Help()
{
	printf(" Texas Instruments TUSB926x Erase Flash syntax :\n\n");
	printf(
			" 	TUSB926x_EraseFlash.exe [-s <Instance Number>] [-sp <Path>] [-p] [-?]\n\n");

	printf(
			" -ef       - Enables the utility to erase a device's flash. This\n");
	printf("			  	    parameter is used to ensure a device flash is not\n");
	printf("				    accidentally erased.\n");
	printf(
			" -s <Instance>    - Selects an specific TUSB9260x Firmware device instance\n");
	printf(
			"                    to erase. Available device instances can be displayed\n");
	printf("                    using the /p command line parameter.\n");
	printf(" -sp <Path>       - Selects a TUSB9260x device path.\n");
	printf("                    Available device paths\n");
	printf(
			"                    can be displayed using the -p command line parameter.\n");
	printf(
			" -p               - Print a list of the available TUSB926x Firmware\n");
	printf("                    instances in your system.\n");
	printf(" -?               - Show this help screen.\n");

	printf("Examples:\n");
	printf("- Erasing all the TUSB926x devices in your system :\n");
	printf("./TUSB926x_EraseFlash.exe -ef\n");
	printf(
			"- Erasing an specific TUSB926x device by selecting its HID device instance:\n");
	printf("./TUSB926x_EraseFlash.exe -ef -s 1\n");
	printf("- Erasing any TUSB926x device having an specific HID Bus Path\n");
	printf("./TUSB926x_EraseFlash.exe -ef -sp 001/002\n");
	printf(
			"- Erasing any TUSB926x device having an specific USB Mass Storage Path\n");
	printf("./TUSB926x_EraseFlash.exe -ef -sp /dev/sg1\n");

}

int main(int argc, char *argv[])
{
	int selection = 0;
	int dev_it = 0;
	int arg_iteration = 0;
	char* seldev;

	/*Validate arguments*/

	arg_iteration = 1;

	printf("\nTUSB926x Lnux Command Line Erase Flash version 1.0.0.0\n 2013 <C> Texas Instruments Inc. All Rights Reserved.\n\n");

	if (argc == 1)
	{
		return 1;
	}
	else
	{ /*Validate arguments*/
		for (arg_iteration = 1; arg_iteration < argc; arg_iteration++)
		{
			if (!strcmp(argv[arg_iteration], "-ef"))
			{
				if (!EFlash)
				{
					EFlash = true;
					printf("/ef - Erase Flash = True\n");
				}
			}
			else if (!strcmp(argv[arg_iteration], "-s"))
			{
				if (!Sel)
				{
					arg_iteration++;

					if (arg_iteration < argc)
					{
						Sel = true;

						selection = atoi(argv[arg_iteration]);
						printf("/s - Selected Device = %s\n",
								argv[arg_iteration]);
						if (!selection)
						{
							printf("\nInvalid device\n");
							return 0;
						}
					}
					else
					{
						printf("/s - Selected Device = Missing Argument\n");
						return 1;
					}
				}
			}
			else if (!strcmp(argv[arg_iteration], "-sp"))
			{
				if (!SelPath)
				{
					arg_iteration++;

					if (arg_iteration < argc)
					{
						SelPath = true;
						printf("/sp - Selected Device Path = %s\n",
								argv[arg_iteration]);
						Device = argv[arg_iteration];
					}
					else
						printf("/sp - Selected Device Path = Missing Argument\n");
				}
			}
			else if (!strcmp(argv[arg_iteration], "-l"))
			{
				if (!Loop)
				{
					Loop = true;
					printf("/l - Loop execution = True\n");
				}
			}
			else if (!strcmp(argv[arg_iteration], "-p"))
			{
				if (!Print)
				{
					Print = true;
				}
			}
			else if (!strcmp(argv[arg_iteration], "-?"))
			{
				if (!Help)
				{
					Help = true;
				}
			}
			else
			{
				if (argc != 0)
					printf("Unknow Parameter = %s\n", argv[arg_iteration]);
				return 1;
			}

		}

		if (Loop && (Sel || SelPath))
		{
			printf("\nYou cannot loop the utility execution and specify a device instance too\n");
			return 1;
		}

		//execute selected commands

		if (Help)
		{
			Print_Help();
		}

		if (Print)
		{
			UpdateSCSIList();

			if (!UpdateDeviceList())
			{
				return 1;
			}

			if (SCSIDevicesCount > 0 && Print)
			{
				printf("\nTUSB926x Firmware Device Instances (USB Mass Storage) :\n====================================================\n");

				for (dev_it = 1; dev_it <= SCSIDevicesCount; dev_it++)
				{
					status = SCSI_inq(SCSIdevlist[dev_it], dev_it,
							HIDDevicesCount);
				}
			}

			Print = false;
		}

		if (Sel)
		{
			if (EFlash)
			{
				UpdateSCSIList();
				if (!UpdateDeviceList())
				{
					//return 0;
				}

				printf("DevicesCount=%d\n", HIDDevicesCount + SCSIDevicesCount);
				if (selection > (HIDDevicesCount + SCSIDevicesCount))
				{
					printf("\nInvalid Selected device\n");
					return 1;
				}
				else if (selection > HIDDevicesCount)
				{
					printf("Erasing flash of SCSI device %d\n", selection);
					EraseFlash_SCSI(selection - HIDDevicesCount);
				}
				else
				{
					printf("Erasing flash of HID device %d\n", selection);
				}
			}
		}

		if (SelPath)
		{
			char* DevicePath = NULL;
			int DevIteration = 0;

			if (EFlash)
			{
				UpdateSCSIList();

				if (!UpdateDeviceList())
				{
					return 1;
				}
				// search for the specified path

				for (DevIteration = HIDDevicesCount; DevIteration > 0; DevIteration--)
				{
					if (DevicePath != NULL)
					{
						delete[] DevicePath;
					}
					DevicePath = GetDevName(DevIteration);

					if (!strcmp(DevicePath, Device))
						break;
				}

				if (DevIteration != 0)
				{
					EraseFlash(0);
				}
				else
				{
					delete[] DevicePath;
					for (DevIteration = SCSIDevicesCount; DevIteration > 0; DevIteration--)
					{
						DevicePath = SCSIdevlist[DevIteration];

						if (!strcmp(DevicePath, Device))
							break;
					}
					if (DevIteration != 0)
					{
						EraseFlash_SCSI(DevIteration);
					}
					else
						printf("Invalid Device Selected\n");

				}

				if (DevIteration != 0)
				{
					delete[] DevicePath;

				}

			}

		}

		if (EFlash && !SelPath && !Sel)
		{
			int DevIteration = 0;
			bool status;

			SCSIDevicesCount = 1;

			if (!UpdateDeviceList())
			{

				if (HIDDevicesCount > 0)
				{
					for (DevIteration = HIDDevicesCount; DevIteration > 0;
							DevIteration--)
					{
						EraseFlash(DevIteration);
					}
				}
			}

			SCSIDevicesCount = 0;

			UpdateSCSIList();

			if (SCSIDevicesCount > 0)
			{
				for (DevIteration = SCSIDevicesCount; DevIteration > 0; DevIteration--)
				{
					EraseFlash_SCSI(DevIteration);
				}
			}

		}

		if ((HIDDevicesCount == 0) && (SCSIDevicesCount == 0))
		{
			printf("No TUSB9260 firmware devices have been detected.\n\n");
			return 1;
		}
		return 0;

	}

	return 0;
}
