/*
 * Texas Instruments EEPROM Burner Driver (Based on USB Skeleton driver - 2.0)
 *
 * Copyright (C) 2015 Texas Instruments Incorporated - http://www.ti.com/
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation version 2.
 *
 * This program is distributed "as is" WITHOUT ANY WARRANTY of any
 * kind, whether express or implied; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
 
//#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/module.h>
#include <linux/kref.h>
#include <linux/fs.h>
#include <linux/usb.h>
#include <asm/uaccess.h>
#include <linux/ioctl.h>
#include "eBurner.h"

// Define the Bulk Out endpoint we'll be writing to.
#define BULK_OUT_ENDPOINT 0

// Define Vendor and Product ID values to match our device
#define USB_SKEL_VENDOR_ID	0x0451
#define USB_SKEL_PRODUCT_ID	0x926B

// Table of devices that work with this driver
static struct usb_device_id eBurner_table [] = {
	{ USB_DEVICE(USB_SKEL_VENDOR_ID, USB_SKEL_PRODUCT_ID) },
	{ }															// Terminating entry 
};
MODULE_DEVICE_TABLE (usb, eBurner_table);

// Get a minor range for our device from the usb maintainer 
#define USB_SKEL_MINOR_BASE	192

// Structure to hold all of our device specific stuff 
struct usb_eBurner
 {
	struct	usb_device *	udev;			// USB device for this device
	struct	usb_interface *	interface;		// Interface for this device
	__u8	bulk_out_endpointAddr;			// Address of the bulk out endpoint
	struct	kref kref;
 };
#define to_eBurner_dev(d) container_of(d, struct usb_eBurner, kref)
struct usb_eBurner *dev = NULL;

static struct usb_driver eBurner_driver;


static void eBurner_delete(struct kref *kref)
{	
	struct usb_eBurner *dev = to_eBurner_dev(kref);

	usb_put_dev(dev->udev);
    kfree (dev);
}

static int eBurner_open(struct inode *inode, struct file *file)
{
	struct usb_eBurner *dev;
	struct usb_interface *interface;
	int subminor;
	int retval = 0;
	
	subminor = iminor(inode);

	interface = usb_find_interface(&eBurner_driver, subminor);
	if (!interface) 
	{
		pr_err ("%s - error, can't find device for minor %d", __FUNCTION__, subminor);
		retval = -ENODEV;
		goto exit;
	}

	dev = usb_get_intfdata(interface);
	if (!dev) 
	{
		retval = -ENODEV;
		goto exit;
	}
	
	// increment our usage count for the device 
	kref_get(&dev->kref);

	//save our object in the file's private structure 
	file->private_data = dev;

exit:
	return retval;
}

static int eBurner_release(struct inode *inode, struct file *file)
{
	struct usb_eBurner *dev;
	
	dev = (struct usb_eBurner *)file->private_data;
	if (dev == NULL)
	{
		return -ENODEV;
	}

	// decrement the count on our device 
	kref_put(&dev->kref, eBurner_delete);
	return 0;
}

/*
static void eBurner_write_bulk_callback(struct urb *urb, struct pt_regs *regs)
{
	//sync/async unlink faults aren't errors 
	if (urb->status && 
	    !(urb->status == -ENOENT || 
	      urb->status == -ECONNRESET ||
	      urb->status == -ESHUTDOWN)) {
		dbg("%s - nonzero write bulk status received: %d",
		    __FUNCTION__, urb->status);
	}

	// free up our allocated buffer 
	usb_buffer_free(urb->dev, urb->transfer_buffer_length, 
			urb->transfer_buffer, urb->transfer_dma);
}

static ssize_t eBurner_write(struct file *file, const char __user *user_buffer, size_t count, loff_t *ppos)
{
	struct usb_eBurner *dev;
	int retval = 0;
	struct urb *urb = NULL;
	char *buf = NULL;

	dev = (struct usb_eBurner *)file->private_data;

	//verify that we actually have some data to write 
	if (count == 0)
		goto exit;

	//create a urb, and a buffer for it, and copy the data to the urb 
	urb = usb_alloc_urb(0, GFP_KERNEL);
	if (!urb) {
		retval = -ENOMEM;
		goto error;
	}

	buf = usb_buffer_alloc(dev->udev, count, GFP_KERNEL, &urb->transfer_dma);
	if (!buf) {
		retval = -ENOMEM;
		goto error;
	}
	if (copy_from_user(buf, user_buffer, count)) {
		retval = -EFAULT;
		goto error;
	}

	//initialize the urb properly 
	usb_fill_bulk_urb(urb, dev->udev,
			  usb_sndbulkpipe(dev->udev, dev->bulk_out_endpointAddr),
			  buf, count, eBurner_write_bulk_callback, dev);
	urb->transfer_flags |= URB_NO_TRANSFER_DMA_MAP;

	//send the data out the bulk port 
	retval = usb_submit_urb(urb, GFP_KERNEL);
	if (retval) {
		err("%s - failed submitting write urb, error %d", __FUNCTION__, retval);
		goto error;
	}

	// release our reference to this urb, the USB core will eventually free it entirely 
	usb_free_urb(urb);

exit:
	return count;

error:
	usb_buffer_free(dev->udev, count, buf, urb->transfer_dma);
	usb_free_urb(urb);
	kfree(buf);
	return retval;
}*/

int eBurner_ioctl (/*struct inode *inode,*/ struct file *filp, unsigned int cmd, Output_Buffer *arg)
{
//	struct file *OutFile;	
	int retval = 0;
	size_t count;
	unsigned char *LocalBuffer;
	
	printk(KERN_DEBUG "calling ioctl.----");
	printk(KERN_DEBUG "cmd = %d",cmd);
	switch(cmd) 
	{
	    case DOWNLOAD_ROM:
			printk(KERN_DEBUG "DOWNLOAD_ROM ioctl.");			
						
			LocalBuffer = kmalloc (arg->BufferSize, GFP_KERNEL);
			memcpy(LocalBuffer, arg->Buffer, arg->BufferSize);			
			
			retval = usb_bulk_msg(dev->udev,
			      usb_sndbulkpipe(dev->udev, dev->bulk_out_endpointAddr),
			      LocalBuffer,
			      arg->BufferSize,
			      &count, HZ*10);

			if(retval == 0)
			{
				printk(KERN_DEBUG "Buffer Data has been sent successfully.");
			}			

			return 0;
	}
	return -ENOTTY; // unknown command 
}

static struct file_operations eBurner_fops = {
	.owner =	THIS_MODULE,
	.unlocked_ioctl =	eBurner_ioctl,
//	.ioctl =	eBurner_ioctl,
	//.write =	eBurner_write,
	.open =		eBurner_open,
	.release =	eBurner_release
};

static struct usb_class_driver eBurner_class = {
	.name = "usb/TUSB9260_%d",
	.fops = &eBurner_fops,
	.minor_base = USB_SKEL_MINOR_BASE,
};

static int eBurner_probe(struct usb_interface *interface, const struct usb_device_id *id)
{	
	struct usb_host_interface *iface_desc;
	struct usb_endpoint_descriptor *endpoint;
	//size_t buffer_size;
	int retval = -ENOMEM;

	// Allocate memory for our device state and initialize it
	dev = kmalloc(sizeof(struct usb_eBurner), GFP_KERNEL);
	if (dev == NULL) 
	{
		pr_err("Not enough memory to initialize our device.");
		goto error;
	}
	memset(dev, 0x00, sizeof (*dev));
	kref_init(&dev->kref);

	dev->udev = usb_get_dev(interface_to_usbdev(interface));
	dev->interface = interface;
	
	iface_desc = interface->cur_altsetting;
	endpoint = &iface_desc->endpoint[BULK_OUT_ENDPOINT].desc;
	if (!dev->bulk_out_endpointAddr && usb_endpoint_is_bulk_out(endpoint))
	{
		// We found a bulk out endpoint		
		dev->bulk_out_endpointAddr = endpoint->bEndpointAddress;
		pr_err("We've found our Bulk Out Endpoint at %d.\n", endpoint->bEndpointAddress);
	}
	
	if (!dev->bulk_out_endpointAddr)
	{
		pr_err("Couldn't find the selected bulk-out endpoint.");
		goto error;
	}

	// Save our data pointer in this interface device
	usb_set_intfdata(interface, dev);

	// We can register the device now, as it is ready
	retval = usb_register_dev(interface, &eBurner_class);
	if (retval) 
	{
		//something prevented us from registering this driver 
		pr_err("Not able to get a minor for this device.");
		usb_set_intfdata(interface, NULL);
		goto error;
	}

	// let the user know what node this device is now attached
	pr_err("TUSB9260 device is now attached - %d", interface->minor);
	return 0;

error:
	if (dev)
	{
		kref_put(&dev->kref, eBurner_delete);
	}
	return retval;
}

static void eBurner_disconnect(struct usb_interface *interface)
{
	struct usb_eBurner *dev;
	int minor = interface->minor;

	// Prevent eBurner_open() from racing eBurner_disconnect()
//	lock_kernel();

	dev = usb_get_intfdata(interface);
	usb_set_intfdata(interface, NULL);

	// Give back our minor
	usb_deregister_dev(interface, &eBurner_class);

//	unlock_kernel();

	// Decrement our usage count
	kref_put(&dev->kref, eBurner_delete);

	pr_err("TUSB9260 #%d now disconnected", minor);
}

static struct usb_driver eBurner_driver = {
	.name 		= 	"eBurner",
	.id_table 	= 	eBurner_table,
	.probe 		= 	eBurner_probe,
	.disconnect =	eBurner_disconnect,
};

static int __init usb_eBurner_init(void)
{
	int result;

	// Register this driver with the USB subsystem
	result = usb_register(&eBurner_driver);
	if (result)
	{
		pr_err("usb_register failed. Error number %d", result);
	}
	return result;
}

static void __exit usb_eBurner_exit(void)
{
	// Deregister this driver with the USB subsystem
	pr_err("De-registering TUSB9260 driver");
	usb_deregister(&eBurner_driver);
}

module_init (usb_eBurner_init);
module_exit (usb_eBurner_exit);

MODULE_LICENSE("GPL");
