#include <i2c_t3.h>
#include "INA233.h"

INA233::INA233(uint8_t addr) {
  ina233_i2caddr = addr;
}

bool INA233::begin(long Speed) {
  Wire.begin();
  //if(Speed > 100000){ Speed = 100000; }// INA233 has weird bugs above this speed http://e2e.ti.com/support/amplifiers/f/14/p/780582/2890588#2890588
  if(Speed < 10000){ Speed = 10000; }
  Wire.setClock(Speed);
  delay(1);
  if(!checkChipInfo()){ return 0; }
  resetRegisters();
  clearFaults();
  EINAutoClear(true); //required for this library to work properly
  uint8_t val = 0;
  wireReadByte(MFR_DEVICE_CONFIG, &val);
  //EINAccumulatorMode(0);
  //ADCtriggered(true);
}

void INA233::ADCtrigger(){
  wireWriteWord(MFR_ADC_CONFIG,MFR_ADC_CONFIG_VAL);
}

//---------------------------------------------------------------------------------------------------------------------------------------------------
//---------------------------------------------------------------------------------------------------------------------------------------------------
//-------     Calibration and Configuration         -------------------------------------------------------------------------------------------------
//---------------------------------------------------------------------------------------------------------------------------------------------------
//---------------------------------------------------------------------------------------------------------------------------------------------------

void INA233::setAlertPinMode(bool ConversionReady_en,bool ADCoverflow_en,bool PORevent_en, bool ComFault_en, bool OPwarn_en, bool OCwarn_en, bool OVwarn_en, bool UVwarn_en){
  uint8_t val = 0;//default
  val = val | (((uint8_t)(!ConversionReady_en)) << 7);
  val = val | (((uint8_t)(!ADCoverflow_en)    ) << 6);
  val = val | (((uint8_t)(!PORevent_en)       ) << 5);
  val = val | (((uint8_t)(!ComFault_en)       ) << 4);
  val = val | (((uint8_t)(!OPwarn_en)         ) << 3);
  val = val | (((uint8_t)(!OCwarn_en)         ) << 2);
  val = val | (((uint8_t)(!OVwarn_en)         ) << 1);
  val = val | (((uint8_t)(!UVwarn_en)         ) << 0);
  wireWriteByte(MFR_ALERT_MASK, val);
}

bool INA233::EINPolarityIncorrect(){
  uint8_t in = 0;
  wireReadByte(MFR_DEVICE_CONFIG, &in);
  wireModifyByte(MFR_DEVICE_CONFIG, 0, 1, 7); //reset the bit to 0
  return ((bool)(in & 128)); //1000000
}

//0 = Pos and Neg Currents add to the accumulator
//1 = Pos Currents add to the accumulator. Neg currents latches EINPolarityIncorrect true.
//2 = Neg Currents add to the accumulator. Pos currents latches EINPolarityIncorrect true.
void INA233::EINAccumulatorMode(uint8_t mode){
  wireModifyByte(MFR_DEVICE_CONFIG, mode, 2, 4);
}


void INA233::I2CFilter(bool en){
  wireModifyByte(MFR_DEVICE_CONFIG, en, 1, 3);
}
void INA233::EINAutoClear(bool en){
  wireModifyByte(MFR_DEVICE_CONFIG, en, 1, 2);
}
void INA233::alertBehavior(bool en){
  wireModifyByte(MFR_DEVICE_CONFIG, en, 1, 1);
}
void INA233::alertPolarity(bool en){
  wireModifyByte(MFR_DEVICE_CONFIG, en, 1, 0);
}

//0=140us 1=204us 2=332us 3=588us 4=1.1ms 5=2.116ms 6=4.156ms 7=8.244ms
void INA233::ADCShuntConversionTime(uint8_t mode){
  MFR_ADC_CONFIG_VAL = wireModifyWord(MFR_ADC_CONFIG,mode,3,3);
}
//0=140us 1=204us 2=332us 3=588us 4=1.1ms 5=2.116ms 6=4.156ms 7=8.244ms
void INA233::ADCVoltageConversionTime(uint8_t mode){
  MFR_ADC_CONFIG_VAL = wireModifyWord(MFR_ADC_CONFIG,mode,3,6);
}
//0=1 1=4 2=16 3=64 4=128 5=256 6=512 7=1024
void INA233::ADCAveraging(uint8_t mode){
  MFR_ADC_CONFIG_VAL = wireModifyWord(MFR_ADC_CONFIG,mode,3,9);
}
void INA233::ADCTriggeredMode(bool en){
  MFR_ADC_CONFIG_VAL = wireModifyWord(MFR_ADC_CONFIG,(!en),1,2); //sends 0 for triggerd mode
}

void INA233::ADCEnable(bool bus_en, bool shunt_en){
  uint16_t val = 0; //00
  val = val | ((uint16_t)bus_en) << 1;
  val = val | ((uint16_t)shunt_en);
  MFR_ADC_CONFIG_VAL = wireModifyWord(MFR_ADC_CONFIG,val,2,0);
}

void INA233::clearFaults(){ //working
  wireSendCmd(CLEAR_FAULTS);
}

void INA233::resetRegisters(){
  wireSendCmd(RESTORE_DEFAULT_ALL);
}

float INA233::setOverCurrentThreshold(float current){ //working
  uint16_t code = 0;
  uint16_t maskBits = 32760; //01111111,11111000
  float MAX = CURRENT_LSB * maskBits;

       if(current <= 0)  { code = 0; }
  else if(current >= MAX){ code = maskBits; } 
  else                   { code = (uint16_t)abs(current / CURRENT_LSB); }
  code = code & maskBits;

  wireWriteWord(IOUT_OC_WARN_LIMIT,code);
  
  return (CURRENT_LSB * code); //the actual limit set
}

float INA233::setOverVoltageThreshold(float voltage){
  uint16_t code = 0;
  uint16_t maskBits = 32760; //01111111,11111000
  float MAX = VOLTAGE_LSB * maskBits;

       if(voltage <= 0)  { code = 0; }
  else if(voltage >= MAX){ code = maskBits; } 
  else                   { code = (uint16_t)abs(voltage / VOLTAGE_LSB); }
  code = code & maskBits;

  wireWriteWord(VIN_OV_WARN_LIMIT,code);
  
  return (VOLTAGE_LSB * code);//the actual limit set
}

float INA233::setUnderVoltageThreshold(float voltage){ //working
  uint16_t code = 0;
  uint16_t maskBits = 32760; //01111111,11111000
  float MAX = VOLTAGE_LSB * maskBits;

       if(voltage <= 0)  { code = 0; }
  else if(voltage >= MAX){ code = maskBits; } 
  else                   { code = (uint16_t)abs(voltage / VOLTAGE_LSB); }
  code = code & maskBits;

  wireWriteWord(VIN_UV_WARN_LIMIT,code);
  
  return (VOLTAGE_LSB * code);//the actual limit set
}

float INA233::setOverPowerThreshold(float power){ //working
  uint16_t code = 0;
  uint16_t maskBits = 65520; //11111111,11110000
  float MAX = POWER_LSB * maskBits;

       if(power <= 0)  { code = 0; }
  else if(power >= MAX){ code = maskBits; } 
  else                 { code = (uint16_t)abs(power / POWER_LSB); }
  code = code & maskBits;

  wireWriteWord(PIN_OP_WARN_LIMIT,code);
  
  return (POWER_LSB * code);//the actual limit set
}


bool INA233::checkChipInfo(){
  uint16_t IDval =0;
  uint16_t MODELval = 0;

  wireReadWord(TI_MFR_ID,&IDval);
  wireReadWord(TI_MFR_MODEL,&MODELval);

  if(IDval != 21577) { return false; } //"TI" in ASCII
  if(MODELval != 13107) { return false; } //"33" in ASCII
  //Revision number is not checked
  return true;
}

byte INA233::updateFaults(){
  wireReadByte(STATUS_MFR_SPECIFIC, &FAULT_BYTE);
  return FAULT_BYTE;
}

bool INA233::getConversionReady(){ 
  return (bool)( FAULT_BYTE & 128 ); //10000000
}
bool INA233::getArithmeticOverflow(){
  return (bool)( FAULT_BYTE & 64 ); //01000000
}
bool INA233::getPowerOnReset(){
  return (bool)( FAULT_BYTE & 32 ); //00100000
}
bool INA233::getComFault(){
  return (bool)( FAULT_BYTE & 16 ); //00010000
}
bool INA233::getOverPower(){
  return (bool)( FAULT_BYTE & 8 ); //00001000
}
bool INA233::getOverCurrent(){
  return (bool)( FAULT_BYTE & 4 ); //00000100
}
bool INA233::getOverVoltage(){
  return (bool)( FAULT_BYTE & 2 ); //00000010
}
bool INA233::getUnderVoltage(){
  return (bool)( FAULT_BYTE & 1 ); //00000001
}


void INA233::printCalibration(){
  Serial.print("CAL:\t\t");   Serial.println(CAL);
  Serial.print("CURRENT_LSB:\t\t");         Serial.println(CURRENT_LSB,5);
  Serial.print("CURRENT_SCALE_FACTOR:\t");  Serial.println(CURRENT_SCALE_FACTOR,5);
  Serial.print("CURRENT_GAIN_TWEEK:\t");    Serial.println(CURRENT_GAIN_TWEEK,5);
  Serial.print("CURRENT_OFFSET:\t\t");      Serial.println(CURRENT_OFFSET,5);
  Serial.print("VOLTAGE_LSB:\t\t");         Serial.println(VOLTAGE_LSB,5);
  Serial.print("VOLTAGE_SCALE_FACTOR:\t");  Serial.println(VOLTAGE_SCALE_FACTOR,5);
  Serial.print("VOLTAGE_GAIN_TWEEK:\t");    Serial.println(VOLTAGE_GAIN_TWEEK,5);
  Serial.print("VOLTAGE_OFFSET:\t\t");      Serial.println(VOLTAGE_OFFSET,5);
  Serial.print("POWER_LSB:\t\t");           Serial.println(POWER_LSB,5);

}

//CAL number other than 2048 can make the output skip CODES like 0,3,5,8,10 instead of 1,2,3,4,5
void INA233::calibrateCurrent(uint16_t cal, float C_scale, float C_gainADJ, float C_offset){
  CURRENT_SCALE_FACTOR = C_scale;
  CURRENT_GAIN_TWEEK = C_gainADJ;
  CURRENT_OFFSET = C_offset;
  CURRENT_LSB = (CURRENT_SCALE_FACTOR * CURRENT_GAIN_TWEEK) / 1000;
  CURRENT_MAX = CURRENT_LSB * 32768;
  CURRENT_MIN = 0 - CURRENT_MAX;
  
  CAL = cal;
  wireWriteWord(MFR_CALIBRATION, (uint16_t)CAL);

  calibratePower();
}

void INA233::calibrateVoltage(float V_scale, float V_gainADJ, float V_offset){
  
  VOLTAGE_SCALE_FACTOR = V_scale;
  VOLTAGE_GAIN_TWEEK = V_gainADJ;
  VOLTAGE_OFFSET = V_offset;
  VOLTAGE_LSB = (1.25 * VOLTAGE_SCALE_FACTOR * VOLTAGE_GAIN_TWEEK) / 1000; //thats 1.25mV LSB, then its multipyed by our correction
  VOLTAGE_MAX = VOLTAGE_LSB * 32768;
  VOLTAGE_MIN = 0 - VOLTAGE_MAX;

  calibratePower();
}

void INA233::calibratePower(){
  
 float wlsb = 25 *  CURRENT_LSB; 
 POWER_LSB =  wlsb * VOLTAGE_SCALE_FACTOR * VOLTAGE_GAIN_TWEEK; //we need to add the voltage scaling correction that the INA dosent know about;
}


//---------------------------------------------------------------------------------------------------------------------------------------------------
//---------------------------------------------------------------------------------------------------------------------------------------------------
//-------     Data Gets      ------------------------------------------------------------------------------------------------------------------------
//---------------------------------------------------------------------------------------------------------------------------------------------------
//---------------------------------------------------------------------------------------------------------------------------------------------------
void INA233::serviceEnergyMonitor(){
  double wattSeconds = 0;
  double ampSeconds = 0;
  uint32_t rawEnergy = 0;
  uint32_t sampleCount = 0;
  uint32_t sampleLength = 0;
  
  float current = getCurrent_A();
  float voltage = getBusVoltage_V();
  uint32_t readTime = micros();
  getEnergy_raw(&sampleCount,&rawEnergy);

  float averageVoltage = (voltage + prevVoltage) / 2;
  bool currentDirection = (current > 0); //positive Current = 1
  
  if(readTime < prevReadTime){ //micros rollover protection
    sampleLength = (4294967295 - prevReadTime) + readTime;
  }else{
    sampleLength = readTime - prevReadTime;
  }
  
  if(sampleCount!=0){
    wattSeconds = (POWER_LSB * ((double)rawEnergy/(double)sampleCount)) * ((double)sampleLength/(double)1000000);
    ampSeconds = wattSeconds / averageVoltage;
  }
  


  if(currentDirection){
    WATT_SECONDS_POS = WATT_SECONDS_POS + wattSeconds;
    AMP_SECONDS_POS = AMP_SECONDS_POS + ampSeconds;
    WATT_SECONDS_TOT = WATT_SECONDS_TOT + wattSeconds;
    AMP_SECONDS_TOT = AMP_SECONDS_TOT + ampSeconds;
  }else{
    WATT_SECONDS_NEG = WATT_SECONDS_NEG + wattSeconds;
    AMP_SECONDS_NEG = AMP_SECONDS_NEG + ampSeconds;
    WATT_SECONDS_TOT = WATT_SECONDS_TOT - wattSeconds;
    AMP_SECONDS_TOT = AMP_SECONDS_TOT - ampSeconds;
  }


  prevReadTime = readTime;
  prevVoltage = voltage;

  p("aV: "); p(averageVoltage,3);p("\t");
  p("wS: "); p(wattSeconds,3);p("\t");
  p("aS: "); p(ampSeconds,4);p("\t");
  p("rawE: "); p(rawEnergy);p("\t");
  p("Cnt: "); p(sampleCount);p("\t");
  p("Len: "); p(sampleLength);p("\t");
  
  pl();
  p("WsTOT: ");p(WATT_SECONDS_TOT,3);p("\t");
  p("WsPOS: ");p(WATT_SECONDS_POS,3);p("\t");
  p("WsNEG: ");p(WATT_SECONDS_NEG,3);p("\t");
  p("AsTOT: ");p(AMP_SECONDS_TOT,3);p("\t");
  p("AsPOS: ");p(AMP_SECONDS_POS,3);p("\t");
  p("AsNEG: ");p(AMP_SECONDS_NEG,3);p("\t");
  pl();

}

int16_t INA233::getBusVoltage_raw() {
  uint16_t value;
  wireReadWord(READ_VIN, &value);
  return (int16_t)value;
}

int16_t INA233::getShuntVoltage_raw() {
  uint16_t value;
  wireReadWord(MFR_READ_VSHUNT, &value);
  return (int16_t)value;
}

int16_t INA233::getCurrent_raw() {
  uint16_t value;
  wireReadWord(READ_IIN, &value);
  return (int16_t)value;
}

int16_t INA233::getPower_raw() {
  uint16_t value;
  wireReadWord(READ_PIN, &value);
  return (int16_t)value;
}

void INA233::getEnergy_raw(uint32_t* sampleCount,uint32_t* rawEnergy) {
  uint8_t value[6];
  wireReadBlock(READ_EIN, value);
  *sampleCount = ((uint32_t)value[5] << 16) | ((uint32_t)value[4] << 8) | ((uint32_t)value[3]);
  *rawEnergy = ((uint32_t)value[2] << 16) | ((uint32_t)value[1] << 8) | ((uint32_t)value[0]);
}



float INA233::getShuntVoltage_mV() {
  int16_t value=getShuntVoltage_raw();
  float vshunt;
  vshunt=(value*pow(10,-R_vs)-b_vs)/m_vs;    //m_vs=4  R_vs=5  b_vs=0
  return vshunt * 1000;
}

float INA233::getBusVoltage_V() {
  int16_t value = getBusVoltage_raw();
  float vbus;
  vbus =((value*pow(10,-R_vb)-b_vb)/m_vb   * VOLTAGE_SCALE_FACTOR * VOLTAGE_GAIN_TWEEK) + VOLTAGE_OFFSET;  //m_vb=8  R_vb=2  b_vb=0
  return vbus;
}

float INA233::getCurrent_A() {
  int16_t value = getCurrent_raw();
  float current;
  current = ((value * CURRENT_LSB) + CURRENT_OFFSET);
  return current;
}

float INA233::getPower_W() {
  uint16_t value=getPower_raw();
  float power;
  power = (value * POWER_LSB);
  return power;
}


//---------------------------------------------------------------------------------------------------------------------------------------------------
//---------------------------------------------------------------------------------------------------------------------------------------------------
//-------        Low Level I2C                      -------------------------------------------------------------------------------------------------
//---------------------------------------------------------------------------------------------------------------------------------------------------
//---------------------------------------------------------------------------------------------------------------------------------------------------
void INA233::wireSendCmd (uint8_t reg)
{
  Wire.beginTransmission(ina233_i2caddr);
  Wire.write(reg);                       // PMBus command
  Wire.endTransmission();

  if(debug){debugI2C(2,reg,0);}
}

void INA233::wireWriteByte (uint8_t reg, uint8_t value)
{
  Wire.beginTransmission(ina233_i2caddr);
  Wire.write(reg);                       // PMBus command
  Wire.write(value);                     // byte to write
  Wire.endTransmission();

  if(debug){debugI2C(1,reg,value);}
}

void INA233::wireWriteWord (uint8_t reg, uint16_t value)
{
  Wire.beginTransmission(ina233_i2caddr);
  Wire.write(reg);                       // PMBus command
  Wire.write(value & 0xFF);              // Lower 8-bits
  Wire.write((value >> 8) & 0xFF);       // Upper 8-bits
  Wire.endTransmission();

  if(debug){debugI2C(1,reg,value);}
}

void INA233::wireReadBlock(uint8_t reg, uint8_t value[6])
{
  int i;
  uint8_t block_size;

  Wire.beginTransmission(ina233_i2caddr);
  Wire.write(reg);
  Wire.endTransmission(false);
  Wire.requestFrom(ina233_i2caddr,7, true);
  
  block_size=Wire.read();
  for (i=0;i<block_size;i++)
  {
    value[i]=Wire.read();
  }
  
  if(debug){
    debugI2C(0,reg,value[0]);
    debugI2C(0,0,value[1]);
    debugI2C(0,0,value[2]);
    debugI2C(0,0,value[3]);
    debugI2C(0,0,value[4]);
    debugI2C(0,0,value[5]);
    
 }
}

void INA233::wireReadWord(uint8_t reg, uint16_t *value)
{

  Wire.beginTransmission(ina233_i2caddr);
  Wire.write(reg);
  Wire.endTransmission(false);
  Wire.requestFrom(ina233_i2caddr, 2, true);

  uint8_t tempval1 = Wire.read();
  uint8_t tempval2 = Wire.read();
  
  *value = tempval1;
  *value=((tempval2 << 8) | *value);

  if(debug){debugI2C(0,reg,*value);}
}

void INA233::wireReadByte(uint8_t reg, uint8_t *value)
{

  Wire.beginTransmission(ina233_i2caddr);
  Wire.write(reg);
  Wire.endTransmission(false);
  Wire.requestFrom(ina233_i2caddr, 1, true);
  *value = Wire.read();
  
  if(debug){debugI2C(0,reg,*value);}
}


uint16_t INA233::wireModifyWord(uint8_t reg, uint16_t value,uint8_t numOfBits, uint8_t placement){
  uint16_t in = 0;
  uint16_t out = 0;
  uint16_t maskBits = 0;
  
  maskBits = pow(2,numOfBits) - 1;
  
  value = value & maskBits; //00000111 clear out any extraneous bits
  
  wireReadWord(reg, &in);

  in = in & (~(maskBits << placement));//111 or whatever, shift right by "placement" then invert to make like "11111111,11000111" then use it to clear bits in "in"
  out = in | (value << placement); //place the bits in empty spot
  
  wireWriteWord(reg, out);
  return out;
}

uint8_t INA233::wireModifyByte(uint8_t reg, uint8_t value,uint8_t numOfBits, uint8_t placement){
  uint8_t in = 0;
  uint8_t out = 0;
  uint8_t maskBits = 0;
  
  maskBits = pow(2,numOfBits) - 1;
  
  value = value & maskBits; //00000111 clear out any extra bits
  
  wireReadByte(reg, &in);

  in = in & (~(maskBits << placement));//111 or whatever, shift right by "placement" then invert to make like "11111111,11000111" then use it to clear bits in "in"
  out = in | (value << placement); //place the bits in empty spot
  
  wireWriteByte(reg, out);
  return out;
}

void INA233::debugI2C(uint8_t RWC,uint8_t reg,uint16_t value){
  Serial.print("[");
  if(RWC == 0){Serial.print("R");}
  if(RWC == 1){Serial.print("W");}
  if(RWC == 2){Serial.print("C");}
  Serial.print(",0x"); Serial.print(reg,HEX); Serial.print(",0x"); Serial.print(value,HEX);
  Serial.print("]");
  Serial.println();
}



