/*
 * rf_driver.h
 *
 *  Created on: Feb 6, 2020
 *      Author: suleyman.eskil
 */

#ifndef INC_RF_DRIVER_H_
#define INC_RF_DRIVER_H_

#include <stdio.h>
#ifndef __STM32L4xx_HAL_H
#include "stm32l4xx_hal.h"
#endif
#include "dw_stm32_delay.h"

/** Pinout -----------------------------------------------/
 * VDD --> 3.3V
 * GND --> GND
 * CS --> PD14
 * MOSI --> PA7
 * MISO --> PA6
 * SCK --> PA5
 * GDO0 --> PD15 // PD5 (UART2 TX)
 * GDO2 --> PF12 // PA3 (UART2 RX)
*/

/* Private define ------------------------------------------------------------*/

#define TRUE 1
#define FALSE 0

/*---------------------------[DATA]------------------------------------------*/
#define FIFO_SIZE 64  //Size of TX and RX FIFO buffers
#define FIXED_LENGTH_LIMIT 255 //For packets with length>255, infinite packet length mode must be used
#define DATA_CHUNK_SIZE 32   //The data is read from and write to the buffer in chunks of 32
#define BUFFER_SIZE (1<<16) //Change this to be able to receive longer (or shorter) frames. _get_frame_length should be modified too.

/*---------------------------[CC1100 - R/W offsets]---------------------------*/
#define WRITE(ADDR) 		ADDR
#define WRITE_BURST(ADDR) 	ADDR | 0x40
#define READ(ADDR) 			ADDR | 0x80
#define READ_BURST(ADDR) 	ADDR | 0xC0
#define CRC_OK				0x80
#define RSSI_OFFSET			0x4A
/*---------------------------[END R/W offsets]--------------------------------*/

/*------------------------[CC1100 - FIFO commands]----------------------------*/

/*Not using write/read and burst offset in order to point out that TX and RX even having the "same" address are 2 different buffers.
 * So instead of writing WRITE(FIFO), TXFIFO
*/
#define TXFIFO_BURST        	0x7F    //write burst only
#define TXFIFO					0x3F    //write single only
#define RXFIFO_BURST        	0xFF    //read burst only
#define RXFIFO 					0xBF    //read single only

#define PATABLE_BURST       	0x7E    //power control read/write
#define PATABLE_SINGLE_BYTE 	0xFE    //power control read/write
/*---------------------------[END FIFO commands]------------------------------*/

/*----------------------[CC1100 - config register]----------------------------*/

#define	IOCFG2_W	0x00
#define	IOCFG2_WB	0x40
#define	IOCFG2_R	0x80
#define	IOCFG2_RB	0xC0

#define	IOCFG1_W	0x01
#define	IOCFG1_WB	0x41
#define	IOCFG1_R	0x81
#define	IOCFG1_RB	0xC1	//	 GDO1 output pin configuration

#define	IOCFG0_W	0x02
#define	IOCFG0_WB	0x42
#define	IOCFG0_R	0x82
#define	IOCFG0_RB	0xC2	//	 GDO0 output pin configuration

#define	FIFOTHR_W	0x03
#define	FIFOTHR_WB	0x43
#define	FIFOTHR_R	0x83
#define	FIFOTHR_RB	0xC3	//	 RX FIFO and TX FIFO thresholds

#define	SYNC1_W		0x04
#define	SYNC1_WB	0x44
#define	SYNC1_R		0x84
#define	SYNC1_RB	0xC4	//	 Sync word, high byte

#define	SYNC0_W		0x05
#define	SYNC0_WB	0x45
#define	SYNC0_R		0x85
#define	SYNC0_RB	0xC5	//	 Sync word, low byte

#define	PKTLEN_W	0x06
#define	PKTLEN_WB	0x46
#define	PKTLEN_R	0x86
#define	PKTLEN_RB	0xC6	//	 Packet length

#define	PKTCTRL1_W	0x07
#define	PKTCTRL1_WB	0x47
#define	PKTCTRL1_R	0x87
#define	PKTCTRL1_RB	0xC7	//	 Packet automation control

#define	PKTCTRL0_W	0x08
#define	PKTCTRL0_WB	0x48
#define	PKTCTRL0_R	0x88
#define	PKTCTRL0_RB	0xC8	//	 Packet automation control

#define	ADDR_W		0x09
#define	ADDR_WB		0x49
#define	ADDR_R		0x89
#define	ADDR_RB		0xC9	//	 Device address

#define	CHANNR_W	0x0A
#define	CHANNR_WB	0x4A
#define	CHANNR_R	0x8A
#define	CHANNR_RB	0xCA	//	 Channel number

#define	FSCTRL1_W	0x0B
#define	FSCTRL1_WB	0x4B
#define	FSCTRL1_R	0x8B
#define	FSCTRL1_RB	0xCB	//	 Frequency synthesizer control

#define	FSCTRL0_W	0x0C
#define	FSCTRL0_WB	0x4C
#define	FSCTRL0_R	0x8C
#define	FSCTRL0_RB	0xCC	//	 Frequency synthesizer control

#define	FREQ2_W		0x0D
#define	FREQ2_WB	0x4D
#define	FREQ2_R		0x8D
#define	FREQ2_RB	0xCD	//	 Frequency control word, high byte

#define	FREQ1_W		0x0E
#define	FREQ1_WB	0x4E
#define	FREQ1_R		0x8E
#define	FREQ1_RB	0xCE	//	 Frequency control word, middle byte

#define	FREQ0_W		0x0F
#define	FREQ0_WB	0x4F
#define	FREQ0_R		0x8F
#define	FREQ0_RB	0xCF	//	 Frequency control word, low byte

#define	MDMCFG4_W	0x10
#define	MDMCFG4_WB	0x50
#define	MDMCFG4_R	0x90
#define	MDMCFG4_RB	0xD0	//	 Modem configuration

#define	MDMCFG3_W	0x11
#define	MDMCFG3_WB	0x51
#define	MDMCFG3_R	0x91
#define	MDMCFG3_RB	0xD1	//	 Modem configuration

#define	MDMCFG2_W	0x12
#define	MDMCFG2_WB	0x52
#define	MDMCFG2_R	0x92
#define	MDMCFG2_RB	0xD2	//	 Modem configuration

#define	MDMCFG1_W	0x13
#define	MDMCFG1_WB	0x53
#define	MDMCFG1_R	0x93
#define	MDMCFG1_RB	0xD3	//	 Modem configuration

#define	MDMCFG0_W	0x14
#define	MDMCFG0_WB	0x54
#define	MDMCFG0_R	0x94
#define	MDMCFG0_RB	0xD4	//	 Modem configuration

#define	DEVIATN_W	0x15
#define	DEVIATN_WB	0x55
#define	DEVIATN_R	0x95
#define	DEVIATN_RB	0xD5	//	 Modem deviation setting

#define	MCSM2_W		0x16
#define	MCSM2_WB	0x56
#define	MCSM2_R		0x96
#define	MCSM2_RB	0xD6	//	 Main Radio Cntrl State Machine config

#define	MCSM1_W		0x17
#define	MCSM1_WB	0x57
#define	MCSM1_R		0x97
#define	MCSM1_RB	0xD7	//	 Main Radio Cntrl State Machine config

#define	MCSM0_W		0x18
#define	MCSM0_WB	0x58
#define	MCSM0_R		0x98
#define	MCSM0_RB	0xD8	//	 Main Radio Cntrl State Machine config

#define	FOCCFG_W	0x19
#define	FOCCFG_WB	0x59
#define	FOCCFG_R	0x99
#define	FOCCFG_RB	0xD9	//	 Frequency Offset Compensation config

#define	BSCFG_W		0x1A
#define	BSCFG_WB	0x5A
#define	BSCFG_R		0x9A
#define	BSCFG_RB	0xDA	//	 Bit Synchronization configuration

#define	AGCCTRL2_W	0x1B
#define	AGCCTRL2_WB	0x5B
#define	AGCCTRL2_R	0x9B
#define	AGCCTRL2_RB	0xDB	//	 AGC control

#define	AGCCTRL1_W	0x1C
#define	AGCCTRL1_WB	0x5C
#define	AGCCTRL1_R	0x9C
#define	AGCCTRL1_RB	0xDC	//	 AGC control

#define	AGCCTRL0_W	0x1D
#define	AGCCTRL0_WB	0x5D
#define	AGCCTRL0_R	0x9D
#define	AGCCTRL0_RB	0xDD	//	 AGC control

#define	WOREVT1_W	0x1E
#define	WOREVT1_WB	0x5E
#define	WOREVT1_R	0x9E
#define	WOREVT1_RB	0xDE	//	 High byte Event 0 timeout

#define	WOREVT0_W	0x1F
#define	WOREVT0_WB	0x5F
#define	WOREVT0_R	0x9F
#define	WOREVT0_RB	0xDF	//	 Low byte Event 0 timeout

#define	WORCTRL_W	0x20
#define	WORCTRL_WB	0x60
#define	WORCTRL_R	0xA0
#define	WORCTRL_RB	0xE0	//	 Wake On Radio control

#define	FREND1_W	0x21
#define	FREND1_WB	0x61
#define	FREND1_R	0xA1
#define	FREND1_RB	0xE1	//	 Front end RX configuration

#define	FREND0_W	0x22
#define	FREND0_WB	0x62
#define	FREND0_R	0xA2
#define	FREND0_RB	0xE2	//	 Front end TX configuration

#define	FSCAL3_W	0x23
#define	FSCAL3_WB	0x63
#define	FSCAL3_R	0xA3
#define	FSCAL3_RB	0xE3	//	 Frequency synthesizer calibration

#define	FSCAL2_W	0x24
#define	FSCAL2_WB	0x64
#define	FSCAL2_R	0xA4
#define	FSCAL2_RB	0xE4	//	 Frequency synthesizer calibration

#define	FSCAL1_W	0x25
#define	FSCAL1_WB	0x65
#define	FSCAL1_R	0xA5
#define	FSCAL1_RB	0xE5	//	 Frequency synthesizer calibration

#define	FSCAL0_W	0x26
#define	FSCAL0_WB	0x66
#define	FSCAL0_R	0xA6
#define	FSCAL0_RB	0xE6	//	 Frequency synthesizer calibration

#define	RCCTRL1_W	0x27
#define	RCCTRL1_WB	0x67
#define	RCCTRL1_R	0xA7
#define	RCCTRL1_RB	0xE7	//	 RC oscillator configuration

#define	RCCTRL0_W	0x28
#define	RCCTRL0_WB	0x68
#define	RCCTRL0_R	0xA8
#define	RCCTRL0_RB	0xE8	//	 RC oscillator configuration

#define	FSTEST_W	0x29
#define	FSTEST_WB	0x69
#define	FSTEST_R	0xA9
#define	FSTEST_RB	0xE9	//	 Frequency synthesizer cal control

#define	PTEST_W		0x2A
#define	PTEST_WB	0x6A
#define	PTEST_R		0xAA
#define	PTEST_RB	0xEA	//	 Production test

#define	AGCTEST_W	0x2B
#define	AGCTEST_WB	0x6B
#define	AGCTEST_R	0xAB
#define	AGCTEST_RB	0xEB	//	 AGC test

#define	TEST2_W		0x2C
#define	TEST2_WB	0x6C
#define	TEST2_R		0xAC
#define	TEST2_RB	0xEC	//	 Various test settings

#define	TEST1_W		0x2D
#define	TEST1_WB	0x6D
#define	TEST1_R		0xAD
#define	TEST1_RB	0xED	//	 Various test settings

#define	TEST0_W		0x2E
#define	TEST0_WB	0x6E
#define	TEST0_R		0xAE
#define	TEST0_RB	0xEE	//	 Various test settings


/*
#define IOCFG2   0x00
#define IOCFG1   0x01         // GDO1 output pin configuration
#define IOCFG0   0x02         // GDO0 output pin configuration
#define FIFOTHR  0x03         // RX FIFO and TX FIFO thresholds
#define SYNC1    0x04         // Sync word, high byte
#define SYNC0    0x05         // Sync word, low byte
#define PKTLEN   0x06         // Packet length
#define PKTCTRL1 0x07         // Packet automation control
#define PKTCTRL0 0x08         // Packet automation control
#define ADDR     0x09         // Device address
#define CHANNR   0x0A         // Channel number
#define FSCTRL1  0x0B         // Frequency synthesizer control
#define FSCTRL0  0x0C         // Frequency synthesizer control
#define FREQ2    0x0D         // Frequency control word, high byte
#define FREQ1    0x0E         // Frequency control word, middle byte
#define FREQ0    0x0F         // Frequency control word, low byte
#define MDMCFG4  0x10         // Modem configuration
#define MDMCFG3  0x11         // Modem configuration
#define MDMCFG2  0x12         // Modem configuration
#define MDMCFG1  0x13         // Modem configuration
#define MDMCFG0  0x14         // Modem configuration
#define DEVIATN  0x15         // Modem deviation setting
#define MCSM2    0x16         // Main Radio Cntrl State Machine config
#define MCSM1    0x17         // Main Radio Cntrl State Machine config
#define MCSM0    0x18         // Main Radio Cntrl State Machine config
#define FOCCFG   0x19         // Frequency Offset Compensation config
#define BSCFG    0x1A         // Bit Synchronization configuration
#define AGCCTRL2 0x1B         // AGC control
#define AGCCTRL1 0x1C         // AGC control
#define AGCCTRL0 0x1D         // AGC control
#define WOREVT1  0x1E         // High byte Event 0 timeout
#define WOREVT0  0x1F         // Low byte Event 0 timeout
#define WORCTRL  0x20         // Wake On Radio control
#define FREND1   0x21         // Front end RX configuration
#define FREND0   0x22         // Front end TX configuration
#define FSCAL3   0x23         // Frequency synthesizer calibration
#define FSCAL2   0x24         // Frequency synthesizer calibration
#define FSCAL1   0x25         // Frequency synthesizer calibration
#define FSCAL0   0x26         // Frequency synthesizer calibration
#define RCCTRL1  0x27         // RC oscillator configuration
#define RCCTRL0  0x28         // RC oscillator configuration
#define FSTEST   0x29         // Frequency synthesizer cal control
#define PTEST    0x2A         // Production test
#define AGCTEST  0x2B         // AGC test
#define TEST2    0x2C         // Various test settings
#define TEST1    0x2D         // Various test settings
#define TEST0    0x2E         // Various test settings
*/

/*-------------------------[END config register]------------------------------*/


/*------------------------[CC1100-command strobes]----------------------------*/

#define	SRES_W		0x30
#define	SRES_R		0xB0	//	 Reset chip
#define	SFSTXON_W	0x31
#define	SFSTXON_R	0xB1	//	 Enable
#define	SXOFF_W		0x32
#define	SXOFF_R		0xB2	//	 Turn off crystal oscillator.
#define	SCAL_W		0x33
#define	SCAL_R		0xB3	//	 Calibrate freq synthesizer & disable
#define	SRX_W		0x34
#define	SRX_R		0xB4	//	 Enable RX.
#define	STX_W		0x35
#define	STX_R		0xB5	//	 Enable TX.
#define	SIDLE_W		0x36
#define	SIDLE_R		0xB6	//	 Exit RX
#define	SWOR_W		0x38
#define	SWOR_R		0xB8	//	 Start automatic RX polling sequence
#define	SPWD_W		0x39
#define	SPWD_R		0xB9	//	 Enter pwr down mode when CSn goes hi
#define	SFRX_W		0x3A
#define	SFRX_R		0xBA	//	 Flush the RX FIFO buffer.
#define	SFTX_W		0x3B
#define	SFTX_R		0xBB	//	 Flush the TX FIFO buffer.
#define	SWORRST_W	0x3C
#define	SWORRST_R	0xBC	//	 Reset real time clock.
#define	SNOP_W		0x3D
#define	SNOP_R		0xBD	//	 No operation.


/*#define SRES     0x30         // Reset chip
#define SFSTXON  0x31         // Enable/calibrate freq synthesizer
#define SXOFF    0x32         // Turn off crystal oscillator.
#define SCAL     0x33         // Calibrate freq synthesizer & disable
#define SRX      0x34         // Enable RX.
#define STX      0x35         // Enable TX.
#define SIDLE    0x36         // Exit RX / TX
#define SAFC     0x37         // AFC adjustment of freq synthesizer //Datasheet böyle bir register yok
#define SWOR     0x38         // Start automatic RX polling sequence
#define SPWD     0x39         // Enter pwr down mode when CSn goes hi
#define SFRX     0x3A         // Flush the RX FIFO buffer.
#define SFTX     0x3B         // Flush the TX FIFO buffer.
#define SWORRST  0x3C         // Reset real time clock.
#define SNOP     0x3D         // No operation.*/
/*-------------------------[END command strobes]------------------------------*/

/*----------------------[CC1100 - status register]----------------------------*/

#define	PARTNUM_R			0xF0	//	Part	number
#define	VERSION_R			0xF1	//	Current	version	number
#define	FREQEST_R			0xF2	//	Frequency	offset	estimate
#define	LQI_R				0xF3	//	Demodulator	estimate	for	link	quality
#define	RSSI_R				0xF4	//	Received	signal	strength	indication
#define	MARCSTATE_R			0xF5	//	Control	state	machine	state
#define	WORTIME1_R			0xF6	//	High	byte	of	WOR	timer
#define	WORTIME0_R			0xF7	//	Low	byte	of	WOR	timer
#define	PKTSTATUS_R			0xF8	//	Current	GDOx	status	and	packet	status
#define	VCO_VC_DAC_R		0xF9	//	Current	setting	from	PLL	cal	module
#define	TXBYTES_R			0xFA	//	Underflow	and	#	of	bytes	in	TXFIFO
#define	RXBYTES_R			0xFB	//	Overflow	and	#	of	bytes	in	RXFIFO
#define	RCCTRL1_STATUS_R	0xFC	//Last	RC	Oscillator	Calibration	Result
#define	RCCTRL0_STATUS_R	0xFD	//Last	RC	Oscillator	Calibration	Result


/*#define PARTNUM        0xF0   // Part number
#define VERSION        0xF1   // Current version number
#define FREQEST        0xF2   // Frequency offset estimate
#define LQI            0xF3   // Demodulator estimate for link quality
#define RSSI           0xF4   // Received signal strength indication
#define MARCSTATE      0xF5   // Control state machine state
#define WORTIME1       0xF6   // High byte of WOR timer
#define WORTIME0       0xF7   // Low byte of WOR timer
#define PKTSTATUS      0xF8   // Current GDOx status and packet status
#define VCO_VC_DAC     0xF9   // Current setting from PLL cal module
#define TXBYTES        0xFA   // Underflow and # of bytes in TXFIFO
#define RXBYTES        0xFB   // Overflow and # of bytes in RXFIFO
#define RCCTRL1_STATUS 0xFC   //Last RC Oscillator Calibration Result
#define RCCTRL0_STATUS 0xFD   //Last RC Oscillator Calibration Result*/

//--------------------------[END status register]-------------------------------

/*----------------------[CC1100 - misc]---------------------------------------*/
#define CRYSTAL_FREQUENCY 26000
#define CRYSTAL_FREQUENCY_M CRYSTAL_FREQUENCY/1000
#define CFG_REGISTER              0x2F  //47 registers
#define MAX_CHANNEL_SPACING (0xFF+256)*CRYSTAL_FREQUENCY*(1<<3)/(1<<18)
#define MIN_CHANNEL_SPACING (0x0+256)*CRYSTAL_FREQUENCY/(1<<18) //Min value possible, but min legal value?

#define RSSI_OFFSET_868MHZ        0x4E  //dec = 74
#define TX_RETRIES_MAX            0x05  //tx_retries_max
#define ACK_TIMEOUT                200  //ACK timeout in ms
#define CC1100_COMPARE_REGISTER   0x00  //register compare 0=no compare 1=compare
//#define BROADCAST_ADDRESS         0x00  //broadcast address
//#define CC1100_FREQ_315MHZ        0x01
//#define CC1100_FREQ_434MHZ        0x02
//#define CC1100_FREQ_868MHZ        0x03
//#define CC1100_FREQ_915MHZ        0x04
//#define CC1100_FREQ_2430MHZ       0x05
#define CC1100_TEMP_ADC_MV        3.225 //3.3V/1023 . mV pro digit
#define CC1100_TEMP_CELS_CO 2.47 //Temperature coefficient 2.47mV per Grad Celsius

/* Private typedef ---------------------------------------------------------*/
typedef enum{
	//SLEEP = 0x00, //Can't be accessed
	IDLE = 0x01,
	//XOFF = 0x02,
	VCOON_MC = 0x03, //MANCAL
	REGON_MC = 0x04, //MANCAL
	MANCAL = 0x05,
	VCOON = 0x06, //FS_WAKEUP
	REGON = 0x07, //FS_WAKEUP
	STARTCAL = 0x08, //CALIBRATE
	BWBOOST = 0x09, //SETTLIG
	FS_LOCK = 0x0A, //SETTLIG
	IFADCON = 0x0B, //SETTLIG
	ENDCAL = 0x0C, //CALIBRATE
	RX = 0x0D,
	RX_END = 0x0E,	//RX
	RX_RST = 0x0F,	//RX
	TXRX_SWITCH = 0x10, //TXRX_SETTLING
	RXFIFO_OVERFLOW = 0x11,
	FSTXON = 0x12,
	TX = 0x13,
	TX_END = 0x14, //TX
	RXTX_SWITCH = 0x15,	//RXTX_SETTLING
	TXFIFO_UNDERFLOW = 0x16
}STATE_TypeDef;

typedef enum{
	GFSK_1_2_kb = 1,
	GFSK_38_4_kb,
	GFSK_100_kb,
	MSK_250_kb,
	MSK_500_kb,
	OOK_4_8_kb
}MODULATION_TypeDef;

typedef enum{
	MHz315 = 1,
	MHz434,
	MHz868,
	MHz915,
}ISMBAND_TypeDef;

typedef enum{
	TIMEOUT = -1,
	FRAME_BAD = 0, //CRC bad when receiving
	FRAME_OK = 1, //CRC OK when receiving, frame transmited when transmitting
}FRAMESTATUS_TypeDef;


/* Private function prototypes -----------------------------------------------*/

/* SPI data flow-------------*/
void rf_spi_write(uint8_t reg, uint8_t data);
uint8_t rf_spi_read(uint8_t reg);
void rf_spi_read_data(uint8_t addr, uint8_t* data, uint8_t size);
void rf_spi_write_data(uint8_t addr, uint8_t* data, uint8_t size);
void rf_spi_write_strobe(uint8_t strobe);

void rf_write_strobe(uint8_t strobe);
uint8_t rf_read_register(uint8_t reg);
void rf_write_register(uint8_t reg, uint8_t data);
void rf_read_data(uint8_t addr, uint8_t* data, uint8_t size);
void rf_write_data(uint8_t addr, uint8_t* data, uint8_t size);
HAL_StatusTypeDef __spi_write(uint8_t* addr, uint8_t *pData, uint16_t size);
HAL_StatusTypeDef __spi_read(uint8_t* addr, uint8_t *pData, uint16_t size);


/* Driver init-------------*/
uint8_t rf_check();
void rf_reset();
void Power_up_reset();

uint8_t rf_begin(SPI_HandleTypeDef* hspi, MODULATION_TypeDef mode, ISMBAND_TypeDef ism_band,  GPIO_TypeDef* cs_port, uint16_t cs_pin, uint16_t gdo0_pin);

/* Driver Config -------*/
void rf_set_modulation_mode(MODULATION_TypeDef mod);
void rf_set_ISMband(ISMBAND_TypeDef band);
void rf_set_channel(uint8_t channel);
void rf_set_output_power_level(int8_t);
void rf_set_syncword(uint16_t syncword);
//packet control
void rf_set_addr(uint8_t addr);
//FSCTRL1 – Frequency Synthesizer Control??
float rf_set_carrier_offset(float offset);
float rf_set_carrier_frequency(float target_freq);
float rf_set_channel_spacing(float cpsacing);
void rf_set_preamble(uint8_t nbytes);
void rf_set_preamble_threshold(uint8_t nbytes);

/*Gonder-Al*/
void SpiPaketgonder(uint8_t* TxBuffer, uint8_t size);
uint8_t SpiPaketal(uint8_t* RxBuffer, uint8_t* size);
uint8_t SpiDurumOku(uint8_t Addr);

/*State --------------*/
void rf_sidle();
void rf_power_down();
void rf_wakeup();
void rf_receive();
uint8_t rf_get_settings();
void rf_wor_enable();
void rf_wor_disable();
void rf_wor_reset();



/* Comm -------------*/
uint16_t _get_frame_size(uint8_t* header, uint8_t data_len_loc, uint8_t data_len_size);
uint8_t _keep_transmiting_data(uint8_t* data, int len);
FRAMESTATUS_TypeDef send_frame(uint8_t* frame, int len);
uint8_t  _keep_receiving_data(uint8_t* data, int len);
FRAMESTATUS_TypeDef receive_frame(uint8_t* frame_buffer, uint16_t* len, uint8_t data_len_loc, uint8_t data_len_size, uint8_t* lqi, uint8_t* rssi);
uint8_t polling_while_lower(uint8_t reg, uint8_t size);
uint8_t polling_while_bigger(uint8_t reg, uint8_t size);
uint8_t rf_incoming_packet();
void send_packet_rf(uint8_t* txBuffer, uint8_t size);

/* Misc -------------*/
void init_serial(UART_HandleTypeDef* huart);
int __io_putchar(int ch);
int _write(int file,char *ptr, int len);

void init_led();




#endif /* INC_RF_DRIVER_H_ */
