/*
 * Copyright (c) 2019, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/***** Includes *****/
/* Standard C Libraries */
#include <stdlib.h>

/* TI Drivers */
#include <ti/drivers/rf/RF.h>
#include <ti/drivers/GPIO.h>
#include <ti/display/Display.h>

/* Driverlib Header files */
#include DeviceFamily_constructPath(driverlib/rf_prop_mailbox.h)

/* Board Header files */
#include "ti_drivers_config.h"

/* Application Header files */
#include "RFQueue.h"
#include <ti_radio_config.h>

#include DeviceFamily_constructPath(driverlib/rf_prop_mailbox.h)
#include DeviceFamily_constructPath(driverlib/rf_prop_cmd.h)
#include DeviceFamily_constructPath(driverlib/rf_mailbox.h)

/***** Defines *****/

#define DATA_ENTRY_HEADER_SIZE 8    // Constant header size of a Generic Data Entry
#define NUM_DATA_ENTRIES       2    // NOTE: Only two data entries supported at the moment

#define CRC                     0   // 2 if .rxConf.bIncludeCrc = 0x1, 0 otherwise
#define RSSI                    0   // 1 if .rxConf.bAppendRssi = 0x1, 0 otherwise
#define TIMESTAMP               0   // 4 if .rxConf.bAppendTimestamp = 0x1, 0 otherwise
#define STATUS                  0   // 1 if .rxConf.bAppendStatus = 0x1, 0 otherwise

#define NUM_APPENDED_BYTES      1 + CRC + RSSI + TIMESTAMP + STATUS // The length is 1 byte

#define MAX_LENGTH              0x0E // Max length byte the radio will accept
                                     // Must be less than 256 if SIZE_OF_LENGHT_FIELD = 1

static void callback(RF_Handle h, RF_CmdHandle ch, RF_EventMask e);

static RF_Object rfObject;
static RF_Handle rfHandle;
Display_Handle display;

/* Buffer which contains all Data Entries for receiving data.
 * Pragmas are needed to make sure this buffer is 4 byte aligned (requirement from the RF Core) */
#if defined(__TI_COMPILER_VERSION__)
#pragma DATA_ALIGN (rxDataEntryBuffer, 4);
static uint8_t
rxDataEntryBuffer[RF_QUEUE_DATA_ENTRY_BUFFER_SIZE(NUM_DATA_ENTRIES,
                                                  MAX_LENGTH,
                                                  NUM_APPENDED_BYTES)];
#elif defined(__IAR_SYSTEMS_ICC__)
#pragma data_alignment = 4
static uint8_t
rxDataEntryBuffer[RF_QUEUE_DATA_ENTRY_BUFFER_SIZE(NUM_DATA_ENTRIES,
                                                  MAX_LENGTH,
                                                  NUM_APPENDED_BYTES)];
#elif defined(__GNUC__)
static uint8_t
rxDataEntryBuffer[RF_QUEUE_DATA_ENTRY_BUFFER_SIZE(NUM_DATA_ENTRIES,
                                                  MAX_LENGTH,
                                                  NUM_APPENDED_BYTES)]
                                                  __attribute__((aligned(4)));
#else
#error This compiler is not supported.
#endif

// Receive dataQueue for RF Core to fill in data
dataQueue_t dataQueue;
rfc_dataEntryGeneral_t* currentDataEntry;
uint16_t packetLength;
uint8_t* packetDataPointer;
rfc_propRxOutput_t rxStatistics;

uint8_t packet[MAX_LENGTH + NUM_APPENDED_BYTES - 1]; // Length is stored separately

uint16_t AddressList[] = {0xBED6, 0xAAAA, 0xBBBB, 0xAABB, 0xBBAA};

/***** Function definitions *****/

void *mainThread(void *arg0)
{
    RF_Params rfParams;
    RF_Params_init(&rfParams);

    /* Open the display for output */
    display = Display_open(Display_Type_UART, NULL);
    if (display == NULL) {
        /* Failed to open display driver */
        while (1) {
        }
    }

    Display_printf(display, 0, 0, "Starting as RX\n");

    GPIO_setConfig(CONFIG_GPIO_RLED, GPIO_CFG_OUT_STD | GPIO_CFG_OUT_LOW);
    GPIO_write(CONFIG_GPIO_RLED, CONFIG_GPIO_LED_OFF);

    if( RFQueue_defineQueue(&dataQueue,
                            rxDataEntryBuffer,
                            sizeof(rxDataEntryBuffer),
                            NUM_DATA_ENTRIES,
                            MAX_LENGTH + NUM_APPENDED_BYTES))
    {
        // Failed to allocate space for all data entries
        while(1);
    }

    // Modify CMD_PROP_RX command for application needs
    // Set the Data Entity queue for received data
    RF_cmdPropRxAdv.pQueue = &dataQueue;
    // Discard ignored packets from Rx queue
    RF_cmdPropRxAdv.rxConf.bAutoFlushIgnored = 1;
    // Discard packets with CRC error from Rx queue
    RF_cmdPropRxAdv.rxConf.bAutoFlushCrcErr = 1;
    // Implement packet length filtering to avoid PROP_ERROR_RXBUF
    RF_cmdPropRxAdv.maxPktLen = MAX_LENGTH;
    RF_cmdPropRxAdv.pktConf.bRepeatOk = 1;
    RF_cmdPropRxAdv.pktConf.bRepeatNok = 1;
    RF_cmdPropRxAdv.pOutput = (uint8_t*)&rxStatistics;

    RF_cmdPropRxAdv.condition.rule = 1;
    RF_cmdPropRxAdv.pktConf.bUseCrc = 0x1;
    RF_cmdPropRxAdv.rxConf.bIncludeHdr = 0x1;
    RF_cmdPropRxAdv.endTrigger.triggerType = 0x1;

    RF_cmdPropRxAdv.pktConf.bCrcIncHdr = 0x1;

    RF_cmdPropRxAdv.hdrConf.numHdrBits = 24;
    RF_cmdPropRxAdv.hdrConf.numLenBits = 8;
    RF_cmdPropRxAdv.hdrConf.lenPos = 0;
    RF_cmdPropRxAdv.lenOffset = -2;

    RF_cmdPropRxAdv.rxConf.bIncludeCrc = 0x0;
    RF_cmdPropRxAdv.rxConf.bAppendRssi = 0x0;
    RF_cmdPropRxAdv.rxConf.bAppendTimestamp = 0x0;
    RF_cmdPropRxAdv.rxConf.bAppendStatus = 0x0;

    RF_cmdPropRxAdv.addrConf.addrType = 1;  // Address in header
    RF_cmdPropRxAdv.addrConf.addrSize = 16; // size of address (in bits)
    RF_cmdPropRxAdv.addrConf.addrPos = 8;
    RF_cmdPropRxAdv.addrConf.numAddr = 5;   // Number of addresses in address list
    RF_cmdPropRxAdv.pAddr = (uint8_t*)&AddressList;

    // Request access to the radio
    rfHandle = RF_open(&rfObject, &RF_prop, (RF_RadioSetup*)&RF_cmdPropRadioDivSetup, &rfParams);

    // Set the frequency
    RF_postCmd(rfHandle, (RF_Op*)&RF_cmdFs, RF_PriorityNormal, NULL, 0);

    RF_runCmd(rfHandle, (RF_Op*)&RF_cmdPropRxAdv, RF_PriorityNormal, &callback, RF_EventRxEntryDone);

    while(1);
}

void callback(RF_Handle h, RF_CmdHandle ch, RF_EventMask e)
{
    if (e & RF_EventRxEntryDone)
    {
        // Toggle pin to indicate RX
        GPIO_toggle(CONFIG_GPIO_RLED);

        // Get current unhandled data entry
        currentDataEntry = RFQueue_getDataEntry();

        packetLength      = (uint16_t)(*(uint8_t*)(&currentDataEntry->data));
        packetDataPointer = (uint8_t*)(&currentDataEntry->data + 1);

        // Copy the payload + the status byte to the packet variable
        memcpy(packet, packetDataPointer, (packetLength + NUM_APPENDED_BYTES - 1));

        RFQueue_nextEntry();
    }
}
