/* XDC Module Headers */
#include <xdc/std.h>
#include <xdc/runtime/System.h>
#include <xdc/runtime/Error.h>
//#include <xdc/runtime/Types.h>

/* RF Drivers */
#include <ti/drivers/rf/RF.h>
#include <ti/drivers/pin/PINCC26XX.h>

/* Driver Header files */
#include <ti/drivers/GPIO.h>

#include "smartrf_settings.h"

/* BIOS Module Headers */
#include <ti/sysbios/BIOS.h>
#include <ti/sysbios/hal/Timer.h>
#include <ti/sysbios/hal/Hwi.h>
#include <ti/sysbios/knl/Queue.h>
#include <ti/sysbios/knl/Clock.h>
#include <ti/sysbios/knl/Task.h>
#include <ti/sysbios/knl/Semaphore.h>

/* Stack size in bytes */
#define THREADSTACKSIZE    768

/* Example/Board Header files */
#include "Board.h"


/* RF SETTINGS */

/* Packet RX Configuration */
#define DATA_ENTRY_HEADER_SIZE 8  /* Constant header size of a Generic Data Entry */
#define MAX_LENGTH             30 /* Max length byte the radio will accept */
#define NUM_DATA_ENTRIES       2  /* NOTE: Only two data entries supported at the moment */
#define NUM_APPENDED_BYTES     2  /* The Data Entries data field will contain:
                                   * 1 Header byte (RF_cmdPropRx.rxConf.bIncludeHdr = 0x1)
                                   * Max 30 payload bytes
                                   * 1 status byte (RF_cmdPropRx.rxConf.bAppendStatus = 0x1) */
#include "RFQueue.h"
static void callback(RF_Handle h, RF_CmdHandle ch, RF_EventMask e);

/***** Variable declarations *****/
RF_Handle rfHandle;
RF_Object rfObject;
RF_CmdHandle RF_rxCmd;
RF_Params rfParams;

Uint8 rssi;




/* Buffer which contains all Data Entries for receiving data.
 * Pragmas are needed to make sure this buffer is 4 byte aligned (requirement from the RF Core) */
#if defined(__TI_COMPILER_VERSION__)
#pragma DATA_ALIGN (rxDataEntryBuffer, 4);
static uint8_t
rxDataEntryBuffer[RF_QUEUE_DATA_ENTRY_BUFFER_SIZE(NUM_DATA_ENTRIES,
                                                  MAX_LENGTH,
                                                  NUM_APPENDED_BYTES)];
#elif defined(__IAR_SYSTEMS_ICC__)
#pragma data_alignment = 4
static uint8_t
rxDataEntryBuffer[RF_QUEUE_DATA_ENTRY_BUFFER_SIZE(NUM_DATA_ENTRIES,
                                                  MAX_LENGTH,
                                                  NUM_APPENDED_BYTES)];
#elif defined(__GNUC__)
static uint8_t
rxDataEntryBuffer[RF_QUEUE_DATA_ENTRY_BUFFER_SIZE(NUM_DATA_ENTRIES,
                                                  MAX_LENGTH,
                                                  NUM_APPENDED_BYTES)]
                                                  __attribute__((aligned(4)));
#else
#error This compiler is not supported.
#endif

/* Receive dataQueue for RF Core to fill in data */
static dataQueue_t dataQueue;
static rfc_dataEntryGeneral_t* currentDataEntry;
static uint8_t packetLength;
static uint8_t* packetDataPointer;


static uint8_t packet[MAX_LENGTH + NUM_APPENDED_BYTES - 1]; /* The length byte is stored in a separate variable */
/* END RADIO */

/* NOTE: See tistdtypes.h, for data types */

Void clk0Fxn(UArg arg0);
Void clk1Fxn(UArg arg0);

Clock_Struct clk0Struct, clk1Struct;
Clock_Handle clk2Handle;


Clock_Handle clockButtonDebounceHandle;

Task_Handle LEDBlinkHandle;



Semaphore_Handle sem;

Void LEDBlinkThread(UArg arg0, UArg arg1);

Void gpioButtonFxn0(uint_least8_t index);

Void debounceButtons(UArg arg);


/*
 *  ======== main ========
 */
int main()
{
  /* Construct BIOS Objects */
  Clock_Params clkParams;
  Task_Params taskParams;
  RF_Params rfParams;
  RF_Stat rfStat;

  /*Define RF QUEUE*/
  if( RFQueue_defineQueue(&dataQueue,
                          rxDataEntryBuffer,
                          sizeof(rxDataEntryBuffer),
                          NUM_DATA_ENTRIES,
                          MAX_LENGTH + NUM_APPENDED_BYTES))
  {
      /* Failed to allocate space for all data entries */
      while(1);
  }

  /* Modify CMD_PROP_RX command for application needs */
  /* Set the Data Entity queue for received data */
  RF_cmdPropRx.pQueue = &dataQueue;
  /* Discard ignored packets from Rx queue */
  RF_cmdPropRx.rxConf.bAutoFlushIgnored = 1;
  /* Discard packets with CRC error from Rx queue */
  RF_cmdPropRx.rxConf.bAutoFlushCrcErr = 1;
  /* Implement packet length filtering to avoid PROP_ERROR_RXBUF */
  RF_cmdPropRx.maxPktLen = MAX_LENGTH;
  RF_cmdPropRx.pktConf.bRepeatOk = 1;
  RF_cmdPropRx.pktConf.bRepeatNok = 1;

  /* Set Error Blocks */
  Error_Block eb;

  /* Initialize Error Block */
  Error_init(&eb);

  /* Initialize Board */
  Board_initGeneral();

  /* Initialize GPIO */
  GPIO_init();

  /* Configure the LED pin */
  GPIO_setConfig(Board_GPIO_LED0, GPIO_CFG_OUT_STD | GPIO_CFG_OUT_LOW);
  GPIO_setConfig(Board_GPIO_LED1, GPIO_CFG_OUT_STD | GPIO_CFG_OUT_LOW);

  /* Configure the Button Input */
  GPIO_setConfig(Board_GPIO_BTN1, GPIO_CFG_IN_PU | GPIO_CFG_IN_INT_FALLING);

  /* Install Button callback */
  GPIO_setCallback(Board_GPIO_BTN1, gpioButtonFxn0);

  /* Enable Button interrupts */
  GPIO_enableInt(Board_GPIO_BTN1);

  RF_Params_init(&rfParams);

  /* Initialize RF Object */
  rfHandle = RF_open(&rfObject, &RF_prop, (RF_RadioSetup*)&RF_cmdPropRadioDivSetup, &rfParams);

  /* Turn on radio and Set the frequency */
  RF_postCmd(rfHandle, (RF_Op*)&RF_cmdFs, RF_PriorityNormal, NULL, 0);

  /* Enter RX mode and stay forever in RX */
  RF_rxCmd = RF_postCmd(rfHandle, (RF_Op*)&RF_cmdPropRx,RF_PriorityNormal, &callback, RF_EventRxEntryDone);


  /* Construct a periodic Clock Instance */
  Clock_Params_init(&clkParams);
  //Default clock is 10us per clock tick, so 1000ms = 100000 clock ticks
  clkParams.period = 100000;
  clkParams.startFlag = TRUE;
  Clock_construct(&clk0Struct, (Clock_FuncPtr)clk0Fxn, 1, &clkParams);

  /* Construct Button Debounce Clock */
  Clock_Params_init(&clkParams);
  clkParams.period = 0;
  clkParams.startFlag = FALSE;
  clockButtonDebounceHandle = Clock_create(debounceButtons, 50, &clkParams, NULL);

  /* Construct writer/reader Task threads */
  Task_Params_init(&taskParams);
  taskParams.stackSize = THREADSTACKSIZE;
  taskParams.priority = 1;
  LEDBlinkHandle = Task_create((Task_FuncPtr)LEDBlinkThread, &taskParams, Error_IGNORE);

  /* Construct Semaphore */
  sem = Semaphore_create(0, NULL, &eb);

  BIOS_start();    /* does not return */
}

/*
 *  ======== clk0Fxn =======
 */
Void clk0Fxn(UArg arg0)
{

  //We get a reading at 0x0393, ie. 915MHz off my frequency generator
  rssi =  RF_getRssi(rfHandle);

  if(rssi > 200)
  {
    RF_flushCmd(rfHandle, RF_CMDHANDLE_FLUSH_ALL, 0);

    //RF_cancelCmd(rfHandle, RF_rxCmd, 0);

    RF_close(rfHandle);

    RF_Params_init(&rfParams);

    //Let's shift the frequency to 914MHz now
    RF_cmdPropRadioDivSetup.centerFreq = 0x0392;

    /* Initialize RF Object */
    rfHandle = RF_open(&rfObject, &RF_prop, (RF_RadioSetup*)&RF_cmdPropRadioDivSetup, &rfParams);

    //Set frequency to 914MHz
    RF_cmdFs.frequency = 0x0392;
    RF_cmdFs.fractFreq = 0x0000;

    /* Turn on radio and Set the frequency */
    RF_postCmd(rfHandle, (RF_Op*)&RF_cmdFs, RF_PriorityHigh, NULL, 0);

    /* Turn on radio and Set the frequency */
    RF_postCmd(rfHandle, (RF_Op*)&RF_cmdFs, RF_PriorityHigh, NULL, 0);

    /* Enter RX mode and stay forever in RX */
    RF_rxCmd = RF_postCmd(rfHandle, (RF_Op*)&RF_cmdPropRx,RF_PriorityNormal, &callback, RF_EventRxEntryDone);

    /* Turn on user LED */
    GPIO_toggle(Board_GPIO_LED0);
  }

  Semaphore_post((Semaphore_Object *)sem);
}

void LEDBlinkThread(UArg arg0, UArg arg1)
{
   while(1)
   {
       //GPIO_toggle(Board_GPIO_LED1);

       //Sleep for 500ms
       //Task_sleep(50000);

       Semaphore_pend(sem, BIOS_WAIT_FOREVER);
   }
}

void gpioButtonFxn0(uint_least8_t index)
{
    //Set our one shot time to delay for 50ms, and than go to debounceButtons() to check if our button is still pressed down
    Clock_start(clockButtonDebounceHandle);
}

void debounceButtons(UArg arg0)
{

    if(GPIO_read(Board_GPIO_BTN1) == 0)
    {
        GPIO_toggle(Board_GPIO_LED1);
    }

    /* Clear the GPIO interrupt */
    GPIO_clearInt(Board_GPIO_BTN1);
}

void callback(RF_Handle h, RF_CmdHandle ch, RF_EventMask e)
{

        /* Get current unhandled data entry */
        currentDataEntry = RFQueue_getDataEntry();

        /* Handle the packet data, located at &currentDataEntry->data:
         * - Length is the first byte with the current configuration
         * - Data starts from the second byte */
        packetLength      = *(uint8_t*)(&currentDataEntry->data);
        packetDataPointer = (uint8_t*)(&currentDataEntry->data + 1);

        /* Copy the payload + the status byte to the packet variable */
        memcpy(packet, packetDataPointer, (packetLength + 1));

        //Check first byte
        if(packet[0] == 0xFF)
        {

            GPIO_toggle(Board_GPIO_LED1);
        }

        RFQueue_nextEntry();


}


