// based on the eusci_a/b_spi_ex1_master.c samples from TI, which bears the following copyright notice:
/* --COPYRIGHT--,BSD
 * Copyright (c) 2017, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/

#include <mbed.hpp>
#include "config.h"
#include "driverlib.h"

#ifdef USE_SPI_A
#ifdef USE_SPI_B
#error "Can't use both SPI A and B at once. Define (#define) *either* USE_SPI_A *or* USE_SPI_B"
#endif
#endif

#ifndef USE_SPI_A
#ifndef USE_SPI_B
#error "Must pick either SPI A or B. Define (#define) either USE_SPI_A or USE_SPI_B.
#endif
#endif

#ifdef USE_SPI_A
#error "TODO"
#else
#define SPI_CLK_GPIO_PORT GPIO_PORT_P2
#define SPI_CLK_GPIO_PIN GPIO_PIN2
#define SPI_DATA_GPIO_PORT GPIO_PORT_P1
#define SPI_DATA_GPIO_PINS (GPIO_PIN6 + GPIO_PIN7)
#endif

SPI::SPI(uint32_t freq_hz) {
    //Set DCO frequency to max DCO setting
    CS_setDCOFreq(CS_DCORSEL_0,CS_DCOFSEL_3);
    //Select DCO as the clock source for SMCLK with no frequency divider
    CS_initClockSignal(CS_SMCLK,CS_DCOCLK_SELECT,CS_CLOCK_DIVIDER_1);


    GPIO_setAsPeripheralModuleFunctionInputPin(
        SPI_CLK_GPIO_PORT,
        SPI_CLK_GPIO_PIN,
        GPIO_SECONDARY_MODULE_FUNCTION
    );

    GPIO_setAsPeripheralModuleFunctionInputPin(
        SPI_DATA_GPIO_PORT,
        SPI_DATA_GPIO_PINS,
        GPIO_SECONDARY_MODULE_FUNCTION
    );

    /*
    * Disable the GPIO power-on default high-impedance mode to activate
    * previously configured port settings
    */
    PMM_unlockLPM5();

    #ifdef USE_SPI_A
    #error "unimplemented"
    #else
    EUSCI_B_SPI_initMasterParam param = {0};
    param.selectClockSource = EUSCI_B_SPI_CLOCKSOURCE_SMCLK;
    param.msbFirst = EUSCI_B_SPI_MSB_FIRST;
    param.clockPhase = EUSCI_B_SPI_PHASE_DATA_CHANGED_ONFIRST_CAPTURED_ON_NEXT;
    //param.clockPhase = EUSCI_B_SPI_PHASE_DATA_CAPTURED_ONFIRST_CHANGED_ON_NEXT;
    param.clockPolarity = EUSCI_B_SPI_CLOCKPOLARITY_INACTIVITY_LOW;
    //param.clockPolarity = EUSCI_B_SPI_CLOCKPOLARITY_INACTIVITY_HIGH;
    param.spiMode = EUSCI_B_SPI_3PIN;
    #endif
    
    param.clockSourceFrequency = CS_getSMCLK();
    param.desiredSpiClock = freq_hz;

    #ifdef USE_SPI_A
    #error "TODO"
    #else
    EUSCI_B_SPI_initMaster(EUSCI_B0_BASE, &param);
    
    //Enable SPI module
    EUSCI_B_SPI_enable(EUSCI_B0_BASE);

    EUSCI_B_SPI_clearInterrupt(EUSCI_B0_BASE,
            EUSCI_B_SPI_RECEIVE_INTERRUPT);
    // Enable USCI_B0 RX interrupt
    EUSCI_B_SPI_enableInterrupt(EUSCI_B0_BASE,
        EUSCI_B_SPI_RECEIVE_INTERRUPT);
    
    //Wait for slave to initialize
    __delay_cycles(100);
    #endif
}

uint8_t SPI::write(uint8_t value) volatile {
    //Delay between transmissions for slave to process information
    //__delay_cycles(40);

    #ifdef USE_SPI_A
    #error "TODO"
    #else
    //USCI_B0 TX buffer ready?
    while (!EUSCI_B_SPI_getInterruptStatus(EUSCI_B0_BASE,
        EUSCI_B_SPI_TRANSMIT_INTERRUPT)) ;
    
    EUSCI_B_SPI_transmitData(EUSCI_B0_BASE, value);
    #endif

    __bis_SR_register(LPM0_bits + GIE);      // CPU off, enable interrupts
    __no_operation();                       // Remain in LPM0

    while (!EUSCI_B_SPI_getInterruptStatus(EUSCI_B0_BASE,
                            EUSCI_B_SPI_TRANSMIT_INTERRUPT));

    __delay_cycles(200);

    uint8_t ret = EUSCI_B_SPI_receiveData(EUSCI_B0_BASE);
    __no_operation();
    return ret;
}

SPI init_spi_and_i_promise_to_only_call_this_once(uint32_t freq_hz) {
    return SPI(freq_hz);
}

// TODO: test if we can get away without doing any of this (maybe the interrupt still interrupts)
#ifdef USE_SPI_A
#error "TODO"
#else

#if defined(__TI_COMPILER_VERSION__) || defined(__IAR_SYSTEMS_ICC__)
#pragma vector=USCI_B0_VECTOR
__interrupt
#elif defined(__GNUC__)
__attribute__((interrupt(USCI_B0_VECTOR)))
#endif
void USCI_B0_ISR (void)
{
    switch (__even_in_range(UCB0IV, USCI_SPI_UCTXIFG))
        {
            case USCI_SPI_UCRXIFG:      // UCRXIFG
                //USCI_B0 TX buffer ready?
                if (EUSCI_B_SPI_getInterruptStatus(EUSCI_B0_BASE,
                            EUSCI_B_SPI_TRANSMIT_INTERRUPT)) {
                    __bic_SR_register_on_exit(LPM0_bits + GIE); // leave LPM0 and continue processing
                    // TODO: confirm no race exists with other interrupts
                }
                break;
            default:
                break;
        }
}

#endif
