﻿///////////////////////////////////////////////////////////////////////////////
//                      LORA Main
//
//      printf()가 컴파일 되면 부팅조차 안함
//
//CPU    STM32L083CZTx, LowPower Cortex-M0 MCU, 32 MHz
//FLASH: 0x08000000 0x00030000 (192K)
//RAM    0x20000000 0x00005000 (20K)
//
// 2017/02/01 v1.0 JEONG OJANG
// 2018/12/27 v1.01 Battery Saving Test & Modify [Result: 80uA -> 2.8uA]
///////////////////////////////////////////////////////////////////////////////

#include "JLIB.H"
#include "DRIVER.H"
#include "JOS.H"
#include "MONITOR.H"
#include "MAIN.H"
#include "RFModule.h"
#include "RFCommunication.h"
#include "KEY.H"
#include "INIDATA.H"
#include "PlayBuzzer.h"
#include "SHT20.H"
#include "SPIAutoMan.h"
#include "DetectFire.h"
#include "AES128.H"
#include "RV3032.H"     //정밀RTC
#include "LOG.H"


#define FORCE_FIXED_HEAT_DETECTOR   0       //차동식 하드웨어를 정온식으로 동작하게 하는 펌웨어
#define KFI_INSPECTION          0           //1로 하면 정온식 열감지를 4초마다 처리하고, 실시간 통신점검을 위해 WOR기능이 활성화됨
#define RECOVERRFCHIPALARM      0           //RF칩을 복구한 경우 멜로디 알림


#define CORECLOCK               8000000
#define REPEATTEST              0           //KC인증할 때 사용

#define WAKEUPINTERVALTIME      24*60*60    //Alive보내는 시간 기본값, 단위:초
//#define WAKEUPINTERVALTIME_KR 24*60*60*7  //2020-11-20 한국형도 1주일마다 1회 보내도록 수정
#define WAKEUPINTERVALTIME_KR   24*60*60    //2021-02-17 Sniff를 끄고 매일 1회 보내도록 수정
#define WAKEUPINTERVALTIME_KFI  180         //통신점검을 대신하기 위해 감지기가 180초마다 Alive를 보내는 방식
#define BELLSHORTONTIME         10*100      //단위10ms, 경종을 짧게 울리는 기간
#define DEFREPEATERMODE         0
#define DEFGROUPNO              1
#define DEFCHANNELNO            1   //2018-09-17 1->26, 2020-10-14 26->1 (26은 소방용임)
#define DEFAULTDEVICENO         1
#define DEFRFPOWER              15  //RFPOWER_10mW, 2017-11-21 16dBm으로 수정, 2018-09-17 15로 수정
#define CHECKFIREINTERVAL       8   //2020-11-03 정온식 열감지기를 KFI 특종으로 설계하기 4초로
                                    //2021-04-30 정오장이 8초로 고정함

#define FIXEDHEATDETECTQTY      2   //열 감지가 2초 동안 연속 감지된 경우 화재로 인정 (수정일: 2020-07-27)
#define SMOKEDETECTQTY          10  //2021-09-15 축척기능으로 인증받기 위해 8->10
#define FIREDETECTQTY           8   //열 연기 복합형 감지 횟수
#define DEFCONSTDETECTTEMP      64  //기본 정온식 감지 온도 (KFI 형식은 65도로), 2020-11-04 65->64
#define DETECTTEMPDIFF          10  //평균온도와 10도 차이가 나면 화재 감지
#define SMOKE_DETECT_LEVEL      200 //1종은 123, 2종160, 3종265,
                                    //2021-11-23 160->200 KFI인증들어가면서 이이사의 요청으로 변경

#define LOWBATTERYSENDINTERVAL  3*60*60 //배터리 저전압 알림 간격, 기본 3시간
#define RORHEATDETECTQTY        3   //차동식 열 감지가 3초 동안 연속 감지된 경우 화재로 인정 (수정일: 2020-11-24)
#define SNIFFCYCLE              60  //86400을 나눌 수 있어야 함
#define SNIFFPERIODII           1700    //ms = 500(Sniff)+1000(캐리어)+200(데이터)
#define RESPONSETIMEUNIT        400 //ms
#define ACKTIMEUNIT             300 //ms, 실제 패킷이 190ms임
#define HALFRESPONSETIMEUNIT    200 //ms
#define RESPONSETOLERLANCE      51  //ms
#define DETECTORQTY             20  //개
#define REPEATERQTY             6   //개
#define WAKEUPINTERVAL          5   //400ms 단위
#define REPEATEROCCUPYTIME      ((DETECTORQTY+WAKEUPINTERVAL)*RESPONSETIMEUNIT) //10,000ms=10초
#define RTCDEVIATIONMAX         30000   //ms단위, 내부RTC 1일 편차5초, 실제로 확인한 RV3032 최대 편차 13초


       CONST CHAR InternalStr[]="Internal";
static CONST CHAR GroupStr[]="Group";
static CONST CHAR ChannelStr[]="Channel";
static CONST CHAR PowerStr[]="Power";
static CONST CHAR DevNoStr[]="DevNo";
static CONST CHAR RepeaterStr[]="Repeater";
static CONST CHAR RemapKeyStr[]="RemapKey";
static CONST CHAR SensorAlarmFgStr[]="SensorAlarmFg";
static CONST CHAR SmokeLevelStr[]="SmokeLevel";
static CONST CHAR SysDebugStr[]="SysDebug";
static CONST CHAR DetectConstTempStr[]="DetectConstTemp";
static CONST CHAR DetectTempDiffStr[]="DetectTempDiff";
static CONST CHAR FireDetectPolicyStr[]="FireDetectPolicy";
static CONST CHAR RtcCalStr[]="RtcCal";

static JTIME FireDetectTime[4];
static JTIME AliveSendTime;         //Alive보낸 시각
static JTIME CommTestStartTime;     //통신 시험 시작 시간, 통신점검 명령을 받았을 때 녹색 LED를 켜기 위함
static DWORD FirstFireTimeStart;    //최초 화재를 발생한 시간
static int   FirstFireTimeIntvl;    //1.5초 + 랜덤1.5초
static LPVOID IniSavePtr;

static BYTE NoReplyAlarm;
static BYTE FireState;              //화재가 감지되면 1, 복구신호가 들어오면 0
#define FIRESTATE_SMOKE     0x01
#define FIRESTATE_HEAT      0x02
#define FIRESTATE_FLAME     0x04
#define FIRESTATE_FIRE      0x08    //연기와 열이 동시에 감지되었을 때
static BYTE RFRxLedTime;
static BYTE RFTxLedTime;
static BYTE LowBattLedOnTime;
static BYTE FireStateOffWhenFinishSound;
static BYTE RfmTaskTimer;
static BYTE ReceiverAckSound;       //수신기 ACK/Retry소리를 내도록 함
static BYTE TestFireCnt;            //화재시험 버튼을 눌러서 화재를 발생시킨 경우 1, 해제신호를 지연하기 위함
static BYTE InstalledSht;
static BYTE SmokeDetectRslt;        //열과 연기를 동시에 측정할 때 한쪽이 먼저 측정완료 되는 경우 재측정을 하지 않도록 이전 측정결과를 보존함
static BYTE HeatDetectRslt;         //      "
static BYTE OftenCheckFg;           //차동식 열감지기에서 온도차가 목표치보다 2도 차이가 나면 자주 체크하도록 하는 플래그임
static BYTE AutoReleaseFg;          //자동복귀형, 화재의 근원이 사라지면 자동 복구됨, 국내형은 수신기가 클리어함
static BYTE SelfFireSirenFg;        //화재발생시 자체 사이렌 출력 여부
static WORD DetectConstTemp;        //정온식 감지 온도
static WORD DetectTempDiff;         //차동식 감지 온도차
static WORD SmokeDetectLevel;       //연기감지 레벨
static WORD PowerOffWaitTime=300;   //화재관련 패킷을 다보낸 후 이 값이 1이되면 전원을 끔
static WORD RFTxLedWink;            //캐리어 송출동안 빠르게 깜빡거리는 모드
static BYTE SmokeDetectCnt;         //연기감지 후 축적기능을 처리함
static BYTE HeatDetectQty;          //열감지 횟수
static BYTE HeatDetectCnt;          //열감지 후 축적기능을 처리함
static BYTE FireDetectCnt;          //화재감지 후 축적기능을 처리함
static BYTE UseHighPrecisionRtc;    //고정밀도 RTC사용보드
static BYTE UseLowResistNtc;        //0:기존 1㏁ NTC, 1:50㏀ NTC
static BYTE SynchronizeTimeFg;      //동기된 시간임
static BYTE SynchronizedVector;     //시간동기 벡터 (400ms단위의 0~149의 타임순번임)  (중계기에게 받음)
static BYTE CommTestReplyOrder;     //통신점검할 때 응답하는 순번 (400ms단위)  (중계기에게 받음)
static BYTE ITP_SendCnt;            //2: 첫번째 보내는 패킷, 1:두번째 보내는 패킷, 모둠 ACK를 받을 시간에 감소함, 0이되면 SleepMode로
static BYTE ITP_WaitAck;            //ACK를 기다리는 중
static BYTE NoSleepMode;            //1이면 SleepMode로 진입하지 않음
static BYTE RfTrafficCnt;           //RF트래픽 수
static WORD AdcCalValue;
static WORD AdcBoardID;

#define NOW_SLEEP           1


static BYTE SensorAlarmFg;          //조건적 알람 Flag
//SensorAlarmFg
#define SAF_SECMAN          0x01    //수동경비
#define SAF_SECAUTO         0x02    //시간에의한 자동경비 (수동경비가 꺼져있을 때 작동)
#define SAF_LOWTEMP         0x04    //과냉 알림
#define SAF_HIGHTEMP        0x08    //과열 알림
#define SAF_BREAKDOWN       0x10    //정전 알림
#define SAF_FIREPREDETECT   0x20    //화재징후감지
#define SAF_SMOKESUPPLEMENT 0x40    //연기센서수증기보완
#define SAF_COHIGH          0x80    //이산화탄소 알람

static BYTE FireDetectPolicy;
#define SAF_FIREDISABLE         0
#define SAF_FIREHEATONLY        1
#define SAF_FIRESMOKEONLY       2
#define SAF_FIREHEATORSMOKE     3
#define SAF_FIREHEATANDSMOKE    4
#define SAF_FIREDEFMAX          5

static BYTE BatteryStatus;          //하위 2비트만 사용
#define BATSTAT_NONE    0
#define BATSTAT_LOW     1
#define BATSTAT_MIDDLE  2
#define BATSTAT_FULL    3
//Bit1-0: 메인배터리 0:없거나 불량, 1:Low, 2:Middle, 3:Full
//Bit3-2: 열감지센서 배터리
//Bit5-4: 연기센서 배터리
//Bit7-6: 예약

       BYTE BoardID=SPIRFM_BOARD;   //RFMODULE.C 에서 오토만 모듈을 구분하는 데 사용
       BYTE VolcanoBoardID;         //=VKBOARD_KOR (0:기본)
       BYTE DetectorType;           //=SMOKE_DETECTOR (0:기본)

static BYTE WakeupSource;

#define WKUPSRC_NONE        0
#define WKUPSRC_KEY         1
#define WKUPSRC_FIRE        2
#define WKUPSRC_RFRECV      3
#define WKUPSRC_RFCARR      4
#define WKUPSRC_SMOKECHK    5

SAVED_DATA SavedData;

#define CHANGESENSOR_QTY    10
typedef struct _CHANGESENSOR
    {
    BYTE OrgKeyNo;
    BYTE ChgKeyNo;
    } CHANGESENSOR;

static CHANGESENSOR ChangeSensors[CHANGESENSOR_QTY];



//-----------------------------------------------------------------------------
//      다른 모듈과 인터페이스 함수들
//-----------------------------------------------------------------------------
VOID  WINAPI SetOftenCheckFg(BOOL Fg) {OftenCheckFg=Fg;}
VOID  WINAPI UART_ErrUsrHandler(int Port, int ErrCode) {}           //Serial Error 사용자 처리
int   WINAPI GetSensorDetectLevel(VOID) {return SmokeDetectLevel;}
int   WINAPI GetDetectConstTemp(VOID) {return DetectConstTemp;}
int   WINAPI GetDetectTempDiff(VOID) {return DetectTempDiff;}
int   WINAPI GetNoReplyAlarm(VOID) {return NoReplyAlarm;}
VOID  WINAPI SetNoReplyAlarm(int NewValue) {NoReplyAlarm=NewValue;}
BOOL  WINAPI IsSpiRfmBoard(VOID) {return BoardID>=SPIRFM_BOARD;}
BOOL  WINAPI IsUseHighPrecisionRtc(VOID) {return UseHighPrecisionRtc;}
BOOL  WINAPI IsUseLowResistNtc(VOID) {return UseLowResistNtc;}
VOID  WINAPI GoSleepMode(VOID) {PowerOffWaitTime=NOW_SLEEP;}
LPVOID WINAPI GetIniSavePtr(VOID) {return IniSavePtr;}
int   WINAPI GetDeviceRegOrder(VOID) {return CommTestReplyOrder;}
VOID  WINAPI SetSendInterval(int Second) {}
VOID  WINAPI GetSensorTempAlarm(int *lpSensorFlag, int *lpLowTemp, int *lpHighTemp) {}
VOID  WINAPI SetSensorTempAlarm(int SensorFlag, int LowTemp, int HighTemp) {}
BOOL  WINAPI SensorTempAlarmRF(int *lpSensorFlag, int *lpLowTemp, int *lpHighTemp) {return 0;}
DWORD WINAPI SetWakeupInterval(DWORD Sec) {return 0;}
int   WINAPI SetDemoMode(int Mode) {return 0;}
int   WINAPI SetExchangeHeatFg(int Mode) {return 0;}
int   WINAPI GetAntenna(VOID) {return INTERNALANTENNA;}   //안테나를 선택함 (모과형보드에서만)
VOID  WINAPI SetAntenna(int Mode) {}
int   WINAPI GetBatteryStatus(VOID) {return BatteryStatus;}




//-----------------------------------------------------------------------------
//      통신점검 명령을 받으면 호출됨
//-----------------------------------------------------------------------------
VOID WINAPI SetNeedReplyStateFg(VOID)
    {
    if (FireState==0)
        {
        ITP_SendCnt=0;
        CommTestStartTime=GetTodayToSecond();
        PortOut(PO_GREEN_LED, ON);
        }
    }




//-----------------------------------------------------------------------------
//      통신점검을 종료함
//-----------------------------------------------------------------------------
LOCAL(VOID) FinishCommTest(VOID)
    {
    ITP_SendCnt=ITP_WaitAck=0;
    CommTestStartTime=0;
    PortOut(PO_GREEN_LED, OFF);
    }




//-----------------------------------------------------------------------------
//      Reset
//-----------------------------------------------------------------------------
VOID WINAPI Reboot(LPCSTR Reason)
    {
    WriteIniStr(InternalStr, "ResetReason", Reason);
    FlushIniFile();
    JOS_ENTER_CRITICAL();       //WatchDog Int가 꺼짐으로 재시작됨
    for (;;);
    }




//-----------------------------------------------------------------------------
//      치명적인 오류가 생겼을 때 LED 번쩍거림
//-----------------------------------------------------------------------------
VOID WINAPI FatalError(int Err)
    {
    switch (Err)
        {
        case FERR_SYSCLKCFG:    //0.5초 간격으로 번쩍임
            for (;;)
                {
                PortOut(PO_RED_LED, ON);  Delay1ms(10);
                PortOut(PO_RED_LED, OFF); Delay1ms(490);
                }

        case FERR_EXCEPTION:    //1초에 5번 깜빡임
            for (;;)
                {
                PortOut(PO_RED_LED, ON);  Delay1ms(10);
                PortOut(PO_RED_LED, OFF); Delay1ms(190);
                }

        case FERR_MEMORY:       //빠르게 두번을 깜빡이는데 1초마다 반복함
            for (;;)
                {
                PortOut(PO_RED_LED, ON);  Delay1ms(10);
                PortOut(PO_RED_LED, OFF); Delay1ms(190);
                PortOut(PO_RED_LED, ON);  Delay1ms(10);
                PortOut(PO_RED_LED, OFF); Delay1ms(790);
                }
        }
    }




//-----------------------------------------------------------------------------
//      감지기가 가진 시간을 중계기에게 전달하도록 함
//      (중계기가 시간을 요구할 때 호출됨)
//-----------------------------------------------------------------------------
VOID WINAPI SendSyncTime(int Sec)
    {
    AliveSendTime=GetTodayToSecond()+Sec;
    }




//-----------------------------------------------------------------------------
//      레지스터 검사에는 이상이 없는 데도 Sniff가 안되어 RFM칩을 리셋하고 재설정함
//-----------------------------------------------------------------------------
LOCAL(VOID) RFM_Reset(VOID)
    {
    AUTOMANSPI_Reinit();
    LOG_Add(LOG_RFMRECOVERY, 0, 0, 0);
    }




//-----------------------------------------------------------------------------
//      시간 교정 요청을 보낼 때인지 알려줌 (8초마다 깨어나 호출됨)
//
//      RFM이 뻗는 이유를 찾지 못해 임시방편으로
//      4시간 마다 RFM 리셋 (자동 통신점검을 4시간마다 하기 때문)
//-----------------------------------------------------------------------------
#define RFMRESETINTERVAL    (3600*4)
LOCAL(BOOL) IsSendTimeRtc(JTIME CurrTime)
    {
    BOOL   Rslt=FALSE;
    JTIME  JTime, CTime;
    static JTIME PrevTime=~0;

    CTime=CurrTime % 86400;
    if (PrevTime==~0) PrevTime=CTime;

    if (AliveSendTime!=0)
        {
        if (CurrTime>=AliveSendTime) {AliveSendTime=0; Rslt++;}
        }
    else{
        JTime=(SynchronizedVector+CommTestReplyOrder)*60+180;                   //자정 이후 시간동기할 때인지 확인
        if (PrevTime<JTime && CTime>=JTime) {RFM_Reset(); Rslt++;}
        else{
            JTime=PrevTime/RFMRESETINTERVAL*RFMRESETINTERVAL+RFMRESETINTERVAL;  //4시간단위로 RFM 리셋할 때가 되었는지 확인
            if (PrevTime<JTime && CTime>=JTime) RFM_Reset();                    //JTime>=86400일 수는 없기 때문에 자정은 제외됨
            }
        }

    PrevTime=CTime;
    return Rslt;
    }



//-----------------------------------------------------------------------------
//      암호화키를 리턴함
//-----------------------------------------------------------------------------
#if defined(ROZETAENCRYPTION) || defined(ROZETAENCRYPTION2)
DWORD WINAPI GetEncrypCode(VOID)
    {
    DWORD GroupNo=(DWORD)-1;

    if (RFC_IsSettingMode()==FALSE) GroupNo=SavedData.GroupNo;
    return CalculateCRC((LPCBYTE)&GroupNo, sizeof(DWORD), 0xE7A95F8D);
    }
#endif

#ifdef AES128ENCRYPTION
LPCBYTE WINAPI GetAesEncrypCode(VOID)
    {
    static BYTE AesKey[16];

    if (Peek(AesKey)==0)
        {
        if (RFC_IsSettingMode()==FALSE)
            {
            Poke(AesKey+0,  CalculateCRC((LPCBYTE)&SavedData.GroupNo, sizeof(DWORD), 0xE7A95F8D));
            Poke(AesKey+4,  CalculateCRC((LPCBYTE)&SavedData.ChannelNo, 1, 0x75F8DA9E));
            Poke(AesKey+8,  Peek(AesKey+0) ^ 0xE872FAC9);
            Poke(AesKey+12, Peek(AesKey+4) ^ 0xF5E67E8A);
            }
        else FillMem(AesKey, sizeof(AesKey), -1);
        }
    return AesKey;
    }
#endif




//-----------------------------------------------------------------------------
//      문제가 생긴 RF모듈이 어떤 장치인지 소리와 LED로 표현
//-----------------------------------------------------------------------------
LOCAL(VOID) NotifyReinitRFM(VOID)
    {
    PlayBeep(RecoveredSnd);

    PortOut(PO_GREEN_LED, ON);
    PortOut(PO_ORANGE_LED, ON);
    while (IsPlayingBeep())
        {
        Sleep(200);
        PortOut(PO_GREEN_LED, PO_NOT);
        PortOut(PO_ORANGE_LED, PO_NOT);
        }
    PortOut(PO_GREEN_LED, OFF);
    PortOut(PO_ORANGE_LED, OFF);
    }




//-----------------------------------------------------------------------------
//      화재 표시 LED 제어 (슬립모드를 위해 Off시 전류 흐르지 않도록 함)
//-----------------------------------------------------------------------------
LOCAL(VOID) FireLedOnOff(BOOL OnOff)
    {
    PortOut(PO_RED_LED, OnOff);
    }

LOCAL(VOID) FireLedFlashOnce(VOID)
    {
    FireLedOnOff(ON);
    LowBattLedOnTime=5;
    }


//-----------------------------------------------------------------------------
//      경종 제어
//-----------------------------------------------------------------------------
VOID WINAPI RingBell(int OnOff)
    {
    if (OnOff==BELL_ON) PutKey(KC_TestDblClk);  //실험을 위해 모든 감지기를 화재감지 상태로 만듦
    }



//-----------------------------------------------------------------------------
//      모든 화재발생 마킹/언마킹
//-----------------------------------------------------------------------------
VOID WINAPI SetFireState(int State)
    {
    LOG_Add(LOG_SETFIRESTATE, State, 0, 0);

    if (State!=0)
        {
        FireState|=State;
        FinishCommTest();                       //통신점검중 화재가 발생하면 통신점검은 중단, 화재가 우선임
        }
    else{
        FireState=0;
        ITP_WaitAck=ITP_SendCnt=0;
        ResetTempHistory();
        RingBell(BELL_OFF);     //PT_RESET 처리에서 SetFireState(0)을 호출함
        }
    FireLedOnOff(FireState!=0);
    if ((FireState & FIRESTATE_SMOKE)==0) SmokeDetectCnt=0;
    if ((FireState & FIRESTATE_HEAT)==0)  HeatDetectCnt=0;
    if ((FireState & FIRESTATE_FIRE)==0)  FireDetectCnt=0;
    }



//-----------------------------------------------------------------------------
//      주어진 화재 플래그만 끔
//-----------------------------------------------------------------------------
LOCAL(VOID) ClearFireState(int State)
    {
    FireState&=~State;
    FireLedOnOff(FireState!=0);
    if ((FireState & FIRESTATE_SMOKE)==0) SmokeDetectCnt=0;
    if ((FireState & FIRESTATE_HEAT)==0)  HeatDetectCnt=0;
    if ((FireState & FIRESTATE_FIRE)==0)  FireDetectCnt=0;
    }



//-----------------------------------------------------------------------------
//      사운드 재생
//-----------------------------------------------------------------------------
VOID WINAPI PlaySound(LPCBYTE Data)
    {
    PlayBeep(Data);
    }

VOID WINAPI MorseOut(LPCSTR Str)
    {
    PlayMorseBuzzer(Str);
    }




LOCAL(BOOL) IsSoundPlaying(VOID)
    {
    return IsPlayingBeep();
    }




//-----------------------------------------------------------------------------
//      RTC 교정에 사용할 편차를 로딩함
//-----------------------------------------------------------------------------
LOCAL(VOID) LoadRtcDeviation(int *lpDeviation, int *lpPeriod)
    {
    LPCSTR lp;
    CHAR Buff[40];

    if (GetIniStr(ConfigStr, RtcCalStr, Buff, sizeof(Buff))>0)
        {
        lp=ScanInt(Buff, lpDeviation);
        ScanInt(lp, lpPeriod);
        }
    }



//-----------------------------------------------------------------------------
//      RTC 교정 데이터를 저장함
//-----------------------------------------------------------------------------
LOCAL(VOID) SaveRtcDeviation(VOID)
    {
    int A,B, IniDeviation, IniPeriod=0, CurrDeviation, CurrPeriod;
    CHAR Buff[40];

    LoadRtcDeviation(&IniDeviation, &IniPeriod);   A=MulDiv(IniDeviation, 24*3600*1000, IniPeriod);
    RTC_GetDeviation(&CurrDeviation, &CurrPeriod); B=MulDiv(CurrDeviation, 24*3600*1000, CurrPeriod);
    Printf("SENSOR: Deviation (Ini=%d, Curr=%d)" CRLF, A,B);
    if (GetDiff(A,B)>=40)       //플래시 저장빈도를 줄이기 위해 하루의 편차로 환산한 후 40이상 차이가 날 때만 저장
        {
        wsprintf(Buff, Psnt2d, CurrDeviation, CurrPeriod);
        WriteIniStr(ConfigStr, RtcCalStr, Buff);
        FlushIniFile();
        }
    }




//-----------------------------------------------------------------------------
//      Flash에 저장된 환경 Data를 가져옴
//-----------------------------------------------------------------------------
LOCAL(VOID) LoadConfigData(VOID)
    {
    int Deviation, Period=0;

    SavedData.GroupNo=GetIniInt(InternalStr, GroupStr, DEFGROUPNO);
    SavedData.ChannelNo=GetIniInt(InternalStr, ChannelStr, DEFCHANNELNO);
    SavedData.RFPower=GetIniInt(InternalStr, PowerStr, DEFRFPOWER);
    SavedData.DeviceNo=GetIniInt(InternalStr, DevNoStr, DEFAULTDEVICENO);
    SavedData.RepeaterMode=GetIniInt(InternalStr, RepeaterStr, DEFREPEATERMODE);    //태국형에서 혹시 쓸지몰라서
    SysDebugFg=GetIniInt(InternalStr, SysDebugStr, 0);

    SmokeDetectLevel=GetIniInt(InternalStr, SmokeLevelStr, SMOKE_DETECT_LEVEL);
    DetectConstTemp=GetIniInt(InternalStr, DetectConstTempStr, DEFCONSTDETECTTEMP)*100;
    DetectTempDiff=GetIniInt(InternalStr, DetectTempDiffStr, DETECTTEMPDIFF)*100;

    SensorAlarmFg=GetIniInt(ConfigStr, SensorAlarmFgStr, SAF_SECAUTO);
    FireDetectPolicy=GetIniInt(ConfigStr, FireDetectPolicyStr, SAF_FIREHEATORSMOKE);

    LoadRtcDeviation(&Deviation, &Period);
    if (Period!=0)
        {
        int A;
        A=MulDiv(Deviation, 24*3600*1000, Period);                      //하루 편차가 30초를 초과하면 사용하지 않음
        if (A>-RTCDEVIATIONMAX && A<RTCDEVIATIONMAX) RTC_SetDeviation(Deviation, Period);   //최초 RTC_CalcDeviation()함수가 호출될 때 적용됨
        }
    }


LOCAL(VOID) SaveConfigData(VOID)
    {
    WriteIniInt(InternalStr, GroupStr, SavedData.GroupNo);
    WriteIniInt(InternalStr, ChannelStr, SavedData.ChannelNo);
    WriteIniInt(InternalStr, PowerStr, SavedData.RFPower);
    WriteIniInt(InternalStr, DevNoStr, SavedData.DeviceNo);

    WriteIniInt(ConfigStr, SensorAlarmFgStr, SensorAlarmFg);
    WriteIniInt(ConfigStr, FireDetectPolicyStr, FireDetectPolicy);
    }




//-----------------------------------------------------------------------------
//      키변환 테이블을 저장하고 불러옵니다
//-----------------------------------------------------------------------------
LOCAL(VOID) LoadKeyRemapTable(VOID)
    {
    int  I, Org, Chg;
    char Buff[20];

    for (I=0; I<CHANGESENSOR_QTY; I++)
        {
        GetIniStrN(RemapKeyStr, I, Buff, sizeof(Buff));
        Org=Chg=0;
        Jsscanf(Buff, Psnt2d, &Org, &Chg);
        ChangeSensors[I].OrgKeyNo=(BYTE)Org;
        ChangeSensors[I].ChgKeyNo=(BYTE)Chg;
        }
    }



//-----------------------------------------------------------------------------
//      키를 바꿈
//-----------------------------------------------------------------------------
LOCAL(int) RemapKey(int Key)
    {
    int I;
    char Buff[64];

    for (I=0; I<CHANGESENSOR_QTY; I++)
        {
        if (ChangeSensors[I].OrgKeyNo==Key)
            {
            Key=ChangeSensors[I].ChgKeyNo;
            Printf("SENSOR: Remapped Key=%d"CRLF, Key);
            if (Key==255)
                {
                if (GetIniStrN(RemapKeyStr, I, Buff, sizeof(Buff))>0)
                    {
                    Mon_Control(COM_DEBUG, "CONTROL", (LPSTR)NextWord(Buff, 2), NullStr);
                    PlaySound(KeyClickSnd);
                    ReceiverAckSound=1;
                    }
                }
            break;
            }
        }
    return Key;
    }



//-----------------------------------------------------------------------------
//      리뱁테이블에 추가 (테이블이 꽉차면 FALSE 리턴)
//-----------------------------------------------------------------------------
LOCAL(BOOL) AddRemapKey(int OrgKey, int ChgKey, LPCSTR Option)
    {
    int  I, Rslt=FALSE;
    char Buff[20];

    for (I=0; I<CHANGESENSOR_QTY; I++)
        {
        if (ChangeSensors[I].OrgKeyNo==OrgKey) goto PutChgKey;
        }

    //테이블에 없는 경우 빈곳에 넣음
    for (I=0; I<CHANGESENSOR_QTY; I++)  //위의 루프와 합치면 중복으로 추가될 수 있음
        {
        if (ChangeSensors[I].OrgKeyNo==0)
            {
            PutChgKey:
            ChangeSensors[I].OrgKeyNo=OrgKey;
            ChangeSensors[I].ChgKeyNo=ChgKey;
            if (ChgKey==255) WriteIniStrN(RemapKeyStr, I, Option);
            else{
                wsprintf(Buff, Psnt2d, OrgKey, ChgKey);
                WriteIniStrN(RemapKeyStr, I, Buff);
                }
            Rslt=TRUE;
            break;
            }
        }
    return Rslt;
    }



//-----------------------------------------------------------------------------
//      리맵테이블 삭제
//-----------------------------------------------------------------------------
LOCAL(BOOL) DelRemapKey(int OrgKey)
    {
    int I, Rslt=FALSE;

    for (I=0; I<CHANGESENSOR_QTY; I++)
        {
        if (ChangeSensors[I].OrgKeyNo==OrgKey)
            {
            ChangeSensors[I].OrgKeyNo=0;
            ChangeSensors[I].ChgKeyNo=0;
            WriteIniStrN(RemapKeyStr, I, "0 0");
            Rslt=TRUE;
            break;
            }
        }
    return Rslt;
    }



//-----------------------------------------------------------------------------
//      시간이 경과하면 LED Off (10ms마다 틱 인터럽트에서 호출함)
//-----------------------------------------------------------------------------
LOCAL(VOID) LedOnOff(VOID)
    {
    #if KFI_INSPECTION
    static BYTE BootLedOnTimer=30;
    #else
    static BYTE BootLedOnTimer=90;      //RF Sniff가 Off 된것을 알리기 위해 LED를 3번 번쩍임
    #endif

    if (BootLedOnTimer!=0)
        {
        BootLedOnTimer--;
        if (BootLedOnTimer==60 || BootLedOnTimer==0)
            {
            PortOut(PO_RED_LED, OFF);
            PortOut(PO_GREEN_LED, OFF);
            PortOut(PO_ORANGE_LED, OFF);
            }
        else if (BootLedOnTimer==30)
            {
            PortOut(PO_RED_LED, ON);
            PortOut(PO_GREEN_LED, ON);
            PortOut(PO_ORANGE_LED, ON);
            }
        goto ProcExit;
        }

    if (RFRxLedTime!=0 && --RFRxLedTime==0) {if (CommTestStartTime==0) PortOut(PO_GREEN_LED, OFF);}
    if (RFTxLedTime!=0 && --RFTxLedTime==0) PortOut(PO_ORANGE_LED, OFF);

    if (LowBattLedOnTime!=0 && --LowBattLedOnTime==0 && FireState==0) FireLedOnOff(OFF);

    if (RFTxLedWink)
        {
        RFTxLedWink++;
        if ((RFTxLedWink & 0x0F)==0)
            {
            PortOut(PO_ORANGE_LED, ON);
            RFTxLedTime=1;
            }
        }

    if (FireStateOffWhenFinishSound && IsSoundPlaying()==FALSE) {SetFireState(OFF); FireStateOffWhenFinishSound=0;}

    ProcExit:;
    }


VOID WINAPI RxLedOn(VOID)
    {
    PortOut(PO_GREEN_LED, ON);
    RFRxLedTime=10;
    }


VOID WINAPI TxLedControl(int Mode)
    {
    if (Mode!=TXLED_OFF)
        {
        PortOut(PO_ORANGE_LED, ON);
        RFTxLedTime=10;
        }
    RFTxLedWink=(Mode==TXLED_WINK);
    }




//-----------------------------------------------------------------------------
//      키상태 수집
//-----------------------------------------------------------------------------
static KEYCODETABLE KeyCodeTable[]=
    {//DownCode,    DblClickCode,   LongDown1Code,  LongDown2Code,  KeyUpCode
    {KC_TestDown,   KC_TestDblClk,  KC_TestLong1,   KC_TestLong2,   KC_TestUp},
    };

LOCAL(UINT) CollectKeyState(VOID)
    {
    UINT Key=0;

    if (PortIn(PI_TESTKEYII)==0 || PortIn(PI_EXTWKUP)==0) Key|=1;
    return Key;
    }



//-----------------------------------------------------------------------------
//      전원 켜기전에 이미 눌려있는 키를 어떻게 할지를 결정해줌
//      ScanKey() 에서 호출함
//-----------------------------------------------------------------------------
UINT WINAPI IgnoreFirstPushedKey(int Key)
    {
    UINT OldKeyII=0;
    //if (Key & 1) OldKeyII|=1;         //이키를 누른채로 전원을 켤 경우 다른 기능을 처리하기 위함
    //if (Key & 0x20) OldKeyII|=0x20;   //전원ON시1이였다가 시간 경과루 0이됨, 전원을 빼면 몇초지난후 1이됨
    return OldKeyII;
    }




//-----------------------------------------------------------------------------
//      키클릭음 처리, ScanKey()에서 호출함
//-----------------------------------------------------------------------------
VOID WINAPI PlayKeyClickSound(int KeyCode)
    {
    if (KeyCode==KC_TestDown) PlaySound(KeyClickSnd);
    }




//-----------------------------------------------------------------------------
//      처리할 키가 있거나 키가 눌려있으면 TRUE리턴
//-----------------------------------------------------------------------------
LOCAL(BOOL) IsBusyKey(VOID)
    {
    int I, Rslt;

    if ((Rslt=CheckKey())==FALSE)
        {
        if (CollectKeyState() & 1) Rslt++;  //Test키가 눌린경우
        else{
            for (I=0; I<countof(KeyCodeTable); I++)
                {
                if (KeyCodeTable[I].DblClickTime!=0) {Rslt++; break;}
                }
            }
        }
    return Rslt;
    }




//-----------------------------------------------------------------------------
//      1ms 마다 처리 (틱 인터럽트에서 호출함)
//-----------------------------------------------------------------------------
VOID WINAPI _1msProc(VOID)
    {
    static BYTE _10ms;

    if (++_10ms>=10)
        {
        _10ms=0;
        RfmTaskTimer=1;
        if (PowerOffWaitTime>NOW_SLEEP) PowerOffWaitTime--;

        ScanKey(CollectKeyState(), KeyCodeTable, countof(KeyCodeTable));
        K_TimerProc();
        PlayBeepProc();
        LedOnOff();
        #ifdef USE_JOS
        JOSTimeTick();
        #endif
        }
    }



//-----------------------------------------------------------------------------
//      반응 소리를 낼지를 알려줌
//-----------------------------------------------------------------------------
LOCAL(BOOL) IsNeedReaction(int PacketType)
    {
    return PacketType==PT_SWITCH || (PacketType==PT_CONTROL && ReceiverAckSound!=0);
    }



//-----------------------------------------------------------------------------
//      화재발생시 경고음를 재생함
//-----------------------------------------------------------------------------
VOID WINAPI PlayFireOccurSound(VOID)
    {
    }



//-----------------------------------------------------------------------------
//      화재 패킷에 해당하는 어레이 변수용 인덱스를 알려줌
//-----------------------------------------------------------------------------
static CONST BYTE FirePacketTypeList[]={PT_SMOKE, PT_HEAT, PT_FLAME, PT_FIRE};
LOCAL(int) GetIndexFromPT(int PT)
    {
    int I=4;

    if      (PT==PT_SMOKE) I=0;
    else if (PT==PT_HEAT)  I=1;
    else if (PT==PT_FLAME) I=2;
    else if (PT==PT_FIRE)  I=3;
    return I;
    }




//-----------------------------------------------------------------------------
//      화재 패킷에 해당하는 FIRESTATE를 알려줌
//-----------------------------------------------------------------------------
LOCAL(int) GetFireStateFromIndex(int I)
    {
    static CONST BYTE FStateList[]={FIRESTATE_SMOKE, FIRESTATE_HEAT, FIRESTATE_FLAME, FIRESTATE_FIRE};
    return FStateList[I];
    }




//-----------------------------------------------------------------------------
//      고정밀도 시간을 RTC에 설정함
//-----------------------------------------------------------------------------
LOCAL(UINT) SetLocalTimeHighPrecision(JTIME NewTime_Sec, UINT NewTime_ms)
    {
    SYSTEMTIME ST, PrevST;

    if (NewTime_ms!=0) {Sleep(1000-NewTime_ms); NewTime_Sec++;}     //RV3032는 1/100초 자리는 항상 0으로 설정되기 때문
    UnpackTotalSecond(&ST, NewTime_Sec);
    GetLocalTime(&PrevST);
    SetLocalTime(&ST);
    SynchronizeTimeFg=1;
    return (Limit(PackTotalSecond(&PrevST)-NewTime_Sec, -127, 127)<<8) | (PrevST.wMilliseconds/10);
    }




//-----------------------------------------------------------------------------
//      RFManagerTask()에서 호출함
//
//      RF 전송/수신 후 호출됨
//-----------------------------------------------------------------------------
#define REPEATER_RECVTIME_ADJUST    258 //스코프로 측정한 결과
LOCAL(VOID) RFSendRecvCB(int CBReason, CONST RF_PACKET *P)
    {
    int PacketType=0;
    static BYTE AliveRetryCnt;

    if (P) PacketType=P->PacketType;
    switch (CBReason)
        {
        case RFCB_SENDOK:           //RFM에 전송을 마칠 때마다 호출됨
            if (PacketType==PT_STATE) {FinishCommTest(); GoSleepMode();}        //통신점검의 응답인데 ACK를 받지 않으므로 바로 슬립모드 진입하도록 함
            break;

        case RFCB_RECVACK:          //전송 후 ACK응답을 받음
            AliveRetryCnt=0;
            if (PacketType==PT_COLLECTACK) {ITP_SendCnt=ITP_WaitAck=0; FirstFireTimeStart=0;}   //FirstFireTimeStart=0은 최초화재 집중전송처리 전에 COLLACK를 받은 경우 Sleep모드 진입을 하는데 그렇게 되면 GetTickCount()가 동작하지 않아 재전송ACK를 보내지 않게 되고 그로인해 화재복구가 안되는 현상이 생김

            NoReplyAlarm=0;
            PacketType=P->Data[0];  //송신 PacketType

            #if REPEATTEST==0
            if (P->PacketType==PT_ACK && (PacketType==PT_SWITCH || PacketType==PT_EXTINGUISH)) PlaySound(DoMiSol); //아래에서 시간 맞추느라고 응답 소리가 늦게 늘려서 미리 소리를 냄
            #endif

            if (P->DataLen>=7 && P->Data[2]==OC_SETTIME)
                {
                UINT  ms, D;
                JTIME JT;

                JT=Peek(P->Data+3); ms=0;
                if (P->DataLen>=10)
                    {
                    ms=P->Data[7]*10;
                    if ((ms+=REPEATER_RECVTIME_ADJUST)>=1000) {ms-=1000; JT++;} //패킷 날라온 시간 보정
                    SynchronizedVector=P->Data[8];
                    CommTestReplyOrder=P->Data[9];
                    }
                RTC_CalcDeviation(JT, ms);
                D=SetLocalTimeHighPrecision(JT, ms);
                SaveRtcDeviation();
                LOG_Add(LOG_RECVTIME1, 0, D>>8, D&0xFF);
                }

            if (P->PacketType==PT_ACK && PacketType==PT_SWITCH)
                {
                #if REPEATTEST
                PutKey(KC_TestUp);
                #endif
                }
            GoSleepMode();
            break;

        case RFCB_PKTRECV:          //수신패킷을 처리하고 호출됨
            RfTrafficCnt++;
            GoSleepMode();          //RF를 수신하여 처리하고 바로 전원을 끄기 위함, P==NULL이면 깨졌거나 다른 중계기 데이터임
            break;

        case RFCB_NOREPLY:
            if (IsNeedReaction(PacketType)) PlaySound(NoReplySnd);
            break;

        case RFCB_NOACK:            //10회 시도 후 호출
            NoReplyAlarm=1;
            if (IsNeedReaction(PacketType))
                {
                ReceiverAckSound=OFF;
                #if REPEATTEST
                PutKey(KC_TestUp);
                #else
                PlaySound(NoReplySndII);
                #endif
                break;
                }
            if (PacketType==PT_BATTERY)
                {
                if (++AliveRetryCnt>=3) AliveRetryCnt=0;
                else SendSyncTime(CommTestStartTime ? 10:180);                  //Alive 패킷의 응답을 못받은 경우 3분 후에 다시 보내도록 함
                }
            RFM_Reset();    //AUTOMANSPI_RecoverChip(TRUE);
            GoSleepMode();
            break;
        }
    }




//-----------------------------------------------------------------------------
//          지연된 포트처리
//-----------------------------------------------------------------------------
#define DELAYEDPORTQTY      2

typedef struct _DELAYEDPORT
    {
    BYTE PortNo;        //송신측에서 준 포트번호임
    BYTE OnOff;
    BYTE DelayedTime;   //초단위
    DWORD StartTick;
    } DELAYEDPORT;

static DELAYEDPORT DelayedPortList[DELAYEDPORTQTY];
static BYTE AuxIoCtlPorts[DELAYEDPORTQTY]={255, 255};

LOCAL(VOID) AddDelayedPort(int PortNo, int OnOff, int DelayedTime)
    {
    int I;
    DELAYEDPORT *DP;

    for (I=0; I<DELAYEDPORTQTY; I++)
        {
        DP=DelayedPortList+I;
        if (DP->StartTick==0)
            {
            DP->PortNo=PortNo;
            DP->OnOff=OnOff;
            DP->DelayedTime=DelayedTime;
            DP->StartTick=GetTickCount();
            break;
            }
        }
    }


LOCAL(VOID) DelayedPortProc(VOID)
    {
    int I;
    DELAYEDPORT *DP;

    for (I=0; I<DELAYEDPORTQTY; I++)
        {
        DP=DelayedPortList+I;
        if (DP->StartTick!=0 && GetTickCount()-DP->StartTick >= DP->DelayedTime*1000)
            {
            PortOut(AuxIoCtlPorts[DP->PortNo], DP->OnOff);
            DP->StartTick=0;
            }
        }
    }



//-----------------------------------------------------------------------------
//      외부 I/O포트 제어
//-----------------------------------------------------------------------------
VOID WINAPI AuxIoControl(LPBYTE RetData, LPCBYTE SetData)
    {
    int PortNo, Data, IsGet, IsAnalog;

    IsGet=SetData[0] & 0x40;
    IsAnalog=SetData[0] & 0x80;
    PortNo=SetData[0] & 0x3F;
    Data=PeekW(SetData+1);

    RetData[0]=SetData[0] & ~0x40;

    if (IsAnalog)
        {
        if (IsGet) Printf("SENSOR: Analog Input (Port=%d)"CRLF, PortNo);
        else       Printf("SENSOR: Analog Out (Port=%d, Data=%d)"CRLF, PortNo, Data);
        }
    else{
        int Bits, OutData, DelayedTime, OnOff;

        Bits=Data>>12;
        OutData=Data&0xFFF;

        if (IsGet) Printf("SENSOR: Digital Input (Port=%d, Bits=%d)"CRLF, PortNo, Bits);
        else{
            Printf("SENSOR: Digital Out (Port=%d, Bits=%d, Data=%X)"CRLF, PortNo, Bits, OutData);
            if (PortNo<DELAYEDPORTQTY && Bits==1)
                {
                OnOff=OutData & 1;
                DelayedTime=OutData>>4;
                Data=PortOut(AuxIoCtlPorts[PortNo], OnOff);
                if (DelayedTime>0) AddDelayedPort(PortNo, OnOff^1, DelayedTime);
                }
            }
        }
    PokeW(RetData+1, Data);         //실제는 현재 포트 상태를 리턴할 것
    }



//-----------------------------------------------------------------------------
//      화재감지 정책 설정
//-----------------------------------------------------------------------------
int WINAPI SetFireDetectPolicy(int NewPolicy)
    {
    int Policy=SAF_FIREDISABLE;

    Policy=FireDetectPolicy;
    if (NewPolicy>=0)
        {
        if (Policy<SAF_FIREDEFMAX) FireDetectPolicy=Policy;
        }
    else{
        if (DetectorType==SMOKE_DETECTOR)
            {
            if (Policy!=SAF_FIREDISABLE) Policy=SAF_FIRESMOKEONLY;
            }
        else if (DetectorType==FIXED_HEAT_DETECTOR || DetectorType==ROR_HEAT_DETECTOR)
            {
            if (Policy!=SAF_FIREDISABLE) Policy=SAF_FIREHEATONLY;
            }
        }
    return Policy;
    }



//-----------------------------------------------------------------------------
//      센서 설정을 함
//-----------------------------------------------------------------------------
VOID WINAPI SensorSetting(int *lpSetData1, int *lpSetData2)
    {
    int SetData1;

    SetData1=*lpSetData1;
    *lpSetData1=SensorAlarmFg;
    *lpSetData2=SetFireDetectPolicy(-1);

    if (SetData1>=0)
        {
        SensorAlarmFg&=SAF_SECMAN|SAF_SECAUTO|SAF_LOWTEMP|SAF_HIGHTEMP;  //이 4개를 제외하구 모두 0으로
        SensorAlarmFg|=SetData1 & ~(SAF_SECMAN|SAF_SECAUTO|SAF_LOWTEMP|SAF_HIGHTEMP);
        WriteIniInt(ConfigStr, SensorAlarmFgStr, SensorAlarmFg);
        }
    }




//-----------------------------------------------------------------------------
//      CPU의 내부 온도를 읽음
//-----------------------------------------------------------------------------
#define TEMP30_CAL  (*(WORD*)0x1FF8007A)    //=678 ... RM0367-p985
#define TEMP130_CAL (*(WORD*)0x1FF8007E)    //=913, 칩마다 다름
LOCAL(int) GetCpuTemp(VOID)
    {
    int Temp, VRef;

    Temp=ADC_GetValueAverage(ADC_CHSELR_CHSEL18, 16);
    VRef=ADC_GetValueAverage(ADC_CHSELR_CHSEL17, 16);
    Temp=Temp*1600/VRef;
    Temp=(Temp-TEMP30_CAL) * (130-30) / (TEMP130_CAL - TEMP30_CAL) + 30;
    //Printf("CPU Temp=%d (30:%d~130:%d)"CRLF, Temp, TEMP30_CAL, TEMP130_CAL);
    return Temp;
    }




//-----------------------------------------------------------------------------
//  Vdd를 측정하여 mV단위로 리턴함
//
//  Vdd측정 보정 ... 실제전압 3.01 -> 측정전압 2.91
//                   실제전압 2.50 -> 측정전압 2.428
//
// 위 2개의 자료로 보정식을 만들면, 보정전압= 측정전압*1.058-0.069
//-----------------------------------------------------------------------------
#define ADCMAXVALUE             4095
#define ADCREFINTVALUE          1224
int WINAPI MeasureVdd(int AdcCal)
    {
    int Vdd;

    Vdd=ADC_GetValueAverage(ADC_CHSELR_CHSEL17, 16)+AdcCal;
    Vdd=DIV_ROUNDUP(ADCREFINTVALUE*ADCMAXVALUE, Vdd);
    return ((Vdd*8667+4096)>>13)-69;        //1058/1000==8667/8192
    }




//-----------------------------------------------------------------------------
//      배터리 전압검사 (CPU가 깨어날 때마다 호출됨, 보통 8초마다)
//-----------------------------------------------------------------------------
LOCAL(VOID) CheckBatteryLevel(VOID)
    {
    int Vdd;
    static INT8 Judge3000, Judge2700;

    if (DetectorType!=SMOKE_HEAT_DETECTOR && VolcanoBoardID<VKBOARD_UAEALONE && PortIn(PI_BREAKDOWN)==0) BatteryStatus=BATSTAT_NONE;
    else{
        Vdd=MeasureVdd(AdcCalValue);
        Judge3000=CompareUsingHysteresis(Judge3000, Vdd, 3000, 50);
        Judge2700=CompareUsingHysteresis(Judge2700, Vdd, 2700, 50);

        if (Judge3000>0) BatteryStatus=BATSTAT_FULL;
        else if (Judge3000<0 && Judge2700>0) BatteryStatus=BATSTAT_MIDDLE;
        else BatteryStatus=BATSTAT_LOW; //if (Judge2700<0)
        }
    }



//-----------------------------------------------------------------------------
//      RTC 인터럽트에서 Alarm이 발생했을 때 호출됨
//      여기 호출을 위해서는 RTC_SetAlarmTime(&ST, TRUE) 초기화를 해야함
//-----------------------------------------------------------------------------
VOID WINAPI RTC_AlarmAEventCB(VOID)
    {
    PutKeyIT(KC_RtcAlarm);
    }



//-----------------------------------------------------------------------------
//      SystemCoreClock=4MHz
//-----------------------------------------------------------------------------
LOCAL(VOID) SystemClock_HSI(VOID)
    {
    RCC_ClkInitTypeDef CI;
    RCC_OscInitTypeDef OI;

    OI.OscillatorType=RCC_OSCILLATORTYPE_HSI;
    OI.HSEState=RCC_HSE_OFF;
    OI.HSIState=RCC_HSI_ON;
    OI.PLL.PLLState=RCC_PLL_ON;
    OI.PLL.PLLSource=RCC_PLLSOURCE_HSI;
    OI.PLL.PLLMUL=RCC_PLL_MUL4;
    #if CORECLOCK==8000000
    OI.PLL.PLLDIV=RCC_PLL_DIV4;
    #else
    OI.PLL.PLLDIV=RCC_PLL_DIV2;
    #endif
    OI.HSICalibrationValue=0x10;
    if (HAL_RCC_OscConfig(&OI)!=HAL_OK) FatalError(FERR_SYSCLKCFG);

    CI.ClockType=(RCC_CLOCKTYPE_SYSCLK|RCC_CLOCKTYPE_HCLK|RCC_CLOCKTYPE_PCLK1|RCC_CLOCKTYPE_PCLK2);
    CI.SYSCLKSource=RCC_SYSCLKSOURCE_PLLCLK;
    #if CORECLOCK==8000000
    CI.AHBCLKDivider=RCC_SYSCLK_DIV2;
    #else
    CI.AHBCLKDivider=RCC_SYSCLK_DIV1;
    #endif
    CI.APB1CLKDivider=RCC_HCLK_DIV1;
    CI.APB2CLKDivider=RCC_HCLK_DIV1;
    if (HAL_RCC_ClockConfig(&CI, FLASH_LATENCY_1)!=HAL_OK) FatalError(FERR_SYSCLKCFG);

    __HAL_RCC_PWR_CLK_ENABLE();
    __HAL_PWR_VOLTAGESCALING_CONFIG(PWR_REGULATOR_VOLTAGE_SCALE1);
    }




#if 0
//-----------------------------------------------------------------------------
//      SystemCoreClock=2.097152MHz
//-----------------------------------------------------------------------------
LOCAL(VOID) SystemClock_MSI(VOID)
    {
    RCC_ClkInitTypeDef CI;
    RCC_OscInitTypeDef OI;

    OI.OscillatorType=RCC_OSCILLATORTYPE_MSI;
    OI.MSIState=RCC_MSI_ON;
    OI.MSIClockRange=RCC_MSIRANGE_5;
    OI.MSICalibrationValue=0x00;
    OI.PLL.PLLState=RCC_PLL_NONE;
    HAL_RCC_OscConfig(&OI);

    CI.ClockType=(RCC_CLOCKTYPE_SYSCLK|RCC_CLOCKTYPE_HCLK|RCC_CLOCKTYPE_PCLK1|RCC_CLOCKTYPE_PCLK2);
    CI.SYSCLKSource=RCC_SYSCLKSOURCE_MSI;
    CI.AHBCLKDivider=RCC_SYSCLK_DIV1;
    CI.APB1CLKDivider=RCC_HCLK_DIV1;
    CI.APB2CLKDivider=RCC_HCLK_DIV1;
    HAL_RCC_ClockConfig(&CI, FLASH_LATENCY_0);

    __HAL_RCC_PWR_CLK_ENABLE();
    __HAL_PWR_VOLTAGESCALING_CONFIG(PWR_REGULATOR_VOLTAGE_SCALE3);
    }
#endif





//-----------------------------------------------------------------------------
//      Wake Up 관련 함수
//-----------------------------------------------------------------------------
#define PI_TESTKEYII_BIT    (1<<(PI_TESTKEYII&0xF)) //PA9
#define PI_RFWKUP1_BIT      (1<<(PI_RFWKUP1&0xF))   //PA10
#define PI_EXRTC_INT_BIT    (1<<(PI_EXRTC_INT&0xF)) //PA4
#define PI_EXTWKUP_BIT      (1<<(PI_EXTWKUP&0xF))   //PA12


VOID EXTI4_15_IRQHandler(VOID)
    {
    UINT PR;

    JOS_ENTER_CRITICAL();
    PR=EXTI->PR;
    if (PR & PI_TESTKEYII_BIT)
        {
        EXTI->PR=PI_TESTKEYII_BIT;
        if (WakeupSource==0) WakeupSource=WKUPSRC_KEY;
        }

    if (PR & PI_RFWKUP1_BIT)
        {
        AUTOMANSPI_MarkExistRecvData();
        EXTI->PR=PI_RFWKUP1_BIT;
        if (WakeupSource==0) WakeupSource=WKUPSRC_RFRECV;
        }

    if (PR & PI_EXRTC_INT_BIT)
        {
        EXTI->PR=PI_EXRTC_INT_BIT;
        if (WakeupSource==0) WakeupSource=WKUPSRC_SMOKECHK;
        }

    if (PR & PI_EXTWKUP_BIT)
        {
        EXTI->PR=PI_EXTWKUP_BIT;
        if (WakeupSource==0) WakeupSource=WKUPSRC_KEY;
        }

    JOS_EXIT_CRITICAL();
    }



//-----------------------------------------------------------------------------
//      내부 RTC 알람이나 타이머로 깨어날 때 호출됨
//-----------------------------------------------------------------------------
VOID WINAPI RTC_WakeupCB(int CBReason)
    {
    if (WakeupSource==0) WakeupSource=WKUPSRC_SMOKECHK;
    }



//-----------------------------------------------------------------------------
//      ADC를 활성 또는 비활성화함
//-----------------------------------------------------------------------------
LOCAL(VOID) AdcOnOff(BOOL OnOff)
    {
    if (OnOff)
        {
        InitPort(PI_IR_SMOKE_SIG, GPIO_MODE_ANALOG, GPIO_NOPULL, 0);
        if (DetectorType==SMOKE_HEAT_DETECTOR)
            {
            InitPort(PI_INT_NTC, GPIO_MODE_ANALOG, GPIO_NOPULL, 0);
            #if USE2NTC
            InitPort(PI_INT_NTC2, GPIO_MODE_ANALOG, GPIO_NOPULL, 0);
            #endif
            }
        InitADC(ADC_SAMPLETIME_39CYCLES_5);
        AdcCalValue=ADC_GetCalibrationValue();
        }
    else{
        ReleaseADC();
        GPIO_DeInit(PI_IR_SMOKE_SIG);
        if (DetectorType==SMOKE_HEAT_DETECTOR)
            {
            GPIO_DeInit(PI_INT_NTC);
            #if USE2NTC
            GPIO_DeInit(PI_INT_NTC2);
            #endif
            }
        }
    }



//-----------------------------------------------------------------------------
//      DAC를 활성 또는 비활성화함
//-----------------------------------------------------------------------------
LOCAL(VOID) DacOnOff(BOOL OnOff)
    {
    #if !defined(STM32L071xx) && !defined(STM32L081xx)
    if (VolcanoBoardID!=VKBOARD_KOR)    //한국형 New보드에서는 PO_SOUND_DACP를 RTCINT로 사용함
        {
        if (OnOff)
            {
            InitPort(PO_SOUND_DACP, GPIO_MODE_ANALOG, GPIO_NOPULL, 0);
            InitPortOutputPP(PO_SPEAKEREN);
            InitDAC(WAVEOUT_DAC_CH);
            }
        else{
            ReleaseDAC(WAVEOUT_DAC_CH);
            GPIO_DeInit(PO_SOUND_DACP);
            InitPortOutputPP(PO_SPEAKEREN);
            }
        }
    #endif
    }



//-----------------------------------------------------------------------------
//      온도센서 전원을 켜고 온습도를 읽음
//-----------------------------------------------------------------------------
VOID WINAPI ReadTempHumi(int *lpTemp, int *lpHumi)
    {
    *lpTemp=*lpHumi=0;

    if (InstalledSht)
        {
        PortOut(PO_SENSORPWR, ON);
        Sleep(10);      //메뉴얼에 최대 1ms, 하지만 하드웨어 콘덴서 충전타임도 필요함
        SHT_Read(lpTemp, lpHumi);
        PortOut(PO_SENSORPWR, OFF);
        }
    }




//-----------------------------------------------------------------------------
//      테스트 키를 눌렀을 때 어떤 종류의 화재를 전송할지를 알려줌
//-----------------------------------------------------------------------------
LOCAL(int) GetTestKeyPktType(VOID)
    {
    int PT=0;
    static BYTE Toggle;

    switch (DetectorType)
        {
        case CALLPOINT_DETECTOR: PT=PT_MANFIRE; break;
        case SMOKE_DETECTOR: PT=PT_SMOKE; break;
        case FIXED_HEAT_DETECTOR:
        case ROR_HEAT_DETECTOR:  PT=PT_HEAT; break;
        case FLAME_DETECTOR: PT=PT_FLAME; break;

        case SMOKE_HEAT_DETECTOR:
            switch (FireDetectPolicy)
                {
                case SAF_FIRESMOKEONLY: PT=PT_SMOKE; break;
                case SAF_FIREHEATORSMOKE: PT=(Toggle==0) ? PT_SMOKE:PT_HEAT; Toggle^=1; break;
                case SAF_FIREHEATONLY: PT=PT_HEAT; break;
                case SAF_FIREHEATANDSMOKE: PT=PT_FIRE; //break;
                }
            //break;
        }
    return PT;
    }



//-----------------------------------------------------------------------------
//          RV3032 시간과 온도를 표시 (테스트용)
//-----------------------------------------------------------------------------
LOCAL(VOID) RV3032_PrintTimeTemp(VOID)
    {
    SYSTEMTIME ST;

    GetLocalTime(&ST);
    Printf("RV3032 RTC Time: %d-%02d-%02d %02d:%02d:%02d %.01d degree"CRLF, ST.wYear, ST.wMonth, ST.wDay, ST.wHour, ST.wMinute, ST.wSecond, RV3032_GetTemp());
    }




//-----------------------------------------------------------------------------
//      10분동안 화재 재인식을 막는 화재 발생 시각을 지움
//-----------------------------------------------------------------------------
LOCAL(VOID) ClearFireDetectTime(VOID) {ZeroMem(FireDetectTime, sizeof(FireDetectTime));}




//-----------------------------------------------------------------------------
//      나의 시간 배정 번호(TAN)을 리턴 (400ms단위 값 리턴)
//      (1분당 RF 사용시간 배정표 참조)
//-----------------------------------------------------------------------------
#define ISIN_FIRST  0       //첫번째 보내는 패킷
#define ISIN_RETRY  1       //ACK를 못받아 재시도하는 패킷
LOCAL(int) GetMyTAN(VOID)
    {
    int TAN;

    TAN=SynchronizedVector+CommTestReplyOrder;
    if (FireState==0) TAN+=WAKEUPINTERVAL;      //+5는 타임테이블III 참조 (0~4는 Wakeup기간)
    return TAN;
    }




//-----------------------------------------------------------------------------
//      나의 모둠 ACK 보낼 시간의 옵션을 리턴함
//-----------------------------------------------------------------------------
LOCAL(int) GetCollAckSendTime(VOID)
    {
    return SynchronizedVector/(DETECTORQTY+WAKEUPINTERVAL)*ACKTIMEUNIT + DETECTORQTY*RESPONSETIMEUNIT;
    }




//-----------------------------------------------------------------------------
//      ACK 보내는 배정시간을 리턴함 (1분 안에서 ms시간을 리턴)
//-----------------------------------------------------------------------------
LOCAL(int) GetMyAckTime(int CurrTime_ms)
    {
    return CurrTime_ms/REPEATEROCCUPYTIME*REPEATEROCCUPYTIME + GetCollAckSendTime();
    }




//-----------------------------------------------------------------------------
//      모둠 ACK가 올 시간이 지나갔는지 알려줌
//-----------------------------------------------------------------------------
LOCAL(BOOL) IsPassedAckTime(VOID)
    {
    UINT ms;
    SYSTEMTIME ST;

    GetLocalTime(&ST);
    ms=ST.wSecond*1000+ST.wMilliseconds;
    return ms>=GetMyAckTime(ms)+ACKTIMEUNIT;
    }




//-----------------------------------------------------------------------------
//      정시에 보낼 패킷(ITP)을 지금 보낼 때인지 알려줌 (LBT로직에서 호출)
//      NeedReplyStateCnt==0 인 경우에는 ACK를 못받아 재시도하는 패킷임
//      (1분당 RF 사용시간 배정표 참조)
//-----------------------------------------------------------------------------
BOOL WINAPI IsSendTimeITP(VOID)
    {
    BOOL Rslt;
    UINT TAN, Remain, ms, Start, End;
    SYSTEMTIME ST;

    GetLocalTime(&ST);
    ms=ST.wSecond*1000+ST.wMilliseconds;
    if (FireState==0)
        {
        TAN=UDivMod(ms, RESPONSETIMEUNIT, &Remain);
        Rslt=GetMyTAN()==TAN && Remain<RESPONSETOLERLANCE+RESPONSETOLERLANCE;
        //if (Rslt==FALSE && PackTotalSecond(&ST)-CommTestStartTime>=DETECTORQTY*RESPONSETIMEUNIT/1000) Rslt++;
        }
    else{
        Start=SynchronizedVector*RESPONSETIMEUNIT;
        End=Start+REPEATEROCCUPYTIME;
        if (ms>=Start && ms<End)        //원래 보내는 시각
            {
            TAN=UDivMod(ms, RESPONSETIMEUNIT, &Remain);
            Rslt=GetMyTAN()==TAN && Remain<RESPONSETOLERLANCE;
            }
        else{                           //재시도 보내는 시각
            Start=ms/REPEATEROCCUPYTIME*REPEATEROCCUPYTIME + CommTestReplyOrder*RESPONSETIMEUNIT + HALFRESPONSETIMEUNIT;
            End=Start+RESPONSETOLERLANCE;
            Rslt=ms>=Start && ms<End;
            }
        }
    return Rslt;
    }




//-----------------------------------------------------------------------------
//      재전송화재나 통신점검의 응답을 제외한 패킷이 전송을 피해야 하는 기간
//
//      (1) 수신부로 부터 재전송화재의 ACK를 받을 시간임을 알려줌
//         이 기간일 때 ACK를 받기 위해 송출을 피하기 위함
//
//      (2) 지금이 중계기의 웨이컵신호를 받는 기간인지 체크함 (이 때는 송신을 유보)
//
//      TRUE를 리턴하면 전송을 유보함
//-----------------------------------------------------------------------------
BOOL WINAPI IsRecvCollectAckTime(VOID)
    {
    int  Rslt=FALSE;
    UINT Remain, ms, Start, End;
    SYSTEMTIME ST;

    GetLocalTime(&ST);
    ms=ST.wSecond*1000+ST.wMilliseconds;
    UDivMod(ms, RESPONSETIMEUNIT, &Remain);                 //TAN:0~149
    Start=ms/REPEATEROCCUPYTIME*REPEATEROCCUPYTIME;
    if (FireState)                                          //1초간Rf트래픽(기획안2).png 참조
        {
        Start+=DETECTORQTY*RESPONSETIMEUNIT;
        End=Start+REPEATERQTY*ACKTIMEUNIT;
        if (ms<Start) Rslt=Remain<HALFRESPONSETIMEUNIT;
        else if (ms<End) Rslt++;                            //else는 Ack보내는 기간임
        }
    else{                                                   //1초간Rf트래픽(기획안3).png 참조
        End=Start+WAKEUPINTERVAL*RESPONSETIMEUNIT;
        if (ms<End) Rslt++;                                 //통신점검용 Wakeup을 발송할 수 있는 기간, 20개씩 등록된 KFI인증용 세트일 경우에 한함
        else if (Remain<HALFRESPONSETIMEUNIT) Rslt++;       //통신점검의 응답 전송기간
        }
    return Rslt;
    }




//-----------------------------------------------------------------------------
//      키 및 센서 처리
//-----------------------------------------------------------------------------
LOCAL(VOID) KeyProc(VOID)
    {
    int I, Key, PT;
    JTIME CurrTime;

    if ((Key=GetKey())>0)   //RFC_IsSendIdle()... 2019-04-30 화재신호를 긴급으로 발송하기 위함
        {
        Printf("SENSOR: Key=%d"CRLF, Key);

        if ((Key=RemapKey(Key))==255) goto ProcExit;
        CurrTime=GetTodayToSecond();
        switch (Key)
            {
            case KC_BootTmr:
                SystemReset("User Booting"CRLF);
                break;

            case KC_SendStatus:     //정기적으로 아날로그 값들을 전송함
                break;

            case KC_TestUp:
                if (UseHighPrecisionRtc) RV3032_PrintTimeTemp();

                if (TestFireCnt) {TestFireCnt=0; break;}

                //if (NoReplyAlarm) {NoReplyAlarm=0; break;}
                if (RFC_IsSettingMode()!=0)
                    {
                    RFC_ExitSetupMode();
                    PlaySound(SolMiDo);
                    break;
                    }
                //TIM14->EGR|=2;        //소프트웨어에서 Int발생

                //InitRfModInGSensor();

                if (FireState) {PostExtinguishFirePacket(ON); break;}
                PostSwitchPacket(SynchronizeTimeFg);
                break;

            case KC_TestLong1:
                //PostRetryFirePacket(PT_SMOKE, 0, 0); break;
                CallMonFunc(COM_DEBUG, Mon_AutomanSpiRFM, "RFM ALL");
                //CallMonFunc(COM_DEBUG, Mon_AutomanSpiRFM, "RFM");
                ClearFireDetectTime();
                PlaySound(DoMiSol);
                RFC_EnterSetupMode();
                break;

            case KC_TestDblClk:
                if (FireState!=0) break;
                if ((PT=GetTestKeyPktType())==0) break;
                goto FireDetect;

            case KC_FlameDetect:  PT=PT_FLAME; TestFireCnt=0; goto FireDetect;  //불꽃감지
            case KC_FlameRelease: PT=PT_FLAME; goto FireRelease;                //불꽃감지 해제

            case KC_FireDetect:   PT=PT_FIRE; TestFireCnt=0; goto FireDetect;   //화재감지
            case KC_FireRelease:  PT=PT_FIRE; goto FireRelease;                 //화재감지 해제

            case KC_HeatDetect:   PT=PT_HEAT;  TestFireCnt=0; goto FireDetect;  //과열 감지
            case KC_HeatRelease:  PT=PT_HEAT;  goto FireRelease;                //과열 감지 해제

            case KC_SmokeDetect:  PT=PT_SMOKE; TestFireCnt=0; //goto FireDetect;//연기감지
                //화재감지 처리
                FireDetect:
                I=GetIndexFromPT(PT);
                if (TestFireCnt==0 && VolcanoBoardID!=VKBOARD_KOR)
                    {
                    if (FireDetectTime[I]!=0 && CurrTime-FireDetectTime[I]<60*10) break;
                    FireDetectTime[I]=CurrTime;
                    }
                SetFireState(GetFireStateFromIndex(I));
                PostFireDetectPacket(PT);

                FirstFireTimeStart=GetTickCount();
                FirstFireTimeIntvl=FirstFireTimeStart%1500+1500;
                RfTrafficCnt=0;
                break;

            case KC_SmokeRelease: PT=PT_SMOKE;  //goto FireRelease;             //연기감지해제
                FireRelease:
                I=GetIndexFromPT(PT);
                if (FireDetectTime[I]!=0 && CurrTime-FireDetectTime[I]<=3) FireDetectTime[I]=0;  //짧은 화재신호는 속보기에서 무시하기 때문에 이후 제대로 된 화재신호를 받을 것에 대비하여 10분 무시기능을 꺼야함
                ClearFireState(GetFireStateFromIndex(I));
                PostFirePacket(PT, OFF);
                break;

            case KC_GasDetect:          //센서포트 재정의하여 사용
            case KC_GasRelease:
                PostGasPacket(Key==KC_GasDetect);
                break;

            case KC_ElectricLeakDetect: //누전, 센서포트 재정의하여 사용
            case KC_ElectricLeakRelease:
                PostElectricLeakPacket(Key==KC_ElectricLeakDetect);
                break;

            case KC_SaveConfig:         //FTDI 무선으로 설정시 저장
                SaveConfigData();
                FlushIniFile();

                PlaySound(KeyClickSnd);
                break;

            #if 0
            case KC_TestFunc:
                if (TestChangeReg(900000)!=0) Printf("Task Register Changed"CRLF);
                else Printf("Task Register No Changed"CRLF);
                //break;
            #endif
            }
        }
    ProcExit:;
    }



//-----------------------------------------------------------------------------
//      8초마다 슬립에서 깨어나 호출됨
//      일정 시간 경과시마다 할일들을 처리함
//-----------------------------------------------------------------------------
LOCAL(VOID) DoIntervalProc(JTIME CurrTime)
    {
    //CurrTime=GetTodayToSecond();              //RV3032는 시간을 억세스하는데 3.4ms걸림
    CheckBatteryLevel();

    if (FireState==0 && CommTestStartTime==0 && IsSendTimeRtc(CurrTime))
        {
        #if 0                                   //IsSendTimeRtc()안에서 4시간마다 RFM을 리셋시킴
        if (AUTOMANSPI_RecoverChip(TRUE))       //24ms소요
            {
            #if RECOVERRFCHIPALARM
            NotifyReinitRFM();
            #endif
            LOG_Add(LOG_RFMRECOVERY, 0, 0, 0);
            }
        #endif
        PostBattPacket(BatteryStatus, SynchronizeTimeFg);
        }
    }




//-----------------------------------------------------------------------------
//      감지중인지 알려줌
//-----------------------------------------------------------------------------
LOCAL(BOOL) IsDetectingSensor(VOID)
    {
    return  (SmokeDetectCnt>0 && SmokeDetectCnt<SMOKEDETECTQTY) ||
            (HeatDetectCnt>0 && HeatDetectCnt<HeatDetectQty) ||
            (FireDetectCnt>0 && FireDetectCnt<SMOKEDETECTQTY) ||
            OftenCheckFg!=0;
    }




//-----------------------------------------------------------------------------
//      얼마 후에 깨어나야 할지를 ms단위로 리턴함 (500ms~8000ms 사이의 값)
//      8초 단위로 깨어나서 화재 감지를 해야 함
//-----------------------------------------------------------------------------
LOCAL(int) GetSleepingTime(CONST SYSTEMTIME *ST)
    {
    int SleepTime;

    SleepTime=CHECKFIREINTERVAL-(ST->wMinute*60+ST->wSecond)%CHECKFIREINTERVAL;
    if (IsDetectingSensor()) SleepTime=1;
    if ((SleepTime=SleepTime*1000-ST->wMilliseconds)<500) SleepTime=CHECKFIREINTERVAL*1000-ST->wMilliseconds;
    return SleepTime;
    }




//-----------------------------------------------------------------------------
//      이번 Sleep 기간 안에 Sniff 타임이 포함되어 있는지 알려줌
//      Sniff를 해야하는 Sleep기간이면 TRUE리턴
//
//      WakeupStart, SleepTime: ms
//-----------------------------------------------------------------------------
LOCAL(BOOL) IsInSniffTime(int WakeupStart, int SleepTime, int *lpAdjSleepTime)
    {
    int I, Rslt=FALSE, WakeupEnd, SniffStart, SniffEnd;     //, F=SleepTime;

    SniffStart=SynchronizedVector*RESPONSETIMEUNIT;
    SniffEnd=SniffStart+SNIFFPERIODII;
    WakeupEnd=WakeupStart+SleepTime;

    for (I=0; I<2; I++)
        {
        if (SniffEnd>WakeupStart && SniffStart<WakeupEnd)   //슬립기간과 스니프 시간이 만나는 경우
            {
            if (SniffStart<=WakeupStart)                    //0.4초단위로 딱 맞춰들어와도 MCU처리시간 때문 Sniff시간을 조금 지나쳤을 수가 있음
                {
                SleepTime=SniffEnd-WakeupStart;
                Rslt++;
                }
            else SleepTime=SniffStart-WakeupStart;
            goto ProcExit;
            }

        //Wakeup:56~04초 이고 Sniff:00~02초인경우
        SniffStart+=60000;
        SniffEnd+=60000;
        }

    ProcExit:
    *lpAdjSleepTime=SleepTime;
    //Printf("WakeupStart=%d, SleepTime=%d, AdjSleepTime=%d, Rslt=%d"CRLF, WakeupStart, F, SleepTime, Rslt);
    return Rslt;
    }




//-----------------------------------------------------------------------------
//      시간 동기화 정확성을 검사하기 위해 찍어봄
//-----------------------------------------------------------------------------
LOCAL(VOID) PrintTime(LPCSTR Title)
    {
    SYSTEMTIME ST;
    CHAR TimeStr[40];

    GetLocalTime(&ST);
    //Printf("T=%u.%03d (%d)"CRLF, PackSecondExceptDate(&ST) % SNIFFCYCLE, ST.wMilliseconds, ST.wSecond);
    if (SysDebugFg) MakeYMDHMS(TimeStr, &ST); else wsprintf(TimeStr, Psnt1d, ST.wSecond);
    Printf("%s T=%s.%03d"CRLF, Title, TimeStr, ST.wMilliseconds);
    }




//-----------------------------------------------------------------------------
//      STOP 모드로 들어감 (화재센서 체크를 위해 깨어나면 TRUE리턴)
//      RTC Alarm을 동작시키면 이걸 안하면 총 2.4uA소비함, 이걸하면 3.9uA
//-----------------------------------------------------------------------------
LOCAL(BOOL) EnterStopMode(VOID)
    {
    int Rslt=FALSE, SleepTime, RtcTimerEn=0, RV3032TmrEn=0, NoFireChk=0, SleepMode, AdjSleepTime;
    SYSTEMTIME ST;

    Printf(SysDebugFg ? "STOPMode..."CRLF:"!"CRLF);

    GPIO_DeInit(PO_BUZZER);             //72.2uA Down
    if (CommTestStartTime==0) PortOut(PO_GREEN_LED, OFF);
    PortOut(PO_ORANGE_LED, OFF);
    AdcOnOff(OFF);
    DacOnOff(OFF);

    //__HAL_RCC_???_CLK_DISABLE();      //저전력에 아무 효과없음

    GetLocalTime(&ST);
    SleepTime=GetSleepingTime(&ST);

    SleepMode=AUTOMANSPI_IsInSleep();
    if (UseHighPrecisionRtc==0)
        {
        if (SleepMode==HASBEENAWAKEUP)
            {
            #if KFI_INSPECTION
            AUTOMANSPI_GoSleepForWOR(3712);
            #else
            AUTOMANSPI_GoSleep();
            #endif
            }
        if (SleepTime>1000) RTC_SetAlarmAfterSecond(SleepTime/1000, TRUE, RTC_ALARM_A); //이처리가 1.5uA를 증가시킴
        else {RTC_SetWakeupTimer(SleepTime); RtcTimerEn++;}                             //RTC 알람이 1초미만의 오차가 있기 때문
        }
    else{
        if (IsInSniffTime(ST.wSecond*1000+ST.wMilliseconds, SleepTime, &AdjSleepTime))
            {
            if (SleepMode!=INSLEEPFORWOR) AUTOMANSPI_GoSleepForWOR(500);        //0.5초마다 Sniff해라
            }
        else{
            if (SleepMode!=INSLEEP) AUTOMANSPI_GoSleep();
            }
        if (SleepTime!=AdjSleepTime) NoFireChk=1;                               //Sniff끌라고 깨어날 것임
        SleepTime=AdjSleepTime;

        //Printf("SleepTime=%u"CRLF, SleepTime);
        RV3032_StartTimer(GetMax(SleepTime*64/1000, 3));                        //0이 설정되면 영원히 깨어나지 못함
        RV3032TmrEn++;
        }

    if (FireState==0 && LowBattLedOnTime!=0) FireLedOnOff(OFF);
    LowBattLedOnTime=0;
    UART_WaitAllSendIT(COM_DEBUG, 10);  //13자("STOPMode..."CRLF) * (1/115200*10=86us) = 1.13ms

    WakeupSource=0;
    PWR_EnterSTOPMode(PWR_LOWPOWERREGULATOR_ON, PWR_STOPENTRY_WFI);     //스펙은 1uA
    SystemClock_HSI();

    if (RtcTimerEn) RTC_SetWakeupTimer(0);                              //Stop을 안하면 계속 1초마다 깨어남
    if (RV3032TmrEn) RV3032_StartTimer(0);
    if (FireState==0 && BatteryStatus==BATSTAT_LOW) FireLedFlashOnce(); //CPU가 길게 살아있는 경우 LowBatt가 오래 켜져서 화재로 오인되기 때문에 꺼야함

    InitPort(PO_BUZZER, GPIO_MODE_AF_PP, GPIO_PULLUP, GPIO_AF2_LPTIM1);

    AdcOnOff(ON);
    DacOnOff(ON);

    if (SysDebugFg) PrintTime("Wakeup");
    Printf("WakeupSource=%s"CRLF, GetToken("None\0Key\0Fire\0RfRecv\0RfCarr\0FireChk\0Pir", WakeupSource));

    if (WakeupSource!=WKUPSRC_SMOKECHK) AUTOMANSPI_WakeupFromSleep();
    //if (SysDebugFg) Printf("CpuTemp=%d degree"CRLF, GetCpuTemp());
    switch (WakeupSource)
        {
        case WKUPSRC_RFRECV:
        case WKUPSRC_KEY:
            KeyPushed:
            PowerOffWaitTime=50;        //0.5초동안 기다리면서 RF 송수신 처리를 함
            break;

        case WKUPSRC_FIRE:              //콜포인트가 아닐 때는 TESTKEYII 임
            if (DetectorType!=CALLPOINT_DETECTOR) goto KeyPushed;
            Rslt++;
            break;

        case WKUPSRC_RFCARR:
            PowerOffWaitTime=450;       //캐리어 4초 후에 데이터를 보내므로 4.5초를 기다려야 함
            break;

        case WKUPSRC_SMOKECHK:
            if (NoFireChk==0) Rslt++;   //Sniff를 끄기 위해 깨어난 경우는 화재감지 Interval이 아니므로 화재감지를 못하게 함
            GoSleepMode();
            break;

        default: //WKUPSRC_NONE
            __HAL_RCC_DBGMCU_CLK_ENABLE();
            DBGMCU->CR=0;               //ST-Link Downlaod 툴로 다운로드하면 전원을 끄지 전까지 DBGMCU->CR=7로 설정되어 Stop모드에 들어가도 MCU에 클럭이 인가되고 그로 인해 SysTick인터럽트가 발생하고 바로 깨어나는 문제가 생김
            __HAL_RCC_DBGMCU_CLK_DISABLE();
            GoSleepMode();
        }

    DoIntervalProc(PackTotalSecond(&ST)+(SleepTime+ST.wMilliseconds)/1000);     //깨어나서 자정 이후 시간동기화를 위해 RTC를 억세스 해야 하는데 RV3032는 억세스시간이 3.4ms나 걸려서 이 값을 가지고 시간동기화 시간인지 확인하기 위해 저장함
    return Rslt;
    }




//-----------------------------------------------------------------------------
//      연기 감지 처리
//-----------------------------------------------------------------------------
LOCAL(VOID) SmokeDetectProc(int Vdd)
    {
    static BYTE NoDetectCnt;

    if (SmokeDetectRslt==CS_MEASUREING) SmokeDetectRslt=CheckSmokeSensor(Vdd, AdcCalValue);
    if (SmokeDetectRslt==CS_SENSORFAULT) PostFaultPacket(FAULT_SMOKESENSOR);

    if (SmokeDetectRslt==CS_MEASUREING) goto ProcExit;
    if (SmokeDetectRslt==CS_DETECTED)
        {
        if (SmokeDetectCnt<SMOKEDETECTQTY)
            {
            FireLedFlashOnce(); PlaySound(FireDetectingSnd);
            if (NoDetectCnt) {NoDetectCnt=0; SmokeDetectCnt--;} //연기가 잠시 소멸됐을 때 1의 값을 유지했으므로
            if (++SmokeDetectCnt==SMOKEDETECTQTY)
                {
                PutKey(KC_SmokeDetect);
                SetFireState(FIRESTATE_SMOKE);                  //KC_SmokeDetect처리 에서도 하지만 Q 처리 시간차에 의해 (A)에서 꺼버리는 버그가 생겼음
                }
            if (SysDebugFg) Printf("SENSOR: SmokeDetectCnt=%d"CRLF, SmokeDetectCnt);
            }
        if (SelfFireSirenFg!=0 && (FireState & FIRESTATE_SMOKE)!=0) PlayFireOccurSound();   //계속 연기가 있는 경우 8초마다 화재발생 알림 소리 송출
        }
    else{   //DetectRslt==CS_NOTDETECTED (감지안됨)
        if (FireState & FIRESTATE_SMOKE)                        //AutoReleaseFg==0이면 이 함수를 호출 조차 안함
            {
            PutKey(KC_SmokeRelease);                            //화재 시험버튼을 눌러 화재를 발생시킨 경우 연동알람을 시험해야 해서 해제신호를 지연시켜서 보냄
            ClearFireState(FIRESTATE_SMOKE);                    //KC_SmokeRelease처리 에서도 하지만 Q 처리 시간차에 의해 이곳에 다시 올 수 있음
            }
        else if (SmokeDetectCnt!=0)
            {
            SmokeDetectCnt=1;
            if (++NoDetectCnt<=2) goto ProcExit;                //연기가 감지되었다가 소멸된 경우 2초간은 1초마다 깨어나게 하기 위함
            }
        SmokeDetectCnt=NoDetectCnt=0;
        }
    ProcExit:;
    }




//-----------------------------------------------------------------------------
//      열 감지 처리
//-----------------------------------------------------------------------------
LOCAL(VOID) HeatDetectProc(int Vdd)
    {
    static BYTE NoDetectCnt;

    if (HeatDetectRslt==CS_MEASUREING) HeatDetectRslt=CheckTempSensor(Vdd, AdcCalValue, DetectorType, CHECKFIREINTERVAL);
    if (HeatDetectRslt==CS_SENSORFAULT) PostFaultPacket(FAULT_HEATSENSOR);

    if (HeatDetectRslt==CS_MEASUREING) goto ProcExit;
    if (HeatDetectRslt==CS_DETECTED)
        {
        if (HeatDetectCnt<HeatDetectQty)
            {
            FireLedFlashOnce(); PlaySound(FireDetectingSnd);
            if (NoDetectCnt) {NoDetectCnt=0; HeatDetectCnt--;}  //연기가 잠시 소멸됐을 때 1의 값을 유지했으므로
            if (++HeatDetectCnt==HeatDetectQty)
                {
                PutKey(KC_HeatDetect);
                SetFireState(FIRESTATE_HEAT);                   //KC_HeatDetect처리 에서도 하지만 Q 처리 시간차에 의해 (A)에서 꺼버리는 버그가 생겼음
                }
            if (SysDebugFg) Printf("SENSOR: HeatDetectCnt=%d"CRLF, HeatDetectCnt);
            }
        if (SelfFireSirenFg!=0 && (FireState & FIRESTATE_HEAT)!=0) PlayFireOccurSound();    //계속 연기가 있는 경우 8초마다 화재발생 알림 소리 송출
        }
    else{   //DetectRslt==CS_NOTDETECTED (감지안됨)
        if (FireState & FIRESTATE_HEAT)                         //AutoReleaseFg==0이면 이 함수를 호출 조차 안함
            {
            PutKey(KC_HeatRelease);                             //화재 시험버튼을 눌러 화재를 발생시킨 경우 연동알람을 시험해야 해서 해제신호를 지연시켜서 보냄
            ClearFireState(FIRESTATE_HEAT);                     //KC_HeatRelease처리 에서도 하지만 Q 처리 시간차에 의해 이곳에 다시 올 수 있음
            }
        else if (HeatDetectCnt!=0)
            {
            HeatDetectCnt=1;
            if (++NoDetectCnt<=2) goto ProcExit;                //연기가 감지되었다가 소멸된 경우 2초간은 1초마다 깨어나게 하기 위함
            }
        HeatDetectCnt=NoDetectCnt=0;
        }
    ProcExit:;
    }




//-----------------------------------------------------------------------------
//      열과 연기를 동시에 감지 처리
//-----------------------------------------------------------------------------
LOCAL(VOID) SmokeHeatDetectProc(int Vdd)
    {
    static BYTE NoDetectCnt;

    if (SmokeDetectRslt==CS_MEASUREING) SmokeDetectRslt=CheckSmokeSensor(Vdd, AdcCalValue);
    if (HeatDetectRslt==CS_MEASUREING)  HeatDetectRslt=CheckTempSensor(Vdd, AdcCalValue, DetectorType, CHECKFIREINTERVAL);

    if (SmokeDetectRslt==CS_SENSORFAULT) PostFaultPacket(FAULT_SMOKESENSOR);
    if (HeatDetectRslt==CS_SENSORFAULT) PostFaultPacket(FAULT_HEATSENSOR);

    if (SmokeDetectRslt==CS_MEASUREING || HeatDetectRslt==CS_MEASUREING) goto ProcExit;         //두개의 센서가 측정이 완료되기를 기다림
    if (SmokeDetectRslt==CS_DETECTED && HeatDetectRslt==CS_DETECTED)
        {
        if (FireDetectCnt<FIREDETECTQTY)
            {
            FireLedFlashOnce(); PlaySound(FireDetectingSnd);
            if (NoDetectCnt) {NoDetectCnt=0; FireDetectCnt--;}  //감지가 잠시 소멸됐을 때 1의 값을 유지했으므로
            if (++FireDetectCnt==FIREDETECTQTY)
                {
                PutKey(KC_FireDetect);
                SetFireState(FIRESTATE_FIRE);                   //KC_FireDetect처리 에서도 하지만 Q 처리 시간차에 의해 (A)에서 꺼버리는 버그가 생겼음
                }
            if (SysDebugFg) Printf("SENSOR: FireDetectCnt=%d"CRLF, FireDetectCnt);
            }
        if (SelfFireSirenFg!=0 && (FireState & FIRESTATE_FIRE)!=0) PlayFireOccurSound();    //계속 연기가 있는 경우 8초마다 화재발생 알림 소리 송출
        }
    else{
        if (FireState & FIRESTATE_FIRE)                         //AutoReleaseFg==0이면 이 함수를 호출 조차 안함
            {
            PutKey(KC_FireRelease);                             //화재 시험버튼을 눌러 화재를 발생시킨 경우 연동알람을 시험해야 해서 해제신호를 지연시켜서 보냄
            ClearFireState(FIRESTATE_FIRE);                     //KC_FireRelease처리 에서도 하지만 Q 처리 시간차에 의해 이곳에 다시 올 수 있음
            }
        else if (FireDetectCnt!=0)
            {
            FireDetectCnt=1;
            if (++NoDetectCnt<=2) goto ProcExit;                //연기가 감지되었다가 소멸된 경우 2초간은 1초마다 깨어나게 하기 위함
            }
        FireDetectCnt=NoDetectCnt=0;
        }
    ProcExit:;
    }




//-----------------------------------------------------------------------------
//      전송할 때가 다음 Sleep 기간 안에 포함되어 있는지 알려줌
//
// Mode==ISIN_FIRST  Sleep기간 안에 재전송 화재나 통신점검의 응답을 송출할 시간이 포함되었는지 알려줌
// Mode==ISIN_RETRY  재시도 패킷을 Q에 넣을 때인지 알려줌
//-----------------------------------------------------------------------------
LOCAL(BOOL) IsIncludeSendTermInNextWakeup(int Mode)
    {
    int CurrTime, SendTime, MySendTime1, MySendTime2, NextWakeupTime, SleepTime, Minute_ms, Second_ms;
    SYSTEMTIME ST;

    GetLocalTime(&ST);
    Second_ms=ST.wSecond*1000+ST.wMilliseconds;
    Minute_ms=ST.wMinute*60*1000;
    SleepTime=GetSleepingTime(&ST);
    IsInSniffTime(Second_ms, SleepTime, &SleepTime);

    CurrTime=Minute_ms+Second_ms;                               //분을 포함한 현재시간
    NextWakeupTime=CurrTime+SleepTime;                          //다음 깨어날 시간

    if (Mode==ISIN_FIRST)
        SendTime=GetMyTAN()*RESPONSETIMEUNIT;
    else
        SendTime=Second_ms/REPEATEROCCUPYTIME*REPEATEROCCUPYTIME + CommTestReplyOrder*RESPONSETIMEUNIT + HALFRESPONSETIMEUNIT;

    MySendTime1=Minute_ms + SendTime;
    MySendTime2=MySendTime1+60*1000;                            //MySendTime1='10:34:56'이고 NextWakeupTime='10:35:04' 인 경우 대비

    return (MySendTime1+RESPONSETOLERLANCE>CurrTime && MySendTime1<NextWakeupTime) ||
           (MySendTime2+RESPONSETOLERLANCE>CurrTime && MySendTime2<NextWakeupTime);
    }




//-----------------------------------------------------------------------------
//      현재 감지기가 감지한 화재 종류를 판단하여 재전송 화재패킷을 Q에 넣음
//-----------------------------------------------------------------------------
LOCAL(VOID) PostResendFirePkt(VOID)
    {
    int I;

    for (I=0; I<countof(FirePacketTypeList); I++)
        {
        if (FireState & GetFireStateFromIndex(I)) PostRetryFirePacket(FirePacketTypeList[I]);
        }
    }




//-----------------------------------------------------------------------------
//      1분마다 화재신호 재전송
//-----------------------------------------------------------------------------
LOCAL(VOID) ResendFirePacket(VOID)
    {
    if (FirstFireTimeStart!=0)                      //최초화재 집중전달모드 (2번 발송)
        {
        if (GetTickCount()-FirstFireTimeStart>=FirstFireTimeIntvl)
            {
            FirstFireTimeStart=0;
            if (RfTrafficCnt<3)
                {
                if (RFC_SendFireRightNow()==FALSE) FirstFireTimeStart=GetTickCount();
                }
            }
        }
    else{
        if (ITP_WaitAck!=0)
            {
            if (IsPassedAckTime())                  //모둠ACK 기간이 지나갔는지 체크
                {
                PrintTime("PassedAckTime");
                ITP_WaitAck=0;
                if (--ITP_SendCnt==0) RFM_Reset();  //AUTOMANSPI_RecoverChip(TRUE);
                }
            }
        else if (ITP_SendCnt==0)
            {
            if (IsIncludeSendTermInNextWakeup(ISIN_FIRST))
                {
                ITP_SendCnt=4;

                PostResendFire:
                PostResendFirePkt();
                ITP_WaitAck=1;
                PrintTime("PostResendFire");
                }
            }
        else{
            if (IsIncludeSendTermInNextWakeup(ISIN_RETRY)) goto PostResendFire;
            }
        }
    }




//-----------------------------------------------------------------------------
//          통신점검의 응답을 보내야 하는 시각에 보내도록 함
//-----------------------------------------------------------------------------
LOCAL(VOID) SendStatePacket(VOID)
    {
    if (ITP_SendCnt==0 &&
        IsIncludeSendTermInNextWakeup(ISIN_FIRST))                      //다음 깨어날 기간안에 보내야 하는 시간이 포함되어 있는지 ?
        {
        PostDiagCommAckPacket((BatteryStatus<<DIAGSTAT_BATT_BIT)|(FireState!=0));   //패킷을 Q에 넣으면 Sleep모드로 들어가지 않음
        PrintTime("PostState");
        ITP_SendCnt=1;
        }
    }




//-----------------------------------------------------------------------------
//          화재 감지 처리 (센서 측정이 끝나면 TRUE리턴)
//-----------------------------------------------------------------------------
LOCAL(BOOL) DetectFireProc(VOID)
    {
    int  Rslt=TRUE, Vdd;

    if (TestFireCnt!=0)
        {
        if (--TestFireCnt==0)                                           //화재 시험버튼을 눌러 화재를 발생시킨 경우 연동알람을 시험해야 해서 해제신호를 지연시켜서 보냄
            {
            if (VolcanoBoardID!=VKBOARD_KOR && FireState!=0)            //국내형은 수신기가 클리어함
                {
                if (FireState & FIRESTATE_SMOKE) PutKey(KC_SmokeRelease);
                if (FireState & FIRESTATE_HEAT)  PutKey(KC_HeatRelease);
                if (FireState & FIRESTATE_FLAME) PutKey(KC_FlameRelease);
                if (FireState & FIRESTATE_FIRE)  PutKey(KC_FireRelease);
                SetFireState(OFF);
                }
            }
        goto ProcExit;
        }

    if (AutoReleaseFg==0 && FireState!=0) goto ProcExit;
    if ((Vdd=MeasureVdd(AdcCalValue))<2400) goto ProcExit;              //Vdd가 2.4V미만: 오동작 가능성이 있어 화재감지동작을 하지않음

    switch (SetFireDetectPolicy(-1))
        {
        case SAF_FIRESMOKEONLY:
            SmokeDetectProc(Vdd);
            Rslt=SmokeDetectRslt!=CS_MEASUREING;
            break;

        case SAF_FIREHEATONLY:
            HeatDetectProc(Vdd);
            Rslt=HeatDetectRslt!=CS_MEASUREING;
            break;

        case SAF_FIREHEATORSMOKE:
            if (SmokeDetectRslt==CS_MEASUREING) SmokeDetectProc(Vdd);
            if (HeatDetectRslt==CS_MEASUREING)  HeatDetectProc(Vdd);
            goto ChkDone;

        case SAF_FIREHEATANDSMOKE:
            SmokeHeatDetectProc(Vdd);
            ChkDone:
            Rslt=SmokeDetectRslt!=CS_MEASUREING && HeatDetectRslt!=CS_MEASUREING;   //연기와 열 모두 측정이 끝난 경우 슬립모드에 진입하도록 함
            //break;
        }

    ProcExit:
    return Rslt;
    }




//-----------------------------------------------------------------------------
//      시작음 발생
//-----------------------------------------------------------------------------
LOCAL(VOID) StartSound(VOID)
    {
    CHAR Code[8];

    //PlaySound(SavedData.RepeaterMode ? RepeaterOnSnd:StartSnd3);    //RozetaSongSnd

    lstrcpy(Code, "SM");    //SMOKE_DETECTOR
    if (DetectorType==FIXED_HEAT_DETECTOR) lstrcpy(Code, "HC");
    else if (DetectorType==ROR_HEAT_DETECTOR) lstrcpy(Code, "HD");
    else if (DetectorType==SMOKE_HEAT_DETECTOR) lstrcpy(Code, "SH");
    else if (DetectorType==CALLPOINT_DETECTOR)  lstrcpy(Code, "CP");

    //if (VolcanoBoardID==VKBOARD_KOR) AddCha(Code, 'K'); else
    if (VolcanoBoardID==VKBOARD_UAE) AddCha(Code, 'U');
    else if (VolcanoBoardID==VKBOARD_THAI) AddCha(Code, 'T');
    else if (VolcanoBoardID==VKBOARD_UAEALONE) AddCha(Code, 'A');

    if (SavedData.RepeaterMode) AddCha(Code, 'R');
    MorseOut(Code);
    }




//-----------------------------------------------------------------------------
//      시간이 잘못된 경우 컴파일 시간을 설정함
//-----------------------------------------------------------------------------
LOCAL(VOID) SetupRTC(VOID)
    {
    int  I, IsRV3032=0;
    CHAR Buff[32];
    SYSTEMTIME ST;

    //내부 RTC 클럭 교정
    RTC_Calibration();      //1초정도 소요
    RTC_Calibration();      //1초정도 소요

    if (UseHighPrecisionRtc)
        {
        //RV3032는 전원을 켠 후 어느 정도 시간이 지나야 I2C가 동작함
        for (I=0; I<10; I++)
            {
            //Printf("SENSOR: Init RV3032 Retry=%d"CRLF, I);
            if ((IsRV3032=RV3032_EnableTimerInt(RV3032REG_TD64Hz))!=FALSE) break;
            }
        if (I<10) SetExtRtcFP(RV3032_GetTime, RV3032_SetTime);
        else      Printf("SENSOR: Init RV3032 Fail..."CRLF);
        }

    GetLocalTime(&ST);
    if (ST.wYear<2021)
        {
        GetAppCompileDate(&ST);
        SetLocalTime(&ST);
        ClearFireDetectTime();      //시간이 초기화 되면 화재센서 10분동안 꺼둔 기능을 다시 켠다
        }
    MakeYMDHMS(Buff, &ST);
    Printf("SENSOR: %s Rtc %s"CRLF, IsRV3032 ? "RV3032":"Internal", Buff);
    }




//-----------------------------------------------------------------------------
//      설정값들을 읽은 후 주요 RF파라메터 등을 표시
//-----------------------------------------------------------------------------
LOCAL(VOID) PrintBootMagII(VOID)
    {
    Printf("SENSOR: Group=%u, DevNo=%u, Channel=%d, Power=%d, Repeater=%d, BoardID=%d,%02X"CRLF,
            SavedData.GroupNo, SavedData.DeviceNo, SavedData.ChannelNo, SavedData.RFPower, SavedData.RepeaterMode, VolcanoBoardID, AdcBoardID>>8);

    if (DetectorType==SMOKE_HEAT_DETECTOR) Printf("SENSOR: DetectPolicy is %s"CRLF, GetToken("DetectDisable\0HeatOnly\0SmokeOnly\0HeatOrSmoke\0HeatAndSmoke\0", FireDetectPolicy));
    }



//-----------------------------------------------------------------------------
//      Main Task
//-----------------------------------------------------------------------------
static VOID CALLBACK MainTask(LPVOID lpData)
    {
    int I, Wakeuped=0, FireDetectingFg, FireDetTime;                            //FireDetectingFg: 1이면 센서 측정중임
    #if 0
    DWORD StartTick;

    StartTick=GetTickCount();
    Delay1ms(1000); Printf("Delay1ms=%d"CRLF, GetTickCount()-StartTick);        //루프딜레이 캘리브레이션
    #endif

    Printf("SENSOR: JOS Ver=%s"CRLF, lpData);
    InstalledSht=SHT_Init();
    PortOut(PO_SENSORPWR, OFF);

    LoadConfigData();
    LoadKeyRemapTable();

    PrintBootMagII();                                       //설정값들을 읽은 후 주요 RF파라메터 등을 표시

    HeatDetectQty=(DetectorType==ROR_HEAT_DETECTOR) ? RORHEATDETECTQTY:FIXEDHEATDETECTQTY;

    if (GetTodayToSecond()<10) ClearFireDetectTime();       //전원 껏다 켜면 날짜가 2000-1-1 0:0:0 부터 시작하므로 생산시 테스트했던 감지 시각을 무효화함

    if (SavedData.DeviceNo!=MASTERRECEIVER) EchoEnableFg=1; //디버거 터미널 응답형으로 전환

    if (__HAL_PWR_GET_FLAG(PWR_FLAG_SB))
        {
        __HAL_PWR_CLEAR_FLAG(PWR_FLAG_SB);                  //시스템이 Standby mode에서 재개된 경우
        Wakeuped=1;
        }
    else if (MeasureVdd(AdcCalValue)>=2100) StartSound();   //배터리가 소모되어 계속 재부팅되면서 부저에서 소리나는 현상을 막기 위함

    SetupRTC();

    //TIM6_InitTick(10);                                    //WatchDog
    RFC_Init(RFSendRecvCB);

    if (Wakeuped)
        {
        for (I=0; I<10; I++)
            {
            Printf("Wakeuped (RSSI=%d)"CRLF, RFM_ReadRSSI());
            HAL_Delay(100);
            }
        }

    SendSyncTime(30);                                       //전원을 켠 경우 30초후에 Alive를 보내도록함
    FireDetectingFg=1;                                      //연기의 기준값을 측정하도록 함
    FireDetTime=GetTickCount();

    //MonitorRoutinue("RFM MOUT 86400000");                 //CE인증을 위함

    #if REPEATTEST
    PutKey(KC_TestUp);
    #endif
    for (;;)
        {
        KeyProc();
        DelayedPortProc();
        MonitorProc();

        if (RFC_IsPowerOnReady()!=0 && RfmTaskTimer!=0)     //10mSec 주기로 실행
            {
            RfmTaskTimer=0;
            RFC_ManagerTask(NULL);
            }

        if (FireDetectingFg)
            {
            #if COMMRANGETEST
            PutKey(C_KeyTestDblClk); FireDetectingFg=0;
            #else
            if (DetectFireProc()!=FALSE) FireDetectingFg=0; //측정종료
            #endif
            }

        I=IsDetectingSensor() ? 1000:CHECKFIREINTERVAL*1000;
        if (GetTickCount()-FireDetTime>=I)                  //Run모드에 장시간 있을 때에도 화재감지를 위함
            {
            SmokeDetectRslt=HeatDetectRslt=CS_MEASUREING;
            FireDetectingFg=1;
            FireDetTime=GetTickCount();
            }

        if (AutoReleaseFg==0)
            {
            if (FireState) ResendFirePacket();              //이전엔 이로직이 DetectFireProc()안에 있었으나, 스니프 기간에 다른 감지기의 재전송 화재가 수신되어 계속 켜있는 경우에도 재전송화재를 보내게 하기 위함.
            else if (CommTestStartTime) SendStatePacket();
            }

        if (IsBusyKey()==FALSE &&                           //설정모드 들어가기 위해 오래누르는 동작 때문
            PowerOffWaitTime==NOW_SLEEP &&
            RFC_IsSettingMode()==FALSE &&
            RFC_IsSendIdle() &&
            AUTOMANSPI_IsExistRecvData()==FALSE &&
            IsSoundPlaying()==FALSE &&
            ITP_SendCnt==0 &&
            ITP_WaitAck==0 &&
            FireDetectingFg==0 &&
            NoSleepMode==0)
            {
            FireDetectingFg=EnterStopMode();                //깨어날 시간을 예약한 후 전원을 끔
            SmokeDetectRslt=HeatDetectRslt=CS_MEASUREING;
            FireDetTime=GetTickCount();
            }
        }
    }



//-----------------------------------------------------------------------------
//      RFM SPI-MISO 제어
//-----------------------------------------------------------------------------
VOID WINAPI RFMMISO_SetInput(VOID) {InitPort(PI_RFMMISO, GPIO_MODE_INPUT, GPIO_PULLUP, 0);}
VOID WINAPI RFMMISO_SetSPI(VOID)   {InitPortEx(PI_RFMMISO, GPIO_MODE_AF_PP, GPIO_PULLDOWN, GPIO_AF0_SPI2, GPIO_SPEED_FREQ_VERY_HIGH);}



LOCAL(VOID) Delay_us(int T)
    {
    volatile int I;
    for (I=0; I<T; I++);
    }



//-----------------------------------------------------------------------------
//      PCB ID를 읽음
//-----------------------------------------------------------------------------
LOCAL(VOID) ReadPCBID(VOID)
    {
    InitPortInput(PI_BOARD_ID1, GPIO_PULLUP);
    InitPortInput(PI_BOARD_ID2, GPIO_PULLUP);
    InitPortInput(PI_BOARD_ID3, GPIO_PULLUP);
    if (PortIn(PI_BOARD_ID1)) DetectorType=FIXED_HEAT_DETECTOR;
    if (PortIn(PI_BOARD_ID2)) VolcanoBoardID|=1;
    if (PortIn(PI_BOARD_ID3)) VolcanoBoardID|=2;
    InitPortOutputOD(PI_BOARD_ID1); //InitPortFloating() 이게 전류를 더 소비함
    InitPortOutputOD(PI_BOARD_ID2);
    InitPortOutputOD(PI_BOARD_ID3);

    if (VolcanoBoardID==VKBOARD_Ext)
        {
        InitADC(ADC_SAMPLETIME_39CYCLES_5);
        InitPort(PI_BOARD_ID3, GPIO_MODE_ANALOG, GPIO_NOPULL, 0);   //PA7
        PortOut(PI_BOARD_ID2, LOW);                                 //평소 전류를 소모하지 않게 하기 위해 전압분배 저항을 Low함
        Delay_us(1000);
        AdcBoardID=ADC_GetValueAverage(ADC_CHSELR_CHSEL7, 16);
        GPIO_DeInit(PI_BOARD_ID1);
        GPIO_DeInit(PI_BOARD_ID2);
        GPIO_DeInit(PI_BOARD_ID3);

        VolcanoBoardID=VKBOARD_KOR;
        switch (AdcBoardID>>8)
            {
            case 0x00: VolcanoBoardID=VKBOARD_UAEALONE; DetectorType=SMOKE_DETECTOR; break;
            case 0x01: VolcanoBoardID=VKBOARD_UAEALONE; DetectorType=FIXED_HEAT_DETECTOR; break;
            case 0x02: VolcanoBoardID=VKBOARD_PIR;      DetectorType=SMOKE_DETECTOR; break;
            case 0x03: VolcanoBoardID=VKBOARD_PIR;      DetectorType=FIXED_HEAT_DETECTOR; break;
            case 0x04: VolcanoBoardID=VKBOARD_ELECLEAK; break;

            case 0x05:  //5.23K                     //1M NTC사용 보드
                DetectorType=SMOKE_HEAT_DETECTOR;   //복합형
                UseHighPrecisionRtc=1;
                break;

            case 0x08:  //11K                       //1M NTC사용 보드
                UseHighPrecisionRtc=1;
                DetectorType=FIXED_HEAT_DETECTOR;   //정온식
                break;

            case 0x09:  //15K                       //1M NTC사용 보드
                UseHighPrecisionRtc=1;
                DetectorType=ROR_HEAT_DETECTOR;     //차동식
                break;

            case 0x06:  //6.8K  스타쉽
            case 0x0A:  //19.1K 볼케이노
                UseHighPrecisionRtc=1;
                DetectorType=SMOKE_DETECTOR;        //광전식 연기감지기
                break;

            case 0x0B:  //25.5K 과제용 열연기CO 복합형
                UseHighPrecisionRtc=1;
                UseLowResistNtc=1;                  //50K NTC사용 보드
                DetectorType=SMOKE_HEAT_CO_DETECTOR; //복합형 with CO
                break;

            case 0x07:  //9.1K 스타쉽
            case 0x0C:  //36K 볼케이노
                UseHighPrecisionRtc=1;
                UseLowResistNtc=1;                  //50K NTC사용 보드
                DetectorType=FIXED_HEAT_DETECTOR;   //정온식
                break;

            case 0x0D:  //54.9K
                UseHighPrecisionRtc=1;
                UseLowResistNtc=1;                  //50K NTC사용 보드
                DetectorType=SMOKE_HEAT_DETECTOR;   //복합형
                break;

            case 0x0E:  //97.6K 볼케이노
            case 0x0F:  //309K  스타쉽
                UseHighPrecisionRtc=1;
                UseLowResistNtc=1;                  //50K NTC사용 보드
                DetectorType=ROR_HEAT_DETECTOR;     //차동식
                //break;
            }
        }
    }




//-----------------------------------------------------------------------------
//      부팅 메세지를 표시함
//-----------------------------------------------------------------------------
LOCAL(VOID) PrintEqualLine(VOID) {Printf("SENSOR: ================================================================================="CRLF);}
LOCAL(VOID) PrintBootMag(VOID)
    {
    CHAR Buff[32];

    Printf(CRLF);
    PrintEqualLine();
    GetAppCompileDateStr(Buff);
    Printf("SENSOR: %s Detector RozetaTech for JW-Receiver (Build %s)" CRLF, GetToken("Smoke\0Fixed Heat\0Differential Heat\0Smoke+FixedHeat\0Flame\0CallPoint", DetectorType), Buff);
    PrintEqualLine();
    Printf("SENSOR: SystemCoreClock=%uHz"CRLF, SystemCoreClock);
    Printf("SENSOR: Analog Board ID=%03X"CRLF, AdcBoardID);
    }





//-----------------------------------------------------------------------------
//      Main
//-----------------------------------------------------------------------------
int main(VOID)
    {
    static BYTE AllocMemory[8092];

    JOS_ENTER_CRITICAL();

    HAL_Init();
    SystemClock_HSI();
    SET_BIT(PWR->CR, PWR_CR_ULP);   //HAL_PWREx_EnableUltraLowPower();
    SET_BIT(PWR->CR, PWR_CR_FWU);   //HAL_PWREx_EnableFastWakeUp();

    IniSavePtr=AllocFlash(INIFILESIZEMAX);
    if (MemAllocInit((UINT)AllocMemory, sizeof(AllocMemory))==FALSE) FatalError(FERR_MEMORY);   //메모리 할당 초기화
    InitDriver();                   //JOS - Lock()/Unlock() 오브젝트 생성

    ReadPCBID();

    #if FORCE_FIXED_HEAT_DETECTOR
    if (DetectorType==ROR_HEAT_DETECTOR) DetectorType=FIXED_HEAT_DETECTOR;      //정온식
    #endif

    InitPortOutputPP(PO_SMOKE_SEN_IR);
    InitPortOutputPP(PO_SEN_LDO_EN);

    InitPortOutputPP(PO_SENSORPWR);  PortOut(PO_SENSORPWR, ON);

    InitPortOutputPP(PO_RED_LED);    PortOut(PO_RED_LED, ON);           //생산시 LED점검을 위해 잠시 모두 켬
    InitPortOutputPP(PO_GREEN_LED);  PortOut(PO_GREEN_LED, ON);         //  "
    InitPortOutputPP(PO_ORANGE_LED); PortOut(PO_ORANGE_LED, ON);        //  "

    InitPort(PO_I2CSCL,   GPIO_MODE_AF_OD, GPIO_PULLUP, GPIO_AF4_I2C1); //GPIO_NOPULL 로 설정하면 아무것도 연결 안되어 있는 경우 수십uA~수100uA 왔다 갔다하면서 전류를 소비함
    InitPort(PO_I2CSDA,   GPIO_MODE_AF_OD, GPIO_PULLUP, GPIO_AF4_I2C1); //  "

    InitPort(PO_LPCOM1TX, GPIO_MODE_AF_PP, GPIO_NOPULL, GPIO_AF7_LPUART1);
    InitPort(PI_LPCOM1RX, GPIO_MODE_AF_PP, GPIO_PULLUP, GPIO_AF7_LPUART1);

    if (VolcanoBoardID<VKBOARD_UAEALONE)
        {
        if (DetectorType!=SMOKE_HEAT_DETECTOR)
            InitPortInput(PI_BREAKDOWN, GPIO_PULLUP);                   //PA2, 정전감지, 0:외부전원, 1:배터리
        }

    InitPort(PI_RFWKUP1,   GPIO_MODE_IT_FALLING, GPIO_NOPULL, 0);
    InitPort(PI_TESTKEYII, GPIO_MODE_IT_FALLING, GPIO_PULLUP, 0);
    InitPort(PI_EXTWKUP,   GPIO_MODE_IT_FALLING, GPIO_PULLUP, 0);
    if (UseHighPrecisionRtc) InitPort(PI_EXRTC_INT,  GPIO_MODE_IT_FALLING, GPIO_PULLUP, 0); //PA4

    InitPortOutputPP(PO_SMOKE_SEN_PW);

    InitPortOutputPP(PO_RFMNSS); PortOut(PO_RFMNSS, HIGH);
    InitPortOutputPP(PO_RFMRESET);                  //리셋 시간을 조금이라도 늘리기 위해 여기서 부터 'L'로
    InitPortEx(PO_RFMSCK,     GPIO_MODE_AF_PP, GPIO_NOPULL, GPIO_AF0_SPI2, GPIO_SPEED_FREQ_VERY_HIGH);
    //InitPortEx(PI_RFMMISO,  GPIO_MODE_AF_PP, GPIO_PULLDOWN, GPIO_AF0_SPI2, GPIO_SPEED_FREQ_VERY_HIGH);    //CC112xWaitMisoLow() 에서 핀모드를 재설정함
    InitPortEx(PO_RFMMOSI,    GPIO_MODE_AF_PP, GPIO_NOPULL, GPIO_AF0_SPI2, GPIO_SPEED_FREQ_VERY_HIGH);

    InitPort(PO_BUZZER, GPIO_MODE_AF_PP, GPIO_PULLUP, GPIO_AF2_LPTIM1);

    NVIC_SetPriority(EXTI4_15_IRQn, 0);     //RF수신인터럽트, PI_RFWKUP1(PA10), PI_RFWKUP2(PB5), PI_TESTKEYII(PA9)
    NVIC_EnableIRQ(EXTI4_15_IRQn);

    InitUart(LPCOM1, 115200, UART_PARITY_NONE, UART_WORDLENGTH_8B, UART_STOPBITS_1, UART_INTCLOCK);

    InitI2C(IIC1, MAKEI2CTIMING(1,3,2,3,9));            //(PRESC, SCLDEL, SDADEL, SCLH, SCLL), RTC, 온도센서가 사용
                                                        //MAKEI2CTIMING(1,10,1,62,86) => 50KHz
                                                        //MAKEI2CTIMING(1,3,2,3,9)    => 400KHz
    #if CORECLOCK==8000000
    SPI_InitMaster(SPI_CH2, SPI_BAUDRATEPRESCALER_2);   //8M/2=4MHz
    #else
    SPI_InitMaster(SPI_CH2, SPI_BAUDRATEPRESCALER_8);   //32M/8=4MHz
    #endif
    InitRTC(RTCCLKSRC_INTERNAL);

    InitBuzzerTimer();
    AdcOnOff(ON);
    DacOnOff(ON);

    JOS_EXIT_CRITICAL();

    PrintBootMag();
    MainTask("v100");
    }



//-----------------------------------------------------------------------------
//      키 리맵 (키코드 변경)
//-----------------------------------------------------------------------------
int WINAPI Mon_RemapKey(int PortNo, LPCSTR MonCmd, LPCSTR lpArg, LPCSTR CmdLine)
    {
    int I, Org, Chg, Rslt=MONRSLT_SYNTAXERR;
    LPCSTR lp;

    if (lpArg[0]=='?')
        {
        Printf("REMAP OrgKeyCode RemapKeyCode ... Remapping key code"CRLF);
        Rslt=MONRSLT_EXIT;
        goto ProcExit;
        }

    Org=0;  lp=ScanInt(lpArg, &Org); if (Org==0) goto ProcExit;
    Chg=-1; lp=ScanInt(lp, &Chg); if (Chg<0) goto ProcExit;
    if (Chg!=0) I=AddRemapKey(Org, Chg, lpArg);
    else        I=DelRemapKey(Org);
    if (I!=FALSE)
        {
        Rslt=MONRSLT_OK;
        }
    else{
        Printf(Chg!=0 ? "Key Remap Table Full!"CRLF:"Not Found Remaped Key"CRLF);
        Rslt=MONRSLT_EXIT;
        }
    ProcExit:
    return Rslt;
    }



//-----------------------------------------------------------------------------
//      파라메터 설정
//-----------------------------------------------------------------------------
int WINAPI Mon_Set(int PortNo, LPCSTR MonCmd, LPCSTR lpArg, LPCSTR CmdLine)
    {
    int  Rslt=MONRSLT_SYNTAXERR;
    CHAR VarName[16];

    if (lpArg[0]=='?')
        {
        Printf("SET INSLEEP ON/OFF"CRLF);       //디버거 포트로 설정을 할 때 슬립모드 못들어가게하기 위해 만듬
        Rslt=MONRSLT_EXIT;
        goto ProcExit;
        }

    lpArg=(LPSTR)ScanWord(lpArg, VarName, sizeof(VarName));
    if (lstrcmpi(VarName, "INSLEEP")==0)
        {
        if (lstrcmpi(lpArg, "ON")==0) NoSleepMode=0;
        else if (lstrcmpi(lpArg, "OFF")==0) NoSleepMode=1;
        else goto ProcExit;
        Rslt=MONRSLT_OK;
        }

    ProcExit:
    return Rslt;
    }



//-----------------------------------------------------------------------------
//      중요변수 내용 표시 ("PRINT" 명령에 대한 처리)
//-----------------------------------------------------------------------------
int WINAPI Mon_DispImpVar(int PortNo, LPCSTR MonCmd, LPCSTR lpArg, LPCSTR CmdLine)
    {
    int  Vdd, BkupSysDebugFg, Rslt=MONRSLT_SYNTAXERR;

    if (lpArg[0]=='?')
        {
        Printf("Print ... Print Import Varient"CRLF);
        Rslt=MONRSLT_EXIT;
        goto ProcExit;
        }

    PrintBootMag();
    PrintBootMagII();
    Printf(CRLF);

    Printf("SyncVct=%d"CRLF, SynchronizedVector);
    Printf("DevRegOrder=%d"CRLF, CommTestReplyOrder);
    Printf("CpuTemp=%d degree"CRLF, GetCpuTemp());
    Printf("SensorAlarmFg=%Xh"CRLF, SensorAlarmFg);
    Printf("Flash IniData=%Xh"CRLF, GetIniSavePtr());
    #if ANIMALTEMPCONTROL || SENDIRTEMPIMAGE
    PrintAnimalTempVar();
    #endif
    Printf(CRLF);

    RFC_PrintVar();

    //임계값을 보기 위함임
    BkupSysDebugFg=SysDebugFg;
    SysDebugFg=1;
    Vdd=MeasureVdd(AdcCalValue);
    switch (SetFireDetectPolicy(-1))
        {
        case SAF_FIRESMOKEONLY:
            CheckSmokeSensor(Vdd, AdcCalValue);
            break;

        case SAF_FIREHEATONLY:
            CheckTempSensor(Vdd, AdcCalValue, DetectorType, CHECKFIREINTERVAL);
            break;

        case SAF_FIREHEATORSMOKE:
        case SAF_FIREHEATANDSMOKE:
            CheckSmokeSensor(Vdd, AdcCalValue);
            CheckTempSensor(Vdd, AdcCalValue, DetectorType, CHECKFIREINTERVAL);
        }
    SysDebugFg=BkupSysDebugFg;

    //DispMemBlock();
    Rslt=MONRSLT_OK;

    ProcExit:
    return Rslt;
    }



//-----------------------------------------------------------------------------
//      모니터 명령 테스트 함수
//-----------------------------------------------------------------------------
int WINAPI Mon_FuncTest(int PortNo, LPCSTR MonCmd, LPCSTR lpArg, LPCSTR CmdLine)
    {
    int Rslt=MONRSLT_SYNTAXERR;

    if (lpArg[0]=='?')
        {
        Printf("TEST ... Programmer Test Function Call"CRLF);
        Rslt=MONRSLT_EXIT;
        goto ProcExit;
        }

    if (AUTOMANSPI_RecoverChip(TRUE)) NotifyReinitRFM();

    Rslt=MONRSLT_OK;

    ProcExit:
    return Rslt;
    }




