﻿///////////////////////////////////////////////////////////////////////////////
//                      LORA MESH 로직
//
// 2017/02/20
// 2017-11-06 SEND_DEVICE_QTY 를 20->100 으로 수정
///////////////////////////////////////////////////////////////////////////////

#include "JLIB.H"
#include "DRIVER.H"
#include "JOS.H"
#include "MAIN.H"
#include "RFModule.h"
#include "SPIAutoMan.h"
#include "RFCommunication.h"
#include "KEY.H"
#include "MONITOR.H"
#include "INIDATA.H"
#include "PlayBuzzer.h"
#include "LOG.H"


#define ROLASENDPACKETQTY       20
#define SEND_DEVICE_QTY         100     //센서의 총 갯수
#define SEND_HISTORY_QTY        80      //10->80 전송 히스토리 갯수
#define REPLY_CHECK_TIME        10*JOS_TICKS_PER_SEC    //5->10 응답 체크 타임아웃 (10초)
#define RETRY_ACK_TIME          5       //송신기가 ACK가 없을 때 재전송을 하는데 그에 대응한 ACK를 보낼 시간, 중계기에 의해 같은 패킷이 여러번 오는 것을 거르기 위함
#define ADDROUTERSENDTIME       2*JOS_TICKS_PER_SEC     //라우터 추가 Data 전송시간
#define PACKET_HISTORY_TIME     180     //전송 히스토리 생명주기 (초단위), 2018-01-29 10->180초 (심플중계기 영향으로 늘림)
#define LBTTIMEOUT              5000    //ms단위
#define SENDRETRYCNT            5       //전송재시도횟수
#define FIRERESENDRETRYCNT      2       //매분마다의 재전송 화재
#define SCANDEVICEMAX           10      //개별 통신점검 최대 갯수
#define LBT_IMPORTANT           255
#define RFC_SEND_ABORT          (LPVOID)1
#define COMTESTINTERVAL         90      //이전통신 점검 후 새로운 통신점검인지 재시인지 확인하는 시간


#define MEMOWNER_CreatePacket       (MEMOWNER_ROLA+0)
#define MEMOWNER_PostCopyPacket     (MEMOWNER_ROLA+1)
#define MEMOWNER_CreatePacketMini   (MEMOWNER_ROLA+2)


typedef struct _SEND_HISTORY
    {
    JTIME RecvTime;     //CPU가 SleepMode에 있을 때 GetTickCount()의 값은 증가되지 않음
    WORD  PublishNo;
    BYTE  DeviceNo;
    } SEND_HISTORY;

static JOS_EVENT *LoRaSendQ;

static SEND_HISTORY SendHistoryList[SEND_HISTORY_QTY];
static CONST CHAR SetPacketStart[]="ROZETA";
static CONST CHAR PublishNoStr[]="PublishNo";

static WORD ReplyTimer;
static WORD SendWaitTimer;
static WORD SendFireDelay;              //화재복구 후 다시 화재가 발생한 경우 수신기 처리를 돕기 위해 신규화재를 지연시킴
static DWORD SettingStartTime, SettingStartCode;
static WORD G_PublishNo;                //패킷을 발행할 때마다 증가함
static BYTE NeedAckPublishNo;           //ACK받을 패킷의 발행번호
static BYTE NeedAckPktType;             //ACK받을 패킷의 PT
static BYTE RetrySendCnt;               //ACK응답이 없는 경우 재시도 카운트
static BYTE GettingToSendData;          //Q에서 보낼 데이터를 가져와서 RetrySendCnt값이 설정되는 기간에 1임
static BYTE SendTimeTimer;              //이시간이 경과하면 시간을 전송해라
static BYTE PowerOnReady;               //전원이 켜진 후 RFM 모듈이 정상동작함을 표시
static BYTE TempRfChannelNo;            //설정용으로 잠시 사용할 채널번호
static JTIME NominatedCommTestTime;     //지명한 통신 점검

static INT8 CanceledRssi;
static int  SendVariableTimer=JOS_TICKS_PER_SEC*5;  //아날로그 값을 주기적으로 전송
static int  SendVariableInterval=JOS_TICKS_PER_SEC*600; //틱단위값, 아날로그값 전송 시간, 100=1초
static INT16 LBT_Level, LbtCurrRssi;
static ftRFSendRecvCB RFSendRecvCB;


BOOL WINAPI RFC_IsSettingMode(VOID)             {return SettingStartTime;}
VOID WINAPI RFC_SendTime(int AfterTime)         {SendTimeTimer=AfterTime;}
BOOL WINAPI RFC_IsPowerOnReady(VOID)            {return PowerOnReady;}
VOID WINAPI RFC_SetSendAnalogInterval(int Sec)  {SendVariableInterval=Sec*JOS_TICKS_PER_SEC;}
int  WINAPI RFC_GetSendAnalogInterval(VOID)     {return SendVariableInterval/JOS_TICKS_PER_SEC;}
VOID WINAPI RFC_SendSensorValue(VOID)           {SendVariableTimer=SavedData.DeviceNo*3*JOS_TICKS_PER_SEC + 3*JOS_TICKS_PER_SEC;} //센서 값을 전송하게 함



//-----------------------------------------------------------------------------
//      지금 즉시 화재패킷을 발송하도록 함 (아직 최초 전송도 못한 경우 FALSE리턴)
//-----------------------------------------------------------------------------
BOOL WINAPI RFC_SendFireRightNow(VOID)
    {
    if (RetrySendCnt==0) SendWaitTimer=0;   //아직 첫째 발송을 안한 경우
    else ReplyTimer=1;                      //재전송할 패킷을 10초 기다리기 전에 지금 재시도 패킷을 전송해라
    return RetrySendCnt;
    }




//-----------------------------------------------------------------------------
//      현재 전송관련된 부분이 대기상태는지 체크함
//-----------------------------------------------------------------------------
BOOL WINAPI RFC_IsSendIdle(VOID)
    {
    RF_PACKET *CurrP;

    RFC_ManagerTask(&CurrP);
    return SendWaitTimer==0 && GettingToSendData==0 && RetrySendCnt==0 && (LoRaSendQ==NULL || JOSQEntries(LoRaSendQ)==0) && CurrP==NULL;
    }




//-----------------------------------------------------------------------------
//      중요변수 내용 표시 ("PRINT" 명령에 대한 처리)
//-----------------------------------------------------------------------------
VOID WINAPI RFC_PrintVar(VOID)
    {
    int I;
    JTIME NowTime;
    SEND_HISTORY *SH;

    NowTime=GetTodayToSecond();

    Printf("LbtLevel=%d"CRLF, LBT_Level);
    Printf("LbtCurrRssi=%d"CRLF, LbtCurrRssi);
    Printf("CanceledRssi=%d"CRLF, CanceledRssi);

    Printf("----SendHistoryList----"CRLF);
    for (I=0,SH=SendHistoryList; I<SEND_HISTORY_QTY; I++,SH++)
        {
        if (SH->RecvTime!=0) Printf("%d: %02X %02X %u"CRLF, I, SH->DeviceNo, SH->PublishNo, NowTime-SH->RecvTime);
        }
    }



//-----------------------------------------------------------------------------
//      패킷의 총 길이를 구함
//-----------------------------------------------------------------------------
LOCAL(int) GetPacketLen(int DataLen)
    {
    return DataLen + GetMemberOffset(RF_PACKET, Data) +1;   //+1은 CheckSum
    }

int WINAPI GetPacketLenII(CONST RF_PACKET *P)
    {
    return GetPacketLen(P->DataLen);
    }




//-----------------------------------------------------------------------------
//      패킷을 찍어봄
//-----------------------------------------------------------------------------
LOCAL(VOID) PrintTxPacket(CONST RF_PACKET *P)
    {
    int I,J;

    Printf("SENSOR:[%u]Tx", GetTickCount());
    Printf(" %08X", PeekBI((LPCBYTE)P));
    Printf(" %04X", PeekBIW((LPCBYTE)P+4));
    Printf(" %02X<-%02X", *((LPCBYTE)P+6), *((LPCBYTE)P+7));
    Printf(" P=%02X", *((LPCBYTE)P+8));
    Printf(" L=%02X", *((LPCBYTE)P+9));

    J=GetPacketLenII(P);
    for (I=10; I<J; I++) Printf(" %02X", *((LPBYTE)P+I));
    Printf(CRLF);
    }

LOCAL(VOID) PrintRxPacket(LPCBYTE RecvBuff, int RecvLen, int Rssi)
    {
    int I;

    Printf("SENSOR:[%u]Rx", GetTickCount());
    if (AUTOMANSPI_ChkPublicTimePkt(RecvBuff, RecvLen, NULL, NULL))
        {
        Printf(" %08X %08X %02X %02X %04X", PeekBI(RecvBuff), PeekBI(RecvBuff+4), RecvBuff[8], RecvBuff[9], PeekBIW(RecvBuff+10));
        goto ProcExit;
        }

    Printf(" %08X", PeekBI(RecvBuff));
    Printf(" %04X", PeekBIW(RecvBuff+4));
    Printf(" %02X<-%02X", RecvBuff[6], RecvBuff[7]);
    Printf(" P=%02X", RecvBuff[8]);
    Printf(" L=%02X", RecvBuff[9]);
    for (I=10; I<RecvLen; I++) Printf(" %02X", RecvBuff[I]);

    ProcExit:
    Printf(" (%d)"CRLF, Rssi);
    }



//-----------------------------------------------------------------------------
//      체크썸을 계산함 (패킷의 총길이 리턴)
//-----------------------------------------------------------------------------
LOCAL(int) CalcPacketCheckSum(RF_PACKET *P)
    {
    int ChkSumPos;

    ChkSumPos=GetPacketLenII(P)-1;
    *((LPBYTE)P+ChkSumPos)=GetJ8ChkSum((LPCBYTE)P, ChkSumPos);
    return ChkSumPos+1;
    }



//-----------------------------------------------------------------------------
//      패킷 헤더 설정
//-----------------------------------------------------------------------------
LOCAL(VOID) MakePacketHeader(RF_PACKET *P, int Receiver, int PacketType, int DataLen)
    {
    P->GroupNo=SavedData.GroupNo;
    P->Receiver=Receiver;
    P->Sender=SavedData.DeviceNo;
    P->PublishNo=++G_PublishNo;
    P->PacketType=PacketType;
    P->DataLen=DataLen;
    }



//-----------------------------------------------------------------------------
//      패킷 생성
//-----------------------------------------------------------------------------
LOCAL(POST_RF_PACKET*) CreatePacket(int Receiver, int PacketType, int DataLen)
    {
    POST_RF_PACKET *PRP;

    if ((PRP=(POST_RF_PACKET*)AllocMem(GetPacketLen(DataLen)+sizeof(POST_RF_HEADER), MEMOWNER_CreatePacket))!=NULL)
        {
        ZeroMem(&PRP->H, sizeof(POST_RF_HEADER));
        MakePacketHeader(&PRP->P, Receiver, PacketType, DataLen);
        }
    return PRP;
    }



//-----------------------------------------------------------------------------
//      On/Off Packet을 만듬
//-----------------------------------------------------------------------------
LOCAL(POST_RF_PACKET*) MakeOnOffPacket(int PacketType, BOOL OnOff)
    {
    POST_RF_PACKET *PRP;

    if ((PRP=CreatePacket(MASTERRECEIVER, PacketType, 1))!=NULL) PRP->P.Data[0]=(BYTE)OnOff;
    return PRP;
    }



//-----------------------------------------------------------------------------
//      16비트 정수 전송 Packet을 만듬
//-----------------------------------------------------------------------------
LOCAL(POST_RF_PACKET*) Make16ValuePacket(int PacketType, int Value)
    {
    POST_RF_PACKET *PRP;

    if ((PRP=CreatePacket(MASTERRECEIVER, PacketType, sizeof(WORD)))!=NULL) PokeW(PRP->P.Data, Value);
    return PRP;
    }



//-----------------------------------------------------------------------------
//      32비트 정수 전송 Packet을 만듬
//-----------------------------------------------------------------------------
LOCAL(POST_RF_PACKET*) Make32ValuePacket(int PacketType, int Value)
    {
    POST_RF_PACKET *PRP;

    if ((PRP=CreatePacket(MASTERRECEIVER, PacketType, sizeof(int)))!=NULL) Poke(PRP->P.Data, Value);
    return PRP;
    }




//-----------------------------------------------------------------------------
//      Time Packet을 만듬
//-----------------------------------------------------------------------------
LOCAL(VOID) MakeTimePacket(RF_PACKET *P)
    {
    MakePacketHeader(P, BROADCASTDEVICENO, PT_TIME, 4);
    Poke(P->Data, GetTodayToSecond());  //P->Data가 4의 경계에 있지 않음
    CalcPacketCheckSum(P);
    }



//-----------------------------------------------------------------------------
//      패킷을 Q에 넣음
//-----------------------------------------------------------------------------
LOCAL(VOID) PostPacket(POST_RF_PACKET *PRP)
    {
    if (PRP!=NULL)
        {
        CalcPacketCheckSum(&PRP->P);
        if (JOSQPost(LoRaSendQ, (DWORD)PRP, 0)!=JOS_ERR_NONE) FreeMem(PRP);
        }
    }



VOID WINAPI PostExtinguishFirePacket(int Reserved)  {PostPacket(MakeOnOffPacket(PT_EXTINGUISH, Reserved));}
VOID WINAPI PostManFirePacket(BOOL OnOff)           {PostPacket(MakeOnOffPacket(PT_MANFIRE, OnOff));}
VOID WINAPI PostFaultPacket(int Fault)              {PostPacket(MakeOnOffPacket(PT_FAULT, Fault));}
VOID WINAPI PostFireCryPacket(VOID)                 {PostPacket(MakeOnOffPacket(PT_CRYFIRE, ON));}
VOID WINAPI PostHelpCryPacket(VOID)                 {PostPacket(MakeOnOffPacket(PT_CRYHELP, ON));}
VOID WINAPI PostDoorPacket(BOOL OnOff)              {PostPacket(MakeOnOffPacket(PT_DOOR, OnOff));}
VOID WINAPI PostTrespassPacket(BOOL OnOff)          {PostPacket(MakeOnOffPacket(PT_TRESPASS, OnOff));}
VOID WINAPI PostFirePacket(int FireType, BOOL OnOff){PostPacket(MakeOnOffPacket(FireType, OnOff));}      //2019-01-25 해제 신호일 때만 사용함
VOID WINAPI PostGasPacket(BOOL OnOff)               {PostPacket(MakeOnOffPacket(PT_GAS, OnOff));}
VOID WINAPI PostHighTempPacket(VOID)                {PostPacket(MakeOnOffPacket(PT_HIGHTEMP, ON));}
VOID WINAPI PostLowTempPacket(VOID)                 {PostPacket(MakeOnOffPacket(PT_LOWTEMP, ON));}
VOID WINAPI PostHighCOPacket(VOID)                  {PostPacket(MakeOnOffPacket(PT_HIGHCO, ON));}
VOID WINAPI PostBreakDownPacket(BOOL OnOff)         {PostPacket(MakeOnOffPacket(PT_BREAKDOWN, OnOff));}
VOID WINAPI PostFirePredetectPacket(BOOL OnOff)     {PostPacket(MakeOnOffPacket(PT_FIREPREDETECT, OnOff));}
VOID WINAPI PostElectricLeakPacket(BOOL OnOff)      {PostPacket(MakeOnOffPacket(PT_ELECTRICLEAK, OnOff));}
VOID WINAPI PostLandslidePacket(int DangerLevel)    {PostPacket(MakeOnOffPacket(PT_LANDSLIDE, DangerLevel));}
VOID WINAPI PostOxygenDensityPacket(int Psnt)       {PostPacket(MakeOnOffPacket(PT_OXYGEN, Psnt));}
VOID WINAPI PostTemperaturePacket(int Value)        {PostPacket(Make16ValuePacket(PT_TEMP, Value));}
VOID WINAPI PostO2ConcentrationPacket(int Value)    {PostPacket(Make32ValuePacket(PT_O2CONC, Value));}
VOID WINAPI PostPirDetectedPacket(VOID)             {PostPacket(MakeOnOffPacket(PT_PIRDETECT, ON));}
VOID WINAPI PostPirSosPacket(VOID)                  {PostPacket(MakeOnOffPacket(PT_PIRSOS, ON));}




//-----------------------------------------------------------------------------
//      테스트 스위치를 눌렀을 때
//-----------------------------------------------------------------------------
VOID WINAPI PostSwitchPacket(BOOL SyncedTimeFg)
    {
    POST_RF_PACKET *PRP;

    if ((PRP=CreatePacket(MASTERRECEIVER, PT_SWITCH, SyncedTimeFg ? 6:1))!=NULL)
        {
        PRP->P.Data[0]=ON;
        PostPacket(PRP);
        }
    }



//-----------------------------------------------------------------------------
//      2021-06-21 RealTime추가
//-----------------------------------------------------------------------------
VOID WINAPI PostBattPacket(int BattStat, BOOL SyncedTimeFg)
    {
    POST_RF_PACKET *PRP;

    if ((PRP=CreatePacket(MASTERRECEIVER, PT_BATTERY, SyncedTimeFg ? 6:1))!=NULL)
        {
        PRP->P.Data[0]=(BYTE)BattStat;
        if (SyncedTimeFg) PRP->P.Data[0]|=0x80; //0x80은 뒤에 RealTime을 포함한 경우임
        PostPacket(PRP);
        }
    }



//-----------------------------------------------------------------------------
//      통신점검 3차 재시도 때 응답보내는 패킷
//-----------------------------------------------------------------------------
LOCAL(VOID) PostBattPacketII(int BattStat, int Order)
    {
    POST_RF_PACKET *PRP;

    if ((PRP=CreatePacket(MASTERRECEIVER, PT_BATTERY, 1))!=NULL)
        {
        PRP->H.SendDelay=Order*100;
        PRP->P.Data[0]=(BYTE)BattStat;
        PostPacket(PRP);
        }
    }



//-----------------------------------------------------------------------------
//      전송할 모든 패킷을 꺼내어 버림
//-----------------------------------------------------------------------------
VOID WINAPI RFC_DestroySendQ(VOID)
    {
    POST_RF_PACKET *PRP;

    while ((PRP=(POST_RF_PACKET*)JOSQPend(LoRaSendQ, CHECKACCEPT, NULL))!=NULL) FreeMem(PRP);
    RFC_ManagerTask(RFC_SEND_ABORT);        //지금 재시도 중인 송출 패킷 Abort
    }



//-----------------------------------------------------------------------------
//      국내 인증형 볼케이노에서 사용 (화재패킷은 긴급으로 발송하도록함)
//-----------------------------------------------------------------------------
VOID WINAPI PostFireDetectPacket(int PacketType)
    {
    POST_RF_PACKET *PRP;

    if ((PRP=CreatePacket(MASTERRECEIVER, PacketType, 1))!=NULL)
        {
        PRP->P.Data[0]=SEN_DETECT;
        PRP->H.LBTDelta=LBT_IMPORTANT;
        if ((PRP->H.SendDelay=SendFireDelay)==0) PRP->H.SendDelay=(GetDeviceRegOrder()%10)*20;  //10개 동시화재에서 5초안에 2개의 화재가 들어와야 한다는 조건을 통과하기 위해 10개 감지기별로 차등을 200ms 간격으로 두고 신규화재를 발송함
        RFC_DestroySendQ();                 //화재신호는 초긴급이므로 기존에 보낼 패킷은 빼버림 (Alive, BattLow 신호등이 삭제됨)
        PostPacket(PRP);
        }
    }



//-----------------------------------------------------------------------------
//      1분마다 재송출하는 화재신호
//-----------------------------------------------------------------------------
VOID WINAPI PostRetryFirePacket(int PacketType)
    {
    POST_RF_PACKET *PRP;

    if ((PRP=CreatePacket(MASTERRECEIVER, PacketType, 1))!=NULL)
        {
        PRP->P.Data[0]=SEN_DETECTRESEND;
        RFC_DestroySendQ();                 //아직 못 보낸 본화재 제거
        PostPacket(PRP);
        }
    }



//-----------------------------------------------------------------------------
//      Ack Packet을 Q에 넣음
//-----------------------------------------------------------------------------
LOCAL(VOID) PostAckPacket(int RecvDevID, int RecvPktType, int RecvPktSN, int PacketAttr)
    {
    POST_RF_PACKET *PRP;

    if ((PRP=CreatePacket(RecvDevID, PT_ACK, 2))!=NULL)
        {
        PRP->P.Data[0]=(BYTE)RecvPktType;
        PRP->P.Data[1]=(BYTE)RecvPktSN;
        PostPacket(PRP);
        }
    }



//-----------------------------------------------------------------------------
//      Control Ack Packet을 Q에 넣음
//-----------------------------------------------------------------------------
LOCAL(VOID) PostControlAckPacket(int RecvDevID, LPCBYTE AckData, int PacketAttr)
    {
    POST_RF_PACKET *PRP;

    if ((PRP=CreatePacket(RecvDevID, PT_CTLACK, 3))!=NULL)
        {
        CopyMem(PRP->P.Data, AckData, 3);
        PostPacket(PRP);
        }
    }



//-----------------------------------------------------------------------------
//      Setting Ack Packet을 Q에 넣음
//-----------------------------------------------------------------------------
LOCAL(VOID) PostSettingAckPacket(int RecvDevID, int Data1, int Data2, int PacketAttr)
    {
    POST_RF_PACKET *PRP;

    if ((PRP=CreatePacket(RecvDevID, PT_SETACK, 2))!=NULL)
        {
        PRP->P.Data[0]=Data1;
        PRP->P.Data[1]=Data2;
        PostPacket(PRP);
        }
    }



//-----------------------------------------------------------------------------
//      통신전검 응답패킷을 만듬
//-----------------------------------------------------------------------------
VOID WINAPI PostDiagCommAckPacket(UINT State)
    {
    POST_RF_PACKET *PRP;

    if ((PRP=CreatePacket(MASTERRECEIVER, PT_STATE, 1))!=NULL)
        {
        PRP->P.Data[0]=State;
        PostPacket(PRP);
        }
    }



//-----------------------------------------------------------------------------
//      주어진 패킷을 복사하여 큐에 넣음
//-----------------------------------------------------------------------------
VOID WINAPI PostCopyPacket(CONST RF_PACKET *OrgP, int SendDelay)
    {
    int PktLen;
    POST_RF_PACKET *PRP;

    PktLen=GetPacketLenII(OrgP);
    if ((PRP=(POST_RF_PACKET*)AllocMem(PktLen+sizeof(POST_RF_HEADER), MEMOWNER_PostCopyPacket))!=NULL)
        {
        ZeroMem(&PRP->H, sizeof(POST_RF_HEADER));
        PRP->H.SendDelay=SendDelay;
        CopyMem(&PRP->P, OrgP, PktLen);
        PostPacket(PRP);
        }
    }



//-----------------------------------------------------------------------------
//      SendHistory 기록
//-----------------------------------------------------------------------------
LOCAL(VOID) PutSendHistory(CONST RF_PACKET *P)
    {
    int I;
    JTIME NowTime;
    DWORD LongTime=0, ElapseTime;
    SEND_HISTORY *SH, *PutSH=NULL;

    NowTime=GetTodayToSecond();
    for (I=0,SH=SendHistoryList; I<SEND_HISTORY_QTY; I++,SH++)
        {
        if (SH->PublishNo==P->PublishNo &&
            SH->DeviceNo==P->Sender)
            {
            PutSH=SH;
            break;
            }
        if ((ElapseTime=NowTime-SH->RecvTime)>LongTime) //가장 오래된 패킷 히스토리 사용
            {
            PutSH=SH;
            LongTime=ElapseTime;
            }
        }
    PutSH->RecvTime=NowTime;
    PutSH->PublishNo=P->PublishNo;
    PutSH->DeviceNo=P->Sender;
    }



//-----------------------------------------------------------------------------
//      ACK가 필요한 패킷인지 알려줌
//-----------------------------------------------------------------------------
LOCAL(BOOL) IsNeedAckPacketType(int PacketType)
    {
    return  PacketType!=PT_SENSOR &&        //센서값도 응답임
            PacketType!=PT_O2CONC &&        //주기적으로 발송하는 산소농도
            PacketType!=PT_CTLACK &&        //제어에 대한 응답임
            PacketType!=PT_SECACK &&        //방범설정에 대한 응답임
            PacketType!=PT_SETACK &&        //센서설정에 대한 응답임
            PacketType!=PT_TEMPACK &&       //온도알람설정에 대한 응답임
            PacketType!=PT_COALARMACK &&    //CO알람설정에 대한 응답임
            PacketType!=PT_SETINTVLACK &&   //전송시간간격설정에 대한 응답임
            PacketType!=PT_STATE &&         //통신점검에 대한 응답임, 2021-06-29 정원중계기에서 ACK를 받음, 2021-08-30 1초간Rf트래픽(기획안3)을 적용하면서 ACK를 보낼 틈이 없어서 ACK를 제거함
            PacketType!=PT_ACK &&           //PT_ACK는 센서모듈 테스트할 때 발생함
            PacketType!=PT_ADDROUTER &&
            PacketType!=PT_CONTROL &&
            PacketType!=PT_SECURITY &&
            PacketType!=PT_SETTING &&
            PacketType!=PT_TEMPALARM &&
            PacketType!=PT_COALARM &&
            PacketType!=PT_SCAN &&          //통신점검
            PacketType!=PT_RESET &&         //화재상태클리어(화재복구)
            PacketType!=PT_BELL &&          //경종 울려라
            PacketType!=PT_SBELL &&         //경종 짧게 울려라
            PacketType!=PT_SETINTERVAL;
            //PacketType!=PT_DALIVEACK;     //감지기 전용 ALIVE 패킷의 응답
            //PacketType!=PT_COIRTEMP;      //2018-11-07 자주 전송하는 대신 재시도를 뺌
                                            //2018-11-16 재시도 3회로 변경
    }



//-----------------------------------------------------------------------------
//      ACK가 필요한 패킷인지 알려줌
//-----------------------------------------------------------------------------
LOCAL(BOOL) IsNeedAckPacket(CONST RF_PACKET *P)
    {
    return P->Receiver!=BROADCASTDEVICENO &&    //방송 패킷은 ACK안보냄
           (P->PacketType!=PT_ROUTEDATA && IsNeedAckPacketType(P->PacketType)) ||
           (P->PacketType==PT_ROUTEDATA && IsNeedAckPacketType(P->Data[0]));
    }




//-----------------------------------------------------------------------------
//      5초안에 전송한 기록이 있는지 확인
//-----------------------------------------------------------------------------
LOCAL(BOOL) IsInSendHistory(CONST RF_PACKET *P)
    {
    int I, Rslt=FALSE;
    JTIME NowTime;
    DWORD HisTime;
    SEND_HISTORY *SH;

    HisTime=(VolcanoBoardID==VKBOARD_KOR) ? 15:PACKET_HISTORY_TIME;

    NowTime=GetTodayToSecond();
    for (I=0,SH=SendHistoryList; I<SEND_HISTORY_QTY; I++,SH++)
        {
        if (SH->RecvTime!=0 &&
            NowTime-SH->RecvTime<HisTime &&
            SH->PublishNo==P->PublishNo &&
            SH->DeviceNo==P->Sender)
            {
            Rslt=TRUE;
            break;
            }
        }
    return Rslt;
    }



//-----------------------------------------------------------------------------
//      설정 패킷의 총 길이를 구함
//-----------------------------------------------------------------------------
LOCAL(int) GetSettingPacketLen(LPCBYTE lp)
    {
    return lp[6] +6 +1 +1;      //6:Len("ROZETA"), +1:Command Len, +1CheckSum
    }



//-----------------------------------------------------------------------------
//      셋팅 패킷을 보냄
//
// "ROZETA" 길이 "Host에서받음문자열" 체크썸
//-----------------------------------------------------------------------------
LOCAL(VOID) SendSettingPacket(LPCSTR HostApply)
    {
    int  I, Len;
    CHAR Buff[MAXPACKETSIZE];

    lstrcpy(Buff, SetPacketStart);
    if (SettingStartCode!=0) Poke(Buff+2, SettingStartCode);
    Buff[6]=lstrlen(HostApply);
    lstrcpyn(Buff+7, HostApply, MAXPACKETSIZE-10);  //7=Len("ROZETA")+LenByte

    Len=GetSettingPacketLen((LPCBYTE)Buff)-1;
    Buff[Len]=GetJ8ChkSum((LPCBYTE)Buff, Len);

    Printf("SENSOR: TxA");
    for (I=0; I<=Len; I++) Printf(" %02X", *((LPBYTE)Buff+I));
    Printf(CrLfStr);

    if (RFM_SendPacket((LPCBYTE)Buff, Len+1)!=FALSE) TxLedControl(TXLED_FLASH);
    }



//-----------------------------------------------------------------------------
//      LoRa 무선으로 설정하는 명령 처리
//-----------------------------------------------------------------------------
#define SP_IGNORE       0
#define SP_SETOK        1
#define SP_INVALID      2
#define SP_EXIT         3
LOCAL(VOID) SettingProc(LPBYTE RecvP, int RecvBytes)
    {
    int I,J, Rslt=SP_IGNORE;
    DWORD PairCode;
    LPSTR lp;
    char  Buff[80];

    I=GetSettingPacketLen(RecvP);
    PairCode=Peek(RecvP+2);
    if (SettingStartCode==0) SettingStartCode=PairCode;
    if (PairCode!=SettingStartCode || I!=RecvBytes || CompMemStr((LPSTR)RecvP, "RO")!=0) goto ProcExit;

    I=(GetJ8ChkSum(RecvP, RecvBytes-1)==RecvP[RecvBytes-1]);
    Printf("SENSOR: LoRa Recv %s"CRLF, I ? "OK":"Crc Err");     //" (RSSI=%d)", RFM_ReadRSSI()
    if (I==FALSE) goto ProcExit;

    PlayBeep(BeeBeeSnd);
    lp=(LPSTR)RecvP+7;
    lp[RecvP[6]]=0;

    Printf("SENSOR: HostCmd: '%s'"CRLF, lp);
    if (lstrcmp(lp, "SAVE")==0) {PutKey(KC_SaveConfig); PutKey(KC_TestUp); Rslt=SP_SETOK;}
    else if (lstrcmp(lp, "EXIT")==0) {PutKey(KC_TestUp); Rslt=SP_SETOK;}
    else if (lstrcmp(lp, "BOOT")==0) {K_SetTimer(KC_BootTmr, 2000); Rslt=SP_SETOK;}
    else if (lstrcmp(lp, "GET")==0)
        {
        wsprintf(Buff, "G=%d&D=%d&C=%d&P=%d&R=%d",
            SavedData.GroupNo, SavedData.DeviceNo, SavedData.ChannelNo,
            SavedData.RFPower, SavedData.RepeaterMode);
        SendSettingPacket(Buff);
        Rslt=SP_EXIT;
        }
    else if (lstrcmp(lp, "GET1")==0)
        {
        wsprintf(Buff, "FDP=%d&WI=%u&ANT=%d",
                        SetFireDetectPolicy(-1),
                        SetWakeupInterval(0),
                        GetAntenna());
        SendSettingPacket(Buff);
        Rslt=SP_EXIT;
        }
    else if (lstrcmp(lp, "GET2")==0)
        {
        int SenFg=-1, LowTemp, HighTemp;

        SensorTempAlarmRF(&SenFg, &LowTemp, &HighTemp);
        wsprintf(Buff, "&TE=%d %d %d", SenFg, LowTemp, HighTemp);
        SendSettingPacket(Buff);
        Rslt=SP_EXIT;
        }
    else if (lstrcmp(lp, "GET3")==0)
        {
        wsprintf(Buff, "DM=%d&EH=%d", SetDemoMode(-1), SetExchangeHeatFg(-1));
        SendSettingPacket(Buff);
        Rslt=SP_EXIT;
        }
    else{
        Rslt=SP_INVALID;
        if (GetUrlVarText(lp, "G", Buff, sizeof(Buff))!=FALSE)
            {
            I=AtoI(Buff, &J); if (J==0) {Invalid: Rslt=SP_INVALID; goto ProcExit;}
            SavedData.GroupNo=I;
            Rslt=SP_SETOK;
            }

        if (GetUrlVarText(lp, "D", Buff, sizeof(Buff))!=FALSE)
            {
            I=AtoI(Buff, &J); if (J==0 || I<0 || I>254) goto Invalid;
            SavedData.DeviceNo=(BYTE)I;
            Rslt=SP_SETOK;
            }

        if (GetUrlVarText(lp, "C", Buff, sizeof(Buff))!=FALSE)
            {
            I=AtoI(Buff, &J); if (J==0 || I<0 || I>255) goto Invalid;
            SavedData.ChannelNo=(BYTE)I;
            Rslt=SP_SETOK;
            }

        if (GetUrlVarText(lp, "P", Buff, sizeof(Buff))!=FALSE)
            {
            I=AtoI(Buff, &J); if (J==0 || I<0) goto Invalid;
            SavedData.RFPower=(BYTE)I;
            Rslt=SP_SETOK;
            }

        if (GetUrlVarText(lp, "R", Buff, sizeof(Buff))!=FALSE)
            {
            I=AtoI(Buff, &J); if (J==0 || I<0 || I>1) goto Invalid;
            SavedData.RepeaterMode=(BYTE)I;
            Rslt=SP_SETOK;
            }

        if (GetUrlVarText(lp, "WI", Buff, sizeof(Buff))!=FALSE)
            {
            SetWakeupInterval(AtoI(Buff, &J));
            Rslt=SP_SETOK;
            }

        if (GetUrlVarText(lp, "FDP", Buff, sizeof(Buff))!=FALSE)
            {
            if (SetFireDetectPolicy(AtoI(Buff, &J))<0) goto Invalid;
            Rslt=SP_SETOK;
            }

        if (GetUrlVarText(lp, "TE", Buff, sizeof(Buff))!=FALSE)
            {
            int SenFg, LowTemp, HighTemp;
            LPCSTR lp2;

            SenFg=AtoI(Buff, &J); if (J==0) goto Invalid;
            lp2=(LPSTR)SkipSpace(Buff+J);
            LowTemp=AtoI(lp2, &J); if (J==0) goto Invalid;
            lp2=(LPSTR)SkipSpace(lp2+J);
            HighTemp=AtoI(lp2, &J); if (J==0) goto Invalid;
            if (SensorTempAlarmRF(&SenFg, &LowTemp, &HighTemp)==FALSE) goto Invalid;
            Rslt=SP_SETOK;
            }

        if (GetUrlVarText(lp, "ANT", Buff, sizeof(Buff))!=FALSE)
            {
            I=AtoI(Buff, &J); if (J==0 || I<0 || I>1) goto Invalid;
            SetAntenna(I);
            Rslt=SP_SETOK;
            }

        if (GetUrlVarText(lp, "DM", Buff, sizeof(Buff))!=FALSE)
            {
            I=AtoI(Buff, &J); if (J==0 || I<0 || I>1) goto Invalid;
            SetDemoMode(I);
            Rslt=SP_SETOK;
            }

        if (GetUrlVarText(lp, "EH", Buff, sizeof(Buff))!=FALSE)
            {
            I=AtoI(Buff, &J); if (J==0 || I<0 || I>1) goto Invalid;
            SetExchangeHeatFg(I);
            Rslt=SP_SETOK;
            }
        }

    ProcExit:
    if (Rslt!=SP_IGNORE && Rslt!=SP_EXIT) SendSettingPacket(Rslt==SP_SETOK ? "SETOK":"INVALID");
    }



//-----------------------------------------------------------------------------
//      RF 설정모드로 들어값니다
//-----------------------------------------------------------------------------
VOID WINAPI RFC_EnterSetupMode(VOID)
    {
    RFM_SetChannel(TempRfChannelNo!=0 ? TempRfChannelNo:SETUPCHANNELNO);
    TempRfChannelNo=0;
    SettingStartTime=GetTickCount();
    SettingStartCode=0;
    Printf("Enter RF Setting Mode"CRLF);
    }

VOID WINAPI RFC_ExitSetupMode(VOID)
    {
    RFM_SetChannel(SavedData.ChannelNo);
    SettingStartTime=0;
    Printf("Exit RF Setting Mode"CRLF);
    }




//-----------------------------------------------------------------------------
//      인증체크를 한 후에 설정모드로 진입
//  Data:
//    DW: JTIME ^ 그룹번호
//    DW: JTIME의 CRC32
//    DB: 설정할 채널번호
//-----------------------------------------------------------------------------
LOCAL(BOOL) StartSettingMode(RF_PACKET *P)
    {
    BOOL Rslt=FALSE;
    DWORD Time;

    if (P->DataLen<9) goto ProcExit;
    Time=Peek(P->Data) ^ SavedData.GroupNo;
    if (CalculateCRC((LPCBYTE)&Time, 4, 0x3FA0B1CF)!=Peek(P->Data+4))
        {
        Printf("SENSOR: TimeCRC Error"CRLF);
        goto ProcExit;
        }
    #if 0
    if (UGetDiff(GetTodayToSecond(), Time)>20*60)   //현재 RTC시간과 20분이상 차이남
        {
        Printf("SENSOR: Timeout"CRLF);
        goto ProcExit;
        }
    #endif
    TempRfChannelNo=P->Data[8];
    PutKey(KC_TestLong1);
    Rslt=TRUE;

    ProcExit:
    return Rslt;
    }



//-----------------------------------------------------------------------------
//      화재를 클리어함
//-----------------------------------------------------------------------------
LOCAL(VOID) ClearFire(VOID)
    {
    SendFireDelay=3*JOS_TICKS_PER_SEC;      //신규화재 송출을 3초간 지연
    SetFireState(FALSE);
    RFC_DestroySendQ();
    }



//-----------------------------------------------------------------------------
//      화재패킷인지 알려줌
//-----------------------------------------------------------------------------
BOOL WINAPI RFC_IsFirePacket(int PT)
    {
    return PT==PT_FIRE || PT==PT_HEAT || PT==PT_SMOKE || PT==PT_FLAME || PT==PT_MANFIRE;
    }



//-----------------------------------------------------------------------------
//      나에게 온 ACK인지 확인
//      2021-05-17 부터는 모둠 ACK가 방송모드로 올 수도 있음
//
//Tx 99110300 14FFFFFF FF<-00 41 08 98 01 05 55 54 51 45 11 53
//   Group    Time     R   S  P  L  |  |  |  +------------+ Crc
//                            T  E  |  |  |   감지기당 2비트의 ACK정보 (감지기 20개이면 5바이트)
//                               N  |  |  |
//                                  |  |  +-- 05이면 뒤에 데이터가 모둠ACK, 02이면 RealTime
//                                  |  +----- 뒤에 비트맵 데이터의 시작 감지기 번호
//                                  +-------- 어떤 패킷의 ACK인지 (원 패킷의 Type)
//-----------------------------------------------------------------------------
LOCAL(BOOL) IsMyAck(CONST RF_PACKET *P, BOOL *lpNeedClrFire)
    {
    int I, MyAck=0, NeedClrFire=0, FirstDevNo, GADataLen;
    LPCBYTE lpGA;

    if (P->Receiver==SavedData.DeviceNo)
        {
        if (P->Data[0]==NeedAckPktType &&
            P->Data[1]==NeedAckPublishNo)
            {
            MyAck=1;
            if (P->DataLen>=3 && P->Data[2]==OC_RESET) NeedClrFire=1;           //화재상태클리어(화재복구), 많은 감지기가 화재상태일 경우 RF 트래픽잼으로 인해 화재복구가 잘 안되어 수신기의 복구중에 ACK에 화재복구 명령을 싫어보냄
            }
        }
    else if (P->Receiver==BROADCASTDEVICENO)
        {
        if (P->PacketType==PT_ACK)
            {
            if (P->DataLen>3 &&
                (P->Data[0]==NeedAckPktType || (RFC_IsFirePacket(P->Data[0])!=0 && RFC_IsFirePacket(NeedAckPktType)!=0)) &&
                P->Data[2]==OC_GATHEREDACK)
                {
                FirstDevNo=P->Data[1];
                lpGA=P->Data+3;
                GADataLen=P->DataLen-3;
                ChkBitmap:
                if (FirstDevNo!=0)  //비트방식
                    {
                    if (P->PacketType==PT_COLLECTACKT && FirstDevNo==SavedData.DeviceNo) SendSyncTime(3);
                    GADataLen<<=2;                                  //1바이트당 4개 감지기의 모둠ACK 정보가 있음
                    if (SavedData.DeviceNo>=FirstDevNo && SavedData.DeviceNo<FirstDevNo+GADataLen)
                        {
                        I=(SavedData.DeviceNo-FirstDevNo)<<1;       //1개 감지기당 2비트씩 할애됨
                        if (CHKBITBYTEARRAY(lpGA, I)!=0)
                            {
                            MyAck=1;
                            I++;
                            if (CHKBITBYTEARRAY(lpGA, I)!=0) NeedClrFire=1;
                            }
                        }
                    }
                else{
                    for (I=0; I<GADataLen; I++)
                        {
                        if ((lpGA[I] & 0x7F)==SavedData.DeviceNo)
                            {
                            if (I==0 && P->PacketType==PT_COLLECTACKT) SendSyncTime(3);
                            MyAck=1;
                            if (lpGA[I] & 0x80) NeedClrFire=1;
                            break;
                            }
                        }
                    }
                }
            }
        else if (P->PacketType==PT_COLLECTACK || P->PacketType==PT_COLLECTACKT)
            {
            FirstDevNo=P->Data[0];
            lpGA=P->Data+1;
            GADataLen=P->DataLen-1;
            goto ChkBitmap;
            }
        }

    *lpNeedClrFire=NeedClrFire;
    return MyAck;
    }



//-----------------------------------------------------------------------------
//      수신패킷 처리 (센서모듈)
//-----------------------------------------------------------------------------
LOCAL(VOID) SensorPacketProc(RF_PACKET *P)
    {
    BOOL NeedClrFire;
    SYSTEMTIME ST;
    BYTE AckData[4];

    switch (P->PacketType)
        {
        case PT_TIME:
            UnpackTotalSecond(&ST, Peek(P->Data)); //P->Data가 4의 경계에 들지 않아 한꺼번에 못읽음
            if (ST.wYear>=2018)
                {
                SetLocalTime(&ST);
                if (SavedData.DeviceNo!=MASTERRECEIVER)
                    Printf("HOSTCMD ALIVE %u"CRLF, PackTotalSecond(&ST));       //중계기인경우 수신부에게 하부 센서들에게 시간을 전송하도록함
                }
            break;

        case PT_ACK:
        case PT_COLLECTACK:
        case PT_COLLECTACKT:
            if (IsMyAck(P, &NeedClrFire))   //RetrySendCnt!=0 && ... 이걸 뺀 이유는 PT_COLLECTACK는 RetrySendCnt==0일 때 처리됨
                {
                RetrySendCnt=0;
                if (RFSendRecvCB) RFSendRecvCB(RFCB_RECVACK, P);
                if (NeedClrFire) ClearFire();                                   //화재상태클리어(화재복구), 많은 감지기가 화재상태일 경우 RF 트래픽잼으로 인해 화재복구가 잘 안되어 수신기의 복구중에 ACK에 화재복구 명령을 싫어보냄
                }
            break;

        case PT_ALIVE:          //송수신기 시험용
            PostAckPacket(P->Sender, P->PacketType, P->PublishNo, PA_NORMAL);
            break;

        case PT_CONTROL:
            AuxIoControl(AckData, P->Data);
            PostControlAckPacket(P->Sender, AckData, PA_NORMAL);
            break;

        case PT_SETTING:
            {
            int Data1, Data2;

            Data1=Data2=-1;
            if (P->DataLen>=1) Data1=P->Data[0];
            if (P->DataLen>=2) Data2=P->Data[1];
            SensorSetting(&Data1, &Data2);
            PostSettingAckPacket(P->Sender, Data1, Data2, PA_NORMAL);
            break;
            }

        case PT_SCAN:
        case PT_RESET:
        case PT_BELL:
        case PT_SBELL:
            {
            int   OrderNo, ThirdRetry;
            JTIME CurrTime;

            OrderNo=ThirdRetry=0;
            if (P->DataLen==0 || (OrderNo=SearchMemByte(P->Data, P->DataLen, SavedData.DeviceNo)+1)>0)
                {
                if (P->DataLen==0) NominatedCommTestTime=0;                     //첫번째 방송모드 통신점검
                else{
                    CurrTime=GetTodayToSecond();
                    if (NominatedCommTestTime!=0 && CurrTime-NominatedCommTestTime>=COMTESTINTERVAL) NominatedCommTestTime=0;  //감지기가 10개 이하인 경우에는 최초 통신점검 때에도 지정방식을 사용함
                    else{
                        if (NominatedCommTestTime==0) NominatedCommTestTime=CurrTime;
                        else ThirdRetry=1;
                        }
                    }

                if (P->PacketType==PT_RESET) ClearFire();                       //화재상태클리어(화재복구) 2019-11-06 응답을 보낼 때까지 화재 상태 유지, 신규화재 발생을 저지할려고
                else if (P->PacketType==PT_BELL) RingBell(BELL_ON);
                else if (P->PacketType==PT_SBELL) RingBell(BELL_SHORTON);

                #if 0
                if (ThirdRetry==0) SetNeedReplyStateFg();
                else{
                    AUTOMANSPI_Reinit();
                    LOG_Add(LOG_RFMRECOVERY, 0, 0, 0);
                    PostBattPacketII(GetBatteryStatus(), OrderNo);              //해당 감지기가 보내야 할 시간이 있음
                    }
                #else
                if (ThirdRetry==0)
                    {
                    if (NominatedCommTestTime==0) SetNeedReplyStateFg();
                    else PostBattPacketII(GetBatteryStatus(), OrderNo);         //해당 감지기가 보내야 할 시간이 있음
                    }
                else{
                    AUTOMANSPI_Reinit();
                    LOG_Add(LOG_RFMRECOVERY, 0, 0, 0);
                    SetNeedReplyStateFg();
                    }
                #endif
                }
            break;
            }

        case PT_STARTSETTING:
            StartSettingMode(P);
            //break;
        }
    }



//-----------------------------------------------------------------------------
//      내가 받아서 처리해야할 패킷인지 알려줌
//-----------------------------------------------------------------------------
LOCAL(BOOL) IsToMe(RF_PACKET *P)
    {
    return P->Receiver==SavedData.DeviceNo || P->Receiver==BROADCASTDEVICENO;
    }



//-----------------------------------------------------------------------------
//      정해진 시간에 보내야하는 패킷인지 알려줌
//-----------------------------------------------------------------------------
LOCAL(BOOL) IsInTimePacket(CONST RF_PACKET *P)
    {
    return P->PacketType==PT_STATE ||       //통신점검의 응답
           (RFC_IsFirePacket(P->PacketType)!=FALSE && P->Data[0]==SEN_DETECTRESEND);
    }

LOCAL(BOOL) IsFirstFirePacket(CONST RF_PACKET *P)
    {
    return RFC_IsFirePacket(P->PacketType)!=FALSE && P->Data[0]==SEN_DETECT;
    }




//-----------------------------------------------------------------------------
//      LBT를 체크함 (리턴값은 10ms 단위의 지연값임)
//      Listen before talk (LBT)
//-----------------------------------------------------------------------------
#define LBT_OK          0
#define LBT_CANCEL      -1
#define LBT_TIMEOUT     -2
LOCAL(int) CheckLBT(CONST RF_PACKET *P, int LBTDelta, LPBYTE LBTStart)
    {
    int   Rslt, LbtCha='^';
    DWORD NowTime;
    static INT8  LbtLevel;
    static DWORD LbtTime, StartTime, KeepTime, TimeOut, TearDispTime;

    NowTime=GetTickCount();
    if (LBTStart[0]!=0)
        {
        TimeOut=LBTTIMEOUT;
        KeepTime=RFM_GetWaitForRfOut()*(SavedData.DeviceNo%20);                     //송출후 RF로 나오기까지 시간을 기다림, 그래야 RSSI가 측정되니까
        LbtLevel=LBT_Level;
        //if (LBTDelta!=0 && LBTDelta!=LBT_IMPORTANT) {Rslt=LBT_OK; goto ProcExit;} //화재수신기 자동통신점검에서 가끔 오류가 발생하여 시험하기 위한 코드임
        if (LBTDelta!=0 && LBTDelta!=LBT_IMPORTANT) {TimeOut=20; LbtLevel=-90;}     //통신 점검 신호는 자신이 송출할 시간에 이미 누가 송출하고 있으면 송출을 취소함
        if (LBTDelta==LBT_IMPORTANT) KeepTime=RFM_GetWaitForRfOut();                //바로 송출하니까 Wakeup후 보내는 신호와 충돌하여 서로 깨뜨림 => 30으로 해봤지만 효과 없었음
        else if (LBTDelta!=0) KeepTime=0;                                           //통신 점검 신호와 중요한 화재신호는 주변 RSSI값이 낮으면 바로 발송
        StartTime=LbtTime=NowTime;
        LBTStart[0]=0;
        }

    if (IsInTimePacket(P))
        {
        if (IsSendTimeITP()==FALSE) {LbtCha='#'; goto Wait;}
        Rslt=LBT_OK;
        goto ProcExit;
        }

    if (IsFirstFirePacket(P)==FALSE && IsRecvCollectAckTime()) {LbtTime=NowTime; LbtCha='$';}   //수신기가 재전송 화재의 ACK를 보낼 시간일 때는 전송을 하지 않음
    else if ((LbtCurrRssi=RFM_ReadRSSI())>=LbtLevel) {LbtTime=NowTime; LbtCha='&';}
    //Printf("LBT RSSI=%d"CRLF, LbtCurrRssi);

    if (NowTime-LbtTime>KeepTime) {Rslt=LBT_OK; goto ProcExit;}                     //신호가 없어진지 5ms후에 발송해야 함
    if (NowTime-StartTime>=TimeOut)
        {
        Rslt=(LBTDelta==0 || LBTDelta==LBT_IMPORTANT) ? LBT_TIMEOUT:LBT_CANCEL;
        if (Rslt==LBT_CANCEL) CanceledRssi=LbtCurrRssi;
        goto ProcExit;
        }

    Wait:
    Rslt=1;     //10ms지연 후 이루틴을 다시 호출하도록함
    if (NowTime-TearDispTime>=500)
        {
        Printf("%c"CRLF, LbtCha);
        TearDispTime=NowTime;
        }

    ProcExit:
    return Rslt;
    }




//-----------------------------------------------------------------------------
//      10ms 단위가 변할 때 현재 시간을 리턴
//      RTC가 10ms단위의 상세 시간을 리턴하기 때문
//-----------------------------------------------------------------------------
LOCAL(VOID) GetLocalTimeAlign10ms(SYSTEMTIME *ST)
    {
    UINT  ms;
    DWORD Tick;

    GetLocalTime(ST); ms=ST->wMilliseconds;
    Tick=GetTickCount();
    for (;;)
        {
        GetLocalTime(ST);
        if (ST->wMilliseconds!=ms) break;
        if (GetTickCount()-Tick>=12) break;
        }
    }




//-----------------------------------------------------------------------------
//      송신모드로 전환하고 패킷을 송신함
//-----------------------------------------------------------------------------
LOCAL(BOOL) SendPacket(RF_PACKET *P)
    {
    int Rslt=FALSE, PktLen;
    SYSTEMTIME ST;

    if (P->PacketType==PT_SWITCH && P->DataLen>=6) goto SetTime;
    else if (P->PacketType==PT_BATTERY && P->DataLen>=6 && (P->Data[0]&0x80)!=0)
        {
        SetTime:
        GetLocalTimeAlign10ms(&ST);
        Poke(P->Data+1, PackTotalSecond(&ST));
        P->Data[5]=(BYTE)(ST.wMilliseconds/10);
        CalcPacketCheckSum(P);
        }

    if (SysDebugFg!=0) PrintTxPacket(P);
    PktLen=GetPacketLenII(P);
    if ((Rslt=RFM_SendPacket((LPCBYTE)P, PktLen))!=FALSE) TxLedControl(TXLED_FLASH); else TxLedControl(OFF);
    LOG_Add(LOG_SENDPACKET, P->Receiver, P->PacketType, P->Data[0]);

    if (NominatedCommTestTime!=0 && P->PacketType==PT_STATE) RFM_SendPacket((LPCBYTE)P, PktLen);
    return Rslt;
    }




//-----------------------------------------------------------------------------
//      디버깅을 위해 표시할 패킷이름을 만듬
//-----------------------------------------------------------------------------
LOCAL(VOID) GetPacketDebugName(LPSTR Buff, RF_PACKET *P)
    {
    if (P->PacketType==PT_ROUTEDATA)
        {
        lstrcpy(Buff, "PT_ROUTEDATA (PT_");
        RFC_PacketTypeToText(GetStrLast(Buff), P->Data[0]);
        AddCha(Buff, ')');
        }
    else{
        lstrcpy(Buff, "PT_");
        RFC_PacketTypeToText(GetStrLast(Buff), P->PacketType);
        }
    }




//-----------------------------------------------------------------------------
//      ACK기다리는 시간 리턴
//-----------------------------------------------------------------------------
LOCAL(int) GetReplyTime(VOID)
    {
    return RFM_GetWaitForRfOut()*(SavedData.DeviceNo%20)+REPLY_CHECK_TIME;
    }




//-----------------------------------------------------------------------------
//      패킷 전송하는 쓰레드
//-----------------------------------------------------------------------------
#define LBTRETRYMAX     20
VOID CALLBACK RFC_ManagerTask(LPVOID lpData)
    {
    int I,J, Rssi;
    RF_PACKET *SendP, *RecvP;
    POST_RF_PACKET *PRP;
    CHAR PktName[40];
    static BYTE LBTDelta;
    static BYTE LBTStart;
    static int  LBTRetryCnt;
    static RF_PACKET *CurrP;
    static BYTE RecvPktBuff[MAXPACKETSIZE], SendPktBuff[MAXPACKETSIZE];

    if (SendWaitTimer!=0) SendWaitTimer--;
    if (SendFireDelay!=0) SendFireDelay--;

    if (lpData!=NULL)
        {
        if (lpData==RFC_SEND_ABORT)
            {
            SendWaitTimer=0;
            RetrySendCnt=0;
            CurrP=NULL;
            }
        else *(RF_PACKET**)lpData=CurrP;
        goto ProcExit;
        }

    SendP=(RF_PACKET*)SendPktBuff;
    RecvP=(RF_PACKET*)RecvPktBuff;

    if (SettingStartTime==0 && SendWaitTimer==0)
        {
        if (CurrP!=NULL)
            {
            if ((I=CheckLBT(CurrP, LBTDelta, &LBTStart))==LBT_OK)
                {
                if (SendPacket(CurrP)!=FALSE)
                    {
                    GetPacketDebugName(PktName, CurrP);
                    if (CurrP->Sender==SavedData.DeviceNo)  //내가 보내는 패킷
                        {
                        if (RFSendRecvCB!=NULL) RFSendRecvCB(RFCB_SENDOK, CurrP);
                        Printf("SENSOR: Send %s (%d->%d, Retry=%d)"CRLF, PktName, CurrP->Sender, CurrP->Receiver, RetrySendCnt);
                        }
                    else{
                        Printf("SENSOR: Send %s (Relay %d->%d)"CRLF, PktName, CurrP->Sender, CurrP->Receiver);
                        }
                    CurrP=NULL;
                    }
                }
            else{
                if (I==LBT_CANCEL)
                    {
                    SendCancel:
                    GetPacketDebugName(PktName, CurrP);
                    Printf("SENSOR: %s (LBT TimeOut, Send Canceled, Retry=%d)"CRLF, PktName, RetrySendCnt);
                    CurrP=NULL;     //LBT로직에서 계속 못보내면 소멸시킴
                    }
                else if (I==LBT_TIMEOUT)
                    {
                    I=(RetrySendCnt!=0) ? 1:LBTRETRYMAX;
                    if (++LBTRetryCnt>=I) goto SendCancel;
                    GetPacketDebugName(PktName, CurrP);
                    Printf("SENSOR: %s (LBT TimeOut, Retry=%d)"CRLF, PktName, LBTRetryCnt);
                    LBTStart=1;
                    }
                else SendWaitTimer=I;
                }
            }
        else if (RetrySendCnt!=0)           //응답없음
            {
            if (--ReplyTimer==0)
                {
                if (RFSendRecvCB) RFSendRecvCB(RFCB_NOREPLY, SendP);

                CurrP=SendP;
                ReplyTimer=GetReplyTime();
                LBTRetryCnt=0;
                LBTStart=1;
                if (RetrySendCnt==SENDRETRYCNT)
                    {
                    CurrP=NULL;
                    Printf("SENSOR: No Reply!!!"CRLF);
                    RetrySendCnt=(BYTE)-1;  //-1:아래서 0이됨
                    if (RFSendRecvCB) RFSendRecvCB(RFCB_NOACK, SendP);
                    }
                RetrySendCnt++;
                }
            }
        else{
            GettingToSendData=1;
            if ((PRP=(POST_RF_PACKET*)JOSQPend(LoRaSendQ, CHECKACCEPT, NULL))!=NULL)
                {
                LBTRetryCnt=0;
                LBTStart=1;
                SendWaitTimer=PRP->H.SendDelay;
                LBTDelta=PRP->H.LBTDelta;
                CopyMem(CurrP=SendP, &PRP->P, GetPacketLenII(&PRP->P)); FreeMem(PRP);
                if (SendP->Sender==SavedData.DeviceNo)   //내가 보내는 패킷
                    {
                    if (IsInTimePacket(SendP)==FALSE && IsNeedAckPacket(SendP)!=FALSE)
                        {
                        ReplyTimer=GetReplyTime();
                        RetrySendCnt=1;
                        NeedAckPktType=SendP->PacketType;
                        NeedAckPublishNo=SendP->PublishNo;
                        }
                    }
                GettingToSendData=0;
                if (SendWaitTimer>0) Printf("SendDelay %dms"CRLF, SendWaitTimer*10);
                }
            else{
                GettingToSendData=0;
                if (SendTimeTimer!=0 && --SendTimeTimer==0) MakeTimePacket(CurrP=SendP);    //시간패킷은 바로보내야 의미가 있으므로 큐에 넣지 않음
                else if (SendVariableTimer!=0 && --SendVariableTimer==0)
                    {
                    PutKey(KC_SendStatus);
                    SendVariableTimer=SavedData.DeviceNo*3*JOS_TICKS_PER_SEC+SendVariableInterval;
                    }
                }
            }
        }

    if ((I=RFM_RecvPacket((LPBYTE)RecvP, MAXPACKETSIZE, &Rssi))>0)
        {
        RxLedOn();

        if (SysDebugFg!=0) PrintRxPacket((LPCBYTE)RecvP, I, Rssi);

        if (SettingStartTime)
            {
            if (I<GetSettingPacketLen((LPCBYTE)RecvP))
                {
                DWORD Tick=GetTickCount();
                for (;;)
                    {
                    if ((J=RFM_RecvPacket((LPBYTE)RecvP+I, MAXPACKETSIZE-I, &Rssi))>0) {RxLedOn(); I+=J; break;}
                    if (GetTickCount()-Tick>=2000) {Printf("No Remain Data"CRLF); break;}
                    JOSTimeDly(1);
                    }
                }
            SettingProc((LPBYTE)RecvP, I);
            goto RecvEnd;
            }

        if ((J=GetPacketLenII(RecvP))>I)   //깨진 데이터임
            {
            RecvInvalid:
            if (RFSendRecvCB) RFSendRecvCB(RFCB_PKTRECV, NULL);
            goto RecvEnd;
            }

        if (RecvP->GroupNo!=SavedData.GroupNo) goto RecvInvalid;        //다른 그룹의 데이터
        if (RecvP->Sender==SavedData.DeviceNo) goto RecvInvalid;        //내가 발송한 데이터

        if (GetJ8ChkSum((LPCBYTE)RecvP, J-1)!=((LPCBYTE)RecvP)[J-1])
            {
            Printf("SENSOR: Recv Crc Err"CRLF);
            goto RecvInvalid;                                           //CRC 에러
            }

        LBT_Level=GetMax(GetMin(LBT_Level, Rssi), -120);
        GetPacketDebugName(PktName, RecvP);
        Printf("SENSOR: Recv %s (%d<-%d)"CRLF, PktName, RecvP->Receiver, RecvP->Sender);
        LOG_Add(LOG_RECVPACKET, RecvP->Sender, RecvP->PacketType, RecvP->Data[0]);

        if (RFSendRecvCB) RFSendRecvCB(RFCB_PKTRECV, RecvP);
        if (SavedData.DeviceNo!=MASTERRECEIVER)
            {   //일반 센서 및 중계기
            if (IsInSendHistory(RecvP)==FALSE)                  //한번도 보낸적이 없던 패킷인 경우 재전송
                {
                PutSendHistory(RecvP);
                if (IsToMe(RecvP)) SensorPacketProc(RecvP);
                }
            }
        }

    RecvEnd:
    if (SettingStartTime!=0 && GetTickCount()-SettingStartTime>=60000*5)
        {
        SettingStartTime=GetTickCount();    //이걸 안하면 설정모드에서 나오는 동안 키가 3~4번 눌림, SettingStartTime=0; 이걸 하면 KC_TestUp키로 해제가 안됨
        PutKey(KC_TestUp);                  //1분이 경과하면 설정모드에서 나옴
        }

    ProcExit:;
    }




//-----------------------------------------------------------------------------
//      이모듈 초기화
//-----------------------------------------------------------------------------
VOID WINAPI RFC_Init(ftRFSendRecvCB CB)
    {
    if (PowerOnReady==0)
        {
        RFSendRecvCB=CB;
        if (LoRaSendQ==NULL) LoRaSendQ=JOSQCreate(ROLASENDPACKETQTY, JOSQTYPE_DWORD);
        //JOSTaskCreateExt(RFC_ManagerTask, NULL, BASE_TASK_STK_SIZE, PRIO_SENSROLATASK, "RFM");

        Sleep(800);         //RF모듈 부팅 시간 (Spec 500ms)
        RFM_Init(SavedData.ChannelNo, SavedData.RFPower);

        G_PublishNo=GetIniInt(InternalStr, PublishNoStr, 0);
        LBT_Level=RFM_GetLbtLevel();

        PowerOnReady=1;
        }
    }



VOID WINAPI RFC_PowerOff(VOID)
    {
    PowerOnReady=0;
    #ifdef PO_RFMPOWER
    PortOut(PO_RFMPOWER, OFF);
    #endif
    }



///////////////////////////////////////////////////////////////////////////////
//                  모니터 프로그램 관련
///////////////////////////////////////////////////////////////////////////////




//-----------------------------------------------------------------------------
//      PacketType을 문자열로 변환
//-----------------------------------------------------------------------------
typedef struct _PACKET_TYPE
    {
    BYTE Type;
    CHAR Name[10];
    } PACKET_TYPE;

static CONST PACKET_TYPE PacketList[]=
    {//             "123456789"
    {PT_SENSOR,     "STATUS"},
    {PT_SWITCH,     "SWITCH"},      //긴급버튼을 눌렀습니다
    {PT_DOOR,       "DOOR"},        //문이 열렸습니다
    {PT_FIRE,       "FIRE"},        //화재감지
    {PT_HEAT,       "HEAT"},        //과열감지
    {PT_SMOKE,      "SMOKE"},       //연기감지
    {PT_HIGHTEMP,   "HIGHTEMP"},    //기온이 설정 온도보다 상승했습니다
    {PT_LOWTEMP,    "LOWTEMP"},     //기온이 설정 온도 이하로 떨어졌습니다
    {PT_HIGHCO,     "HIGHCO"},      //일산화탄소가 설정치보다 상승했습니다
    {PT_GAS,        "GAS"},         //유해가스 감지
    {PT_TRESPASS,   "TRESPASS"},    //동체감지(침입감지)
    {PT_BREAKDOWN,  "BREAKDOWN"},   //정전
    {PT_EXCEEDPOWER,"EXCEEDPOW"},   //전력사용량초과
    {PT_CRYFIRE,    "CRYFIRE"},     //불이야 외치는 소리를 감지했습니다
    {PT_CRYHELP,    "CRYHELP"},     //도와줘라고 외치는 소리를 감지했습니다
    {PT_TEMP,       "TEMP"},        //중요한 온도값
    {PT_CO2,        "CO2"},         //이산화탄소 감지
    {PT_NH3,        "NH3"},         //암모니아 감지
    {PT_BATTERY,    "BATTSTAT"},    //배터리 상태
    {PT_FIREPREDETECT,"FIRPREDET"}, //화재 징후 감지
    {PT_ELECTRICLEAK,"ELECLEAK"},   //누전 감지
    {PT_CONTROL,    "CONTROL"},     //제어
    {PT_CTLACK,     "CTLACK"},      //제어응답
    {PT_SECURITY,   "SECURITY"},    //방범설정
    {PT_SECACK,     "SECACK"},      //방범설정 응답
    {PT_SETTING,    "SETTING"},     //센서설정
    {PT_SETACK,     "SETACK"},      //센서설정 응답
    {PT_TEMPALARM,  "TEMPALARM"},   //온도 알람 설정
    {PT_TEMPACK,    "TEMPACK"},     //온도 알람 설정 응답
    {PT_LANDSLIDE,  "LANDSLIDE"},   //산사태 감지
    {PT_OXYGEN,     "OXYGEN"},      //산소량 %
    {PT_FLAME,      "FLAME"},       //불꽃감지:
    {PT_O2CONC,     "O2CONC"},      //산소농도(Concentration) 주기적 전송 (ppm단위)
    {PT_COIRTEMP,   "COIRTEMP"},    //CO농도센서1개, IR온도센서1~3
    {PT_COIRTEMPIMP,"COIRTEMPI"},   //CO농도센서1개, IR온도센서1~3 (중요한 패킷)
    {PT_SETINTERVAL,"SETINTVAL"},   //전송시간 간격 설정
    {PT_SETINTVLACK,"SETINTACK"},   //전송시간 간격 Ack
    {PT_COALARM,    "COALARM"},     //CO 알람 설정
    {PT_COALARMACK, "COALMACK"},    //CO 알람 설정 응답
    {PT_SCAN,       "SCAN"},        //통신점검
    {PT_RESET,      "RESET"},       //화재상태 클리어(화재복구)
    {PT_STATE,      "STATE"},       //통신점검/화재복구 응답
    {PT_SCANALL,    "SCANALL"},     //모든 디바이스 통신점검
    {PT_RESETALL,   "RESETALL"},    //모든 디바이스 화재복구
    {PT_STATEALL,   "STATEALL"},    //모든 디바이스 응답 중 동작상태
    {PT_BATTSTATE,  "BATTSTATE"},   //모든 디바이스 응답 중 LowBattery상태
    {PT_FIRESTATE,  "FIRESTATE"},   //모든 디바이스 응답 중 화재상태
    {PT_MANFIRE,    "MANFIRE"},     //수동화재신고
    {PT_BELL,       "BELL"},        //경종 울려라
    {PT_SBELL,      "SBELL"},       //경종 짧게 울려라
    {PT_BELLALL,    "BELLALL"},     //모든 경종 울려라
    {PT_SBELLALL,   "SBELLALL"},    //모든 경종 짧게 울려라
    {PT_TIME,       "TIME"},        //시간 설정 패킷
    {PT_ADDROUTER,  "ADDROUTER"},   //라우팅 테이블 추가
    {PT_BLOCK,      "BLOCK"},       //블럭데이터 전송
    {PT_PIRDETECT,  "PIRDETECT"},   //PIR 감지
    {PT_PIRSOS,     "PIRSOS"},      //화장실에서 쓰러짐 감지
    {PT_ELECALARM,  "ELECALARM"},   //누전감지기 알람
    {PT_ELECINFO,   "ELECINFO"},    //누전감지기 정보
    {PT_ACK,        "ACK"},         //ACK
    //{PT_DETALIVE,   "DETALIVE"},  //미니패킷으로 전송되는 감지기 ALIVE
    //{PT_DALIVEACK,  "DALIVEACK"}, //미니패킷으로 응답하는 감지기 ALIVE ACK
    {PT_FAULT,      "FAULT"},       //감지기 결함 (불꽃감지기에서 처음 사용)
    {PT_EXTINGUISH, "EXTINGUIS"},   //자체 화재 진압 완료 / 화재감지 상황종료 / 오탐 화재 였음
    {PT_COLLECTACK, "COLLACK"},     //모둠 ACK
    {PT_COLLECTACKT,"COLLACKT"},    //모둠 ACK (장기간 꺼져서 RTC시간을 잃어버린 중계기가 화재 발생한 감지기에게 RTC시간을 달라고 요청하는 모둠ACK)
    };


BOOL WINAPI RFC_PacketTypeToText(LPSTR Buff, int PacketType)
    {
    int I, Rslt=FALSE;

    for (I=0; I<countof(PacketList); I++)
        {
        if (PacketList[I].Type==PacketType) {lstrcpy(Buff, PacketList[I].Name); Rslt=TRUE; break;}
        }
    if (Rslt==FALSE) lstrcpy(Buff, "UNKNOWN");
    return Rslt;
    }


//MAIN.C에서 사용
int WINAPI Mon_Control(int PortNo, LPCSTR MonCmd, LPCSTR lpArg, LPCSTR CmdLine) {return MONRSLT_EXIT;}


