/**
  **************************************************************************************************
  * @project    Keypad
  * @file       Debug.с
  * @date       12.2016
  * @prefix     DBG_
  * @modify     Suslov Petr, Evdokimov Igor
  * @brief      Debug print function
  **************************************************************************************************
**/


//Includes
#include "Debug.h"
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <ti/drivers/UART2.h>
/* Board Header files */
#include "ti_drivers_config.h"

UART2_Handle uart;

void UART_Init()
{
    UART2_Params uartParams;

    /* Create a UART where the default read and write mode is BLOCKING */
    UART2_Params_init(&uartParams);
    uartParams.baudRate = 115200;

    uart = UART2_open(CONFIG_UART2_0, &uartParams);

    if (uart == NULL) {
        /* UART2_open() failed */
        while (1);
    }
}

/**
  *************************************************************************************************
  * @brief      Platform put buffer to UART
  *************************************************************************************************
*/
void UART_PutBuff( char* buffer, uint16_t len )
{
    //if( uart == NULL ) return;
    UART2_write(uart, buffer, len, NULL);
}


/**
  *************************************************************************************************
  * @brief      Platform put string to UART
  *************************************************************************************************
*/
void UART_PutString( const char* str )
{
    //if( uart == NULL ) return;
    UART2_write(uart, str, strlen(str), NULL);
}


/**
  **************************************************************************************************
  * @brief      Printf() realization
  **************************************************************************************************
**/
#if (!defined USE_UART_TEST_MODE) && (!(defined DeviceFamily_CC13X2) || !(defined DeviceFamily_CC13X0))
void DBG_printf( const char* fmt, ... )
{
    char buff[DEBUG_STR_LENGTH];

    va_list args;
    va_start(args, fmt);
	vsnprintf(buff, sizeof(buff), fmt, args);
/* Для использования расширенного драйвера, нужно задефайнить в проекте следующие параметры:
  #define UART_Ex 
  #define UART_Ex_DEBUG	USARTx  (x - номер юарта) */
#ifdef UART_Ex
	UARTx_PutBuff( UART_Ex_DEBUG, (uint8_t*)buff, strlen(buff) );
#else
    UART_PutBuff( (char*)buff, strlen(buff) );
#endif
    va_end(args);
}
#endif

/**
  **************************************************************************************************
  * @brief      Print HEX
  **************************************************************************************************
**/
void DBGH( char* title, void* buf, uint32_t buf_len )
{
    uint32_t i;
    char* bufc = (char*)buf;
    DBGF("%s (%03d,0x%02x):",title,buf_len,buf_len);
    for(i=0;i<buf_len;i++) DBGF("%02x ", bufc[i]);
    DBGF("\n");
}

/**
  **************************************************************************************************
  * @brief      ssprintf realization
  **************************************************************************************************
**/

#ifndef SMALL_PRINTF
char* ssprintf( const char *sfmt, ... )
{
    static char bufstring[DEBUG_STR_LENGTH];

    va_list argp;
    va_start( argp, sfmt );
    vsnprintf( bufstring, DEBUG_STR_LENGTH, sfmt, argp );
    va_end( argp );
    return bufstring;
}
#else
#define INF     32766
unsigned char Isdigit(char c)   // проверка является ли символ цифрой
{
    if((c >= 0x30) && (c <= 0x39))      return true;
    else                                return false;
}

// возвращает указатель на форматированую строку
char * ssprintf(const char *sfmt, ...)
{
    static unsigned char bufstring[DEBUG_STR_LENGTH];   // были обявлены глобально
    unsigned char buf[DEBUG_STR_LENGTH];                // были обявлены глобально // --- static

    register unsigned char *f, *bp, *str;
    register long l;
    register unsigned long u;
    register int i;
    register int fmt;
    register unsigned char pad = ' ';
    int     flush_left = 0, f_width = 0, prec = INF, hash = 0, do_long = 0;
    int     sign = 0;

    va_list ap;
    va_start(ap, sfmt);

    str = bufstring;
    f = (unsigned char *) sfmt;

    for (; *f; f++)
    {
        if (*f != '%')
        {                               // not a format character
            *str++ = (*f);          // then just output the char
        }
        else
        {
            f++;                        // if we have a "%" then skip it
            if (*f == '-')
            {
                flush_left = 1; // minus: flush left
                f++;
            }
            if (*f == '0' || *f == '.')
                {
                    // padding with 0 rather than blank
                    pad = '0';
                    f++;
            }
            if (*f == '*')
                {   // field width
                    f_width = va_arg(ap, int);
                    f++;
            }
            else if (Isdigit(*f))
                {
                    f_width = atoi((char *) f);
                    while (Isdigit(*f))
                        f++;        // skip the digits
            }
            if (*f == '.')
                {   // precision
                    f++;
                    if (*f == '*')
                    {
                        prec = va_arg(ap, int);
                        f++;
                    }
                    else if (Isdigit(*f))
                    {
                        prec = atoi((char *) f);
                        while (Isdigit(*f))
                            f++;    // skip the digits
                    }
                }
            if (*f == '#')
                {   // alternate form
                    hash = 1;
                    f++;
            }
            if (*f == 'l')
                {   // long format
                    do_long = 1;
                    f++;
            }

                fmt = *f;
                bp = buf;
                switch (fmt) {      // do the formatting
                case 'd':           // 'd' signed decimal
                    if (do_long)
                        l = va_arg(ap, long);
                    else
                        l = (long) (va_arg(ap, int));
                    if (l < 0)
                    {
                        sign = 1;
                        l = -l;
                    }
                    do  {
                        *bp++ = l % 10 + '0';
                    } while ((l /= 10) > 0);
                    if (sign)
                        *bp++ = '-';
                    f_width = f_width - (bp - buf);
                    if (!flush_left)
                        while (f_width-- > 0)
                            *str++ = (pad);
                    for (bp--; bp >= buf; bp--)
                        *str++ = (*bp);
                    if (flush_left)
                        while (f_width-- > 0)
                            *str++ = (' ');
                    break;
            case 'o':           // 'o' octal number
            case 'x':           // 'x' hex number
            case 'u':           // 'u' unsigned decimal
                    if (do_long)
                        u = va_arg(ap, unsigned long);
                    else
                        u = (unsigned long) (va_arg(ap, unsigned));
                    if (fmt == 'u')
                    {   // unsigned decimal
                        do {
                            *bp++ = u % 10 + '0';
                        } while ((u /= 10) > 0);
                    }
                    else if (fmt == 'o')
                    {  // octal
                        do {
                            *bp++ = u % 8 + '0';
                        } while ((u /= 8) > 0);
                        if (hash)
                            *bp++ = '0';
                    }
                    else if (fmt == 'x')
                    {   // hex
                        do {
                            i = u % 16;
                            if (i < 10)
                                *bp++ = i + '0';
                            else
                                *bp++ = i - 10 + 'a';
                        } while ((u /= 16) > 0);
                        if (hash)
                        {
                            *bp++ = 'x';
                            *bp++ = '0';
                        }
                    }
                    i = f_width - (bp - buf);
                    if (!flush_left)
                        while (i-- > 0)
                            *str++ = (pad);
                    for (bp--; bp >= buf; bp--)
                        *str++ = ((int) (*bp));
                    if (flush_left)
                        while (i-- > 0)
                            *str++ = (' ');
                    break;
            case 'c':           // 'c' character
                    i = va_arg(ap, int);
                    *str++ = ((int) (i));
                    break;
            case 's':           // 's' string
                    bp = va_arg(ap, unsigned char *);
                    if (!bp)
                        bp = (unsigned char *) "(nil)";
                    f_width = f_width - strlen((char *) bp);
                    if (!flush_left)
                        while (f_width-- > 0)
                            *str++ = (pad);
                    for (i = 0; *bp && i < prec; i++)
                    {
                        *str++ = (*bp);
                        bp++;
                    }
                    if (flush_left)
                        while (f_width-- > 0)
                            *str++ = (' ');
                    break;
            case '%':           // '%' character
                    *str++ = ('%');
                    break;
            }
            flush_left = 0, f_width = 0, prec = INF, hash = 0, do_long = 0;
            sign = 0;
            pad = ' ';
        }
    }

    va_end(ap);
    // terminate string with null
    *str++ = '\0';
    return (char*)bufstring;
}

#endif
