/*!
 *  @file       main.c
 *
 *  \mainpage ADCBuf_open() error
 * 
 *  @brief The sample project to demonstrate problem with ADCBuf_open() function
 *         with ARM compiler v18.x.x vs. v16.9.x
 *
 *  @author Marek Klimeš <klimes@metra-su.cz>
 *   
 *  @version 1.0.
 *              
 *  @date 07.02.2019.
 *
 *  @bug
 */
 
 
 

/* XDCtools Header files */
#include <xdc/runtime/System.h>
#include <xdc/cfg/global.h>

/* BIOS Header files */
#include <ti/sysbios/BIOS.h>
#include <ti/sysbios/knl/Task.h>
#include <ti/sysbios/knl/Clock.h>

/* TI-RTOS Header files */
//#include <ti/drivers/PIN.h>
#include <ti/drivers/pin/PINCC26xx.h>
#include <ti/drivers/Power.h>
#include <ti/drivers/power/PowerCC26XX.h>
#include <ti/drivers/ADCBuf.h>

/* Board Header files */
#include <Board.h>

/* Project header files */
#include "RFU40.h"


static PIN_Handle pinHandle;
static PIN_State pinState;


/** @brief PIN config table
 */
const PIN_Config pinTable[] = {
    Board_PIN_LED0| PIN_GPIO_OUTPUT_EN | PIN_GPIO_LOW | PIN_PUSHPULL | PIN_DRVSTR_MAX, // LED initially off
    Board_PIN_LED1| PIN_GPIO_OUTPUT_EN | PIN_GPIO_LOW | PIN_PUSHPULL | PIN_DRVSTR_MAX, // LED initially off
    PIN_TERMINATE                                                               /* Terminate list                  */
};

ADCBuf_Conversion oneShotConversion;// Structure for OneShto conversion
uint16_t sampleBufferOne[ADCBUFFERSIZE];  /* First buffer for ADC, if ONE_SHOT this is only used buffer */
uint16_t sampleBufferTwo[ADCBUFFERSIZE];  /* Second buffer for ADC, if ONE_SHOT this is not used */
uint32_t microVoltBuffer[ADCBUFFERSIZE];  /* Buffer for corrected and converted values to microVolts */


/** @brief Main function
 *
 * @return 0
 */
int main(void)
{
    /* Call board init functions */
    Board_initGeneral();
    ADCBuf_init();

    // Tasks, clocks etc. are defined in cfg file

    /* Open GPIO pins */
    pinHandle = PIN_open(&pinState, pinTable);
    if(!pinHandle) {
        System_abort("Error initializing board pins!!!\n");
    }

    // Turn all used LEDs ON
    PIN_setOutputValue(pinHandle, Board_PIN_GLED, Board_PIN_LED_ON);
    PIN_setOutputValue(pinHandle, Board_PIN_RLED, Board_PIN_LED_ON);

    /* Start BIOS */
    BIOS_start();

    return (0);
}

/* clockMeasVbatFcn()
 * Clock callback funkce pro časování měření napětí baterie
 */
void clockMeasVbatFcn(void){
    // odblokovat semafor, clock nevypínat, běží stále
    Semaphore_post(measVbatSemHandle);
}


/*
 * measVbatTaskFcn(void)
 * - task for battery voltage measurement
 * */
void measVbatTaskFcn(void){
    ADCBuf_Handle adcBuf;               // Handle to ADCBuf
    ADCBuf_Params adcBufParams;         // Structure with config. parameters
    uint32_t voltage;                   // Measured voltage on ADC0

    // Conversion setup
    // Fixed ref. voltage, input scaling enabled - set in board file
    ADCBuf_Params_init(&adcBufParams);
    adcBufParams.callbackFxn = adcBufFullCallback;
    adcBufParams.recurrenceMode = ADCBuf_RECURRENCE_MODE_ONE_SHOT;
    adcBufParams.returnMode = ADCBuf_RETURN_MODE_CALLBACK;
    adcBufParams.samplingFrequency = 1000;

    PIN_setOutputValue(pinHandle, Board_PIN_GLED, Board_PIN_LED_OFF);

    adcBuf = ADCBuf_open(Board_ADCBUF0, &adcBufParams);

    PIN_setOutputValue(pinHandle, Board_PIN_RLED, Board_PIN_LED_OFF);

    // Conversion structure setup
    oneShotConversion.arg = NULL;
    oneShotConversion.adcChannel = Board_ADCBUF0CHANNEL0;
    oneShotConversion.sampleBuffer = sampleBufferOne;
    oneShotConversion.sampleBufferTwo = NULL;
    oneShotConversion.samplesRequestedCount = ADCBUFFERSIZE;

    if (!adcBuf){
        System_abort("Error in open ADC!");
    }

    // Configure clock to unblock this task at specified time intervals
    Clock_stop(clkMeasVbatHandle);
    Clock_setTimeout(clkMeasVbatHandle, MEAS_VBAT_PERIOD);
    Clock_setPeriod(clkMeasVbatHandle, MEAS_VBAT_PERIOD);
    Clock_start(clkMeasVbatHandle);

    while (1) { //
        voltage = 0;     // If ADC fails, voltage will be 0 V

        // Conversion start in callback mode
        if (ADCBuf_convert(adcBuf, &oneShotConversion, 1) != ADCBuf_STATUS_SUCCESS) {
            System_abort("Error in ADC_conversion start!");
        }
        else { // Conversion started correctly
            // Wait for conversion done
            Semaphore_pend(adcSemHandle, BIOS_WAIT_FOREVER);

            // Adjust raw adc values and convert them to microvolts
            ADCBuf_adjustRawValues(adcBuf, &sampleBufferOne, ADCBUFFERSIZE,
                                   Board_ADCBUF0CHANNEL0);

            // Convert adjusted values to micro volts
            ADCBuf_convertAdjustedToMicroVolts(adcBuf, Board_ADCBUF0CHANNEL0,
                    &sampleBufferOne, microVoltBuffer, ADCBUFFERSIZE);

            // Use 4 values to compute the mean voltage value
            voltage = microVoltBuffer[0] + microVoltBuffer[1] + microVoltBuffer[2] +
                      microVoltBuffer[3];
            voltage = voltage >> 2; // MOD 4

            if (voltage >= 3000000) {
                PIN_setOutputValue(pinHandle, Board_PIN_GLED, Board_PIN_LED_ON);
            }
            else {
                PIN_setOutputValue(pinHandle, Board_PIN_GLED, Board_PIN_LED_OFF);
            }

        }

        Semaphore_pend(measVbatSemHandle, BIOS_WAIT_FOREVER);    // Wait to next measurement

    } // while(1);

} /* measVbatTaskFcn */


/*
 * adcBufFullCallback(...)
 * ADC callback function
 */
void adcBufFullCallback(ADCBuf_Handle adcHandle, ADCBuf_Conversion *conversionStruct,
    void *completedADCBuffer, uint32_t completedChannel) {

    // unblock the task
    Semaphore_post(adcSemHandle);
}











