/*
 * Copyright (c) 2015-2016, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 *  ======== rfEasyLinkRx.c ========
 */
/* XDCtools Header files */
#include <xdc/std.h>
#include <xdc/runtime/System.h>
#include <xdc/runtime/Error.h>

/* BIOS Header files */
#include <ti/sysbios/BIOS.h>
#include <ti/sysbios/knl/Task.h>
#include <ti/sysbios/knl/Semaphore.h>
#include <ti/sysbios/knl/Clock.h>
#include <ti/sysbios/knl/Mailbox.h>

#include <ti/devices/cc13x0/driverlib/sys_ctrl.h>

/* TI-RTOS Header files */
#include <ti/drivers/PIN.h>

/* Board Header files */
#include "Board.h"

/* EasyLink API Header files */
#include "easylink/EasyLink.h"


#include "sys_intf.h"
#include "version.h"

#include <stdlib.h>
#include <string.h>

/***** Defines *****/

/* Undefine to remove address filter and async mode */
//#define RFEASYLINKRX_ASYNC
#define RFEASYLINKRX_ADDR_FILTER

#define RFEASYLINKEX_TASK_STACK_SIZE 1024
#define RFEASYLINKEX_TASK_PRIORITY   2

/* Pin driver handle */
//static PIN_Handle ledPinHandle;
//static PIN_State ledPinState;

/*
 * Application LED pin configuration table:
 *   - All LEDs board LEDs are off.
 */
//PIN_Config pinTable[] = {
//    CC1310_LAUNCHXL_PIN_LED0 | PIN_GPIO_OUTPUT_EN | PIN_GPIO_LOW | PIN_OPENDRAIN | PIN_DRVSTR_MAX,
//    CC1310_LAUNCHXL_PIN_LED1 | PIN_GPIO_OUTPUT_EN | PIN_GPIO_LOW | PIN_OPENDRAIN | PIN_DRVSTR_MAX,
//    CC1310_LAUNCHXL_PIN_LED2 | PIN_GPIO_OUTPUT_EN | PIN_GPIO_LOW | PIN_OPENDRAIN | PIN_DRVSTR_MAX,
//    PIN_TERMINATE
//};

/***** Variable declarations *****/
static Task_Params rxTaskParams;
Task_Struct rxTask;    /* not static so you can see in ROV */
static uint8_t rxTaskStack[RFEASYLINKEX_TASK_STACK_SIZE];

static Task_Params uartRxTaskParams;
Task_Struct uartRxTask;    /* not static so you can see in ROV */
static uint8_t uartRxTaskStack[RFEASYLINKEX_TASK_STACK_SIZE];

static Task_Params uartTxTaskParams;
Task_Struct uartTxTask;    /* not static so you can see in ROV */
static uint8_t uartTxTaskStack[RFEASYLINKEX_TASK_STACK_SIZE];

static Mailbox_Handle mbox0;
static Mailbox_Params mbox0Params;


/* The RX Output struct contains statistics about the RX operation of the radio */
PIN_Handle pinHandle;

#ifdef RFEASYLINKRX_ASYNC
static Semaphore_Handle rxDoneSem;
#endif

/***** Function definitions *****/
void process_optosonde_2017(EasyLink_RxPacket* rxPacket) {
    optosonde_sys_intf_t data;
    memset(&data, 0, sizeof(data));
    data.magic_byte = OPTOSONDE_SYS_INTF_MAGIC_BYTE;
    data.len = sizeof(data);
    data.version = 0;
    data.gw_fw_version = VERSION_HASH;

    memcpy(data.node_addr, rxPacket->payload + 1, 8);

    data.reason = rxPacket->payload[9];
    data.temp_deg = (rxPacket->payload[10] << 8) | rxPacket->payload[11];
    data.bat_mv = rxPacket->payload[12] * 20;
    memcpy(&data.node_fw_version, rxPacket->payload + 13, sizeof(data.node_fw_version));
    data.rssi = rxPacket->rssi;

    Mailbox_post(mbox0, &data, BIOS_NO_WAIT);

}

#ifdef RFEASYLINKRX_ASYNC
void rxDoneCb(EasyLink_RxPacket * rxPacket, EasyLink_Status status)
{
    if (status == EasyLink_Status_Success)
    {
        /* Toggle LED2 to indicate RX */
//        PIN_setOutputValue(pinHandle, Board_PIN_LED2,!PIN_getOutputValue(Board_PIN_LED2));
        if (rxPacket->payload[0] == 0) {
            process_optosonde_2017(rxPacket);
        } else {
            // Unknown payload version
            /* Toggle LED1 and LED2 to indicate error */
//            PIN_setOutputValue(pinHandle, Board_PIN_LED1,!PIN_getOutputValue(Board_PIN_LED1));
//            PIN_setOutputValue(pinHandle, Board_PIN_LED2,!PIN_getOutputValue(Board_PIN_LED2));
        }
    }
    else if(status == EasyLink_Status_Aborted)
    {
        /* Toggle LED1 to indicate command aborted */
//        PIN_setOutputValue(pinHandle, Board_PIN_LED1,!PIN_getOutputValue(Board_PIN_LED1));
    }
    else
    {
        /* Toggle LED1 and LED2 to indicate error */
//        PIN_setOutputValue(pinHandle, Board_PIN_LED1,!PIN_getOutputValue(Board_PIN_LED1));
//        PIN_setOutputValue(pinHandle, Board_PIN_LED2,!PIN_getOutputValue(Board_PIN_LED2));
    }

    Semaphore_post(rxDoneSem);
}
#endif

static void rfEasyLinkRxFnx(UArg arg0, UArg arg1)
{
#ifndef RFEASYLINKRX_ASYNC
    static EasyLink_RxPacket rxPacket = {0};
#endif

#ifdef RFEASYLINKRX_ASYNC
    /* Create a semaphore for Async*/
    Semaphore_Params params;
    Error_Block eb;

    /* Init params */
    Semaphore_Params_init(&params);
    Error_init(&eb);

    /* Create semaphore instance */
    rxDoneSem = Semaphore_create(0, &params, &eb);
#endif //RFEASYLINKRX_ASYNC

//    EasyLink_init(EasyLink_Phy_625bpsLrm);
//    EasyLink_init(EasyLink_Phy_5kbpsSlLr);
    EasyLink_init(EasyLink_Phy_Custom);

    /*
     * If you wish to use a frequency other than the default, use
     * the following API:
     * EasyLink_setFrequency(868000000);
     */

#ifdef RFEASYLINKRX_ADDR_FILTER
    uint8_t addrFilter = OPTOSONDE_ADDR;
    EasyLink_enableRxAddrFilter(&addrFilter, 1, 1);
#endif //RFEASYLINKRX_ADDR_FILTER

    GPIO_setOutputEnableDio(CC1310_LAUNCHXL_PIN_LED2, true);

    while (true) {
#ifdef RFEASYLINKRX_ASYNC
        EasyLink_receiveAsync(rxDoneCb, 0);

        /* Wait for Rx */
        if(!Semaphore_pend(rxDoneSem, BIOS_WAIT_FOREVER)) {
        }
#else
        memset(&rxPacket, 0, sizeof(rxPacket));
        rxPacket.absTime = 0;
        rxPacket.rxTimeout = 0;
        EasyLink_Status result = EasyLink_receive(&rxPacket);

        if (result == EasyLink_Status_Success)
        {
            GPIO_toggleDio(CC1310_LAUNCHXL_PIN_LED2);
            if (rxPacket.payload[0] == 0) {
                process_optosonde_2017(&rxPacket);
            } else {
                // Unknown payload version
            }
        }
        else
        {
            /* Toggle LED1 and LED2 to indicate error */
//            PIN_setOutputValue(pinHandle, Board_PIN_LED1,!PIN_getOutputValue(Board_PIN_LED1));
//            PIN_setOutputValue(pinHandle, Board_PIN_LED2,!PIN_getOutputValue(Board_PIN_LED2));
        }
#endif //RX_ASYNC
    }
}

static UART_Handle uart;

void uartRxFxn(UArg arg0, UArg arg1) {
    static optosonde_sys_intf_t  data = {0};

    while (true) {
        UART_read(uart, &data, sizeof(data));

        if (data.magic_byte == OPTOSONDE_SYS_INTF_MAGIC_BYTE) {
            if (data.reason == optosonde_reason_reset) {
                SysCtrlSystemReset();
            }
        }

    }
}

/*
 *  ======== echoFxn ========
 *  Task for this function is created statically. See the project's .cfg file.
 */
void uartTxFxn(UArg arg0, UArg arg1)
{
    UART_Params uartParams;
    UART_init();
    /* Create a UART with data processing off. */
    UART_Params_init(&uartParams);
    uartParams.writeDataMode = UART_DATA_BINARY;
    uartParams.readDataMode = UART_DATA_BINARY;
    uartParams.readReturnMode = UART_RETURN_FULL;
    uartParams.readEcho = UART_ECHO_OFF;
    uartParams.baudRate = 115200;
    uartParams.readMode = UART_MODE_BLOCKING;
    uart = UART_open(Board_UART0, &uartParams);

    if (uart == NULL) {
        System_abort("Error opening the UART");
    }

    Task_construct(&uartRxTask, uartRxFxn, &uartRxTaskParams, NULL);

    static optosonde_sys_intf_t  data;

    GPIO_setOutputEnableDio(CC1310_LAUNCHXL_PIN_LED1, true);

    data.magic_byte = OPTOSONDE_SYS_INTF_MAGIC_BYTE;
    data.reason = 0xFE;
    UART_write(uart, &data, sizeof(optosonde_sys_intf_t));

    while (true) {
        GPIO_setDio(CC1310_LAUNCHXL_PIN_LED1);
        if (Mailbox_pend(mbox0, &data, (30 * 1000 * 1000) / Clock_tickPeriod)) {
            GPIO_clearDio(CC1310_LAUNCHXL_PIN_LED1);

        } else {
            memset(&data, 0, sizeof(data));
            data.magic_byte = OPTOSONDE_SYS_INTF_MAGIC_BYTE;
            data.reason = optosonde_reason_watchdog;
            data.len = sizeof(data);
        }

        UART_write(uart, &data, sizeof(optosonde_sys_intf_t));
    }
}

/*
 *  ======== main ========
 */
int main(void)
{
    /* Call driver init functions */
    Board_initGeneral();

    GPIO_setOutputEnableDio(CC1310_LAUNCHXL_PIN_LED0, true);
    GPIO_clearDio(CC1310_LAUNCHXL_PIN_LED0);

    Mailbox_Params_init(&mbox0Params);
    mbox0 = Mailbox_create(sizeof(optosonde_sys_intf_t), 10, &mbox0Params, NULL);

    Task_Params_init(&rxTaskParams);
    rxTaskParams.stackSize = RFEASYLINKEX_TASK_STACK_SIZE;
    rxTaskParams.priority = RFEASYLINKEX_TASK_PRIORITY;
    rxTaskParams.stack = &rxTaskStack;
    rxTaskParams.arg0 = (UInt)1000000;

    Task_construct(&rxTask, rfEasyLinkRxFnx, &rxTaskParams, NULL);

    // UART TX Task
    Task_Params_init(&uartTxTaskParams);
    uartTxTaskParams.stackSize = RFEASYLINKEX_TASK_STACK_SIZE;
    uartTxTaskParams.priority = RFEASYLINKEX_TASK_PRIORITY + 1;
    uartTxTaskParams.stack = &uartTxTaskStack;
    uartTxTaskParams.arg0 = (UInt)1000000;
    Task_construct(&uartTxTask, uartTxFxn, &uartTxTaskParams, NULL);

    // UART RX Task
    Task_Params_init(&uartRxTaskParams);
    uartRxTaskParams.stackSize = RFEASYLINKEX_TASK_STACK_SIZE;
    uartRxTaskParams.priority = RFEASYLINKEX_TASK_PRIORITY + 1;
    uartRxTaskParams.stack = &uartRxTaskStack;
    uartRxTaskParams.arg0 = (UInt)1000000;

    /* Start BIOS */
    BIOS_start();

    return (0);
}
