/*
 * oad_image_header.h
 *
 *  Created on: Mar 16, 2022
 *      Author: system
 */

#ifndef OAD_NATIVE_OAD_OAD_IMAGE_HEADER_H_
#define OAD_NATIVE_OAD_OAD_IMAGE_HEADER_H_

/*******************************************************************************
 * INCLUDES
 ******************************************************************************/
#include <stddef.h>
#include <stdint.h>

/*******************************************************************************
 * CONSTANTS
 ******************************************************************************/


#define OAD_IMG_FULL_HDR_LEN         sizeof(imgHdr_t)                     //!< The length of the OAD Header + Segments
#define OAD_IMG_ID_LEN               8                                    //!< OAD image identification bytes length
#define OAD_SW_VER_LEN               4                                    //!< OAD software version length in bytes


#define OAD_IMG_ID_VAL                    {'C', 'C', '1', '3', 'x', '2', 'R', '1'}

/*!
 * Magic number used by entries in the image header table at the beginning of
 * external flash. Note this is different from @ref OAD_IMG_ID_VAL so that
 * the application can determine the difference between an entry in the header
 * table and the start of an OAD image
 *
 * \note This is only used by off-chip OAD
 */
#define OAD_EXTFL_ID_VAL                  {'O', 'A', 'D', ' ', 'N', 'V', 'M', '1'}

/*!
 * Length of OAD image ID field, this is the length of @ref OAD_IMG_ID_VAL and
 * @ref OAD_EXTFL_ID_VAL
 */
#define OAD_IMG_ID_LEN               8

/*!
 * Offset in bytes of imgType into the structure @ref imgFixedHdr_t
 */
#define IMG_TYPE_OFFSET              offsetof(imgHdr_t, fixedHdr.imgType)

/*!
 * OAD on-chip persistent application image
 *
 * The persistent app is the permanently resident app used in on-chip OAD
 * The persistent app is responsible for updating the user application
 */
#define OAD_IMG_TYPE_PERSISTENT_APP       0

/*!
 * OAD network processor image
 *
 * A network processor image. This image is intended to run an external
 * processor
 */
#define OAD_IMG_TYPE_FACTORY              5

/*!
 * OAD Boot Image Manager (BIM) image
 *
 * A BIM image
 * \note BIM upgrade is not currently supported
 */
#define OAD_IMG_TYPE_BIM                  6

/*!
 * OAD reserved image type begin
 *
 * \warning These fields are RFU
 */
#define OAD_IMG_TYPE_RSVD_BEGIN           64

/*!
 * Offset in bytes of crc32 into the structure @ref imgFixedHdr_t
 */
#define CRC_OFFSET                   offsetof(imgHdr_t, fixedHdr.crc32)
/*!
 * Offset in bytes of CRC status into the structure @ref imgFixedHdr_t
 */
#define CRC_STAT_OFFSET              offsetof(imgHdr_t, fixedHdr.crcStat)

/*!
 * Flag for crcStat indicating the CRC of the image is
 * valid
 */
#define CRC_VALID                    0xFE

/*!
 * Flag for crcStat indicating the CRC of the image is
 * invalid
 */
#define CRC_INVALID                  0xFC

/*!
 * Offset in bytes of bimVer into the structure
 * @ref imgFixedHdr_t. This is the start of data after the CRC bytes
 */
#define IMG_DATA_OFFSET              BIM_VER_OFFSET

/*!
 * Offset in bytes of bimVer into the structure @ref imgFixedHdr_t
 */
#define BIM_VER_OFFSET               offsetof(imgHdr_t, fixedHdr.bimVer)

/*******************************************************************************
 * Typedefs
 */

/// @cond NODOC
#if defined (__IAR_SYSTEMS_ICC__)
  #define TYPEDEF_STRUCT_PACKED        __packed typedef struct
#elif defined __TI_COMPILER_VERSION || defined __TI_COMPILER_VERSION__ || __clang__
  #define TYPEDEF_STRUCT_PACKED        typedef struct __attribute__((packed))
#elif defined (__GNUC__)
  #define TYPEDEF_STRUCT_PACKED        typedef struct __attribute__((__packed__))
#endif
/// @endcond // NODOC

/*!
 * OAD Core image header structure definition
 * \note This header is required to be part of all OAD images
 */
TYPEDEF_STRUCT_PACKED
{
  uint8_t   imgID[8];       //!< User-defined Image Identification bytes
  uint32_t  crc32;          //!< Image's 32-bit CRC value
  uint8_t   bimVer;         //!< BIM version
  uint8_t   metaVer;        //!< Metadata version
  uint16_t  techType;       //!< Wireless protocol type BLE/TI-MAC/ZIGBEE etc.
  uint8_t   imgCpStat;      //!< Image copy status
  uint8_t   crcStat;        //!< CRC status
  uint8_t   imgType;        //!< Image Type
  uint8_t   imgNo;          //!< Image number of 'image type'
  uint32_t  imgVld;         //!< Image validation bytes, used by BIM.
  uint32_t  len;            //!< Image length in bytes.
  uint32_t  prgEntry;       //!< Program entry address
  uint8_t   softVer[4];     //!< Software version of the image
  uint32_t  imgEndAddr;     //!< Address of the last byte of a contiguous image
  uint16_t  hdrLen;         //!< Total length of the image header
  uint16_t  rfu;            //!< Reserved bytes
}imgFixedHdr_t;

/*!
 * OAD continuous image payload segment
 *
 * This segment is used to describe a contiguous region of an image
 * The code data in this blob will be flashed to sequential addresses
 */
TYPEDEF_STRUCT_PACKED
{
  uint8_t   segTypeImg;     //!< Segment type - for Contiguous image payload
  uint16_t  wirelessTech;   //!< Wireless technology type
  uint8_t   rfu;            //!< Reserved byte
  uint32_t  imgSegLen;      //!< Payload segment length
  uint32_t  startAddr;      //!< Start address of image on internal flash
}imgPayloadSeg_t;
/*!
 * OAD Image Header
 *
 * This structure represents the complete header to be prepended to an OAD image
 *
 * It is an amalgamation of the structures defined above, and must contain a
 * core header
 */
TYPEDEF_STRUCT_PACKED
{
   imgFixedHdr_t         fixedHdr;    //!< Required core image header
#if (defined(SECURITY))
   securityInfoSeg_t     secInfoSeg;  //!< Security information segment
#endif
#if (!defined(STACK_LIBRARY) && (defined(SPLIT_APP_STACK_IMAGE)))
   boundarySeg_t         boundarySeg; //!< Optional boundary segment
#endif
   imgPayloadSeg_t       imgPayload;  //!< Required contiguous image segment
} imgHdr_t;

#endif /* OAD_NATIVE_OAD_OAD_IMAGE_HEADER_H_ */
