﻿///////////////////////////////////////////////////////////////////////////////
//                      STM32F0XX Driver
//
// 시작일: 2016-06-23 ~
// 프로그래머: 정오장
///////////////////////////////////////////////////////////////////////////////


#include "JLIB.H"
#include "DRIVER.H"
#include "JOS.H"


#define RTCDEVIATION_DEBUG      1           //시간동기 할 때 현재가지고 있던 시간의 편파를 표시함. 양수이면 빠른 것이고 음수이면 느린 것임
#define SWFLOWCONTROL           0
//#define BAUD_4800_SUPPORT



__IO DWORD FineTickCnt; //JOS에서 사용함
static BYTE DebugPort=COM_DEBUG;
VOID WINAPI _1msProc(VOID);
VOID WINAPI UART_ErrUsrHandler(int Port, int ErrCode);

VOID WINAPI SetPendSV(VOID) {SCB->ICSR|=SCB_ICSR_PENDSVSET_Msk;}        //Bit28
//VOID WINAPI ClearPendSV(VOID) {SCB->ICSR|=SCB_ICSR_PENDSVCLR_Msk;}    //Bit27



//#define MODIFYBIT(Reg, ClearMask, SetMask) {Reg=(Reg & ~(ClearMask)) | (SetMask);}    //이거보다 아래가 코드가 짧음
#define MODIFYBIT(Reg, Clear, Set) {UINT R=Reg & ~(Clear); R|=Set; Reg=R;}
#define BITSET(Reg, Bit, OnOff)    {UINT R=Reg; R&=~(Bit); if (OnOff) R|=Bit; Reg=R;}



VOID WINAPI Delay_us(int T)
    {
    volatile int I;
    for (I=0; I<T; I++);
    }


//-----------------------------------------------------------------------------
//      명령어 반복만으로 ms단위 지연을 만듬
//-----------------------------------------------------------------------------
VOID WINAPI Delay1ms(UINT ms)
    {
    volatile UINT Cnt;
    while (ms--) {Cnt=6820; while (Cnt--);}
    }




HAL_StatusTypeDef HAL_InitTick(UINT TickPriority)
    {
    SysTick_Config(HAL_RCC_GetHCLKFreq()/1000);         //SysTick_Config()은 CORE_CM0.H안에 있는 함수임
    NVIC_SetPriority(SysTick_IRQn, TickPriority);
    return HAL_OK;
    }




HAL_StatusTypeDef HAL_Init(VOID)
    {
    #if (PREFETCH_ENABLE!=0)
    __HAL_FLASH_PREFETCH_BUFFER_ENABLE();
    #endif

    HAL_InitTick(TICK_INT_PRIORITY);
    return HAL_OK;
    }



VOID HAL_Delay(volatile UINT Delay)
    {
    DWORD Time=HAL_GetTick();
    while ((HAL_GetTick()-Time)<Delay);
    }


UINT HAL_GetTick(VOID)
    {
    if (FineTickCnt==0) return 1;   //플래그로 쓰기 때문
    return FineTickCnt;
    }




#ifdef STM32F042x6
//-----------------------------------------------------------------------------
//         The system Clock is configured as follow
//-----------------------------------------------------------------------------
VOID WINAPI InitSystemClock(VOID)
    {
    RCC_ClkInitTypeDef CLKI;
    RCC_OscInitTypeDef OSCI;

    //Enable HSE Oscillator and Activate PLL with HSE as source
    OSCI.OscillatorType=RCC_OSCILLATORTYPE_NONE;
    OSCI.PLL.PLLState=RCC_PLL_ON;
    OSCI.PLL.PLLSource=RCC_PLLSOURCE_HSI;
    OSCI.PLL.PREDIV=RCC_PREDIV_DIV1;
    OSCI.PLL.PLLMUL=RCC_PLL_MUL6;
    if (HAL_RCC_OscConfig(&OSCI)!=HAL_OK) FatalError(FERR_SYSCLKCFG);

    //Select PLL as system clock source and configure the HCLK, PCLK1 clocks dividers
    CLKI.ClockType=(RCC_CLOCKTYPE_SYSCLK|RCC_CLOCKTYPE_HCLK|RCC_CLOCKTYPE_PCLK1);
    CLKI.SYSCLKSource=RCC_SYSCLKSOURCE_PLLCLK;
    CLKI.AHBCLKDivider=RCC_SYSCLK_DIV1;
    CLKI.APB1CLKDivider=RCC_HCLK_DIV1;
    if (HAL_RCC_ClockConfig(&CLKI, FLASH_LATENCY_1)!=HAL_OK) FatalError(FERR_SYSCLKCFG);
    }
#endif



#ifdef STM32F091xC
//-----------------------------------------------------------------------------
//         The system Clock is configured as follow :
//
//            System Clock source            = PLL (HSE)
//            SYSCLK(Hz)                     = 48000000
//            HCLK(Hz)                       = 48000000
//            AHB Prescaler                  = 1
//            APB1 Prescaler                 = 1
//            HSE Frequency(Hz)              = 8000000
//            PREDIV                         = 1
//            PLLMUL                         = 6
//            Flash Latency(WS)              = 1
//-----------------------------------------------------------------------------
VOID WINAPI InitSystemClock(VOID)
    {
    RCC_ClkInitTypeDef CLKI;
    RCC_OscInitTypeDef OSCI;

    //Enable HSE Oscillator and Activate PLL with HSE as source
    OSCI.OscillatorType=RCC_OSCILLATORTYPE_HSE;
    OSCI.HSEState=RCC_HSE_ON;
    OSCI.PLL.PLLState=RCC_PLL_ON;
    OSCI.PLL.PLLSource=RCC_PLLSOURCE_HSE;   //=8000000
    OSCI.PLL.PREDIV=RCC_PREDIV_DIV1;
    OSCI.PLL.PLLMUL=RCC_PLL_MUL6;           //8000000/1*6 = 48M
    if (HAL_RCC_OscConfig(&OSCI)!=HAL_OK) FatalError(FERR_SYSCLKCFG);

    //Select HSI48 Oscillator as PLL source
    OSCI.OscillatorType=RCC_OSCILLATORTYPE_HSI48;
    OSCI.HSI48State=RCC_HSI48_ON;
    OSCI.PLL.PLLState=RCC_PLL_ON;
    OSCI.PLL.PLLSource=RCC_PLLSOURCE_HSI48;
    OSCI.PLL.PREDIV=RCC_PREDIV_DIV2;
    OSCI.PLL.PLLMUL=RCC_PLL_MUL2;
    if (HAL_RCC_OscConfig(&OSCI)!=HAL_OK) FatalError(FERR_SYSCLKCFG);

    //Select PLL as system clock source and configure the HCLK, PCLK1 clocks dividers
    CLKI.ClockType=RCC_CLOCKTYPE_SYSCLK|RCC_CLOCKTYPE_HCLK|RCC_CLOCKTYPE_PCLK1;
    CLKI.SYSCLKSource=RCC_SYSCLKSOURCE_PLLCLK;
    CLKI.AHBCLKDivider=RCC_SYSCLK_DIV1;
    CLKI.APB1CLKDivider=RCC_HCLK_DIV1;
    if (HAL_RCC_ClockConfig(&CLKI, FLASH_LATENCY_1)!=HAL_OK) FatalError(FERR_SYSCLKCFG);
    }
#endif




///////////////////////////////////////////////////////////////////////////////
//                              GPIO
///////////////////////////////////////////////////////////////////////////////


//-----------------------------------------------------------------------------
//      GPIO Port Base를 얻음
//-----------------------------------------------------------------------------
LOCAL(GPIO_TypeDef*) GetGpioBase(int PinNo)
    {
    GPIO_TypeDef *GB=NULL;

    switch (PinNo>>4)
        {
        case 0: GB=GPIOA; break;
        case 1: GB=GPIOB; break;
        case 2: GB=GPIOC; break;
        #if defined(STM32F091xC) || defined(STM32F072xB)
        case 3: GB=GPIOD; break;
        case 4: GB=GPIOE; break;
        #endif
        case 5: GB=GPIOF; break;
        }
    return GB;
    }



//-----------------------------------------------------------------------------
//      GPIO Pin 초기화
//-----------------------------------------------------------------------------
#define GPIO_MODE             0x00000003
#define RISING_EDGE           0x00100000
#define FALLING_EDGE          0x00200000
#define GPIO_MODE_IT          0x00010000
#define EXTI_MODE             0x10000000
#define GPIO_OUTPUT_TYPE      0x00000010
#define GPIO_MODE_EVT         0x00020000
VOID HAL_GPIO_Init(GPIO_TypeDef *GPIOx, GPIO_InitTypeDef *GI)
    {
    UINT Shift, BitPos, CurrBit;

    for (BitPos=0; GI->Pin>>BitPos; BitPos++)
        {
        if ((CurrBit=GI->Pin & (1<<BitPos))!=0)
            {
            if (GI->Mode==GPIO_MODE_AF_PP || GI->Mode==GPIO_MODE_AF_OD)
                {
                //GPIO_PinAFConfig()
                Shift=(BitPos&7)<<2;
                MODIFYBIT(GPIOx->AFR[BitPos>>3], 0x0F<<Shift, GI->Alternate << Shift);
                }

            MODIFYBIT(GPIOx->MODER, GPIO_MODER_MODER0<<BitPos*2, (GI->Mode&GPIO_MODE)<<BitPos*2);

            if (GI->Mode==GPIO_MODE_OUTPUT_PP || GI->Mode==GPIO_MODE_AF_PP ||
                GI->Mode==GPIO_MODE_OUTPUT_OD || GI->Mode==GPIO_MODE_AF_OD)
                {
                MODIFYBIT(GPIOx->OSPEEDR, GPIO_OSPEEDER_OSPEEDR0<<BitPos*2, GI->Speed<<BitPos*2);
                MODIFYBIT(GPIOx->OTYPER, GPIO_OTYPER_OT_0<<BitPos, ((GI->Mode&GPIO_OUTPUT_TYPE)>>4)<<BitPos);
                }

            MODIFYBIT(GPIOx->PUPDR, GPIO_PUPDR_PUPDR0<<BitPos*2, (GI->Pull)<<BitPos*2);

            if (GI->Mode & EXTI_MODE)
                {
                __HAL_RCC_SYSCFG_CLK_ENABLE();

                Shift=(BitPos&3)<<2;
                MODIFYBIT(SYSCFG->EXTICR[BitPos>>2], 0x0F<<Shift, GPIO_GET_INDEX(GPIOx)<<Shift);

                BITSET(EXTI->IMR,  CurrBit, GI->Mode & GPIO_MODE_IT);
                BITSET(EXTI->EMR,  CurrBit, GI->Mode & GPIO_MODE_EVT);
                BITSET(EXTI->RTSR, CurrBit, GI->Mode & RISING_EDGE);
                BITSET(EXTI->FTSR, CurrBit, GI->Mode & FALLING_EDGE);
                }
            }
        }
    }




//-----------------------------------------------------------------------------
//      GPIO Pin을 리셋 상태로 되돌림
//-----------------------------------------------------------------------------
VOID WINAPI GPIO_DeInit(int PinNo)
    {
    UINT T, Pin, Shift, BitPos, CurrBit;
    GPIO_TypeDef *GB;

    if ((GB=GetGpioBase(PinNo))==NULL) goto ProcExit;

    Pin=1<<(PinNo & 0x1F);
    for (BitPos=0; Pin>>BitPos; BitPos++)
        {
        if ((CurrBit=Pin & (1<<BitPos))!=0)
            {
            CLEAR_BIT(GB->MODER, GPIO_MODER_MODER0<<(BitPos<<1));
            CLEAR_BIT(GB->AFR[BitPos>>3], 0x0F<<((BitPos&7)<<2));
            CLEAR_BIT(GB->OSPEEDR, GPIO_OSPEEDER_OSPEEDR0<<(BitPos<<1));
            CLEAR_BIT(GB->OTYPER, GPIO_OTYPER_OT_0<<BitPos);
            CLEAR_BIT(GB->PUPDR, GPIO_PUPDR_PUPDR0<<(BitPos<<1));

            Shift=(BitPos&3)<<2;
            T=SYSCFG->EXTICR[BitPos>>2] & (0x0F<<Shift);
            if (GPIO_GET_INDEX(GB)<<Shift==T)
                {
                CLEAR_BIT(SYSCFG->EXTICR[BitPos>>2], 0x0F<<Shift);
                CLEAR_BIT(EXTI->IMR, CurrBit);
                CLEAR_BIT(EXTI->EMR, CurrBit);
                CLEAR_BIT(EXTI->RTSR, CurrBit);
                CLEAR_BIT(EXTI->FTSR, CurrBit);
                }
            }
        }
    ProcExit:;
    }




//-----------------------------------------------------------------------------
//      Init GPIO
//-----------------------------------------------------------------------------
VOID WINAPI InitPortEx(int PinNo, int PinMode, int PullUpDown, int AltFnc, int Speed)
    {
    int Port, Dummy;
    GPIO_InitTypeDef GITD;
    GPIO_TypeDef     *GB;

    Port=PinNo>>4;
    if ((GB=GetGpioBase(PinNo))!=NULL)
        {
        //해당 GPIO Clock Enable
        RCC->AHBENR|=RCC_AHBENR_GPIOAEN<<Port;
        Dummy=RCC->AHBENR & (RCC_AHBENR_GPIOAEN<<Port); //Delay after an RCC peripheral clock enabling
        UNUSED(Dummy);

        ZeroMem(&GITD, sizeof(GPIO_InitTypeDef));
        GITD.Mode=PinMode;
        GITD.Pull=PullUpDown;
        GITD.Speed=Speed;
        GITD.Pin=1<<(PinNo & 0x0F);
        GITD.Alternate=AltFnc;
        HAL_GPIO_Init(GB, &GITD);
        }
    }

VOID WINAPI InitPort(int PinNo, int PinMode, int PullUpDown, int AltFnc)
    {
    InitPortEx(PinNo, PinMode, PullUpDown, AltFnc, GPIO_SPEED_FREQ_HIGH);
    }

VOID WINAPI InitPortOutputPP(int PinNo)
    {
    InitPort(PinNo, GPIO_MODE_OUTPUT_PP, GPIO_NOPULL, 0);
    }

VOID WINAPI InitPortOutputOD(int PinNo)
    {
    InitPort(PinNo, GPIO_MODE_OUTPUT_OD, GPIO_NOPULL, 0);
    }

VOID WINAPI InitPortInput(int PinNo, int PullUpDown)
    {
    InitPort(PinNo, GPIO_MODE_INPUT, PullUpDown, 0);
    }



//-----------------------------------------------------------------------------
//      GPIO Out
//-----------------------------------------------------------------------------
BOOL WINAPI PortOut(int PinNo, int PinData)
    {
    int PinBit, Rslt=0, BitNo;
    GPIO_TypeDef *GB;

    if (PinNo<I2CPORT)
        {
        BitNo=PinNo & 0x0F;
        PinBit=1<<BitNo;
        if ((GB=GetGpioBase(PinNo))!=NULL)
            {
            JOS_ENTER_CRITICAL();
            switch (PinData)
                {
                case LOW: GB->BRR=PinBit; break;        //HAL_GPIO_WritePin(GB, PinBit, (GPIO_PinState)PinData);
                case HIGH: GB->BSRR=PinBit; break;
                case PO_NOT: GB->ODR^=PinBit; break;    //HAL_GPIO_TogglePin(GB, PinBit);
                case PO_GETSTATE: Rslt=(GB->ODR >> BitNo) & 1;
                }
            JOS_EXIT_CRITICAL();
            }
        }
    return Rslt;
    }



BOOL WINAPI PortIn(int PinNo)
    {
    BOOL Rslt=FALSE;
    GPIO_TypeDef *GB;

    if (PinNo<I2CPORT)
        {
        if ((GB=GetGpioBase(PinNo))!=NULL)
            Rslt=(GB->IDR & (1<<(PinNo & 0x0F)))!=0;
        }

    return Rslt;
    }

DWORD WINAPI PortAIn(VOID) {return GPIOA->IDR;}
DWORD WINAPI PortBIn(VOID) {return GPIOB->IDR;}

VOID WINAPI PortAByteOut(int Data)
    {
    GPIOA->BRR=0xFF;
    GPIOA->BSRR=Data & 0xFF;
    }

VOID WINAPI PortBByteOut(int Data)
    {
    GPIOB->BRR=0xFF;
    GPIOB->BSRR=Data & 0xFF;
    }

VOID WINAPI PortBHighByteOut(int Data)
    {
    GPIOB->BRR=0xFF00;
    GPIOB->BSRR=(Data & 0xFF)<<8;
    }



///////////////////////////////////////////////////////////////////////////////
//                              UART, RS232
///////////////////////////////////////////////////////////////////////////////


typedef struct
    {
    USART_TypeDef *Instance;
    UART_AdvFeatureInitTypeDef AdvancedInit;
    JOS_EVENT *hEvGroupTx;
    JOS_EVENT *RxQ;
    JOS_EVENT *TxQ;
    BYTE PO_RTS;
    BYTE PI_CTS;
    BYTE PO_DTR;
    BYTE PI_DSR;
    BYTE NowReceived;
    UINT BaudRate;
    UINT Mode;
    UINT OverSampling;
    UINT OneBitSampling;
    UINT ErrorCode;
    } UART_HANDLE;


typedef struct
    {
    USART_TypeDef *Instance;
    } UART_INST;



static UART_HANDLE UartHandle[USART_QTY];


LOCAL(BOOL) UART_TxGroupLock(int Port)
    {
    return JOSSemPend((UartHandle+Port)->hEvGroupTx, JOS_TICKS_PER_SEC*3)!=JOS_ERR_TIMEOUT;
    }

LOCAL(VOID) UART_TxGroupUnlock(int Port)
    {
    JOSSemPost((UartHandle+Port)->hEvGroupTx);
    }


//-----------------------------------------------------------------------------
//      UART 클럭을 구함
//-----------------------------------------------------------------------------
LOCAL(DWORD) UART_GetClock(USART_TypeDef *UB)
    {
    DWORD InClk=0;
    UART_INST UINST={UB};
    UART_ClockSourceTypeDef ClockSrc=UART_CLOCKSOURCE_UNDEFINED;

    UART_GETCLOCKSOURCE(&UINST, ClockSrc);
    switch (ClockSrc)
        {
        case UART_CLOCKSOURCE_PCLK1:  InClk=HAL_RCC_GetPCLK1Freq(); break;
        case UART_CLOCKSOURCE_HSI:    InClk=HSI_VALUE; break;
        case UART_CLOCKSOURCE_SYSCLK: InClk=HAL_RCC_GetSysClockFreq(); break;
        case UART_CLOCKSOURCE_LSE:    InClk=LSE_VALUE; break;
        }
    return InClk;
    }


//-----------------------------------------------------------------------------
//      Baudrate 설정
//-----------------------------------------------------------------------------
LOCAL(VOID) SetBaudrate(USART_TypeDef *UB, DWORD NewBaud, UINT OverSampling)
    {
    DWORD Div, HDiv, InClk;

    HDiv=NewBaud>>1;

    if ((InClk=UART_GetClock(UB))!=0)
        {
        if (OverSampling==UART_OVERSAMPLING_8)
            {
            Div=(InClk*2+HDiv)/NewBaud;
            Div=(Div&0xFFF0) | ((Div&0xF)>>1);
            }
        else{
            Div=(InClk+HDiv)/NewBaud;
            }
        UB->BRR=Div;
        }
    }

VOID WINAPI UART_SetBaudrate(int Port, DWORD NewBaud)
    {
    UART_HANDLE *hUart=UartHandle+Port;

    SetBaudrate(hUart->Instance, NewBaud, hUart->OverSampling);
    }



//-----------------------------------------------------------------------------
//      현재 Baudrate가 얼마인지 계산
//-----------------------------------------------------------------------------
DWORD WINAPI UART_GetBaudrate(int Port)
    {
    int UartDiv, HDiv;
    DWORD BaudRate=0, InClk;
    UART_HANDLE *hUart;
    USART_TypeDef *UB;

    hUart=UartHandle+Port;
    UB=hUart->Instance;
    if ((InClk=UART_GetClock(UB))!=0)
        {
        // Check UART Over Sampling to set Baud Rate Register
        UartDiv=UB->BRR;
        if (hUart->OverSampling==UART_OVERSAMPLING_8)
            {
            UartDiv=(UartDiv & 0xFFF0) | ((UartDiv & 7)<<1);
            HDiv=UartDiv>>1;
            BaudRate=(InClk*2+HDiv)/UartDiv;
            }
        else{
            UartDiv&=0xFFFF;
            HDiv=UartDiv>>1;
            BaudRate=(InClk+HDiv)/UartDiv;
            }
        }
    return BaudRate;
    }



//-----------------------------------------------------------------------------
//      자동 Baudrate Ebanle/Disable
//-----------------------------------------------------------------------------
LOCAL(VOID) SetAutoBaudrate(USART_TypeDef *UB, UINT EnBit, UINT Mode)
    {
    MODIFYBIT(UB->CR2, USART_CR2_ABREN, EnBit);
    if (EnBit!=0) MODIFYBIT(UB->CR2, USART_CR2_ABRMODE, Mode);
    }

VOID WINAPI UART_SetAutoBaudrate(int Port, BOOL En, UINT Mode)
    {
    USART_TypeDef *UB;

    if (En!=0) En=USART_CR2_ABREN;

    UB=(UartHandle+Port)->Instance;
    UB->CR1&= ~USART_CR1_UE;
    SetAutoBaudrate(UB, En, Mode);
    UB->CR1 |=  USART_CR1_UE;
    }



//-----------------------------------------------------------------------------
//      현재 자동 Baudrate 모드인지 알려줌
//-----------------------------------------------------------------------------
BOOL WINAPI UART_IsAutoBaudrate(int Port)
    {
    return (UartHandle+Port)->Instance->CR2 & USART_CR2_ABREN;
    }



//-----------------------------------------------------------------------------
//      H/W는  AutoBaud를 한번 설정하여 한문자를 받으면 AutoBaud가 동작하지 않지만
//      이 레지스터 값은 그대로 있음
//      이 값을 플래그로 사용하기 위해 Clear함
//-----------------------------------------------------------------------------
VOID WINAPI UART_ClearAutoBaudrate(int Port)
    {
    (UartHandle+Port)->Instance->CR2 &= ~USART_CR2_ABREN;
    }




//-----------------------------------------------------------------------------
//      DPLC나 485에서 다량의 데이터를 전송할 때 수신을 막기 위해 필요
//-----------------------------------------------------------------------------
VOID WINAPI UART_DisableRx(int Port)
    {
    (UartHandle+Port)->Instance->CR1 &= ~USART_CR1_RE;
    }

VOID WINAPI UART_EnableRx(int Port)
    {
    (UartHandle+Port)->Instance->CR1 |= USART_CR1_RE;
    }




LOCAL(VOID) UART_AdvFeatureCfg(USART_TypeDef *UB, UART_AdvFeatureInitTypeDef *AFI)
    {
    UINT Feat;

    Feat=AFI->AdvFeatureInit;
    if (Feat & UART_ADVFEATURE_TXINVERT_INIT)           MODIFYBIT(UB->CR2, USART_CR2_TXINV, AFI->TxPinLevelInvert);
    if (Feat & UART_ADVFEATURE_RXINVERT_INIT)           MODIFYBIT(UB->CR2, USART_CR2_RXINV, AFI->RxPinLevelInvert);
    if (Feat & UART_ADVFEATURE_DATAINVERT_INIT)         MODIFYBIT(UB->CR2, USART_CR2_DATAINV, AFI->DataInvert);
    if (Feat & UART_ADVFEATURE_SWAP_INIT)               MODIFYBIT(UB->CR2, USART_CR2_SWAP, AFI->Swap);
    if (Feat & UART_ADVFEATURE_RXOVERRUNDISABLE_INIT)   MODIFYBIT(UB->CR3, USART_CR3_OVRDIS, AFI->OverrunDisable);
    if (Feat & UART_ADVFEATURE_DMADISABLEONERROR_INIT)  MODIFYBIT(UB->CR3, USART_CR3_DDRE, AFI->DMADisableonRxError);
    if (Feat & UART_ADVFEATURE_AUTOBAUDRATE_INIT)       SetAutoBaudrate(UB, AFI->AutoBaudRateEnable, AFI->AutoBaudRateMode);

    if (Feat & UART_ADVFEATURE_MSBFIRST_INIT)           MODIFYBIT(UB->CR2, USART_CR2_MSBFIRST, AFI->MSBFirst);
    }


#define UART_CR1_FIELDS  ((DWORD)(USART_CR1_M | USART_CR1_PCE | USART_CR1_PS | USART_CR1_TE|USART_CR1_RE|USART_CR1_OVER8))//UART or USART CR1 fields of parameters set by UART_SetConfig API




#if !defined(STM32F030x6) &&\
    !defined(STM32F030x8) &&\
    !defined(STM32F070xB) &&\
    !defined(STM32F070x6) &&\
    !defined(STM32F030xC)


LOCAL(BOOL) UART_WaitOnFlag(USART_TypeDef *UB, DWORD Flag, DWORD Timeout)
    {
    BOOL  Rslt=TRUE;
    DWORD StartTime=HAL_GetTick();

    while ((UB->ISR & Flag)==0)
        {
        if (HAL_GetTick()-StartTime>=Timeout) {Rslt=FALSE; break;}
        }
    return Rslt;
    }



#define UART_TEACK_REACK_TIMEOUT        1000
LOCAL(BOOL) UART_ChkIdleState(USART_TypeDef *UB)
    {
    BOOL Rslt=FALSE;

    if (IS_UART_WAKEUP_INSTANCE(UB))
        {
        if (UB->CR1 & USART_CR1_TE)
            {
            if (UART_WaitOnFlag(UB, USART_ISR_TEACK, UART_TEACK_REACK_TIMEOUT)==FALSE) goto ProcExit;
            }

        if (UB->CR1 & USART_CR1_RE)
            {
            if (UART_WaitOnFlag(UB, USART_ISR_REACK, UART_TEACK_REACK_TIMEOUT)==FALSE) goto ProcExit;
            }
        }
    Rslt=TRUE;

    ProcExit:
    return Rslt;
    }
#else

LOCAL(BOOL) UART_ChkIdleState(USART_TypeDef *UB)
    {
    return TRUE;
    }

#endif



//-----------------------------------------------------------------------------
//      인터럽트 수신버퍼 설정
//-----------------------------------------------------------------------------
LOCAL(VOID) UART_PrepareRecvIT(UART_HANDLE *hUart)
    {
    USART_TypeDef *UB;

    hUart->RxQ=JOSQCreate(UART_RECV_BUFF_SIZE, JOSQTYPE_BYTE);
    UB=hUart->Instance;
    UB->CR1|=USART_CR1_PEIE;    //Enable the UART Parity Error Interrupt
    UB->CR3|=USART_CR3_EIE;     //Enable the UART Error Interrupt: (Frame error, noise error, overrun error)
    UB->CR1|=USART_CR1_RXNEIE;  //Enable the UART Data Register not empty Interrupt
    }



//-----------------------------------------------------------------------------
//      인터럽트 송신버퍼 설정
//-----------------------------------------------------------------------------
LOCAL(VOID) UART_PrepareSendIT(UART_HANDLE *hUart)
    {
    USART_TypeDef *UB;

    hUart->TxQ=JOSQCreate(UART_SEND_BUFF_SIZE, JOSQTYPE_BYTE);
    hUart->hEvGroupTx=JOSSemCreate(1);
    UB=hUart->Instance;
    UB->CR3|=USART_CR3_EIE;     //Enable the UART Error Interrupt: (Frame error, noise error, overrun error)
    UB->CR1|=USART_CR1_TXEIE;   //Enable the UART Transmit Data Register Empty Interrupt
    }



//-----------------------------------------------------------------------------
//      RS232 초기화
//-----------------------------------------------------------------------------
VOID WINAPI InitUart(int ComNo, DWORD BaudRate, int Parity, int DataLen, int StopBit, int HwFlowCtl, BOOL AutoBaud)
    {
    USART_TypeDef *UB;
    UART_HANDLE *hUart;

    hUart=UartHandle+ComNo;
    hUart->Mode=UART_MODE_TX_RX;    //=USART_CR1_TE|USART_CR1_RE
    //hUart->OverSampling=UART_OVERSAMPLING_8;
    hUart->AdvancedInit.AdvFeatureInit=UART_ADVFEATURE_NO_INIT;
    if (AutoBaud)
        {
        hUart->AdvancedInit.AdvFeatureInit=UART_ADVFEATURE_AUTOBAUDRATE_INIT; //UART_ADVFEATURE_NO_INIT;
        hUart->AdvancedInit.AutoBaudRateEnable=UART_ADVFEATURE_AUTOBAUDRATE_ENABLE;
        hUart->AdvancedInit.AutoBaudRateMode=UART_ADVFEATURE_AUTOBAUDRATE_ONSTARTBIT;
        }

    switch (ComNo)
        {
        #ifdef PO_COM1TX
        case COM1:
            hUart->Instance=USART1;
            __HAL_RCC_USART1_CLK_ENABLE();
            NVIC_SetPriority(USART1_IRQn, 1);
            NVIC_EnableIRQ(USART1_IRQn);

            UART_PrepareRecvIT(hUart);  //수신버퍼 설정
            UART_PrepareSendIT(hUart);
            break;
        #endif

        #ifdef PO_COM2TX
        case COM2:
            hUart->Instance=USART2;
            __HAL_RCC_USART2_CLK_ENABLE();
            NVIC_SetPriority(USART2_IRQn, 1);
            NVIC_EnableIRQ(USART2_IRQn);

            UART_PrepareRecvIT(hUart);  //수신버퍼 설정
            UART_PrepareSendIT(hUart);
            break;
        #endif

        #ifdef PO_COM3TX
        case COM3:
            hUart->Instance=USART3;
            __HAL_RCC_SYSCFG_CLK_ENABLE();
            __HAL_RCC_USART3_CLK_ENABLE();
            NVIC_SetPriority(USART3_8_IRQn, 0);
            NVIC_EnableIRQ(USART3_8_IRQn);

            UART_PrepareRecvIT(hUart);  //수신버퍼 설정
            UART_PrepareSendIT(hUart);
            break;
        #endif

        #ifdef PO_COM4TX
        case COM4:
            hUart->Instance=USART4;
            __HAL_RCC_SYSCFG_CLK_ENABLE();
            __HAL_RCC_USART4_CLK_ENABLE();
            NVIC_SetPriority(USART3_8_IRQn, 0);
            NVIC_EnableIRQ(USART3_8_IRQn);

            UART_PrepareRecvIT(hUart);  //수신버퍼 설정
            UART_PrepareSendIT(hUart);
            break;
        #endif

        #ifdef PO_COM5TX
        case COM5:
            hUart->Instance=USART5;
            __HAL_RCC_SYSCFG_CLK_ENABLE();
            __HAL_RCC_USART5_CLK_ENABLE();
            NVIC_SetPriority(USART3_8_IRQn, 0);
            NVIC_EnableIRQ(USART3_8_IRQn);

            UART_PrepareRecvIT(hUart);  //수신버퍼 설정
            UART_PrepareSendIT(hUart);
            break;
        #endif

        #ifdef PO_COM6TX
        case COM6:
            hUart->Instance=USART6;
            __HAL_RCC_SYSCFG_CLK_ENABLE();
            __HAL_RCC_USART6_CLK_ENABLE();
            NVIC_SetPriority(USART3_8_IRQn, 0);
            NVIC_EnableIRQ(USART3_8_IRQn);

            UART_PrepareRecvIT(hUart);  //수신버퍼 설정
            UART_PrepareSendIT(hUart);
            break;
        #endif

        #ifdef PO_COM7TX
        case COM7:
            hUart->Instance=USART7;
            __HAL_RCC_SYSCFG_CLK_ENABLE();
            __HAL_RCC_USART7_CLK_ENABLE();
            NVIC_SetPriority(USART3_8_IRQn, 0);
            NVIC_EnableIRQ(USART3_8_IRQn);

            UART_PrepareRecvIT(hUart);  //수신버퍼 설정
            UART_PrepareSendIT(hUart);
            break;
        #endif

        #ifdef PO_COM8TX
        case COM8:
            hUart->Instance=USART8;
            __HAL_RCC_SYSCFG_CLK_ENABLE();
            __HAL_RCC_USART8_CLK_ENABLE();
            NVIC_SetPriority(USART3_8_IRQn, 0);
            NVIC_EnableIRQ(USART3_8_IRQn);

            UART_PrepareRecvIT(hUart);  //수신버퍼 설정
            UART_PrepareSendIT(hUart);
            break;
        #endif
        }

    UB=hUart->Instance;

    UB->CR1&= ~USART_CR1_UE;    //__HAL_UART_DISABLE
    MODIFYBIT(UB->CR1, UART_CR1_FIELDS, DataLen|Parity|hUart->Mode|hUart->OverSampling);
    MODIFYBIT(UB->CR2, USART_CR2_STOP, StopBit);
    MODIFYBIT(UB->CR3, USART_CR3_RTSE|USART_CR3_CTSE|USART_CR3_ONEBIT, HwFlowCtl|hUart->OneBitSampling);

    if (BaudRate!=0) SetBaudrate(UB, BaudRate, hUart->OverSampling);
    if (hUart->AdvancedInit.AdvFeatureInit!=UART_ADVFEATURE_NO_INIT) UART_AdvFeatureCfg(UB, &hUart->AdvancedInit);

    #ifdef USART_CR2_LINEN
    UB->CR2&=~(USART_CR2_LINEN|USART_CR2_CLKEN);
    #else
    UB->CR2&=~USART_CR2_CLKEN;
    #endif

    #if defined(USART_CR3_SCEN) && defined(USART_CR3_IREN)
    UB->CR3&=~(USART_CR3_SCEN|USART_CR3_HDSEL|USART_CR3_IREN);
    #elif defined(USART_CR3_IREN)
    UB->CR3&=~(USART_CR3_HDSEL|USART_CR3_IREN));
    #else
    UB->CR3&=~USART_CR3_HDSEL;
    #endif

    UB->CR1|= USART_CR1_UE;     //__HAL_UART_ENABLE

    if (UART_ChkIdleState(UB)==FALSE)
        {
        __HAL_UART_DISABLE_IT(hUart, UART_IT_TXE);
        __HAL_UART_DISABLE_IT(hUart, UART_IT_RXNE);
        __HAL_UART_DISABLE_IT(hUart, UART_IT_PE);
        __HAL_UART_DISABLE_IT(hUart, UART_IT_ERR);
        }
    }




//-----------------------------------------------------------------------------
//      NMI
//-----------------------------------------------------------------------------
VOID NMI_Handler(VOID)
    {
    LowPrintf(CRLF"NMI Occured, Halted...");
    for (;;);
    }



//-----------------------------------------------------------------------------
//      익섹션 처리
//-----------------------------------------------------------------------------
LOCAL(VOID) DefaultExpHandler(UINT *lpStack, LPCSTR FaultMsg)
    {
    LowPrintf(CRLF"%s"CRLF, FaultMsg);
    LowPrintf(" R12=%08X LR=%08X PC=%08X SR=%08X"CRLF, lpStack[8], lpStack[9], lpStack[10], lpStack[11]);
    LowPrintf("  R0=%08X R1=%08X R2=%08X R3=%08X"CRLF, lpStack[4], lpStack[5], lpStack[6], lpStack[7]);
    LowPrintf("  R4=%08X R5=%08X R6=%08X R7=%08X"CRLF, lpStack[0], lpStack[1], lpStack[2], lpStack[3]);

    #ifdef USE_JOS
    LowPrintf("  SP=%08X Task=%d"CRLF, lpStack+12, JOSCurrPrio);
    #else
    LowPrintf("  SP=%08X"CRLF, lpStack+12);
    #endif

    LowPrintf("Halted...");
    FatalError(FERR_EXCEPTION);
    }



//-----------------------------------------------------------------------------
//      Fault exception.
//      익셉션이 발생할 때 CPU가 Push하는 순서 xPSR,PC,LR,R12,R3,R2,R1,R0, R7,R6,R5,R4
//-----------------------------------------------------------------------------
VOID NMI_Handler_C(UINT *lpStack)           {DefaultExpHandler(lpStack, "NMI Occured");}
VOID HardFault_Handler_C(UINT *lpStack)     {DefaultExpHandler(lpStack, "Hard Fault");}
VOID SVC_Handler_C(UINT *lpStack)           {DefaultExpHandler(lpStack, "Called SVC");}
VOID PendSV_Handler_C(UINT *lpStack)        {DefaultExpHandler(lpStack, "PendSV Exception");}
VOID UndefinedIntHandler_C(UINT *lpStack)   {DefaultExpHandler(lpStack, "Undefined Interrupt");}




//-----------------------------------------------------------------------------
//      이 핸들러의 우선 순위보다 높은 것은 SVC_Handler와 HardFault 밖에 없으므로
//      아래 처리중에 인터럽트가 발생할 일은 없음
//-----------------------------------------------------------------------------
VOID SysTickHandler_C(VOID)
    {
    FineTickCnt++;
    _1msProc();
    }




#if 0
//-----------------------------------------------------------------------------
//      TIM interrupts requests handler
//-----------------------------------------------------------------------------
LOCAL(VOID) TIM_IRQHandler(TIM_TypeDef*TIMx)
    {
    // Capture compare 1 event
    if ((TIMx->SR & TIM_FLAG_CC1)!=0 && (TIMx->DIER & TIM_IT_CC1)!=0)
        {
        TIMx->SR=~TIM_IT_CC1;

        if (TIMx->CCMR1 & TIM_CCMR1_CC1S)
            {   //Input capture event
            //HAL_TIM_IC_CaptureCallback(TIMx, HAL_TIM_ACTIVE_CHANNEL_1);
            }
        else{   //Output compare event
            //HAL_TIM_OC_DelayElapsedCallback(TIMx, HAL_TIM_ACTIVE_CHANNEL_1);
            //HAL_TIM_PWM_PulseFinishedCallback(TIMx, HAL_TIM_ACTIVE_CHANNEL_1);
            }
        }

    //Capture compare 2 event
    if ((TIMx->SR & TIM_FLAG_CC2)!=0 && (TIMx->DIER & TIM_IT_CC2)!=0)
        {
        TIMx->SR = ~TIM_IT_CC2;

        if (TIMx->CCMR1 & TIM_CCMR1_CC2S)
            {
            //HAL_TIM_IC_CaptureCallback(TIMx, HAL_TIM_ACTIVE_CHANNEL_2);
            }
        else{
            //HAL_TIM_OC_DelayElapsedCallback(TIMx, HAL_TIM_ACTIVE_CHANNEL_2);
            //HAL_TIM_PWM_PulseFinishedCallback(TIMx, HAL_TIM_ACTIVE_CHANNEL_2);
            }
        }


    // Capture compare 3 event
    if ((TIMx->SR & TIM_FLAG_CC3)!=0 && (TIMx->DIER & TIM_IT_CC3)!=0)
        {
        TIMx->SR = ~TIM_IT_CC3;

        if (TIMx->CCMR2 & TIM_CCMR2_CC3S)
            {
            //HAL_TIM_IC_CaptureCallback(TIMx, HAL_TIM_ACTIVE_CHANNEL_3);
            }
        else{
            //HAL_TIM_OC_DelayElapsedCallback(TIMx, HAL_TIM_ACTIVE_CHANNEL_3);
            //HAL_TIM_PWM_PulseFinishedCallback(TIMx, HAL_TIM_ACTIVE_CHANNEL_3);
            }
        }


    // Capture compare 4 event
    if ((TIMx->SR & TIM_FLAG_CC4)!=0 && (TIMx->DIER & TIM_IT_CC4)!=0)
        {
        TIMx->SR = ~TIM_IT_CC4;

        if ((TIMx->CCMR2&TIM_CCMR2_CC4S)!=0x00)
            {
            //HAL_TIM_IC_CaptureCallback(TIMx, HAL_TIM_ACTIVE_CHANNEL_4);
            }
        else{
            //HAL_TIM_OC_DelayElapsedCallback(TIMx, HAL_TIM_ACTIVE_CHANNEL_4);
            //HAL_TIM_PWM_PulseFinishedCallback(TIMx, HAL_TIM_ACTIVE_CHANNEL_4);
            }
        }

    //TIM Update event
    if ((TIMx->SR & TIM_FLAG_UPDATE)!=0 && (TIMx->DIER & TIM_IT_UPDATE)!=0)
        {
        TIMx->SR = ~TIM_IT_UPDATE;
        //HAL_TIM_PeriodElapsedCallback(TIMx);
        }

    //TIM Break input event
    if ((TIMx->SR & TIM_FLAG_BREAK)!=0 && (TIMx->DIER & TIM_IT_BREAK)!=0)
        {
        TIMx->SR = ~TIM_IT_BREAK;
        //HAL_TIMEx_BreakCallback(TIMx);
        }

    //TIM Trigger detection event
    if ((TIMx->SR & TIM_FLAG_TRIGGER)!=0 && (TIMx->DIER & TIM_IT_TRIGGER)!=0)
        {
        TIMx->SR = ~TIM_IT_TRIGGER;
        //HAL_TIM_TriggerCallback(TIMx);
        }

    //TIM commutation event
    if ((TIMx->SR & TIM_FLAG_COM)!=0 && (TIMx->DIER & TIM_IT_COM)!=0)
        {
        TIMx->SR = ~TIM_IT_COM;
        //HAL_TIMEx_CommutationCallback(TIMx);
        }
    }
#endif



//#ifdef TIM6 //main()에서 작성함       //STM32F042는 없음
#if 0
//-----------------------------------------------------------------------------
//      SysTick 이 없는 칩에서 사용
//-----------------------------------------------------------------------------
VOID TIM6_DAC_IRQHandler(VOID)
    {
    //TIM_IRQHandler(TIM6);

    if (TIM6->SR & TIM_FLAG_UPDATE)
        {
        }

    TIM6->SR=0;
    }
#endif



//-----------------------------------------------------------------------------
//      RS232 에러 표시
//-----------------------------------------------------------------------------
VOID WINAPI UART_DisplayError(VOID)
    {
    int I, ErrCode, HwErr;
    UART_HANDLE *hUart;

    for (I=COM1; I<USART_QTY; I++)
        {
        if ((hUart=UartHandle+I)!=NULL &&
            (ErrCode=hUart->ErrorCode)!=0)
            {
            HwErr=0;
            if (ErrCode & HAL_UART_ERROR_PE) {Printf("SENSOR: COM%d Parity Err"CRLF, I+1); HwErr=1;}
            if (ErrCode & HAL_UART_ERROR_FE) {Printf("SENSOR: COM%d Frame Err"CRLF, I+1); HwErr=1;}
            if (ErrCode & HAL_UART_ERROR_NE) {Printf("SENSOR: COM%d Noise Err"CRLF, I+1); HwErr=1;}
            if (ErrCode & HAL_UART_ERROR_ORE) Printf("SENSOR: COM%d Over Run"CRLF, I+1);
            if (HwErr) UART_ErrUsrHandler(I, ErrCode);
            hUart->ErrorCode=0;
            }
        }
    }



//-----------------------------------------------------------------------------
//      Handle UART interrupt request.
//-----------------------------------------------------------------------------
LOCAL(VOID) UART_IRQHandler(UART_HANDLE *hUart)
    {
    DWORD ISR;
    USART_TypeDef *UB;

    UB=hUart->Instance;
    ISR=UB->ISR;

    if (ISR & USART_ISR_PE)             //Parity error
        {
        UB->ICR=USART_ICR_PECF;
        hUart->ErrorCode|=HAL_UART_ERROR_PE;
        }

    if (ISR & USART_ISR_FE)             //Frame error
        {
        UB->ICR=USART_ICR_FECF;
        hUart->ErrorCode|=HAL_UART_ERROR_FE;
        }

    if (ISR & USART_ISR_NE)             //Noise error
        {
        UB->ICR=USART_ICR_NCF;
        hUart->ErrorCode|=HAL_UART_ERROR_NE;
        }

    if (ISR & USART_ISR_ORE)            //Over-Run
        {
        UB->ICR=USART_ICR_ORECF;
        hUart->ErrorCode|=HAL_UART_ERROR_ORE;
        }

    if (ISR & USART_ISR_CTSIF)
        {
        UB->ICR=USART_ICR_CTSCF;
        UB->CR1|=USART_CR1_TXEIE;
        }

    #if !defined(STM32F030x6) && !defined(STM32F030x8)&& !defined(STM32F070xB)&& !defined(STM32F070x6)&& !defined(STM32F030xC)
    if (ISR & USART_ISR_WUF)            //Wakeup from Stop mode
        {
        UB->ICR=USART_ICR_WUCF;
        //HAL_UARTEx_WakeupCallback(hUart);
        }
    #endif

    if (ISR & USART_ISR_RXNE)
        {
        hUart->NowReceived=1;           //RS485를 송신을 위해 추가
        if (JOSQPost(hUart->RxQ, UB->RDR, 0)!=JOS_ERR_NONE) hUart->ErrorCode|=HAL_UART_ERROR_ORE;
        //else UB->CR1&=~USART_CR1_RXNEIE;              //버퍼가 꽉참, 수신 Byte를 읽어가지 않으면 계속 Int가 걸림
        }


    if (ISR & USART_ISR_TXE)                            //송신레지스터 빔
        {
        DWORD Data;
        JOS_RESULT QRslt;
        Data=JOSQPend(hUart->TxQ, CHECKACCEPT, &QRslt);
        if (QRslt==JOS_ERR_NONE)
            {
            UB->TDR=Data;
            }
        else{
            UB->CR1&=~USART_CR1_TXEIE;                  //Disable the UART Transmit Data Register Empty Interrupt
            //UB->CR1|=USART_CR1_TCIE;                  //Enable the UART Transmit Complete Interrupt
            }
        }

    #if 0
    if (ISR & USART_ISR_TC) UART_EndTransmit_IT(hUart); //송신완료
    #endif
    }




//-----------------------------------------------------------------------------
//                 STM32F0xx Peripherals Interrupt Handlers
//  Add here the Interrupt Handler for the used peripheral(s) (PPP), for the
//  available peripheral interrupt handler's name please refer to the startup
//  file (startup_stm32f0xx.s).
//
//      This function handles UART interrupt request.
//      This function is redefined in "main.h" and related to DMA used for USART data transmission
//-----------------------------------------------------------------------------
VOID USART1_IRQHandler(VOID)
    {
    JOSIntEnterII();
    UART_IRQHandler(UartHandle+COM1);
    JOSIntExitII();
    }


VOID USART2_IRQHandler(VOID)
    {
    JOSIntEnterII();
    UART_IRQHandler(UartHandle+COM2);
    JOSIntExitII();
    }



//-----------------------------------------------------------------------------
//      SYSCFG interrupt line 29 status register (SYSCFG_ITLINE29)
//-----------------------------------------------------------------------------
VOID USART3_8_IRQHandler(VOID)
    {
    #if defined(STM32F091xC)
    JOSIntEnterII();
    if (SYSCFG->IT_LINE_SR[29] & SYSCFG_ITLINE29_SR_USART3_GLB) UART_IRQHandler(UartHandle+COM3);
    if (SYSCFG->IT_LINE_SR[29] & SYSCFG_ITLINE29_SR_USART4_GLB) UART_IRQHandler(UartHandle+COM4);
    if (SYSCFG->IT_LINE_SR[29] & SYSCFG_ITLINE29_SR_USART5_GLB) UART_IRQHandler(UartHandle+COM5);
    if (SYSCFG->IT_LINE_SR[29] & SYSCFG_ITLINE29_SR_USART6_GLB) UART_IRQHandler(UartHandle+COM6);
    if (SYSCFG->IT_LINE_SR[29] & SYSCFG_ITLINE29_SR_USART7_GLB) UART_IRQHandler(UartHandle+COM7);
    if (SYSCFG->IT_LINE_SR[29] & SYSCFG_ITLINE29_SR_USART8_GLB) UART_IRQHandler(UartHandle+COM8);
    JOSIntExitII();
    #endif
    }



///////////////////////////////////////////////////////////////////////////////
//                      RS232, 시리얼포트
///////////////////////////////////////////////////////////////////////////////



//-----------------------------------------------------------------------------
//      UART H/W FlowControl을  켜거나 끕니다
//-----------------------------------------------------------------------------
VOID WINAPI UART_SetHwFlow(int Port, BOOL En)
    {
    USART_TypeDef *UB;

    UB=(UartHandle+Port)->Instance;
    if (En) UB->CR3|=USART_CR3_RTSE|USART_CR3_CTSE;             //RTSE:CR3_Bit8, CTES_CR3_Bit9
    else    UB->CR3&=~(USART_CR3_RTSE|USART_CR3_CTSE);
    }



//-----------------------------------------------------------------------------
//      Comport 레지스터 표시
//-----------------------------------------------------------------------------
VOID WINAPI UART_DispReg(int Port)
    {
    USART_TypeDef *UB;

    UB=(UartHandle+Port)->Instance;
    Printf("ISR=%08X"CRLF
           "CR1=%08X"CRLF
           "CR2=%08X"CRLF
           "CR3=%08X"CRLF, UB->ISR, UB->CR1, UB->CR2, UB->CR3);
    }



//-----------------------------------------------------------------------------
//      CTS핀상태를 알려줌
//-----------------------------------------------------------------------------
int WINAPI UART_InCTS(int Port)
    {
    int CtsPin;

    //return (UartHandle+Port)->Instance->ISR & UART_FLAG_CTS;
    if ((CtsPin=(UartHandle+Port)->PI_CTS)==0) return 0;
    return PortIn(CtsPin);
    }



//-----------------------------------------------------------------------------
//      RTS 출력
//-----------------------------------------------------------------------------
int WINAPI UART_OutRTS(int Port, BOOL EnableFg)
    {
    int RtsPin, Rslt=0;

    if ((RtsPin=(UartHandle+Port)->PO_RTS)!=0)
        {
        //Printf("COM%d RTS<-%d"CRLF, Port+1, EnableFg);
        Rslt=PortOut(RtsPin, EnableFg);
        }
    return Rslt;
    }



//-----------------------------------------------------------------------------
//      DSR핀상태를 알려줌
//-----------------------------------------------------------------------------
int WINAPI UART_InDSR(int Port)
    {
    int DsrPin;

    if ((DsrPin=(UartHandle+Port)->PI_DSR)==0) return 0;
    return PortIn(DsrPin);
    }



//-----------------------------------------------------------------------------
//      DTR 출력
//-----------------------------------------------------------------------------
int WINAPI UART_OutDTR(int Port, BOOL EnableFg)
    {
    int DtrPin, Rslt=0;

    if ((DtrPin=(UartHandle+Port)->PO_DTR)!=0) Rslt=PortOut(DtrPin, EnableFg);
    return Rslt;
    }



//-----------------------------------------------------------------------------
//      송신 Q에 1Byte를 넣음
//-----------------------------------------------------------------------------
BOOL WINAPI USB_TxByte(int Cha);
BOOL WINAPI UART_TxByteIT(int Port, int Ch)
    {
    JOS_RESULT Rslt;
    USART_TypeDef *UB;
    UART_HANDLE *hUart;

    //if (Port==COM_USB) return USB_TxByte(Ch);
    hUart=UartHandle+Port;
    UB=hUart->Instance;
    Rslt=JOSQPost(hUart->TxQ, Ch, 0);
    UB->CR1|=USART_CR1_TXEIE;           //Enable the UART Transmit Data Register Empty Interrupt
    return Rslt==JOS_ERR_NONE;
    }



//-----------------------------------------------------------------------------
//      폴링방식으로 주어진 Data를 전송함
//-----------------------------------------------------------------------------
BOOL WINAPI UART_TxByte(int Port, int Ch)
    {
    BOOL Rslt=FALSE;
    USART_TypeDef *UB;

    UB=(UartHandle+Port)->Instance;
    if (UB->ISR & USART_ISR_TXE)
        {
        UB->TDR=(BYTE)Ch;
        Rslt=TRUE;
        }
    return Rslt;
    }



//-----------------------------------------------------------------------------
//      Serial Port로 출력
//-----------------------------------------------------------------------------
VOID WINAPI UART_TxStr(int Port, LPCSTR Str)
    {
    int Ch;
    DWORD EndTime;

    EndTime=lstrlen(Str)+GetTickCount()+1;      //+1은 1자 보낼 때 GetCount()로 1ms를 바로 넘어갈 수도 있음
    while ((Ch=*Str++)!=0)
        {
        while (UART_TxByte(Port, Ch)==FALSE)
            {
            if (GetTickCount()>=EndTime) break;
            }
        }
    }



//-----------------------------------------------------------------------------
//      Serial Port로 출력 (인터럽트이용)
//-----------------------------------------------------------------------------
VOID WINAPI UART_TxBinIT(int Port, LPCBYTE Data, int SendBytes)
    {
    int Ch;
    DWORD EndTime;

    if (UART_TxGroupLock(Port)!=FALSE)
        {
        EndTime=GetTickCount()+SendBytes+1; //+1은 1자 보낼 때 GetCount()로 1ms를 바로 넘어갈 수도 있음
        while (SendBytes--)
            {
            Ch=*Data++;
            while (UART_TxByteIT(Port, Ch)==FALSE)
                {
                if (GetTickCount()>=EndTime) return;
                }
            }
        UART_TxGroupUnlock(Port);
        }
    }



//-----------------------------------------------------------------------------
//      Serial Port로 출력
//-----------------------------------------------------------------------------
VOID WINAPI UART_TxBin(int Port, LPCBYTE Data, int SendBytes)
    {
    int Ch;
    DWORD EndTime;

    EndTime=GetTickCount()+SendBytes+1; //+1은 1자 보낼 때 GetCount()로 1ms를 바로 넘어갈 수도 있음
    #ifdef BAUD_4800_SUPPORT
    EndTime<<=1;        //4800은 2ms가 필요함
    #endif
    while (SendBytes--)
        {
        Ch=*Data++;
        while (UART_TxByte(Port, Ch)==FALSE)
            {
            if (GetTickCount()>=EndTime) return;
            }
        }
    }



VOID WINAPI UART_TxStrIT(int Port, LPCSTR Str)
    {
    int Ch;
    DWORD EndTime;

    EndTime=lstrlen(Str)*2+GetTickCount();

    if (UART_TxGroupLock(Port)!=FALSE)
        {
        while ((Ch=*Str++)!=0)
            {
            while (UART_TxByteIT(Port, Ch)==FALSE)
                {
                if (GetTickCount()>=EndTime)
                    {
                    if (Port!=COM_DEBUG) Printf("COM%d Tx Timeout"CRLF, Port+1);
                    UART_TxGroupUnlock(Port);
                    goto ProcExit;
                    }
                }
            }
        UART_TxGroupUnlock(Port);
        }
    ProcExit:;
    }




VOID WINAPI UART_TxChrIT(int Port, int Cha)
    {
    CHAR Buff[4];

    Buff[0]=(CHAR)Cha;
    Buff[1]=0;
    UART_TxStrIT(Port, Buff);
    }


VOID Printf(LPCSTR DispStr, ...)
    {
    va_list VA;
    CHAR Buff[128];

    va_start(VA, DispStr);
    Vsprintf(Buff, DispStr, VA);
    va_end(VA);
    UART_TxStrIT(DebugPort, Buff);
    }


VOID LowPrintf(LPCSTR DispStr, ...)
    {
    va_list VA;
    CHAR Buff[128];

    va_start(VA, DispStr);
    Vsprintf(Buff, DispStr, VA);
    va_end(VA);
    UART_TxStr(DebugPort, Buff);
    }



VOID PrintfII(int Port, LPCSTR DispStr, ...)
    {
    va_list VA;
    CHAR Buff[128];

    if (Port<USART_QTY)
        {
        va_start(VA, DispStr);
        Vsprintf(Buff, DispStr, VA);
        va_end(VA);
        UART_TxStrIT(Port, Buff);
        }
    }

int  WINAPI GetChar(VOID) {return UART_RxByteIT(DebugPort);}
int  WINAPI GetDebugPort(VOID) {return DebugPort;}
VOID WINAPI SetDebugPort(int PortNo) {DebugPort=PortNo;}




//-----------------------------------------------------------------------------
//      시리얼 포트로 부터 1Byte를 읽음 (수신된 Data가 없으면 -1리턴)
//-----------------------------------------------------------------------------
int WINAPI UART_RxByte(int Port)
    {
    int RcvBytes=-1;
    USART_TypeDef *UB;

    UB=(UartHandle+Port)->Instance;
    if (UB->ISR & USART_ISR_RXNE) RcvBytes=UB->RDR & 0xFF;

    return RcvBytes;
    }




//-----------------------------------------------------------------------------
//      인터럽트 수신
//
//      RTOS의 Task에서 호출하기 위한 UART 수신함수
//      Timeout은 ms단위, Timeout==INFINITE 이면 무한 대기
//      Time Out이 되면 -1리턴
//-----------------------------------------------------------------------------
int WINAPI USB_RxByte(VOID);
int WINAPI UART_RxByteRT(int Port, UINT TimeOut)
    {
    int RcvByte;
    JOS_RESULT QRslt;
    UART_HANDLE *hUart;

    //if (Port==COM_USB) return USB_RxByte();
    hUart=UartHandle+Port;
    RcvByte=JOSQPend(hUart->RxQ, TimeOut, &QRslt);
    if (QRslt!=JOS_ERR_NONE) RcvByte=-1;
    return RcvByte;
    }

int WINAPI UART_RxByteIT(int Port) {return UART_RxByteRT(Port, CHECKACCEPT);}




//-----------------------------------------------------------------------------
//      읽은 바이트수 리턴
//-----------------------------------------------------------------------------
int WINAPI UART_ReceiveCntIT(int Port)
    {
    return JOSQEntries((UartHandle+Port)->RxQ);
    }



//-----------------------------------------------------------------------------
//     모두 송신했는지 체크함
//-----------------------------------------------------------------------------
VOID WINAPI UART_WaitAllSendIT(int ComPort, DWORD TimeOut)
    {
    UART_HANDLE *hUart;
    DWORD Time;

    hUart=UartHandle+ComPort;
    Time=GetTickCount();
    for (;;)
        {
        if (JOSQEntries(hUart->TxQ)==0 && (hUart->Instance->ISR & USART_ISR_TC)!=0) break;
        if (GetTickCount()-Time>TimeOut)    //200ms: 2400 Baud에서 40자 전송시간은 166ms
            {
            Printf("COM%d: WaitAllSendIT Timeout"CRLF, ComPort+1);
            break;
            }
        }
    }




//-----------------------------------------------------------------------------
//      485 전송을 함
//-----------------------------------------------------------------------------
BOOL WINAPI UART_485BinOut(int ComPort, LPCBYTE Data, int ToSendBytes, BOOL FirstDelayFg, int RetryQty, int DEPin)
    {
    int I, Retry, Rslt=FALSE;
    DWORD Delay, IdleStartTime, FirstTime, CurrTime;
    UART_HANDLE *hUart;
    //USART_TypeDef *UB;

    hUart=UartHandle+ComPort;
    //UB=hUart->Instance;

    for (Retry=0; Retry<RetryQty; Retry++)
        {
        while (UART_RxByteIT(ComPort)>=0);

        if (FirstDelayFg!=FALSE)
            {
            IdleStartTime=FirstTime=GetTickCount();
            Delay=((IdleStartTime % 10)+1)<<2;          //4~40ms Delay, 다른 장치와 485 충돌을 피하기 위함
            hUart->NowReceived=0;
            for (;;)
                {
                CurrTime=GetTickCount();
                //if (UB->ISR & USART_ISR_RXNE) IdleStartTime=CurrTime;     //수신부가 인터럽트를 쓰기 때문에 USART_ISR_RXNE를 사용할 수 없음
                if (hUart->NowReceived) {hUart->NowReceived=0; IdleStartTime=CurrTime;} //수신이 없는지 일정시간 후 전송
                if (CurrTime-IdleStartTime>=Delay) break;
                if (CurrTime-FirstTime>=1000)
                    {
                    Printf("SENSOR: 485 Line Busy"CRLF);
                    goto Cont;
                    }
                }
            }

        if (DEPin>=0) PortOut(DEPin, HIGH);             //DPLC를 사용할 때는 DE핀이 없음
        UART_TxBin(ComPort, Data, ToSendBytes);
        UART_WaitAllSendIT(ComPort, 200);               //송신버퍼도 비고 시리얼레지스터도 빌때까지 기다림
        if (DEPin>=0) PortOut(DEPin, LOW);

        for (I=0; I<ToSendBytes; I++)
            {
            if (UART_RxByteIT(ComPort)!=Data[I]) break;
            }

        if (I==ToSendBytes) {Rslt++; break;}            //{Printf("SENSOR: Send 485 OK"CRLF); break;}
        Printf("SENSOR: Retry 485 Out (%d)"CRLF, Retry+1);
        Cont:;
        }
    return Rslt;
    }

BOOL WINAPI UART_485StrOut(int ComPort, LPCSTR Str, int DEPin) {return UART_485BinOut(ComPort, (LPCBYTE)Str, lstrlen(Str), TRUE, 3, DEPin);}





#ifdef HAL_I2C_MODULE_ENABLED
///////////////////////////////////////////////////////////////////////////////
//              I2C
///////////////////////////////////////////////////////////////////////////////

static I2C_HandleTypeDef I2cHandle1;
#ifdef PO_I2C2SCL
static I2C_HandleTypeDef I2cHandle2;
#endif
static JOS_EVENT* I2C_Sem;


#define TIMING_CLEAR_MASK       0xF0FFFFFF      //I2C TIMING clear register Mask
#define I2C_TIMEOUT_STOPF       25



LOCAL(I2C_HandleTypeDef*) GetI2cHandle(int I2C_Ch)
    {
    I2C_HandleTypeDef *hI2c=NULL;

    if (I2C_Ch==IIC1) hI2c=&I2cHandle1;
    #ifdef PO_I2C2SCL
    else              hI2c=&I2cHandle2;
    #endif
    return hI2c;
    }




LOCAL(I2C_TypeDef*) GetI2cCtrlAddr(int I2C_Ch)
    {
    I2C_TypeDef *I2CA=NULL;

    if (I2C_Ch==IIC1) I2CA=I2cHandle1.Instance;
    #ifdef PO_I2C2SCL
    else              I2CA=I2cHandle2.Instance;
    #endif
    return I2CA;
    }




//-----------------------------------------------------------------------------
//      I2C 초기화 (Timing==0을 주면 기본값 0x10420F13 사용)
//-----------------------------------------------------------------------------
VOID WINAPI InitI2C(int I2C_Ch, DWORD Timing)
    {
    I2C_TypeDef *I2CA;
    I2C_HandleTypeDef *hI2C;

    hI2C=GetI2cHandle(I2C_Ch);

    if (I2C_Ch==IIC1)
        {
        __HAL_RCC_I2C1_CLK_ENABLE();
        hI2C->Instance=I2CA=I2C1;
        }
    #ifdef PO_I2C2SCL
    else{
        __HAL_RCC_I2C2_CLK_ENABLE();
        hI2C->Instance=I2CA=I2C2;
        }
    #endif

    if (Timing==0) Timing=0x10420F13;
    hI2C->Init.Timing=Timing;       //0x10420F13:100Khz / 0x00E0D3FF:16KHz
    hI2C->Init.DualAddressMode=I2C_DUALADDRESS_DISABLE;
    hI2C->Init.OwnAddress1=0;
    hI2C->Init.OwnAddress2=0;
    hI2C->Init.AddressingMode=I2C_ADDRESSINGMODE_7BIT;
    hI2C->Init.GeneralCallMode=I2C_GENERALCALL_DISABLE;     //CR1
    hI2C->Init.NoStretchMode=I2C_NOSTRETCH_DISABLE;         //CR1

    I2CA->CR1&=~I2C_CR1_PE;     //__HAL_I2C_DISABLE(hI2C);
    I2CA->TIMINGR=hI2C->Init.Timing&TIMING_CLEAR_MASK;
    I2CA->OAR1&=~I2C_OAR1_OA1EN;
    if (hI2C->Init.OwnAddress1!=0)
        {
        I2CA->OAR1=I2C_OAR1_OA1EN|hI2C->Init.OwnAddress1;
        if (hI2C->Init.AddressingMode!=I2C_ADDRESSINGMODE_7BIT) I2CA->OAR1|=I2C_OAR1_OA1MODE;
        }

    if (hI2C->Init.AddressingMode==I2C_ADDRESSINGMODE_10BIT) I2CA->CR2=I2C_CR2_ADD10;

    I2CA->CR2|=I2C_CR2_AUTOEND|I2C_CR2_NACK;
    I2CA->OAR2=hI2C->Init.DualAddressMode|hI2C->Init.OwnAddress2|(hI2C->Init.OwnAddress2Masks<<8);
    I2CA->CR1=hI2C->Init.GeneralCallMode|hI2C->Init.NoStretchMode;
    I2CA->CR1|=I2C_CR1_PE;      //__HAL_I2C_ENABLE(hI2C);

    if (I2C_Sem==NULL) I2C_Sem=JOSSemCreate(1); //이 함수가 반복 호출됨
    }

BOOL WINAPI I2C_Lock(VOID) {return JOSSemPend(I2C_Sem, JOS_TICKS_PER_SEC*3)==JOS_ERR_NONE;}
VOID WINAPI I2C_Unlock(VOID) {JOSSemPost(I2C_Sem);}





LOCAL(VOID) I2C_Stop(I2C_TypeDef *I2CA)
    {
    I2CA->ICR=I2C_ISR_STOPF;
    I2CA->CR2 &= ~(I2C_CR2_SADD|I2C_CR2_HEAD10R|I2C_CR2_NBYTES|I2C_CR2_RELOAD|I2C_CR2_RD_WRN);  //I2C_RESET_CR2(hI2C);
    }



//-----------------------------------------------------------------------------
//      I2C Master가 SCL라인을 Low잡고 있을 때가 가끔 생김 (원인은 모름)
//      이렇게 하면 복구됨
//-----------------------------------------------------------------------------
LOCAL(VOID) I2C_CheckLineBusy(I2C_TypeDef *I2CA)
    {
    if (I2CA->ISR & I2C_ISR_BUSY)
        {
        I2CA->CR1&=~I2C_CR1_PE;
        I2CA->CR1|=I2C_CR1_PE;
        Sleep(20);
        Printf("SENSOR: Recovered I2C BUSY"CRLF);
        }
    }



LOCAL(VOID) I2C_SetCR2(I2C_TypeDef *I2CA, int DevAddr, int Size, DWORD ModeRequest)
    {
    DWORD Dw;

    Dw=I2CA->CR2;
    Dw&=~(I2C_CR2_SADD|I2C_CR2_NBYTES|I2C_CR2_RELOAD|I2C_CR2_AUTOEND|I2C_CR2_RD_WRN|I2C_CR2_START|I2C_CR2_STOP);
    Dw|=(DevAddr&I2C_CR2_SADD)|((Size<<16)&I2C_CR2_NBYTES)|ModeRequest;
    I2CA->CR2=Dw;
    }


LOCAL(int) I2C_TransferConfig(I2C_TypeDef *I2CA, int DevAddr, int Size, DWORD Request)
    {
    int Mode=I2C_AUTOEND_MODE;

    if (Size>255)
        {
        Size=255;
        Mode=I2C_RELOAD_MODE;
        }
    I2C_SetCR2(I2CA, DevAddr, Size, Mode|Request);
    return Size;
    }



//-----------------------------------------------------------------------------
//      HAL_TIMEOUT 으로 리턴하는 경우 I2C가 계속 BUSY가 되는 버그를 수정함 (2020-11-16)
//-----------------------------------------------------------------------------
LOCAL(HAL_StatusTypeDef) I2C_IsAcknowledgeFailed(I2C_TypeDef *I2CA, DWORD Timeout)
    {
    DWORD Time;
    HAL_StatusTypeDef Rslt=HAL_OK;

    Time=GetTickCount();
    if (I2CA->ISR & I2C_FLAG_AF)                //Acknowledge failure received flag
        {
        Rslt=HAL_ERROR;
        while ((I2CA->ISR & I2C_ISR_STOPF)==0)  //STOP detection flag
            {
            if (GetTickCount()-Time>Timeout) {Rslt=HAL_TIMEOUT; break;}
            }
        I2CA->ICR=I2C_FLAG_AF;
        I2CA->ICR=I2C_FLAG_STOPF;

        //I2C_Flush_TXDR();
        if (I2CA->ISR & I2C_FLAG_TXIS) I2CA->TXDR=0;
        I2CA->ISR|=I2C_FLAG_TXE;

        //I2C_RESET_CR2(hi2c);  //I2C_Stop()에서 처리됨
        }
    return Rslt;
    }


//원이름: I2C_WaitOnTXISFlagUntilTimeout
LOCAL(HAL_StatusTypeDef) I2C_WaitOnTXISFlag(I2C_TypeDef *I2CA, DWORD Timeout)
    {
    DWORD Time;
    HAL_StatusTypeDef Rslt=HAL_OK;

    Time=GetTickCount();
    while ((I2CA->ISR & I2C_FLAG_TXIS)==0)      //Transmit interrupt status
        {
        if ((Rslt=I2C_IsAcknowledgeFailed(I2CA, Timeout))!=HAL_OK) break;
        if (GetTickCount()-Time>Timeout) {Rslt=HAL_TIMEOUT; break;}
        }
    return Rslt;
    }




//원이름: I2C_WaitOnFlagUntilTimeout
LOCAL(HAL_StatusTypeDef) I2C_WaitOnFlag(I2C_TypeDef *I2CA, DWORD Flag, DWORD Timeout)
    {
    DWORD Time;
    HAL_StatusTypeDef Rslt=HAL_OK;

    Time=GetTickCount();
    while ((I2CA->ISR & Flag)==0)
        {
        if (GetTickCount()-Time>Timeout) {Rslt=HAL_TIMEOUT; break;}
        }
    return Rslt;
    }



//원이름: I2C_WaitOnSTOPFlagUntilTimeout
LOCAL(HAL_StatusTypeDef) I2C_WaitOnSTOPFlag(I2C_TypeDef *I2CA, DWORD Timeout)
    {
    DWORD Time;
    HAL_StatusTypeDef Rslt=HAL_OK;

    Time=GetTickCount();
    while ((I2CA->ISR & I2C_ISR_STOPF)==0)
        {
        if ((Rslt=I2C_IsAcknowledgeFailed(I2CA, Timeout))!=HAL_OK) break;
        if (GetTickCount()-Time>Timeout) {Rslt=HAL_TIMEOUT; break;}
        }
    return Rslt;
    }



//HAL_I2C_Master_Transmit
BOOL WINAPI I2C_MasterTx(int I2C_Ch, int DevAddr, LPCBYTE lpData, int Size, int Timeout)
    {
    int BlockSize;
    I2C_TypeDef *I2CA;
    HAL_StatusTypeDef Rslt=HAL_TIMEOUT;

    I2CA=GetI2cCtrlAddr(I2C_Ch);
    I2C_CheckLineBusy(I2CA);
    BlockSize=I2C_TransferConfig(I2CA, DevAddr, Size, I2C_GENERATE_START_WRITE);
    Size-=BlockSize;

    for (;;)
        {
        if (I2C_WaitOnTXISFlag(I2CA, Timeout)!=HAL_OK) goto ProcExit;
        I2CA->TXDR=*lpData++;
        if (--BlockSize==0)
            {
            if (Size==0) break;
            if (I2C_WaitOnFlag(I2CA, I2C_ISR_TCR, Timeout)!=HAL_OK) goto ProcExit;
            BlockSize=I2C_TransferConfig(I2CA, DevAddr, Size, I2C_NO_STARTSTOP);
            Size-=BlockSize;
            }
        }

    Rslt=I2C_WaitOnSTOPFlag(I2CA, Timeout);

    ProcExit:
    I2C_Stop(I2CA);
    return Rslt==HAL_OK;
    }



//HAL_I2C_Master_Receive
BOOL WINAPI I2C_MasterRx(int I2C_Ch, int DevAddr, LPBYTE lpData, int Size, int Timeout)
    {
    int BlockSize;
    HAL_StatusTypeDef Rslt=HAL_TIMEOUT;
    I2C_TypeDef *I2CA;

    I2CA=GetI2cCtrlAddr(I2C_Ch);
    I2C_CheckLineBusy(I2CA);
    BlockSize=I2C_TransferConfig(I2CA, DevAddr, Size, I2C_GENERATE_START_READ);
    Size-=BlockSize;

    for (;;)
        {
        if (I2C_WaitOnFlag(I2CA, I2C_ISR_RXNE, Timeout)!=HAL_OK) goto ProcExit;
        *lpData++=I2CA->RXDR;
        if (--BlockSize==0)
            {
            if (Size==0) break;
            if (I2C_WaitOnFlag(I2CA, I2C_ISR_TCR, Timeout)!=HAL_OK) goto ProcExit;
            BlockSize=I2C_TransferConfig(I2CA, DevAddr, Size, I2C_NO_STARTSTOP);
            Size-=BlockSize;
            }
        }

    Rslt=I2C_WaitOnSTOPFlag(I2CA, I2C_TIMEOUT_STOPF);

    ProcExit:
    I2C_Stop(I2CA);
    return Rslt==HAL_OK;
    }




LOCAL(BOOL) I2C_SetMemAddr(I2C_TypeDef *I2CA, UINT MemAddr, UINT MemAddSize, UINT Timeout)
    {
    BOOL Rslt=FALSE;

    if (I2C_WaitOnTXISFlag(I2CA, Timeout)!=HAL_OK) goto ProcExit;
    if (MemAddSize!=I2C_MEMADD_SIZE_8BIT)
        {
        I2CA->TXDR=MemAddr>>8;
        if (I2C_WaitOnTXISFlag(I2CA, Timeout)!=HAL_OK) goto ProcExit;
        }
    I2CA->TXDR=MemAddr&0xFF;
    Rslt++;

    ProcExit:
    return Rslt;
    }




//-----------------------------------------------------------------------------
//      MemAddSize : I2C_MEMADD_SIZE_8BIT / I2C_MEMADD_SIZE_16BIT
//      원이름: HAL_I2C_Mem_Read
//-----------------------------------------------------------------------------
BOOL WINAPI I2C_MemRead(int I2C_Ch, UINT DevAddr, UINT MemAddr, UINT MemAddSize, LPBYTE lpData, UINT Size, DWORD Timeout)
    {
    int Rslt=FALSE, BlockSize;
    I2C_TypeDef *I2CA;

    I2CA=GetI2cCtrlAddr(I2C_Ch);
    I2C_CheckLineBusy(I2CA);

    //if (I2C_RequestMemoryRead(I2CA, DevAddr, MemAddr, MemAddSize, Timeout)!=HAL_OK) goto ProcExit;
    I2C_SetCR2(I2CA, DevAddr, MemAddSize, I2C_SOFTEND_MODE|I2C_GENERATE_START_WRITE);
    if (I2C_SetMemAddr(I2CA, MemAddr, MemAddSize, Timeout)==FALSE) goto ProcExit;
    if (I2C_WaitOnFlag(I2CA, I2C_ISR_TC, Timeout)!=HAL_OK) goto ProcExit;

    BlockSize=I2C_TransferConfig(I2CA, DevAddr, Size, I2C_GENERATE_START_READ);
    Size-=BlockSize;
    for (;;)
        {
        if (I2C_WaitOnFlag(I2CA, I2C_ISR_RXNE, Timeout)!=HAL_OK) goto ProcExit;
        *lpData++=I2CA->RXDR;
        if (--BlockSize==0)
            {
            if (Size==0) break;
            if (I2C_WaitOnFlag(I2CA, I2C_ISR_TCR, Timeout)!=HAL_OK) goto ProcExit;
            BlockSize=I2C_TransferConfig(I2CA, DevAddr, Size, I2C_NO_STARTSTOP);
            Size-=BlockSize;
            }
        }

    Rslt=I2C_WaitOnSTOPFlag(I2CA, I2C_TIMEOUT_STOPF)==HAL_OK;

    ProcExit:
    I2C_Stop(I2CA);
    return Rslt;
    }




//-----------------------------------------------------------------------------
//      원이름: HAL_I2C_Mem_Write
//-----------------------------------------------------------------------------
BOOL WINAPI I2C_MemWrite(int I2C_Ch, UINT DevAddr, UINT MemAddr, UINT MemAddSize, LPCBYTE lpData, UINT Size, DWORD Timeout)
    {
    int Rslt=FALSE, BlockSize;
    I2C_TypeDef *I2CA;

    I2CA=GetI2cCtrlAddr(I2C_Ch);
    I2C_CheckLineBusy(I2CA);

    //if (I2C_RequestMemoryWrite(I2CA, DevAddr, MemAddr, MemAddSize, Timeout)!=HAL_OK) goto ProcExit;
    I2C_SetCR2(I2CA, DevAddr, MemAddSize, I2C_RELOAD_MODE|I2C_GENERATE_START_WRITE);
    if (I2C_SetMemAddr(I2CA, MemAddr, MemAddSize, Timeout)==FALSE) goto ProcExit;
    if (I2C_WaitOnFlag(I2CA, I2C_ISR_TCR, Timeout)!=HAL_OK) goto ProcExit;

    BlockSize=I2C_TransferConfig(I2CA, DevAddr, Size, I2C_NO_STARTSTOP);
    Size-=BlockSize;
    for (;;)
        {
        if (I2C_WaitOnTXISFlag(I2CA, Timeout)!=HAL_OK) goto ProcExit;
        I2CA->TXDR=*lpData++;
        if (--BlockSize==0)
            {
            if (Size==0) break;
            if (I2C_WaitOnFlag(I2CA, I2C_ISR_TCR, Timeout)!=HAL_OK) goto ProcExit;
            BlockSize=I2C_TransferConfig(I2CA, DevAddr, Size, I2C_NO_STARTSTOP);
            Size-=BlockSize;
            }
        }

    Rslt=I2C_WaitOnSTOPFlag(I2CA, I2C_TIMEOUT_STOPF)==HAL_OK;

    ProcExit:
    I2C_Stop(I2CA);
    return Rslt;
    }




//-----------------------------------------------------------------------------
//      원이름: HAL_I2CEx_ConfigAnalogFilter
//-----------------------------------------------------------------------------
VOID WINAPI I2C_ConfigAnalogFilter(int I2C_Ch, UINT AnalogFilter)
    {
    I2C_TypeDef *I2CA;

    I2CA=GetI2cCtrlAddr(I2C_Ch);
    I2CA->CR1&=~I2C_CR1_PE;     //__HAL_I2C_DISABLE(hI2C);
    I2CA->CR1&=~I2C_CR1_ANFOFF;
    I2CA->CR1|=AnalogFilter;
    I2CA->CR1|=I2C_CR1_PE;      //__HAL_I2C_ENABLE(hI2C);
    }




//-----------------------------------------------------------------------------
//      원이름: HAL_I2CEx_ConfigDigitalFilter
//-----------------------------------------------------------------------------
VOID WINAPI I2C_ConfigDigitalFilter(int I2C_Ch, UINT DigitalFilter)
    {
    I2C_TypeDef *I2CA;

    I2CA=GetI2cCtrlAddr(I2C_Ch);
    I2CA->CR1&=~I2C_CR1_PE;     //__HAL_I2C_DISABLE(hI2C);
    I2CA->CR1&=~I2C_CR1_DNF;
    I2CA->CR1|=DigitalFilter<<8;
    I2CA->CR1|=I2C_CR1_PE;      //__HAL_I2C_ENABLE(hI2C);
    }




//-----------------------------------------------------------------------------
//      원이름: HAL_I2C_IsDeviceReady
//-----------------------------------------------------------------------------
BOOL WINAPI I2C_IsDeviceReady(int I2C_Ch, UINT DevAddress, UINT Trials, UINT Timeout)
    {
    int  Rslt=FALSE, Cnt=0;
    UINT TickStart;
    I2C_TypeDef *I2CA;
    I2C_HandleTypeDef *hI2c;

    hI2c=GetI2cHandle(I2C_Ch);
    I2CA=hI2c->Instance;

    if (I2CA->ISR & I2C_FLAG_BUSY) goto ProcExit;

    TickStart=HAL_GetTick();
    for (;;)
        {
        I2CA->CR2=I2C_GENERATE_START(hI2c->Init.AddressingMode, DevAddress);

        while ((I2CA->ISR & I2C_FLAG_STOPF)==0 && (I2CA->ISR & I2C_FLAG_AF)==0)
            {
            if (HAL_GetTick()-TickStart>Timeout) goto ProcExit;
            }

        if ((I2CA->ISR & I2C_FLAG_AF)==0)
            {
            if (I2C_WaitOnFlag(I2CA, I2C_FLAG_STOPF, Timeout)==FALSE) goto ProcExit;
            I2CA->ICR=I2C_FLAG_STOPF;
            Rslt++;
            goto ProcExit;
            }

        if (I2C_WaitOnFlag(I2CA, I2C_FLAG_STOPF, Timeout)==FALSE) goto ProcExit;

        I2CA->ICR=I2C_FLAG_AF;
        I2CA->ICR=I2C_FLAG_STOPF;

        if (++Cnt==Trials)
            {
            I2CA->CR2|=I2C_CR2_STOP;
            if (I2C_WaitOnFlag(I2CA, I2C_FLAG_STOPF, Timeout)==FALSE) goto ProcExit;
            I2CA->ICR=I2C_FLAG_STOPF;
            break;
            }
        }

    ProcExit:
    return Rslt;
    }


#endif //HAL_I2C_MODULE_ENABLED





#ifdef HAL_SPI_MODULE_ENABLED
///////////////////////////////////////////////////////////////////////////////
//              SPI
///////////////////////////////////////////////////////////////////////////////


typedef struct _SPI_HANDLE
    {
    SPI_TypeDef *Instance;
    JOS_EVENT *RxQ;
    JOS_EVENT *TxQ;
    SPI_InitTypeDef Init;
    UINT ErrorCode;
    } SPI_HANDLE;


static SPI_HANDLE Spi1Handle;
static SPI_HANDLE Spi2Handle;



LOCAL(VOID) DisableSpiWhen1LineMode(SPI_HANDLE *hSpi, SPI_TypeDef *SPIx)
    {
    if (hSpi->Init.Mode==SPI_MODE_MASTER &&
        (hSpi->Init.Direction==SPI_DIRECTION_1LINE ||
         hSpi->Init.Direction==SPI_DIRECTION_2LINES_RXONLY))
        SPIx->CR1&=~SPI_CR1_SPE;   //__HAL_SPI_DISABLE(hspi);
    }




LOCAL(BOOL) SPI_EndRxTxTransaction(SPI_HANDLE *hSpi, SPI_TypeDef *SPIx, UINT Timeout, UINT StartTick)
    {
    BOOL Rslt=FALSE;

    while ((SPIx->SR & SPI_SR_FTLVL)!=SPI_FTLVL_EMPTY)
        {
        if (HAL_GetTick()-StartTick>=Timeout) goto ErExit;
        }

    while (SPIx->SR & SPI_SR_BSY)
        {
        if (HAL_GetTick()-StartTick>=Timeout) goto ErExit;
        }

    while ((SPIx->SR & SPI_SR_FRLVL)!=SPI_FRLVL_EMPTY)
        {
        *(__IO BYTE*)&SPIx->DR;

        if (HAL_GetTick()-StartTick>=Timeout)
            {
            ErExit:
            SPIx->CR2&=~(SPI_IT_TXE|SPI_IT_RXNE|SPI_IT_ERR);    //__HAL_SPI_DISABLE_IT()
            DisableSpiWhen1LineMode(hSpi, SPIx);
            goto ProcExit;
            }
        }
    Rslt++;

    ProcExit:
    return Rslt;
    }



//-----------------------------------------------------------------------------
//      8Bit SPI 송수신 (HAL_SPI_TransmitReceive() 에서 8비트 처리만 떼어냄)
//-----------------------------------------------------------------------------
BOOL WINAPI SPI_TransmitReceive8(int SPI_Ch, LPCBYTE TxData, int TxSize, LPBYTE RxData, int RxSize, int RecvSkip, UINT Timeout)
    {
    int Rslt=FALSE, TxAllowed=1, Data;
    DWORD StartTick;
    SPI_TypeDef *SPIx;
    SPI_HANDLE  *hSpi;

    hSpi=(SPI_Ch==SPI_CH1) ? &Spi1Handle:&Spi2Handle;
    SPIx=hSpi->Instance;
    StartTick=HAL_GetTick();

    SPIx->CR2|=SPI_RXFIFO_THRESHOLD;
    SPIx->CR1|=SPI_CR1_SPE;             //__HAL_SPI_ENABLE(hSpi);

    if (hSpi->Init.Mode==SPI_MODE_SLAVE || TxSize==1)
        {
        *(__IO BYTE*)&SPIx->DR=*TxData++;
        TxSize--;
        TxAllowed=0;
        }

    while (TxSize>0 || RxSize>0)
        {
        if (TxAllowed!=0 && (SPIx->SR & SPI_SR_TXE)!=0)
            {
            Data=0;     //데이터를 수신하기 위한 더미, 전송해야 클럭이 발생함
            if (TxSize>0) {Data=*TxData++; TxSize--;}
            *(__IO BYTE*)&SPIx->DR=Data;
            TxAllowed=0;
            }

        if (SPIx->SR & SPI_SR_RXNE)
            {
            Data=SPIx->DR;
            //if (RecvSkip>0) RecvSkip--;
            //else
            if (RxSize>0) {*RxData++=Data; RxSize--;}
            TxAllowed=1;
            }

        if (HAL_GetTick()-StartTick>=Timeout) goto ProcExit;
        }

    Rslt=SPI_EndRxTxTransaction(hSpi, SPIx, Timeout, StartTick);

    ProcExit:
    return Rslt;
    }



//원이름 HAL_SPI_Receive
BOOL WINAPI SPI_Receive8(int SPI_Ch, LPBYTE RxBuff, UINT Size, UINT Timeout)
    {
    int  Rslt=FALSE;
    UINT StartTick;
    SPI_TypeDef *SPIx;
    SPI_HANDLE  *hSpi;

    hSpi=(SPI_Ch==SPI_CH1) ? &Spi1Handle:&Spi2Handle;
    SPIx=hSpi->Instance;

    if (hSpi->Init.Mode==SPI_MODE_MASTER && hSpi->Init.Direction==SPI_DIRECTION_2LINES)
        {
        return SPI_TransmitReceive8(SPI_Ch, NULL, 0, RxBuff, Size, 0, Timeout);
        }

    StartTick=HAL_GetTick();
    SPIx->CR2|=SPI_RXFIFO_THRESHOLD;

    if (hSpi->Init.Direction==SPI_DIRECTION_1LINE) SPIx->CR1&=~SPI_CR1_BIDIOE;  //SPI_1LINE_RX(hSpi);
    SPIx->CR1|=SPI_CR1_SPE;                                                     //__HAL_SPI_ENABLE(hSpi);

    while (Size>0)
        {
        if (SPIx->SR & SPI_FLAG_RXNE)   //__HAL_SPI_GET_FLAG()
            {
            *RxBuff++=*(__IO BYTE*)&SPIx->DR;
            Size--;
            }
        else if (HAL_GetTick()-StartTick>=Timeout) goto ProcExit;
        }

    //SPI_EndRxTransaction()
    DisableSpiWhen1LineMode(hSpi, SPIx);

    while (SPIx->SR & SPI_SR_BSY)
        {
        if (HAL_GetTick()-StartTick>=Timeout) goto ErExit;
        }

    if (hSpi->Init.Mode==SPI_MODE_MASTER &&
        (hSpi->Init.Direction==SPI_DIRECTION_1LINE ||
         hSpi->Init.Direction==SPI_DIRECTION_2LINES_RXONLY))
        {
        //SPI_WaitFifoStateUntilTimeout()
        while ((SPIx->SR & SPI_SR_FRLVL)!=SPI_FRLVL_EMPTY)
            {
            *(__IO BYTE*)&SPIx->DR;

            if (HAL_GetTick()-StartTick>=Timeout)
                {
                ErExit:
                SPIx->CR2&=~(SPI_IT_TXE|SPI_IT_RXNE|SPI_IT_ERR);
                DisableSpiWhen1LineMode(hSpi, SPIx);
                goto ProcExit;
                }
            }
        }
    Rslt++;

    ProcExit:
    return Rslt;
    }




//원이름: HAL_SPI_Transmit
BOOL WINAPI SPI_Transmit8(int SPI_Ch, LPCBYTE TxBuff, UINT Size, UINT Timeout)
    {
    int  Rslt=FALSE;
    UINT StartTick;
    SPI_TypeDef *SPIx;
    SPI_HANDLE  *hSpi;

    hSpi=(SPI_Ch==SPI_CH1) ? &Spi1Handle:&Spi2Handle;
    SPIx=hSpi->Instance;
    StartTick=HAL_GetTick();

    if (hSpi->Init.Direction==SPI_DIRECTION_1LINE) SPIx->CR1|=SPI_CR1_BIDIOE;   //SPI_1LINE_TX(hSpi);
    SPIx->CR1|=SPI_CR1_SPE;                                                     //__HAL_SPI_ENABLE(hSpi);

    while (Size>0)
        {
        if (SPIx->SR & SPI_FLAG_TXE)
            {
            *(__IO BYTE*)&SPIx->DR=*TxBuff++;
            Size--;
            }
        else if (HAL_GetTick()-StartTick>=Timeout) goto ProcExit;
        }

    if (SPI_EndRxTxTransaction(hSpi, SPIx, Timeout, StartTick)==FALSE) goto ProcExit;

    if (hSpi->Init.Direction==SPI_DIRECTION_2LINES)
        {
        __IO UINT T; T=SPIx->DR; T=SPIx->SR; UNUSED(T); //__HAL_SPI_CLEAR_OVRFLAG
        }
    Rslt++;

    ProcExit:
    return Rslt;
    }



//-----------------------------------------------------------------------------
//      SPI 초기화
//-----------------------------------------------------------------------------
VOID WINAPI SPI_InitMaster(int SPI_Ch, int BaudPrescaler)
    {
    SPI_TypeDef *SPIx;
    SPI_HANDLE *hSpi;

    if (SPI_Ch==SPI_CH1)
        {
        __HAL_RCC_SPI1_CLK_ENABLE();
        hSpi=&Spi1Handle;
        SPIx=SPI1;
        }
    else{
        __HAL_RCC_SPI2_CLK_ENABLE();
        hSpi=&Spi2Handle;
        SPIx=SPI2;
        }

    hSpi->Instance=SPIx;
    hSpi->Init.BaudRatePrescaler=BaudPrescaler; //SPI_BAUDRATEPRESCALER_256;
    hSpi->Init.Direction=SPI_DIRECTION_2LINES;
    hSpi->Init.CLKPhase=SPI_PHASE_1EDGE;
    hSpi->Init.CLKPolarity=SPI_POLARITY_LOW;
    hSpi->Init.CRCCalculation=SPI_CRCCALCULATION_DISABLE;
    hSpi->Init.CRCPolynomial=7;
    hSpi->Init.DataSize=SPI_DATASIZE_8BIT;
    hSpi->Init.FirstBit=SPI_FIRSTBIT_MSB;
    hSpi->Init.NSS=SPI_NSS_SOFT;
    hSpi->Init.TIMode=SPI_TIMODE_DISABLE;
    hSpi->Init.NSSPMode=SPI_NSS_PULSE_DISABLE;
    hSpi->Init.CRCLength=SPI_CRC_LENGTH_8BIT;
    hSpi->Init.Mode=SPI_MODE_MASTER;

    SPIx->CR1&=~SPI_CR1_SPE;  //__HAL_SPI_DISABLE()

    SPIx->CR1=hSpi->Init.Mode|hSpi->Init.Direction|
        hSpi->Init.CLKPolarity|hSpi->Init.CLKPhase|(hSpi->Init.NSS&SPI_CR1_SSM)|
        hSpi->Init.BaudRatePrescaler|hSpi->Init.FirstBit|hSpi->Init.CRCCalculation;

    SPIx->CR2=(((hSpi->Init.NSS>>16)&SPI_CR2_SSOE)|hSpi->Init.TIMode|hSpi->Init.NSSPMode|hSpi->Init.DataSize)|SPI_RXFIFO_THRESHOLD_QF;

    #if defined(SPI_I2SCFGR_I2SMOD)
    SPIx->I2SCFGR&=~SPI_I2SCFGR_I2SMOD;
    #endif
    }





#endif //HAL_SPI_MODULE_ENABLED



#ifdef HAL_TIM_MODULE_ENABLED
///////////////////////////////////////////////////////////////////////////////
//                      Timer/PWM
///////////////////////////////////////////////////////////////////////////////



//-----------------------------------------------------------------------------
//hTmr->Instance ... TIM1/TIM2/TIM3/TIM14/TIM16/TIM17
//hTmr->Init.CounterMode
//      TIM_COUNTERMODE_UP
//      TIM_COUNTERMODE_DOWN
//      TIM_COUNTERMODE_CENTERALIGNED1
//      TIM_COUNTERMODE_CENTERALIGNED2
//      TIM_COUNTERMODE_CENTERALIGNED3
//hTmr->Init.ClockDivision
//      TIM_CLOCKDIVISION_DIV1
//      TIM_CLOCKDIVISION_DIV2
//      TIM_CLOCKDIVISION_DIV4
//-----------------------------------------------------------------------------
VOID WINAPI TIM_PWM_Init(TIM_HandleTypeDef *hTmr)
    {
    DWORD CR1;
    TIM_TypeDef *TIMx;

    TIMx=hTmr->Instance;

    CR1=TIMx->CR1;
    if (IS_TIM_COUNTER_MODE_SELECT_INSTANCE(TIMx))      //TIM1 or TIM2 or TIM3
        {
        CR1&=~(TIM_CR1_DIR|TIM_CR1_CMS);
        CR1|=hTmr->Init.CounterMode;
        }
    if (IS_TIM_CLOCK_DIVISION_INSTANCE(TIMx))           //TIM1 or TIM2 or TIM3 or TIM14 or TIM16 or TIM17
        {
        CR1&=~TIM_CR1_CKD;
        CR1|=hTmr->Init.ClockDivision;
        }
    TIMx->CR1=CR1;

    TIMx->ARR=hTmr->Init.Period;
    TIMx->PSC=hTmr->Init.Prescaler;

    if (IS_TIM_REPETITION_COUNTER_INSTANCE(TIMx))       //TIM1,TIM15,TIM16,TIM17
        TIMx->RCR=hTmr->Init.RepetitionCounter;

    TIMx->EGR=TIM_EGR_UG;
    }




//-----------------------------------------------------------------------------
//      HAL_TIM_PWM_ConfigChannel
//-----------------------------------------------------------------------------
VOID WINAPI TIM_PWM_ConfigChannel(TIM_HandleTypeDef *hTmr, TIM_OC_InitTypeDef *OCCfg, int Channel)
    {
    DWORD CCMR1, CCER, CR2;
    TIM_TypeDef *TIMx;

    TIMx=hTmr->Instance;
    switch (Channel)
        {
        case TIM_CHANNEL_1:     //TIM1/TIM2/TIM3/TIM14/TIM16/TIM17
            TIMx->CCER&=~TIM_CCER_CC1E;
            CCER=TIMx->CCER;
            CR2=TIMx->CR2;
            CCMR1=TIMx->CCMR1;

            CCMR1&=~TIM_CCMR1_OC1M;
            CCMR1&=~TIM_CCMR1_CC1S;
            CCMR1|=OCCfg->OCMode;

            CCER&=~TIM_CCER_CC1P;
            CCER|=OCCfg->OCPolarity;

            if (TIMx==TIM1 || TIMx==TIM16 || TIMx==TIM17)
                {
                CCER&=~TIM_CCER_CC1NP;
                CCER|=OCCfg->OCNPolarity;
                CCER&=~TIM_CCER_CC1NE;
                }

            if (IS_TIM_BREAK_INSTANCE(TIMx))
                {
                CR2&=~TIM_CR2_OIS1;
                CR2&=~TIM_CR2_OIS1N;
                CR2|=OCCfg->OCIdleState;
                CR2|=OCCfg->OCNIdleState;
                }
            TIMx->CR2=CR2;
            TIMx->CCMR1=CCMR1;
            TIMx->CCR1=OCCfg->Pulse;
            TIMx->CCER=CCER;

            TIMx->CCMR1|=TIM_CCMR1_OC1PE;
            TIMx->CCMR1&=~TIM_CCMR1_OC1FE;
            TIMx->CCMR1|=OCCfg->OCFastMode;
            break;

        case TIM_CHANNEL_2:     //TIM1/TIM2/TIM3
            TIMx->CCER&=~TIM_CCER_CC2E;

            CCER=TIMx->CCER;
            CR2=TIMx->CR2;
            CCMR1=TIMx->CCMR1;

            CCMR1&=~TIM_CCMR1_OC2M;
            CCMR1&=~TIM_CCMR1_CC2S;
            CCMR1|=OCCfg->OCMode<<8;

            CCER&=~TIM_CCER_CC2P;
            CCER|=OCCfg->OCPolarity<<4;

            if (TIMx==TIM1)
                {
                CCER&=~TIM_CCER_CC2NP;
                CCER|=OCCfg->OCNPolarity<<4;
                CCER&=~TIM_CCER_CC2NE;
                }

            if (IS_TIM_BREAK_INSTANCE(TIMx))
                {
                CR2&=~TIM_CR2_OIS2;
                CR2&=~TIM_CR2_OIS2N;
                CR2|=OCCfg->OCIdleState<<2;
                CR2|=OCCfg->OCNIdleState<<2;
                }

            TIMx->CR2=CR2;
            TIMx->CCMR1=CCMR1;
            TIMx->CCR2=OCCfg->Pulse;
            TIMx->CCER=CCER;


            TIMx->CCMR1|=TIM_CCMR1_OC2PE;
            TIMx->CCMR1&=~TIM_CCMR1_OC2FE;
            TIMx->CCMR1|=OCCfg->OCFastMode<<8;
            break;

        case TIM_CHANNEL_3:     //TIM1/TIM2/TIM3
            TIMx->CCER&=~TIM_CCER_CC3E;

            CCER=TIMx->CCER;
            CR2=TIMx->CR2;
            CCMR1=TIMx->CCMR2;

            CCMR1&=~TIM_CCMR2_OC3M;
            CCMR1&=~TIM_CCMR2_CC3S;
            CCMR1|=OCCfg->OCMode;

            CCER&=~TIM_CCER_CC3P;
            CCER|=OCCfg->OCPolarity<<8;

            if (TIMx==TIM1)
                {
                CCER&=~TIM_CCER_CC3NP;
                CCER|=OCCfg->OCNPolarity<<8;
                CCER&=~TIM_CCER_CC3NE;
                }

            if (IS_TIM_BREAK_INSTANCE(TIMx))
                {
                CR2&=~TIM_CR2_OIS3;
                CR2&=~TIM_CR2_OIS3N;
                CR2|=OCCfg->OCIdleState<<4;
                CR2|=OCCfg->OCNIdleState<<4;
                }

            TIMx->CR2=CR2;
            TIMx->CCMR2=CCMR1;
            TIMx->CCR3=OCCfg->Pulse;
            TIMx->CCER=CCER;


            TIMx->CCMR2|=TIM_CCMR2_OC3PE;
            TIMx->CCMR2&=~TIM_CCMR2_OC3FE;
            TIMx->CCMR2|=OCCfg->OCFastMode;
            break;

        case TIM_CHANNEL_4:     //TIM1/TIM2/TIM3
            TIMx->CCER&=~TIM_CCER_CC4E;

            CCER=TIMx->CCER;
            CR2=TIMx->CR2;

            CCMR1=TIMx->CCMR2;
            CCMR1&=~TIM_CCMR2_OC4M;
            CCMR1&=~TIM_CCMR2_CC4S;

            CCMR1|=OCCfg->OCMode<<8;

            CCER&=~TIM_CCER_CC4P;
            CCER|=OCCfg->OCPolarity<<12;

            if (IS_TIM_BREAK_INSTANCE(TIMx))
                {
                CR2&=~TIM_CR2_OIS4;
                CR2|=OCCfg->OCIdleState<<6;
                }

            TIMx->CR2=CR2;
            TIMx->CCMR2=CCMR1;
            TIMx->CCR4=OCCfg->Pulse;
            TIMx->CCER=CCER;


            TIMx->CCMR2|=TIM_CCMR2_OC4PE;
            TIMx->CCMR2&=~TIM_CCMR2_OC4FE;
            TIMx->CCMR2|=OCCfg->OCFastMode<<8;
            //break;
        }
    }



//-----------------------------------------------------------------------------
//      듀티폭만 바꿈
//-----------------------------------------------------------------------------
VOID WINAPI TIM_PWM_Duty(TIM_HandleTypeDef *hTmr, int Channel, int Permil)
    {
    int Pulse;
    TIM_TypeDef *TIMx;

    TIMx=hTmr->Instance;

    Pulse=hTmr->Init.Period*Permil/1000;
    switch (Channel)
        {
        case TIM_CHANNEL_1: TIMx->CCR1=Pulse; break;
        case TIM_CHANNEL_2: TIMx->CCR2=Pulse; break;
        case TIM_CHANNEL_3: TIMx->CCR3=Pulse; break;
        case TIM_CHANNEL_4: TIMx->CCR4=Pulse; //break;
        }
    }



//-----------------------------------------------------------------------------
//      도중에 출력 극성만 설정
//-----------------------------------------------------------------------------
VOID WINAPI TIM_PWM_SetOutPolarity(TIM_HandleTypeDef *hTmr, int Channel, int OCPolarity)
    {
    DWORD CCER;
    TIM_TypeDef *TIMx;

    TIMx=hTmr->Instance;
    CCER=TIMx->CCER;
    switch (Channel)
        {
        case TIM_CHANNEL_1: CCER&=~TIM_CCER_CC1P; CCER|=OCPolarity; break;      //TIM1/TIM2/TIM3/TIM14/TIM16/TIM17
        case TIM_CHANNEL_2: CCER&=~TIM_CCER_CC2P; CCER|=OCPolarity<<4; break;   //TIM1/TIM2/TIM3
        case TIM_CHANNEL_3: CCER&=~TIM_CCER_CC3P; CCER|=OCPolarity<<8; break;   //TIM1/TIM2/TIM3
        case TIM_CHANNEL_4: CCER&=~TIM_CCER_CC4P; CCER|=OCPolarity<<12; //break; //TIM1/TIM2/TIM3
        }
    TIMx->CCER=CCER;
    }



//-----------------------------------------------------------------------------
//      HAL_TIM_PWM_Start()
//
//      TIM1,TIM2,TIM3: TIM_CHANNEL_1~4
//      TIM16,TIM17: TIM_CHANNEL_1 한개 만 가능
//-----------------------------------------------------------------------------
VOID WINAPI TIM_PWM_Start(TIM_HandleTypeDef *hTmr, int Channel)
    {
    TIM_TypeDef *TIMx;

    TIMx=hTmr->Instance;

    //HAL_TIM_CCxChannelCmd()
    TIMx->CCER|=TIM_CCx_ENABLE<<Channel;//Capture/Compare output enable

    if (IS_TIM_BREAK_INSTANCE(TIMx))    //TIM1 or TIM16 or TIM17
        TIMx->BDTR|=TIM_BDTR_MOE;       //__HAL_TIM_MOE_ENABLE()

    TIMx->CR1|=TIM_CR1_CEN;             //__HAL_TIM_ENABLE()
    }




//-----------------------------------------------------------------------------
//      HAL_TIM_PWM_Stop()
//-----------------------------------------------------------------------------
VOID WINAPI TIM_PWM_Stop(TIM_HandleTypeDef *hTmr, int Channel)
    {
    TIM_TypeDef *TIMx;

    TIMx=hTmr->Instance;
    TIMx->CCER&=~(TIM_CCER_CC1E<<Channel);

    if (IS_TIM_BREAK_INSTANCE(TIMx))    //TIM1 or TIM16 or TIM17
        {
        //__HAL_TIM_MOE_DISABLE(hTmr);
        if ((TIMx->CCER & TIM_CCER_CCxE_MASK)==0 &&
            (TIMx->CCER & TIM_CCER_CCxNE_MASK)==0) TIMx->BDTR&=~TIM_BDTR_MOE;
        }

    //__HAL_TIM_DISABLE(hTmr);
    if ((TIMx->CCER & TIM_CCER_CCxE_MASK)==0 &&
        (TIMx->CCER & TIM_CCER_CCxNE_MASK)==0) TIMx->CR1&=~TIM_CR1_CEN;
    }



#ifdef TIM6     //STM32F042는 없음
//-----------------------------------------------------------------------------
//      SysTick 대신에 사용할 경우 이함수 호출
//      TIM6_DAC_IRQHandler() 로 인터럽트 발생
//-----------------------------------------------------------------------------
//HAL_StatusTypeDef HAL_InitTick(UINT TickPriority)
VOID WINAPI TIM6_InitTick(int Freq)
    {
    RCC_ClkInitTypeDef ClkCfg;
    UINT Timclock, FLatency;
    TIM_HandleTypeDef TimInst;

    NVIC_SetPriority(TIM6_DAC_IRQn, TICK_INT_PRIORITY); //Configure the TIM6 IRQ priority
    NVIC_EnableIRQ(TIM6_DAC_IRQn);                      //Enable the TIM6 global Interrupt
    __HAL_RCC_TIM6_CLK_ENABLE();                        //=RCC_APB1PeriphClockCmd(RCC_APB1Periph_TIM6, ENABLE);

    HAL_RCC_GetClockConfig(&ClkCfg, &FLatency);         //Get clock configuration

    //TIM6 clock 계산
    Timclock=HAL_RCC_GetPCLK1Freq();
    if (ClkCfg.APB1CLKDivider!=RCC_HCLK_DIV1) Timclock<<=1; //APB1 prescaler

    TimInst.Instance=TIM6;
    TimInst.Init.Period=(1000000/Freq)-1;               //Period = [(TIM6CLK/1000) - 1]. to have a (1/1000) s time base.
    TimInst.Init.Prescaler=(Timclock/1000000)-1;        //Prescaler to have a 1MHz counter clock. Compute the prescaler value to have TIM6 counter clock equal to 1MHz
    //TimInst.Init.ClockDivision=0;                     //TIM6에서는 안씀
    //TimInst.Init.CounterMode=TIM_COUNTERMODE_UP;      //TIM6에서는 안씀
    TIM_PWM_Init(&TimInst);

    //HAL_TIM_Base_Start_IT(&TimInst)
    TimInst.Instance->DIER|=TIM_IT_UPDATE;              //__HAL_TIM_ENABLE_IT(TIM6, TIM_IT_UPDATE); Enable the TIM Update interrupt
    TimInst.Instance->CR1|=TIM_CR1_CEN;                 //TIM_Cmd(TIM6, ENABLE); = __HAL_TIM_ENABLE(TIM6)
    }
#endif


//__HAL_TIM_DISABLE
VOID WINAPI TIM_Disable(TIM_TypeDef *TIMx)
    {
    if ((TIMx->CCER & TIM_CCER_CCxE_MASK)==0 &&
        (TIMx->CCER & TIM_CCER_CCxNE_MASK)==0) TIMx->CR1&=~TIM_CR1_CEN;
    }


VOID WINAPI TIM_Stop(TIM_TypeDef *TIMx)
    {
    TIMx->DIER&=~TIM_IT_UPDATE; //__HAL_TIM_DISABLE_IT(hTim, TIM_IT_UPDATE);
    TIM_Disable(TIMx);          //__HAL_TIM_DISABLE()
    }



VOID WINAPI TIM_Release(TIM_TypeDef *TIMx)
    {
    #ifdef TIM2
    if (TIMx==TIM2)
        {
        TIM_Stop(TIM2);
        __HAL_RCC_TIM2_FORCE_RESET();
        __HAL_RCC_TIM2_RELEASE_RESET();
        }
    #endif

    #ifdef TIM6
    if (TIMx==TIM6)
        {
        TIM_Stop(TIM6);
        __HAL_RCC_TIM6_FORCE_RESET();
        __HAL_RCC_TIM6_RELEASE_RESET();
        }
    #endif
    }



VOID WINAPI TIM_SetPrescalerFreq(TIM_TypeDef *TIMx, UINT Freq)
    {
    TIMx->PSC=DIV_ROUNDUP(SystemCoreClock, Freq)-1;
    }

VOID WINAPI TIM_SetDivisor(TIM_TypeDef *TIMx, UINT Divisor)
    {
    TIMx->ARR=Divisor;
    }




LOCAL(VOID) TIM_BaseSetConfig(TIM_TypeDef*TIMx, TIM_Base_InitTypeDef *TI)
    {
    UINT CR1;

    CR1=TIMx->CR1;

    if (IS_TIM_COUNTER_MODE_SELECT_INSTANCE(TIMx))
        {
        CR1&=~(TIM_CR1_DIR|TIM_CR1_CMS);
        CR1|=TI->CounterMode;
        }

    if (IS_TIM_CLOCK_DIVISION_INSTANCE(TIMx))
        {
        CR1&=~TIM_CR1_CKD;
        CR1|=TI->ClockDivision;
        }

    MODIFY_REG(CR1, TIM_CR1_ARPE, TI->AutoReloadPreload);

    TIMx->CR1=CR1;
    TIMx->ARR=TI->Period;
    TIMx->PSC=TI->Prescaler;

    if (IS_TIM_REPETITION_COUNTER_INSTANCE(TIMx)) TIMx->RCR=TI->RepetitionCounter;
    TIMx->EGR=TIM_EGR_UG;
    }



//-----------------------------------------------------------------------------
//      주어진 타이머를 업카운터 모드, 인터럽트로 설정함
//-----------------------------------------------------------------------------
VOID WINAPI TIM_UpCounterSetup(TIM_TypeDef *TIMx, UINT Freq, UINT Divisor)
    {
    IRQn_Type Irq;
    TIM_HandleTypeDef THT;

    #ifdef TIM2
    if (TIMx==TIM2)
        {
        __HAL_RCC_TIM2_CLK_ENABLE();
        Irq=TIM2_IRQn;
        }
    #endif
    #ifdef TIM6
    else if (TIMx==TIM6)
        {
        __HAL_RCC_TIM6_CLK_ENABLE();
        Irq=TIM6_DAC_IRQn;
        }
    #endif
    #if defined(TIM2) || defined(TIM6)
    else goto ProcExit;
    #endif

    ZeroMem(&THT, sizeof(THT));
    THT.Instance=TIMx;
    THT.Init.AutoReloadPreload=TIM_AUTORELOAD_PRELOAD_ENABLE;
    THT.Init.ClockDivision=TIM_CLOCKDIVISION_DIV1;
    THT.Init.CounterMode=TIM_COUNTERMODE_UP;
    THT.Init.Prescaler=DIV_ROUNDUP(SystemCoreClock, Freq)-1;
    THT.Init.Period=Divisor;
    TIM_BaseSetConfig(TIMx, &THT.Init);

    NVIC_SetPriority(Irq, 3);
    NVIC_EnableIRQ(Irq);

    //TIM_Base_Start_IT(TIMx);
    TIMx->DIER=TIM_IT_UPDATE;       //__HAL_TIM_ENABLE_IT()
    if ((TIMx->SMCR & TIM_SMCR_SMS)!=TIM_SLAVEMODE_TRIGGER) //IS_TIM_SLAVEMODE_TRIGGER_ENABLED
        TIMx->CR1|=TIM_CR1_CEN;     //__HAL_TIM_ENABLE()

    #if defined(TIM2) || defined(TIM6)
    ProcExit:;
    #endif
    }

#endif //HAL_TIM_MODULE_ENABLED






///////////////////////////////////////////////////////////////////////////////
//              FLASH
///////////////////////////////////////////////////////////////////////////////



#ifdef HAL_FLASH_MODULE_ENABLED
//-----------------------------------------------------------------------------
//원이름 FLASH_WaitForLastOperation
//-----------------------------------------------------------------------------
LOCAL(int) FLASH_WaitBusy(DWORD Timeout)
    {
    int Rslt=HAL_OK;
    DWORD Time;

    Time=GetTickCount();
    while (FLASH->SR & FLASH_FLAG_BSY)
        {
        if (GetTickCount()-Time>Timeout) {Rslt=HAL_TIMEOUT; goto ProcExit;}
        }

    if (FLASH->SR & FLASH_FLAG_EOP) FLASH->SR=FLASH_FLAG_EOP;

    if (FLASH->SR & (FLASH_FLAG_WRPERR|FLASH_FLAG_PGERR))
        {
        if (FLASH->SR & FLASH_FLAG_WRPERR) Rslt|=HAL_FLASH_ERROR_WRP;
        if (FLASH->SR & FLASH_FLAG_PGERR)  Rslt|=HAL_FLASH_ERROR_PROG;
        FLASH->SR=FLASH_FLAG_WRPERR|FLASH_FLAG_PGERR;
        }

    ProcExit:
    return Rslt;
    }




//-----------------------------------------------------------------------------
//      2Byte 기록
//-----------------------------------------------------------------------------
LOCAL(int) FLASH_ProgramHalfWord(UINT Address, int Data)
    {
    int Rslt=HAL_ERROR;

    if ((Rslt=FLASH_WaitBusy(FLASH_TIMEOUT_VALUE))==HAL_OK)
        {
        FLASH->CR|=FLASH_CR_PG;
        *(__IO WORD*)Address=Data;
        Rslt=FLASH_WaitBusy(FLASH_TIMEOUT_VALUE);
        FLASH->CR&=~FLASH_CR_PG;
        }
    return Rslt;
    }



//-----------------------------------------------------------------------------
//      FLASH Erase
//-----------------------------------------------------------------------------
LOCAL(int) FLASH_Erase(int EraseType, UINT PageAddress, int NbPages, UINT *lpErrPageAddr)
    {
    int Rslt=HAL_ERROR;
    UINT Addr, EndAddr;

    if (FLASH_WaitBusy(FLASH_TIMEOUT_VALUE)!=HAL_OK) goto ProcExit;
    *lpErrPageAddr=(UINT)-1;

    if (EraseType==FLASH_TYPEERASE_MASSERASE)
        {
        FLASH->CR|=FLASH_CR_MER;
        FLASH->CR|=FLASH_CR_STRT;
        Rslt=FLASH_WaitBusy(FLASH_TIMEOUT_VALUE);
        FLASH->CR&=~FLASH_CR_MER;
        }
    else{
        EndAddr=PageAddress+NbPages*FLASH_PAGE_SIZE;
        //if (PageAddress<FLASH_BASE || EndAddr>FLASH_BANK1_END+1) goto ProcExit;

        for (Addr=PageAddress; Addr<EndAddr; Addr+=FLASH_PAGE_SIZE)
            {
            FLASH->CR|=FLASH_CR_PER;    //Page Erase
            FLASH->AR=Addr;             //Page Address
            FLASH->CR|=FLASH_CR_STRT;
            Rslt=FLASH_WaitBusy(FLASH_TIMEOUT_VALUE);
            FLASH->CR&=~FLASH_CR_PER;

            if (Rslt!=HAL_OK) {*lpErrPageAddr=Addr; break;}
            }
        }

    ProcExit:
    return Rslt;
    }




//-----------------------------------------------------------------------------
//      FLASH 기록
//      인터럽트를 막고 호출하므로 HAL_Delay(), GetTickCount()를 사용하면 안됨
//-----------------------------------------------------------------------------
BOOL WINAPI FLASH_Program(UINT StartAddr, LPCVOID WriteBuff, int WriteBytes)
    {
    BOOL Rslt=FALSE;
    UINT Addr, EndAddr, EraseErrPageAddr;
    CONST WORD *lpW;

    if (StartAddr<FLASH_BASE || StartAddr+WriteBytes>FLASH_BANK1_END+1) goto ProcExit;

    if (FLASH_Erase(FLASH_TYPEERASE_PAGES, StartAddr, (WriteBytes+FLASH_PAGE_SIZE-1)/FLASH_PAGE_SIZE, &EraseErrPageAddr)!=HAL_OK) goto ProcExit;        //FLASH_TYPEERASE_MASSERASE

    EndAddr=StartAddr+WriteBytes;
    lpW=(CONST WORD*)WriteBuff;
    for (Addr=StartAddr; Addr<EndAddr; Addr+=2)
        {
        if (FLASH_ProgramHalfWord(Addr, *lpW++)!=HAL_OK) goto ProcExit;
        }

    lpW=(CONST WORD*)WriteBuff;
    for (Addr=StartAddr; Addr<EndAddr; Addr+=2)
        {
        if (*(__IO WORD*)Addr!=*lpW++) goto ProcExit;
        }
    Rslt=TRUE;

    ProcExit:
    return Rslt;
    }




BOOL WINAPI FLASH_Unlock(VOID)
    {
    BOOL Rslt=FALSE;

    if (FLASH->CR & FLASH_CR_LOCK)
        {
        FLASH->KEYR=FLASH_KEY1;
        FLASH->KEYR=FLASH_KEY2;
        Rslt=TRUE;
        }
    return Rslt;
    }



VOID WINAPI FLASH_Lock(VOID)
    {
    FLASH->CR|=FLASH_CR_LOCK;
    }




//-----------------------------------------------------------------------------
//      INI 데이터를 기록할 때 사용함
//-----------------------------------------------------------------------------
BOOL WINAPI FLASH_WriteIniData(UINT StartAddr, LPCVOID WriteData, int WriteBytes)
    {
    BOOL Rslt;
    JOS_CRITICAL_VAR;

    JOS_ENTER_CRITICAL();
    FLASH_Unlock();
    Rslt=FLASH_Program(StartAddr, WriteData, WriteBytes);
    FLASH_Lock();
    JOS_EXIT_CRITICAL();
    return Rslt;
    }




//-----------------------------------------------------------------------------
//      FLASH 맨뒤에서 부터 지정한 크기를 할당해줌
//-----------------------------------------------------------------------------
LPVOID WINAPI AllocFlash(UINT Size)
    {
    UINT First;
    static UINT FlashBottom=FLASH_BASE+FLASH_SIZE;

    First=FlashBottom;
    for (;;)
        {
        FlashBottom-=FLASH_PAGE_SIZE;
        if (First-FlashBottom>=Size) break;
        }
    return (LPVOID)FlashBottom;
    }


#endif //HAL_FLASH_MODULE_ENABLED



///////////////////////////////////////////////////////////////////////////////
//              ADC
///////////////////////////////////////////////////////////////////////////////
#ifdef HAL_ADC_MODULE_ENABLED

#define ADC_STAB_DELAY_US       1
#define ADC_ENABLE_TIMEOUT      2
#define ADC_DISABLE_TIMEOUT     2
#define ADC_CALIBRATION_TIMEOUT 2
#define ADC_STOP_CONVERSION_TIMEOUT 2
#define ADC_TEMPSENSOR_DELAY_US 10

static ADC_HandleTypeDef AdcHandle;



//-----------------------------------------------------------------------------
//      Adc 초기화
//-----------------------------------------------------------------------------
LOCAL(BOOL) IsAdcEnable(ADC_TypeDef *Adc)
    {
    return (Adc->CR & (ADC_CR_ADEN|ADC_CR_ADDIS))==ADC_CR_ADEN &&
           ((Adc->ISR & ADC_FLAG_RDY)!=0 || (Adc->CFGR1 & ADC_CFGR1_AUTOFF)!=0);
    }



//-----------------------------------------------------------------------------
//      ADC 초기화
//-----------------------------------------------------------------------------
VOID WINAPI ADC_Init(ADC_HandleTypeDef *hAdc)
    {
    DWORD CfgR1;
    ADC_TypeDef *Adc;

    Adc=hAdc->Instance;

    __HAL_RCC_ADC1_CLK_ENABLE();

    if ((Adc->CR & ADC_CR_ADSTART)==0)
        {
        if (IsAdcEnable(Adc)==FALSE)
            {
            MODIFYBIT(Adc->CFGR1, ADC_CFGR1_RES, hAdc->Init.Resolution);
            MODIFYBIT(Adc->CFGR2, ADC_CFGR2_CKMODE, hAdc->Init.ClockPrescaler);
            }

        Adc->CFGR1&=~(ADC_CFGR1_DISCEN|
            ADC_CFGR1_AUTOFF|
            ADC_CFGR1_AUTDLY|
            ADC_CFGR1_CONT|
            ADC_CFGR1_OVRMOD|
            ADC_CFGR1_EXTSEL|
            ADC_CFGR1_EXTEN|
            ADC_CFGR1_ALIGN|
            ADC_CFGR1_SCANDIR|
            ADC_CFGR1_DMACFG);

        CfgR1=(hAdc->Init.LowPowerAutoWait<<14)|
              (hAdc->Init.LowPowerAutoPowerOff<<15)|
              (hAdc->Init.ContinuousConvMode<<13)|
              hAdc->Init.DataAlign|
              (hAdc->Init.DMAContinuousRequests<<1);

        if (hAdc->Init.Overrun!=ADC_OVR_DATA_PRESERVED) CfgR1|=ADC_CFGR1_OVRMOD;
        if (hAdc->Init.ScanConvMode==ADC_SCAN_DIRECTION_BACKWARD) CfgR1|=ADC_CFGR1_SCANDIR;

        if ((hAdc->Init.DiscontinuousConvMode==ENABLE) &&
            (hAdc->Init.ContinuousConvMode==DISABLE)) CfgR1|=ADC_CFGR1_DISCEN;

        if (hAdc->Init.ExternalTrigConv!=ADC_SOFTWARE_START)
            CfgR1|=hAdc->Init.ExternalTrigConv | hAdc->Init.ExternalTrigConvEdge;

        Adc->CFGR1|=CfgR1;

        if (hAdc->Init.SamplingTimeCommon!=0)
            {
            Adc->SMPR&=~ADC_SMPR_SMP;
            Adc->SMPR|=hAdc->Init.SamplingTimeCommon & ADC_SMPR_SMP;
            }

        //if ((Adc->CFGR1 & ~(ADC_CFGR1_AWDCH|ADC_CFGR1_AWDEN|ADC_CFGR1_AWDSGL|ADC_CFGR1_RES))==CfgR1) 이면 NoErr
        }
    }



LOCAL(BOOL) ADC_Enable(ADC_TypeDef *Adc)
    {
    BOOL Rslt=FALSE;
    DWORD Time;
    __IO DWORD Delay;

    if (IsAdcEnable(Adc)==FALSE)
        {
        if (Adc->CR & (ADC_CR_ADCAL|ADC_CR_ADSTP|ADC_CR_ADSTART|ADC_CR_ADDIS|ADC_CR_ADEN)) goto ProcExit; //ADC_ERROR_INTERNAL

        Adc->CR|=ADC_CR_ADEN;

        Delay=(SystemCoreClock/1000000)*ADC_STAB_DELAY_US;
        while (Delay--);

        Time=HAL_GetTick();
        while ((Adc->ISR & ADC_FLAG_RDY)==0)
            {
            if ((HAL_GetTick()-Time)>ADC_ENABLE_TIMEOUT) goto ProcExit;
            }
        }
    Rslt=TRUE;

    ProcExit:
    return Rslt;
    }



LOCAL(VOID) ADC_Start(ADC_HandleTypeDef*hAdc)
    {
    ADC_TypeDef *Adc;

    Adc=hAdc->Instance;
    if ((Adc->CR & ADC_CR_ADSTART)==0)
        {
        if (hAdc->Init.LowPowerAutoPowerOff!=0 ||
            ADC_Enable(Adc)!=FALSE)
            {
            Adc->ISR=ADC_FLAG_EOC|ADC_FLAG_EOS|ADC_FLAG_OVR;
            Adc->CR|=ADC_CR_ADSTART;
            }
        }
    }


LOCAL(HAL_StatusTypeDef) ADC_ConversionStop(ADC_TypeDef *Adc)
    {
    DWORD Time;

    if (Adc->CR & ADC_CR_ADSTART)
        {
        if ((Adc->CR & ADC_CR_ADSTART)!=0 &&
            (Adc->CR & ADC_CR_ADDIS)==0) Adc->CR|=ADC_CR_ADSTP;

        Time=HAL_GetTick();
        while (Adc->CR & ADC_CR_ADSTART)
            {
            if (HAL_GetTick()-Time>ADC_STOP_CONVERSION_TIMEOUT) return HAL_ERROR;
            }
        }
    return HAL_OK;
    }



LOCAL(HAL_StatusTypeDef) ADC_Disable(ADC_TypeDef *Adc)
    {
    DWORD Time;

    if (IsAdcEnable(Adc)!=FALSE)
        {
        if ((Adc->CR & (ADC_CR_ADSTART|ADC_CR_ADEN))!=ADC_CR_ADEN) return HAL_ERROR;
        Adc->CR|=ADC_CR_ADDIS;
        Adc->ISR=ADC_FLAG_EOSMP|ADC_FLAG_RDY;

        Time=HAL_GetTick();
        while (Adc->CR & ADC_CR_ADEN)
            {
            if (HAL_GetTick()-Time>ADC_DISABLE_TIMEOUT) return HAL_ERROR;
            }
        }
    return HAL_OK;
    }



LOCAL(VOID) ADC_Stop(ADC_HandleTypeDef *hAdc)
    {
    ADC_TypeDef *Adc;

    Adc=hAdc->Instance;
    if (ADC_ConversionStop(Adc)==HAL_OK &&
        ADC_Disable(Adc)==HAL_OK)
        {
        //ADC_STATE_CLR_SET(hAdc->State, HAL_ADC_STATE_REG_BUSY, HAL_ADC_STATE_READY);
        }
    }




LOCAL(VOID) ADC_ConfigChannel(ADC_HandleTypeDef *hAdc, int Ch, int Rank, int SamplingTime)
    {
    __IO DWORD Delay;
    ADC_TypeDef *ADCx;

    ADCx=hAdc->Instance;
    if ((ADCx->CR & ADC_CR_ADSTART)==0)
        {
        if (Rank!=ADC_RANK_NONE)
            {
            ADCx->CHSELR|=1<<Ch;

            if (IS_ADC_SAMPLE_TIME(hAdc->Init.SamplingTimeCommon)==FALSE)
                {
                if ((ADCx->SMPR & ADC_SMPR_SMP)!=SamplingTime)
                    {
                    ADCx->SMPR&=~ADC_SMPR_SMP;
                    ADCx->SMPR|=SamplingTime & ADC_SMPR_SMP;
                    }
                }

            if (ADC_IS_CHANNEL_INTERNAL(Ch))
                {
                ADC->CCR|=ADC_CHANNEL_INTERNAL_PATH(Ch);
                if (Ch==ADC_CHANNEL_TEMPSENSOR)
                    {
                    Delay=(SystemCoreClock/1000000)*ADC_TEMPSENSOR_DELAY_US;
                    while (Delay!=0) Delay--;
                    }
                }
            }
        else{
            ADCx->CHSELR&=~(1<<Ch);
            if (ADC_IS_CHANNEL_INTERNAL(Ch)) ADC->CCR&=~ADC_CHANNEL_INTERNAL_PATH(Ch);
            }
        }
    }




LOCAL(VOID) ADCEx_CalibrationStart(ADC_TypeDef *Adc)
    {
    DWORD Time;

    if (IsAdcEnable(Adc)==FALSE)
        {
        Adc->CR|=ADC_CR_ADCAL;

        Time=HAL_GetTick();
        while (Adc->CR & ADC_CR_ADCAL)
            {
            if (HAL_GetTick()-Time>ADC_CALIBRATION_TIMEOUT) goto ProcExit;
            }
        }
    ProcExit:;
    }



//-----------------------------------------------------------------------------
//      ADC 초기화
//-----------------------------------------------------------------------------
VOID WINAPI InitADC(VOID)
    {
    __HAL_RCC_ADC1_CLK_ENABLE();

    AdcHandle.Instance=ADC1;
    AdcHandle.Init.ClockPrescaler=ADC_CLOCK_SYNC_PCLK_DIV4;     //ADC_CLOCK_ASYNC_DIV1 / ADC_CLOCK_SYNC_PCLK_DIV2
    AdcHandle.Init.Resolution=ADC_RESOLUTION_12B;               //_6B / _8B / _10B /_12B
    AdcHandle.Init.ScanConvMode=DISABLE;                        //ADC_SCAN_DIRECTION_BACKWARD / _FORWARD
    AdcHandle.Init.ContinuousConvMode=DISABLE;
    AdcHandle.Init.DiscontinuousConvMode=DISABLE;
    AdcHandle.Init.ExternalTrigConv=ADC_SOFTWARE_START;         //ADC_EXTERNALTRIGCONV_T3_TRGO /_T1_TRGO / _T1_CC4
    AdcHandle.Init.ExternalTrigConvEdge=ADC_EXTERNALTRIGCONVEDGE_NONE; //_RISING / _NONE / _FALLING / _RISINGFALLING
    //AdcHandle.Init.DataAlign=ADC_DATAALIGN_RIGHT;             //ADC_DATAALIGN_RIGHT(0) / ADC_DATAALIGN_LEFT
    AdcHandle.Init.DMAContinuousRequests=DISABLE;
    AdcHandle.Init.EOCSelection=ADC_EOC_SINGLE_CONV;            //_SEQ_CONV / _SINGLE_CONV / _SINGLE_SEQ_CONV
    AdcHandle.Init.Overrun=ADC_OVR_DATA_OVERWRITTEN;            //_PRESERVED / _OVERWRITTEN
    AdcHandle.Init.LowPowerAutoWait=DISABLE;
    AdcHandle.Init.LowPowerAutoPowerOff=DISABLE;
    ADC_Init(&AdcHandle);

    //NVIC_SetPriority(ADC1_COMP_IRQn, 1);
    //NVIC_EnableIRQ(ADC1_COMP_IRQn);
    //AdcHandle.Instance->IER|=ADC_IT_OVR;                      //__HAL_ADC_ENABLE_IT

    ADCEx_CalibrationStart(AdcHandle.Instance);
    //HAL_ADCEx_Calibration_Start(&AdcHandle);
    }




VOID WINAPI ADC_PollForConversion(ADC_HandleTypeDef*hAdc, DWORD Timeout)
    {
    DWORD Time, EOC;
    ADC_TypeDef *Adc;

    Adc=hAdc->Instance;

    EOC=ADC_FLAG_EOS;
    if (hAdc->Init.EOCSelection!=ADC_EOC_SEQ_CONV)
        {
        if (Adc->CFGR1 & ADC_CFGR1_DMAEN) goto ProcExit;
        EOC=ADC_FLAG_EOC|ADC_FLAG_EOS;
        }

    Time=HAL_GetTick();
    while ((Adc->ISR & EOC)==0)
        {
        if (HAL_GetTick()-Time>Timeout) goto ProcExit;        //HAL_TIMEOUT
        }

   if ((Adc->CFGR1 & ADC_CFGR1_EXTEN)==0 && hAdc->Init.ContinuousConvMode==DISABLE)
        {
        if (Adc->ISR & ADC_FLAG_EOS)
            {
            if ((Adc->CR & ADC_CR_ADSTART)==0)
                Adc->IER &= ~(ADC_IT_EOC|ADC_IT_EOS);
            }
        }

    if (hAdc->Init.LowPowerAutoWait==DISABLE) Adc->ISR=ADC_FLAG_EOC|ADC_FLAG_EOS;

    ProcExit:;
    }



DWORD WINAPI ADC_GetValue(int Channel)
    {
    DWORD Value;

    ADC_ConfigChannel(&AdcHandle, Channel, ADC_RANK_CHANNEL_NUMBER, ADC_SAMPLETIME_28CYCLES_5);

    ADC_Start(&AdcHandle);
    ADC_PollForConversion(&AdcHandle, 10);
    AdcHandle.Instance->ISR=ADC_FLAG_EOS;
    Value=AdcHandle.Instance->DR;
    ADC_Stop(&AdcHandle);

    ADC_ConfigChannel(&AdcHandle, Channel, ADC_RANK_NONE, 0);   //이걸 안해주면 채널이 다중선택되어 여러 채널을 사용할 경우 각각의 값이 나오지 않음
    return Value;
    }


#if 0
DWORD WINAPI ADC_GetValue(int Channel)
    {
    DWORD Value;
    ADC_ChannelConfTypeDef ACC;

    ACC.Channel=Channel;
    ACC.Rank=ADC_RANK_NONE; //ADC_RANK_CHANNEL_NUMBER;
    ACC.SamplingTime=ADC_SAMPLETIME_28CYCLES_5;
    HAL_ADC_ConfigChannel(&AdcHandle, &ACC);

    HAL_ADC_Start(&AdcHandle);
    HAL_ADC_PollForConversion(&AdcHandle, 10);
    Value=HAL_ADC_GetValue(&AdcHandle);
    HAL_ADC_Stop(&AdcHandle);

    return Value;
    }
#endif



//-----------------------------------------------------------------------------
//      평균값 리턴
//-----------------------------------------------------------------------------
int WINAPI ADC_GetAverage(int ChNo, ADCAVERAGE *AA)
    {
    int I, Value;

    Value=ADC_GetValue(ChNo);
    if (AA->Cnt<ADCAVERAGEQTY)
        {
        AA->Data[AA->Cnt++]=Value;
        }
    else{
        MoveMem(AA->Data, AA->Data+1, sizeof(short)*(ADCAVERAGEQTY-1));
        AA->Data[ADCAVERAGEQTY-1]=Value;
        }

    for (I=Value=0; I<AA->Cnt; I++) Value+=AA->Data[I];
    return Value/AA->Cnt;
    }

#endif //HAL_ADC_MODULE_ENABLED




#ifdef HAL_RTC_MODULE_ENABLED
///////////////////////////////////////////////////////////////////////////////
//                      RTC
///////////////////////////////////////////////////////////////////////////////



#define RTC_WRITEPROTECTION_DISABLE(Inst) (Inst)->WPR=0xCA; (Inst)->WPR=0x53
#define RTC_WRITEPROTECTION_ENABLE(Inst)  (Inst)->WPR=0xFF

static RTC_HandleTypeDef RtcHandle;


HAL_StatusTypeDef HAL_RCCEx_PeriphCLKConfig(RCC_PeriphCLKInitTypeDef*PCI)
    {
    UINT Time, BDCR;

    if (PCI->PeriphClockSelection & RCC_PERIPHCLK_RTC)
        {
        if ((RCC->BDCR&RCC_BDCR_RTCSEL)!=(PCI->RTCClockSelection&RCC_BDCR_RTCSEL))
            {
            __HAL_RCC_PWR_CLK_ENABLE();

            PWR->CR|=PWR_CR_DBP;
            Time=HAL_GetTick();
            while ((PWR->CR&PWR_CR_DBP)==0)
                {
                if (HAL_GetTick()-Time>RCC_DBP_TIMEOUT_VALUE) return HAL_TIMEOUT;
                }

            BDCR=RCC->BDCR & ~RCC_BDCR_RTCSEL;
            RCC->BDCR|=RCC_BDCR_BDRST;  //__HAL_RCC_BACKUPRESET_FORCE();
            RCC->BDCR&=~RCC_BDCR_BDRST; //__HAL_RCC_BACKUPRESET_RELEASE();
            RCC->BDCR=BDCR;

            if (BDCR & RCC_BDCR_LSERDY)
                {
                Time=HAL_GetTick();
                while (__HAL_RCC_GET_FLAG(RCC_FLAG_LSERDY)==0)
                    {
                    if (HAL_GetTick()-Time>RCC_LSE_TIMEOUT_VALUE) return HAL_TIMEOUT;
                    }
                }
            RCC->BDCR=(RCC->BDCR & ~RCC_BDCR_RTCSEL) | PCI->RTCClockSelection;
            }
        }


    if (PCI->PeriphClockSelection & RCC_PERIPHCLK_USART1)
        RCC->CFGR3=(RCC->CFGR3 & ~RCC_CFGR3_USART1SW) | PCI->Usart1ClockSelection;      //__HAL_RCC_USART1_CONFIG

    #if defined(STM32F071xB) || defined(STM32F072xB) || defined(STM32F078xx) || defined(STM32F091xC) || defined(STM32F098xx)
    if (PCI->PeriphClockSelection & RCC_PERIPHCLK_USART2)
        RCC->CFGR3=(RCC->CFGR3 & ~RCC_CFGR3_USART2SW) | PCI->Usart2ClockSelection;      //__HAL_RCC_USART2_CONFIG
    #endif

    #if defined(STM32F091xC) || defined(STM32F098xx)
    if (PCI->PeriphClockSelection & RCC_PERIPHCLK_USART3)
        RCC->CFGR3=(RCC->CFGR3 & ~RCC_CFGR3_USART3SW) | PCI->Usart3ClockSelection;      //__HAL_RCC_USART3_CONFIG
    #endif

    if (PCI->PeriphClockSelection & RCC_PERIPHCLK_I2C1)
        RCC->CFGR3=(RCC->CFGR3 & ~RCC_CFGR3_I2C1SW) | PCI->I2c1ClockSelection;          //__HAL_RCC_I2C1_CONFIG

    #if defined(STM32F042x6) || defined(STM32F048xx) || defined(STM32F072xB) || defined(STM32F078xx) || defined(STM32F070xB) || defined(STM32F070x6)
    if (PCI->PeriphClockSelection & RCC_PERIPHCLK_USB)
        RCC->CFGR3=(RCC->CFGR3 & ~RCC_CFGR3_USBSW) | PCI->UsbClockSelection;            //__HAL_RCC_USB_CONFIG
    #endif

    #if defined(STM32F042x6) || defined(STM32F048xx)\
         || defined(STM32F051x8) || defined(STM32F058xx)\
         || defined(STM32F071xB) || defined(STM32F072xB) || defined(STM32F078xx)\
         || defined(STM32F091xC) || defined(STM32F098xx)
    if (PCI->PeriphClockSelection & RCC_PERIPHCLK_CEC)
        RCC->CFGR3=(RCC->CFGR3 & ~RCC_CFGR3_CECSW) | PCI->CecClockSelection;            //__HAL_RCC_CEC_CONFIG
    #endif

    return HAL_OK;
    }




LOCAL(BOOL) RTC__EnterInitMode(RTC_TypeDef *RTCx)
    {
    DWORD Time;

    if ((RTCx->ISR & RTC_ISR_INITF)==0)
        {
        RTCx->ISR=RTC_INIT_MASK;

        Time=HAL_GetTick();
        while ((RTCx->ISR & RTC_ISR_INITF)==0)
            {
            if (HAL_GetTick()-Time > RTC_TIMEOUT_VALUE) return FALSE;
            }
        }
    return TRUE;
    }




LOCAL(BOOL) RTC_WaitForSynchro(RTC_TypeDef *RTCx)
    {
    DWORD Time;

    RTCx->ISR&=RTC_RSF_MASK;

    Time=HAL_GetTick();
    while ((RTCx->ISR & RTC_ISR_RSF)==0)
        {
        if (HAL_GetTick()-Time>RTC_TIMEOUT_VALUE) return FALSE;
        }
    return TRUE;
    }



VOID WINAPI InitRTC(int RtcClock)
    {
    RTC_TypeDef *RTCx;
    RCC_OscInitTypeDef       RCC_OI;
    RCC_PeriphCLKInitTypeDef RCC_PCI;

    __HAL_RCC_PWR_CLK_ENABLE();

    #ifdef STM32L051xx
    __HAL_RCC_LSEDRIVE_CONFIG(RCC_LSEDRIVE_HIGH); //L시리즈는 모두 해당
    #endif

    PWR->CR|=PWR_CR_DBP;        //HAL_PWR_EnableBkUpAccess();

    RCC_OI.PLL.PLLState=RCC_PLL_NONE;
    if (RtcClock==RTC_CLOCK_LSI)
        {
        RCC_OI.OscillatorType=RCC_OSCILLATORTYPE_LSI;
        RCC_OI.LSIState=RCC_LSI_ON;
        }
    else{
        RCC_OI.OscillatorType=RCC_OSCILLATORTYPE_LSE;
        RCC_OI.LSEState=RCC_LSE_ON;
        }
    if (HAL_RCC_OscConfig(&RCC_OI)!=HAL_OK) Printf("RTC: OSC Config Error"CRLF);

    RCC_PCI.PeriphClockSelection=RCC_PERIPHCLK_RTC;
    RCC_PCI.RTCClockSelection=(RtcClock==RTC_CLOCK_LSI) ? RCC_RTCCLKSOURCE_LSI:RCC_RTCCLKSOURCE_LSE;
    if (HAL_RCCEx_PeriphCLKConfig(&RCC_PCI)!=HAL_OK) Printf("RTC: Select Clock Error"CRLF);

    __HAL_RCC_RTC_ENABLE();

    RtcHandle.Init.HourFormat=RTC_HOURFORMAT_24;
    RtcHandle.Init.AsynchPrediv=0x7F;
    RtcHandle.Init.SynchPrediv=(RtcClock==RTC_CLOCK_LSI) ? 0x0130:0x00FF;
    RtcHandle.Init.OutPut=RTC_OUTPUT_DISABLE;                   //RTC_OUTPUT_ALARMA, RTC_OUTPUT_WAKEUP
    RtcHandle.Init.OutPutPolarity=RTC_OUTPUT_POLARITY_HIGH;     //RTC_OUTPUT_POLARITY_LOW
    RtcHandle.Init.OutPutType=RTC_OUTPUT_TYPE_OPENDRAIN;        //RTC_OUTPUT_TYPE_PUSHPULL
    RtcHandle.Instance=RTCx=RTC;

    RTC_WRITEPROTECTION_DISABLE(RTCx);
    if (RTC__EnterInitMode(RTCx)!=FALSE)
        {
        RTCx->CR&=~(RTC_CR_FMT|RTC_CR_OSEL|RTC_CR_POL);
        RTCx->CR|=RtcHandle.Init.HourFormat|RtcHandle.Init.OutPut|RtcHandle.Init.OutPutPolarity;
        RTCx->PRER=RtcHandle.Init.SynchPrediv;
        RTCx->PRER|=RtcHandle.Init.AsynchPrediv<<16;
        RTCx->ISR&=~RTC_ISR_INIT;

        #ifndef STM32L051xx
        if ((RTCx->CR & RTC_CR_BYPSHAD)==0)
            {
            if (RTC_WaitForSynchro(RTCx)==FALSE) goto InitExit;
            }
        RTCx->TAFCR&=~RTC_TAFCR_ALARMOUTTYPE;
        RTCx->TAFCR|=RtcHandle.Init.OutPutType;
        #else
        RTCx->OR&=~(RTC_OR_ALARMOUTTYPE|RTC_OR_OUT_RMP);
        RTCx->OR|=RtcHandle.Init.OutPutType|RtcHandle.Init.OutPutRemap;
        #endif
        }

    InitExit:
    RTC_WRITEPROTECTION_ENABLE(RTCx);
    }




//-----------------------------------------------------------------------------
//      외부 RTC 콜백설정
//-----------------------------------------------------------------------------
static GETTIMEFT ExtRTC_GetTime;
static SETTIMEFT ExtRTC_SetTime;
VOID WINAPI SetExtRtcFP(GETTIMEFT GetTimeFP, SETTIMEFT SetTimeFP)
    {
    ExtRTC_GetTime=GetTimeFP;
    ExtRTC_SetTime=SetTimeFP;
    }



//-----------------------------------------------------------------------------
//      RTC의 시간을 얻어옴
//-----------------------------------------------------------------------------
static SYSTEMTIME TempST;
#ifdef CORRECT_RTC
LOCAL(VOID) GetLocalTime_(SYSTEMTIME *ST)
#else
VOID WINAPI GetLocalTime(SYSTEMTIME *ST)
#endif
    {
    UINT DR, TR, PreDivS;
    RTC_TypeDef *RTCx;

    if (ExtRTC_GetTime) ExtRTC_GetTime(ST);
    else{
        if ((RTCx=RtcHandle.Instance)!=NULL)
            {
            PreDivS=RTC->PRER&0xFFFF;
            ST->wMilliseconds=(PreDivS-RTCx->SSR) * 1000 / (PreDivS+1);

            TR=RTCx->TR;
            ST->wHour=Bcd2Bin((TR&(RTC_TR_HT|RTC_TR_HU))>>16);
            ST->wMinute=Bcd2Bin((TR&(RTC_TR_MNT|RTC_TR_MNU))>>8);
            ST->wSecond=Bcd2Bin(TR&(RTC_TR_ST|RTC_TR_SU));
            //ST->TimeFormat=(BYTE)((TR&(RTC_TR_PM))>>16);

            DR=RTCx->DR;
            ST->wYear=Bcd2Bin((DR&(RTC_DR_YT|RTC_DR_YU))>>16)+2000;
            ST->wMonth=Bcd2Bin((DR&(RTC_DR_MT|RTC_DR_MU))>>8);
            ST->wDay=Bcd2Bin(DR&(RTC_DR_DT|RTC_DR_DU));
            //ST->WeekDay=((DR&(RTC_DR_WDU))>>13);
            }
        else *ST=TempST;
        }
    }


#ifdef CORRECT_RTC
#include "CorrectRtc.c"
#endif



//-----------------------------------------------------------------------------
//      현재 시간을 2000/1/1을 기준으로 해서 초로 계산해줌
//-----------------------------------------------------------------------------
JTIME WINAPI GetTodayToSecond(VOID)
    {
    SYSTEMTIME ST;
    GetLocalTime(&ST);
    return PackTotalSecond(&ST);
    }



//-----------------------------------------------------------------------------
//      RTC의 시간을 설정함
//-----------------------------------------------------------------------------
BOOL WINAPI SetLocalTime(CONST SYSTEMTIME *ST)
    {
    int Rslt=FALSE, Week, PreDivS;
    RTC_TypeDef *RTCx;

    if (ExtRTC_SetTime) Rslt=ExtRTC_SetTime(ST);
    else{
        if ((RTCx=RtcHandle.Instance)!=NULL)
            {
            //if ((RTCx->Instance->CR&RTC_CR_FMT)==0) ST->TimeFormat=0;
            RTC_WRITEPROTECTION_DISABLE(RTCx);

            if (RTC__EnterInitMode(RTCx)==FALSE) goto ProtEn;

            RTCx->TR=(Bin2Bcd(ST->wHour)<<16)|
                     (Bin2Bcd(ST->wMinute)<<8)|
                      Bin2Bcd(ST->wSecond); //| ((ST->TimeFormat)<<16);

            if ((Week=GetWeek(ST->wYear, ST->wMonth, ST->wDay))==0) Week=7; //주일
            RTCx->DR=(Bin2Bcd(ST->wYear-2000)<<16) |
                     (Bin2Bcd(ST->wMonth)<<8) |
                      Bin2Bcd(ST->wDay) |
                     (Week<<13);

            PreDivS=RTC->PRER&0xFFFF;
            RTCx->SSR=PreDivS - ST->wMilliseconds*(PreDivS+1)/1000;

            RTCx->CR&=~RTC_CR_BCK;
            //RTCx->CR|=ST->DayLightSaving|ST->StoreOperation;

            RTCx->ISR&=~RTC_ISR_INIT;
            if ((RTCx->CR & RTC_CR_BYPSHAD)==0)
                {
                if (RTC_WaitForSynchro(RTCx)==FALSE) goto ProtEn;
                }
            Rslt++;

            ProtEn:
            RTC_WRITEPROTECTION_ENABLE(RTCx);
            }
        else TempST=*ST;
        }
    return Rslt;
    }




//-----------------------------------------------------------------------------
//  RTC_FORMAT_BIN 만지원, 24시간제만 지원
//-----------------------------------------------------------------------------
LOCAL(HAL_StatusTypeDef) RTC_SetAlarm_IT(RTC_TypeDef *RTCx, RTC_AlarmTypeDef *AT)
    {
    HAL_StatusTypeDef Rslt=HAL_ERROR;
    UINT Tick;

    RTC_WRITEPROTECTION_DISABLE(RTCx);

    if (RTCx->CR & RTC_CR_FMT) goto ProcExit;   //12시간제 모드만 지원

    RTCx->CR&=~RTC_CR_ALRAE;                    //__HAL_RTC_ALARMA_DISABLE(hRtc);
    RTCx->ISR= ~(RTC_FLAG_ALRAF|RTC_ISR_INIT) | (RTCx->ISR & RTC_ISR_INIT);  //=__HAL_RTC_ALARM_CLEAR_FLAG(hRtc, RTC_FLAG_ALRAF);

    Tick=HAL_GetTick();
    while ((RTCx->ISR & RTC_FLAG_ALRAWF)==0)
        {
        if (HAL_GetTick()-Tick > RTC_TIMEOUT_VALUE) {Rslt=HAL_TIMEOUT; goto ProcExit;}
        }

    RTCx->ALRMAR=(Bin2Bcd(AT->AlarmTime.Hours)<<16) |
                 (Bin2Bcd(AT->AlarmTime.Minutes)<<8)|
                  Bin2Bcd(AT->AlarmTime.Seconds)|
                 (AT->AlarmTime.TimeFormat<<16)|
                 (Bin2Bcd(AT->AlarmDateWeekDay)<<24) | AT->AlarmDateWeekDaySel | AT->AlarmMask;

    RTCx->ALRMASSR=AT->AlarmTime.SubSeconds|AT->AlarmSubSecondMask;

    RTCx->CR|=RTC_CR_ALRAE;                     //__HAL_RTC_ALARMA_ENABLE(hRtc);
    RTCx->CR|=RTC_IT_ALRA;                      //__HAL_RTC_ALARM_ENABLE_IT(hRtc, RTC_IT_ALRA);
    __HAL_RTC_ALARM_EXTI_ENABLE_IT();           //Enable interrupt on the RTC Alarm associated Exti line.
    __HAL_RTC_ALARM_EXTI_ENABLE_RISING_EDGE();  //Enable rising edge trigger on the RTC Alarm associated Exti line.
    Rslt=HAL_OK;

    ProcExit:
    RTC_WRITEPROTECTION_ENABLE(RTCx);
    return Rslt;
    }



//-----------------------------------------------------------------------------
//      알람설정
//-----------------------------------------------------------------------------
VOID WINAPI RTC_SetAlarmTime(SYSTEMTIME *ST, BOOL EnableIT)
    {
    RTC_AlarmTypeDef AT;

    AT.AlarmDateWeekDay=ST->wDay;                           //요일인경우 1:월,2:화...일:7
    AT.AlarmDateWeekDaySel=RTC_ALARMDATEWEEKDAYSEL_DATE;    //RTC_ALARMDATEWEEKDAYSEL_WEEKDAY
    AT.AlarmMask=RTC_ALARMMASK_DATEWEEKDAY;
    AT.AlarmSubSecondMask=RTC_ALARMSUBSECONDMASK_NONE;
    AT.AlarmTime.TimeFormat=0;                              //24시간제일 땐 0
    AT.AlarmTime.Hours=ST->wHour;
    AT.AlarmTime.Minutes=ST->wMinute;
    AT.AlarmTime.Seconds=ST->wSecond;
    AT.AlarmTime.SubSeconds=0;
    if (RTC_SetAlarm_IT(RtcHandle.Instance, &AT)==HAL_OK)
        {
        if (EnableIT)           //Alarm Wakeup을 사용할 때는 Interrupt가 필요치 않음
            {
            NVIC_SetPriority(RTC_IRQn, 0);
            NVIC_EnableIRQ(RTC_IRQn);
            }
        }
    else Printf("RTC: Set Alarm Error"CRLF);
    }




//-----------------------------------------------------------------------------
//      RTC IRQ 핸들러, Wakeup만 위해서는 필요치 않음
//-----------------------------------------------------------------------------
__weak VOID WINAPI RTC_AlarmAEventCB(VOID) {}
VOID RTC_IRQHandler(VOID)
    {
    RTC_TypeDef *RTCx;

    JOSIntEnterII();
    RTCx=RtcHandle.Instance;
    //HAL_RTCEx_WakeUpTimerIRQHandler(&RtcHandle);

    //HAL_RTC_AlarmIRQHandler(&RtcHandle);
    if (RTCx->CR & RTC_IT_ALRA)         //__HAL_RTC_ALARM_GET_IT_SOURCE()
        {
        if (RTCx->ISR & RTC_FLAG_ALRAF) //__HAL_RTC_ALARM_GET_FLAG
            {
            RTC_AlarmAEventCB();
            RTCx->ISR = ~(RTC_FLAG_ALRAF|RTC_ISR_INIT) | (RTCx->ISR & RTC_ISR_INIT);  //=__HAL_RTC_ALARM_CLEAR_FLAG(hRtc, RTC_FLAG_ALRAF);
            }
        }
    __HAL_RTC_ALARM_EXTI_CLEAR_FLAG();
    JOSIntExitII();
    }



VOID WINAPI RTC_Calibration(VOID)
    {
    RTC_TypeDef *RTCB;
    DWORD SubSec, Meas_ms, StartTime, NewValue;

    if ((RTCB=RtcHandle.Instance)!=NULL)
        {
        StartTime=GetTickCount();
        SubSec=(WORD)RTC->SSR;
        HAL_Delay(100);
        while ((WORD)RTC->SSR!=SubSec); //다음 같은 값이 되면 RTC의 1초임
        Meas_ms=GetTickCount()-StartTime;
        //Printf("Measure Time=%u"CRLF, Meas_ms);

        NewValue=(((WORD)RTC->PRER+1)*1000+(Meas_ms>>1))/Meas_ms;

        RTC_WRITEPROTECTION_DISABLE(RTCB);
        if (RTC__EnterInitMode(RTCB)!=FALSE)
            {
            RTC->PRER=(RTC->PRER & 0xFFFF0000) | ((NewValue-1) & 0xFFFF);

            RTCB->CR&=~RTC_CR_BKP;
            RTCB->ISR&=~RTC_ISR_INIT;
            if ((RTCB->CR & RTC_CR_BYPSHAD)==0) RTC_WaitForSynchro(RTCB);
            }
        RTC_WRITEPROTECTION_ENABLE(RTCB);
        }
    }




#endif //HAL_RTC_MODULE_ENABLED



#ifdef HAL_PWR_MODULE_ENABLED
VOID WINAPI PWR_EnterStopMode(int Regulator, int StopEntry)
    {
    PWR->CR=(PWR->CR & ~(PWR_CR_PDDS|PWR_CR_LPDS)) | Regulator;
    SCB->SCR|=SCB_SCR_SLEEPDEEP_Msk;

    if (StopEntry==PWR_STOPENTRY_WFI)
        {
        __WFI();
        }
    else{
        __SEV();
        __WFE();
        __WFE();
        }

    SCB->SCR&=~SCB_SCR_SLEEPDEEP_Msk;

    #ifdef HAL_RTC_MODULE_ENABLED
    RtcHandle.Instance->ISR = ~(RTC_FLAG_ALRAF|RTC_ISR_INIT) | (RtcHandle.Instance->ISR & RTC_ISR_INIT);  //=__HAL_RTC_ALARM_CLEAR_FLAG(hRtc, RTC_FLAG_ALRAF);
    __HAL_RTC_ALARM_EXTI_CLEAR_FLAG();
    #endif
    }
#endif //HAL_PWR_MODULE_ENABLED





///////////////////////////////////////////////////////////////////////////////
//                      DAC
///////////////////////////////////////////////////////////////////////////////

#ifdef HAL_DAC_MODULE_ENABLED
#if defined(STM32F071xB) || defined(STM32F072xB) || defined(STM32F078xx) || defined(STM32F091xC) || defined(STM32F098xx)

VOID WINAPI DAC_SetValue(UINT Channel, UINT Alignment, UINT Data)
    {
    __IO UINT T;

    T=(UINT)DAC;
    if (Channel==DAC_CHANNEL_1) T+=DAC_DHR12R1_ALIGNMENT(Alignment);
    else                        T+=DAC_DHR12R2_ALIGNMENT(Alignment);
    *(__IO UINT*)T=Data;
    }



LOCAL(VOID) DAC_ConfigChannel(UINT Channel, UINT Trigger, UINT OutputBuff)
    {
    UINT T;

    T=DAC->CR & ~((DAC_CR_MAMP1|DAC_CR_WAVE1|DAC_CR_TSEL1|DAC_CR_TEN1|DAC_CR_BOFF1)<<Channel);
    DAC->CR=T | ((Trigger|OutputBuff)<<Channel);
    }



LOCAL(VOID) DAC_Start(UINT Channel)
    {
    DAC->CR|=DAC_CR_EN1<<Channel;  //__HAL_DAC_ENABLE

    if (Channel==DAC_CHANNEL_1)
        {
        if ((DAC->CR & (DAC_CR_TEN1|DAC_CR_TSEL1))==(DAC_CR_TEN1|DAC_CR_TSEL1)) SET_BIT(DAC->SWTRIGR, DAC_SWTRIGR_SWTRIG1);
        }
    else{
        if ((DAC->CR & (DAC_CR_TEN2|DAC_CR_TSEL2))==(DAC_CR_TEN2|DAC_CR_TSEL2)) SET_BIT(DAC->SWTRIGR, DAC_SWTRIGR_SWTRIG2);
        }
    }



//-----------------------------------------------------------------------------
//      DAC 초기화
//-----------------------------------------------------------------------------
VOID WINAPI InitDAC(UINT Channel)
    {
    __HAL_RCC_DAC1_CLK_ENABLE();

    DAC_ConfigChannel(Channel, DAC_TRIGGER_NONE, DAC_OUTPUTBUFFER_DISABLE);
    DAC_SetValue(Channel, DAC_ALIGN_12B_R, 0);
    DAC_Start(Channel);
    }



#endif //defined(STM32F071xB) || defined(STM32F072xB) || defined(STM32F078xx) || defined(STM32F091xC) || defined(STM32F098xx)
#endif //HAL_DAC_MODULE_ENABLED



//-----------------------------------------------------------------------------
//          32비트 랜덤 수치를 얻어냄
//-----------------------------------------------------------------------------
VOID WINAPI TM_RNG_Init(VOID)
    {
    }


#ifdef HAL_RTC_MODULE_ENABLED
UINT WINAPI TM_RNG_Get(VOID)
    {
    DWORD Dw;
    Dw=GetTodayToSecond();
    return CalculateCRC((LPCBYTE)&Dw, 4, 0xC0B3FA1F);
    }
#endif



#ifdef HAL_IWDG_MODULE_ENABLED
///////////////////////////////////////////////////////////////////////////////
//                              WatchDog
///////////////////////////////////////////////////////////////////////////////


VOID WINAPI IWDG_Refresh(VOID)
    {
    IWDG->KR=IWDG_KEY_RELOAD;   //__HAL_IWDG_RELOAD_COUNTER()
    }


#define HAL_IWDG_DEFAULT_TIMEOUT            39
BOOL WINAPI IWDG_Init(int Prescaler, int Reload, int Window)
    {
    BOOL Rslt=FALSE;
    UINT Tick;

    //__HAL_RCC_CLEAR_RESET_FLAGS();

    IWDG->KR=IWDG_KEY_ENABLE;               //__HAL_IWDG_START()
    IWDG->KR=IWDG_KEY_WRITE_ACCESS_ENABLE;  //IWDG_ENABLE_WRITE_ACCESS()
    IWDG->PR=Prescaler;
    IWDG->RLR=Reload;

    Tick=HAL_GetTick();
    while (IWDG->SR!=0)
        {
        if (HAL_GetTick()-Tick>HAL_IWDG_DEFAULT_TIMEOUT) goto ProcExit;
        }
    if (IWDG->WINR!=Window) IWDG->WINR=Window; else IWDG_Refresh();
    Rslt++;

    ProcExit:
    return Rslt;
    }


#endif //HAL_IWDG_MODULE_ENABLED




//-----------------------------------------------------------------------------
//      시스템 리셋
//-----------------------------------------------------------------------------
typedef VOID(*ft)(VOID);
VOID WINAPI SystemReset(LPCSTR BootReason)
    {
    #ifdef HAL_IWDG_MODULE_ENABLED
    ft Pgm;

    Printf(BootReason);
    Printf("SENSOR: Shutdown..."CRLF);
    UART_WaitAllSendIT(DebugPort, 10);      //문자열 찍을 시간

    JOS_ENTER_CRITICAL();
    SYSCFG->CFGR1 &= ~(SYSCFG_CFGR1_MEM_MODE);
    Pgm=(ft)(*(DWORD*)(FLASH_BASE+4));
    Pgm();
    #else
    Printf(BootReason);
    Printf("SENSOR: Shutdown..."CRLF);
    UART_WaitAllSendIT(DebugPort, 10);      //문자열 찍을 시간

    IWDG_Init(IWDG_PRESCALER_32, 10, IWDG_WINDOW_DISABLE);
    for (;;);
    #endif
    }





VOID WINAPI InitDriver(VOID)
    {
    }





