﻿///////////////////////////////////////////////////////////////////////////////
//      No CPU Automan Module 드라이버
//
//  2018-11-07
///////////////////////////////////////////////////////////////////////////////


#include "JLIB.H"
#include "DRIVER.H"
#include "JOS.H"
#include "MONITOR.H"
#include "MAIN.H"
#include "SPIAutoMan.h"
#ifdef AES128ENCRYPTION
#include "AES128.H"
#endif

#define REINITNOTIFY            0       //RFM을 다시 초기화 한경우 알려라
#define KOREA_CERTIFICATION     0       //KC인증 받는데 정전기에 RFM이 죽어서 4초마다 리셋하는 코드가 동작함
#define MINPACKETSIZE           6       //RF_PACKET_MINI의 크기는 7, Public Scan Packet는 6
                                        //세팅패킷 "ROZETA"+Len+Data
#define CHIPDEBUG               0
#define RFM_DELAY               1000



//Configuration registers
#define CC112X_IOCFG3                   0x0000
#define CC112X_IOCFG2                   0x0001
#define CC112X_IOCFG1                   0x0002
#define CC112X_IOCFG0                   0x0003
#define CC112X_SYNC3                    0x0004
#define CC112X_SYNC2                    0x0005
#define CC112X_SYNC1                    0x0006
#define CC112X_SYNC0                    0x0007
#define CC112X_SYNC_CFG1                0x0008
#define CC112X_SYNC_CFG0                0x0009
#define CC112X_DEVIATION_M              0x000A
#define CC112X_MODCFG_DEV_E             0x000B
#define CC112X_DCFILT_CFG               0x000C
#define CC112X_PREAMBLE_CFG1            0x000D
#define CC112X_PREAMBLE_CFG0            0x000E
#define CC112X_FREQ_IF_CFG              0x000F
#define CC112X_IQIC                     0x0010
#define CC112X_CHAN_BW                  0x0011
#define CC112X_MDMCFG1                  0x0012
#define CC112X_MDMCFG0                  0x0013
#define CC112X_DRATE2                   0x0014
#define CC112X_DRATE1                   0x0015
#define CC112X_DRATE0                   0x0016
#define CC112X_AGC_REF                  0x0017
#define CC112X_AGC_CS_THR               0x0018
#define CC112X_AGC_GAIN_ADJUST          0x0019
#define CC112X_AGC_CFG3                 0x001A
#define CC112X_AGC_CFG2                 0x001B
#define CC112X_AGC_CFG1                 0x001C
#define CC112X_AGC_CFG0                 0x001D
#define CC112X_FIFO_CFG                 0x001E
#define CC112X_DEV_ADDR                 0x001F
#define CC112X_SETTLING_CFG             0x0020
#define CC112X_FS_CFG                   0x0021
#define CC112X_WOR_CFG1                 0x0022
#define CC112X_WOR_CFG0                 0x0023
#define CC112X_WOR_EVENT0_MSB           0x0024
#define CC112X_WOR_EVENT0_LSB           0x0025
#define CC112X_PKT_CFG2                 0x0026
#define CC112X_PKT_CFG1                 0x0027
#define CC112X_PKT_CFG0                 0x0028
#define CC112X_RFEND_CFG1               0x0029
#define CC112X_RFEND_CFG0               0x002A
#define CC112X_PA_CFG2                  0x002B
#define CC112X_PA_CFG1                  0x002C
#define CC112X_PA_CFG0                  0x002D
#define CC112X_PKT_LEN                  0x002E

//Extended Configuration Registers
#define CC112X_IF_MIX_CFG               0x2F00
#define CC112X_FREQOFF_CFG              0x2F01
#define CC112X_TOC_CFG                  0x2F02
#define CC112X_MARC_SPARE               0x2F03
#define CC112X_ECG_CFG                  0x2F04
#define CC112X_SOFT_TX_DATA_CFG         0x2F05
#define CC112X_EXT_CTRL                 0x2F06
#define CC112X_RCCAL_FINE               0x2F07
#define CC112X_RCCAL_COARSE             0x2F08
#define CC112X_RCCAL_OFFSET             0x2F09
#define CC112X_FREQOFF1                 0x2F0A
#define CC112X_FREQOFF0                 0x2F0B
#define CC112X_FREQ2                    0x2F0C
#define CC112X_FREQ1                    0x2F0D
#define CC112X_FREQ0                    0x2F0E
#define CC112X_IF_ADC2                  0x2F0F
#define CC112X_IF_ADC1                  0x2F10
#define CC112X_IF_ADC0                  0x2F11
#define CC112X_FS_DIG1                  0x2F12
#define CC112X_FS_DIG0                  0x2F13
#define CC112X_FS_CAL3                  0x2F14
#define CC112X_FS_CAL2                  0x2F15
#define CC112X_FS_CAL1                  0x2F16
#define CC112X_FS_CAL0                  0x2F17
#define CC112X_FS_CHP                   0x2F18
#define CC112X_FS_DIVTWO                0x2F19
#define CC112X_FS_DSM1                  0x2F1A
#define CC112X_FS_DSM0                  0x2F1B
#define CC112X_FS_DVC1                  0x2F1C
#define CC112X_FS_DVC0                  0x2F1D
#define CC112X_FS_LBI                   0x2F1E
#define CC112X_FS_PFD                   0x2F1F
#define CC112X_FS_PRE                   0x2F20
#define CC112X_FS_REG_DIV_CML           0x2F21
#define CC112X_FS_SPARE                 0x2F22
#define CC112X_FS_VCO4                  0x2F23
#define CC112X_FS_VCO3                  0x2F24
#define CC112X_FS_VCO2                  0x2F25
#define CC112X_FS_VCO1                  0x2F26
#define CC112X_FS_VCO0                  0x2F27
#define CC112X_GBIAS6                   0x2F28
#define CC112X_GBIAS5                   0x2F29
#define CC112X_GBIAS4                   0x2F2A
#define CC112X_GBIAS3                   0x2F2B
#define CC112X_GBIAS2                   0x2F2C
#define CC112X_GBIAS1                   0x2F2D
#define CC112X_GBIAS0                   0x2F2E
#define CC112X_IFAMP                    0x2F2F
#define CC112X_LNA                      0x2F30
#define CC112X_RXMIX                    0x2F31
#define CC112X_XOSC5                    0x2F32
#define CC112X_XOSC4                    0x2F33
#define CC112X_XOSC3                    0x2F34
#define CC112X_XOSC2                    0x2F35
#define CC112X_XOSC1                    0x2F36
#define CC112X_XOSC0                    0x2F37
#define CC112X_ANALOG_SPARE             0x2F38
#define CC112X_PA_CFG3                  0x2F39
#define CC112X_IRQ0M                    0x2F3F
#define CC112X_IRQ0F                    0x2F40

//Status Registers
#define CC112X_WOR_TIME1                0x2F64
#define CC112X_WOR_TIME0                0x2F65
#define CC112X_WOR_CAPTURE1             0x2F66
#define CC112X_WOR_CAPTURE0             0x2F67
#define CC112X_BIST                     0x2F68
#define CC112X_DCFILTOFFSET_I1          0x2F69
#define CC112X_DCFILTOFFSET_I0          0x2F6A
#define CC112X_DCFILTOFFSET_Q1          0x2F6B
#define CC112X_DCFILTOFFSET_Q0          0x2F6C
#define CC112X_IQIE_I1                  0x2F6D
#define CC112X_IQIE_I0                  0x2F6E
#define CC112X_IQIE_Q1                  0x2F6F
#define CC112X_IQIE_Q0                  0x2F70
#define CC112X_RSSI1                    0x2F71
#define CC112X_RSSI0                    0x2F72
#define CC112X_MARCSTATE                0x2F73
#define CC112X_LQI_VAL                  0x2F74
#define CC112X_PQT_SYNC_ERR             0x2F75
#define CC112X_DEM_STATUS               0x2F76
#define CC112X_FREQOFF_EST1             0x2F77
#define CC112X_FREQOFF_EST0             0x2F78
#define CC112X_AGC_GAIN3                0x2F79
#define CC112X_AGC_GAIN2                0x2F7A
#define CC112X_AGC_GAIN1                0x2F7B
#define CC112X_AGC_GAIN0                0x2F7C
#define CC112X_SOFT_RX_DATA_OUT         0x2F7D
#define CC112X_SOFT_TX_DATA_IN          0x2F7E
#define CC112X_ASK_SOFT_RX_DATA         0x2F7F
#define CC112X_RNDGEN                   0x2F80
#define CC112X_MAGN2                    0x2F81
#define CC112X_MAGN1                    0x2F82
#define CC112X_MAGN0                    0x2F83
#define CC112X_ANG1                     0x2F84
#define CC112X_ANG0                     0x2F85
#define CC112X_CHFILT_I2                0x2F86
#define CC112X_CHFILT_I1                0x2F87
#define CC112X_CHFILT_I0                0x2F88
#define CC112X_CHFILT_Q2                0x2F89
#define CC112X_CHFILT_Q1                0x2F8A
#define CC112X_CHFILT_Q0                0x2F8B
#define CC112X_GPIO_STATUS              0x2F8C
#define CC112X_FSCAL_CTRL               0x2F8D
#define CC112X_PHASE_ADJUST             0x2F8E
#define CC112X_PARTNUMBER               0x2F8F
#define CC112X_PARTVERSION              0x2F90
#define CC112X_SERIAL_STATUS            0x2F91
#define CC112X_RX_STATUS                0x2F92
#define CC112X_TX_STATUS                0x2F93
#define CC112X_MARC_STATUS1             0x2F94
#define CC112X_MARC_STATUS0             0x2F95
#define CC112X_PA_IFAMP_TEST            0x2F96
#define CC112X_FSRF_TEST                0x2F97
#define CC112X_PRE_TEST                 0x2F98
#define CC112X_PRE_OVR                  0x2F99
#define CC112X_ADC_TEST                 0x2F9A
#define CC112X_DVC_TEST                 0x2F9B
#define CC112X_ATEST                    0x2F9C
#define CC112X_ATEST_LVDS               0x2F9D
#define CC112X_ATEST_MODE               0x2F9E
#define CC112X_XOSC_TEST1               0x2F9F
#define CC112X_XOSC_TEST0               0x2FA0

#define CC112X_RXFIRST                  0x2FD2
#define CC112X_TXFIRST                  0x2FD3
#define CC112X_RXLAST                   0x2FD4
#define CC112X_TXLAST                   0x2FD5
#define CC112X_NUM_TXBYTES              0x2FD6      //Number of bytes in TXFIFO
#define CC112X_NUM_RXBYTES              0x2FD7      //Number of bytes in RXFIFO
#define CC112X_FIFO_NUM_TXBYTES         0x2FD8
#define CC112X_FIFO_NUM_RXBYTES         0x2FD9

//DATA FIFO Access
#define CC112X_SINGLE_TXFIFO            0x003F      //TXFIFO  - Single access to Transmit FIFO

#define CC112X_LQI_CRC_OK_BM            0x80
#define CC112X_LQI_EST_BM               0x7F


//Command strobe registers
#define CC112X_SRES                     0x30        //SRES    - Reset chip.
#define CC112X_SFSTXON                  0x31        //SFSTXON - Enable and calibrate frequency synthesizer.
#define CC112X_SXOFF                    0x32        //SXOFF   - Turn off crystal oscillator.
#define CC112X_SCAL                     0x33        //SCAL    - Calibrate frequency synthesizer and turn it off.
#define CC112X_SRX                      0x34        //SRX     - Enable RX. Perform calibration if enabled.
#define CC112X_STX                      0x35        //STX     - Enable TX. If in RX state, only enable TX if CCA passes.
#define CC112X_SIDLE                    0x36        //SIDLE   - Exit RX/TX, turn off frequency synthesizer.
#define CC112X_AFC                      0x37        //AFC     - Automatic Frequency Correction
#define CC112X_SWOR                     0x38        //SWOR    - Start automatic RX polling sequence (Wake-on-Radio)
#define CC112X_SPWD                     0x39        //SPWD    - Enter power down mode when CSn goes high.
#define CC112X_SFRX                     0x3A        //SFRX    - Flush the RX FIFO buffer.
#define CC112X_SFTX                     0x3B        //SFTX    - Flush the TX FIFO buffer.
#define CC112X_SWORRST                  0x3C        //SWORRST - Reset real time clock.
#define CC112X_SNOP                     0x3D        //SNOP    - No operation. Returns status byte.

//Chip states returned in status byte
#define CC112X_STATE_IDLE               0
#define CC112X_STATE_RX                 0x10
#define CC112X_STATE_TX                 0x20
#define CC112X_STATE_FSTXON             0x30
#define CC112X_STATE_CALIBRATE          0x40
#define CC112X_STATE_SETTLING           0x50
#define CC112X_STATE_RXFIFO_ERROR       0x60
#define CC112X_STATE_TXFIFO_ERROR       0x70

#define RADIO_BURST_ACCESS              0x40
#define RADIO_SINGLE_ACCESS             0
#define RADIO_READ_ACCESS               0x80
#define RADIO_WRITE_ACCESS              0

//Bit fields in the chip status byte
#define STATUS_CHIP_RDYn_BM             0x80
#define STATUS_STATE_BM                 0x70
#define STATUS_FIFO_BYTES_AVAILABLE_BM  0x0F

//Defines used for the manual calibration
#define VCDAC_START_OFFSET 2
#define FS_VCO2_INDEX 0
#define FS_VCO4_INDEX 1
#define FS_CHP_INDEX 2


//Minimum definitions for reading chip ID and chip VERSION
#define CC1120_READ_BURST               0xC0
#define CC1120_EXT_MEM_ACCESS           0x2F
#define CC1120_PARTNUMBER_ADDR          0x8F
#define CC1120_PARTVERSION_ADDR         0x90



#define CC112x_FreqValue424p7000 0x6A2CCD  //424.7000MHz
#define CC112x_FreqValue447p2625 0x6FD0CD  //447.2625MHz
#define CC112x_FreqValue447p6750 0x6FEB33  //447.6750MHz
#define CC112x_FreqValue447p8625 0x6FF733  //447.8625MHz

#define CC112xRssiOffset  142  //102에서 수정


static int ExistRecvCnt;
static int CarrierTime_ms;
static int CurrentChannel;
static int CurrentPower;
static BYTE Calibrated;
static BYTE NowInSleep;
static JTIME RFDataReceiveTime;     //Sleep Mode 에서도 동작해야 하므로 GetTickCount()는 사용할 수 없음


typedef struct
    {
    WORD Addr;
    WORD Data;
    }InitReg_T;


//Register settings imported directly from SmartRF Studio
//Modem settings for 1.2kbps 2-FSK  7.812kHz RXBW and 4 Byte preamble(")
static CONST InitReg_T PacketModeCC112xRfSettings[]=
    {
    {CC112X_IOCFG3, 0x11},      //Reg00     //CARRIER_SENSE_VALID, 0x11: 캐리어가 감지되면 'H', 스코프로 찍어보기 위함
    {CC112X_IOCFG2, 0x06},      //Reg01     //6:PKT_SYNC_RXTX, 7:CRCOK, CRC가 잡히면 'H', 데이터 수신이 끝나면 Low됨
    {CC112X_IOCFG0, 0x3C},      //Reg03     //RSSI_VALID  ;20120418
    {CC112X_FS_DIG1, 0},        //Reg2F12
    {CC112X_FS_DIG0, 0x5F},     //Reg2F13
    {CC112X_FS_CAL0, 0x0E},     //Reg2F17
    {CC112X_FS_DIVTWO, 0x03},   //Reg2F19
    {CC112X_FS_DSM0, 0x33},     //Reg2F1B
    {CC112X_FS_DVC0, 0x17},     //Reg2F1D
    {CC112X_FS_PFD, 0x50},      //Reg2F1F
    {CC112X_FS_PRE, 0x6E},      //Reg2F20
    {CC112X_FS_REG_DIV_CML, 0x14}, //2F21
    {CC112X_FS_SPARE, 0xAC},    //Reg2F22
    {CC112X_XOSC5, 0x0E},       //Reg2F32
    {CC112X_XOSC4, 0xA0},       //Reg2F33
    {CC112X_XOSC3, 0xC7},       //Reg2F34
    {CC112X_XOSC1, 0x03},       //Reg2F36
    {CC112X_ANALOG_SPARE, 0},   //Reg2F38
    {CC112X_FIFO_CFG, 0x80},    //Reg1E     //Automatically flushes the last packet received in the RX FIFO if a CRC error
                                            //이값을 Bit7을 0으로 하면 자체 CRC가 빠짐
    {CC112X_DEV_ADDR, 0},       //Reg1F
    {CC112X_SETTLING_CFG, 0x03},//Reg20
    {CC112X_FS_CFG, 0x14},      //Reg21
    {CC112X_PKT_CFG2, 0},       //Reg26
    {CC112X_PKT_CFG1, 0x05},    //Reg27     //Address check off and CRC check on
    {CC112X_PKT_CFG0, 0x20},    //Reg28
    {CC112X_PKT_LEN, 0xFF},     //Reg2E
    {CC112X_RFEND_CFG1, 0x0F},  //Reg29     //Stay in RX after RX, No timeout for sync word search
    {CC112X_RFEND_CFG0, 0},     //Reg2A     //IDle after TX, no interferring from MARC, AUTOMANSPI_WakeupFromSleep() 에도 있음
    //{CC112X_FREQ2, 0x6C},     //Reg2F0C   //frf = fvco/LO_DIV where fvco = (FREQ/(2^16)*fxosc) + (FREQOFF/(2^18)*fxosc)
    //{CC112X_FREQ1, 0x80},     //Reg2F0D
    //{CC112X_FREQ0, 0},        //Reg2F0E
    {CC112X_SYNC3, 0x93},       //Reg04
    {CC112X_SYNC2, 0x0B},       //Reg05
    {CC112X_SYNC1, 0x51},       //Reg06
    {CC112X_SYNC0, 0xDE},       //Reg07
    {CC112X_SYNC_CFG1, 0x4B},   //Reg08     //0x0B;20120418
    {CC112X_SYNC_CFG0, 0x14},   //Reg09     //0X17;20120418
    {CC112X_DEVIATION_M, 0x06}, //Reg0A     //(2000 Hz)
    {CC112X_MODCFG_DEV_E, 0x02},//Reg0B     //(2000 Hz)     //0X03
    {CC112X_DCFILT_CFG, 0x1C},  //Reg0C
    {CC112X_PREAMBLE_CFG1, 0x18},  //0D     //4 byte preamble, PreambleWord=0xAA, 매뉴얼 p74
    {CC112X_PREAMBLE_CFG0, 0x2A},  //0E     //0x2A ;20120418
    {CC112X_FREQ_IF_CFG, 0x40}, //Reg0F     //(62500 Hz)
    {CC112X_IQIC, 0xC6},        //Reg10
    {CC112X_CHAN_BW, 0x50},     //Reg11     //0x14:10k, 0x15:9.6k, 0x50:7.812k  //20120418
    {CC112X_MDMCFG1, 0x46},     //Reg12
    {CC112X_MDMCFG0, 0x05},     //Reg13
    {CC112X_DRATE2, 0x43},      //Reg14     //(1200 bps)
    {CC112X_DRATE1, 0xA9},      //Reg15     //(1200 bps)
    {CC112X_DRATE0, 0x2A},      //Reg16     //(1200 bps)
    {CC112X_AGC_REF, 0x23},     //Reg17     //10LOG10(7.812*10^3)-4 ;20120418
    {CC112X_AGC_CS_THR, 0xEF},  //Reg18
    {CC112X_AGC_GAIN_ADJUST, 0x00},//19     //-119
            //AGC_CS_THR=0x10, GAIN_ADJUST=0x23 ... RSSI -119 dbm 에서 깨어나라
            //AGC_CS_THR=0x10, GAIN_ADJUST=0x1A ... RSSI -110 dbm 에서 깨어나라
            //AGC_CS_THR=0x10, GAIN_ADJUST=0x13 ... RSSI -103 dbm 에서 깨어나라
            //AGC_CS_THR=0x10, GAIN_ADJUST=0x04 ... RSSI -86 dbm 에서 깨어나라
            //AGC_CS_THR=0x19, GAIN_ADJUST=0x00 ... RSSI -72 dbm 에서 깨어나라
            //AGC_CS_THR=0x19, GAIN_ADJUST=0x04 ... RSSI -76 dbm 에서 깨어나라

            //오토만에서 제공한 다른 값
            //AGC_CS_THR=0x02, GAIN_ADJUST=0x00 ... about -100dBm
            //AGC_CS_THR=0x00, GAIN_ADJUST=0x00 ... about -102dBm
            //AGC_CS_THR=0xFF, GAIN_ADJUST=0x00 ... about -103dBm
            //AGC_CS_THR=0xFA, GAIN_ADJUST=0x00 ... about -108dBm
            //AGC_CS_THR=0xF8, GAIN_ADJUST=0x00 ... about -110dBm
            //AGC_CS_THR=0xEF, GAIN_ADJUST=0x00 ... about -119dBm
            //AGC_CS_THR=0xEB, GAIN_ADJUST=0x00 ... about -123dBm (1200bps에서 최저)

    {CC112X_AGC_CFG3, 0x91},    //Reg1A
    {CC112X_AGC_CFG2, 0x20},    //Reg1B
    {CC112X_AGC_CFG1, 0xA9},    //Reg1C
    {CC112X_AGC_CFG0, 0xCF},    //Reg1D     //AUTOMANSPI_WakeupFromSleep()에도 있음
    //{CC112X_PA_CFG2, 0x74},   //Reg2B
    {CC112X_PA_CFG1, 0x56},     //Reg2C
    //{CC112X_PA_CFG0, 0x6C},   //Reg2D
    {CC112X_IF_MIX_CFG, 0},     //Reg2F00
    {CC112X_FREQOFF_CFG, 0x22}, //Reg2F01
    {CC112X_TOC_CFG, 0x0B},     //Reg2F02
    {CC112X_SOFT_TX_DATA_CFG, 0}   //2F05
    };



LOCAL(VOID) CC112xSpiBegin(VOID)
    {
    PortOut(PO_RFMNSS, LOW);
    //Delay_us(10);             //50ns
    }

LOCAL(VOID) CC112xSpiEnd(VOID)
    {
    //Delay_us(10);
    PortOut(PO_RFMNSS, HIGH);   //Disable chip
    }


LOCAL(int) CC112xSpiTx(int WriteData)
    {
    #if SPI_GPIOCONTROL
    int I, ReadData=0;

    //Printf("SpiOut=%02X"CRLF, WriteData);
    for (I=0; I<8; I++)
        {
        if ((WriteData>>7)&1) RFMSPIMOSItoHIGH; else RFMSPIMOSItoLOW;   //PortOut(PO_RFMMOSI, (WriteData>>7)&1);
        RFMSPICLKtoHIGH;                    //PortOut(PO_RFMSCK, HIGH); //Delay_us(5);
        ReadData<<=1;
        if (RFMSPIMISOCHECK) ReadData|=1;   //if (PortIn(PI_RFMMISO)!=0) ReadData|=1;
        RFMSPICLKtoLOW;                     //PortOut(PO_RFMSCK, LOW); //Delay_us(5);
        WriteData<<=1;
        }
    #else
    int ReadData=0; //상위 24비트 Clear
    SPI_TransmitReceive8(SPI_RFM, (LPCBYTE)&WriteData, 1, (LPBYTE)&ReadData, 1, 0, 50);
    #endif
    return ReadData;
    }



LOCAL(int) CC112xSpiRx(VOID)
    {
    #if SPI_GPIOCONTROL
    int I, Data=0;

    for (I=0; I<8; I++)
        {
        RFMSPICLKtoHIGH;                    //PortOut(PO_RFMSCK, HIGH); //Delay_us(5);
        Data<<=1;
        if (RFMSPIMISOCHECK) Data|=1;       //if (PortIn(PI_RFMMISO)!=0) Data|=1;
        RFMSPICLKtoLOW;                     //PortOut(PO_RFMSCK, LOW); //Delay_us(5);
        }
    //Printf("SpiIn=%02X"CRLF, Data);
    #else
    int Data=0; //상위 24비트 Clear
    SPI_TransmitReceive8(SPI_RFM, NULL, 0, (LPBYTE)&Data, 1, 0, 50);
    #endif
    return Data;
    }



LOCAL(BOOL) CC112xWaitMisoLow(VOID)     //소요시간 208uS
    {
    int I=200;

    #if !SPI_GPIOCONTROL
    RFMMISO_SetInput();
    #endif
    while (PortIn(PI_RFMMISO))
        {
        Delay_us(10);
        if (--I==0) break;
        }
    #if !SPI_GPIOCONTROL
    RFMMISO_SetSPI();
    #endif
    return I;
    }




LOCAL(int) CC112xSpiAccessReg(int AccessType, int Addr, LPBYTE lpData, int Len)
    {
    int ExtAddr, RegAddr, RfStatus=0;

    ExtAddr=Addr>>8;
    RegAddr=Addr & 0xFF;

    CC112xSpiBegin();
    CC112xWaitMisoLow();

    if (ExtAddr==0)
        {
        AccessType|=RegAddr;
        RfStatus=CC112xSpiTx(AccessType);
        }
    else{
        AccessType|=ExtAddr;
        RfStatus=CC112xSpiTx(AccessType);
        CC112xSpiTx(RegAddr);
        }

    if (AccessType & RADIO_READ_ACCESS)
        {
        if (AccessType & RADIO_BURST_ACCESS)
            {
            #if SPI_GPIOCONTROL
            int I;
            for (I=0; I<Len; I++) *lpData++=CC112xSpiRx();
            #else
            SPI_TransmitReceive8(SPI_RFM, NULL,0, lpData,Len, 0, 100);
            #endif
            }
        else *lpData=CC112xSpiRx();
        }
    else{
        if (AccessType & RADIO_BURST_ACCESS)
            {
            #if SPI_GPIOCONTROL
            int I;
            for (I=0; I<Len; I++) CC112xSpiTx(*lpData++);
            #else
            SPI_TransmitReceive8(SPI_RFM, lpData,Len, NULL,0, 0, 100);
            #endif
            }
        else CC112xSpiTx(*lpData);
        }

    CC112xSpiEnd();
    return RfStatus;
    }


LOCAL(int) CC112xReadReg(int Addr)
    {
    BYTE Buff[4];

    CC112xSpiAccessReg(RADIO_BURST_ACCESS|RADIO_READ_ACCESS, Addr, Buff, 1);
    #if CHIPDEBUG
    Printf("CC112x ReadReg  %02Xh -> %02Xh"CRLF, Addr, Buff[0]);
    #endif
    return Buff[0];
    }

LOCAL(int) CC112xWriteReg(int Addr, int Data)
    {
    BYTE Buff[4];

    #if CHIPDEBUG
    Printf("CC112x WriteReg %02Xh <- %02Xh"CRLF, Addr, Data);
    #endif
    Buff[0]=(BYTE)Data;
    return CC112xSpiAccessReg(RADIO_BURST_ACCESS|RADIO_WRITE_ACCESS, Addr, Buff, 1);
    }

LOCAL(int) CC112xReadRxFifo(LPBYTE lpData, int Len)
    {
    #if CHIPDEBUG
    int I, Rslt;

    Rslt=CC112xSpiAccessReg(RADIO_BURST_ACCESS|RADIO_READ_ACCESS, CC112X_SINGLE_TXFIFO, lpData, Len);
    Printf("CC112x ReadReg  %02Xh ->", CC112X_SINGLE_TXFIFO);
    for (I=0; I<Len; I++) Printf(" %02Xh", lpData[I]);
    Printf(CrLfStr);
    return Rslt;
    #else
    return CC112xSpiAccessReg(RADIO_BURST_ACCESS|RADIO_READ_ACCESS, CC112X_SINGLE_TXFIFO, lpData, Len);
    #endif
    }

LOCAL(int) CC112xWriteTxFifo(LPCBYTE lpData, int Len)
    {
    #if CHIPDEBUG
    int I;

    Printf("CC112x WriteReg %02Xh <-", CC112X_SINGLE_TXFIFO);
    for (I=0; I<Len; I++) Printf(" %02Xh", lpData[I]);
    Printf(CrLfStr);
    #endif
    return CC112xSpiAccessReg(RADIO_BURST_ACCESS|RADIO_WRITE_ACCESS, CC112X_SINGLE_TXFIFO, (LPBYTE)lpData, Len);
    }




//-----------------------------------------------------------------------------
//      trxSpiCmdStrobe()
//
//      Send command strobe to the radio. Returns status byte read
//      during transfer of command strobe. Validation of provided
//      is not done. Function assumes chip is ready.
//-----------------------------------------------------------------------------
LOCAL(int) CC112xWriteCmd(int Cmd)
    {
    int RfStatus;

    #if CHIPDEBUG
    Printf("CC112x WriteCmd %02Xh"CRLF, Cmd);
    #endif
    CC112xSpiBegin();
    CC112xWaitMisoLow();
    RfStatus=CC112xSpiTx(Cmd);
    CC112xSpiEnd();
    return RfStatus;
    }




//-----------------------------------------------------------------------------
//      CC112x칩 다시 초기화
//-----------------------------------------------------------------------------
VOID WINAPI AUTOMANSPI_Reinit(VOID)
    {
    Printf("SENSOR: Reinit CC112x"CRLF);
    Calibrated=0;
    AUTOMANSPI_Init(CurrentChannel, CurrentPower);
    #if REINITNOTIFY
    NotifyDebugAlarm();     //버그를 잡기위해 알람울리고 Led 빠르게 번쩍이게 함
    #endif
    }




//-----------------------------------------------------------------------------
// SNOP (No Operation Strobe)를 전송하여 라디오의 상태와 TX FIFO의 빈 바이트 수를 가져옴
//
//  Status byte:
//   -------------------------------------------------------------------------
//  | CHIP_RDY | STATE[2:0] | FIFO_BYTES_AVAILABLE (free bytes in the TX FIFO |
//   -------------------------------------------------------------------------
LOCAL(int) CC112xGetTxStatus(VOID)
    {
    return CC112xWriteCmd(CC112X_SNOP);     //SNOP (No Operation)
    }



//-----------------------------------------------------------------------------
//      Chip이 Idle 상태가 될 때까지 기다림
//
//응답 Bit7    1: 전원과 크리스탈이 안정될 동안 1을 유지, 이 값이 0일 때 SPI를 사용 가능
//     Bit6-4  0:Idle, 1:Rx, 2:Tx, 3:FstXon, 4:Calibrate, 5:처리중, 6:Rx Fifo Err, 7:Tx Fifo Err
//     Bit3-0  Reserved
//-----------------------------------------------------------------------------
LOCAL(VOID) CC112xWaitChipReady(VOID)
    {
    int   ChipStatus;
    DWORD Time;

    Time=GetTickCount();
    while ((ChipStatus=CC112xGetTxStatus()) & 0xF0)
        {
        if (GetTickCount()-Time>=RFM_DELAY)
            {
            Printf("SENSOR: CC112xWaitChipReady() Timeout (ChipStatus=0x%02X)"CRLF, ChipStatus);
            AUTOMANSPI_Reinit();
            break;
            }
        }
    }



//-----------------------------------------------------------------------------
//      칩내부의 MCU가 Idle상태가 될 때까지 기다림
//
//  응답 Bit6-5  0:처리중, 1:TX, 2:Idle, 3:RX
//       Bit4-0  0:Sleep, 1:Idle, 2:XOff, 13:Rx, 14:RxEnd, 19:Tx, 20:TxEnd ... 매뉴얼 101p 참조
//-----------------------------------------------------------------------------
LOCAL(VOID) CC112xWaitMarcIdle(VOID)
    {
    int   MarcState;
    DWORD Time;

    Time=GetTickCount();
    while ((MarcState=CC112xReadReg(CC112X_MARCSTATE))!=0x41)
        {
        if (GetTickCount()-Time>=RFM_DELAY)
            {
            Printf("SENSOR: CC112xWaitMarcIdle() Timeout (MarcState=0x%02X)"CRLF, MarcState);
            AUTOMANSPI_Reinit();
            break;
            }
        }
    }



#if 0
//-----------------------------------------------------------------------------
//      라디오의 상태 및 RXFIFO의 사용 가능한 바이트 수를 얻음
//
//  Status byte:
//   ------------------------------------------------------------------------------
//  | CHIP_RDY | STATE[2:0] | FIFO_BYTES_AVAILABLE (available bytes in the RX FIFO |
//   ------------------------------------------------------------------------------
LOCAL(int) CC112xGetRxStatus(VOID)
    {
    return CC112xWriteCmd(CC112X_SNOP|RADIO_READ_ACCESS);
    }
#endif




LOCAL(int) CC112xGetChipID(VOID)
    {
    int DevID, DevVer;

    //Pull CSn low and wait for MISO goes low => clock ready
    CC112xSpiBegin();
    CC112xWaitMisoLow(); //Wait for MISO Low
    CC112xSpiTx(CC1120_READ_BURST|CC1120_EXT_MEM_ACCESS);     //[7:6]=READ_BURST, [5:0]=extended memory access address
    CC112xSpiTx(CC1120_PARTNUMBER_ADDR);                      //[7:0]=Partnumber address
    DevID=CC112xSpiRx();
    DevVer=CC112xSpiRx();
    CC112xSpiEnd();
    return (DevID<<8)|DevVer;
    }



//-----------------------------------------------------------------------------
//          Put radio in RX and enables interrupt
//-----------------------------------------------------------------------------
LOCAL(VOID) CC112xIdleRx(BOOL FlushRxBuffFg)
    {
    if (FlushRxBuffFg) CC112xWriteCmd(CC112X_SFRX);
    CC112xWriteCmd(CC112X_SIDLE);
    CC112xWaitChipReady();
    CC112xWriteCmd(CC112X_SRX);
    }



LOCAL(VOID) CC112xManualCal(VOID)
    {
    int  OriginalFsCal2;
    BYTE CalRsltForVcdacStartHigh[3], CalRsltForVcdacStartMid[3];

    // 1) Set VCO cap-array to 0 (FS_VCO2 = 0)
    CC112xWriteReg(CC112X_FS_VCO2, 0);

    // 2) Start with high VCDAC (original VCDAC_START + 2):
    OriginalFsCal2=CC112xReadReg(CC112X_FS_CAL2);
    CC112xWriteReg(CC112X_FS_CAL2, OriginalFsCal2+VCDAC_START_OFFSET);

    // 3) Calibrate and wait for calibration to be done (radio back in IDLE state)
    CC112xWriteCmd(CC112X_SCAL);
    CC112xWaitMarcIdle();

    // 4) Read FS_VCO2, FS_VCO4 and FS_CHP register obtained with high VCDAC_START value
    CalRsltForVcdacStartHigh[FS_VCO2_INDEX]=CC112xReadReg(CC112X_FS_VCO2);
    CalRsltForVcdacStartHigh[FS_VCO4_INDEX]=CC112xReadReg(CC112X_FS_VCO4);
    CalRsltForVcdacStartHigh[FS_CHP_INDEX]=CC112xReadReg(CC112X_FS_CHP);

    // 5) Set VCO cap-array to 0 (FS_VCO2 = 0)
    CC112xWriteReg(CC112X_FS_VCO2, 0);

    // 6) Continue with mid VCDAC (original VCDAC_START):
    CC112xWriteReg(CC112X_FS_CAL2, OriginalFsCal2);

    // 7) Calibrate and wait for calibration to be done (radio back in IDLE state)
    CC112xWriteCmd(CC112X_SCAL);
    CC112xWaitMarcIdle();

    // 8) Read FS_VCO2, FS_VCO4 and FS_CHP register obtained with mid VCDAC_START value
    CalRsltForVcdacStartMid[FS_VCO2_INDEX]=CC112xReadReg(CC112X_FS_VCO2);
    CalRsltForVcdacStartMid[FS_VCO4_INDEX]=CC112xReadReg(CC112X_FS_VCO4);
    CalRsltForVcdacStartMid[FS_CHP_INDEX]=CC112xReadReg(CC112X_FS_CHP);

    // 9) Write back highest FS_VCO2 and corresponding FS_VCO and FS_CHP result
    if (CalRsltForVcdacStartHigh[FS_VCO2_INDEX]>CalRsltForVcdacStartMid[FS_VCO2_INDEX])
        {
        CC112xWriteReg(CC112X_FS_VCO2, CalRsltForVcdacStartHigh[FS_VCO2_INDEX]);
        CC112xWriteReg(CC112X_FS_VCO4, CalRsltForVcdacStartHigh[FS_VCO4_INDEX]);
        CC112xWriteReg(CC112X_FS_CHP,  CalRsltForVcdacStartHigh[FS_CHP_INDEX]);
        }
    else{
        CC112xWriteReg(CC112X_FS_VCO2, CalRsltForVcdacStartMid[FS_VCO2_INDEX]);
        CC112xWriteReg(CC112X_FS_VCO4, CalRsltForVcdacStartMid[FS_VCO4_INDEX]);
        CC112xWriteReg(CC112X_FS_CHP,  CalRsltForVcdacStartMid[FS_CHP_INDEX]);
        }
    }



LOCAL(VOID) CC112xSendPacket(LPCBYTE lpData)
    {
    CC112xWriteCmd(CC112X_SIDLE);           //소요시간:264uS, Be sure radio is in IDLE state
    CC112xWaitChipReady();                  //소요시간:288uS
    CC112xWriteTxFifo(lpData, lpData[0]+1); //소요시간:768uS, Write array to FIFO
    CC112xWriteCmd(CC112X_STX);             //소요시간:264uS
    CC112xWaitMarcIdle();
    }




//-----------------------------------------------------------------------------
//      WOR: Wake on Radio
//
//      Set up radio for WOR operation terminating on RSSI
//          Radio state 는 Idle에서 WOR로 전환됨
//          Note: assumes chip is ready
//-----------------------------------------------------------------------------
VOID WINAPI AUTOMANSPI_GoSleepForWOR(int SniffInterval_ms)
    {
    CC112xWriteCmd(CC112X_SIDLE);

    //CC112xWriteReg(CC112X_AGC_CS_THR, 0x37);      //Set carrier sense gate mode on recevier, and set AGC carrier sense threshold

    CC112xWriteReg(CC112X_AGC_CFG0, 0xC3);          //Set RSSI valid count to minimum: 2 COMPUTATIONS
    CC112xWriteReg(CC112X_AGC_CFG1, 0xA0);          //Set AGC window size to 8 samples, Set AGC settle wait time to 24 SAMPLES
    CC112xWriteReg(CC112X_SETTLING_CFG, 0x03);      //Turning off autocal
    //CC112xWriteReg(CC112X_WOR_CFG1, 0x08);        //Setting WOR mode to NORMAL_MODE & WOR timer resolution to HIGH_RESOLUTION
    CC112xWriteReg(CC112X_WOR_EVENT0_MSB, SniffInterval_ms>>8); //스니프간격: 0xE80 -> 3.73Sec (1ms단위)
    CC112xWriteReg(CC112X_WOR_EVENT0_LSB, SniffInterval_ms);    //            0x3E8 -> 1초

    CC112xWriteReg(CC112X_WOR_CFG1, 0x4E);          //0x0E->0x4E, Event1 timeout set to 1 ms to allow XOSC anddigital regulator to startup and settle

    if (Calibrated==0)
        {
        CC112xWriteReg(CC112X_WOR_CFG0, 0x24);      //Disable Event2 configuration. RC_OSC calibration interval disabled.
                                                    //Setting RC osc enable
                                                    //Setting RC calibration mode to continuous calibration
        CC112xWriteCmd(CC112X_SIDLE);               //Strobe idle to calibrate
        CC112xWriteReg(CC112X_WOR_CFG0, 0x20);      //Disable RC osc calibration
        Calibrated=1;
        }

    CC112xWriteReg(CC112X_RFEND_CFG1, 0x0F);        //Setting RX timeout to never
                                                    //in case of noise making the radio hang in RX this can be set to force powerdown after a certain time in RX
    CC112xWriteReg(CC112X_RFEND_CFG0, 0x09);        //Setting RX_TERMINATE_ON_RSSI

    //CC112xWriteReg(CC112X_IOCFG2, 17);            //20:MCU_WAKEUP, GPIO2 핀을 Wakeup 핀으로

    //trxIdleWOR()
    CC112xWriteCmd(CC112X_SWORRST);                 //Reset WOR otherwise we might get stuck in powerdown
    CC112xWriteCmd(CC112X_SWOR);

    NowInSleep=INSLEEPFORWOR;
    Printf("RFM Power down for WOR"CRLF);
    }




//-----------------------------------------------------------------------------
//      RF칩을 거의 꺼놓음 (Sniff 안됨)
//-----------------------------------------------------------------------------
VOID WINAPI AUTOMANSPI_GoSleep(VOID)
    {
    CC112xWriteCmd(CC112X_SIDLE);
    CC112xWaitChipReady();
    CC112xWriteCmd(CC112X_SPWD);

    NowInSleep=INSLEEP;
    Printf("RFM Sleep In"CRLF);
    }




//-----------------------------------------------------------------------------
//      WOR에서 깨어나면 연속 수신이 안되어 레지스터 복귀
//-----------------------------------------------------------------------------
VOID WINAPI AUTOMANSPI_WakeupFromSleep(VOID)
    {
    CC112xWriteReg(CC112X_AGC_CFG0, 0xCF);
    CC112xWriteReg(CC112X_WOR_CFG1, 0x08);      //0x0E->0x4E, Event1 timeout set to 1 ms to allow XOSC anddigital regulator to startup and settle
    CC112xWriteReg(CC112X_RFEND_CFG0, 0x00);    //Setting RX_TERMINATE_ON_RSSI
    CC112xIdleRx(FALSE);                        //RF모듈에서 데이터 수신으로 깨어나는 경우 수신FIFO를 지우면 안됨, 캐리어로 깨어나는 경우에는 지워도 됨) (2019-10-01)
    NowInSleep=HASBEENAWAKEUP;
    }




//-----------------------------------------------------------------------------
//      출력 파워 설정
//-----------------------------------------------------------------------------
VOID WINAPI AUTOMANSPI_SetOutputPower(int Pow)
    {
    int Value;
    static CONST WORD TxRfPowerSettings[]=
        {
        0x5D3C, //0 dBm
        0x5F3C, //1 dBm
        0x6244, //2 dBm
        0x644C, //3 dBm
        0x664C, //4 dBm
        0x6954, //5 dBm
        0x6B54, //6 dBm
        0x6D5C, //7 dBm
        0x6F5C, //8 dBm
        0x7264, //9 dBm
        0x746C, //10 dBm
        0x776C, //11 dBm
        0x7974, //12 dBm
        0x7B74, //13 dBm
        0x7D7C, //14 dBm
        0x7F7C  //15 dBm
        };

    Value=TxRfPowerSettings[Pow];
    CC112xWriteReg(CC112X_PA_CFG2, Value>>8);
    CC112xWriteReg(CC112X_PA_CFG0, Value);
    }




//-----------------------------------------------------------------------------
//      채널에 해당하는 주파수 설정
//
//      24Bit레지스터값=출력주파수*0x10000/4000000  (4000000=fxosc)
//
//      채널주파수=12500*Ch+424700000
//      CH00~20    ... 424,700,000 ~ 424,950,000
//      CH25~49    ... 447,262,500 ~ 447,562,500 (한국소방용)
//      CH50~70    ... 447,600,000 ~ 447,850,000 (71~은 75~와 중복됨)
//      CH75~85    ... 447,862,500 ~ 447,987,500 (한국일반용)
//        86~99    ... 448,000,000 ~ 448,162,500 (개발용으로 사용)
//      CH100~223  ... 433,052,500 ~ 434,590,000 (UAE소방용)
//
//  12500*23+433,052,500*65536/4000000 = 6C55C2h
//-----------------------------------------------------------------------------
#define CHWIDTH     12500
#define FREQXOSC    4000000
LOCAL(VOID) SetChannel(int Ch)
    {
    DWORD FreqOut, FreqValue;

    if (Ch<25)       FreqOut=Ch      *CHWIDTH+424700000;    //~425,000,000
    else if (Ch<50)  FreqOut=(Ch-25) *CHWIDTH+447262500;    //~447,562,500
    else if (Ch<75)  FreqOut=(Ch-50) *CHWIDTH+447600000;    //
    else if (Ch<100) FreqOut=(Ch-75) *CHWIDTH+447862500;    //~448,125,000
    else             FreqOut=(Ch-100)*CHWIDTH+433052500;

    FreqValue=UMulDiv(FreqOut, 65536, FREQXOSC);
    if (SysDebugFg>=2) Printf("SENSOR: Ch=%u, Freq=%,uHz, RegValue=%Xh"CRLF, Ch, FreqOut, FreqValue);

    CC112xWriteReg(CC112X_FREQ2, FreqValue>>16);
    CC112xWriteReg(CC112X_FREQ1, FreqValue>>8);
    CC112xWriteReg(CC112X_FREQ0, FreqValue);
    }


VOID WINAPI AUTOMANSPI_SetChannel(int MyCh)
    {
    CC112xWriteCmd(CC112X_SIDLE);
    SetChannel(MyCh);
    CC112xManualCal();
    CC112xIdleRx(TRUE);     //수신모드로 설정
    }


LOCAL(UINT) GetOutFrequency(VOID)
    {
    DWORD FreqValue;

    FreqValue=CC112xReadReg(CC112X_FREQ2);
    FreqValue<<=8; FreqValue|=CC112xReadReg(CC112X_FREQ1);
    FreqValue<<=8; FreqValue|=CC112xReadReg(CC112X_FREQ0);
    return UMulDiv(FreqValue, FREQXOSC, 65536);
    }



LOCAL(int) ConvRssiValue(int RegRssi)
    {
    if (RegRssi>127) RegRssi-=256;
    return RegRssi-CC112xRssiOffset;
    }

int WINAPI AUTOMANSPI_ReadRSSI(VOID)
    {
    return ConvRssiValue(CC112xReadReg(CC112X_RSSI1));
    }



#if 0
//-----------------------------------------------------------------------------
//      패킷을 찍어봄
//-----------------------------------------------------------------------------
LOCAL(VOID) PrintPacket(LPCBYTE Buff, int Len)
    {
    int I;

    Printf("RFM: (%d)",Len);
    for (I=0; I<Len; I++) Printf(" %02X", Buff[I]);
    Printf(CRLF);
    }
#endif




//-----------------------------------------------------------------------------
//      Public 시각 패킷을 만듦
//          PktLen   1Byte
//          TIMESIGN 4Byte
//          JTIME    4Byte
//          1/100sec 1Byte
//          Vector   1Byte
//          CRC16    2Byte
//
//  2021-09-13 ST==NULL 이면 JTIME을 0으로 송출하며 시간을 달라는 의미임
//-----------------------------------------------------------------------------
#define PUBLICTIMEPKTSIGN   0xFE4FDA00
#define PUBLICTIMEPKTCRC    0xBDA6
VOID WINAPI AUTOMANSPI_MakePublicTimePkt(LPBYTE Buff, CONST SYSTEMTIME *ST)
    {
    Buff[0]=PUBLICTIMEPKTSIZE;
    Poke(Buff+0+1, CurrentChannel|PUBLICTIMEPKTSIGN);   //채널별로 시간을 관리하는데 시간이 다른 옆 채널의 하모닉이 수신되어 시간을 흩어지는 현상이 생김
    Poke(Buff+4+1, ST!=NULL ? PackTotalSecond(ST):0);
    Buff[8+1]=ST!=NULL ? ST->wMilliseconds/10:0;
    Buff[9+1]=0;
    Buff[4+1]^=Buff[8+1];
    Buff[5+1]^=Buff[8+1];
    Poke(Buff+6+1, Peek(Buff+6+1)^Peek(Buff+1));
    PokeW(Buff+10+1, CalculateCrc16(Buff+1, 10, PUBLICTIMEPKTCRC));
    }




//-----------------------------------------------------------------------------
//          PublicTimePacket 인지 확인함
//      (14) 4E 3B 6D BE E7 C1 B6 03 F1 21 E0 4A CE CF
//          TIMESIGN 4Byte
//          JTIME    4Byte
//          1/100sec 1Byte
//          Vector   1Byte
//          CRC16    2Byte
//-----------------------------------------------------------------------------
BOOL WINAPI AUTOMANSPI_ChkPublicTimePkt(LPCBYTE RecvBuff, int Len, JTIME *JT, int *lpms)
    {
    BOOL  Rslt=FALSE;
    DWORD TimeSign;
    BYTE  Buff[PUBLICTIMEPKTSIZE];

    if ((TimeSign=Peek(RecvBuff))==(CurrentChannel|PUBLICTIMEPKTSIGN) && Len==12 && CalculateCrc16(RecvBuff, 10, PUBLICTIMEPKTCRC)==PeekW(RecvBuff+10))
        {
        if (JT!=NULL)
            {
            CopyMem(Buff, RecvBuff, sizeof(Buff));
            Poke(Buff+6, Peek(Buff+6)^TimeSign);
            Buff[4]^=Buff[8];
            Buff[5]^=Buff[8];
            *JT=Peek(Buff+4);
            *lpms=Buff[8]*10;
            }
        Rslt++;
        }
    return Rslt;
    }




//-----------------------------------------------------------------------------
// RecvTotal 17
// Data  14 22 54 1B 05 97 5B 82 9F E0 43 84 BC 6A 7C 5D 2E FE C2 9E FA 62 81
//-----------------------------------------------------------------------------
int WINAPI AUTOMANSPI_RecvPacket(LPBYTE Buff, int BuffSize, int *lpRssI)
    {
    int RecvTotal, DataLen=0;
    #if defined(ROZETAENCRYPTION) || defined(ROZETAENCRYPTION2)
    DWORD EC;
    #endif
    #ifdef AES128ENCRYPTION
    //DWORD Time;
    BYTE Tmp[64];
    #endif
    JOS_CRITICAL_VAR;
    #if KOREA_CERTIFICATION
    static DWORD RecvTime;
    #endif

    if (ExistRecvCnt>0)
        {
        JOS_ENTER_CRITICAL();
        ExistRecvCnt--;
        JOS_EXIT_CRITICAL();

        if (NowInSleep) AUTOMANSPI_WakeupFromSleep();

        if ((RecvTotal=CC112xReadReg(CC112X_NUM_RXBYTES))>0)    //RecvTotal는 실제 데이터크기+3 임
            {
            #if KOREA_CERTIFICATION
            RecvTime=GetTickCount();
            #endif
            if (RecvTotal>BuffSize)
                {
                CC112xReadRxFifo(Buff, BuffSize);               //어떻게 깨졌는지 찍어보기 위해 버퍼만큼만 읽어서 리턴함
                *lpRssI=0;

                //FIFO does not match what we expect, flush FIFO put radio in IDLE
                //CC112xWriteCmd(CC112X_SFRX);    //Flush the RX FIFO buffer. CC112xIdleRx() 안에서 처리함

                Printf("SENSOR: Cleared FIFO"CRLF);
                DataLen=BuffSize;       //호출한 쪽에서 깨진 패킷으로 에러처리하게 하기 위함
                }
            else{
                CC112xReadRxFifo(Buff, RecvTotal);
                //PrintPacket(Buff, RecvTotal);
                *lpRssI=ConvRssiValue(Buff[RecvTotal-2]);

                if ((DataLen=RecvTotal-3)<MINPACKETSIZE)    //-3: 맨앞에 길이, 뒤에 RSSI,LQI
                    {
                    //깨진 패킷이 수신된 경우 RecvTotal이 3이하인 경우가 발생하여 CopyMem()에서 죽는 경우가 발생했음
                    Printf("SENSOR: PacketSize is too small"CRLF);
                    if (DataLen<0) DataLen=0;
                    }
                else{
                    if (AUTOMANSPI_ChkPublicTimePkt(Buff+1, DataLen, NULL, NULL)!=FALSE)
                        {
                        MoveMem(Buff, Buff+1, DataLen);
                        goto RecvEnd;
                        }
                    #ifdef ROZETAENCRYPTION
                    EC=GetEncrypCode();
                    Buff[0]^=(BYTE)EC;          //전송할 때 암호화하지 않았기 때문
                    EncryptDecrypt(Buff, DataLen+1, EC, ED_DECRYPT);    //+1은 맨앞에 길이
                    MoveMem(Buff, Buff+1, DataLen);                     //CopyLength가 음수가 되지 않도록 조심할 것
                    #endif

                    #ifdef ROZETAENCRYPTION2
                    EC=GetEncrypCode();
                    Buff[0]^=(BYTE)EC;          //전송할 때 암호화하지 않았기 때문
                    EncryptDecryptII(Buff, DataLen+1, EC, ED_DECRYPT);  //+1은 맨앞에 길이
                    MoveMem(Buff, Buff+1, DataLen);                     //CopyLength가 음수가 되지 않도록 조심할 것
                    #endif

                    #ifdef AES128ENCRYPTION
                    //Time=GetTickCount();
                    CopyMem(Tmp, Buff+1, DataLen);
                    AES_ECB_Decrypt(Tmp, GetAesEncrypCode(), Buff, DataLen);
                    DataLen=GetMin(Buff[0], BuffSize-1);                //수신 데이터가 깨지면 이상한 값이 크기가 되어 버퍼를 깨뜨림
                    MoveMem(Buff, Buff+1, DataLen);
                    //Printf("DecTime=%d"CRLF,GetTickCount()-Time);     //DecTime=6ms(STM32F091), 8ms(STM32L083)
                    #endif

                    //PrintPacket(Buff, DataLen);
                    //Printf("RssI=%d"CRLF, CC112xReadReg(CC112X_RSSI1));  //데이터 끝에 있는 값과 동일
                    }
                RecvEnd:
                RFDataReceiveTime=GetTodayToSecond();
                }
            AUTOMANSPI_MarkExistRecvData();     //뒤에 수신된 데이터가 있는데 인터럽트를 수신하지 못해서 수신FIFO데이터를 가져가지 않으면 이후 수신이 중단된는 현상이 생김
            }
        CC112xIdleRx(TRUE);                     //수신모드로 설정
        }
    #if KOREA_CERTIFICATION
    else{
        if (RecvTime==0) RecvTime=GetTickCount();
        if (GetTickCount()-RecvTime>=4000) {RecvTime=GetTickCount(); AUTOMANSPI_Reinit();}  //정전기를 맞으면 모듈이 죽어서 주기적으로 리셋함
        }
    #endif
    return DataLen;
    }




//-----------------------------------------------------------------------------
// RecvTotal 17
// Data  14 22 54 1B 05 97 5B 82 9F E0 43 84 BC 6A 7C 5D 2E FE C2 9E FA 62 81
//-----------------------------------------------------------------------------
int WINAPI AUTOMANSPI_RecvPacketRaw(LPBYTE Buff, int BuffSize, int *lpRssI)
    {
    int RecvTotal, DataLen=0;
    JOS_CRITICAL_VAR;

    if (ExistRecvCnt>0)
        {
        JOS_ENTER_CRITICAL();
        ExistRecvCnt--;
        JOS_EXIT_CRITICAL();

        if (NowInSleep) AUTOMANSPI_WakeupFromSleep();

        if ((RecvTotal=CC112xReadReg(CC112X_NUM_RXBYTES))>0)    //RecvTotal는 실제 데이터크기+3 임
            {
            if (RecvTotal>BuffSize)
                {
                *lpRssI=0;
                Printf("SENSOR: Cleared FIFO"CRLF);
                DataLen=0;                                      //호출한 쪽에서 깨진 패킷으로 에러처리하게 하기 위함
                }
            else{
                CC112xReadRxFifo(Buff, RecvTotal);
                *lpRssI=ConvRssiValue(Buff[RecvTotal-2]);

                if ((DataLen=RecvTotal-3)<MINPACKETSIZE)        //-3: 맨앞에 길이, 뒤에 RSSI,LQI
                    {
                    Printf("SENSOR: PacketSize is too small"CRLF);
                    DataLen=0;
                    }
                else{
                    MoveMem(Buff, Buff+1, DataLen);             //CopyLength가 음수가 되지 않도록 조심할 것
                    }
                RFDataReceiveTime=GetTodayToSecond();
                }
            AUTOMANSPI_MarkExistRecvData();                     //뒤에 수신된 데이터가 있는데 인터럽트를 수신하지 못해서 수신FIFO데이터를 가져가지 않으면 이후 수신이 중단된는 현상이 생김
            }
        CC112xIdleRx(TRUE);                                     //수신모드로 설정
        }
    return DataLen;
    }




//-----------------------------------------------------------------------------
//      캐리어를 송출시작 또는 송출 중단
//-----------------------------------------------------------------------------
VOID WINAPI AUTOMANSPI_CarrierOut(BOOL OnOff)
    {
    static int  MDMCfg0, MDMCfg1, DevE;
    static BYTE CarrierOut;

    if (OnOff)
        {
        if (CarrierOut==0)
            {
            MDMCfg0=CC112xReadReg(CC112X_MDMCFG0);
            MDMCfg1=CC112xReadReg(CC112X_MDMCFG1);
            DevE=   CC112xReadReg(CC112X_MODCFG_DEV_E);

            CC112xWriteReg(CC112X_MDMCFG1, MDMCfg1&0xBF);   //FIFO_EN (Data in/Out through the serial pin)
            CC112xWriteReg(CC112X_MDMCFG0, MDMCfg0|0x40);   //Transparent mode enabled
            CC112xWriteReg(CC112X_MODCFG_DEV_E, DevE&0xFC); //Deviation_E=0

            CC112xWriteCmd(CC112X_STX);     //캐리어송신 시작
            CarrierOut=1;
            }
        }
    else{
        if (CarrierOut!=0)
            {
            CC112xWriteCmd(CC112X_SIDLE);   //캐리어송신 중단

            // 원래값으로 설정
            CC112xWriteReg(CC112X_MDMCFG1, MDMCfg1);        //FIFO_EN (Data in/Out through the FIFO)
            CC112xWriteReg(CC112X_MDMCFG0, MDMCfg0);        //Transparent mode disable
            CC112xWriteReg(CC112X_MODCFG_DEV_E, DevE);      //Deviation_E=2
            CarrierOut=0;
            }
        }
    }




//-----------------------------------------------------------------------------
//      주어진 기간동안 캐리어를 송출합니다
//-----------------------------------------------------------------------------
LOCAL(VOID) CC112xUnmodulation(UINT ms)
    {
    #ifndef USE_JOS
    DWORD StartTime;
    #endif
    Printf("SENSOR: Carrier Out %dms"CRLF, ms);

    AUTOMANSPI_CarrierOut(ON);
    #ifdef USE_JOS
    Sleep(ms);
    #else
    StartTime=GetTickCount();
    for (;;)
        {
        if (GetTickCount()-StartTime>=ms) break;
        MonitorProc();
        }
    #endif
    AUTOMANSPI_CarrierOut(OFF);
    Printf("SENSOR: Finish Carrier Out"CRLF);
    }



//-----------------------------------------------------------------------------
//      주어진 기간동안 변조된 데이터를 송출합니다
//-----------------------------------------------------------------------------
LOCAL(VOID) CC112xModulation(UINT ms)
    {
    DWORD StartTime;
    BYTE  Buff[128];

    Printf("SENSOR: Modulated RF Out %dms"CRLF, ms);

    StartTime=GetTickCount();
    FillMem(Buff, sizeof(Buff), 0xAA);
    Buff[0]=127;
    for (;;)
        {
        AUTOMANSPI_SendPacketRaw(Buff);
        if (GetTickCount()-StartTime>=ms) break;
        #ifdef USE_JOS
        Sleep(1);
        #else
        MonitorProc();
        #endif
        }
    Printf("SENSOR: Finish Modulated RF Out"CRLF);
    }



//-----------------------------------------------------------------------------
//      캐리어를 미리 출력하고 시간이 경과한 후 이함수를 호출할 수도 있음
//-----------------------------------------------------------------------------
BOOL WINAPI AUTOMANSPI_SendPacket(LPCBYTE SendData, int SendSize)
    {
    int  Rslt=FALSE;
    BYTE Buff[RMPACKETSIZEMAX+4];
    #ifdef AES128ENCRYPTION
    //DWORD Time;
    BYTE Tmp[RMPACKETSIZEMAX+4];
    #endif

    if (NowInSleep) AUTOMANSPI_WakeupFromSleep();

    #ifdef ROZETAENCRYPTION
    if (SendSize>sizeof(Buff)-1) {Printf("SENSOR: Send date Too big"CRLF); goto ProcExit;}  //-1은 실제길이가 추가되므로
    Buff[0]=SendSize;
    CopyMem(Buff+1, SendData, SendSize);
    EncryptDecrypt(Buff, SendSize+1, GetEncrypCode(), ED_ENCRYPT);
    Buff[0]=SendSize;       //맨앞에 길이는 암호화하면 RF모듈이 처리 못함
    #endif

    #ifdef ROZETAENCRYPTION2
    if (SendSize>sizeof(Buff)-1) {Printf("SENSOR: Send date Too big"CRLF); goto ProcExit;}  //-1은 실제길이가 추가되므로
    Buff[0]=SendSize;
    CopyMem(Buff+1, SendData, SendSize);
    EncryptDecryptII(Buff, SendSize+1, GetEncrypCode(), ED_ENCRYPT);
    Buff[0]=SendSize;       //맨앞에 길이는 암호화하면 RF모듈이 처리 못함
    #endif

    #ifdef AES128ENCRYPTION
    //Time=GetTickCount();
    if ((Tmp[0]=(SendSize+1+15) & ~0x0F)>sizeof(Tmp)-2) {Printf("SENSOR: Send date Too big"CRLF); goto ProcExit;}  //SendSize+1은 실제길이가 추가되므로
    Buff[0]=SendSize;
    CopyMem(Buff+1, SendData, SendSize);
    AES_ECB_Encrypt(Buff, GetAesEncrypCode(), Tmp+1, Tmp[0]);
    CopyMem(Buff, Tmp, Tmp[0]+1);
    //Printf("EncTime=%d"CRLF,GetTickCount()-Time);   //EncTime: <1ms
    #endif

    if (CarrierTime_ms)
        {
        AUTOMANSPI_CarrierOut(ON);
        Sleep(CarrierTime_ms);
        CarrierTime_ms=0;
        }
    AUTOMANSPI_CarrierOut(OFF);

    CC112xSendPacket(Buff);
    CC112xIdleRx(TRUE);             //이걸 안해주면 다음 데이수신은 문제 없으나, 캐리어 디텍트 기능이 안됨
    //PrintPacket(Buff, Buff[0]+1);
    Rslt=TRUE;

    ProcExit:
    return Rslt;
    }



//-----------------------------------------------------------------------------
//      주어진 데이터를 그대로 전송함 (첫바이트는 길이임)
//-----------------------------------------------------------------------------
VOID WINAPI AUTOMANSPI_SendPacketRaw(LPCBYTE SendData)
    {
    if (NowInSleep) AUTOMANSPI_WakeupFromSleep();

    if (CarrierTime_ms)
        {
        AUTOMANSPI_CarrierOut(ON);
        Sleep(CarrierTime_ms);
        CarrierTime_ms=0;
        }
    AUTOMANSPI_CarrierOut(OFF);

    CC112xSendPacket(SendData);
    CC112xIdleRx(TRUE);
    //PrintPacket(Buff, Buff[0]+1);
    }



BOOL WINAPI AUTOMANSPI_GetVersion(LPBYTE Buff)
    {
    wsprintf((LPSTR)Buff, "%X", CC112xGetChipID());
    return TRUE;
    }


BOOL WINAPI AUTOMANSPI_Init(int Channel, int Power)
    {
    int I, Rslt=FALSE;

    CurrentChannel=Channel;
    CurrentPower=Power;

    PortOut(PO_RFMRESET, LOW); HAL_Delay(20);
    PortOut(PO_RFMRESET, HIGH); HAL_Delay(3);

    I=CC112xGetChipID();
    if ((I>>8)!=0x48)
        {
        Printf("SENSOR: CC112x Not found"CRLF);
        goto ProcExit;
        }
    Printf("SENSOR: RF Device CC112x, Rev=%02X"CRLF, I&0xFF);

    CC112xWriteCmd(CC112X_SIDLE);

    //CC112xRegConfig()
    for (I=0; I<countof(PacketModeCC112xRfSettings); I++)
        {
        CHAR Name[8], Data[8];

        wsprintf(Name, "%02X", PacketModeCC112xRfSettings[I].Addr);
        GetIniStr("RFMREG", Name, Data, sizeof(Data));
        CC112xWriteReg(PacketModeCC112xRfSettings[I].Addr,
                       Data[0]!=0 ? AtoH(Data, NULL):PacketModeCC112xRfSettings[I].Data);
        }

    AUTOMANSPI_SetOutputPower(Power);
    SetChannel(Channel);
    CC112xManualCal();

    CC112xIdleRx(TRUE);         //수신모드로 설정
    Rslt++;

    ProcExit:
    return Rslt;
    }




//-----------------------------------------------------------------------------
//      레지스터 덤프
//-----------------------------------------------------------------------------
LOCAL(VOID) CC112xDumpReg(UINT Start, UINT End)
    {
    UINT RegAdd;

    for (RegAdd=Start; RegAdd<=End; RegAdd++)
        {
        Printf("%02X:%02X"CRLF, RegAdd, CC112xReadReg(RegAdd));
        //if ((RegAdd & 7)==7) Printf(CrLfStr);
        }
    Printf(CrLfStr);
    }




//-----------------------------------------------------------------------------
//      RFM 모니터 커멘더 처리
//-----------------------------------------------------------------------------
int WINAPI Mon_AutomanSpiRFM(int PortNo, LPCSTR MonCmd, LPCSTR lpArg, LPCSTR CmdLine)
    {
    int I, RegAddr, RegData, Rslt=MONRSLT_SYNTAXERR;

    if (lpArg[0]=='?')
        {
        Printf("RFM Reg [Data] ... Read/Write CC1120 Register"CRLF);
        Printf("RFM CMD Command ... Write CC1120 Command"CRLF);
        Rslt=MONRSLT_EXIT;
        goto ProcExit;
        }

    if (lpArg[0]==0)
        {
        Rslt=MONRSLT_EXIT;
        I=CC112xGetChipID();
        if ((I>>8)!=0x48)
            {
            Printf("SENSOR: CC112x Not found"CRLF);
            goto ProcExit;
            }
        Printf("SENSOR: RF Device CC112x, Rev=%02X"CRLF, I&0xFF);
        Printf("SENSOR: RSSI=%d"CRLF, AUTOMANSPI_ReadRSSI());
        Printf("SENSOR: Chip Status=0x%02X"CRLF, CC112xWriteCmd(CC112X_SNOP) & 0xF0);   //평소값 1F
        Printf("SENSOR: Marc State =0x%02X"CRLF, CC112xReadReg(CC112X_MARCSTATE));      //평소값 6D
        Printf("SENSOR: Channel=%d"CRLF, CurrentChannel);
        Printf("SENSOR: Power=%d"CRLF, CurrentPower);
        Printf("SENSOR: Out Freq=%,u"CRLF, GetOutFrequency());
        goto ProcExit;
        }

    if (CompMemStr(lpArg, "CMD")==0)
        {
        RegData=AtoH(SkipSpace(lpArg+3), &I); if (I==0) goto ProcExit;
        Printf("SENSOR: WriteCmd()=%02X"CRLF, CC112xWriteCmd(RegData));
        Rslt=MONRSLT_OK;
        goto ProcExit;
        }

    if (CompMemStr(lpArg, "CH")==0)
        {
        RegData=AtoI(SkipSpace(lpArg+2), &I); if (I==0) goto ProcExit;
        AUTOMANSPI_SetChannel(RegData);
        Rslt=MONRSLT_OK;
        goto ProcExit;
        }

    if (CompMemStr(lpArg, "PW")==0)
        {
        RegData=AtoI(SkipSpace(lpArg+2), &I); if (I==0) goto ProcExit;
        AUTOMANSPI_SetOutputPower(RegData);
        Rslt=MONRSLT_OK;
        goto ProcExit;
        }

    if (CompMemStr(lpArg, "OUT")==0)    //캐리어 출력 테스트 (무변조)
        {
        UINT Time;
        Time=AtoI(SkipSpace(lpArg+3), &I);
        if (I==0) Time=10*60*1000;
        CC112xUnmodulation(Time);
        CC112xIdleRx(TRUE);
        Rslt=MONRSLT_OK;
        goto ProcExit;
        }

    if (CompMemStr(lpArg, "MOUT")==0)   //변조 데이터 출력 테스트 (CE인증에 필요)
        {
        UINT Time;
        Time=AtoI(SkipSpace(lpArg+4), &I);
        if (I==0) Time=10*60*1000;
        CC112xModulation(Time);
        Rslt=MONRSLT_OK;
        goto ProcExit;
        }

    if (CompMemStr(lpArg, "RX")==0)     //수신모드
        {
        CC112xIdleRx(TRUE);
        Rslt=MONRSLT_OK;
        goto ProcExit;
        }

    if (CompMemStr(lpArg, "ALL")==0)    //모든 레지스터 덤프
        {
        Printf("Base Reg: 00~2E"CRLF);    CC112xDumpReg(0, 0x2E);
        Printf("Ext Reg: 2F00~2F39"CRLF); CC112xDumpReg(0x2F00, 0x2F39);
        Rslt=MONRSLT_OK;
        goto ProcExit;
        }

    RegAddr=AtoH(lpArg, &I); if (I==0) goto ProcExit;
    lpArg=(LPSTR)SkipSpace(lpArg+I);
    if (lpArg[0]==0)
        {
        Printf("SENSOR: Reg%02X=>%02X"CRLF, RegAddr, CC112xReadReg(RegAddr));
        Rslt=MONRSLT_OK;
        }
    else{
        CHAR Name[8], Data[8];

        RegData=AtoH(lpArg, &I); if (I==0) goto ProcExit;
        CC112xWriteReg(RegAddr, RegData);

        wsprintf(Name, "%02X", RegAddr);
        wsprintf(Data, "%02X", RegData);
        WriteIniStr("RFMREG", Name, Data);

        Printf("SENSOR: Reg%02X<=%02X"CRLF, RegAddr, RegData);
        Rslt=MONRSLT_OK;
        }

    ProcExit:
    return Rslt;
    }



//-----------------------------------------------------------------------------
//      LBT 레벨을 리턴함 (이 함수 리턴값 보다 큰 신호가 들어오면 송출을 지연함)
//-----------------------------------------------------------------------------
//int WINAPI AUTOMANSPI_GetLbtLevel(VOID) {return -120;}  //안테나 뽑고 측정한 임계값, 중계기 재전송 패킷과 직접 전송된 패킷을 구분하기 위해 사용했으나 KFI에서는 이것 때문에 LBT Timeout이 발생함
//int WINAPI AUTOMANSPI_GetLbtLevel(VOID) {return -111;}  //5m거리
//int WINAPI AUTOMANSPI_GetLbtLevel(VOID) {return -73;}   //바로 옆에서
int WINAPI AUTOMANSPI_GetLbtLevel(VOID) {return -90;}   //KFI 에서는 이값을 사용해야 함



//-----------------------------------------------------------------------------
//      RF가 출력되기까지의 시간을 기다림
//      심플리피터에서 사용
//-----------------------------------------------------------------------------
int WINAPI AUTOMANSPI_GetWaitForRfOut(VOID) {return 10;}        //2019-10-30 5->10으로 변경


VOID WINAPI AUTOMANSPI_MarkExistRecvData(VOID) {ExistRecvCnt++;}
BOOL WINAPI AUTOMANSPI_IsExistRecvData(VOID) {return ExistRecvCnt;}
VOID WINAPI AUTOMANSPI_SetCarrierTime(int ms) {CarrierTime_ms=ms;}  //1회 송출하면 Clear됨
int  WINAPI AUTOMANSPI_IsInSleep(VOID) {return NowInSleep;}         //WOR 상태 리턴



//-----------------------------------------------------------------------------
//      RFM이 수신 불능상태가 되었을 때 복구하는 루틴 (다시 초기화한 경우 TRUE리턴)
//
// 원인: 신호가 애매하게 작은 경우 CC1120내부의 프로세서가 미쳐서
//       레지스터 값을 바꾸어 버린 것으로 추청함.
//
// 처리: 10분간 수신이 없으면 레지스터값과 설정값을 비교하고 달라진 경우 칩 재설정
//-----------------------------------------------------------------------------
BOOL WINAPI AUTOMANSPI_RecoverChip(BOOL RightNow)
    {
    int I, RegAdd, RegData, SetData, Rslt=FALSE;
    JTIME CurrTime;

    CurrTime=GetTodayToSecond();
    if (RightNow==FALSE && CurrTime-RFDataReceiveTime<10*60) goto ProcExit;     //10분
    RFDataReceiveTime=CurrTime;

    Printf("SENSOR: Check CC112x Register"CRLF);
    for (I=0; I<countof(PacketModeCC112xRfSettings); I++)
        {
        RegAdd=PacketModeCC112xRfSettings[I].Addr;
        if (RegAdd==CC112X_AGC_CFG1 || RegAdd==CC112X_RFEND_CFG0 || RegAdd==CC112X_AGC_CFG0) continue;  //감지기에서 Sleep Mode로 들어갈 때 수정됨
        SetData=PacketModeCC112xRfSettings[I].Data;
        RegData=CC112xReadReg(RegAdd);
        if (SetData!=RegData)
            {
            Printf("SENSOR: Reg=%02X, Set=%02X, Reg=%02X"CRLF, RegAdd, SetData, RegData);
            break;
            }
        }

    if (I<countof(PacketModeCC112xRfSettings))
        {
        AUTOMANSPI_Reinit();
        Rslt++;
        }

    ProcExit:
    return Rslt;
    }




