/*
 * Copyright (c) 2016-2020, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 *  ======== timerled.c ========
 */

#include <stddef.h>

/* Driver Header files */
#include <ti/drivers/GPIO.h>
#include <ti/drivers/Timer.h>
#include <ti/drivers/timer/GPTimerCC26XX.h>
#include <ti/drivers/timer/TimerCC26XX.h>
#include <ti/drivers/dpl/HwiP.h>

/* Board Header file */
#include "ti_drivers_config.h"
#include DeviceFamily_constructPath(driverlib/timer.h)


Timer_Handle timer0;
Timer_Params params;

GPTimerCC26XX_Handle gptHandle;
/* GPTimer configuration array from application */
extern const GPTimerCC26XX_Config GPTimerCC26XX_config[];

/* Callback used for toggling the LED. */
void timerCallback(Timer_Handle myHandle, int_fast16_t status);
//zero latency callback
void timerCallback_ZL(uintptr_t a0);
/*
 *  ======== mainThread ========
 */
void *mainThread(void *arg0)
{

    /* Call driver init functions */
    GPIO_init();
    Timer_init();

    /* Configure the LED pin */
    GPIO_setConfig(CONFIG_GPIO_LED_0, GPIO_CFG_OUT_STD | GPIO_CFG_OUT_LOW);

    /* Turn off user LED */
    GPIO_write(CONFIG_GPIO_LED_0, CONFIG_GPIO_LED_OFF);

    /*
     * Setting up the timer in continuous callback mode that calls the callback
     * function every 1,000,000 microseconds, or 1 second.
     */
    Timer_Params_init(&params);
    params.period = 1000000;
    params.periodUnits = Timer_PERIOD_US;
    params.timerMode = Timer_CONTINUOUS_CALLBACK;
    params.timerCallback = timerCallback;

    timer0 = Timer_open(CONFIG_TIMER_0, &params);

    if (timer0 == NULL) {
        /* Failed to initialized timer */
        while (1) {}
    }

    if (Timer_start(timer0) == Timer_STATUS_ERROR) {
        /* Failed to start timer */
        while (1) {}
    }

    //Adding custom hardware interrupt
    TimerCC26XX_HWAttrs *timerHwAttrs = (TimerCC26XX_HWAttrs * ) timer0->hwAttrs;
    gptHandle =  (GPTimerCC26XX_Handle) & GPTimerCC26XX_config[timerHwAttrs->gpTimerUnit];

    GPTimerCC26XX_HWAttrs const *gptHwAttrs = gptHandle->hwAttrs;
    GPTimerCC26XX_Object  *gptObject  = gptHandle->object;

    /* Construct RTOS HWI */
    HwiP_Struct *pHwi = &gptObject->hwi[gptHandle->timerPart];
    HwiP_Params  hp;
    HwiP_Params_init(&hp);
    hp.arg       = (uintptr_t)gptHandle;
    hp.enableInt = true;
    hp.priority  = 0;

    GPTimerCC26XX_unregisterInterrupt(gptHandle);
    HwiP_construct(pHwi, gptHwAttrs->intNum, timerCallback_ZL, &hp);

    //HwiP_plug(gptHwAttrs->intNum, timerCallback_ZL);
    HwiP_enableInterrupt(gptHwAttrs->intNum);
    uint32_t ui32Base = gptHwAttrs->baseAddr;
    /* Enable interrupts in timer unit */
    TimerIntEnable(ui32Base, TIMER_TIMA_TIMEOUT);

    return (NULL);
}

/*
 * This callback is called every 1,000,000 microseconds, or 1 second. Because
 * the LED is toggled each time this function is called, the LED will blink at
 * a rate of once every 2 seconds.
 */
void timerCallback(Timer_Handle myHandle, int_fast16_t status)
{
    GPIO_toggle(CONFIG_GPIO_LED_0);
}

/* Lookup table definition for interfacing driverlib and register fields.
   Used to simplify code and to easily look up register fields as several fields
   are not symmetric across timer A and timer B registers (interrupts & dma)
 */
typedef struct GPTimerCC26XX_LUT
{
    uint16_t map;                      /* Timer argument in driverlib (TIMER_A / TIMER_B) */
    uint16_t shift;                    /* Bit shift for registers shared between GPT_A / GPT_B */
    uint16_t offset;                   /* Byte offset for registers sequentially in memory map for GPT_A/GPT_B */
    uint16_t interrupts[GPT_NUM_INTS]; /* Interrupt bitfields for GPTA/B. Order must match GPTimerCC26XX_Interrupt */
} const GPTimerCC26XX_LUT;

/* Lookup table definition for interfacing driverlib and register fields. */
static const GPTimerCC26XX_LUT GPT_LUT[GPT_PARTS_COUNT] =
{
    {
        .map        = TIMER_A,
        .shift      = 0,
        .offset     = 0,
        .interrupts ={ GPT_MIS_TATOMIS,                   GPT_MIS_CAMMIS, GPT_MIS_CAEMIS, GPT_MIS_TAMMIS },
    },
    {
        .map        = TIMER_B,
        .shift      = 8,
        .offset     = 4,
        .interrupts ={ GPT_MIS_TBTOMIS,                   GPT_MIS_CBMMIS, GPT_MIS_CBEMIS, GPT_MIS_TBMMIS },
    },
};

void timerCallback_ZL(uintptr_t a0)
{
    GPTimerCC26XX_Handle         handle  = (GPTimerCC26XX_Handle)gptHandle;
    GPTimerCC26XX_HWAttrs const *hwAttrs = gptHandle->hwAttrs;

    uint32_t timer = GPT_LUT[handle->timerPart].map;

    /* Full width raw interrupt status */
    uint32_t interrupts = HWREG(hwAttrs->baseAddr + GPT_O_MIS);
    /* Interrupt mask to clear (byte 0 or 1) */
    uint32_t interruptClr = timer & interrupts;
    /* Clear interrupts */
    HWREG(hwAttrs->baseAddr + GPT_O_ICLR) = interruptClr;

    GPIO_toggle(CONFIG_GPIO_LED_0);
}
