#include "Gpio.h"
#include "Sequencer.h"

//#define MASK_1_8V
// #define DISABLE_20V

const Gpio_getHandlerFunction Gpio_getEnableTable[GPIO_NUM_QUADRANTS] = {
	Gpio_getEnableQ1,
	Gpio_getEnableQ2
};

const Gpio_getHandlerFunction Gpio_getFeedbackTable[GPIO_NUM_QUADRANTS] = {
	Gpio_getFeedbackQ1,
	Gpio_getFeedbackQ2
};

const Gpio_setHandlerFunction Gpio_setEnableTable[GPIO_NUM_QUADRANTS] = {
	Gpio_setEnableQ1,
	Gpio_setEnableQ2
};

const Gpio_setHandlerFunction Gpio_setFeedbackTable[GPIO_NUM_QUADRANTS] = {
	Gpio_setFeedbackQ1,
	Gpio_setFeedbackQ2
};

const Gpio_setPortHandler Gpio_setPortTable[2] = {
	Gpio_setPortOn,
	Gpio_setPortOff
};


void Gpio_init()
{
	P1DIR = 0xF0;
	P1OUT = 0x00;
	P1REN = 0x00;
	P2SEL = 0x00;
	P2DIR = 0x00;
	P2OUT = 0x00;
	P2REN = 0x00;
	P3DIR = 0x40;
	P3OUT = 0x00;
	P3REN = 0x00;
	P4DIR = 0x7F;
	P4OUT = 0x00;
	P4REN = 0x00;
}

/*
 * @brief Gets the active high power good regulator feedback from the Gpio
 * and puts it in the pointer.
 *
 * @detail The order of the bits are as follows:
 * 		0 : 65V
 * 		1 : -32V
 * 		2 : -9V
 * 		3 : 13V
 * 		4 : 1.8V
 * 		5 : 20V
 */
void Gpio_getFeedbackQ1( uint8_t* const p ) {
	uint8_t temp;
	extern struct Sequencer_t sequencer[2];
	temp =	( P3IN & 0x80 ) >> 7;	// P3.7 >> 7 = 0, 65V
	temp |=	( P2IN & 0x08 ) >> 2;	// P2.3 >> 2 = 1, -32V
	temp |= ( P2IN & 0x10 ) >> 2;	// P2.4 >> 2 = 2, -9V
	temp |= ( P2IN & 0x01 ) << 3;	// P2.0 << 3 = 3, 13V
#ifdef MASK_1_8V
	temp |= ( P4IN & 0x40 ) >> 2; 	// P4.6 >> 2 = 4, 1.8V (use enable)
#else
	temp |= ( P2IN & 0x02 ) << 3;	// P2.1 << 3 = 4, 1.8V
#endif
#ifdef DISABLE_20V
	temp |= ( sequencer[0].output & 0x10 ) << 1;
#else
	temp |= ( P2IN & 0x04 ) << 3;	// P2.2 << 3 = 5, 20V
#endif
	*p = temp;
}

/*
 * @brief Gets the active high power good regulator feedback from the Gpio
 * and puts it in the pointer.
 *
 * @detail The order of the bits are as follows:
 * 		0 : 65V
 * 		1 : -32V
 * 		2 : -9V
 * 		3 : 13V
 * 		4 : 1.8V
 * 		5 : 20V
 */
void Gpio_getFeedbackQ2( uint8_t* const p )
{
	uint8_t temp;
	extern struct Sequencer_t sequencer[2];
	temp =	( P4IN & 0x80 ) >> 7;	// P4.7 >> 7 = 0, 65V
	temp |=	( P1IN & 0x01 ) << 1;	// P1.0 << 1 = 1, -32V
	temp |= ( P1IN & 0x02 ) << 1;	// P1.1 << 1 = 2, -9V
	temp |= ( P1IN & 0x04 ) << 1;	// P1.2 << 1 = 3, 13V
#ifdef MASK_1_8V
	temp |= ( P1IN & 0x20 ) >> 1; 	// P1.5 >> 1 = 4, 1.8V (use enable)
#else
	temp |= ( P1IN & 0x08 ) << 1;	// P1.3 << 1 = 4, 1.8V
#endif
#ifdef DISABLE_20V
	temp |= ( sequencer[1].output & 0x10 ) << 1;
#else
	temp |= ( P2IN & 0x20 );		// P2.5      = 5, 20V
#endif
	*p = temp;
}

/*
 * @brief Returns active LOW enable for respective quadrant.
 */
inline void Gpio_getEnableQ1( uint8_t* const p )
{
	*p = ( ( P2IN & 0x40 ) == 0 );	// Input is active LOW, convert to active high
}

/*
 * @brief Returns active LOW enable for respective quadrant.
 */
inline void Gpio_getEnableQ2( uint8_t* const p )
{
	*p = ( ( P2IN & 0x80 ) == 0 );	// Input is active LOW, convert to active high
}

/*
 * @brief Sets the active high enable to the regulators.
 *
 * @detail The order of the bits are as follows:
 * 		0 : -32V
 * 		1 : -9V
 * 		2 : 13V
 * 		3 : 1.8V
 * 		4 : 20V
 */
void Gpio_setEnableQ1( const uint8_t* const p )
{
	Gpio_setPortTable[ ( *p & 0x01 ) == 0 ]( &P4OUT, BIT3 );	// P4.3, -32V
	Gpio_setPortTable[ ( *p & 0x02 ) == 0 ]( &P4OUT, BIT4 );	// P4.4, -9V
	Gpio_setPortTable[ ( *p & 0x04 ) == 0 ]( &P4OUT, BIT5 );	// P4.5, 13V
	Gpio_setPortTable[ ( *p & 0x08 ) == 0 ]( &P4OUT, BIT6 );	// P4.6, 1.8V
#ifndef DISABLE_20V
	Gpio_setPortTable[ ( *p & 0x10 ) == 0 ]( &P3OUT, BIT6 );	// P3.6, 20V
#endif
}

/*
 * @brief Sets the active high enable to the regulators.
 *
 * @detail The order of the bits are as follows:
 * 		0 : -32V
 * 		1 : -9V
 * 		2 : 13V
 * 		3 : 1.8V
 * 		4 : 20V
 */
void Gpio_setEnableQ2( const uint8_t* const p )
{
	Gpio_setPortTable[ ( *p & 0x01 ) == 0 ]( &P4OUT, BIT0 );	// P4.0, -32V
	Gpio_setPortTable[ ( *p & 0x02 ) == 0 ]( &P4OUT, BIT1 );	// P4.1, -9V
	Gpio_setPortTable[ ( *p & 0x04 ) == 0 ]( &P4OUT, BIT2 );	// P4.2, 13V
	Gpio_setPortTable[ ( *p & 0x08 ) == 0 ]( &P1OUT, BIT5 );	// P1.5, 1.8V
#ifndef DISABLE_20V
	Gpio_setPortTable[ ( *p & 0x10 ) == 0 ]( &P1OUT, BIT4 );	// P3.7, 20V
#endif
}

/*
 * @brief Sets the active LOW feedback output to the FPGA.
 */
void Gpio_setFeedbackQ1( const uint8_t* const p )
{
	// Convert to active LOW
	Gpio_setPortTable[ ( *p & 0x01 ) != 0 ]( &P1OUT, BIT7 );	// P1.7, Q1_GOOD
}

/*
 * @brief Sets the active LOW feedback output to the FPGA.
 */
void Gpio_setFeedbackQ2( const uint8_t* const p )
{
	// Convert to active LOW
	Gpio_setPortTable[ ( *p & 0x01 ) != 0 ]( &P1OUT, BIT6 );	// P1.6, Q2_GOOD
}

void Gpio_setPortOn( volatile unsigned char* const port, const unsigned char bit )
{
	*port |= bit;
}

void Gpio_setPortOff( volatile unsigned char* const port, const unsigned char bit )
{
	*port &= ~bit;
}
