#include "I2c.h"

const uint8_t I2C_SLAVE_ADDRESS[I2C_NUM_SLAVES] = {
	0x40,		// MSP_SLAVE
	0x48,		// ADT PS1 Q1 (temp sensor)
	0x49,		// ADT PS1 Q2
	0x4B,		// ADT PS2 Q1
	0x4A		// ADT PS2 Q2
};

const uint8_t I2C_TRANSACTION_BYTES[I2C_NUM_SLAVES] = {
		18,
		2,
		2,
		2,
		2
};

const uint8_t I2C_BUFFER_OFFSET[I2C_NUM_SLAVES] = {
		0,
		18,
		20,
		22,
		24
};

void I2c_init( struct I2c_t* const p, uint8_t* pBuf )
{
	p->ix = 0;
	p->cb = 0;
	p->pBuf = pBuf;

	//	The recommended USCI initialization or reconfiguration process is:
	//	1. Set UCSWRST (BIS.B #UCSWRST,&UCxCTL1)
	//	2. Initialize all USCI registers with UCSWRST=1 (including UCxCTL1)
	//	3. Configure ports.
	//	4. Clear UCSWRST via software (BIC.B #UCSWRST,&UCxCTL1)
	//	5. Enable interrupts (optional) via UCxRXIE and/or UCxTXIE

	//	1. Set UCSWRST (BIS.B #UCSWRST,&UCxCTL1)
	UCB0CTL1 |= UCSWRST;					// Enable SW reset

	//	2. Initialize all USCI registers with UCSWRST=1 (including UCxCTL1)
    UCB0CTL0 |= UCMST + UCMODE_3 + UCSYNC;  // I2C Master, synchronous mode
    UCB0CTL1 |= UCSSEL_2;          			// Use SMCLK, keep SW reset
    UCB0BR1 = 1;
    UCB0BR0 = 0;

    //	3. Configure ports.
    P3SEL |= 0x06;

	//	4. Clear UCSWRST via software (BIC.B #UCSWRST,&UCxCTL1)
    UCB0CTL1 &= ~UCSWRST;

	//	5. Enable interrupts (optional) via UCxRXIE and/or UCxTXIE
    UCB0I2CIE |= UCNACKIE;
    IE2 |= UCB0RXIE;
}

// ISR for receive
// UCBxTXIFG and UCBxRXIFG from USCI_Bx and UCAxTXIFG from USCI_Ax share a vector
#pragma vector = USCIAB0TX_VECTOR
__interrupt void I2c_irqRx()
{
	extern struct I2c_t i2c;
	extern uint32_t cI2cRxIfg;
	extern uint32_t cSpiTxIfg;

	if ( IFG2 & UCB0RXIFG ) {
		cI2cRxIfg++;
		*(i2c.pBuf + I2C_BUFFER_OFFSET[ i2c.ix ] + i2c.cb ) = UCB0RXBUF;
		i2c.cb++;

		// If this is the next to last byte, generate a stop condition
		if ( i2c.cb == ( I2C_TRANSACTION_BYTES[ i2c.ix ] - 1 ) ) {
			UCB0CTL1 |= UCTXSTP;
		}

		// If this is the last byte, increment the index and see if it needs to be reset
		if ( i2c.cb == I2C_TRANSACTION_BYTES[ i2c.ix ] ) {
			i2c.ix++;
			if ( i2c.ix == I2C_NUM_SLAVES ) {
				i2c.ix = 0;
			}
		}
	}

	if ( IFG2 & UCA0TXIFG ) {
		cSpiTxIfg++;
		UCA0TXBUF = UCA0TXBUF;	// tried 0xA5, UCA0RXBUF as well
	}
}

//// ISR for NACK
//// The I2C transmit and receive interrupt flags UCBxTXIFG and UCBxRXIFG
//// from USCI_Bx and UCAxTXIFG from USCI_Ax share another interrupt vector
//#pragma vector = USCIAB0RX_VECTOR
//__interrupt void I2c_irqNack()
//{
//}
