#include "Sequencer.h"

/*
 * @brief Defines masks for which feedback inputs should be evaluated as
 * precursors.
 *
 * @detail At every state regulator state, there are regulators which 
 * should certainly be on, and certainly be off. This defines the mask
 * for which regulator feedbacks should be evaluated. As an example, in
 * 1.8V state, whether turning on or turning off, all regulators 
 * preceeding 1.8V in the sequence should be ON, and all regulators
 * following 1.8V in the sequence should be OFF. The 1.8V regulator
 */
const uint8_t SEQ_PRECURSOR_EVAL_MASK[SEQ_NUM_STATES] = {
    0xFF,   // OFF,             Check everything
    0xFD,   // -32V,            Check everything except -32V
    0xFB,   // -9V,             Check everything except -9V
    0xF7,   // 13V,             Check everything except 13V
    0xEF,   // 1.8V,            Check everything except 1.8V
    0xDF,   // 20V,             Check everything except 20V
    0xFF    // ON,              Check everything
};

/*
 * @brief
 */
const uint8_t SEQ_PRECURSOR_EXPECTED[SEQ_NUM_STATES] = {
    0x01,   // OFF,             65V should be on at all times
    0x01,   // -32V,            65V must be on before -32V
    0x03,   // -9V,             -32V and preceeding before -9V
    0x07,   // 13V,             -9V and preceeding before 13V
    0x0F,   // 1.8V,            13V and preceeding before 1.8V
    0x1F,   // 20V,             1.8V and preceeding before 20V
    0x3F    // ON,              All on while on
};

const uint8_t SEQ_FEEDBACK_MASK[SEQ_NUM_STATES][2] = {
    // !vEnable     vEnable
    {   0x01,       0x01 },     // OFF,             65V should be on even when everything's off
    {   0x01,       0x03 },     // -32V,            -32V and preceeding
    {   0x03,       0x07 },     // -9V,             -9V and preceeding
    {   0x07,       0x0F },     // 13V,             13V and preceeding
    {   0x0F,       0x1F },     // 1.8V,            1.8V and preceeding
    {   0x1F,       0x3F },     // 20V,             All on
    {   0x3F,       0x3F }      // ON,              All on
};

const enum SequencerState_t SEQ_TRANSITION_TABLE[SEQ_NUM_STATES][3] = {
    //  Sequencer_SEQUENCER_
    //  PREV            SAME            NEXT
    {   SEQ_OFF,       SEQ_OFF,       SEQ_NEG_32V    },  // SEQ_OFF
    {   SEQ_OFF,       SEQ_NEG_32V,   SEQ_NEG_9V     },  // -32V
    {   SEQ_NEG_32V,   SEQ_NEG_9V,    SEQ_13V        },  // -9V
    {   SEQ_NEG_9V,    SEQ_13V,       SEQ_1_8V       },  // 13V
    {   SEQ_13V,       SEQ_1_8V,      SEQ_20V        },  // 1.8V
    {   SEQ_1_8V,      SEQ_20V,       SEQ_ON         },  // 20V
    {   SEQ_20V,       SEQ_ON,        SEQ_ON         }   // SEQ_ON
};

const uint8_t SEQ_OUTPUT_LUT[SEQ_NUM_STATES][2] = {
    //  !vEnable    vEnable
    {   0x00,       0x00    },      // OFF
    {   0x00,       0x01    },      // -32V
    {   0x01,       0x03    },      // -9V
    {   0x03,       0x07    },      // 13V
    {   0x07,       0x0F    },      // 1.8V
    {   0x0F,       0x1F    },      // 20V
    {   0x1F,       0x1F    }       // ON
};

const enum SequencerProcessState_t PROCESS_AFTER_PRECURSORS[] = {
    PROCESS_FEEDBACK,   // OFF, 
    PROCESS_TIME,       // -32V,
    PROCESS_TIME,       // -9V, 
    PROCESS_TIME,       // 13V, 
    PROCESS_TIME,       // 1.8V,
    PROCESS_TIME,       // 20V, 
    PROCESS_FEEDBACK    // ON,  
};

const Sequencer_processHandlerFunction Sequencer_processHandlerTable[SEQ_NUM_PROCESS_STATES-1] = {
    Sequencer_processPrecursors,
    Sequencer_processTime,
    Sequencer_processFeedback
};

//> @brief Minimum time in ms to turn a regulator off/on
const uint8_t SEQ_MIN_TIME[2] = {
    25,      // !vEnable (turning off)
    14      // vEnable (turning on)
};


//> @brief Maximum time in ms to turn a regulator off/on
const uint8_t SEQ_MAX_TIME[2] = {
    30,      // !vEnable (turning off)
    15      // vEnable (turning on)
};

// vEnable = bit 2
// fDone = bit 1
// fFault = bit 0
const uint8_t SEQ_LUT[] = {
    SEQ_SAME,   // 000
    SEQ_PREV,   // 001
    SEQ_PREV,   // 010
    SEQ_PREV,   // 011
    SEQ_SAME,   // 100
    SEQ_PREV,   // 101
    SEQ_NEXT,   // 110
    SEQ_PREV    // 111
};

/*
 * @brief Initializes the sequencer structure (essentially a constructor)
 */
void Sequencer_init( struct Sequencer_t* const p )
{
    p->vEnable      = 0;
    p->feedback     = 1;
    p->fDone        = 0;
    p->fFault       = 0;
    p->fault        = 0;
    p->output       = 0;
    p->processState = PROCESS_PRECURSOR;
    p->state        = SEQ_OFF;
    SysTime_init( &p->tEvent );
    SysTime_init( &p->time );
}

/*
 *
 */
void Sequencer_process( struct Sequencer_t* const p, const struct SysTime_t* const pSysTime )
{
    Sequencer_resetProcessing( p );
	p->time = *pSysTime;
    while( p->processState != PROCESS_FINISHED ) {
        p->processState = Sequencer_processHandlerTable[ p->processState ]( p );
    }
    // Next state
    p->state = SEQ_TRANSITION_TABLE[ p->state ][ SEQ_LUT[ Sequencer_getSequencerInput( p ) ] ];
    // Set the output
    Sequencer_setOutput( p );
}

/*
 * @brief Resets the process states to "idle/safe" values.
 */
void Sequencer_resetProcessing( struct Sequencer_t* const p )
{
    // feedback and time are supplied by higher modules
    // tEvent and state must persist through calls, so don't reset those
    p->output   = 0;
    p->fDone    = 0;
    // Initialize fFault to 0 since later modules will set to 1 IFF there's a fault
    p->fFault   = 0;
    if ( p->vEnableOld != p->vEnable ) {
    	p->tEvent = p->time;
    }
    p->vEnableOld = p->vEnable;
    p->processState = PROCESS_PRECURSOR;
}

/*
 * @brief Checks precursor regulator feedback.
 *
 * @param p Pointer to the Sequencer_t structure (essentially a this pointer)
 */
enum SequencerProcessState_t Sequencer_processPrecursors( struct Sequencer_t* const p ) {
    p->fault = ( SEQ_PRECURSOR_EVAL_MASK[p->state] & p->feedback ) ^ SEQ_PRECURSOR_EXPECTED[p->state];
    if ( p->fault != 0 ) {
        p->fFault = 1;
        p->fDone = 1;
        p->tEvent = p->time;
        p->vEnable = 0;
        return PROCESS_FINISHED;
    } 
    return PROCESS_AFTER_PRECURSORS[p->state];
}

/*
 * @brief Checks to make sure the time is between the minimum and maximum.
 *
 * @param p Pointer to the Sequencer_t structure (essentially a this pointer)
 */
enum SequencerProcessState_t Sequencer_processTime( struct Sequencer_t* const p ) {
    struct SysTime_t delta = SysTime_diff( &p->time, &p->tEvent );
    if ( delta.ms >= SEQ_MIN_TIME[p->vEnable] && delta.ms < SEQ_MAX_TIME[p->vEnable] ) {
        p->tEvent = p->time;
        return PROCESS_FEEDBACK;
    } 
    if ( delta.ms >= SEQ_MAX_TIME[p->vEnable] ) {
        p->fFault = 1;
        p->vEnable = 0;
        p->fDone = 1;
        return PROCESS_FINISHED;
    }
    return PROCESS_FINISHED;
}

/*
 * @brief Processes feedback to determine if the current sequence is complete.
 *
 * @param p Pointer to the Sequencer_t structure (essentially a this pointer)
 */
enum SequencerProcessState_t Sequencer_processFeedback( struct Sequencer_t* const p ) {
    if ( p->feedback == SEQ_FEEDBACK_MASK[p->state][p->vEnable] ) {
        p->fDone = 1;
        p->tEvent = p->time;
    }
    return PROCESS_FINISHED;
}

void Sequencer_setOutput( struct Sequencer_t* const p )
{
    p->output = SEQ_OUTPUT_LUT[ p->state ][ p->vEnable ];
}

uint8_t Sequencer_getSequencerInput( struct Sequencer_t* const p )
{
    uint8_t temp = 0;
    temp += p->vEnable  << 2;
    temp += p->fDone    << 1;
    temp += p->fFault;
    return temp;
}
