#include "SysTime.h"

#ifndef __SEQUENCER_H
#define __SEQUENCER_H

#define SEQ_PREV                0
#define SEQ_SAME                1
#define SEQ_NEXT                2

enum SequencerState_t {
    SEQ_OFF,
    SEQ_NEG_32V,
    SEQ_NEG_9V,
    SEQ_13V,
    SEQ_1_8V,
    SEQ_20V,
    SEQ_ON,
    SEQ_NUM_STATES
};

enum SequencerProcessState_t {
    PROCESS_PRECURSOR,
    PROCESS_TIME,
    PROCESS_FEEDBACK,
    PROCESS_FINISHED,
    SEQ_NUM_PROCESS_STATES
};

/*
 *
 * @param feedback holds a bitmask of all regulator feedback
            0 :  65V
            1 : -32V
            2 : -9V
            3 :  13V
            4 :  1.8V
            5 :  20V
 *
 */
struct Sequencer_t {
    uint8_t                         vEnable;            //> Virtual enable signal
    uint8_t							vEnableOld;			//> Old virtual enable
    uint8_t                         feedback;           //> Feedback from power regulators
    uint8_t                         fDone;              //> Current sequence complete
    uint8_t                         fFault;             //> Fault flag
    uint8_t                         fault;              //> Actual faults
    uint8_t                         output;             //> Enable outputs to regulators (active high)
    struct SysTime_t                time;               //> Time
    struct SysTime_t                tEvent;             //> Time of the last event
    enum SequencerProcessState_t    processState;       //> Processing state
    enum SequencerState_t           state;              //> Sequencer state
};

extern const uint8_t SEQ_PRECURSOR_EVAL_MASK[];
extern const uint8_t SEQ_PRECURSOR_EXPECTED[];
extern const uint8_t SEQ_FEEDBACK_MASK[SEQ_NUM_STATES][2];
extern const uint8_t SEQ_OUTPUT_LUT[SEQ_NUM_STATES][2];
extern const enum SequencerState_t SEQ_TRANSITION_TABLE[SEQ_NUM_STATES][3];
extern const uint8_t SEQ_LUT[];
extern const uint8_t SEQ_MIN_TIME[];
extern const uint8_t SEQ_MIN_TIME[];
extern const enum SequencerProcessState_t PROCESS_AFTER_PRECURSORS[];

// Initialization/constructor
void                            Sequencer_init                          ( struct Sequencer_t* const );

void                            Sequencer_process                       ( struct Sequencer_t* const, const struct SysTime_t* const );
void                            Sequencer_resetProcessing               ( struct Sequencer_t* const );

// Process handlers
enum SequencerProcessState_t    Sequencer_processPrecursors             ( struct Sequencer_t* const );
enum SequencerProcessState_t    Sequencer_processTime                   ( struct Sequencer_t* const );
enum SequencerProcessState_t    Sequencer_processFeedback               ( struct Sequencer_t* const );

// Declare a process handler function type, and a LUT for the handlers (above)
typedef enum SequencerProcessState_t (*Sequencer_processHandlerFunction)( struct Sequencer_t* const );
extern const Sequencer_processHandlerFunction Sequencer_processHandlerTable[];

void                            Sequencer_processSequencerState         ( struct Sequencer_t* const );

uint8_t                         Sequencer_getSequencerInput             ( struct Sequencer_t* const );

void                            Sequencer_setOutput                     ( struct Sequencer_t* const );

#endif // __SEQUENCER_H
