#include "Spi.h"
#include "Power.h"
#include "Build.h"

void Spi_init( struct Spi_t* const p, uint8_t* const pBuf )
{
	p->ix = 0;
	p->pBuf = pBuf;
	*( p->pBuf +  0 ) 	= BUILD_MAJOR;
	*( p->pBuf +  1 )	= BUILD_MINOR;
	*( p->pBuf +  2 )	= (uint8_t)BUILD_NUMBER;
	*( p->pBuf +  3 )	= BUILD_NUMBER >> 8;

	Spi_hwInit();

	// Place default byte in Tx register
    UCA0TXBUF = SPI_DEFAULT_BYTE;
}

void Spi_hwInit()
{
	/*
     * Steps to initialize SPI are
     * 1. Set UCWRST
     * 2. Initialize all USCI registers with UCSWRST=1 (includeing UCxCTL1)
     * 3. Configure ports
     * 4. Clear UCSWRST via software
     * 5. Enable interrupts (optional) via UCxRXIE and/or UCxTXIE
     */

	// 1. Set UCWRST
	UCA0CTL1 |= UCSWRST;                      // **Put state machine in reset**

	// 2. Initialize all USCI registers with UCSWRST=1 (including UCxCTL1)
    UCA0CTL0 = UCCKPL + UCMSB + UCMODE_2 + UCSYNC;

    // 3. Configure ports
    // Set select bits to secondary functions
    P3SEL 	|= 0x39;

  	// 4. Clear UCSWRST via software	  		// Clock polarity high, MSB first
    UCA0CTL1 &= ~UCSWRST;                     	// **Initialize USCI state machine**

    // 5. Enable interrupts (optional) via UCxRXIE and/or UCxTXIE
    IE2 |= UCA0RXIE + UCA0TXIE;                	// Enable USCI_A0 RX/TX interrupt
}


/*
 * @brief SPI receive and I2C NACK interrupt handler
 */
#pragma vector=USCIAB0RX_VECTOR
__interrupt void Spi_irq()
{
	extern struct Spi_t spi;
	extern struct I2c_t i2c;
	extern uint32_t cSpiRxIfg;
	extern uint32_t cI2cNackIfg;

	if ( IFG2 & UCA0RXIFG ) {
		cSpiRxIfg++;
//		spi.ix = UCA0RXBUF;
		UCA0RXBUF = UCA0RXBUF;

//		// If this is the first byte, copy all data from the I2C buffer
//		if ( spi.ix == 0 ) {
//			Spi_getMessage( &spi, &i2c );
//			spi.ix++;
//
//		// If this is one of the message bytes, copy that to the I2C buffer
//		} else if ( spi.ix < SPI_NUM_MSG_BYTES-1 ) {
//			UCA0TXBUF = *(spi.pBuf + spi.ix);
//			spi.ix++;
//
//		// otherwise, send the default byte
//		} else {
//			spi.ix = 0;
//			UCA0TXBUF = SPI_DEFAULT_BYTE;
//		}
	}

	if ( UCB0STAT & UCNACKIFG ) {
		cI2cNackIfg++;
		// We can only get here by receiving a NACK, in which case we'll just move on to the next transaction
		// (assuming we're not on the last slave)
		UCB0CTL1 |= UCTXSTP;
		i2c.ix++;
		i2c.cb = 0;
		if ( i2c.ix == I2C_NUM_SLAVES ) {
			i2c.ix = 0;
		}
		UCB0STAT &= ~UCNACKIFG;
	}

}

/*
 * @brief Get the SPI message
 *
 *
 *
 */
void Spi_getMessage( struct Spi_t* const p, struct I2c_t* const pI2c )
{
	uint8_t i, checksum;
	extern struct Power_t power[2];
	*( p->pBuf +  4 )	= power[0].enable;
	*( p->pBuf +  5 )	= power[0].pSequencer->output;
	*( p->pBuf +  6 )	= power[0].pSequencer->feedback;
	*( p->pBuf +  7 )	= power[0].fault;
	*( p->pBuf +  8 )	= power[1].enable;
	*( p->pBuf +  9 )	= power[1].pSequencer->output;
	*( p->pBuf + 10 )	= power[1].pSequencer->feedback;
	*( p->pBuf + 11 )	= power[1].fault;
	memcpy( p->pBuf + SPI_PS1_OFFSET_BYTES, pI2c->pBuf, I2C_SLAVE_BYTES );
	checksum = 0;
	for ( i = 0; i < SPI_NUM_MSG_BYTES-1; i++ ) {
		checksum ^= *(p->pBuf + i);
	}
	*( p->pBuf + SPI_NUM_MSG_BYTES - 1 ) = checksum;
}
