/*
 * -------------------------------------------
 *    MSP432 DriverLib - v3_21_00_05 
 * -------------------------------------------
 *
 * --COPYRIGHT--,BSD,BSD
 * Copyright (c) 2016, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
/******************************************************************************
 * MSP432 SPI - 3-wire Master Incremented Data
 *
 * This example shows how SPI master talks to SPI slave using 3-wire mode.
 * Incrementing data is sent by the master starting at 0x01. Received data is
 * expected to be same as the previous transmission.  eUSCI RX ISR is used to
 * handle communication with the CPU, normally in LPM0. Because all execution 
 * after LPM0 is in ISRs, initialization waits for DCO to stabilize against 
 * ACLK.
 *
 * Note that in this example, EUSCIB0 is used for the SPI port. If the user
 * wants to use EUSCIA for SPI operation, they are able to with the same APIs
 * with the EUSCI_AX parameters.
 *
 * ACLK = ~32.768kHz, MCLK = SMCLK = DCO 3MHz
 *
 * Use with SPI Slave Data Echo code example.
 *
 *                MSP432P401
 *              -----------------
 *             |                 |
 *             |                 |
 *             |                 |
 *             |             P1.6|-> Data Out (UCB0SIMO)
 *             |                 |
 *             |             P1.7|<- Data In (UCB0SOMI)
 *             |                 |
 *             |             P1.5|-> Serial Clock Out (UCB0CLK)
 * Author: Timothy Logan
*******************************************************************************/
/* DriverLib Includes */
#include "driverlib.h"

/* Standard Includes */
#include <stdint.h>

#include <stdbool.h>

#define  ADS1292_DRDY_PORT  GPIO_PORT_P2
#define  ADS1292_DRDY_PIN   GPIO_PIN3 //J4.34
#define  ADS1292_CS_PORT    GPIO_PORT_P5
#define  ADS1292_CS_PIN     GPIO_PIN1 //J4.33
#define  ADS1292_START_PORT GPIO_PORT_P3
#define  ADS1292_START_PIN  GPIO_PIN5 //J4.32
#define  ADS1292_PWDN_PORT  GPIO_PORT_P3
#define  ADS1292_PWDN_PIN   GPIO_PIN7 //J4.31

//Packet format
#define  CES_CMDIF_PKT_START_1    0x0A
#define  CES_CMDIF_PKT_START_2     0xFA
#define  CES_CMDIF_TYPE_DATA       0x02
#define  CES_CMDIF_PKT_STOP        0x0B

// Register Read Commands
#define RREG    0x20;      //Read n nnnn registers starting at address r rrrr
                                //first byte 001r rrrr (2xh)(2) - second byte 000n nnnn(2)
#define WREG    0x40;       //Write n nnnn registers starting at address r rrrr
                                //first byte 010r rrrr (2xh)(2) - second byte 000n nnnn(2)


#define START       0x08        //Start/restart (synchronize) conversions
#define STOP        0x0A        //Stop conversion
#define RDATAC      0x10        //Enable Read Data Continuous mode.
//This mode is the default mode at power-up.
#define SDATAC      0x11        //Stop Read Data Continuously mode
#define RDATA       0x12        //Read data by command; supports multiple read back.

#define LOW         0x00
#define HIGH        0x01

//register address
#define ADS1292_REG_ID          0x00
#define ADS1292_REG_CONFIG1     0x01
#define ADS1292_REG_CONFIG2     0x02
#define ADS1292_REG_LOFF        0x03
#define ADS1292_REG_CH1SET      0x04
#define ADS1292_REG_CH2SET      0x05
#define ADS1292_REG_RLDSENS     0x06
#define ADS1292_REG_LOFFSENS    0x07
#define ADS1292_REG_LOFFSTAT    0x08
#define ADS1292_REG_RESP1       0x09
#define ADS1292_REG_RESP2       0x0A

/* Statics */
static volatile uint8_t RXData = 0;
static uint8_t TXData = 0;
static uint8_t Read_1292 = 0;
static volatile uint8_t SPI_RX_Buff_Count = 0;
static uint8_t SPI_RX_Buff[8]={};
unsigned long uecgtemp = 0;
signed long secgtemp=0;
volatile signed long s32DaqVals[8];
uint8_t DataPacketHeader[16];
uint8_t data_len = 8;
volatile bool ads1292dataReceived =false;

/* SPI Master Configuration Parameter */
const eUSCI_SPI_MasterConfig spiMasterConfig =
{
        EUSCI_B_SPI_CLOCKSOURCE_SMCLK,             // SMCLK Clock Source
        3000000,                                   // SMCLK = DCO = 3MHZ
        1000000,                                    // SPICLK = 500khz
        EUSCI_B_SPI_MSB_FIRST,                     // MSB First
        EUSCI_B_SPI_PHASE_DATA_CAPTURED_ONFIRST_CHANGED_ON_NEXT,    // Phase 1
        EUSCI_B_SPI_CLOCKPOLARITY_INACTIVITY_LOW, //  polarity 0
        EUSCI_B_SPI_3PIN                           // 4Wire CS active low SPI Mode
};

/* UART Configuration Parameter. These are the configuration parameters to
 * make the eUSCI A UART module to operate with a 115200 baud rate. These
 * values were calculated using the online calculator that TI provides
 * at:
 * http://software-dl.ti.com/msp430/msp430_public_sw/mcu/msp430/MSP430BaudRateConverter/index.html
 */
const eUSCI_UART_Config uartConfig =
{
        EUSCI_A_UART_CLOCKSOURCE_SMCLK,          // SMCLK Clock Source
        13,                                      // BRDIV = 13
        0,                                       // UCxBRF = 0
        37,                                      // UCxBRS = 37
        EUSCI_A_UART_NO_PARITY,                  // No Parity
        EUSCI_A_UART_MSB_FIRST,                  // MSB First
        EUSCI_A_UART_ONE_STOP_BIT,               // One stop bit
        EUSCI_A_UART_MODE,                       // UART mode
        EUSCI_A_UART_OVERSAMPLING_BAUDRATE_GENERATION  // Oversampling
};
void delay(int msec)
{
    int c;
    for(c=0;c<msec;c++);
}
void ads1292_SPI_Command_Data(uint8_t command)
{

   TXData  =  command;
   GPIO_setOutputLowOnPin(ADS1292_CS_PORT, ADS1292_CS_PIN);
   delay(2);
   GPIO_setOutputHighOnPin(ADS1292_CS_PORT, ADS1292_CS_PIN);
   delay(2);
   GPIO_setOutputLowOnPin(ADS1292_CS_PORT,ADS1292_CS_PIN);
   delay(2);
   /* Polling to see if the TX buffer is ready */
    while (!(SPI_getInterruptStatus(EUSCI_B0_BASE,EUSCI_B_SPI_TRANSMIT_INTERRUPT)));

   /* Transmitting data to slave */
   SPI_transmitData(EUSCI_B0_BASE, TXData);

   while (EUSCI_B_SPI_isBusy(EUSCI_B0_BASE));

   GPIO_setOutputHighOnPin(ADS1292_CS_PORT, ADS1292_CS_PIN);
   delay(2);

}
void ads1292_Reg_Write(uint8_t READ_WRITE_ADDRESS,uint8_t DATA)
{
uint8_t dataToSend;
switch (READ_WRITE_ADDRESS)
  {
    case 1:
            DATA = DATA & 0x87;
        break;
    case 2:
            DATA = DATA & 0xFB;
        DATA |= 0x80;
        break;
    case 3:
        DATA = DATA & 0xFD;
        DATA |= 0x10;
        break;
    case 7:
        DATA = DATA & 0x3F;
        break;
    case 8:
            DATA = DATA & 0x5F;
        break;
    case 9:
        DATA |= 0x02;
        break;
    case 10:
        DATA = DATA & 0x87;
        DATA |= 0x01;
        break;
    case 11:
        DATA = DATA & 0x0F;
        break;
    default:
        break;
  }

  // now combine the register address and the command into one byte:
  dataToSend = READ_WRITE_ADDRESS | WREG;

  GPIO_setOutputLowOnPin(ADS1292_CS_PORT, ADS1292_CS_PIN);
  delay(2);
  GPIO_setOutputHighOnPin(ADS1292_CS_PORT, ADS1292_CS_PIN);
  delay(2);
  GPIO_setOutputLowOnPin(ADS1292_CS_PORT,ADS1292_CS_PIN);
  delay(2);

  /*Send Register Location*/
  /* Polling to see if the TX buffer is ready */
  while (!(SPI_getInterruptStatus(EUSCI_B0_BASE,EUSCI_B_SPI_TRANSMIT_INTERRUPT)));
  /* Transmitting data to slave */
  SPI_transmitData(EUSCI_B0_BASE, dataToSend);

  /*Number of Register to Write*/
  /* Polling to see if the TX buffer is ready */
  while (!(SPI_getInterruptStatus(EUSCI_B0_BASE,EUSCI_B_SPI_TRANSMIT_INTERRUPT)));
  /* Transmitting data to slave */
  SPI_transmitData(EUSCI_B0_BASE, 0x00);

  /*Number of Register to Write*/
  /* Polling to see if the TX buffer is ready */
  while (!(SPI_getInterruptStatus(EUSCI_B0_BASE,EUSCI_B_SPI_TRANSMIT_INTERRUPT)));
  /* Transmitting data to slave */
  SPI_transmitData(EUSCI_B0_BASE, DATA);
  delay(2);
   /*EMPTY*/; // Wait for completion of final byte before raising /CS
  while (EUSCI_B_SPI_isBusy(EUSCI_B0_BASE));
  GPIO_setOutputHighOnPin(ADS1292_CS_PORT, ADS1292_CS_PIN);
  delay(2);
}

int Setup_Initialization()
{

        /* Halting WDT  */
       WDT_A_holdTimer();

       /* Initialize the  data ready and chip select pins*/
       GPIO_setAsInputPin(ADS1292_DRDY_PORT,ADS1292_DRDY_PIN);
       GPIO_setAsOutputPin(ADS1292_CS_PORT,ADS1292_CS_PIN);
       GPIO_setAsOutputPin(ADS1292_START_PORT,ADS1292_START_PIN);
       GPIO_setAsOutputPin(ADS1292_PWDN_PORT,ADS1292_PWDN_PIN);

       /* Selecting P1.5 P1.6 and P1.7 in SPI mode */
       GPIO_setAsPeripheralModuleFunctionInputPin(GPIO_PORT_P1,
               GPIO_PIN5 | GPIO_PIN6 | GPIO_PIN7, GPIO_PRIMARY_MODULE_FUNCTION);

       /* Configuring SPI in 3wire master mode */
       SPI_initMaster(EUSCI_B0_BASE, &spiMasterConfig);

       /* Enable SPI module */
       SPI_enableModule(EUSCI_B0_BASE);

       /* Enabling interrupts */
       SPI_enableInterrupt(EUSCI_B0_BASE, EUSCI_B_SPI_RECEIVE_INTERRUPT);
       Interrupt_enableInterrupt(INT_EUSCIB0);
       //Interrupt_enableSleepOnIsrExit();

       /* Selecting P1.2 and P1.3 in UART mode and P1.0 as output (LED) */
       GPIO_setAsPeripheralModuleFunctionInputPin(GPIO_PORT_P1,
                GPIO_PIN2 | GPIO_PIN3, GPIO_PRIMARY_MODULE_FUNCTION);
       GPIO_setAsOutputPin(GPIO_PORT_P1, GPIO_PIN0);
       GPIO_setOutputLowOnPin(GPIO_PORT_P1, GPIO_PIN0);

       /* Setting DCO to 24MHz (upping Vcore) */
       FlashCtl_setWaitState(FLASH_BANK0, 2);
       FlashCtl_setWaitState(FLASH_BANK1, 2);
       PCM_setCoreVoltageLevel(PCM_VCORE1);
       CS_setDCOCenteredFrequency(CS_DCO_FREQUENCY_24);

       /* Configuring UART Module */
       UART_initModule(EUSCI_A0_BASE, &uartConfig);

       /* Enable UART module */
       UART_enableModule(EUSCI_A0_BASE);

       /* Enabling interrupts */
       UART_enableInterrupt(EUSCI_A0_BASE, EUSCI_A_UART_RECEIVE_INTERRUPT);
       Interrupt_enableInterrupt(INT_EUSCIA0);
       //Interrupt_enableSleepOnIsrExit();

       /*ADS 1292 Initialization*/
       /*ADS1292 Reset*/
       GPIO_setOutputHighOnPin(ADS1292_PWDN_PORT, ADS1292_PWDN_PIN);
       delay(100);                 // Wait 100 mSec
       GPIO_setOutputLowOnPin(ADS1292_PWDN_PORT, ADS1292_PWDN_PIN);
       delay(100);
       GPIO_setOutputHighOnPin(ADS1292_PWDN_PORT, ADS1292_PWDN_PIN);
       delay(100);
       /*Disable Start*/
       GPIO_setOutputLowOnPin(ADS1292_START_PORT,ADS1292_START_PIN);
       delay(20);
       /*Enable Start*/
       GPIO_setOutputHighOnPin(ADS1292_START_PORT,ADS1292_START_PIN);
       delay(20);
       /*Hard Stop*/
       GPIO_setOutputLowOnPin(ADS1292_START_PORT,ADS1292_START_PIN);
       delay(100);
       /*Send Command Data START*/
       ads1292_SPI_Command_Data(START);
       /*Soft Stop*/
       ads1292_SPI_Command_Data(STOP);
       delay(50);
       /*Stop read data continuous*/
       ads1292_SPI_Command_Data(SDATAC);
       delay(300);

       ads1292_Reg_Write(ADS1292_REG_CONFIG1, 0x00);        //Set sampling rate to 125 SPS
       delay(10);
       ads1292_Reg_Write(ADS1292_REG_CONFIG2, 0b10100000);   //Lead-off comp off, test signal disabled
       delay(10);
       ads1292_Reg_Write(ADS1292_REG_LOFF, 0b00010000);      //Lead-off defaults
       delay(10);
       ads1292_Reg_Write(ADS1292_REG_CH1SET, 0b01000000);    //Ch 1 enabled, gain 6, connected to electrode in
       delay(10);
       ads1292_Reg_Write(ADS1292_REG_CH2SET, 0b01100000);    //Ch 2 enabled, gain 6, connected to electrode in
       delay(10);
       ads1292_Reg_Write(ADS1292_REG_RLDSENS, 0b00101100);   //RLD settings: fmod/16, RLD enabled, RLD inputs from Ch2 only
       delay(10);
       ads1292_Reg_Write(ADS1292_REG_LOFFSENS, 0x00);        //LOFF settings: all disabled
       delay(10);
                                                               //Skip register 8, LOFF Settings default
       ads1292_Reg_Write(ADS1292_REG_RESP1, 0b11110010);     //Respiration: MOD/DEMOD turned only, phase 0
       delay(10);
       ads1292_Reg_Write(ADS1292_REG_RESP2, 0b00000011);     //Respiration: Calib OFF, respiration freq defaults
       delay(10);
       /*Stop read data continuous*/
       ads1292_SPI_Command_Data(RDATAC);
       delay(10);

       /*Enable Start*/
       GPIO_setOutputHighOnPin(ADS1292_START_PORT,ADS1292_START_PIN);
       delay(20);
return 0;
}

int main(void)
{
    int i, j;
    Setup_Initialization();
    while(1)
    {
        if(GPIO_getInputPinValue(ADS1292_DRDY_PORT,ADS1292_DRDY_PIN) == false)
        {
            Read_1292 = true;
        }

        if (ads1292dataReceived == true)
        {
            j=0;
            for(i=0;i<6;i+=3)                  // data outputs is (24 status bits + 24 bits Respiration data +  24 bits ECG data)
            {

                uecgtemp = (unsigned long) (  ((unsigned long)SPI_RX_Buff[i+3] << 16) | ( (unsigned long) SPI_RX_Buff[i+4] << 8) |  (unsigned long) SPI_RX_Buff[i+5]);
                uecgtemp = (unsigned long) (uecgtemp << 8);
                secgtemp = (signed long) (uecgtemp);
                secgtemp = (signed long) (secgtemp >> 8);

                s32DaqVals[j++]=secgtemp;
            }

            DataPacketHeader[0] = CES_CMDIF_PKT_START_1 ;   // Packet header1 :0x0A
            DataPacketHeader[1] = CES_CMDIF_PKT_START_2;    // Packet header2 :0xFA
            DataPacketHeader[2] = (uint8_t) (data_len);     // data length
            DataPacketHeader[3] = (uint8_t) (data_len>>8);
            DataPacketHeader[4] = CES_CMDIF_TYPE_DATA;      // packet type: 0x02 -data 0x01 -commmand

            DataPacketHeader[5] = s32DaqVals[1];            // 4 bytes ECG data
            DataPacketHeader[6] = s32DaqVals[1]>>8;
            DataPacketHeader[7] = s32DaqVals[1]>>16;
            DataPacketHeader[8] = s32DaqVals[1]>>24;

            DataPacketHeader[9] = s32DaqVals[0];            // 4 bytes Respiration data
            DataPacketHeader[10] = s32DaqVals[0]>>8;
            DataPacketHeader[11] = s32DaqVals[0]>>16;
            DataPacketHeader[12] = s32DaqVals[0]>>24;

            DataPacketHeader[13] = CES_CMDIF_TYPE_DATA;   // Packet footer1:0x00
            DataPacketHeader[14] = CES_CMDIF_PKT_STOP ;   // Packet footer2:0x0B

            for(i=0; i<15; i++){
                UART_transmitData(EUSCI_A0_BASE, DataPacketHeader[i]);     // transmit the data over USB
            }
            ads1292dataReceived = false;
            SPI_RX_Buff_Count = 0;
        }
    }

//    PCM_gotoLPM0();
//    __no_operation();
}

//******************************************************************************
//
//This is the EUSCI_B0 interrupt vector service routine.
//
//******************************************************************************
void EUSCIB0_IRQHandler(void)
{
    uint32_t status = SPI_getEnabledInterruptStatus(EUSCI_B0_BASE);
    uint32_t jj;

    //SPI_clearInterruptFlag(EUSCI_B0_BASE, status);

    if(status & EUSCI_B_SPI_RECEIVE_INTERRUPT)
    {

        RXData = SPI_receiveData(EUSCI_B0_BASE);
        if(Read_1292 == true)
        {
            SPI_RX_Buff[SPI_RX_Buff_Count++] = RXData;  // store the result data in array

            if (SPI_RX_Buff_Count > 8){
                SPI_RX_Buff_Count = 0;
                ads1292dataReceived = true;
                Read_1292 = false;
            }
        }
    /* Delay between transmissions for slave to process information */
        for(jj=50;jj<50;jj++);

    }
}
