
#include <msp430.h>

#include <stdint.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>

#include "timermgr.h"


struct TimerTask
{
    uint32_t DelayValue;  //Defines at what time the timer task should execute
    TimerTaskFunc FuncPtr;
    void* FuncPtrData;
};

struct TimerMgr
{
    uint16_t TaskArrCapacity;
    uint16_t NumberOfTasks;
    struct TimerTask* TasksArr;
    uint32_t CurrentTickCount;
};

void DoTick( struct TimerMgr* Mgr )
{
    Mgr->CurrentTickCount += 1;
}

void DoTimer( struct TimerMgr* Mgr )
{
    uint16_t numexecuted = 0;
    uint16_t numtasks = Mgr->NumberOfTasks; //Cache this in case the number of tsks change during the loop
    while( numexecuted < numtasks )
    {
        if( (Mgr->TasksArr[numexecuted].DelayValue /* This is the timeout value specified in main.c */) > Mgr->CurrentTickCount )
        {
            break;
        }
        else
        {
            (*Mgr->TasksArr[numexecuted].FuncPtr)( Mgr->TasksArr[numexecuted].FuncPtrData );
            numexecuted += 1;
        }
    }

    if( numexecuted )
    {
        memmove( &Mgr->TasksArr[0], &Mgr->TasksArr[numexecuted], sizeof(struct TimerTask) * (Mgr->NumberOfTasks - numexecuted) );
        Mgr->NumberOfTasks -= numexecuted;
        Mgr->TasksArr = realloc( Mgr->TasksArr, Mgr->NumberOfTasks );
    }
}



struct TimerMgr* InitTimerMgr( uint32_t ClockPeriod )
{
    struct TimerMgr* mgr = malloc( sizeof(struct TimerMgr) );
    memset( mgr, 0, sizeof(struct TimerMgr) );
//    mgr->DelayScalingFactor = ClockPeriod / 100000 /* 1 uS Timer */;
    return mgr;
}

void FreeTimerMgr( struct TimerMgr** Mgr )
{
    free( (*Mgr)->TasksArr );
    free( *Mgr );
    *Mgr = NULL;
}

int InsertTimerTask( struct TimerMgr* Mgr, TimerTaskFunc TaskFuncPtr, void* Data, uint32_t DelayValue )
{
    //Find out where to insert
    uint16_t InsertionIndex;
//    for( InsertionIndex = Mgr->NumberOfTasks; InsertionIndex >= 0; InsertionIndex-- )
    for( InsertionIndex = 0; InsertionIndex < Mgr->NumberOfTasks; InsertionIndex++ )
    {
        if( Mgr->TasksArr[ Mgr->NumberOfTasks - InsertionIndex - 1 ].DelayValue < DelayValue )
        {
            break;
        }
    }

    //Resize the storage array, if necessary
    if( Mgr->NumberOfTasks >= Mgr->TaskArrCapacity )
    {
        struct TimerTask* newarr = realloc( Mgr->TasksArr, sizeof(struct TimerTask) * (Mgr->NumberOfTasks + 1) );
        memset( &Mgr->TasksArr[Mgr->NumberOfTasks], 0, sizeof(struct TimerTask) );
        if( newarr == NULL )
        {
            return -1;
        }
        else
        {
            Mgr->TasksArr = newarr;
            Mgr->TaskArrCapacity += 1;
        }
    }

    //Shift the array to accomodate insertion of the task, if necessary
    if( InsertionIndex < Mgr->NumberOfTasks )
    {
        memmove( &Mgr->TasksArr[InsertionIndex+1], &Mgr->TasksArr[InsertionIndex], sizeof(struct TimerTask) * (Mgr->NumberOfTasks - InsertionIndex) );
    }

    //Insert the task
    struct TimerTask* curtask = &Mgr->TasksArr[ InsertionIndex ];
    curtask->DelayValue = Mgr->CurrentTickCount + DelayValue;
    curtask->FuncPtr = TaskFuncPtr;
    curtask->FuncPtrData = Data;

    Mgr->NumberOfTasks += 1;

    return 0;
}
