/*! \file HAL_UART.c
 *  \brief Implementation of UART  module
 */


/*
 * HAL_UART.c
 *
 *  Created on: Feb 9, 2020
 *      Author: first
 */
//****************************************************************************
//
// HAL_UART.c - Hardware abstraction layer for UART with MSP432P401R
//
//****************************************************************************


#include "HAL_UART.h"

#ifdef USE_UART_INTERRUPTS
/*
 * Statically allocate memory for the UART 1 ring buffers
 *  We use in this module the macro defined in util.h to manage the ring buffer
 */

/*
 *========================================================
 * UART 1
 * RX: define buffer and accounting structure
 * TX: define buffer and accounting structure
 *========================================================
 */


/*
 *==================
 *  RX
 *==================
 */


/*
 * Define a ring buffer for the receive on UART 1
 */
static char rx_uart1_ringBuffer[MAX_RB_SIZE];       /*!< receive ring buffer on UART1 */

/*
 * Define ring buffer Accounting structure for receive ring buffer UART 1
*/
RingHdr rx_u1_rb;                                  /*!< ring buffer accounting for receive on UART 1 */


/*
 *==================
 *  TX
 *==================
 */

/*
 * Define a ring buffer for the transmit on UART 1
 */
static char tx_uart1_ringBuffer[MAX_RB_SIZE];           /*!< transmit ring buffer on UART1 */

/*
 * Define ring buffer Accounting structure for transmit ring buffer UART 1
*/
RingHdr tx_u1_rb;                                       /*!< ring buffer accounting for transmit  on UART 1 */


#endif


#ifdef USE_UART0_INTERRUPTS

/*
 * Statically allocate memory for the UART 0 ring buffers
 *  We use in this module the macro defined in util.h to manage the ring buffer
 */

/*
 *========================================================
 * UART 0
 * RX: define buffer and accounting structure
 * TX: define buffer and accounting structure
 *========================================================
 */


/*
 *==================
 *  RX
 *==================
 */


/*
 * Define a ring buffer for the receive on UART 0
 */
static char rx_uart0_ringBuffer[MAX_RB_SIZE];

/*
 * Define ring buffer Accounting structure for  transmit ring buffer UART 0
*/
RingHdr rx_u0_rb;

/*
 *==================
 *  TX
 *==================
 */


/*
 * Define a ring buffer for the transmit on UART 0
 */
static char tx_uart0_ringBuffer[MAX_RB_SIZE];

/*
 * Define ring buffer Accounting structure for  transmit ring buffer UART 0
*/
RingHdr tx_u0_rb;

#endif

/*
 *===================================
 * UART Configuarations
 *===================================
 */

/* UART 1 Configuration Parameters.
 * These are the configuration parameters to
 * make the eUSCI A UART module to operate with a 115200 baud rate.
 * These values were calculated using the online calculator that TI provides at:
 * http://software-dl.ti.com/msp430/msp430_public_sw/mcu/msp430/MSP430BaudRateConverter/index.html
 *
 * Note: We are, for now, using the same configuration parameters for both UART1 and UART0
 */
EUSCI_A_UART_initParam uartParam =
{
        EUSCI_A_UART_CLOCKSOURCE_SMCLK,                     /*!< SMCLK Clock Source */
        4,                                                  /*!< BRDIV = 4          */
        5,                                                  /*!< UCxBRF = 5         */
        85,                                                 /*!< UCxBRS = 85        */
        EUSCI_A_UART_NO_PARITY,                             /*!< No Parity          */
        EUSCI_A_UART_LSB_FIRST,                             /*!< LSB First          */
        EUSCI_A_UART_ONE_STOP_BIT,                          /*!< One stop bit       */
        EUSCI_A_UART_MODE,                                  /*!< UART mode          */
        EUSCI_A_UART_OVERSAMPLING_BAUDRATE_GENERATION       /*!< Oversampling       */
};

/* UART 0 Configuration Parameter.
 * These are the configuration parameters to
 * make the eUSCI A UART module to operate with a 115200 baud rate.
 * These values were calculated using the online calculator that TI provides at:
 * http://software-dl.ti.com/msp430/msp430_public_sw/mcu/msp430/MSP430BaudRateConverter/index.html
 *
 * Note: We are, for now, using the same configuration parameters for both UART1 and UART0
 */
EUSCI_A_UART_initParam uart0Param =
{
        EUSCI_A_UART_CLOCKSOURCE_SMCLK,                     /*!< SMCLK Clock Source */
        4,                                                  /*!< BRDIV = 4          */
        5,                                                  /*!< UCxBRF = 5         */
        85,                                                 /*!< UCxBRS = 85        */
        EUSCI_A_UART_NO_PARITY,                             /*!< No Parity          */
        EUSCI_A_UART_LSB_FIRST,                             /*!< LSB First          */
        EUSCI_A_UART_ONE_STOP_BIT,                          /*!< One stop bit       */
        EUSCI_A_UART_MODE,                                  /*!< UART mode          */
        EUSCI_A_UART_OVERSAMPLING_BAUDRATE_GENERATION       /*!< Oversampling       */
};


/*
 *=====================================================
 * This is the Backchannel Communication UART
 *=====================================================
 */
/*
 * Initializes Backchannel UART GPIO
 */
void UART_initGPIO(void)
{
    /*
     * Selecting UART functions for TXD
     */
    GPIO_setAsPeripheralModuleFunctionInputPin(
            UART_TXD_PORT,
            UART_TXD_PIN,
            UART_SELECT_FUNCTION
    );


    /*
     * Selecting UART functions for RXD
     */
    GPIO_setAsPeripheralModuleFunctionInputPin(
            UART_RXD_PORT,
            UART_RXD_PIN,
            UART_SELECT_FUNCTION
    );

    return;
}


/*
 * Notes:
 */
void UART_init(void)
{
#ifdef USE_UART_INTERRUPTS
    RingHdr *pRHDR;
#endif
    /*
     * Configuring UART Module
     */
    EUSCI_A_UART_init(EUSCI_A1_BASE, &uartParam);

#ifdef USE_UART_INTERRUPTS
    /*
      * Initialize the receive buffer accounting structure
      */
     pRHDR = &rx_u1_rb;
     ringHdrInit(pRHDR,MAX_RB_SIZE);

    /*
     * Initialize the transmit buffer accounting structure
     */
    pRHDR = &tx_u1_rb;
    ringHdrInit(pRHDR,MAX_RB_SIZE);
#endif

    /*
     * Enable UART1 module
     */
    EUSCI_A_UART_enable(EUSCI_A1_BASE);

#ifdef USE_UART_INTERRUPTS
    /*
     * Enable rx interrupts
     */
    EUSCI_A_UART_enableInterrupt(EUSCI_A1_BASE, EUSCI_A_UART_RECEIVE_INTERRUPT);
#endif

    return;
}


/*
 * Transmits String over UART1
 *
 */
void UART_transmitString( char *pStr )
{
#ifdef USE_UART_INTERRUPTS
    int idx = 0;
    RingHdr *pRHDR;

    pRHDR = &tx_u1_rb;
#endif

    while( *pStr != 0)
    {
#ifdef USE_UART_INTERRUPTS
        idx = ringHdrPut(pRHDR);
        if (idx == -1)
        {
            /* this means the ring buffer is full */
            _no_operation();

            /*
             * The current character was not processed
             * go back one on the pointer
             */
            pStr--;

            /*
             * Enable tx interrupts
             * to attempt to flush the ring buffer
             */
            EUSCI_A_UART_enableInterrupt(EUSCI_A1_BASE, EUSCI_A_UART_TRANSMIT_INTERRUPT);
            _no_operation();
        }
        else
        {
            /*
             * Store the character at the calculated index
             */
            tx_uart1_ringBuffer[(unsigned int)idx] = *pStr;
        }
#else
        EUSCI_A_UART_transmitData(EUSCI_A1_BASE, *pStr );
#endif
        pStr++;
    }
#ifdef USE_UART_INTERRUPTS
    /*
     * Enable tx interrupts
     */
    EUSCI_A_UART_enableInterrupt(EUSCI_A1_BASE, EUSCI_A_UART_TRANSMIT_INTERRUPT);
#endif

    return;
}

/*
 * Notes:
 */
void UART_transmitBuffer( char *pBuffer, unsigned int numBytes)
{
#ifdef USE_UART_INTERRUPTS
    int idx = 0;
    RingHdr *pRHDR;

    pRHDR = &tx_u1_rb;
#endif
    char *pCurrByte;        /*!< pointer to the current character from  buffer */

    if (pBuffer)
    {
        pCurrByte = pBuffer;    /*!< Point to the start of the specified buffer */
        while ( numBytes--)
        {
#ifdef USE_UART_INTERRUPTS
            idx = ringHdrPut(pRHDR);
            if (idx == -1)
            {
                /* this means the ring buffer is full */
                _no_operation();

                /*
                 * The current character was not processed
                 * go back one on the pointer
                 */
                pCurrByte--;

                /*
                 * Enable tx interrupts
                 * to attempt to flush the ring buffer
                 */
                EUSCI_A_UART_enableInterrupt(EUSCI_A1_BASE, EUSCI_A_UART_TRANSMIT_INTERRUPT);
                _no_operation();
            }
            else
            {
                /*
                 * Store the character at the calculated index
                 */
                tx_uart1_ringBuffer[(unsigned int)idx] = *pCurrByte;
            }
#else
            EUSCI_A_UART_transmitData(EUSCI_A1_BASE, *pCurrByte );
#endif
            pCurrByte++;
        }
#ifdef USE_UART_INTERRUPTS
        /*
         * Enable tx interrupts
         */
        EUSCI_A_UART_enableInterrupt(EUSCI_A1_BASE, EUSCI_A_UART_TRANSMIT_INTERRUPT);
#endif
    }

    return;
}


/*
 * Transmit a single character from a buffer position
 */
void UART_transmitCharacter( char *pCharacter)
{
#ifdef USE_UART_INTERRUPTS
    int idx = 0;
    RingHdr *pRHDR;

    pRHDR = &tx_u1_rb;
	
#endif

    if (pCharacter != 0)
    {
#ifdef USE_UART_INTERRUPTS
        idx = ringHdrPut(pRHDR);
        if ( idx == -1)
        {
            /*
             * This means the ring buffer is full
             * this character cannot be processed
             */
        }
        else
        {
            /*
             * Store the character at the calculated index
             */
            tx_uart1_ringBuffer[(unsigned int)idx] = *pCharacter;
            /*
             * Enable tx interrupts
             */
            EUSCI_A_UART_enableInterrupt(EUSCI_A1_BASE, EUSCI_A_UART_TRANSMIT_INTERRUPT);
        }
#else
        EUSCI_A_UART_transmitData(EUSCI_A1_BASE, *pCharacter );
#endif
    }

    return;
}

/*
 * Receive a single character
 *
 */
uint8_t UART_receiveCharacter(uint16_t timeout)
{
    uint8_t character = 0xFF;   /*!< this indicate an invalid character */
#ifdef USE_UART_INTERRUPTS
    int idx = -1;
    RingHdr *pRHDR;

    pRHDR = &rx_u1_rb;
    /*
     * When using interrupts
     * the RX U1 Ring Buffer is used to have
     * productor ( interrupt service routine) and consumer (the background function)
     * to work independently without contension of resources
     */
    if ( timeout == 0)
    {
        /*
         *  This case is a blocking call
         *  we will wait until a character is present
         *  in the rx ring buffer
         */
        while (idx == -1)
        {
            idx = ringHdrGet(pRHDR);
            if (idx != -1)
            {
                character = rx_uart1_ringBuffer[(unsigned int)idx];
            }
        }
    }
    else
    {
        /*
         * This case is a non blocking call
         * we try timeout count times to
         * read a character from the rx ring buffer
         * after this count is consumed we return indicating
         * no character available we return 0xFF
         */
        while (idx == -1 && timeout-- != 0)
        {
            idx = ringHdrGet(pRHDR);
            if (idx != -1)
            {
                character = rx_uart1_ringBuffer[(unsigned int)idx];
            }
        }
    }


#else
    character = EUSCI_A_UART_receiveData(EUSCI_A1_BASE);
#endif

    return character;
}

/*! \fn uint8_t UART_receiveString(char *pString)
 *  \brief receive a string of character from the UART
 *  \return a single character in an uint8_t
 */
void UART_receiveString(uint8_t *pStr)
{
    uint8_t character;
    register uint8_t *lp;
    if ( pStr != 0)
    {
        for (lp = pStr;;)
        {
            switch (character = UART_receiveCharacter(0) & 0x7F)
            {
            case '\n':
            case '\r':
                *lp = '\0';
                return;
            case '\b':
            case '\177':
                if ( lp > pStr)
                {
                    char c;
                    lp--;
                    c = '\b';
                    UART_transmitCharacter(&c);
                    c = ' ';
                    UART_transmitCharacter(&c);
                    c = '\b';
                    UART_transmitCharacter(&c);
                }
                break;

            case '#':
            {
                if ( lp > pStr)
                {
                    --lp;
                }
            }
            break;

            case 'r'&037:
            {
                uint8_t *p;
                char c;
                c = '\n';
                UART_transmitCharacter(&c);
                for (p = pStr; p < lp; p++)
                {
                    c = *p;
                    UART_transmitCharacter(&c);
                }
            }
            break;

            case '@':
            case 'u'&037:
            case 'w'&037:
            {
                char c;
                lp = pStr;
                c = '\n';
                UART_transmitCharacter(&c);
            }
            break;
            default:
                UART_transmitCharacter((char *)&character);
                *lp++ = character;
            }
        }
    }

    return;
}


#ifdef USE_UART_INTERRUPTS
//******************************************************************************
// UART 1  Interrupt ***********************************************************
//******************************************************************************
#if defined(__TI_COMPILER_VERSION__) || defined(__IAR_SYSTEMS_ICC__)
#pragma vector=USCI_A1_VECTOR
__interrupt void USCI_A1_ISR(void)
#elif defined(__GNUC__)
void __attribute__ ((interrupt(USCI_A1_VECTOR))) USCI_A1_ISR (void)
#else
#error Compiler not supported!
#endif
{
    switch(__even_in_range(UCA1IV, USCI_UART_UCTXCPTIFG))
    {
      /*
       * No Interrupt
       */
    case USCI_NONE:
    {
        _no_operation();
    }
    break;

      /*
       *================================================================================
       * The UCRXIFG interrupt flag is set each time a character is
       * received and loaded into UCAxRXBUF.
       * An interrupt request is generated if UCRXIE and GIE are also set.
       * UCRXIFG and UCRXIE are reset by a system reset PUC signal or when UCSWRST = 1.
       * UCRXIFG is automatically reset when UCAxRXBUF is read.
       *=================================================================================
       */
    case USCI_UART_UCRXIFG:
    {
        char character = 0xFF;      /*!< local character to hold the next character from the ring buffer    */
        RingHdr *pRHDR;
        pRHDR = &rx_u1_rb;
        int idx = 0;

        /*
         * Read the character from the UART FIFO (1char)
         */
        character = HWREG16(EUSCI_A1_BASE + OFS_UCAxRXBUF);

        /* Clear the interrupt flag */
        EUSCI_A_UART_clearInterrupt(EUSCI_A1_BASE, EUSCI_A_UART_RECEIVE_INTERRUPT_FLAG);

        /*
         * Store the received character in the receive ring buffer
         */
        idx = ringHdrPut(pRHDR);
        if ( idx != -1)
        {
            rx_uart1_ringBuffer[(unsigned int)idx] = character;
        }
        else
        {
            /* this means the receive ring buffer is full */
            /* we have to drop this character */
        }
    }
    break;

      /*
       *=================================================================================
       * The UCTXIFG interrupt flag is set by the transmitter to indicate
       * that UCAxTXBUF is ready to accept another character.
       * An interrupt request is generated if UCTXIE and GIE are also set.
       * UCTXIFG is automatically reset if a character is written to UCAxTXBUF.
       *=================================================================================
       */

    case USCI_UART_UCTXIFG:
    {
        char character;             /*!< local character to hold the next character from the ring buffer    */
        RingHdr *pRHDR;
        pRHDR = &tx_u1_rb;
        int idx = 0;


        idx = ringHdrGet(pRHDR);
        if ( idx == -1)
        {
            /*
             * This means the ring buffer is empty
             * we disable interrupts
             */
            EUSCI_A_UART_disableInterrupt(EUSCI_A1_BASE, EUSCI_A_UART_TRANSMIT_INTERRUPT);

            // Set TXIFG manually back to 1 for next time
            HWREG16(EUSCI_A1_BASE + OFS_UCAxIFG) |= EUSCI_A_UART_TRANSMIT_INTERRUPT;
        }
        else
        {
            /*
             * store the extracted character in the UART transmit buffer
             */
            character = tx_uart1_ringBuffer[(unsigned int)idx];
            HWREG16(EUSCI_A1_BASE + OFS_UCAxTXBUF) = character;
        }

    }
    break;

      /*
       *=================================================================================
       * START byte received interrupt.
       * This flag is set when the UART module receives a START byte.
       * This flag can be cleared by writing 0 to it
       *=================================================================================
       */
    case USCI_UART_UCSTTIFG:
    {
        _no_operation();
    }
    break;

      /*
       *=================================================================================
       * Transmit complete interrupt.
       * This flag is set after the complete UART byte
       *  in the internal shift register including STOP bit is shifted out.
       *  This flag can be cleared by writing 0 to it.
       *=================================================================================
       */
    case USCI_UART_UCTXCPTIFG:
    {
        _no_operation();
    }
    break;
  }
}

#endif /* USE_UART_INTERRUPTS */


/*------------------------------------------------------------------------------------------------------------------------------ */

/*
 * =====================================================
 * This is the Debug UART
 * =====================================================
 */

/*
 * Notes:
 */
void UART0_initGPIO(void)
{
    /* Selecting UART functions for TXD and RXD */
       GPIO_setAsPeripheralModuleFunctionInputPin(
               UART0_TXD_PORT,
               UART0_TXD_PIN,
               UART_SELECT_FUNCTION);

       GPIO_setAsPeripheralModuleFunctionInputPin(
               UART0_RXD_PORT,
               UART0_RXD_PIN,
               UART_SELECT_FUNCTION);
}

/*
 * Notes:
 */
void UART0_init(void)
{
#ifdef USE_UART0_INTERRUPTS
    RingHdr *pRHDR;
#endif

    /* Configuring UART Module */
    EUSCI_A_UART_init(EUSCI_A0_BASE, &uart0Param);

#ifdef USE_UART0_INTERRUPTS
    /*
     * Initialize the receive buffer accounting structure
     */
    pRHDR = &rx_u0_rb;
    ringHdrInit(pRHDR,MAX_RB_SIZE);

    /*
     * Initialize the transmit buffer accounting structure
     */
    pRHDR = &tx_u0_rb;
    ringHdrInit(pRHDR,MAX_RB_SIZE);
#endif

     /* Enable UART module */
     EUSCI_A_UART_enable(EUSCI_A0_BASE);

#ifdef USE_UART0_INTERRUPTS
     /*
      * Enable rx interrupts
      */
     EUSCI_A_UART_enableInterrupt(EUSCI_A0_BASE, EUSCI_A_UART_RECEIVE_INTERRUPT);
#endif

     return;
}

/*
 *
 */
void UART0_transmitString( char *pStr )
{
#ifdef USE_UART0_INTERRUPTS
    int idx = 0;
    RingHdr *pRHDR;

    pRHDR = &tx_u0_rb;
#endif

    while( *pStr != 0)
    {
#ifdef USE_UART0_INTERRUPTS
        idx = ringHdrPut(pRHDR);
        if (idx == -1)
        {
            /* this means the ring buffer is full */
            _no_operation();

            /*
             * The current character was not processed
             * go back one on the pointer
             */
            pStr--;

            /*
             * Enable tx interrupts
             * to attempt to flush the ring buffer
             */
            EUSCI_A_UART_enableInterrupt(EUSCI_A0_BASE, EUSCI_A_UART_TRANSMIT_INTERRUPT);
            _no_operation();
        }
        else
        {
            /*
             * Store the character at the calculated index
             */
            tx_uart0_ringBuffer[(unsigned int)idx] = *pStr;
        }
#else
        EUSCI_A_UART_transmitData(EUSCI_A0_BASE, *pStr );
#endif
        pStr++;
    }
#ifdef USE_UART0_INTERRUPTS
    /*
     * Enable tx interrupts
     */
    EUSCI_A_UART_enableInterrupt(EUSCI_A0_BASE, EUSCI_A_UART_TRANSMIT_INTERRUPT);
#endif

    return;

}

/*
 * Notes:
 */
void UART0_transmitCharacter( char *pCharacter)
{
#ifdef USE_UART0_INTERRUPTS
    int idx = 0;
    RingHdr *pRHDR;

    pRHDR = &tx_u0_rb;

#endif
    if (pCharacter != 0)
    {
#ifdef USE_UART0_INTERRUPTS
        idx = ringHdrPut(pRHDR);
        if (idx == -1)
        {
            /* this means the ring buffer is full */
            _no_operation();

            /*
             * The current character was not processed
             */

            /*
             * Enable tx interrupts
             * to attempt to flush the ring buffer
             */
            EUSCI_A_UART_enableInterrupt(EUSCI_A0_BASE, EUSCI_A_UART_TRANSMIT_INTERRUPT);
            _no_operation();
        }
        else
        {
            /*
             * Store the character at the calculated index
             */
            tx_uart0_ringBuffer[(unsigned int)idx] = *pCharacter;
            /*
             * Enable tx interrupts
             */
            EUSCI_A_UART_enableInterrupt(EUSCI_A0_BASE, EUSCI_A_UART_TRANSMIT_INTERRUPT);
        }
#else
        EUSCI_A_UART_transmitData(EUSCI_A0_BASE, *pCharacter );
#endif
    }

    return;
}

/*
 * Notes:
 */
uint8_t UART0_receiveCharacter(void)
{
    uint8_t character = 0xFF;   /*!< this indicate an invalid character */
#ifdef USE_UART0_INTERRUPTS
    volatile uint16_t timeout = MAX_RECEIVE_TIMEOUT;
    int idx = -1;
    RingHdr *pRHDR;

    pRHDR = &rx_u0_rb;
    /*
     * When using interrupts
     * the RX U0 Ring Buffer is used to have
     * productor ( interrupt service routine) and consumer (the foreground funtion)
     * to work independently without contension of resources
     */
    //while (idx == -1 && timeout-- != 0)
    while (idx == -1 )
    {
        idx = ringHdrGet(pRHDR);
        if (idx != -1)
        {
            character = rx_uart0_ringBuffer[(unsigned int)idx];
        }
    }
#else
    character = EUSCI_A_UART_receiveData(EUSCI_A0_BASE);
#endif

    return character;
}


/*! \fn uint8_t UART0_receiveString(char *pString)
 *  \brief receive a string of character from the UART
 *  \return a single character in an uint8_t
 */
void UART0_receiveString(uint8_t *pStr)
{
    uint8_t character;
    register uint8_t *lp;
    if ( pStr != 0)
    {
        for (lp = pStr;;)
        {
            switch (character = UART0_receiveCharacter() & 0x7F)
            {
            case '\n':
            case '\r':
                *lp = '\0';
                return;
            case '\b':
            case '\177':
                if ( lp > pStr)
                {
                    char c;
                    lp--;
                    c = '\b';
                    UART0_transmitCharacter(&c);
                    c = ' ';
                    UART0_transmitCharacter(&c);
                    c = '\b';
                    UART0_transmitCharacter(&c);
                }
                break;

            case '#':
            {
                if ( lp > pStr)
                {
                    --lp;
                }
            }
            break;

            case 'r'&037:
            {
                uint8_t *p;
                char c;
                c = '\n';
                UART0_transmitCharacter(&c);
                for (p = pStr; p < lp; p++)
                {
                    c = *p;
                    UART0_transmitCharacter(&c);
                }
            }
            break;

            case '@':
            case 'u'&037:
            case 'w'&037:
            {
                char c;
                lp = pStr;
                c = '\n';
                UART0_transmitCharacter(&c);
            }
            break;
            default:
                UART0_transmitCharacter((char *)&character);
                *lp++ = character;
            }
        }
    }

    return;
}



#ifdef USE_UART0_INTERRUPTS
//******************************************************************************
// UART 0  Interrupt ***********************************************************
//******************************************************************************

#if defined(__TI_COMPILER_VERSION__) || defined(__IAR_SYSTEMS_ICC__)
#pragma vector=USCI_A0_VECTOR
__interrupt void USCI_A0_ISR(void)
#elif defined(__GNUC__)
void __attribute__ ((interrupt(USCI_A0_VECTOR))) USCI_A0_ISR (void)
#else
#error Compiler not supported!
#endif
{
  switch(__even_in_range(UCA0IV, USCI_UART_UCTXCPTIFG))
  {

  /*
   *
   */
  case USCI_NONE:
  {
      _no_operation();
  }
  break;

  /*
   *================================================================================
   * The UCRXIFG interrupt flag is set each time a character is
   * received and loaded into UCAxRXBUF.
   * An interrupt request is generated if UCRXIE and GIE are also set.
   * UCRXIFG and UCRXIE are reset by a system reset PUC signal or when UCSWRST = 1.
   * UCRXIFG is automatically reset when UCAxRXBUF is read.
   *=================================================================================
   */
  case USCI_UART_UCRXIFG:
  {
      char character = 0xFF;      /*!< local character to hold the next character from the ring buffer    */
      RingHdr *pRHDR;
      pRHDR = &rx_u0_rb;
      int idx = 0;

      /*
       * Read the character from the UART FIFO (1char)
       */
      character = HWREG16(EUSCI_A0_BASE + OFS_UCAxRXBUF);

      /* Clear the interrupt flag */
      EUSCI_A_UART_clearInterrupt(EUSCI_A0_BASE, EUSCI_A_UART_RECEIVE_INTERRUPT_FLAG);

      /*
       * Store the received character in the receive ring buffer
       */
      idx = ringHdrPut(pRHDR);
      if ( idx != -1)
      {
          rx_uart0_ringBuffer[(unsigned int)idx] = character;
      }
      else
      {
          /* this means the receive ring buffer is full */
          /* we have to drop this character */
      }

  }
  break;

  /*
   *=================================================================================
   * The UCTXIFG interrupt flag is set by the transmitter to indicate
   * that UCAxTXBUF is ready to accept another character.
   * An interrupt request is generated if UCTXIE and GIE are also set.
   * UCTXIFG is automatically reset if a character is written to UCAxTXBUF.
   *=================================================================================
   */

  case USCI_UART_UCTXIFG:
  {
      char character;             /*!< local character to hold the next character from the ring buffer    */
      volatile bool status = false;       /*!< status of the ring buffer operation                                */
      character = character;
      RingHdr *pRHDR;
      pRHDR = &tx_u0_rb;
      int idx = 0;


      idx = ringHdrGet(pRHDR);
      if ( idx == -1)
      {
          /*
           * This means the ring buffer is empty
           * we disable interrupts
           */
          EUSCI_A_UART_disableInterrupt(EUSCI_A0_BASE, EUSCI_A_UART_TRANSMIT_INTERRUPT);

          // Set TXIFG manually back to 1 for next time
          HWREG16(EUSCI_A0_BASE + OFS_UCAxIFG) |= EUSCI_A_UART_TRANSMIT_INTERRUPT;
      }
      else
      {
          /*
           * store the extracted character in the UART transmit buffer
           */
          character = tx_uart0_ringBuffer[(unsigned int)idx];
          HWREG16(EUSCI_A0_BASE + OFS_UCAxTXBUF) = character;
      }
  }
  break;

  /*
   *=================================================================================
   * START byte received interrupt.
   * This flag is set when the UART module receives a START byte.
   * This flag can be cleared by writing 0 to it
   *=================================================================================
   */
  case USCI_UART_UCSTTIFG:
  {
      _no_operation();
  }
  break;

  /*
   *=================================================================================
   * Transmit complete interrupt.
   * This flag is set after the complete UART byte
   * in the internal shift register including STOP bit is shifted out.
   * This flag can be cleared by writing 0 to it.
   *=================================================================================
   */
  case USCI_UART_UCTXCPTIFG:
  {
      _no_operation();
  }
  break;
  }

} /* End of USCI_A0_ISR */

#endif /*  USE_UART0_INTERRUPTS */
