/*! \file HAL_UART.h
 *  \brief definitions of UART  module
 *
 *  This file contains definitions, data structures, function prototypes
 *  for the UART Hardware Abstraction Layer (HAL)

 *  The MightyPuffer uses the MCU UART's  for 2 main purposes:
 *   - logging of debug messages during the development phase of the product
 *   - identity and inspection programming during the life cycle of the product.
 *
 *  Two UART modules are provided by the MCU MSP430FR6989/MSP430FR5989
 *
 *      On the eval board
 *      The backchannel UART is the UART on eUSCI_A1.
 *      This UART channel is separate from the UART on the 20-pin BoosterPack connector (eUSCI_A0).
 *      P4.3 (UCA0RXD)  and P4.2 (UCA0TXD)
 *      For details of the Eval board look at document msp-exp430fr6989slau627a.pdf
 *
 *      On the MightyPuffer board
 *      We should provision for Test Points (TP) for both UART's
 *      We use the MCU msp430fr5989.
 *
 *      The first version of this driver was using a polling method to handle the low
 *      level aspects.
 *      This current version uses Interrupts on both transmit and receive sides
 *      for both UART.
 *      On the transmit side:
 *          - the producer is the background code ( functions)
 *          - the consumer is the foreground code ( ISR)
 *      On the receive side:
 *          - the producer is the foreground  code ( ISR)
 *          - the consumer is the background  code ( functions)
 *
 *      The data to be processed is managed by a ring buffer
 *
 *
 */

/*
 * HAL_UART.h
 *
 *  Created on: Feb 9, 2020
 *      Author: first
 */
//****************************************************************************
//
// HAL_UART.h - Prototypes of hardware abstraction layer for UART
//
//****************************************************************************

#ifndef __HAL_UART_H_
#define __HAL_UART_H_

#include <msp430.h>                      /*!< TI MSP430 Hardware definitions                                        */
#include <driverlib.h>                   /*!< TI low level driver library                                           */

#ifdef USE_UART_INTERRUPTS
#include "util.h"                       /*!< Utilities and useful macros - ring buffer management is located there */
#endif

#ifdef USE_UART0_INTERRUPTS
#include "util.h"                       /*!< Utilities and useful macros - ring buffer management is located there */
#endif


#define MAX_RB_SIZE             (128)  /*!< Maximum size of the ring buffers */


#define MAX_RECEIVE_TIMEOUT     (1024)  /*!< Retry count timeout to determine a receive is not returning data */


#ifdef USE_MOCKUP_LAUNCHPAD
/*
 *=======================================
 * The following Port and Pin assignment
 * are for the MOCKUP_LAUNCHPAD platform
 *
 *=======================================
 */

/*
 * Back Channel UART1 Ports and Pins
 * We use the back channel as the
 * Serial Communication Channel
 */
#define UART_TXD_PORT        (GPIO_PORT_P3)     /*!< Transmit Port on P3 */
#define UART_TXD_PIN         (GPIO_PIN4)        /*!< Transmit Pin 4 */

#define UART_RXD_PORT        (GPIO_PORT_P3)     /*!< Receive Port on P3 */
#define UART_RXD_PIN         (GPIO_PIN5)        /*!< Receive Pin 5 */


/*
 * Debug  UART0 Ports and Pins
 * We use the Debug UART for logging and tracing
 * as Developement assistance tools
 */
#define UART0_TXD_PORT      (GPIO_PORT_P4)      /*!< Transmit Port on P4    */
#define UART0_TXD_PIN       (GPIO_PIN2)         /*!< Transmit Pin 2         */

#define UART0_RXD_PORT      (GPIO_PORT_P4)      /*!< Receive Port on P4     */
#define UART0_RXD_PIN       (GPIO_PIN3)         /*!< Receive Pin 3          */

#else
/*
 *=======================================
 * The following Port and Pin assignment
 * are for the MightyPuffer platform
 * Note:
 *      They are for now identical to the
 *      ones for the MOCKUP_LAUNCHPAD
 *      as the hardware for the
 *      MightyPuffer board in currently
 *      the define phase
 *=======================================
 */

/*
 *  UART1  Ports and Pins
 */


/*
 * Back Channel UART1  Ports and Pins
 * We use the back channel as the
 * Serial Communication Channel
 */


#define UART_TXD_PORT        (GPIO_PORT_P3)     /*!< Transmit Port on P3 */
#define UART_TXD_PIN         (GPIO_PIN4)        /*!< Transmit Pin 4 */

#define UART_RXD_PORT        (GPIO_PORT_P3)     /*!< Receive Port on P3 */
#define UART_RXD_PIN         (GPIO_PIN5)        /*!< Receive Pin 5 */


/*
 * Debug  UART0 Ports and Pins
 * We use the Debug UART for logging and tracing
 * as Developement assistance tools
 */
#define UART0_TXD_PORT      (GPIO_PORT_P4)      /*!< Transmit Port on P4 */
#define UART0_TXD_PIN       (GPIO_PIN2)         /*!< Transmit Pin 2 */

#define UART0_RXD_PORT      (GPIO_PORT_P4)      /*!< Receive Port on P4 */
#define UART0_RXD_PIN       (GPIO_PIN3)         /*!< Receive Pin 3 */


#endif


#define UART_SELECT_FUNCTION GPIO_PRIMARY_MODULE_FUNCTION /*!< Multiplex Function Selector */

/*! \fn void UART_initGPIO(void)
 *  \brief Initialize GPIO's for Back channel UART
 *  no param,  no return
 */
void UART_initGPIO(void);

/*! \fn void UART_init(void)
 *  \brief Initialize the UART module EUSCI_A1
 *  no param, no return
 */
void UART_init(void);


/*! \fn void UART_transmitString( char *pStr)
 *  \brief transmit a string of character to the UART
 *  \param pStr A pointer to the string to send across
 */
void UART_transmitString( char *pStr );


/*! \fn void UART_transmitBuffer( char *pBuffer, unsigned int numBytes)
 *  \brief transmit a buffer of a specified number of bytes  to send the UART
 *  \param pStr A pointer to the buffer to send across
 *  \param numBytes A number of bytes
 */
void UART_transmitBuffer( char *pBuffer, unsigned int numBytes);


/*! \fn void UART_transmitCharacter( char *pBuffer)
 *  \brief transmit a single character located at pBuffer
 *  \param pCharacter A pointer to the character to transmit
 *
 */
void UART_transmitCharacter( char *pCharacter);

/*! \fn uint8_t UART_receiveCharacter(void)
 *  \brief receive a single character from the UART
 *  \param timeout number  of time we are looking for a character
 *
 *  \return a single character an uint8_t
 */
uint8_t UART_receiveCharacter(uint16_t timeout);


/*! \fn uint8_t UART_receiveString(char *pString)
 *  \brief receive a string of character from the UART
 *  \return a single character in an uint8_t
 */
void UART_receiveString(uint8_t *pStr);




/*! \fn void UART0_initGPIO(void)
 *  \brief Initialize GPIO's for Debug channel UART
 *  no param,  no return
 */
void UART0_initGPIO(void);

/*! \fn void UART0_init(void)
 *  \brief Initialize the UART module EUSCI_A0
 *  no param, no return
 */
void UART0_init(void);

/*! \fn void UART0_transmitString( char *pStr)
 *  \brief transmit a string of character to the UART
 *  \param pStr A pointer to the string to send across
 */
void UART0_transmitString( char *pStr );

/*! \fn void UART0_transmitCharacter( char *pBuffer)
 *  \brief transmit a single character located at pBuffer
 *  \param pBuffer A pointer to the character to transmit
 *
 */
void UART0_transmitCharacter( char *pBuffer);

/*! \fn uint8_t UART0_receiveCharacter(void)
 *  \brief receive a single character from the UART
 *  \return a single character in an uint8_t
 */
uint8_t UART0_receiveCharacter(void);


/*! \fn uint8_t UART_receiveString(char *pString)
 *  \brief receive a string of character from the UART
 *  \return a single character in an uint8_t
 */
void UART0_receiveString(uint8_t *pStr);

#endif /* __HAL_UART_H_ */
