/*!	\file log.h
*   \brief definitions of message logging module
*
*	This file contains definitions, data structures, function prototypes
*	for the message logging module of the MightyPuffer Device
*
*/

/*
 * log.h
 *
 *  Created on: Feb 28, 2020
 *      Author: first
 */

#ifndef INCLUDE_LOG_H_
#define INCLUDE_LOG_H_

#include <msp430.h>                 /*!<    */
#include <driverlib.h>              /*!<    */
#include <stdlib.h>                 /*!<    */
#include <stdio.h>                  /*!<    */
#include <stdint.h>                 /*!<    */
#include <string.h>                 /*!<    */
#include <stdarg.h>                 /*!<    */

#include "HAL_UART.h"               /*!<    */
#include "debug.h"                  /*!<    */


#define MAX_LOG_BUFFER_SIZE (350)       /*!< Maximum length of the log buffer */

/**
 * We define the log 4 log levels for sending messages to the log
 */
#define LOG_TRACE 1       /*!< This level is for tracing of code execution    */
#define LOG_INFO 2        /*!< This level is for communicating information    */
#define LOG_WARNING 3     /*!< This level is to indicate non fatal conditions */
#define LOG_ERROR 4       /*!< This level is to declare an error              */
#define LOG_ALWAYS 5      /*!< This loevel is to send inconditional message   */
#define LOG_OFF (LOG_ALWAYS+1)



/*
 * This is the format string for the header of all the messages
 * it uses a running count of milliseconds ( Timer A )
 * to extract seconds and milliseconds.
 * displayed as MP:<numSec>.<numMsec>:[]
 */
#define HEADER_FORMAT "MP:%06d.%03d:[]"



/**
 * \def LOG_DEFAULT_LEVEL - define this value before including the log file if you which to
 * over ride the level for a particular file you are using
 */
#ifndef LOG_DEFAULT_LEVEL
#define LOG_DEFAULT_LEVEL LOG_WARNING
#endif

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * \fn _log_printf
 * \brief private function to support the log printing system
 *
 * \param level the level of the message to create
 * \param .... a variable number of arguments
 */
void _log_printf(int level, ...);

#ifdef __cplusplus
}
#endif

/**
 * \fn log_print
 * \brief allows information to be sent over the debug UART back to the host system
 *
 * \param level - the debug level.
 * There is a global level but it can be overwritten on a
 * file by file basis
 *
 * \param format - the format string, which follows the same syntax as printf
 * \param values - the values required by the format string
 */
#define log_print(level, ...) \
 do { \
 if( (level) >= LOG_DEFAULT_LEVEL ) \
 _log_printf( level, __VA_ARGS__ ); \
 } while(0)

#endif /* INCLUDE_LOG_H_ */
