#include "System/helpers.h"

/**
 * @brief Extrage un bit dintr-un octet.
 *
 * @param byte Octetul verificat
 * @param position Pozitia bitului de extras
 *
 * @return Valoarea bitului extras
**/
unsigned char GetBit(unsigned char byte, unsigned char position)
{
	return ((byte >> position) & 1);
}

/**
 * @brief Extrage un grup de biti dintr-un octet.
 *
 * @param byte Octetul verificat
 * @param numberOfBits Numarul de biti din grup
 * @param firstBitIndex Pozitia primului bit din grup
 *
 * @return Valoarea grupului de biti extras
**/
unsigned char GetBitsFromChar(unsigned char byte, unsigned char numberOfBits, unsigned char firstBitIndex)
{
    int mask;
    switch (numberOfBits) // mask = (2^numberOfBits) - 1; se seteaza masca, nu se calculeaza (optimizare)
    {
        case 0:
            mask = 0;
            break;
        case 1:
            mask = 1;
            break;
        case 2:
            mask = 3;
            break;
        case 3:
            mask = 7;
            break;
        case 4:
            mask = 15;
            break;
        case 5:
            mask = 31;
            break;
        case 6:
            mask = 63;
            break;
        case 7:
            mask = 127;
            break;
    }
	return ((byte >> firstBitIndex) & mask);
}

/**
 * @brief Extrage un grup de biti dintr-un intreg.
 *
 * @param integer Intregul verificat
 * @param numberOfBits Numarul de biti din grup
 * @param firstBitIndex Pozitia primului bit din grup
 *
 * @return Valoarea grupului de biti extras
**/
int GetBitsFromInt(int integer, unsigned char numberOfBits, unsigned char firstBitIndex)
{
    int mask;
    switch (numberOfBits) // mask = (2^numberOfBits) - 1; se seteaza masca, nu se calculeaza (optimizare)
    {
        case 0:
            mask = 0;
            break;
        case 1:
            mask = 1;
            break;
        case 2:
            mask = 3;
            break;
        case 3:
            mask = 7;
            break;
        case 4:
            mask = 15;
            break;
        case 5:
            mask = 31;
            break;
        case 6:
            mask = 63;
            break;
        case 7:
            mask = 127;
            break;
        case 8:
            mask = 255;
            break;
        case 9:
            mask = 511;
            break;
        case 10:
            mask = 1023;
            break;
        case 11:
            mask = 2047;
            break;
        case 12:
            mask = 4095;
            break;
        case 13:
            mask = 8191;
            break;
        case 14:
            mask = 16383;
            break;
        case 15:
            mask = 32767;
            break;
        case 16:
            mask = 65535;
            break;
        case 17:
            mask = 131071;
            break;
        case 18:
            mask = 262141;
            break;
        case 19:
            mask = 524287;
            break;
        case 20:
            mask = 1048575;
            break;
        case 21:
            mask = 2097151;
            break;
        case 22:
            mask = 4194303;
            break;
        case 23:
            mask = 8388607;
            break;
        case 24:
            mask = 16777215;
            break;
        case 25:
            mask = 33554431;
            break;
        case 26:
            mask = 67108863;
            break;
        case 27:
            mask = 134217727;
            break;
        case 28:
            mask = 268435455;
            break;
        case 29:
            mask = 536870911;
            break;
        case 30:
            mask = 1073741823;
            break;
        case 31:
            mask = 2147483647;
            break;
    }
    return ((integer >> firstBitIndex) & mask);
}

/**
 * @brief Seteaza un bit dintr-un octet.
 *
 * @param byte Octetul modificat
 * @param position Pozitia bitului de setat
**/
void SetBit(unsigned char* byte, unsigned char position)
{
	*byte |= 1 << position;
}

/**
 * @brief Deseteaza un bit dintr-un octet.
 *
 * @param byte Octetul modificat
 * @param position Pozitia bitului de desetat
**/
void ClearBit(unsigned char* byte, unsigned char position)
{
	*byte &= ~(1 << position);
}

/**
 * @brief Comuta valoarea unui bit dintr-un octet.
 *
 * @param byte Octetul modificat
 * @param position Pozitia bitului de comutat
**/
void ToggleBit(unsigned char* byte, unsigned char position)
{
	*byte ^= 1 << position;
}

/**
 * @brief Extrag un octet dintr-un intreg.
 *
 * @param number Intregul verificat
 * @param byteNumber Indicele octetului de extras
 *
 * @return Octetul extras
**/
unsigned char GetByte(int number, unsigned char byteNumber)
{
	return (number >> (8 * byteNumber)) & 0xFF;
}

/**
 * @brief Extrag un octet dintr-un intreg unsigned long.
 *
 * @param number Intregul verificat
 * @param byteNumber Indicele octetului de extras
 *
 * @return Octetul extras
**/
unsigned char GetByteULong(unsigned long number, unsigned char byteNumber)
{
    return (number >> (8 * byteNumber)) & 0xFF;
}

/**
 * @brief Ridica un intreg la putere.
 *
 * @param base Baza
 * @param exp Exponent
 *
 * @return Intregul calculat
**/
int intPow(int base, int exp)
{
    int result = 1;
    while (exp)
    {
        if (exp & 1)
            result *= base;
        exp >>= 1;
        base *= base;
    }

    return result;
}

/**
 * @brief Extrage un string din alt string.
 *
 * @param string String initial
 * @param position Primul caracter din substring
 * @param length Lungime substring
 *
 * @return Pointer catre substringul extras
**/
char *substring(volatile char *string, int position, int length)
{
    char *pointer;
    int c;

    pointer = malloc(length + 1);

    if (pointer == 0)
    {
        //printf("Unable to allocate memory.\n");
        //exit(1);
    }

    for (c = 0 ; c < length ; c++)
    {
        *(pointer+c) = *(string+position-1);
        string++;
    }

    *(pointer+c) = '\0';
    return pointer;
}

/**
 * @brief Converste o valoare intreaga CC2 pe 24 de biti
 * in intreg cu semn.
 *
 * @param val Valoarea de convertit
 *
 * @return Valoarea convertita
**/
int Int24BitTwoComplementToSignedInt(unsigned int val)
{
    return (0x800000 & val ? (int)(0x7FFFFF & val) - 0x800000 : val);
}

/**
 * @brief Adauga intregi la string (concatenare) fara a pune terminatorul de string la final.
 *
 * @param dest String destinatie
 * @param bytesArray Buffer de intregi de adaugat
 * @param numberOfInts Lungime buffer
 * @param numberOfBytesPerInt Numar de octeti pe intreg
 * @param newDestLength Noua lungime a stringului destinatie
 *
 * @return Pointer catre stringul destinatie
**/
char* AppendBytesToString(char* dest, int* bytesArray, int numberOfInts, int numberOfBytesPerInt, int* newDestLength)
{
    int i,j,k,l=0;
    for (i = 0; dest[i] != '\0'; i++)
        ;

    for (k = 0; k < numberOfInts; k++)
    {
        for (j = 0; j < numberOfBytesPerInt; j++) // numberOfBytesPerInt octeti
        {
            dest[i+l] = GetByte(bytesArray[k], (numberOfBytesPerInt-1)-j); // MSB primul
            l++;
        }
    }
    //dest[i+j] = '\0';
    //dest[i+l] = '\0';
    *newDestLength = i+l;

    return dest;
}

/**
 * @brief Verifica daca un string contine alt string.
 *
 * @param string Stringul verificat
 * @param response Stringul dupa care se verifica
 * @param stringLength Lungime string verificat
 *
 * @return true/false daca stringul contine/nu contine stringul verificat
**/
bool ContainsString(volatile char* string, const char* response, int stringLength)
{
    int i;
    char* temp;
    temp = (char*) calloc(stringLength, sizeof(char));
    for (i = 0; i < stringLength; i++)
    {
        temp[i] = string[i];
    }

    if (strstr(temp, response) != NULL)
    {
        free(temp);
        return true;
    }

    free(temp);
    return false;
}

/**
 * @brief Compara doi intregi exprimati in CC2. Returneaza numarul mai mare
 * sau unul dintre numere daca sunt egale.
 *
 * @param a Numarul a
 * @param b Numarul b
 * @param integerLength Lungime intreg (in biti)
 *
 * @return true/false daca stringul contine/nu contine stringul verificat
**/
int CompareTwoComplement(int a, int b, int integerLength)
{
    int a_msb = ((a >> integerLength-1) & 1);
    int b_msb = ((b >> integerLength-1) & 1);
    if (a_msb != b_msb) // Semne diferite; numarul pozitiv este mai mare
    {
        if (a_msb == 0) return a; // a > b
        return b; // Else b > a
    }
    else // Semne egale
    {
        int a1 = GetBitsFromInt(a, integerLength-1, 0);
        int b1 = GetBitsFromInt(b, integerLength-1, 0);
        if (a_msb == 0) // Ambele pozitive
        {
            if (a1 > b1) return a; // a > b
            return b; // Else b > a
        }
        else // Ambele negative
        {
            if (-a1 > -b1) return b; // a < b
            return a; // Else b < a
        }
    }
}

