#include "System/timers.h"

volatile bool timeoutA2Flag = false;
volatile bool timeoutA3Flag = false;
volatile bool timer32Finished = false;
volatile int timeoutA0Cycles = 0;

/**
 * @brief Initializeaza Sys timer. Este utilizat pentru resetarea Watchdog-ului.
**/
void System_Timer_init(void)
{
	MAP_SysTick_setPeriod(4800000); // Maximum period is 16777216; 10 ms @ MCLK = 48 MHz
	MAP_SysTick_enableModule();
	MAP_SysTick_enableInterrupt();
}

/**
 * @brief Initializeaza timer 32.
**/
void Timer32_init(void)
{
    MAP_Timer32_initModule(TIMER32_0_BASE, TIMER32_PRESCALER_1, TIMER32_32BIT, TIMER32_PERIODIC_MODE);
    MAP_Timer32_enableInterrupt(TIMER32_0_BASE);
    MAP_Interrupt_enableInterrupt(INT_T32_INTC);
}

/**
 * @brief Trece in sleep procesorul pentru un numar de cicli.
 *
 * @param cycles Numarul de cicli cat sta procesorul in sleep
**/
void Timer32_sleep_cycles(uint32_t cycles)
{
    /* Configuring Timer32 to "uint32_t cycles" cycles of MCLK in periodic mode */
	if (cycles <= 4294967295)
	    MAP_Timer32_setCount(TIMER32_0_BASE, cycles);
	else
	    MAP_Timer32_setCount(TIMER32_0_BASE, 4294967295);

	MAP_Timer32_startTimer(TIMER32_0_BASE, true);

	//MAP_Interrupt_enableSleepOnIsrExit();
	MAP_Interrupt_enableMaster();
	timer32Finished = false;
	while (timer32Finished == false)
	{
	    MAP_PCM_gotoLPM0InterruptSafe(); // Go to sleep until timer reaches the number of cycles; MAP_PCM_gotoLPM0InterruptSafe() disables master interrupts
	    MAP_Interrupt_enableMaster();
	}
}

/*void Timer32_sleep_us(uint32_t microseconds)
{
    // Configuring Timer32 to equivalent cycles of MCLK in periodic mode
    uint32_t cycles;

    MAP_Timer32_initModule(TIMER32_BASE, TIMER32_PRESCALER_1, TIMER32_32BIT, TIMER32_PERIODIC_MODE);
    MAP_Interrupt_enableInterrupt(INT_T32_INT1);

    if (microseconds < MCLK_PERIOD) microseconds = MCLK_PERIOD;
    if (microseconds > TIMER32_MAX_PERIOD) microseconds = TIMER32_MAX_PERIOD;

    cycles = (uint32_t)((double)microseconds / MCLK_PERIOD);

    MAP_Timer32_setCount(TIMER32_BASE, cycles);
    MAP_Timer32_enableInterrupt(TIMER32_BASE);
    MAP_Timer32_startTimer(TIMER32_BASE, true);

    MAP_Interrupt_enableSleepOnIsrExit();
    MAP_Interrupt_enableMaster();
    MAP_PCM_gotoLPM0InterruptSafe(); // Go to sleep until timer reaches the number of cycles; MAP_PCM_gotoLPM0InterruptSafe() disables master interrupts
    MAP_Interrupt_enableMaster();
}*/

/**
 * @brief Opreste timer 32.
**/
void Timer32_turnOff(void)
{
	MAP_Timer32_haltTimer(TIMER32_BASE);
    MAP_Interrupt_disableSleepOnIsrExit();
}

/**
 * @brief Configureaza timer A0.
 *
 * @param timeoutCycles Numarul de cicli dupa care genereaza intrerupere
**/
void TimerA0_init(int timeoutCycles)
{
	Timer_A_UpModeConfig timerA_param = {0};
    timerA_param.clockSource = TIMER_A_CLOCKSOURCE_SMCLK;
    timerA_param.clockSourceDivider = TIMER_A_CLOCKSOURCE_DIVIDER_1;
    timerA_param.timerPeriod = 10000;
    timerA_param.timerInterruptEnable_TAIE = TIMER_A_TAIE_INTERRUPT_DISABLE;
    timerA_param.captureCompareInterruptEnable_CCR0_CCIE = TIMER_A_CCIE_CCR0_INTERRUPT_ENABLE;
    timerA_param.timerClear = TIMER_A_DO_CLEAR;

    timeoutA0Cycles = timeoutCycles;

    MAP_Timer_A_configureUpMode(TIMER_A0_BASE, &timerA_param);
    MAP_Interrupt_enableInterrupt(INT_TA0_0);
    //MAP_Interrupt_enableSleepOnIsrExit(); // Comentat ca daca altfel imi executa urmatoarea functie (in program, dupa locul unde e apelata TimerA0_init), dupa ce trecea timeoutCycles
    MAP_Interrupt_enableMaster();
    MAP_Timer_A_startCounter(TIMER_A0_BASE,TIMER_A_UP_MODE);
}

/**
 * @brief Configureaza timer A1. Timer achizitii periodice, actualizare
 * stare interfete de comunicatie si operare circuit W5500.
**/
void TimerA1_init(void)
{
    Timer_A_UpModeConfig timerA_param = {0};
    timerA_param.clockSource = TIMER_A_CLOCKSOURCE_SMCLK;
    timerA_param.clockSourceDivider = TIMER_A_CLOCKSOURCE_DIVIDER_1;
    timerA_param.timerPeriod = 10000;
    timerA_param.timerInterruptEnable_TAIE = TIMER_A_TAIE_INTERRUPT_DISABLE;
    timerA_param.captureCompareInterruptEnable_CCR0_CCIE = TIMER_A_CCIE_CCR0_INTERRUPT_ENABLE;
    timerA_param.timerClear = TIMER_A_DO_CLEAR;

    MAP_Timer_A_configureUpMode(TIMER_A1_BASE, &timerA_param);
    MAP_Interrupt_enableInterrupt(INT_TA1_0);
    //MAP_Interrupt_enableSleepOnIsrExit();
    MAP_Interrupt_enableMaster();
    MAP_Timer_A_startCounter(TIMER_A1_BASE,TIMER_A_UP_MODE);
    MAP_Interrupt_enableMaster();
}

/**
 * @brief Opreste timer A1.
**/
void TimerA1_turnOff(void)
{
    MAP_Timer_A_stopTimer(TIMER_A1_BASE);
    //MAP_Timer_A_clearTimer(TIMER_A1_BASE);
}

/**
 * @brief Reporneste timer A1 de la valoarea la care a ramas.
**/
void TimerA1_resume(void)
{
    MAP_Interrupt_enableMaster();
    MAP_Timer_A_startCounter(TIMER_A1_BASE,TIMER_A_UP_MODE);
    MAP_Interrupt_enableMaster();
}

/**
 * @brief Reporneste timer A1 de la 0.
**/
void TimerA1_restart(void)
{
    MAP_Timer_A_stopTimer(TIMER_A1_BASE);
    TimerA1_resume();
}

/**
 * @brief Opreste timer A0.
**/
void TimerA0_turnOff(void)
{
    MAP_Timer_A_stopTimer(TIMER_A0_BASE);
    //MAP_Timer_A_clearTimer(TIMER_A0_BASE);
}

/**
 * @brief Reporneste timer A0 de la valoarea la care a ramas.
**/
void TimerA0_resume(void)
{
    MAP_Interrupt_enableMaster();
    MAP_Timer_A_startCounter(TIMER_A0_BASE,TIMER_A_UP_MODE);
    MAP_Interrupt_enableMaster();
}

/**
 * @brief Reporneste timer A0 de la 0.
**/
void TimerA0_restart(void)
{
    MAP_Timer_A_stopTimer(TIMER_A0_BASE);
    TimerA0_resume();
}

/**
 * @brief Porneste timer A2. Timer timeout TX si RX (SPI, UART).
 *
 * @param cycles Timeout SMCLK cicli (maxim 65536)
**/
void TimerA2_startCycles(int cycles)
{
    Timer_A_UpModeConfig timerA_param = {0};
    timerA_param.clockSource = TIMER_A_CLOCKSOURCE_SMCLK;
    timerA_param.clockSourceDivider = TIMER_A_CLOCKSOURCE_DIVIDER_1;
    timerA_param.timerPeriod = cycles; // SMCLK = MCLK / 2; deci se poate astepta minim 2 cicli MCLK (adica 1 ciclu SMCLK)
    timerA_param.timerInterruptEnable_TAIE = TIMER_A_TAIE_INTERRUPT_DISABLE;
    timerA_param.captureCompareInterruptEnable_CCR0_CCIE = TIMER_A_CCIE_CCR0_INTERRUPT_ENABLE;
    timerA_param.timerClear = TIMER_A_DO_CLEAR;

    MAP_Timer_A_configureUpMode(TIMER_A2_BASE, &timerA_param);
    MAP_Interrupt_enableInterrupt(INT_TA2_0);
    //MAP_Interrupt_enableSleepOnIsrExit();
    MAP_Interrupt_enableMaster();
    MAP_Timer_A_startCounter(TIMER_A2_BASE, TIMER_A_UP_MODE);
}

/**
 * @brief Porneste timer A2. Timer timeout TX si RX (SPI, UART).
 *
 * @param hundredsMs Timeout in sute de ms (maxim 642, adica 64 s)
**/
void TimerA2_startHundredsMiliseconds(int hundredsMs)
{
    Timer_A_UpModeConfig timerA_param = {0};
    timerA_param.clockSource = TIMER_A_CLOCKSOURCE_ACLK;
    timerA_param.clockSourceDivider = TIMER_A_CLOCKSOURCE_DIVIDER_32;
    timerA_param.timerPeriod = 102*hundredsMs;
    timerA_param.timerInterruptEnable_TAIE = TIMER_A_TAIE_INTERRUPT_DISABLE;
    timerA_param.captureCompareInterruptEnable_CCR0_CCIE = TIMER_A_CCIE_CCR0_INTERRUPT_ENABLE;
    timerA_param.timerClear = TIMER_A_DO_CLEAR;

    MAP_Timer_A_configureUpMode(TIMER_A2_BASE, &timerA_param);
    MAP_Interrupt_enableInterrupt(INT_TA2_0);
    //MAP_Interrupt_enableSleepOnIsrExit();
    MAP_Interrupt_enableMaster();
    MAP_Timer_A_startCounter(TIMER_A2_BASE, TIMER_A_UP_MODE);
}

/**
 * @brief Opreste timer A2.
**/
void TimerA2_turnOff(void)
{
    MAP_Timer_A_stopTimer(TIMER_A2_BASE);
    //MAP_Timer_A_clearTimer(TIMER_A2_BASE);
}

/**
 * @brief Porneste timer A3. Timer timeout TX si RX (SPI, UART).
 *
 * @param cycles Timeout SMCLK cicli (maxim 65536)
**/
void TimerA3_startCycles(int cycles)
{
    Timer_A_UpModeConfig timerA_param = {0};
    timerA_param.clockSource = TIMER_A_CLOCKSOURCE_SMCLK;
    timerA_param.clockSourceDivider = TIMER_A_CLOCKSOURCE_DIVIDER_1;
    timerA_param.timerPeriod = cycles; // SMCLK = MCLK / 2; deci se poate astepta minim 2 cicli MCLK (adica 1 ciclu SMCLK)
    timerA_param.timerInterruptEnable_TAIE = TIMER_A_TAIE_INTERRUPT_DISABLE;
    timerA_param.captureCompareInterruptEnable_CCR0_CCIE = TIMER_A_CCIE_CCR0_INTERRUPT_ENABLE;
    timerA_param.timerClear = TIMER_A_DO_CLEAR;

    MAP_Timer_A_configureUpMode(TIMER_A3_BASE, &timerA_param);
    MAP_Interrupt_enableInterrupt(INT_TA3_0);
    //MAP_Interrupt_enableSleepOnIsrExit();
    MAP_Interrupt_enableMaster();
    MAP_Timer_A_startCounter(TIMER_A3_BASE, TIMER_A_UP_MODE);
}

/**
 * @brief Porneste timer A3. Timer timeout TX si RX (SPI, UART).
 *
 * @param hundredsMs Timeout in sute de ms (maxim 642, adica 64 s)
**/
void TimerA3_startHundredsMiliseconds(int hundredsMs)
{
    Timer_A_UpModeConfig timerA_param = {0};
    timerA_param.clockSource = TIMER_A_CLOCKSOURCE_ACLK;
    timerA_param.clockSourceDivider = TIMER_A_CLOCKSOURCE_DIVIDER_32;
    timerA_param.timerPeriod = 102*hundredsMs;
    timerA_param.timerInterruptEnable_TAIE = TIMER_A_TAIE_INTERRUPT_DISABLE;
    timerA_param.captureCompareInterruptEnable_CCR0_CCIE = TIMER_A_CCIE_CCR0_INTERRUPT_ENABLE;
    timerA_param.timerClear = TIMER_A_DO_CLEAR;

    MAP_Timer_A_configureUpMode(TIMER_A3_BASE, &timerA_param);
    //MAP_Interrupt_enableSleepOnIsrExit();
    MAP_Interrupt_enableInterrupt(INT_TA3_0);
    MAP_Interrupt_enableMaster();
    MAP_Timer_A_startCounter(TIMER_A3_BASE, TIMER_A_UP_MODE);
}

/**
 * @brief Opreste timer A3.
**/
void TimerA3_turnOff(void)
{
    MAP_Timer_A_stopTimer(TIMER_A3_BASE);
    //MAP_Timer_A_clearTimer(TIMER_A3_BASE);
}
