/*
 * \file   TI_MSPBoot_MI_FRAMDualImg.c
 *
 * \brief  Driver for memory interface using FRAM in F5529
 *         This file supports Dual Image
 */
/* --COPYRIGHT--,BSD
 * Copyright (c) 2017, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/

//
// Include files
//
#include "msp430.h"
#include "TI_MSPBoot_Common.h"
#include "TI_MSPBoot_MI_16Bit.h"

//
// Local Function prototypes
//
static uint8_t TI_MSPBoot_MI_EraseSectorDirect(uint16_t addr);
static uint8_t TI_MSPBoot_MI_WriteByteDirect(uint16_t addr, uint8_t data);
// Function prototypes
void TI_MSPBoot_MI_EraseAppDirect(uint8_t DownArea);
void TI_MSPBoot_MI_ReplaceApp(void);
uint16_t TI_MSPBoot_MI_GetPhysicalAddressFromVirtual(uint16_t addr);

//
//  Functions declarations
//
/******************************************************************************
*
 * @brief   Erase a Flash Sector
 *
 * @param  addr    Address in the sector being erased (sector is 512B)
 *                  The actual address will be in Download area
 *
 * @return  RET_OK when successful,
 *          RET_PARAM_ERROR if address is outside of Application area
 *****************************************************************************/
uint8_t TI_MSPBoot_MI_EraseSector(uint16_t addr)
{
#ifdef CONFIG_MI_MEMORY_RANGE_CHECK
    // Check address to be within Application range
	if ((addr < APP_START_ADDR) || (addr > APP_END_ADDR))
        return RET_PARAM_ERROR;
#endif

    // Erase the corresponding area
    TI_MSPBoot_MI_EraseSectorDirect(TI_MSPBoot_MI_GetPhysicalAddressFromVirtual(addr));

    return RET_OK;
}

/******************************************************************************
*
 * @brief   Erase a Flash Sector directly
 *
 * @param  addr    Address in the sector being erased (sector is 512B)
 *
 * @return  RET_OK when successful,
 *          RET_PARAM_ERROR if address is outside of Application area
 *****************************************************************************/
static uint8_t TI_MSPBoot_MI_EraseSectorDirect(uint16_t addr)
{
    FCTL1 = FWKEY + ERASE;                      // Enable flash erase
#if (MCLK==1000000)
    FCTL2 = FWKEY + FSSEL_2 + 2;              // Flash timing setup (SMCLK/3) = 333.333Khz
#elif (MCLK==4000000)
    FCTL2 = FWKEY + FSSEL_2 + 8;              // Flash timing setup (SMCLK/9) = 444.444Khz
#elif (MCLK==8000000)
    FCTL2 = FWKEY + FSSEL_2 + 16;             // Flash timing setup (SMCLK/17) = 470.588Khz
#else
#error "Please define a valid MCLK or add configuration"
#endif
    FCTL3 = FWKEY;                              // Disable lock
    *(unsigned int *)addr = 0;                  // Dummy write to erase flash
    FCTL1 = FWKEY;
    FCTL3 = FWKEY+LOCK;                         // Disable flash write

    return RET_OK;
}
        
/******************************************************************************
 *
 * @brief   Erase the application area (address obtained from linker file)
 *          Erases the application in Download Area
 *          FRAM doesn't have an "erased" state but this function is added
 *          for compatibility with Flash and in order to calculate CRC
 *
 * @return  none
 *****************************************************************************/
void TI_MSPBoot_MI_EraseApp(void)
{
    // Erase Download Area
    TI_MSPBoot_MI_EraseAppDirect(1);
 
}

/******************************************************************************
 *
 * @brief   Erase the application area (address obtained from linker file)
 *          It can erase application in Download or App Area
 *
 * @param DownArea 1:Erase Download area, 0: Erase App Area
 *
 * @return  none
 *****************************************************************************/
void TI_MSPBoot_MI_EraseAppDirect(uint8_t DownArea)
{
    uint16_t addr;

    for (addr = APP_END_ADDR; addr >APP_START_ADDR; addr -= 512)
    {
        if(DownArea==0)
        {
            TI_MSPBoot_MI_EraseSectorDirect(addr);
        }
        else
        {
            TI_MSPBoot_MI_EraseSector(addr);
        }
    }
}

/******************************************************************************
 *
 * @brief   Write a Byte to Flash memory
 *      This function writes the byte to Download area 
 *
 * @param  addr     Address of the Byte being written in Application area
 *                  The actual address will be in Download area
 * @param  data     Byte being written
 *
 * @return  RET_OK when sucessful,
 *          RET_PARAM_ERROR if address is outside of Application area
 *****************************************************************************/
uint8_t TI_MSPBoot_MI_WriteByte(uint16_t addr, uint8_t data)
{
#ifdef CONFIG_MI_MEMORY_RANGE_CHECK
    // Check address to be within Application range
	if ((addr < APP_START_ADDR) || (addr > APP_END_ADDR))
        return RET_PARAM_ERROR;
#endif

    // Write the byte
    TI_MSPBoot_MI_WriteByteDirect(TI_MSPBoot_MI_GetPhysicalAddressFromVirtual(addr), data);
    
    return RET_OK;
}


/******************************************************************************
 *
 * @brief   Write a Byte Directly to Flash memory
 *
 * @param  addr     Address of the Byte being written in Flash
 * @param  data     Byte being written
 *
 * @return  RET_OK when successful,
 *          RET_PARAM_ERROR if address is outside of Application area
 *****************************************************************************/
static uint8_t TI_MSPBoot_MI_WriteByteDirect(uint16_t addr, uint8_t data)
{
    FCTL3 = FWKEY;
    FCTL1 = FWKEY+ WRT;                         // Enable flash write
#if (MCLK==1000000)
    FCTL2 = FWKEY + FSSEL_2 + 2;              // Flash timing setup (SMCLK/3) = 333.333Khz
#elif (MCLK==4000000)
    FCTL2 = FWKEY + FSSEL_2 + 8;              // Flash timing setup (SMCLK/9) = 444.444Khz
#elif (MCLK==8000000)
    FCTL2 = FWKEY + FSSEL_2 + 16;             // Flash timing setup (SMCLK/17) = 470.588Khz
#else
#error "Please define a valid MCLK or add configuration"
#endif
    *(unsigned char*)addr = data;               // Write data to flash
    FCTL1 = FWKEY;                              // Disable flash write
    FCTL3 = FWKEY + LOCK;

    return RET_OK;
}


/******************************************************************************
 *
 * @brief   Replaces the application area with the contents of download area
 *          This function should only be called after validating the download
 *          area.
 *
 * @return  none
 *****************************************************************************/
void TI_MSPBoot_MI_ReplaceApp(void)
{
    volatile uint16_t addr;

    for (addr = APP_START_ADDR; addr <= APP_END_ADDR; addr++)
    {
        TI_MSPBoot_MI_WriteByteDirect(addr, *(uint8_t *)(TI_MSPBoot_MI_GetPhysicalAddressFromVirtual(addr)));
    }
}


/******************************************************************************
 *
 * @brief   Convert a virtual address of application to a physical address in
 *          download area
 *
 * @param  addr     Address in application memory
 *
 * @return  Physical address in download area
 *
 * @note: the address must be in application area, this function doesn't validate
 *          the input
 *****************************************************************************/
uint16_t TI_MSPBoot_MI_GetPhysicalAddressFromVirtual(uint16_t addr)
{
    volatile uint16_t ret;
    volatile uint16_t address;
    extern uint16_t _Down_Offset_Size;			/*! Download Offset Size */
    extern uint16_t _Down_Offset1;				/*! Download Offset 1 */

    // Application will be downloaded to 2 areas in memory. I.e.
    // data from 4400-7DFF  will be downloaded to BE00-F7FF
    // data from 7E00-BDFF will be downloaded to 10000-13FFF

    address = addr;

	ret = (addr + (uint16_t )&_Down_Offset1);


    return ret;
}

