/* --COPYRIGHT--,BSD
 * Copyright (c) 2016, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
//*****************************************************************************

//*****************************************************************************
//  LongTouch.c
//
//! LongTouch is used for applications that require the user to continuously touch
//! a sensor element for long periods of time (such as a video game controller or
//! headphones). This brings the benefit of being able to use environmental tracking
//! while the user is touching the element. Example: A user picks up a video game
//! controller. After a continuous touch has been detected for 5 seconds, the system
//! will update the LongTouch variable to "true". The LongTouch variable will stay
//! "true" as long as the user continues to touch the controller. When a touch is not
//! detected after 5 seconds, the LongTouch variable will become "false".
//! This is done by evaluating the sensor mode (self or mutual), flipping the
//! Direction of Interest, and reseeding the LTA. This example uses Button 1 on the
//! BSWP EVM board in Self Capacitive Mode.
//
//*****************************************************************************

#include "LongTouch.h"

void LongTouch_Handler(tSensor *pSensor, tLongTouchParams *params)
{
    // How long has the long touch been so far?
    uint32_t ui32Touch_Duration = pSensor->ui16Timeout * g_uiApp.ui16ActiveModeScanPeriod;

    if (params->bLongTouch ^ pSensor->bSensorTouch)
        {
            params->bTrueTouch = true;
        }
        else
        {
            params->bTrueTouch = false;
        }

        if (ui32Touch_Duration > params->ui32LongTouch_Timeout)
        {
            g_bLongTouchTransition = true;

            if(pSensor->DirectionOfInterest == eDOIDown)
                {
                    if(pSensor->SensingMethod == eSelf)
                    {
                        params->bLongTouch = true;
                    }
                    else
                    {
                        params->bLongTouch = false;
                    }
                    pSensor->DirectionOfInterest = eDOIUp;
                }

                else if(pSensor->DirectionOfInterest == eDOIUp)
                {
                    if(pSensor->SensingMethod == eProjected)
                    {
                        params->bLongTouch = true;
                    }
                    else
                    {
                        params->bLongTouch = false;
                    }
                    pSensor->DirectionOfInterest = eDOIDown;
                }

        }

}
