/* --COPYRIGHT--,BSD
 * Copyright (c) 2017, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
//*****************************************************************************
// Development main.c for MSP430FR2633, MSP430FR2533, MSP430FR2632, and
// MSP430FR2532.
//
// This starter application initializes the CapTIvate touch library
// for the touch panel specified by CAPT_UserConfig.c/.h via a call to
// CAPT_appStart(), which initializes and calibrates all sensors in the
// application, and starts the CapTIvate interval timer.
//
// Then, the capacitive touch interface is driven by calling the CapTIvate
// application handler, CAPT_appHandler().  The application handler manages
// whether the user interface (UI) is running in full active scan mode, or
// in a low-power wake-on-proximity mode.
//
// The CapTIvate application handler will return true if proximity was
// detected on any of the sensors in the application, which is used here
// to control the state of LED2. LED1 is set while the background loop enters
// the handler, and is cleared when the background loop leaves the handler.
//
// \version 1.80.00.27
// Released on March 05, 2019
//
//*****************************************************************************

#include <msp430.h>                      // Generic MSP430 Device Include
#include "driverlib.h"                   // MSPWare Driver Library
#include "captivate.h"                   // CapTIvate Touch Software Library
#include "CAPT_App.h"                    // CapTIvate Application Code
#include "CAPT_BSP.h"                    // CapTIvate EVM Board Support Package
#include "LongTouch.h"					 // CapTIcate LongTouch Mode

uint16_t buffer[4];
bool g_bLongTouchTransition = false;
//
//! When using a CapTIvate EVM board, this allows LED1 to illuminate when
//! a TrueTouch is detected. LED2 will illuminate when a LongTouch is detected.
//
void BTN00_updateHandler(tSensor *pSensor)
{
    static tLongTouchParams longTouchParams =
    {
        .ui32LongTouch_Timeout = 5000,
        .bTrueTouch = false,
        .bLongTouch = false
    };

    // Run the long touch handler for this sensor.
    LongTouch_Handler(pSensor, &longTouchParams);

    if((pSensor->bReCalibrateEnable == true) && (g_bLongTouchTransition == true))
    {
        CAPT_calibrateSensor(pSensor);
        g_bLongTouchTransition = false;
    }

	// Use LED1 to indicate when a real touch is present.
	if (longTouchParams.bTrueTouch == true)
	{
		LED1_ON;
		buffer[0] = 100;
	}
	else
	{
		LED1_OFF;
		buffer[0] = 0;
	}

	// Use LED2 to indicate when long touch mode is activated.
	if (longTouchParams.bLongTouch == true)
	{
		LED2_ON;
		buffer[1] = 50;
	}
	else
	{
		LED2_OFF;
		buffer[1] = 0;
	}
	buffer[2] = BTN00.pCycle[0]->pElements[0]->LTA.ui16Natural;
	buffer[3] = BTN00.pCycle[0]->pElements[0]->filterCount.ui16Natural;

	// Send data to User Log Window in GUI to demonstrate timing
	CAPT_writeGeneralPurposeData(buffer, 4);
}


void main(void)
{
	WDT_A_hold(WDT_A_BASE);
	BSP_configureMCU();
	__bis_SR_register(GIE);

	CAPT_appStart();
	MAP_CAPT_registerCallback(&BTN00, &BTN00_updateHandler);

	while(1)
	{
		CAPT_appHandler();
		CAPT_appSleep();
	}
}
