/*	CoRTOSblinkyMSP.c

	2017.03.03 10:00	Initial release

	MSP430FR6989, Texas Instruments Code composer studio

	A stand-alone 'blinky' demonstration of the CoRTOS Cooperative
	Real Time Operating System.

	It demonstrates the kernel and the time delay extension to the
	basic OS.

	Code has been skinnyed down for this demo file.

	(c) 2017 Nicholas O. Lindan,
	Released under GPL V3 https://www.gnu.org/licenses/gpl.html */

// #include <msp430.h>
#include "msp430FR6989.h"

#include "CoRTOScomdefs.h"

void led_on (uint8_t led);
void led_off (uint8_t led);
void initialize_io_and_interrupts (void);
void task0 (void);
void task1 (void);
void service_timers (void);
void delay (uint16_t ticks);
void suspend (void);
void resume_task (uint8_t tn);
void relinquish (void);
int main (void);

/*	Task number of the executing task - this can be useful for
	array indexing and for calls to the OS. */

uint8_t current_task;

/**********************************************************************
*
*	msp430 I/O
*
**********************************************************************/

/*	The interrupt is triggered 10mSec period on output
	compare A using timer 1.  Its function is to decrement
	the timers of any delayed tasks via a call to service_timers(). */

#pragma vector=TIMER1_A0_VECTOR
__interrupt void timerA1_CC0 (void)
{
	/*	This vector is shared with other timer interrupt sources.  At
		present there is only the one source triggered on overflow compare. */
	service_timers ();
	__enable_interrupt ();
}

/*	Change port as needed for your hardware - this works
	with ti's MSP-EXP30FR5969 Explorer board. */
#define red_led 0x00
#define green_led 0x01

void led_on (uint8_t led)
{
	if (led == red_led)
		P1OUT |= 0x01;
	else
		P9OUT |= 0x80;
}

void led_off (uint8_t led)
{
	if (led == red_led)
		P1OUT &= ~0x01;
	else
		P9OUT &= ~0x80;
}

#define development_WDT 0x5A80
void initialize_io_and_interrupts (void)
{

	/*	Stop the watchdog timer. */
	WDTCTL = development_WDT;

	/*	Change port as needed - this is for the the
		ti's MSP-EXP430FR6989 Explorer board. */
	PM5CTL0 = 0xfffe;
	P1DIR = 0x01;
	P9DIR = 0x80;

	/*	Set one wait state for FRAM access, the a5 is the unlock code,
		10 is the number of wait states - in the high nibble.  This
		is needed as the FRAM can only run at 8MHz and so needs the
		wait state when the CPU runs at 16MHz. */
	FRCTL0 = 0xa510; FRCTL0 = 0xa510;

	/*	Setup XT1 */
	PJSEL0 |= BIT4 | BIT5;

	/*	Unlock CS registers */
	CSCTL0 = 0xa500;

	/*	Set DCO to 16MHz */
	CSCTL1 = DCOFSEL_4|DCORSEL;

	/*	set ACLK = XT1; SMCLK= MCLK = DCO = 16MHz */
	CSCTL2 = SELA__LFXTCLK | SELS__DCOCLK | SELM__DCOCLK;

	/*	Set all dividers */
	CSCTL3 = DIVA__1 | DIVS__1 | DIVM__1;
	CSCTL4 &= ~LFXTOFF;

	/*	Timer A1: SMCLK/8, MC_1 upmode, TASSEL_2 : 2 MHz clock source */
	TA1CTL = TASSEL_2 + MC_1 + ID_3;

	/*	10mSec period */
	TA1CCR0 =  20000;

	/*	CCR0 timer interrupt enabled */
	TA1CCTL0 |= CCIE;
}

/**********************************************************************
*
*	Tasks
*
**********************************************************************/

/*	These three lines tell the OS about the tasks. */

typedef void (* task_address_type) (void);
task_address_type start_addresses [2] = {task0, task1};
#define number_of_tasks 2

/*	Task 0 turns the LED on and then waits for 1 second, then
	it does it again.

	Task1 waits 0.5 second and then turns the led off and waits
	another 0.5 second and then repeats.

	As a result the led is on for 0.5 second and then off for 0.5
	second. */

void task0 (void)
{
	while (1)
	{
		led_on (red_led);
		delay (53);
		led_off (red_led);
		delay (53);
	}
}

void task1 (void)
{
	while (1)
	{
		led_on (green_led);
		delay (71);
		led_off (green_led);
		delay (71);
	}
}

/**********************************************************************
*
*	Delay
*
**********************************************************************/

/*	This is a pared-down version of the code in CoRTOSdelay.c,
	but it is an accurate depiction of the action of the delay
	function.

	Note that delay() is independent of the kernel.  If there are
	no calls to delay() in the developed system then the whole
	of the delay code can be left out of the link with no impact
	on the rest of the system.

	This independence is true of all the CoRTOS extensions - link
	in only the features you need. */

#define number_of_timers 2
static uint16_t timer [number_of_timers];
static boolean timer_active [number_of_timers];

/*	Called by the 10mSec interrupt.  If a timer is non-zero it is because
	a task has delayed itself.  When the timer reaches 0 the task is
	resumed/made active so that it will be given control by relinquish(). */

void service_timers (void)
{
	uint8_t tn;

	for (tn = 0; tn < number_of_timers; tn++)
	{
		if (timer_active[tn] == true)
		{
			if (--timer[tn] == 0)
			{
				resume_task (tn);
				timer_active[tn] = false;
			}
		}
	}
}

/*	Called by a task to delay itself.  The task is taken out of action
	for ticks * 10mSec. */

void delay (uint16_t ticks)
{
	/*	A task can only delay itself.  Task numbers == timer
		numbers - this is only in this simplified version.

		The task calls pause to set it to inactive.  It gives up control
		by calling relinquish() - relinquish() will then skip this task
		when its turn to run comes up again.  The task is made active by
		the service_timers() function's call to resume_task().

		The call to pause() is made before the timer itself is made
		active so that if the timer[] setting was not atomic it doesn't
		matter as the ISR won't be decrementing the timer until it is
		active.  */

	timer_active[current_task] = false;
	timer[current_task] = ticks;
	suspend ();
	timer_active[current_task] = true;
	relinquish ();
}

/**********************************************************************
*
*	Assembler macros
*
**********************************************************************/

/*	Specific for each uP.

	N.B. for the MSP CCS V7, a space is needed between " and push/pop */

// - or - " pushm.w #8, r11 " for those msp430's that support it
#define pushall() asm volatile \
	(							\
		" push r4  \n\t"	\
		" push r5  \n\t"	\
		" push r6  \n\t"	\
		" push r7  \n\t"	\
		" push r8  \n\t"	\
		" push r9  \n\t"	\
		" push r10 \n\t"	\
		" push r11 \n\t"	\
	)

// - or - " popm.w #8, r11 \n\t" for those msp430's that support it
#define popall() asm volatile \
	(							\
		" pop r11 \n\t"	\
		" pop r10 \n\t"	\
		" pop r9  \n\t"	\
		" pop r8  \n\t"	\
		" pop r7  \n\t"	\
		" pop r6  \n\t"	\
		" pop r5  \n\t"	\
		" pop r4  \n\t"	\
	)

/**********************************************************************
*
*	Kernel
*
**********************************************************************/

static uint16_t sp_save [number_of_tasks];
static uint16_t starting_stack [number_of_tasks];
static uint8_t start_from_beginning [number_of_tasks];
static uint8_t task_active [number_of_tasks];

/*	pause() and resume() are used to quickly skip tasks that are blocked,
	in this demo they are blocked by being delayed. */

void suspend (void)
{
	task_active[current_task] = false;
}

void resume_task (uint8_t tn)
{
	task_active[tn] = true;
}

/*	Relinquish is the heart of CoRTOS.

	It is a round-robin task scheduler.

	Only active tasks are given control.  Inactive tasks are skipped
	until they are made active again.  Tasks make themselves inactive
	when they are blocked.  The tasks are made active again when the
	block is removed by an interrupt or another task.
*/

void relinquish (void)
{
	/*	Save whatever it is that the compiler expects to be saved.  This,
		and the reference to the stack pointer are specific to the uP. */
	pushall();
	sp_save[current_task] = _get_SP_register();

	while (1)
	{
		do
		{
			/*	Circle through the tasks, the if() construct is much
				faster than using a mod % operator. */
			if (++current_task == number_of_tasks)
				current_task = 0;

		/*	Skipping inactive tasks.  It is not uncommon for the system
			to spend most of it's time chasing its tail going through a
			list of all inactive tasks.  This is a good place to add code
			for putting the processor in a low power state.  It is also
			a good place to modulate an output pin to make a PWM measurement
			of processor loading. */
		} while (task_active[current_task] == false);

		if (start_from_beginning[current_task] == true)
		{
			/*	This is mostly the case for tasks when the system starts.
				task0 (the task at start_addresses[0]) however has been
				started when the system was put into motion. */
			start_from_beginning[current_task] = false;
			_set_SP_register (starting_stack[current_task]);
			start_addresses[current_task] ();

			/*	If a task should execute a return the the uP will come to
				this spot.  The task is taken out of circulation.  The start_
				from_beginning flag is set in case the task is made active. */
			task_active[current_task] = false;
			start_from_beginning[current_task] = true;
		}
		else
		{
			/*	This is the normal round robin return-to-task point.  The
				tasks stack and registers are restored.

				Note that as tasks enter and exit through relinquish any
				code the compiler may add for function entry/exit is automatically
				'cancelled out.'  The GCC compiler will change this code with
				optimization level. */
			 _set_SP_register (sp_save[current_task]);
			popall();
			return;
		}
	}
}

int main (void)
{
	/*	Initialize the CoRTOS variables, a bit skinnyed down for this
		demonstration program. */
	starting_stack[0] = _get_SP_register();
	starting_stack[1] = _get_SP_register() - 60;

	start_from_beginning[0] = false;
	start_from_beginning[1] = true;

	task_active[0] = true;
	task_active[1] = true;

	timer[0] = 0;
	timer[1] = 0;
	timer_active[0] = false;
	timer_active[1] = false;

	initialize_io_and_interrupts ();
	__enable_interrupt ();

	/*	Pass control to task0 - the one whose start address is at position
		0 in the start addresses array. */
	current_task = 0;
	start_addresses[0] ();

	/*	task0 is different from the other tasks - if it executes a return
		then control passes here and the OS stops. */
	while (1)
		;
	return (0);
}
